/* Copyright (C) 2002 Asfand Yar Qazi.

 This file is part of XBobble.

    XBobble is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    XBobble is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XBobble; if not, write to the Free Software Foundation,
    Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

/** @file Video_Output_Manager.cc

    @see Video_Output_Manager.hh */

#include "Video_Output_Manager.hh"
#include "System.hh"
#include "Game_Manager.hh"
#include "Video_Data.hh"
#include <cstdlib>
#include <stdexcept>
#include <iostream>
#include <SDL.h>
#include <SDL_image.h>
#include <SDL_opengl.h>

namespace XBobble
{

// class Test_Video_Data : public Video_Data
// {
// public:
// 	Test_Video_Data(Video_Output_Manager& vo)
// 	{
// 		vo.register_data(this);
// 	}

// 	void
// 	draw_handler() const
// 	{
// 		GLUquadric* q = gluNewQuadric();
// 		gluSphere(q, 50, 50, 50);
// 		gluDeleteQuadric(q);
// 	}
// };

namespace
{

/// @internal Thrown by try_init_if_not_init when we finally have a
/// screen!  Holds valid screen.
class Screen_Is_Init
{
public:
	Screen_Is_Init(SDL_Surface* arg)
	 : screen(arg)
	{
	}

	SDL_Surface* screen;
};

/// @internal Try to init a the given surface if it is currently 0.
/// If it is already non-zero, return it as it is.
void
try_init_if_not_init(int32_t scrw, int32_t scrh,
		     int32_t bpp, uint32_t flags)
	throw(Screen_Is_Init)
{
	SDL_Surface* screen = SDL_SetVideoMode(scrw, scrh, bpp, flags);
	if(screen) throw Screen_Is_Init(screen);
}

} // namespace

/// Pimpl class
class Video_Output_Manager::Pimpl
{
	friend class Video_Output_Manager;

	/// SDL flags used to initialise SDL screen
	uint32_t sdl_flags;

	/// A list of video datas to draw each screen
	std::vector<Video_Data*> data_list;

	/// Convenience
	typedef std::map<std::string, uint32_t> TName_Map;

	/// Give a texture name (a string), get a GL texture index (ie
	/// an int)
	TName_Map tex_list;

}; // class Video_Output_Manager::Pimpl

Video_Output_Manager::Video_Output_Manager(System& arg_sys)
 : Singleton_Class("Video_Output_Manager"), system(arg_sys),
   impl(new Pimpl()) // Throws bad_alloc, removes us from
		     // Singleton_Class list, and does no more.  Which
		     // is good.
{
    try
    {
	impl->data_list.reserve(20);

	// We MUST have doublebuffering.  I refuse to work with a
	// system that doesn't.
	SDL_GL_SetAttribute(SDL_GL_DOUBLEBUFFER, 1);

	// We don't need alpha layer to use alpha blending
	SDL_GL_SetAttribute(SDL_GL_ALPHA_SIZE, 0);

	int scrw = get_width();
	int scrh = get_height();
	bool fullscreen
		= get_fullscreen();

	if(scrw == 0)
		throw Init_Error(get_my_name()
				 + " - screen width too small");

	if(scrh == 0)
		throw Init_Error(get_my_name()
				 + " - screen height too small");

	impl->sdl_flags = SDL_HWSURFACE | SDL_OPENGL;
	if(fullscreen)
		impl->sdl_flags |= SDL_FULLSCREEN;

	try
	{
		// Try for 24-bit depth buffer (i.e. 24/32-bit display
		// mode)
		SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE, 24);
		try_init_if_not_init(scrw, scrh, 0, impl->sdl_flags);

		// Then 16-bit depth buffer (i.e. 16-bit display mode)
		SDL_GL_SetAttribute(SDL_GL_DEPTH_SIZE, 16);
		try_init_if_not_init(scrw, scrh, 0, impl->sdl_flags);

		// Uh-ho.  Screen_Is_Init not thrown.  So, we haven't
		// got a screen!
		throw Init_Error(get_my_name()
				 + " - Couldn't init OpenGL visual.");
	}
	catch(Screen_Is_Init& sii)
	{
		// We're ok now.  Screen is up and running
		screen = sii.screen;
	}

	SDL_WM_SetCaption("XBOBBLE!", "XBOBBLE!");

	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	glEnable(GL_DEPTH_TEST);
	glShadeModel(GL_SMOOTH);
	glEnable(GL_TEXTURE_2D);
	glFrontFace(GL_CCW);
	glCullFace(GL_BACK);
	glEnable(GL_CULL_FACE);

	if(system.game_manager.get_option("alpha_enabled").at(0) == '1') {
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	}

	// glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, 1);


	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	// Note to self: These are NOT Viewport dependant!
	glFrustum(-100, 100, -75, 75, 500, 10000);

	glMatrixMode(GL_MODELVIEW);
    }
    catch(...)
    {
	    delete impl;
	    throw;
    }
}

int
Video_Output_Manager::get_width() const
{
	return atoi(system.game_manager.get_option("width").c_str());
}

int
Video_Output_Manager::get_height() const
{
	return atoi(system.game_manager.get_option("height").c_str());
}

bool
Video_Output_Manager::get_fullscreen() const
{
	return static_cast<bool>(atoi(system.game_manager.get_option(
					      "fullscreen").c_str()));
}

void
Video_Output_Manager::switch_fullscreen()
{
	if(SDL_WM_ToggleFullScreen(screen) == 0)
		return;

	int fs_val = !get_fullscreen();
	char charval[] ="0";
	if(fs_val)
		charval[0] = '1';
	system.game_manager.set_option("fullscreen", charval);

	impl->sdl_flags ^= SDL_FULLSCREEN;
}

void
Video_Output_Manager::resize()
{
	screen = SDL_SetVideoMode(get_width(), get_height(),
				  0, impl->sdl_flags);

	glViewport(0, 0, get_width(), get_height());
}

void
Video_Output_Manager::clear_screen()
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
}

void
Video_Output_Manager::draw_screen()
{
	std::vector<Video_Data*>::iterator it, itend;

	glLoadIdentity();
	clear_screen();

	// Move camera back and reverse y and z axes (corresponding to
	// 'Grid' coordinates)
	glTranslatef(0, 0, -1000);
	glRotatef(180, 1, 0, 0);

	glTranslatef(0, -20, 0);
	// Lean back slightly
	glRotatef(-25, 1, 0, 0);

	for(it = impl->data_list.begin(), itend = impl->data_list.end();
	    it != itend; ++it) {
		glPushMatrix();
		(*it)->draw();
		glPopMatrix();
	}

	glFlush();
	SDL_GL_SwapBuffers();
}

void
Video_Output_Manager::register_data(Video_Data* arg)
{
	if(arg == 0)
		throw std::invalid_argument(
			"Video_Output_Manager::register_data() - "
			"NULL pointer!  wot u finkin'!");

	impl->data_list.push_back(arg);
}

namespace
{

class SDL_Surface_Deleter
{
	SDL_Surface* img;
public:
	SDL_Surface_Deleter(SDL_Surface* arg)
	 : img(arg)
	{
	}

	~SDL_Surface_Deleter()
	{
		if(img) {
			SDL_FreeSurface(img);
			img = 0;
		}
	}
};

} // namespace

uint32_t
Video_Output_Manager::get_texture(const std::string& name)
{
	Game_Manager& gm = system.game_manager;
	Pimpl::TName_Map::iterator it;
	if((it = impl->tex_list.find(name)) == impl->tex_list.end()) {
		std::string actualname
			= gm.get_option("data_root")
			+ "/" + name;
		// load texture, add with given name, and return.
		SDL_Surface* img = IMG_Load(actualname.c_str());
		if(!img) {
			std::string msg
				= std::string() +
				"Video_Output_Manager::get_texture() - " +
				"invalid file " + name;
			throw std::invalid_argument(msg);
		}
		SDL_Surface_Deleter imgdel(img);

		if(img->w < 64)
			throw std::invalid_argument(
				std::string() +
				"Video_Output_Manager::get_texture() - " +
				"width of image " + name + " < 64!");

		if(img->h < 64)
			throw std::invalid_argument(
				std::string() +
				"Video_Output_Manager::get_texture() - " +
				"height of image " + name + " < 64!");

		 bool width_power_of_2 = ((img->w&(img->w-1))==0);
		 bool height_power_of_2 = ((img->h&(img->h-1))==0);

		 if(!width_power_of_2)
			throw std::invalid_argument(
				std::string() +
				"Video_Output_Manager::get_texture() - " +
				"width is not a power of 2");

		 if(!height_power_of_2)
			throw std::invalid_argument(
				std::string() +
				"Video_Output_Manager::get_texture() - " +
				"height is not a power of 2");

		GLuint gltexname;
		glGenTextures(1, &gltexname);
		glBindTexture(GL_TEXTURE_2D, gltexname);
		bool hasalpha = false;
		if(img->format->Aloss == 0) // 8-bit alpha channel
			hasalpha = true;
		glTexImage2D(GL_TEXTURE_2D, 0, (hasalpha)?GL_RGBA8:GL_RGB8,
			     img->w, img->h, 0, (hasalpha)?GL_RGBA:GL_RGB,
			     GL_UNSIGNED_BYTE, img->pixels);
		GLint mag_filter = GL_LINEAR, min_filter = GL_LINEAR;
		std::string op = gm.get_option("GL_TEXTURE_MAG_FILTER");
		if(op == "GL_NEAREST")
			mag_filter = GL_NEAREST;
		else if(op != "GL_LINEAR")
			std::cerr << "Option 'GL_TEXTURE_MAG_FILTER' set to"
				  << " invalid value {" << op << "},\n"
				  << "will default to GL_LINEAR\n";
		op = gm.get_option("GL_TEXTURE_MIN_FILTER");
		if(op == "GL_NEAREST")
			min_filter = GL_NEAREST;
		else if(op != "GL_LINEAR")
			std::cerr << "Option 'GL_TEXTURE_MIN_FILTER' set to"
				  << " invalid value {" << op << "},\n"
				  << "will default to GL_LINEAR\n";
		glTexParameteri(GL_TEXTURE_2D,  GL_TEXTURE_MIN_FILTER,
				min_filter);
		glTexParameteri(GL_TEXTURE_2D,  GL_TEXTURE_MAG_FILTER,
				mag_filter);

		// If this fails, I'll be an uncle's monkey.
		impl->tex_list.insert(make_pair(name, gltexname));
		return gltexname;
	} else
		return it->second;

} // Video_Output_Manager::get_texture()

uint32_t
Video_Output_Manager::get_texture(const char* name)
{
	return get_texture(std::string(name));
}

} // namespace XBobble


