/*
   Copyright (C) 2010, 2012 Oleksiy Chernyavskyy

   This file is part of XDClient.

   XDClient is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   XDClient is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with XDClient.  If not, see <http://www.gnu.org/licenses/>.
*/


#include <stdlib.h>
#include <string.h>
#include <wchar.h>
#include <stdio.h>
#include "hash.h"
#include "utf8.h"
#include "regexp_utf8.h"


nameval_t* hlookup(void* symtab_in, const char* name, int create)
{
  int h;
  nameval_t* nv;
  void **symtab;

  symtab = (void**) symtab_in;

  if (!symtab || !name)
	return NULL;

  h = hash(name);
  for (nv = (nameval_t*) symtab[h]; nv != NULL; nv = nv->next)
	if (!strcmp(name, nv->name)) return nv;
  if (create) {
	nv = (nameval_t*) malloc(sizeof(nameval_t));
	nv->name = strdup(name);
	nv->mbs_val = NULL;
	nv->wcs_val = NULL;
	nv->ival = 0;
	nv->val = NULL;
	nv->val_type = 0;
	nv->val_free = NULL;
	nv->next = symtab[h];
	symtab[h] = nv;
  }
  return nv;
}

unsigned int hash(const char *str)
{
  unsigned int h;
  unsigned char *p;

  if (!str) {
	fprintf(stderr, "error: %s: parameter == NULL\n", __FUNCTION__);
	return 0;
  }

  h = 0;
  for (p = (unsigned char*) str; *p != '\0'; p++)
	h = MULTIPLIER * h + *p;
  return h % NHASH;
}

void htable_free(void *symtab_in, unsigned flags)
{
  nameval_t *nv, *nv_next;
  int i;
  void **symtab;

  symtab = (void**) symtab_in;

  if (!symtab)
	return;

  for (i=0; i<NHASH; i++) {
	if (nv = (nameval_t*) symtab[i]) {
	  while (nv) {
		nv_next = nv->next;
		if (nv->name)
		  free(nv->name);
		if (! (flags & HT_SKIP_STR)) {
		  if (nv->mbs_val)
			free(nv->mbs_val);
		  if (nv->wcs_val)
			free(nv->wcs_val);
		}

		if (nv->val && nv->val_free)
		  (*nv->val_free)(nv->val);
		else if (nv->val && flags & HT_FORCE)
		  free(nv->val);

		free(nv);
		nv = nv_next;
	  }
	}
  }
  free(symtab_in);
}

void htable_regex_free(void *symtab_in, char *pat, unsigned flags)
{
  nameval_t *nv, *nv_prev, *nv_next;
  int i;
  void **symtab;

  symtab = (void**) symtab_in;

  if (!symtab)
	return;

  for (i=0; i<NHASH; i++) {
	if (nv = (nameval_t*) symtab[i]) {
	  nv_prev = NULL;
	  while (nv) {
		nv_next = nv->next;
		if (reg_test_8(nv->name, towcs(pat), RE_PREFIX_DASH | RE_SUFFIX_DOLLAR, NULL)) {
		  if (nv->name)
			free(nv->name);
		  if (! (flags & HT_SKIP_STR)) {
			if (nv->mbs_val)
			  free(nv->mbs_val);
			if (nv->wcs_val)
			  free(nv->wcs_val);
		  }

		  if (nv->val && nv->val_free)
			(*nv->val_free)(nv->val);
		  else if (nv->val && flags & HT_FORCE)
			free(nv->val);

		  free(nv);
		  if (nv_prev)
			nv_prev->next = nv_next;
		  else
			symtab[i] = nv_next;
		} else {
		  nv_prev = nv;
		}
		nv = nv_next;
	  }
	}
  }
}

void* create_htable(void)
{
  void **symtab;

  symtab = (void**) malloc(sizeof(void*) * NHASH);
  hash_init(symtab);
  return (void*) symtab;
}

void hash_init(void* symtab)
{
  memset(symtab, 0, sizeof(void*) * NHASH);
}

void print_htable(void *_symtab, void (*nv_print)(nameval_t*, int), int shift)
{
  nameval_t *nv, *nv_next;
  int i;
  void **symtab;

  symtab = (void**) _symtab;

  if (!symtab)
	return;

  for (i=0; i<NHASH; i++) {
	if (nv = (nameval_t*) symtab[i]) {
	  while (nv) {
		if (nv->name) {
		  ppad(shift);
		  fprintf(stdout, "name: %s\n", nv->name);
		} else {
		  ppad(shift);
		  fprintf(stdout, "no name\n");
		}

		if (nv->mbs_val) {
		  ppad(shift+2);
		  fprintf(stdout, "mbs_val: %s\n", nv->mbs_val);
		}
		if (nv->wcs_val) {
		  ppad(shift+2);
		  fwprintf(stdout, L"wcs_val: %S\n", nv->wcs_val);
		}

		if (nv->val && nv_print) {
		  ppad(shift+2);
		  fprintf(stdout, "val:\n");
		  (*nv_print)(nv, shift+4);
		}

		fprintf(stdout, "\n");

		nv = nv->next;
	  }
	}
  }
}

wnameval_t* whlookup(void* symtab_in, const wchar_t* name, int create)
{
  int h;
  wnameval_t* nv;
  void **symtab;

  symtab = (void**) symtab_in;

  if (!symtab || !name)
	return NULL;

  h = whash(name);
  for (nv = (wnameval_t*) symtab[h]; nv != NULL; nv = nv->next)
	if (!wcscmp(name, nv->name)) return nv;
  if (create) {
	nv = (wnameval_t*) malloc(sizeof(wnameval_t));
	nv->name = wcsdup(name);
	nv->mbs_val = NULL;
	nv->wcs_val = NULL;
	nv->ival = 0;
	nv->val = NULL;
	nv->val_type = 0;
	nv->val_free = NULL;
	nv->next = symtab[h];
	symtab[h] = nv;
  }
  return nv;
}

unsigned int whash(const wchar_t *str)
{
  unsigned int h;
  const wchar_t *p;

  if (!str) {
	fprintf(stderr, "error: %s: parameter == NULL\n", __FUNCTION__);
	return 0;
  }

  h = 0;
  for (p = str; *p != L'\0'; p++)
	h = MULTIPLIER * h + (unsigned int) *p;
  return h % NHASH;
}

void whtable_free(void *symtab_in, unsigned flags)
{
  wnameval_t *nv, *nv_next;
  int i;
  void **symtab;

  symtab = (void**) symtab_in;

  if (!symtab)
	return;

  for (i=0; i<NHASH; i++) {
	if (nv = (wnameval_t*) symtab[i]) {
	  while (nv) {
		nv_next = nv->next;
		if (nv->name)
		  free(nv->name);
		if (! (flags & HT_SKIP_STR)) {
		  if (nv->mbs_val)
			free(nv->mbs_val);
		  if (nv->wcs_val)
			free(nv->wcs_val);
		}

		if (nv->val && nv->val_free)
		  (*nv->val_free)(nv->val);
		else if (nv->val && flags & HT_FORCE)
		  free(nv->val);

		free(nv);
		nv = nv_next;
	  }
	}
  }
  free(symtab_in);
}

void whtable_regex_free(void *symtab_in, wchar_t *pat, unsigned flags)
{
  wnameval_t *nv, *nv_prev, *nv_next;
  int i;
  void **symtab;

  symtab = (void**) symtab_in;

  if (!symtab)
	return;

  for (i=0; i<NHASH; i++) {
	if (nv = (wnameval_t*) symtab[i]) {
	  nv_prev = NULL;
	  while (nv) {
		nv_next = nv->next;
		if (reg_test(nv->name, pat, RE_PREFIX_DASH | RE_SUFFIX_DOLLAR, NULL)) {
		  if (nv->name)
			free(nv->name);
		  if (! (flags & HT_SKIP_STR)) {
			if (nv->mbs_val)
			  free(nv->mbs_val);
			if (nv->wcs_val)
			  free(nv->wcs_val);
		  }

		  if (nv->val && nv->val_free)
			(*nv->val_free)(nv->val);
		  else if (nv->val && flags & HT_FORCE)
			free(nv->val);

		  free(nv);
		  if (nv_prev)
			nv_prev->next = nv_next;
		  else
			symtab[i] = nv_next;
		} else {
		  nv_prev = nv;
		}
		nv = nv_next;
	  }
	}
  }
}

void print_whtable(void *_symtab, void (*nv_print)(wnameval_t*, int), int shift)
{
  wnameval_t *nv, *nv_next;
  int i;
  void **symtab;

  symtab = (void**) _symtab;

  if (!symtab)
	return;

  for (i=0; i<NHASH; i++) {
	if (nv = (wnameval_t*) symtab[i]) {
	  while (nv) {
		if (nv->name) {
		  ppad(shift);
		  fwprintf(stdout, L"name: %S\n", nv->name);
		} else {
		  ppad(shift);
		  fprintf(stdout, "no name\n");
		}

		if (nv->mbs_val) {
		  ppad(shift+2);
		  fprintf(stdout, "mbs_val: %s\n", nv->mbs_val);
		}
		if (nv->wcs_val) {
		  ppad(shift+2);
		  fwprintf(stdout, L"wcs_val: %S\n", nv->wcs_val);
		}

		if (nv->val && nv_print) {
		  ppad(shift+2);
		  fprintf(stdout, "val:\n");
		  (*nv_print)(nv, shift+4);
		}

		fprintf(stdout, "\n");

		nv = nv->next;
	  }
	}
  }
}

