/*
    Copyright (C) 2012 Oleksiy Chernyavskyy

    This file is part of XDClient.

    XDClient is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    XDClient is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XDClient.  If not, see <http://www.gnu.org/licenses/>.
*/

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <wctype.h>
#include <ctype.h>
#include "conf.h"
#include "common.h"
#include "file.h"
#include "utf8.h"
#include "hash.h"
#include "adt_id.h"
#include "regexp_utf8.h"

xdc_conf_t *global_xdc_conf = NULL;

/*ARGSUSED*/
void xdc_conf_init(xdc_conf_t *xdc_conf, int argc, char **argv)
{
  if (!xdc_conf)
	return;

  xdc_conf->symtab = create_htable();
  xdc_conf->wsymtab = create_htable();
  xdc_conf_add_arg(xdc_conf, "argv0", strdup(argv[0]));
  xdc_conf->warn = 1;
  xdc_conf->verbose = 0;
  xdc_conf->skip_abbrev = 1;

  xdc_conf->cmd_lang_from = 0;
  xdc_conf->cmd_lang_from_all = 0;
  xdc_conf->env_lang_from = 0;
  xdc_conf->env_lang_from_all = 0;
  xdc_conf->cfg_lang_from = 0;
  xdc_conf->cfg_lang_from_all = 0;
  xdc_conf->cmd_lang_to = 0;
  xdc_conf->cmd_lang_to_all = 0;
  xdc_conf->env_lang_to = 0;
  xdc_conf->env_lang_to_all = 0;
  xdc_conf->cfg_lang_to = 0;
  xdc_conf->cfg_lang_to_all = 0;
  xdc_conf->cmd_dict_name = 0;
  xdc_conf->env_dict_name = 0;
  xdc_conf->cfg_dict_name = 0;
  xdc_conf->cmd_verbose = 0;
  xdc_conf->env_verbose = 0;
  xdc_conf->cfg_verbose = 0;
  xdc_conf->verbose = 0;
  xdc_conf->cmd_no_verbose = 0;
  xdc_conf->cmd_list = 0;
  xdc_conf->list_dicts = 0;
  xdc_conf->describe = 0;
  xdc_conf->use_colors = 1;
}

void xdc_conf_set_defaults(xdc_conf_t *xdc_conf)
{
  nameval_t *nv;
  char *prog_home;

  if (!xdc_conf)
	return;

  nv = hlookup(xdc_conf->symtab, "argv0", 0);
  if ((prog_home = chop_path(get_prog_dir(nv->mbs_val))))
	xdc_conf_add_arg(xdc_conf, "program_home", prog_home);

  save_arg_path(xdc_conf, pfmbs("~/.xdc/xdc.conf:~/.xdc.conf:~/.config/xdc/xdc.conf:~/.config/xdc.conf:/etc/xdc.conf:/usr/local/etc/xdc.conf:%s:/usr/share/xdclient/xdc.conf:/usr/local/share/xdclient/xdc.conf", prog_home), "default_config_path");

  save_arg_path(xdc_conf, pfmbs("~/.xdc:~/.config/xdc:/usr/share/xdclient/index:/usr/local/share/xdclient/index:%s/xdc_index", \
		prog_home), "default_index_path");
  
  save_arg_path(xdc_conf, pfmbs("~/.xdc:~/.config/xdc:/usr/share/xdclient/dict:/usr/local/share/xdclient/dict:%s/xdc_dict", \
		prog_home), "default_path");

  save_arg_path(xdc_conf, pfmbs("~/.xdc:~/.config/xdc:/tmp/xdc_output", \
		prog_home), "default_output_dir");
}

int xdc_conf_file_read(xdc_conf_t *xdc_conf)
{
  FILE *fp;
  int llen;
  char c;
  int i;
  char *buf;
  int buf_len = 0;
  int line_num;
  int var_start;
  int var_len;
  int val_start;
  int val_len;
  nameval_t *nv;
  int inq, indq;
  long line_start;
  char *conf_path = NULL;
  mbs_ll_t *mbsl;
  char *mbs;

  if (!xdc_conf)
	return 0;

  if ((nv = hlookup(xdc_conf->symtab, "cmd_arg_conf", 0))) {
	if (nv->mbs_val && is_rfile(nv->mbs_val) && !is_dir(nv->mbs_val)) {
	  conf_path = strdup(nv->mbs_val);
	} else {
	  fprintf(stderr, "xdclient: error: invalid config path command line argument\n");
	  return 0;
	}
  }

  if (!conf_path && (nv = hlookup(xdc_conf->symtab, "env_arg_conf_path", 0))) {
	if (nv->mbs_val && is_rfile(nv->mbs_val) && !is_dir(nv->mbs_val)) {
	  conf_path = strdup(nv->mbs_val);
	} else {
	  fprintf(stderr, "xdclient: error: invalid config path environment variable\n");
	  return 0;
	}
  }

  if (!conf_path) {
	nv = hlookup(xdc_conf->symtab, "default_config_path", 0);
	mbsl = (mbs_ll_t*) nv->val;
	while (mbsl && !conf_path) {
	  if (is_rfile(mbsl->mbs) && !is_dir(mbsl->mbs))
		conf_path = strdup(mbsl->mbs);
	  mbsl = mbsl->next;
	}
  }

  /* no config file found. It's OK */
  if (!conf_path)
	return 1;

  fp = fopen(conf_path, "r");
  if (!fp) {
	fprintf(stderr, "xdclient: %s: failed to open config file: %s\n", __FUNCTION__, conf_path);
	return 0;
  }

  line_num = 0;
  buf = NULL;
  buf_len = 0;

RCF_INEXT:
  while(! feof(fp)) {
	line_start = ftell(fp);
	line_num++;
/* read string line into buffer */
	c = fgetc(fp);
	llen = 0;
	while (c != '\n' && !feof(fp)) {
	  llen++;
	  c = fgetc(fp);
	}

	if (! llen)
	  goto RCF_INEXT;

	if (buf_len <= llen) {
	  if (buf)
		free(buf);
	  buf_len = llen*10;
	  buf = (char*) malloc(sizeof(char) * buf_len);
	}

	fseek(fp, line_start, SEEK_SET);
	fread(buf, sizeof(char), llen, fp);
	buf[llen] = '\0';

/* parse buffer */
	inq = indq = 0;

	i=0;
	while (isblank(buf[i]))
	  i++;

	if (buf[i] == '#')
	  goto RCF_INEXT;

	if (buf[i] <= 0x40 || (buf[i] >= 0x5b && buf[i] <= 0x5e) || buf[i] == 0x60 || buf[i] >= 0x7b) {
	  fprintf(stderr, "xdclient: syntax error in config file %s, line %d\n", conf_path, line_num);
	  goto RCF_INEXT;
	}

	var_start = i;
	var_len = 0;
	while (!isblank(buf[i]) && buf[i] != '=' && buf[i]) {
	  var_len++;
	  i++;
	}

	while(isblank(buf[i]))
	  i++;

	if (buf[i] != '=') {
	  fprintf(stderr, "xdclient: syntax error in config file %s, line %d\n", conf_path, line_num);
	  goto RCF_INEXT;
	}
	i++;

	/* read value */
	while (isblank(buf[i]))
	  i++;
	if (!buf[i])
	  goto RCF_INEXT;

	if (buf[i] == '"') {
	  indq=1;
	  i++;
	} else if (buf[i] == '\'') {
	  inq = 1;
	  i++;
	}
	val_start = i;

	val_len = llen - i;
	i = llen-1;
	while(isblank(buf[i])) {
	  i--;
	  val_len--;
	}

	if (inq || indq) {
	  if ((inq && buf[i] == '\'') || (indq && buf[i] == '"')) {
		i--;
		val_len--;
	  } else {
		fprintf(stderr, "xdclient: syntax error in config file %s, line %d: closing quote character missed\n", conf_path, line_num);
		goto RCF_INEXT;
	  }
	}
	if (val_len <= 0) {
	  fprintf(stderr, "xdclient: syntax error in config file %s, line %d\n", conf_path, line_num);
	  goto RCF_INEXT;
	}

	buf[var_start + var_len] = '\0';
	buf[val_start + val_len] = '\0';
	xdc_conf_add_arg(xdc_conf, pfmbs("cfg_arg_%s", &buf[var_start]), strdup(&buf[val_start]));
  }

  fclose(fp);
  if (buf)
	free(buf);
  free(conf_path);

  if ((nv = hlookup(xdc_conf->symtab, "cfg_arg_path", 0)) && nv->mbs_val) {
	mbs = strdup(nv->mbs_val);
	save_arg_path(xdc_conf, mbs, "cfg_arg_path");
	free(mbs);
  }

  if ((nv = hlookup(xdc_conf->symtab, "cfg_arg_add_path", 0)) && nv->mbs_val) {
	mbs = strdup(nv->mbs_val);
	save_arg_path(xdc_conf, mbs, "cfg_arg_add_path");
	free(mbs);
  }

  if ((nv = hlookup(xdc_conf->symtab, "cfg_arg_index_path", 0)) && nv->mbs_val) {
	mbs = strdup(nv->mbs_val);
	save_arg_path(xdc_conf, mbs, "cfg_arg_index_path");
	free(mbs);
  }

  if ((nv = hlookup(xdc_conf->symtab, "cfg_arg_dict_name", 0)) && nv->mbs_val) {
	mbs = strdup(nv->mbs_val);
	save_arg_dict_name(xdc_conf, mbs, "cfg_arg_dict_name");
	free(mbs);
  }

  if ((nv = hlookup(xdc_conf->symtab, "cfg_arg_lang_from", 0)) && nv->mbs_val) {
	mbs = strdup(nv->mbs_val);
	save_arg_lang(xdc_conf, mbs, "cfg", "from");
	free(mbs);
  }

  if ((nv = hlookup(xdc_conf->symtab, "cfg_arg_lang_to", 0)) && nv->mbs_val) {
	mbs = strdup(nv->mbs_val);
	save_arg_lang(xdc_conf, mbs, "cfg", "to");
	free(mbs);
  }

  if ((nv = hlookup(xdc_conf->symtab, "cfg_arg_use_colors", 0)) && nv->mbs_val) {
	i = parse_true_false(nv->mbs_val);
	if (i == 1 || i == -1)
	  nv->ival = 1;
	else
	  nv->ival = 1;
  }

  return 1;
}

void xdc_conf_read_env(xdc_conf_t *xdc_conf)
{
  char *enval;
  int ret;
  nameval_t *nv;

  if ((enval = getenv("XDC_PATH")))
	save_arg_path(xdc_conf, enval, "env_arg_path");

  if ((enval = getenv("XDC_ADD_PATH")))
	save_arg_path(xdc_conf, enval, "env_arg_add_path");

  if ((enval = getenv("XDC_INDEX_PATH")))
	save_arg_path(xdc_conf, enval, "env_arg_index_path");

  if ((enval = getenv("XDC_CONF_PATH")))
	xdc_conf_add_arg(xdc_conf, "env_arg_conf_path", abs_path_8(towcs(enval)));

  if ((enval = getenv("XDC_DICT_SELECT")))
	save_arg_dict_name(xdc_conf, enval, "env_arg_dict_name");

  if ((enval = getenv("XDC_VERBOSE")))
	xdc_conf_add_arg(xdc_conf, "env_arg_verbose", NULL);

  if ((enval = getenv("XDC_USE_COLORS"))) {
	nv = xdc_conf_add_arg(xdc_conf, "env_arg_use_colors", NULL);
	ret = parse_true_false(enval);
	if (ret == -1 || ret == 1)
	  nv->ival = 1;
	else
	  nv->ival = 0;
  }

  if ((enval = getenv("XDC_LANG_FROM")))
	save_arg_lang(xdc_conf, enval, "env", "from");

  if ((enval = getenv("XDC_LANG_TO")))
	save_arg_lang(xdc_conf, enval, "env", "to");

  if ((enval = getenv("XDC_SKIP_ABBREV")))
	xdc_conf_add_arg(xdc_conf, "env_arg_skip_abbrev", NULL);

  if ((enval = getenv("XDC_NO_SKIP_ABBREV")))
	xdc_conf_add_arg(xdc_conf, "env_arg_no_skip_abbrev", NULL);
}

int xdc_conf_read_args(xdc_conf_t *xdc_conf, int argc, char **argv)
{
  int i, j, h;
  nameval_t *nv;
  char *mbs;
  int ret;

  if (!xdc_conf || (argc > 0 && argv == NULL))
	return 0;

  for (i=1; i<argc; i++) {
	if (strncmp(argv[i], "-g", 2) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_gen_index", NULL);
	} else if (strncmp(argv[i], "--gen-index", 11) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_gen_index", NULL);
	} else if (strncmp(argv[i], "--clean-output", 14) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_clean_output", NULL);
	} else if (strncmp(argv[i], "--list-dicts", 12) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_list_dicts", NULL);
	} else if (strncmp(argv[i], "--describe", 10) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_describe", NULL);
	} else if (strncmp(argv[i], "-W", 2) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_no_warn", NULL);
	} else if (strncmp(argv[i], "--no-warn", 9) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_no_warn", NULL);
	} else if (strncmp(argv[i], "-o", 2) == 0) {
	  i++;
	  if (i >= argc)
		return 0;
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_output_dir", strdup(argv[i]));
	} else if (strncmp(argv[i], "--output-dir=", 13) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_output_dir", strdup(&argv[i][13]));
	} else if (strncmp(argv[i], "-V", 2) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_print_version", NULL);
	} else if (strncmp(argv[i], "--version", 9) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_print_version", NULL);
	} else if (strncmp(argv[i], "-h", 2) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_print_help", NULL);
	} else if (strncmp(argv[i], "--help", 6) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_print_help", NULL);
	} else if (strncmp(argv[i], "-l", 2) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_list", NULL);
	} else if (strncmp(argv[i], "--list", 6) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_list", NULL);
	} else if (strncmp(argv[i], "-v", 2) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_verbose", NULL);
	} else if (strncmp(argv[i], "--verbose", 9) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_verbose", NULL);
	} else if (strncmp(argv[i], "-C", 2) == 0) {
	  nv = xdc_conf_add_arg(xdc_conf, "cmd_arg_use_colors", NULL);
	  nv->ival = 0;
	} else if (strncmp(argv[i], "-c", 2) == 0) {
	  nv = xdc_conf_add_arg(xdc_conf, "cmd_arg_use_colors", NULL);
	  nv->ival = 1;
	} else if (strncmp(argv[i], "--use-colors=", 13) == 0) {
	  nv = xdc_conf_add_arg(xdc_conf, "cmd_arg_use_colors", NULL);
	  ret = parse_true_false(&argv[i][13]);
	  if (ret == -1 || ret == 1)
		nv->ival = 1;
	  else
		nv->ival = 0;
	} else if (strncmp(argv[i], "--conf=", 7) == 0) {
	  nv = xdc_conf_add_arg(xdc_conf, "cmd_arg_conf", abs_path_8(towcs(&argv[i][7])));
	  nv->wcs_val = mbs2wcs(&argv[i][7]);
	} else if (strncmp(argv[i], "--skip-abbrev", 13) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_skip_abbrev", NULL);
	} else if (strncmp(argv[i], "--no-skip-abbrev", 16) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_no_skip_abbrev", NULL);
	} else if (strncmp(argv[i], "-p", 2) == 0) {
	  /* dict, index, cache search path */
	  i++;
	  if (i >= argc)
		return 0;

	  if (! save_arg_path(xdc_conf, argv[i], "cmd_arg_path"))
		return 0;
	} else if (strncmp(argv[i], "--path=", 7) == 0) {
	  if (! save_arg_path(xdc_conf, &argv[i][7], "cmd_arg_path"))
		return 0;
	} else if (strncmp(argv[i], "--add-path=", 11) == 0) {
	  if (! save_arg_path(xdc_conf, &argv[i][11], "cmd_arg_add_path"))
		return 0;
	} else if (strncmp(argv[i], "-i", 2) == 0) {
	  /* index search path */
	  i++;
	  if (i >= argc)
		return 0;
	  if (! save_arg_path(xdc_conf, argv[i], "cmd_arg_index_path"))
		return 0;
	} else if (strncmp(argv[i], "--index-path=", 13) == 0) {
	  if (! save_arg_path(xdc_conf, &argv[i][13], "cmd_arg_index_path"))
		return 0;
	} else if (strncmp(argv[i], "-f", 2) == 0) {
	  /* from langs list */
	  i++;
	  if (i >= argc)
		return 0;
	  if (! save_arg_lang(xdc_conf, argv[i], "cmd", "from"))
		return 0;
	} else if (strncmp(argv[i], "--from=", 7) == 0) {
	  if (! save_arg_lang(xdc_conf, &argv[i][7], "cmd", "from"))
		return 0;
	} else if (strncmp(argv[i], "-t", 2) == 0) {
	  /* to langs list */
	  i++;
	  if (i >= argc)
		return 0;
	  if (! save_arg_lang(xdc_conf, argv[i], "cmd", "to"))
		return 0;
	} else if (strncmp(argv[i], "--to=", 5) == 0) {
	  if (! save_arg_lang(xdc_conf, &argv[i][5], "cmd", "to"))
		return 0;
	} else if (strncmp(argv[i], "-d", 2) == 0) {
	  i++;
	  if (i >= argc)
		return 0;
	  if (! save_arg_dict_name(xdc_conf, argv[i], "cmd_arg_dict_name"))
		return 0;
	} else if (strncmp(argv[i], "--dict-name=", 12) == 0) {
	  if (! save_arg_dict_name(xdc_conf, &argv[i][12], "cmd_arg_dict_name"))
		return 0;
	} else if (strncmp(argv[i], "-D", 2) == 0) {
	  i++;
	  if (i >= argc)
		return 0;
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_delete_cost", strdup(argv[i]));
	} else if (strncmp(argv[i], "--delete-cost=", 14) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_delete_cost", strdup(&argv[i][14]));
	} else if (strncmp(argv[i], "-I", 2) == 0) {
	  i++;
	  if (i >= argc)
		return 0;
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_insert_cost", strdup(argv[i]));
	} else if (strncmp(argv[i], "--insert-cost=", 14) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_insert_cost", strdup(&argv[i][14]));
	} else if (strncmp(argv[i], "-S", 2) == 0) {
	  i++;
	  if (i >= argc)
		return 0;
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_substitute_cost", strdup(argv[i]));
	} else if (strncmp(argv[i], "--substitute-cost=", 18) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_substitute_cost", strdup(&argv[i][18]));
	} else if (strncmp(argv[i], "-E", 2) == 0) {
	  i++;
	  if (i >= argc)
		return 0;
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_max_cost", strdup(argv[i]));
	} else if (strncmp(argv[i], "--max-errors=", 13) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_max_cost", strdup(&argv[i][13]));
	} else if (strncmp(argv[i], "-F", 2) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_full_search", NULL);
	} else if (strncmp(argv[i], "--full", 6) == 0) {
	  xdc_conf_add_arg(xdc_conf, "cmd_arg_full_search", NULL);
	} else if (strncmp(argv[i], "-k", 2) == 0) {
	  i++;
	  if (i >= argc)
		return 0;
	  nv = xdc_conf_add_arg(xdc_conf, "cmd_arg_key", NULL);
	  nv->wcs_val = mbs2wcs(argv[i]);
	} else {
	  if (!hlookup(xdc_conf->symtab, "cmd_arg_key", 0)) {
		if (i == argc-1) {
		  nv = xdc_conf_add_arg(xdc_conf, "cmd_arg_key", NULL);
		  nv->wcs_val = mbs2wcs(argv[i]);
		} else {
		  j = strlen(argv[i]);
		  if (argv[i][0] != '-') {
			if (! (j>1 && argv[i][1] == '-')) {
			  nv = xdc_conf_add_arg(xdc_conf, "cmd_arg_key", NULL);
			  nv->wcs_val = mbs2wcs(argv[i]);
			}
		  }
		}
	  }

	  for (j=0; argv[i][j] == '-' && argv[i][j]; j++);
	  for (h=j; argv[i][h] != '=' && argv[i][h]; h++);
	  if (j>0 && h>j) {
		mbs = strndup(&argv[i][j], h-j);
		for(j=0; mbs[j]; j++) {
		  if (mbs[j] == '-')
			mbs[j] = '_';
		}
		nv = xdc_conf_add_arg(xdc_conf, pfmbs("cmd_arg_%s", mbs), NULL);
		free(mbs);
		if (argv[i][h] && argv[i][h+1])
		  nv->mbs_val = strdup(&argv[i][h+1]);
	  }
	}
  }

  return 1;
}

nameval_t* xdc_conf_add_arg(xdc_conf_t *xdc_conf, char *arg, char *val)
{
  nameval_t *nv;

  if (!xdc_conf || !arg || !arg[0])
	return NULL;

  nv = hlookup(xdc_conf->symtab, arg, 1);
  if (nv->mbs_val)
	free(nv->mbs_val);
  if (nv->val && nv->val_free) {
	(*nv->val_free)(nv->val);
	nv->val = NULL;
	nv->val_free = NULL;
  }
  nv->mbs_val = val;
  return nv;
}

xdc_conf_t* get_xdc_conf(void)
{
  return global_xdc_conf;
}

xdc_conf_t* create_xdc_conf(int argc, char *argv[])
{
  xdc_conf_t *xdc_conf;
  nameval_t *nv;

  if (!global_xdc_conf) {
	xdc_conf = (xdc_conf_t*) malloc(sizeof(xdc_conf_t));
	global_xdc_conf = xdc_conf;

	xdc_conf_init(xdc_conf, argc, argv);
	xdc_conf_set_defaults(xdc_conf);

	if (! xdc_conf_read_args(xdc_conf, argc, argv)) {
	  fprintf(stderr, "xdclient: error: parsing arguments\n");
	  free_xdc_conf(xdc_conf);
	  global_xdc_conf = NULL;
	  return NULL;
	}

	xdc_conf_read_env(xdc_conf);

	if (! xdc_conf_file_read(xdc_conf)) {
	  free_xdc_conf(xdc_conf);
	  global_xdc_conf = NULL;
	  return NULL;
	}

  }

  if (hlookup(xdc_conf->symtab, "cmd_arg_no_warn", 0))
	xdc_conf->warn = 0;

  if (hlookup(xdc_conf->symtab, "cmd_arg_no_skip_abbrev", 0)) {
	xdc_conf->skip_abbrev = 0;
  } else if (! hlookup(xdc_conf->symtab, "cmd_arg_skip_abbrev", 0)) {
	if (hlookup(xdc_conf->symtab, "env_arg_no_skip_abbrev", 0)) {
	  xdc_conf->skip_abbrev = 0;
	} else if (! hlookup(xdc_conf->symtab, "env_arg_skip_abbrev", 0)) {
	  if (hlookup(xdc_conf->symtab, "cfg_arg_no_skip_abbrev", 0))
		xdc_conf->skip_abbrev = 0;
	}
  }

  if (hlookup(xdc_conf->symtab, "cmd_arg_lang_from", 0))
	xdc_conf->cmd_lang_from = 1;
  if (hlookup(xdc_conf->symtab, "cmd_val_lang_from_xx", 0))
	xdc_conf->cmd_lang_from_all = 1;
  if (hlookup(xdc_conf->symtab, "env_arg_lang_from", 0))
	xdc_conf->env_lang_from = 1;
  if (hlookup(xdc_conf->symtab, "env_val_lang_from_xx", 0))
	xdc_conf->env_lang_from_all = 1;
  if (hlookup(xdc_conf->symtab, "cfg_arg_lang_from", 0))
	xdc_conf->cfg_lang_from = 1;
  if (hlookup(xdc_conf->symtab, "cfg_val_lang_from_xx", 0))
	xdc_conf->cfg_lang_from_all = 1;
  if (hlookup(xdc_conf->symtab, "cmd_arg_lang_to", 0))
	xdc_conf->cmd_lang_to = 1;
  if (hlookup(xdc_conf->symtab, "cmd_val_lang_to_xx", 0))
	xdc_conf->cmd_lang_to_all = 1;
  if (hlookup(xdc_conf->symtab, "env_arg_lang_to", 0))
	xdc_conf->env_lang_to = 1;
  if (hlookup(xdc_conf->symtab, "env_val_lang_to_xx", 0))
	xdc_conf->env_lang_to_all = 1;
  if (hlookup(xdc_conf->symtab, "cfg_arg_lang_to", 0))
	xdc_conf->cfg_lang_to = 1;
  if (hlookup(xdc_conf->symtab, "cfg_val_lang_to_xx", 0))
	xdc_conf->cfg_lang_to_all = 1;
  if (hlookup(xdc_conf->symtab, "cmd_arg_dict_name", 0))
	xdc_conf->cmd_dict_name = 1;
  if (hlookup(xdc_conf->symtab, "env_arg_dict_name", 0))
	xdc_conf->env_dict_name = 1;
  if (hlookup(xdc_conf->symtab, "cfg_arg_dict_name", 0))
	xdc_conf->cfg_dict_name = 1;

  if (hlookup(xdc_conf->symtab, "cmd_arg_verbose", 0)) {
	xdc_conf->cmd_verbose = 1;
	xdc_conf->verbose = 1;
  }
  if (hlookup(xdc_conf->symtab, "env_arg_verbose", 0)) {
	xdc_conf->env_verbose = 1;
	xdc_conf->verbose = 1;
  }
  if (hlookup(xdc_conf->symtab, "cfg_arg_verbose", 0)) {
	xdc_conf->cfg_verbose = 1;
	xdc_conf->verbose = 1;
  }
  if (hlookup(xdc_conf->symtab, "cmd_arg_no_verbose", 0)) {
	xdc_conf->cmd_no_verbose= 1;
	xdc_conf->verbose = 0;
  }

  if (hlookup(xdc_conf->symtab, "cmd_arg_list", 0))
	xdc_conf->cmd_list = 1;

  if (hlookup(xdc_conf->symtab, "cmd_arg_list_dicts", 0))
	xdc_conf->list_dicts = 1;

  if (hlookup(xdc_conf->symtab, "cmd_arg_describe", 0))
	xdc_conf->describe = 1;

  if ((nv = hlookup(xdc_conf->symtab, "cfg_arg_use_colors", 0)))
	xdc_conf->use_colors = nv->ival;
  if ((nv = hlookup(xdc_conf->symtab, "env_arg_use_colors", 0)))
	xdc_conf->use_colors = nv->ival;
  if ((nv = hlookup(xdc_conf->symtab, "cmd_arg_use_colors", 0)))
	xdc_conf->use_colors = nv->ival;

  return global_xdc_conf;
}

void free_xdc_conf(xdc_conf_t *xdc_conf)
{
  if (! xdc_conf)
	return;

  htable_free(xdc_conf->symtab, 0);
  whtable_free(xdc_conf->wsymtab, 0);

  free(xdc_conf);
}

int save_arg_path(xdc_conf_t *xdc_conf, char *arg, char *name)
{
  nameval_t *nv;
  mbs_ll_t *mbsl, *mbsl2;
  char *mbs;

  if (!xdc_conf || !arg || !arg[0] || !name || !name[0])
	return 0;

  nv = xdc_conf_add_arg(xdc_conf, name, strdup(arg));
  mbsl = mbsl2 = mbs_split(arg, ':');
  while(mbsl2) {
	mbs = mbsl2->mbs;
	mbsl2->mbs = abs_path_8(towcs(mbs));
	free(mbs);
	mbsl2 = mbsl2->next;
  }
  nv->val = (void*) mbsl;
  nv->val_type = Tmbsll;
  nv->val_free = (void (*)(void*)) mbsl_free;

  return 1;
}

int save_arg_lang(xdc_conf_t *xdc_conf, char *arg, char *src, char *fromto)
{
  nameval_t *nv;
  mbs_ll_t *mbsl, *mbsl2;
  int i;

  if (!xdc_conf || !arg || !arg[0] || !fromto || !fromto[0])
	return 0;

  xdc_conf_add_arg(xdc_conf, pfmbs("%s_arg_lang_%s", src, fromto), NULL);
  mbsl = mbs_split(arg, ',');
  while(mbsl) {
	if (strlen(mbsl->mbs) >= 2) {
	  if (reg_test_8(mbsl->mbs, L"all", RE_IGN_CASE | RE_COMP_SPACES | RE_SKIP_NONALPHA | RE_SKIP_START_SPC | RE_PREFIX_DASH, NULL)) {
		nv = xdc_conf_add_arg(xdc_conf, pfmbs("%s_val_lang_%s_%s", src, fromto, "xx"), NULL);
	  } else {
		for (i=0; i < 2; i++)
		  mbsl->mbs[i] = tolower(mbsl->mbs[i]);
		mbsl->mbs[2] = '\0';
		nv = xdc_conf_add_arg(xdc_conf, pfmbs("%s_val_lang_%s_%s", src, fromto, mbsl->mbs), NULL);
	  }
	  nv->ival = 1;
	}
	mbsl2 = mbsl->next;
	free(mbsl->mbs);
	free(mbsl);
	mbsl = mbsl2;
  }

  return 1;
}

int save_arg_dict_name(xdc_conf_t *xdc_conf, char *arg, char *hash_name)
{
  nameval_t *nv;
  wcs_ll_t *wcsl, *wcsl2;
  int i;

  if (!xdc_conf || !arg || !arg[0] || !hash_name || !hash_name[0])
	return 0;

  nv = xdc_conf_add_arg(xdc_conf, hash_name, NULL);
  wcsl = wcsl2 = wcs_split(towcs(arg), L',');
  while(wcsl2) {
	for (i=0; wcsl2->wcs[i]; i++)
	  wcsl2->wcs[i] = towlower(wcsl2->wcs[i]);
	wcsl2 = wcsl2->next;
  }
  nv->val = (void *) wcsl;
  nv->val_type = Twcsll;
  nv->val_free = (void (*)(void*)) wcsl_free;

  return 1;
}

int parse_true_false(char *str)
{
  wchar_t *val;

  if (!str)
	return -1;

  val = pfwcs(L"%s", str);

  if (wcscasecmp(val, L"true") == 0)
	return 1;
  if (wcscasecmp(val, L"yes") == 0)
	return 1;
  if (wcscasecmp(val, L"y") == 0)
	return 1;
  if (wcscasecmp(val, L"on") == 0)
	return 1;
  if (wcscasecmp(val, L"1") == 0)
	return 1;

  if (wcscasecmp(val, L"false") == 0)
	return 0;
  if (wcscasecmp(val, L"no") == 0)
	return 0;
  if (wcscasecmp(val, L"n") == 0)
	return 0;
  if (wcscasecmp(val, L"off") == 0)
	return 0;
  if (wcscasecmp(val, L"0") == 0)
	return 0;

  return -1;
}

