/*
    Copyright (C) 2012 Oleksiy Chernyavskyy

    This file is part of XDClient.

    XDClient is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    XDClient is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with XDClient.  If not, see <http://www.gnu.org/licenses/>.
*/

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <wchar.h>
#ifdef TARGET_FREEBSD
#include <sys/types.h>
#else
#include <stdint.h>
#include <wctype.h>
#endif
#include <dirent.h>
#include <string.h>
#include <ctype.h>
#include "xdxf.h"
#include "xml_utf8.h"
#include "common.h"
#include "file.h"
#include "xdc.h"
#include "hash.h"

wchar_t sbuf[BUFSZ];

int cmpkeys(const void *p1, const void *p2)
{
  key_ll_t **key1, **key2;

  key1 = (key_ll_t**) p1;
  key2 = (key_ll_t**) p2;
  return wcscmp((*key1)->key, (*key2)->key);
}

void kl_base_opt_merge(key_ll_t **keys, int nkeys, key_ll_t *optkeys)
{
  key_ll_t *_key;
  int start, end, j, cmpret;
  key_ll_t *_key_next;

  if (!optkeys || !keys || nkeys <= 0)
	return;

  _key = optkeys;
  while(_key) {
	_key_next = _key->next;
	start = 0;
	end = nkeys;

	for (j = (end - start)/2; ; j = (end - start)/2) {
	  cmpret = wcscmp(_key->key, keys[start+j]->key);
	  if (cmpret > 0) {
		start += j;
	  } else if (cmpret < 0) {
		end -= j;
	  }  else {
		break;
	  }
	  if (j == 0)
		break;
	}
	if (cmpret < 0 && start > 0) {
	  start--;
	} else if (cmpret == 0 && start+j < nkeys-1) {
	  do {
		start++;
		cmpret = wcscmp(_key->key, keys[start+j]->key);
	  } while (cmpret == 0 && start+j < nkeys-1);

	  if (cmpret < 0)
		start--;
	}

	_key->next = keys[start+j]->next;
	if (_key->next)
	  _key->next->prev = _key;
	_key->prev = keys[start+j];
	keys[start+j]->next = _key;

	sort_keys(keys[start+j]);
	_key = _key_next;
  }
}

int count_opt_keys(keypart_t *key_parts, int kp_len)
{
  int i, nopt;

  if (!key_parts)
	return 0;

  for (i=0, nopt=0; i<kp_len && key_parts[i].str; i++)
	if (key_parts[i].is_opt) nopt++;

  return nopt;
}

int count_keys(keypart_t *key_parts, int kp_len)
{
  int i;

  for (i=0; i<kp_len && key_parts[i].str; i++);

  return i;
}


key_ll_t* fill_opts(keypart_t *key_parts, int kp_len, DTag *tag_ar, key_ll_t *optkeys, int ktag_index)
{
  unsigned opt_key_num;
  unsigned opt_mask;
  key_ll_t *optkey_first;
  key_ll_t *optkey_last;
  key_ll_t *_key;
  wchar_t *key_str;
  int nopt;
  int i;

  PRDEBUG("enter")

  nopt = count_opt_keys(key_parts, kp_len);

  if (nopt>0) {
	opt_key_num = 1;
	for(i=0; i<nopt && i<=MAXOPTS; i++) {
	  opt_key_num = opt_key_num * 2; 
	}
  } else {
	opt_key_num = 0;
  }

  optkey_first = optkey_last = NULL;
  for (opt_mask=1; opt_mask < opt_key_num; opt_mask++) {
	key_str = get_key_by_mask(key_parts, kp_len, opt_mask);
	_key = (key_ll_t*) malloc(sizeof(key_ll_t));
	_key->key = key_str;
	_key->ktag_index = ktag_index;
	_key->kmask = opt_mask;
	_key->tag = tag_ar;
	_key->next = NULL;
	if (optkey_last) {
	  optkey_last->next = _key;
	  _key->prev = optkey_last;
	  optkey_last = _key;
	} else {
	  _key->prev = NULL;
	  optkey_first = optkey_last = _key;
	}
  }

  if (optkeys) {
	if (optkey_first) {
	  _key = optkeys;
	  while (_key->next)
		_key = _key->next;
	  _key->next = optkey_first;
	  optkey_first->prev = _key;
	}
  } else {
	optkeys = optkey_first;
  }

  PRDEBUG("exit")

  return optkeys;
}

wchar_t* get_key_by_mask(keypart_t *key_parts, int kp_len, unsigned opt_mask)
{
  int h, k, i;
  unsigned oflag;
  int npart;
  int nopt;
  unsigned opt_key_num;
  wchar_t *str_ret;

  if (!key_parts || kp_len == 0)
	return NULL;

  nopt = count_opt_keys(key_parts, kp_len);
  npart = count_keys(key_parts, kp_len);

  if (nopt>0) {
	opt_key_num = 1;
	for(i=0; i<nopt && i<=MAXOPTS; i++)
	  opt_key_num = opt_key_num * 2; 
  } else {
	opt_key_num = 0;
  }

  if (opt_mask > 0 && opt_mask >= opt_key_num)
	return NULL;

  for (h=0, k=0, oflag = 1; oflag <= MAXOPT && k < npart && h < BUFSZ-1; oflag = oflag << 1, k++) {
	while (! key_parts[k].is_opt && k < npart) {
	  for(i=0; key_parts[k].str[i] != L'\0' && h < BUFSZ-1; i++, h++)
		sbuf[h] = key_parts[k].str[i];
	  k++;
	}
	if (opt_mask & oflag) {
	  if (k<npart) {
		for(i=0; key_parts[k].str[i] != L'\0' && h < BUFSZ-1; i++, h++)
		  sbuf[h] = key_parts[k].str[i];
	  } else {
		fwprintf(stderr, L"xdc: error: %s:%d: key_part index >= npart\n", __FUNCTION__, __LINE__);
	  }
	}
  }
  sbuf[h] = L'\0';

  //  str_ret = wcsdup(sbuf);
  str_ret = (wchar_t*) malloc(sizeof(wchar_t) * (h+1));
  wcscpy(str_ret, sbuf);

  return str_ret;
}

key_ll_t* sort_keys(key_ll_t *key_start)
{
  key_ll_t *sorted_start;
  key_ll_t *sorted_end;
  key_ll_t *_key, *_key2, *_key_next;

  if (!key_start)
	return NULL;

  while (key_start->prev)
	key_start = key_start->prev;

  _key = key_start->next;
  sorted_start = key_start;
  sorted_start->next = NULL;
  sorted_start->prev = NULL;
  sorted_end = sorted_start;

  while(_key) {
	_key_next = _key->next;
	_key->prev = NULL;
	_key->next = NULL;
	_key2 = sorted_end;
	while (1) {
	  if (wcscmp(_key->key, _key2->key) > 0) {
		_key->next = _key2->next;
		_key->prev = _key2;
		_key2->next = _key;
		if (_key->next)
		  _key->next->prev = _key;
		else
		  sorted_end = _key;
		break;
	  }
	  if (! _key2->prev) {
		_key2->prev = _key;
		_key->next = _key2;
		break;
	  }
	  _key2 = _key2->prev;
	}
	_key = _key_next;
  }
  sorted_start = sorted_end;
  while (sorted_start->prev)
	sorted_start = sorted_start->prev;

  return sorted_start;
}

int crop_nonalpha(key_ll_t **keys, int klen)
{
  int j, i, h;
  wchar_t *key;

  if (!keys || klen <=0)
	return 0;

  for (j=0; j<klen && keys[j]; j++) {
	key = keys[j]->key;
	if (key) {
	  for (i=0, h=0; key[h] != L'\0'; h++) {
/*		if (key[h] != L'\'') { */
		if (iswalnum(key[h]) || iswspace(key[h])) {
		  if (i<h)
			key[i] = key[h];
		  i++;
		}
	  }
	  key[i] = L'\0';
	} else {
	  fwprintf(stderr, L"xdc: error: %s:%d\n", __FUNCTION__, __LINE__);
	}
  }
  return 1;
}

int crop_nonalpha2(key_ll_t *key_start)
{
  key_ll_t *_key;
  int i, h;
  wchar_t *key_str;

  if (!key_start)
	return 0;

  _key = key_start;
  while (_key->prev)
	_key = _key->prev;

  while((_key = _key->next)) {
	key_str = _key->key;
	if (key_str) {
	  for (i=0, h=0; key_str[h] != L'\0'; h++) {
/*		if (key_str[h] != L'\'') { */
		if (iswalnum(key_str[h]) || iswspace(key_str[h])) {
		  if (i<h)
			key_str[i] = key_str[h];
		  i++;
		}
	  }
	  key_str[i] = L'\0';
	} else {
	  fwprintf(stderr, L"xdc: error: %s:%d\n", __FUNCTION__, __LINE__);
	}
  }
  return 1;
}


int kll_shrink_norm_spaces(key_ll_t **keys, int klen)
{
  int j, i, h;
  wchar_t *key;

  if (!keys || klen <= 0)
	return 0;

  for (j=0; j<klen && keys[j]; j++) {
	key = keys[j]->key;
	if (key) {
	  for (i=0, h=0; key[h] != L'\0'; h++) {
		if (iswspace(key[h])) {
		  if (i>0)
			key[i++] = L' ';
		  while(iswspace(key[++h]));
		  h--;
		} else {
		  key[i++] = key[h];
		}
	  }
	  i-=1;
	  while (i >= 0 && iswspace(key[i]))
		i--;
	  i++;
	  key[i] = L'\0';
	} else {
	  fwprintf(stderr, L"xdc: error: %s:%d\n", __FUNCTION__, __LINE__);
	}
  }
  return 1;
}

int kll_shrink_norm_spaces2(key_ll_t *key_start)
{
  key_ll_t *_key;
  int i, h;
  wchar_t *key_str;

  if (!key_start)
	return 0;

  _key = key_start;
  while (_key->prev)
	_key = _key->prev;

  while((_key = _key->next)) {
	key_str = _key->key;
	if (key_str) {
	  for (i=0, h=0; key_str[h] != L'\0'; h++) {
		if (iswspace(key_str[h])) {
		  if (i>0)
			key_str[i++] = L' ';
		  while(iswspace(key_str[++h]));
		  h--;
		} else {
		  key_str[i++] = key_str[h];
		}
	  }
	  i-=1;
	  while (i >= 0 && iswspace(key_str[i]))
		i--;
	  i++;
	  key_str[i] = L'\0';
	} else {
	  fwprintf(stderr, L"xdc: error: %s:%d\n", __FUNCTION__, __LINE__);
	}
  }
  return 1;
}

int dup_remove(key_ll_t **keys, int *klen)
{
  int i, j;

  if (!keys || !klen || *klen <=0)
	return 0;

  for (j=0, i=1; i<*klen && keys[i]; i++) {
	if (wcscmp(keys[j]->key, keys[i]->key) != 0) {
	  j++;
	  if (j<i)
		keys[j] = keys[i];
	} else {
	  if (keys[i]->key)
		free(keys[i]->key);
	  free(keys[i]);
	}
  }
  *klen = j+1;
  return 1;
}

int dup_remove2(key_ll_t *key_start)
{
  key_ll_t *_key;
  key_ll_t *_key2;

  if (!key_start)
	return 0;

  _key = key_start;
  while (_key->prev)
	_key = _key->prev;

  while((_key2 = _key->next)) {
	if (wcscmp(_key->key, _key2->key) != 0) {
	  _key = _key2;
	} else {
	  _key->next = _key2->next;
	  if (_key->next)
		_key->next->prev = _key;
	  if (_key2->key)
		free(_key2->key);
	  free(_key2);
	}
  }
  return 1;
}

int keys_tolower(key_ll_t **keys, int nkeys)
{
  int i, j;

  if (!keys || nkeys <= 0)
	return 0;

  for (i=0; i<nkeys; i++) {
	for (j=0; keys[i]->key[j] != L'\0'; j++)
	  keys[i]->key[j] = towlower(keys[i]->key[j]);
  }
  return 1;
}

int keys_tolower2(key_ll_t *key_start)
{

  key_ll_t *_key;
  int j;

  if (!key_start)
	return 0;

  _key = key_start;
  while (_key->prev)
	_key = _key->prev;

  while (_key) {
	for (j=0; _key->key[j] != L'\0'; j++)
	  _key->key[j] = towlower(_key->key[j]);
	_key = _key->next;
  }
  return 1;
}

int start_spaces_remove(key_ll_t **keys, int nkeys, int sorted)
{
  int i, j, h;

  if (!keys || nkeys <= 0)
	return 0;

  for (i=0; i<nkeys; i++) {
	for (j=0; iswspace(keys[i]->key[j]) && keys[i]->key[j] != L'\0'; j++) ;

	if (!j && sorted) {
	  break;
	} else if (j) {
	  for (h=0; keys[i]->key[j] != L'\0'; j++, h++)
		keys[i]->key[h] = keys[i]->key[j];
	  keys[i]->key[h] = L'\0';
	}
  }
  return 1;
}

int start_spaces_remove2(key_ll_t *key_start, int sorted)
{
  key_ll_t *_key;
  int h, j;

  if (!key_start)
	return 0;

  _key = key_start;
  while (_key->prev)
	_key = _key->prev;

  while (_key) {
	for (j=0; iswspace(_key->key[j]) && _key->key[j] != L'\0'; j++) ;

	if (!j && sorted) {
	  break;
	} else if (j) {
	  for (h=0; _key->key[j] != L'\0'; j++, h++)
		_key->key[h] = _key->key[j];
	  _key->key[h] = L'\0';
	}
	_key = _key->next;
  }
  return 1;
}

DTag* read_xdxf(char *path, long shift, long size, FILE *dfp)
{
  FILE *fp;
  int i;
  wchar_t *buf;
  DTag *tag;
  wint_t wc;
  long fsize;
  long mb_len;
  long nbytes;
  long wc_num;

  PRDEBUG("enter")

  if ((!path || !path[0]) && !dfp)
	return NULL;

  dfp = NULL;
  if (dfp)
	fp = dfp;
  else
	fp = fopen(path, "r");

  if (fp) {
	if (size == 0) {
	  wc_num = 0;
	  while((wc = fgetwc(fp)) != WEOF) wc_num++;
	  nbytes = file_size(path);
	  rewind(fp);
	} else {
	  fsize = file_size(path);
	  wc_num = fsize;
	  if (fsize > shift) {
		if (shift + size > fsize)
		  size = fsize - shift;
		fseek(fp, shift, SEEK_SET);
		wc_num = size;
		nbytes = size;
	  } else {
		fwprintf(stderr, L"xdc: error: read_xdxf: file size <= shift\n");
		fwprintf(stderr, L"     file: %s\n", path);
		if (!dfp)
		  fclose(fp);
		return NULL;
	  }
	}
	buf = (wchar_t*) malloc(sizeof(wchar_t) * (wc_num+1));
	for (i=0, mb_len=0; i < wc_num && mb_len < nbytes; i++) {
	  wc = fgetwc(fp);
	  buf[i] = wc;
	  mb_len += wc_mb_len(wc);
	}
	buf[i] = L'\0';
	tag = dexml(buf, XML_SAVE_SPC_BODY);
	if (!tag)
	  free(buf);
	if (!dfp)
	  fclose(fp);
	PRDEBUG("exit")
	return tag;
  } else {
	fwprintf(stderr, L"xdc: error: could not open file %s\n", path);
	return NULL;
  }

  return NULL;
}

int opt_mask_len(void)
{
#if defined(OPTS_8)
  return sizeof(uint8_t);
#elif defined(OPTS_16)
  return sizeof(uint16_t);
#elif defined(OPTS_32)
  return sizeof(uint32_t);
#else
  return 0;
#endif
}

int ktag_index_len(void)
{
#if defined(KTAG_INDEX_16)
  return sizeof(uint16_t);
#else
  return sizeof(uint32_t);
#endif
}

int find_xdxf(char *path_in, xdcdb_t **db_ret, unsigned flags)
{
  int ret;
  DIR *dirp;
  struct dirent *de;
  char path[MAXPATH];
  char _path_in[MAXPATH];
  char fname[MAXPATH];
  xdcdb_t *db_last = NULL;
  xdcdb_t *_db_ret;
  xdcdb_t *_db;

  strncpy(_path_in, path_in, MAXPATH-1);
  _path_in[MAXPATH-1] = '\0';
  chomp_path(_path_in);

  if ((ret = is_xdxf(_path_in))) {
	*db_ret = get_db_stat(_path_in, flags);
	return ret;
  }

  if (! is_dir(_path_in)) {
	*db_ret = NULL;
	return 0;
  }

  dirp = opendir(_path_in);
  if (! dirp) {
	fprintf(stderr, "xdc: warning: could not open %s\n", _path_in);
	*db_ret = NULL;
	return 0;
  }

  while((de = readdir(dirp)) != NULL) {
	if (de->d_name[0] == '.') {
#ifdef _DIRENT_HAVE_D_NAMLEN
	  if (de->d_namlen == 1)
		continue;
	  if (de->d_name[1] == '.' && de->d_namlen == 2)
		continue;
#else
	  if (strlen(de->d_name) == 1)
		continue;
	  if (de->d_name[1] == '.' && strlen(de->d_name) == 2)
		continue;
#endif
	}
#ifdef _DIRENT_HAVE_D_NAMLEN
	strncpy(fname, de->d_name, de->d_namlen);
	fname[de->d_namlen] = '\0';
#else
	strcpy(fname, de->d_name);
#endif
	snprintf(path, MAXPATH, "%s/%s", _path_in, fname);
	ret = find_xdxf(path, &_db_ret, flags);
	switch(ret) {
	  case 1:
	  case 2:
		_db = get_db_stat(path, flags);
		_db->next = db_last;
		db_last = _db;
		break;
	  case 3:
		if (db_last) {
		  _db = db_last;
		  while (_db->next)
			_db = _db->next;
		  _db->next = _db_ret;
		} else {
		  db_last = _db_ret;
		}
		break;
	  default:
		break;
	}
  }

  closedir(dirp);

  if (! db_last) {
	*db_ret = NULL;
	return 0;
  }

  *db_ret = db_last;
  return 3;
}

int get_lang(xdc_conf_t *xdc_conf, fstat_lt *dict, char **lang_from_ret, char **lang_to_ret)
{
  index_lt *index;
  nameval_t *nv;
  fstat_lt *flist;
  char *lang_from, *lang_to;
  DTag *tag_root, *tag;
  DAttr *attr;
  wchar_t *wcs;
  int i;

  lang_from = lang_to = NULL;

  if (!xdc_conf || !dict || !dict->path || (!lang_from_ret && !lang_to_ret))
	return 0;

  if (lang_from_ret)
	*lang_from_ret = NULL;
  if (lang_to_ret)
	*lang_to_ret = NULL;

  index = NULL;
  if (dict->md5_part) {
	if ((nv = hlookup(xdc_conf->symtab, pfmbs("index:dict_md5_part:%s", dict->md5_part), 0))) {
	  if ((flist = (fstat_lt*) nv->val)) {
		if (!flist->udata)
		  load_index(flist);
		index = (index_lt*) flist->udata;
	  }
	}
  }

  if (!index && dict->md5_full) {
	if ((nv = hlookup(xdc_conf->symtab, pfmbs("index:dict_md5_full:%s", dict->md5_full), 0))) {
	  if ((flist = (fstat_lt*) nv->val)) {
		if (!flist->udata)
		  load_index(flist);
		index = (index_lt*) flist->udata;
	  }
	}
  }

  if (index && (index->set_lang_from || index->set_lang_to)) {
	tag_root = fseek_tag(dict->path, NULL, L"xdxf", 0);
	if (!tag_root)
	  return 0;
	tag = tag_seek(NULL, tag_root, L"xdxf", S_ALL);
	if (!tag)
	  return 0;

	attr = tag->attributes;
	while (attr && (!lang_from || !lang_to)) {
	  if (!wcsncmp(attr->name, L"lang_from", attr->nlen) && index->set_lang_from && lang_from_ret) {
		wcs = wcsndup(attr->value, attr->vlen);
		lang_from = wcs2mbs(wcs);
		free(wcs);
	  } else if (!wcsncmp(attr->name, L"lang_to", attr->nlen) && index->set_lang_to && lang_to_ret) {
		wcs = wcsndup(attr->value, attr->vlen);
		lang_to = wcs2mbs(wcs);
		free(wcs);
	  }
	  attr = attr->next;
	}
	tag_free(tag_root);
  }

  if (lang_from_ret && lang_to_ret && !lang_from && !lang_to)
	get_lang_in_path(dict->path, &lang_from, &lang_to);
  else if (lang_from_ret && !lang_from)
	get_lang_in_path(dict->path, &lang_from, NULL);
  else if (lang_to_ret && !lang_to)
	get_lang_in_path(dict->path, NULL, &lang_to);

  if (lang_from) {
	mbs_pref_crop(lang_from, ' ');
	mbs_pref_crop(lang_from, '\t');
	mbs_pref_crop(lang_from, '\n');
	mbs_sufx_crop(lang_from, ' ');
	mbs_sufx_crop(lang_from, '\t');
	mbs_sufx_crop(lang_from, '\n');
	for (i=0; i<3 && i <strlen(lang_from); i++)
	  lang_from[i] = tolower(lang_from[i]);
	lang_from[i] = '\0';
  }

  if (lang_to) {
	mbs_pref_crop(lang_to, ' ');
	mbs_pref_crop(lang_to, '\t');
	mbs_pref_crop(lang_to, '\n');
	mbs_sufx_crop(lang_to, ' ');
	mbs_sufx_crop(lang_to, '\t');
	mbs_sufx_crop(lang_to, '\n');
	for (i=0; i<3 && i <strlen(lang_to); i++)
	  lang_to[i] = tolower(lang_to[i]);
	lang_to[i] = '\0';
  }

  if (lang_from_ret)
	*lang_from_ret = lang_from;
  else if (lang_from)
	free(lang_from);

  if (lang_to_ret)
	*lang_to_ret = lang_to;
  else if (lang_to)
	free(lang_to);

  return 1;
}

int get_full_name(xdc_conf_t *xdc_conf, fstat_lt *dict, wchar_t **name_ret)
{
  index_lt *index;
  nameval_t *nv;
  fstat_lt *flist;
  wchar_t *name;
  DTag *tag, *tag_root;
  wchar_t *wcs;


  if (!xdc_conf || !dict || !dict->path || !name_ret)
	return 0;

  *name_ret = NULL;
  name = NULL;
  index = NULL;

  if (dict->md5_part) {
	if ((nv = hlookup(xdc_conf->symtab, pfmbs("index:dict_md5_part:%s", dict->md5_part), 0))) {
	  if ((flist = (fstat_lt*) nv->val)) {
		if (!flist->udata)
		  load_index(flist);
		index = (index_lt*) flist->udata;
	  }
	}
  }

  if (!index && dict->md5_full) {
	if ((nv = hlookup(xdc_conf->symtab, pfmbs("index:dict_md5_full:%s", dict->md5_full), 0))) {
	  if ((flist = (fstat_lt*) nv->val)) {
		if (!flist->udata)
		  load_index(flist);
		index = (index_lt*) flist->udata;
	  }
	}
  }

  if (index && index->full_name_pos) {
	tag = read_xdxf(dict->path, index->full_name_pos, index->full_name_len, NULL);
	if (!tag)
	  return 0;
	if (tag->body && tag->body->type == DT_Value) {
	  wcs = wcsndup(tag->body->value, tag->body->vlen);
	  wcs_pref_crop(wcs, L' ');
	  wcs_pref_crop(wcs, L'\t');
	  wcs_pref_crop(wcs, L'\n');
	  wcs_sufx_crop(wcs, L' ');
	  wcs_sufx_crop(wcs, L'\t');
	  wcs_sufx_crop(wcs, L'\n');
	  if (wcslen(wcs) == 0) {
		free(wcs);
		wcs = NULL;
	  }
	  name = wcs;
	}
	tag_free(tag);
  } else {
	tag_root = fseek_tag(dict->path, L"full_name", NULL, 0);
	if (!tag_root)
	  return 0;
	tag = tag_seek(NULL, tag_root, L"full_name", S_ALL);
	if (tag && tag->body && tag->body->type == DT_Value) {
	  wcs = wcsndup(tag->body->value, tag->body->vlen);
	  wcs_pref_crop(wcs, L' ');
	  wcs_pref_crop(wcs, L'\t');
	  wcs_pref_crop(wcs, L'\n');
	  wcs_sufx_crop(wcs, L' ');
	  wcs_sufx_crop(wcs, L'\t');
	  wcs_sufx_crop(wcs, L'\n');
	  if (wcslen(wcs) == 0) {
		free(wcs);
		wcs = NULL;
	  }
	  name = wcs;
	}
	tag_free(tag_root);
  }

  if (!name)
	name = get_name_in_path(dict->path);

  *name_ret = name;

  return 1;
}

wchar_t* get_name_in_path(char *path)
{
  int i, j, h;
  wchar_t *name_ret;
  wchar_t *wpath;
  
  if (!path)
	return NULL;

  wpath = mbs2wcs(path);
  if (!wpath)
	return NULL;

  name_ret = NULL;
  for (i=0; wpath[i] != L'\0'; i++);
  for (; i >= 0 && wpath[i] != L'/'; i--);
  for (j=i-1; j >= 0 && wpath[j] != L'/'; j--);
  j++;
  if (j<i) {
	name_ret = (wchar_t*) malloc(sizeof(wchar_t) * (i-j+1));
	for (h=0; j<i; j++, h++)
	  name_ret[h] = wpath[j];
	name_ret[h] = L'\0';
  }
  return name_ret;
}

int get_lang_in_path(char *path, char **lang_from, char **lang_to)
{
  int i, j, h;
  char *lfrom, *lto;
  
  if (!path || (!lang_from && !lang_to))
	return 0;

  if (lang_from)
	*lang_from = NULL;
  if (lang_to)
	*lang_to = NULL;

  lfrom = lto = NULL;
  for (i=0; path[i] != '\0'; i++);
  for (; i >= 0 && path[i] != '/'; i--);
  for (i--; i >= 0 && path[i] != '/'; i--);
  for (j=i-1; j >=0 && path[j] != '/'; j--);
  j++;
  if (i-j == 4) {
	lfrom = (char*) malloc(sizeof(char) *3);
	lto = (char*) malloc(sizeof(char) *3);
	for (h=0; h<2; h++) {
	  lfrom[h] = tolower(path[j+h]);
	  lto[h] = tolower(path[j+h+2]);
	}
	lfrom[2] = lto[2] = '\0';
  }

  if (lang_from)
	*lang_from = lfrom;
  else
	free(lfrom);

  if (lang_to)
	*lang_to = lto;
  else
	free(lto);

  return 1;
}

int get_description(xdc_conf_t *xdc_conf, fstat_lt *dict, wchar_t **descr_ret)
{
  index_lt *index;
  nameval_t *nv;
  fstat_lt *flist;
  wchar_t *descr;
  DTag *tag, *tag_root;
  wchar_t *wcs;


  if (!xdc_conf || !dict || !dict->path || !descr_ret)
	return 0;

  *descr_ret = NULL;
  descr = NULL;
  index = NULL;

  if (dict->md5_part) {
	if ((nv = hlookup(xdc_conf->symtab, pfmbs("index:dict_md5_part:%s", dict->md5_part), 0))) {
	  if ((flist = (fstat_lt*) nv->val)) {
		if (!flist->udata)
		  load_index(flist);
		index = (index_lt*) flist->udata;
	  }
	}
  }

  if (!index && dict->md5_full) {
	if ((nv = hlookup(xdc_conf->symtab, pfmbs("index:dict_md5_full:%s", dict->md5_full), 0))) {
	  if ((flist = (fstat_lt*) nv->val)) {
		if (!flist->udata)
		  load_index(flist);
		index = (index_lt*) flist->udata;
	  }
	}
  }

  if (index && index->descr_pos) {
	tag = read_xdxf(dict->path, index->descr_pos, index->descr_len, NULL);
	if (!tag)
	  return 0;
	if (tag->body && tag->body->type == DT_Value) {
	  wcs = wcsndup(tag->body->value, tag->body->vlen);
	  wcs_pref_crop(wcs, L' ');
	  wcs_pref_crop(wcs, L'\t');
	  wcs_pref_crop(wcs, L'\n');
	  wcs_sufx_crop(wcs, L' ');
	  wcs_sufx_crop(wcs, L'\t');
	  wcs_sufx_crop(wcs, L'\n');
	  if (wcslen(wcs) == 0) {
		free(wcs);
		wcs = NULL;
	  }
	  descr = wcs;
	}
	tag_free(tag);
  } else {
	tag_root = fseek_tag(dict->path, L"description", NULL, 0);
	if (!tag_root)
	  return 0;
	tag = tag_seek(NULL, tag_root, L"description", S_ALL);
	if (tag && tag->body && tag->body->type == DT_Value) {
	  wcs = wcsndup(tag->body->value, tag->body->vlen);
	  wcs_pref_crop(wcs, L' ');
	  wcs_pref_crop(wcs, L'\t');
	  wcs_pref_crop(wcs, L'\n');
	  wcs_sufx_crop(wcs, L' ');
	  wcs_sufx_crop(wcs, L'\t');
	  wcs_sufx_crop(wcs, L'\n');
	  if (wcslen(wcs) == 0) {
		free(wcs);
		wcs = NULL;
	  }
	  descr = wcs;
	}
	tag_free(tag_root);
  }

  *descr_ret = descr;

  return 1;
}

