/*
 * yauap - A simple commandline frontend for GStreamer 
 * Copyright (c) 2006 Sascha Sommer <ssommer@suse.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 */


#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <gst/gst.h>

#include "yauap.h"



/* private data */
typedef struct yauap_priv_s {
    GMainLoop *loop;
    GstElement *play;   /* the playback pipeline */
    GstTagList* tag_list;

    GstElement *src;    /* the source element */
    GstElement *decode; /* the decoder element */
    GstElement *volume;
    GstElement *audio;  /* audio chain (format conversion, resampling, volume, output sink) */
    int rate;
    int channels;

    int use_scope;
#define SAMPLE_BUFFER_SIZE (SCOPE_SIZE * 1000)
    unsigned char sample_buffer[SAMPLE_BUFFER_SIZE];
    unsigned int write_pos;
    unsigned int read_pos;
} yauap_priv_t;

/* send signals to the frontends */
static void signal_frontend(player_t* player,unsigned int signal,char* message){
    yauap_frontend_t* frontend;
    int i=0;
    while((frontend = player->frontends[i])){
        if(frontend->signal_cb)
           frontend->signal_cb(frontend,signal,message);
        ++i;
    }
}


/* functions that interact with gstreamer they are used by the dbus service*/

/* quit the player */
static int player_quit(player_t* player){
    yauap_priv_t* priv = player->yauap_priv;
    g_main_loop_quit(priv->loop);
    return FALSE;
}

/* pause / unpause */
static int player_pause(player_t* player){
    yauap_priv_t* priv = player->yauap_priv;
    GstState state;
    if(!gst_element_get_state(priv->play, &state, NULL, 0)){
        printf("error: player_pause: gst_element_get_state failed\n");
        return FALSE;
    }
    if(state == GST_STATE_PLAYING){
        gst_element_set_state(priv->play,GST_STATE_PAUSED);
        printf("pause                                      \n");
    }else
        gst_element_set_state(priv->play,GST_STATE_PLAYING);
    return TRUE;
}

/* return time length in ms */
static unsigned int player_get_time_length(player_t* player){
    yauap_priv_t* priv = player->yauap_priv;
    GstFormat fmt = GST_FORMAT_TIME;
    gint64 len = 0;
    gst_element_query_duration(priv->play, &fmt, &len);
    return len / GST_MSECOND;
}

/* return current position in ms */
static unsigned int player_get_time_position(player_t* player){
    yauap_priv_t* priv = player->yauap_priv;
    GstFormat fmt = GST_FORMAT_TIME;
    gint64 pos = 0;
    gst_element_query_position(priv->play, &fmt, &pos);
    return pos / GST_MSECOND;
}

/* stop playback */
static int player_stop(player_t* player){
    yauap_priv_t* priv = player->yauap_priv;
    /* stop playback */
    gst_element_set_state(priv->play, GST_STATE_NULL);
    if(priv->tag_list){
        gst_tag_list_free(priv->tag_list);
        priv->tag_list = NULL;
    }
    return TRUE;
}

/* load a new url */
static int player_load(player_t* player,const char* url){
    yauap_priv_t* priv = player->yauap_priv;
    /* uninit */
    player_stop(player);

    printf("loading url %s\n",url);

    /* remove old source element*/
    if(priv->src)
        gst_bin_remove(GST_BIN(priv->play),priv->src);

    /* create a new source element */
    priv->src = gst_element_make_from_uri(GST_URI_SRC, url,"source");


    /* add it to the pipeline and link it to the decoder */
    if(priv->src){
        gst_bin_add (GST_BIN( priv->play),priv->src);
        gst_element_link(priv->src,priv->decode);
        return TRUE;
    }

    return FALSE;
}

/* seek to offset in ms */
static int player_seek(player_t* player,unsigned int offset){
    yauap_priv_t* priv = player->yauap_priv;
    printf("seeking to %i\n",offset);

    /* reset sample buffer (for scope) */
    priv->write_pos = priv->read_pos = 0;

    if(!gst_element_seek(priv->play,1.0,GST_FORMAT_TIME,GST_SEEK_FLAG_ACCURATE|GST_SEEK_FLAG_FLUSH,
                         GST_SEEK_TYPE_SET,offset * GST_MSECOND,GST_SEEK_TYPE_NONE, 0)){
        printf("seek failed\n");
        player->start(player);

    }
    return TRUE;
}

/* return current volume [0-100] */
static float player_get_volume(player_t* player){
    yauap_priv_t* priv = player->yauap_priv;
    gdouble volume = 0.0;
    /* get current volume */
    if(priv->volume)
        g_object_get( G_OBJECT(priv->volume), "volume", &volume, NULL );
    return volume * 25.0; 
}

/* set new volume [0-100] */
static int player_set_volume(player_t* player,float value){
    yauap_priv_t* priv = player->yauap_priv;
    gdouble volume = value; 

    if(!priv->volume)
       return FALSE;

    /* change range from 0 - 100 to 0 - 4 */
    volume *= 0.04;
 
    
    /* ajust values */
    if(volume < 0.0)
        volume = 0.0;
    else if(volume > 4.0)
        volume = 4.0;

    printf("\nsetting volume %f\n",value);
    g_object_set(G_OBJECT(priv->volume), "volume", volume, NULL);
    return TRUE;
}


/* start playback */
#define PLAY_TIMEOUT 1000
static int player_start(player_t* player){
    yauap_priv_t* priv = player->yauap_priv;
    GstState state;
    int timeout = PLAY_TIMEOUT;

    /* reset sample buffer (for scope) */
    priv->write_pos = priv->read_pos = 0;



    /* start playback */
    gst_element_set_state(priv->play, GST_STATE_PLAYING);

    /* wait until we are really playing back */
    while(timeout > 0 && gst_element_get_state(priv->play, &state, NULL, 0) && state != GST_STATE_PLAYING){
    //    printf("waiting...\n");
        usleep(100);
        --timeout;
    }
    if(timeout <= 0){
        printf("timed out waiting for playback to start\n");
        return TRUE;
    }


    return TRUE;
}

/* metadata handling */

static void list_append_string(char*** ret,const char* tag,char* str){
    unsigned int len = strlen(tag)+strlen(str)+2;
    char* tmp = calloc(1,len);
    snprintf(tmp,len,"%s=%s",tag,str);
    (*ret)[0] = strdup(tmp);
    *ret = *ret + 1;
    free(tmp);
}
static void list_append_uint(char*** ret,const char* tag,unsigned int value){
    char buf[100];
    snprintf(buf,sizeof(buf),"%u",value);
    list_append_string(ret,tag,buf);
}


/* read out metadata and append it to the string list in user_data */
static void tag_for_each(const GstTagList *list,const gchar *tag,gpointer user_data){
    char***ret = user_data;
    GType type = gst_tag_get_type(tag);

    if(type==G_TYPE_STRING){
        char* str_value;
        if(gst_tag_list_get_string(list,tag,&str_value)){
            list_append_string(ret,tag,str_value);
            g_free(str_value);
        }
    }else if(type == G_TYPE_UINT){
        unsigned int uint_value=0;
        if(gst_tag_list_get_uint(list,tag,&uint_value))
            list_append_uint(ret,tag,uint_value);
    }else if(type == GST_TYPE_DATE){
        GDate* date=NULL;
        if(gst_tag_list_get_date(list,tag,&date))
            list_append_uint(ret,tag,g_date_get_year(date));
    }
}

/* count the entries in a taglist */
static void taglist_count_entries(const GstTagList *list,const gchar *tag,gpointer user_data){
    unsigned int* cnt = user_data;
    *cnt = *cnt + 1;
}


/* get a list with all meta infos */
static int player_get_metadata(player_t* player,char*** ret){
    yauap_priv_t* priv = player->yauap_priv;
    GstCaps *caps;
    GstStructure *str;
    GstPad *audiopad;
    unsigned int num_tags = 0;   
    printf ("player_get_metadata\n");
    char**  ptr; 
    gst_tag_list_foreach(priv->tag_list,taglist_count_entries,&num_tags); /* count tags */
    *ret = ptr = calloc(num_tags+4,sizeof(char*)); /* allocate taglist for tags + num_channels + samplerate + length */
    /* get tags from tag_list */
    gst_tag_list_foreach(priv->tag_list,tag_for_each,&ptr);


    /* get samplerate and channels */
    audiopad = gst_element_get_pad (priv->audio, "sink");
    /* get negotiated caps */
    if(audiopad && GST_IS_PAD(audiopad)){
        caps = gst_pad_get_negotiated_caps(GST_PAD_CAST(audiopad));

        str = gst_caps_get_structure(caps, 0);

        if(str && gst_structure_get_int(str,"rate",&priv->rate))
            list_append_uint(&ptr,"samplerate",priv->rate);

        if(gst_structure_get_int(str,"channels",&priv->channels))
            list_append_uint(&ptr,"channels",priv->channels);
    }

    /* get length */
    list_append_uint(&ptr,"length",player_get_time_length(player));

    /* terminate the list */
    ptr[0]=NULL;

    /* test */
#if 0
    ptr = *ret;
    while(*ptr){
        printf("%s\n",*ptr);
        ++ptr;
    }
#endif
    return TRUE;
}

/* return current audio sample buffer */
static int player_get_scopedata(player_t* player,char* buf){
    yauap_priv_t* priv = player->yauap_priv;
    if(!priv->use_scope){
        memset(buf,0,SCOPE_SIZE);
        return TRUE;
    }

    memcpy(buf,priv->sample_buffer + priv->read_pos,SCOPE_SIZE);
    priv->read_pos = (priv->read_pos + SCOPE_SIZE) % SAMPLE_BUFFER_SIZE;

    return TRUE;
}




/* function that handles messages from gstreamer */
static gboolean gstreamer_callback(GstBus *bus,GstMessage *msg,gpointer    data){
    player_t* player = data;
    yauap_priv_t* priv = player->yauap_priv;
    switch (GST_MESSAGE_TYPE (msg)) {
        case GST_MESSAGE_TAG:{
            GstTagList* tlist;
            gst_message_parse_tag(msg,&tlist);
            priv->tag_list = gst_tag_list_merge(priv->tag_list,tlist,GST_TAG_MERGE_PREPEND);

            /* signal metadata change */
            signal_frontend(player,SIGNAL_METADATA,"new metadata"); 
            }
            break;
        case GST_MESSAGE_EOS:
            g_print ("\nEnd-of-stream\n");
            player_stop(player);
            signal_frontend(player,SIGNAL_EOS,"end of stream");
            break;
        case GST_MESSAGE_ERROR: {
            gchar *debug;
            GError *err;

            gst_message_parse_error (msg, &err, &debug);
            g_free (debug);

            g_print ("\nError: %s\n", err->message);
            
            signal_frontend(player,SIGNAL_ERROR,err->message); 
            
            g_error_free (err);
            g_main_loop_quit(priv->loop);
            break;
            }
        default:
            break;
    }
    return TRUE;
}

/* callback function that gets called once the output format of the decoder is figured out
   the audio filter chain gets connected to the pipeline here 
*/
static void cb_new_decoded_pad(GstElement *decodebin,GstPad *pad,gboolean last,gpointer data){
    GstCaps *caps;
    GstStructure *str;
    GstPad *audiopad;
    yauap_priv_t* priv = data;

    /* remove already linked element */
    audiopad = gst_element_get_pad(priv->audio, "sink");
    if (GST_PAD_IS_LINKED(audiopad)) {
        g_object_unref(audiopad);
        return;
    }

    caps = gst_pad_get_caps(pad);
    str = gst_caps_get_structure(caps, 0);

    if (!g_strrstr(gst_structure_get_name(str), "audio")) {
        gst_caps_unref(caps);
        gst_object_unref(audiopad);
        return;
    }
    gst_caps_unref(caps);

    /* link */
    gst_pad_link(pad, audiopad);

}


/* get sample buffer (assumes 16-bit 2 channel audio ) */
static void handoff_cb( GstPad* pad, GstBuffer* buf, gpointer arg){
    player_t* player = arg;
    yauap_priv_t* priv = player->yauap_priv;
    unsigned char* data = GST_BUFFER_DATA(buf);
    unsigned int len = buf->size;
    int space = SAMPLE_BUFFER_SIZE - priv->write_pos;

    
    if(len > SAMPLE_BUFFER_SIZE){
        printf("increase sample buffer size !!!!!!!!!\n");
        return;

    }

    /* fill scope ring buffer */
    if(space >= len){   
        memcpy(priv->sample_buffer + priv->write_pos,data,len);
    }else{
        memcpy(priv->sample_buffer + priv->write_pos,data,space);
        memcpy(priv->sample_buffer, data + space, len - space); 
    }
    priv->write_pos = ( priv->write_pos + len ) % SAMPLE_BUFFER_SIZE;

}




/************************ decodeable check ***************************************/

/* how many microseconds shall we wait for the detection to succeed */
#define DETECT_TIMEOUT 100000


/* callback function that checks if a audio decoder has been added */
static void can_decode_new_decode_pad_callback(GstElement* element, GstPad* pad, gboolean a, gpointer data){
    int* can_decode=data;
    GstCaps* caps = gst_pad_get_caps( pad );
    if(gst_caps_get_size(caps)>0) {
        GstStructure* str = gst_caps_get_structure( caps,0 );
        if(g_strrstr(gst_structure_get_name( str ), "audio" ))
            *can_decode = 1;
    }
    gst_caps_unref( caps );
}

/* callback function that terminates the detection process */
static void can_decode_no_more_pads_callback(GstElement* element, gpointer data){
    int* last = (int*)data;
    *last = 1;
}

/* checks if we are able to decode the audio part of the given url */
/* simply starts playback with fake video and audio output devices */
/* returns nonzero on success, zero otherwise */
static int player_can_decode(const char* url){
    int can_decode=0;
    int last = 0;
    unsigned int timeout = 0;
    /* create a simple decode pipeline */
    GstElement* play = gst_element_factory_make("pipeline", "play");
    GstElement* src = gst_element_make_from_uri(GST_URI_SRC, url,"source");
    GstElement* decodebin =  gst_element_factory_make("decodebin", "decode");
    gst_bin_add( GST_BIN( play ), src);
    gst_bin_add( GST_BIN( play ), decodebin );
    gst_element_link( src, decodebin );

    /* connect signal handlers */
    g_signal_connect( G_OBJECT( decodebin ), "new-decoded-pad", G_CALLBACK( can_decode_new_decode_pad_callback ), &can_decode );
    g_signal_connect( G_OBJECT( decodebin ), "no-more-pads", G_CALLBACK( can_decode_no_more_pads_callback ), &last );

    /* start decoding */ 
    gst_element_set_state(play,GST_STATE_PLAYING );

    /* wait a bit */
    while ( !can_decode && !last && timeout < DETECT_TIMEOUT ) {
        timeout += 1000;
        usleep(1000);
    }

    /* stop playback */
    gst_element_set_state(play,GST_STATE_NULL);

    /* destroy pipeline */
    gst_object_unref( GST_OBJECT( play ) );

    return can_decode;
}

/****************************** end decodeabel ***************************************/



static void display_usage(void){
    printf("Usage: yauap [options]\n"
           "options:\n"
           " -h display this help\n");
}   

yauap_frontend_t* init_commandline(int argc, char* argv[],player_t* player);
yauap_frontend_t* init_dbus_service(player_t* player);


int main(int argc, char* argv[]){
    player_t player;
    GstBus *bus;
    GstElement* convert;
    GstElement* resample;
    GstElement* audiosink;
    GstElement* identity;
    GstPad* audiopad;
    GstPad* convertpad;
    
    yauap_priv_t* priv = calloc(1,sizeof(yauap_priv_t));
    int i;
    int run=1;
    

    memset(&player,0,sizeof(player_t));

    /* set function pointers */
    player.quit = player_quit;
    player.pause = player_pause;
    player.can_decode = player_can_decode;
    player.stop = player_stop;
    player.load = player_load;
    player.start = player_start;
    player.get_time_length = player_get_time_length;
    player.get_time_position = player_get_time_position;
    player.seek = player_seek;
    player.get_metadata = player_get_metadata;
    player.get_scopedata = player_get_scopedata;
    player.get_volume = player_get_volume;
    player.set_volume = player_set_volume;
    player.yauap_priv = priv;
    priv->use_scope = 1;


    /* init gstreamer */
    gst_init(&argc, &argv);



    /* parse generic arguments */
    for(i=1;i<argc;i++){
        if(!strcmp(argv[i],"-h")){
            display_usage();
            run=0;
        }
    }


    /* create main loop */
    priv->loop = g_main_loop_new(NULL, FALSE);
    priv->tag_list = gst_tag_list_new();


    /* set up gstreamer pipeline */


    /* create a simple audio pipeline */
    priv->play = gst_element_factory_make("pipeline","play");

    /* create a generic decoder (the src element will be created in player_load) */
    priv->decode = gst_element_factory_make("decodebin", "decoder");

    /* create audio filter chain */
    priv->audio = gst_bin_new("audiobin");

    /* create conversion, resample, volume and audio output filters */
    convert = gst_element_factory_make("audioconvert", "convert");
    resample = gst_element_factory_make ("audioresample", "aresample");
    priv->volume = gst_element_factory_make ("volume", "volume");
    audiosink = gst_element_factory_make("autoaudiosink", "audiosink");
    /* clone the data for the scope */
    identity = gst_element_factory_make("identity","identity");

    /* check if the necessary plugins are really installed */
    if(!convert || !resample || !priv->volume || !audiosink || !identity){
        printf("error couldn't load gstreamer plugins\n");
        printf("convert=%p resample=%p volume=%p audiosink=%p identity=%p\n",convert,resample,priv->volume,audiosink,identity);
        return 1;
    }

    /* add them to the pipeline and link them */
    gst_bin_add_many (GST_BIN(priv->audio), convert, identity,priv->volume,resample,audiosink, NULL);

    /* create a sink pad for our audio filter chain similar to the pad of the conversion filter */
    audiopad = gst_element_get_pad(convert, "sink");
    gst_element_add_pad(priv->audio,gst_ghost_pad_new ("sink", audiopad));
    gst_object_unref(audiopad);
    if(priv->use_scope){
        /* add a data probe to the src of the convert pad */
        convertpad = gst_element_get_pad(convert, "src");
        gst_pad_add_buffer_probe(convertpad, G_CALLBACK(handoff_cb), &player);
        gst_object_unref(convertpad);
    }


    gst_element_link_many(convert,identity,priv->volume,resample,audiosink,NULL);


    /* add the audio decoder and the audio filter chain to our pipeline */
    gst_bin_add_many(GST_BIN(priv->play), priv->decode, priv->audio, NULL);

    /* connect the cb_new_decoded_pad callback */
    g_signal_connect(priv->decode, "new-decoded-pad", G_CALLBACK (cb_new_decoded_pad), priv);

    /* connect bus callback */
    bus = gst_pipeline_get_bus(GST_PIPELINE (priv->play));
    gst_bus_add_watch(bus, gstreamer_callback, &player);
    gst_object_unref(bus);


    /* init frontends */
    player.frontends[0] = init_dbus_service(&player);
    player.frontends[1] = init_commandline(argc, argv,&player);
    player.frontends[2] = NULL; 

    /* fixme currently we require the commandline frontend */
    if(!player.frontends[1])
        run = 0;

    /* run the main loop */
    if(run)
        g_main_loop_run(priv->loop);

    /* make sure that we really stopped */
    player_stop(&player);


    /* uninit frontends */
    i=0;
    while(player.frontends[i]){
       if(player.frontends[i]->free)
           player.frontends[i]->free(player.frontends[i]);
       ++i;
    }    




    /* clean up */
    gst_object_unref(GST_OBJECT (priv->play));

    free(priv);


    gst_deinit();    
    printf("\n");
    return 0;
}
