/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2014-05-17
 * Description : Album Labels Tree View.
 *
 * Copyright (C) 2014 Mohamed Anwer <mohammed dot ahmed dot anwer at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#include "albumlabelstreeview.moc"

// QT includes

#include <QTreeWidget>
#include <QPainter>
#include <QDebug>

// KDE includes

#include <kapplication.h>
#include <KIconLoader>
#include <KUrl>

// Local includes

#include "searchxml.h"
#include "searchtabheader.h"
#include "albummanager.h"
#include "albumtreeview.h"
#include "databaseconstants.h"
#include "imagelister.h"
#include "statesavingobject.h"
#include "databaseaccess.h"
#include "albumdb.h"
#include "colorlabelfilter.h"
#include "picklabelfilter.h"
#include "tagscache.h"
#include "globals.h"
#include "albumsettings.h"

namespace Digikam
{

class AlbumLabelsTreeView::Private
{
public:
    Private() :
        ratings(0),
        picks(0),
        colors(0),
        isCheckableTreeView(false),
        isLoadingState(false),
        iconSizeFromSetting(0)
    {
        starPolygon << QPoint(0,  24);
        starPolygon << QPoint(20, 20);
        starPolygon << QPoint(28,  0);
        starPolygon << QPoint(36, 20);
        starPolygon << QPoint(56, 24);
        starPolygon << QPoint(40, 36);
        starPolygon << QPoint(44, 56);
        starPolygon << QPoint(28, 44);
        starPolygon << QPoint(12, 56);
        starPolygon << QPoint(16, 36);
    }

    QFont                        rootFont;
    QFont                        regularFont;
    QSize                        iconSize;
    QSize                        rootSizeHint;

    QPolygon                     starPolygon;

    QTreeWidgetItem*             ratings;
    QTreeWidgetItem*             picks;
    QTreeWidgetItem*             colors;

    bool                         isCheckableTreeView;
    bool                         isLoadingState;
    int                          iconSizeFromSetting;

    QHash<Labels, QList<int> >   selectedLabels;

    static const QString         configRatingSelectionEntry;
    static const QString         configPickSelectionEntry;
    static const QString         configColorSelectionEntry;
    static const QString         configExpansionEntry;
};

const QString AlbumLabelsTreeView::Private::configRatingSelectionEntry("RatingSelection");
const QString AlbumLabelsTreeView::Private::configPickSelectionEntry("PickSelection");
const QString AlbumLabelsTreeView::Private::configColorSelectionEntry("ColorSelection");
const QString AlbumLabelsTreeView::Private::configExpansionEntry("Expansion");

AlbumLabelsTreeView::AlbumLabelsTreeView(QWidget *parent, bool setCheckable) :
    QTreeWidget(parent), StateSavingObject(this), d(new Private)
{
    d->rootFont            = QFont("Times",18,-1,false);
    d->regularFont         = AlbumSettings::instance()->getTreeViewFont();
    d->iconSizeFromSetting = AlbumSettings::instance()->getTreeViewIconSize();
    d->iconSize            = QSize(d->iconSizeFromSetting,d->iconSizeFromSetting);
    d->rootSizeHint        = QSize(1,40);
    d->isCheckableTreeView = setCheckable;

    setHeaderLabel("My Labels");
    setUniformRowHeights(false);
    initTreeView();

    if(d->isCheckableTreeView)
    {
        QTreeWidgetItemIterator it(this);
        while(*it)
        {
            if((*it)->parent())
            {
                (*it)->setFlags((*it)->flags()|Qt::ItemIsUserCheckable);
                (*it)->setCheckState(0, Qt::Unchecked);
            }
            ++it;
        }
    }
    else
    {
        setSelectionMode(QAbstractItemView::ExtendedSelection);
    }

    connect(AlbumSettings::instance(), SIGNAL(setupChanged()),
            this, SLOT(slotSettingsChanged()));
}

AlbumLabelsTreeView::~AlbumLabelsTreeView()
{
    delete d;
}

bool AlbumLabelsTreeView::isCheckable() const
{
    return d->isCheckableTreeView;
}

bool AlbumLabelsTreeView::isLoadingState() const
{
    return d->isLoadingState;
}

QPixmap AlbumLabelsTreeView::goldenStarPixmap() const
{
    QPixmap pixmap = QPixmap(60, 60);
    pixmap.fill(Qt::transparent);

    QPainter p1(&pixmap);
    p1.setRenderHint(QPainter::Antialiasing, true);
    p1.setBrush(QColor(0xff,0xd7,0x00));
    p1.setPen(palette().color(QPalette::Active, foregroundRole()));
    p1.drawPolygon(d->starPolygon, Qt::WindingFill);
    p1.end();

    return pixmap;
}

QPixmap AlbumLabelsTreeView::colorRectPixmap(QColor color) const
{
    QRect rect(8,8,48,48);
    QPixmap pixmap = QPixmap(60, 60);
    pixmap.fill(Qt::transparent);

    QPainter p1(&pixmap);
    p1.setRenderHint(QPainter::Antialiasing, true);
    p1.setBrush(color);
    p1.setPen(palette().color(QPalette::Active, foregroundRole()));
    p1.drawRect(rect);
    p1.end();

    return pixmap;
}

QHash<AlbumLabelsTreeView::Labels, QList<int> > AlbumLabelsTreeView::selectedLabels()
{
    QHash<Labels, QList<int> > selectedLabelsHash;
    QList<int> selectedRatings;
    QList<int> selectedPicks;
    QList<int> selectedColors;

    if(d->isCheckableTreeView)
    {
        QTreeWidgetItemIterator it(this, QTreeWidgetItemIterator::Checked);
        while(*it)
        {
            QTreeWidgetItem* item = (*it);

            if(item->parent() == d->ratings)
                selectedRatings << indexFromItem(item).row();
            else if(item->parent() == d->picks)
                selectedPicks << indexFromItem(item).row();
            else
                selectedColors << indexFromItem(item).row();

            ++it;
        }
    }
    else
    {
        foreach (QTreeWidgetItem* item, selectedItems())
        {
            if(item->parent() == d->ratings)
                selectedRatings << indexFromItem(item).row();
            else if(item->parent() == d->picks)
                selectedPicks << indexFromItem(item).row();
            else
                selectedColors << indexFromItem(item).row();
        }
    }

    selectedLabelsHash[Ratings] = selectedRatings;
    selectedLabelsHash[Picks]   = selectedPicks;
    selectedLabelsHash[Colors]  = selectedColors;

    return selectedLabelsHash;
}

void AlbumLabelsTreeView::doLoadState()
{
    d->isLoadingState = true;
    KConfigGroup configGroup         = getConfigGroup();
    const QList<int> expansion       = configGroup.readEntry(entryName(d->configExpansionEntry),       QList<int>());
    const QList<int> selectedRatings = configGroup.readEntry(entryName(d->configRatingSelectionEntry), QList<int>());
    const QList<int> selectedPicks   = configGroup.readEntry(entryName(d->configPickSelectionEntry),   QList<int>());
    const QList<int> selectedColors  = configGroup.readEntry(entryName(d->configColorSelectionEntry),  QList<int>());

    d->ratings->setExpanded(true);
    d->picks->setExpanded(true);
    d->colors->setExpanded(true);

    foreach (int parent, expansion)
    {
        switch (parent) {
        case 1:
            d->ratings->setExpanded(false);
            break;
        case 2:
            d->picks->setExpanded(false);
            break;
        case 3:
            d->colors->setExpanded(false);
        default:
            break;
        }
    }

    foreach (int rating, selectedRatings) {
        d->ratings->child(rating)->setSelected(true);
    }

    foreach (int pick, selectedPicks) {
        d->picks->child(pick)->setSelected(true);
    }

    foreach (int color, selectedColors) {
        d->colors->child(color)->setSelected(true);
    }

    d->isLoadingState = false;
}

void AlbumLabelsTreeView::doSaveState()
{
    KConfigGroup configGroup = getConfigGroup();
    QList<int> expansion;

    if(!d->ratings->isExpanded())
    {
        expansion << 1;
    }

    if(!d->picks->isExpanded())
    {
        expansion << 2;
    }

    if(!d->colors->isExpanded())
    {
        expansion << 3;
    }

    QHash<Labels, QList<int> > labels =  selectedLabels();

    configGroup.writeEntry(entryName(d->configExpansionEntry), expansion);
    configGroup.writeEntry(entryName(d->configRatingSelectionEntry), labels[Ratings]);
    configGroup.writeEntry(entryName(d->configPickSelectionEntry),   labels[Picks]);
    configGroup.writeEntry(entryName(d->configColorSelectionEntry),  labels[Colors]);
}

void AlbumLabelsTreeView::setCurrentAlbum()
{
    emit signalSetCurrentAlbum();
}

void AlbumLabelsTreeView::initTreeView()
{
    setIconSize(QSize(d->iconSizeFromSetting*5,d->iconSizeFromSetting));
    initRatingsTree();
    initPicksTree();
    initColorsTree();
    expandAll();
    setRootIsDecorated(false);
}

void AlbumLabelsTreeView::initRatingsTree()
{
    d->ratings = new QTreeWidgetItem(this);
    d->ratings->setText(0, i18n("Rating"));
    d->ratings->setSizeHint(0,d->rootSizeHint);
    d->ratings->setFont(0,d->rootFont);
    d->ratings->setFlags(Qt::ItemIsEnabled);

    QTreeWidgetItem* noRate = new QTreeWidgetItem(d->ratings);
    noRate->setText(0,i18n("No Rating"));
    noRate->setIcon(0,KIconLoader::global()->loadIcon("emblem-unmounted", KIconLoader::NoGroup, 48));
    noRate->setFont(0,d->regularFont);

    for(int rate = 1 ; rate <= 5 ; rate++)
    {
        QTreeWidgetItem* rateWidget = new QTreeWidgetItem(d->ratings);

        QPixmap pix(goldenStarPixmap().width()*rate,goldenStarPixmap().height());
        pix.fill(Qt::transparent);
        QPainter p(&pix);
        int offset = 0;
        p.setRenderHint(QPainter::Antialiasing, true);
        p.setPen(palette().color(QPalette::Active, foregroundRole()));

        for(int i = 0 ; i < rate ; ++i)
        {
            p.drawPixmap(offset,0,goldenStarPixmap());
            offset += goldenStarPixmap().width();
        }

        rateWidget->setIcon(0,QIcon(pix));
        rateWidget->setSizeHint(0,d->iconSize);
    }
}

void AlbumLabelsTreeView::initPicksTree()
{
    d->picks = new QTreeWidgetItem(this);
    d->picks->setText(0, i18n("Pick"));
    d->picks->setSizeHint(0,d->rootSizeHint);
    d->picks->setFont(0,d->rootFont);
    d->picks->setFlags(Qt::ItemIsEnabled);

    QStringList pickSetNames;
    pickSetNames << i18n("No Pick")
                 << i18n("Rejected Item")
                 << i18n("Pending Item")
                 << i18n("Accepted Item");

    QStringList pickSetIcons;
    pickSetIcons << "emblem-unmounted"
                 << "flag-red"
                 << "flag-yellow"
                 << "flag-green";

    foreach (QString pick, pickSetNames) {
        QTreeWidgetItem* pickWidgetItem = new QTreeWidgetItem(d->picks);
        pickWidgetItem->setText(0,pick);
        pickWidgetItem->setFont(0,d->regularFont);
        pickWidgetItem->setIcon(0,KIconLoader::global()->loadIcon(pickSetIcons.at(pickSetNames.indexOf(pick)), KIconLoader::NoGroup, 48));
    }
}

void AlbumLabelsTreeView::initColorsTree()
{
    d->colors = new QTreeWidgetItem(this);
    d->colors->setText(0, i18n("Color"));
    d->colors->setSizeHint(0,d->rootSizeHint);
    d->colors->setFont(0,d->rootFont);
    d->colors->setFlags(Qt::ItemIsEnabled);

    QTreeWidgetItem* noColor = new QTreeWidgetItem(d->colors);
    noColor->setText(0,i18n("No Color"));
    noColor->setFont(0,d->regularFont);
    noColor->setIcon(0,KIconLoader::global()->loadIcon("emblem-unmounted", KIconLoader::NoGroup, 48));

    QStringList colorSet;
    colorSet << "red"      << "orange"
             << "yellow"   << "darkgreen"
             << "darkblue" << "magenta"
             << "darkgray" << "black"
             << "white";

    QStringList colorSetNames;
    colorSetNames << i18n("Red")    << i18n("Orange")
                  << i18n("Yellow") << i18n("Green")
                  << i18n("Blue")   << i18n("Magenta")
                  << i18n("Gray")   << i18n("Black")
                  << i18n("White");

    foreach (QString color, colorSet) {
        QTreeWidgetItem* colorWidgetItem = new QTreeWidgetItem(d->colors);
        colorWidgetItem->setText(0,colorSetNames.at(colorSet.indexOf(color)));
        colorWidgetItem->setFont(0,d->regularFont);
        QPixmap colorIcon = colorRectPixmap(QColor(color));
        colorWidgetItem->setIcon(0,QIcon(colorIcon));
        colorWidgetItem->setSizeHint(0,d->iconSize);
    }
}

void AlbumLabelsTreeView::slotSettingsChanged()
{
    if(d->iconSizeFromSetting != AlbumSettings::instance()->getTreeViewIconSize())
    {
        d->iconSizeFromSetting = AlbumSettings::instance()->getTreeViewIconSize();
        setIconSize(QSize(d->iconSizeFromSetting*5,d->iconSizeFromSetting));
        d->iconSize = QSize(d->iconSizeFromSetting,d->iconSizeFromSetting);
        QTreeWidgetItemIterator it(this);
        while(*it)
        {
            if((*it)->parent())
            {
                (*it)->setSizeHint(0,d->iconSize);
            }
            ++it;
        }
    }

    if(d->regularFont != AlbumSettings::instance()->getTreeViewFont())
    {
        d->regularFont = AlbumSettings::instance()->getTreeViewFont();
        QTreeWidgetItemIterator it(this);
        while(*it)
        {
            if((*it)->parent())
            {
                (*it)->setFont(0,d->regularFont);
            }
            ++it;
        }
    }
}

void AlbumLabelsTreeView::restoreSelectionFromHistory(QHash<Labels, QList<int> > neededLabels)
{
    QTreeWidgetItemIterator it(this, QTreeWidgetItemIterator::Selected);
    while(*it)
    {
        (*it)->setSelected(false);
        ++it;
    }

    foreach (int rateItemIndex, neededLabels[Ratings])
    {
        d->ratings->child(rateItemIndex)->setSelected(true);
    }

    foreach (int pickItemIndex, neededLabels[Picks])
    {
        d->picks->child(pickItemIndex)->setSelected(true);
    }

    foreach (int colorItemIndex, neededLabels[Colors])
    {
        d->colors->child(colorItemIndex)->setSelected(true);
    }
}

// -------------------------------------------------------------------------------

class AlbumLabelsSearchHandler::Private
{
public:

    Private() :
        treeWidget(0),
        restoringSelectionFromHistory(0),
        currentXmlIsEmpty(0),
        albumForSelectedItems(0)
    {}


    AlbumLabelsTreeView* treeWidget;
    bool                 restoringSelectionFromHistory;
    bool                 currentXmlIsEmpty;
    QString              oldXml;
    Album*               albumForSelectedItems;
    QString              generatedAlbumName;
    KUrl::List           urlListForSelectedAlbum;
};

AlbumLabelsSearchHandler::AlbumLabelsSearchHandler(AlbumLabelsTreeView *treeWidget) :
    d(new Private)
{
    d->treeWidget = treeWidget;

    if(!d->treeWidget->isCheckable())
    {
        connect(d->treeWidget, SIGNAL(itemSelectionChanged()),
                this, SLOT(slotSelectionChanged()));

        connect(d->treeWidget, SIGNAL(signalSetCurrentAlbum()),
                this, SLOT(slotSetCurrentAlbum()));
    }
    else
    {
        connect(d->treeWidget, SIGNAL(itemClicked(QTreeWidgetItem*,int)),
                this, SLOT(slotCheckStateChanged()));
    }
}

AlbumLabelsSearchHandler::~AlbumLabelsSearchHandler()
{
    delete d;
}

Album *AlbumLabelsSearchHandler::albumForSelectedItems() const
{
    return d->albumForSelectedItems;
}

KUrl::List AlbumLabelsSearchHandler::imagesUrls()
{
    return d->urlListForSelectedAlbum;
}

QString AlbumLabelsSearchHandler::generatedName() const
{
    return d->generatedAlbumName;
}

void AlbumLabelsSearchHandler::restoreSelectionFromHistory(QHash<AlbumLabelsTreeView::Labels, QList<int> > neededLabels)
{
    d->restoringSelectionFromHistory = true;
    d->treeWidget->restoreSelectionFromHistory(neededLabels);
    d->restoringSelectionFromHistory = false;
    slotSelectionChanged();
}

bool AlbumLabelsSearchHandler::isRestoringSelectionFromHistory() const
{
    return d->restoringSelectionFromHistory;
}

QString AlbumLabelsSearchHandler::createXMLForCurrentSelection(QHash<AlbumLabelsTreeView::Labels, QList<int> > selectedLabels)
{
    SearchXmlWriter writer;
    writer.setFieldOperator(SearchXml::standardFieldOperator());
    QList<int>        ratings;
    QList<int>        colorsAndPicks;

    foreach (int rate, selectedLabels[AlbumLabelsTreeView::Ratings])
    {
        if(rate == 0)
            ratings << -1;
        else
            ratings << rate;
    }

    foreach (int color, selectedLabels[AlbumLabelsTreeView::Colors])
    {
        colorsAndPicks << TagsCache::instance()->tagForColorLabel(color);
    }

    foreach (int pick, selectedLabels[AlbumLabelsTreeView::Picks])
    {
        colorsAndPicks << TagsCache::instance()->tagForPickLabel(pick);
    }

    d->currentXmlIsEmpty = (ratings.isEmpty() && colorsAndPicks.isEmpty()) ? true : false;

    if(!ratings.isEmpty() && !colorsAndPicks.isEmpty())
    {
        foreach (int val, ratings)
        {
            writer.writeGroup();
            writer.writeField("rating", SearchXml::Equal);
            writer.writeValue(val);
            writer.finishField();

            writer.writeField("tagid",SearchXml::InTree);
            writer.writeValue(colorsAndPicks);
            writer.finishField();

            writer.finishGroup();
        }
    }
    else if(!ratings.isEmpty())
    {
        foreach (int rate, ratings) {
            writer.writeGroup();
            writer.writeField("rating", SearchXml::Equal);
            writer.writeValue(rate);
            writer.finishField();
            writer.finishGroup();
        }
    }
    else if(!colorsAndPicks.isEmpty())
    {
        writer.writeGroup();
        writer.writeField("tagid",SearchXml::InTree);
        writer.writeValue(colorsAndPicks);
        writer.finishField();
        writer.finishGroup();
    }
    else
    {
        writer.writeGroup();
        writer.finishGroup();
    }

    writer.finish();

    generateAlbumNameForExporting(selectedLabels[AlbumLabelsTreeView::Ratings],
                                  selectedLabels[AlbumLabelsTreeView::Colors],
                                  selectedLabels[AlbumLabelsTreeView::Picks]);
    return writer.xml();
}

SAlbum* AlbumLabelsSearchHandler::search(const QString &xml) const
{
    SAlbum* album;
    int id;

    if(!d->treeWidget->isCheckable())
    {
        album = AlbumManager::instance()->findSAlbum(SAlbum::getTemporaryTitle(DatabaseSearch::AdvancedSearch));

        if(album)
        {
            id = album->id();
            DatabaseAccess().db()->updateSearch(id,DatabaseSearch::AdvancedSearch,
                                                SAlbum::getTemporaryTitle(DatabaseSearch::AdvancedSearch), xml);
        }
        else
        {
            id = DatabaseAccess().db()->addSearch(DatabaseSearch::AdvancedSearch,
                                                  SAlbum::getTemporaryTitle(DatabaseSearch::AdvancedSearch), xml);
        }
        album = new SAlbum(getDefaultTitle(), id);
    }
    else
    {
        album = AlbumManager::instance()->findSAlbum(getDefaultTitle());

        if(album)
        {
            id = album->id();
            DatabaseAccess().db()->updateSearch(id,DatabaseSearch::AdvancedSearch,
                                                getDefaultTitle(), xml);
        }
        else
        {
            id = DatabaseAccess().db()->addSearch(DatabaseSearch::AdvancedSearch,
                                                  getDefaultTitle(), xml);
        }
        album = new SAlbum(d->generatedAlbumName, id);
    }

    if(!album->isUsedByLabelsTree())
        album->setUsedByLabelsTree(true);

    return album;
}

void AlbumLabelsSearchHandler::generateAlbumNameForExporting(QList<int> ratings, QList<int> colorsList, QList<int> picksList)
{
    QString name;
    QString ratingsString;
    QString picksString;
    QString colorsString;

    if(!ratings.isEmpty())
    {
        ratingsString += i18n("Rating: ");

        QListIterator<int> it(ratings);

        while (it.hasNext())
        {
            int rating = it.next();
            if(rating == -1)
            {
                ratingsString += i18n("No Rating");
            }
            else
            {
                ratingsString += QString::number(rating);
            }

            if(it.hasNext())
            {
                ratingsString +=", ";
            }
        }
    }

    if(!colorsList.isEmpty())
    {
        colorsString += i18n("Colors: ");

        QListIterator<int> it(colorsList);

        while (it.hasNext())
        {
            switch (it.next()) {
            case NoColorLabel:
                colorsString += i18n("No Color");
                break;
            case RedLabel:
                colorsString += i18n("Red");
                break;
            case OrangeLabel:
                colorsString += i18n("Orange");
                break;
            case YellowLabel:
                colorsString += i18n("Yellow");
                break;
            case GreenLabel:
                colorsString += i18n("Green");
                break;
            case BlueLabel:
                colorsString += i18n("Blue");
                break;
            case MagentaLabel:
                colorsString += i18n("Magenta");
                break;
            case GrayLabel:
                colorsString += i18n("Gray");
                break;
            case BlackLabel:
                colorsString += i18n("Black");
                break;
            case WhiteLabel:
                colorsString += i18n("White");
                break;
            default:
                break;
            }

            if(it.hasNext())
            {
                colorsString +=", ";
            }
        }
    }

    if(!picksList.isEmpty())
    {
        picksString += i18n("Picks: ");

        QListIterator<int> it(picksList);
        while (it.hasNext())
        {
            switch (it.next()) {
            case NoPickLabel:
                picksString += i18n("No Pick");
                break;
            case RejectedLabel:
                picksString += i18n("Rejected");
                break;
            case PendingLabel:
                picksString += i18n("Pending");
                break;
            case AcceptedLabel:
                picksString += i18n("Accepted");
                break;
            default:
                break;
            }

            if(it.hasNext())
            {
                picksString +=", ";
            }
        }
    }


    if(ratingsString.isEmpty() && picksString.isEmpty())
    {
        name = colorsString;
    }
    else if(ratingsString.isEmpty() && colorsString.isEmpty())
    {
        name = picksString;
    }
    else if(colorsString.isEmpty() && picksString.isEmpty())
    {
        name = ratingsString;
    }
    else if(ratingsString.isEmpty())
    {
        name = picksString + " | " + colorsString;
    }
    else if(picksString.isEmpty())
    {
        name = ratingsString + " | " + colorsString;
    }
    else if(colorsString.isEmpty())
    {
        name = ratingsString + " | " + picksString;
    }
    else
    {
        name = ratingsString + " | " + picksString + " | " + colorsString;
    }

    d->generatedAlbumName = name;

}

void AlbumLabelsSearchHandler::imagesUrlsForCurrentAlbum()
{
    KUrl url = d->albumForSelectedItems->databaseUrl();
    KIO::TransferJob* job = ImageLister::startListJob(url);
    job->addMetaData("listAlbumsRecursively", "true");

    connect(job, SIGNAL(data(KIO::Job*,QByteArray)),
            this, SLOT(slotData(KIO::Job*,QByteArray)));
}

QString AlbumLabelsSearchHandler::getDefaultTitle() const
{
    if(d->treeWidget->isCheckable())
    {
        return i18n("Exporting Album");
    }
    else
    {
        return i18n("Labels Album");
    }
}

void AlbumLabelsSearchHandler::slotSelectionChanged()
{
    if(d->treeWidget->isLoadingState() || d->restoringSelectionFromHistory)
    {
        return;
    }

    QString xml   = createXMLForCurrentSelection(d->treeWidget->selectedLabels());
    SAlbum* album = search(xml);


    if(album)
    {
        AlbumManager::instance()->setCurrentAlbums(QList<Album*>() << album);
        d->albumForSelectedItems = album;
        d->oldXml = xml;
    }
}

void AlbumLabelsSearchHandler::slotCheckStateChanged()
{
    QString currentXml = createXMLForCurrentSelection(d->treeWidget->selectedLabels());

    if(currentXml == d->oldXml)
    {
        return;
    }

    if(d->albumForSelectedItems)
    {
        emit checkStateChanged(d->albumForSelectedItems,Qt::Unchecked);
    }

    SAlbum* album = search(currentXml);

    if (album)
    {
        if(!d->currentXmlIsEmpty)
        {
            d->albumForSelectedItems = album;
            imagesUrlsForCurrentAlbum();
        }
        else
        {
            d->albumForSelectedItems = 0;
        }

        emit checkStateChanged(album,Qt::Checked);
    }

    d->oldXml   = currentXml;
}

void AlbumLabelsSearchHandler::slotSetCurrentAlbum()
{
    slotSelectionChanged();
}

void AlbumLabelsSearchHandler::slotData(KIO::Job *job, QByteArray data)
{
    Q_UNUSED(job);
    if (data.isEmpty())
    {
        return;
    }

    QByteArray    tmp(data);
    QDataStream   ds(&tmp, QIODevice::ReadOnly);
    KUrl::List urlList;

    while (!ds.atEnd())
    {
        ImageListerRecord record;
        ds >> record;

        ImageInfo info(record);
        urlList << info.fileUrl();
    }

    d->urlListForSelectedAlbum = urlList;
}

} // namespace Digikam
