/*  This file is part of the KDE project
    Copyright (C) 2008 Dario Freddi <drf54321@gmail.com>
    Copyright (C) 2008 Lukas Appelhans <l.appelhans@gmx.de>
    Copyright (C) 2008 Alessandro Diaferia <alediaferia@gmail.com>
    Copyright (C) 2008 Daniel Nicoletti <dantti85-pk@yahoo.com.br>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.

*/

#include "AuthorizationsModel.h"
#include "pkgroup.h"
#include "pkitem.h"
#include "abstractitem.h"

#include <QStandardItem>

#include <KDebug>

namespace PolkitKde
{

AuthorizationsModel::AuthorizationsModel(QObject* parent)
        : QAbstractItemModel(parent)
{
    rootItem = new Group(QString(), QString());
}

AuthorizationsModel::~AuthorizationsModel()
{
    delete rootItem;
}

QModelIndex AuthorizationsModel::index(int row, int column, const QModelIndex &parent) const
{
    if (row > rowCount(parent) || column > columnCount(parent)) {
        kDebug() << "Invalid Index";
        return QModelIndex();
    }

    if (parent.isValid()) {
        Group * group = static_cast<Group*>(parent.internalPointer());
        return createIndex(row, column, group->children().at(row));
    } else {
        return createIndex(row, column, rootItem->childAt(row));
    }
    return QModelIndex();
}

QModelIndex AuthorizationsModel::parent(const QModelIndex &i) const
{
    if (!i.isValid()) {
        return QModelIndex();
    }

    AbstractItem *item = static_cast<AbstractItem*>(i.internalPointer());
    AbstractItem *parentItem = item->parent();
    int row;

    if (!parentItem) {
        return QModelIndex();
    }

    if (parentItem->parent() && parentItem->parent()->isGroup()) {
        row = static_cast<Group*>(parentItem->parent())->children().indexOf(parentItem);
    } else {
        row = rootItem->children().indexOf(parentItem);
    }

    return createIndex(row, i.column(), parentItem); // :)
}

int AuthorizationsModel::rowCount(const QModelIndex &parent) const
{
    if (parent.isValid()) {
        AbstractItem *item = static_cast<AbstractItem*>(parent.internalPointer());
        if (item->isGroup()) {
            return static_cast<Group*>(item)->children().count();
        } else {
            return 0;
        }
    }

    return rootItem->children().count();
}

int AuthorizationsModel::columnCount(const QModelIndex &parent) const
{
    Q_UNUSED(parent)
    return 1; // check this out..
}

QVariant AuthorizationsModel::data(const QModelIndex & index, int role) const
{
    AbstractItem *item = static_cast<AbstractItem*>(index.internalPointer());
    return item->data(index.column(), role);
}

bool AuthorizationsModel::hasChildren(const QModelIndex &parent) const
{
    if (!parent.isValid()) {
        return !rootItem->children().isEmpty();
    }
    AbstractItem *item = static_cast<AbstractItem*>(parent.internalPointer());
    return item->isGroup();
}

void AuthorizationsModel::addEntries(const QList<PolKitPolicyFileEntry *> &entries)
{
    foreach(PolKitPolicyFileEntry *entry, entries) {
        kDebug() << "Inserting" << polkit_policy_file_entry_get_id(entry);

        QString entryId     = polkit_policy_file_entry_get_id(entry);
        QString entryString = entryId;
        QStringList entryPieces = entryString.split('.');
        QString rootString = entryPieces.at(0) + '.' +
                             entryPieces.at(1) + '.' +
                             entryPieces.at(2);
        entryString.remove(rootString + '.');

        //Lets first find out if the group exists
        //if so let's try to find if the action exists
        bool itemInsertedOrChanged = false;
        foreach(AbstractItem *item, rootItem->children()) {
            if (item->isGroup() && static_cast<Group*>(item)->data(0, Qt::UserRole) == rootString) {
                kDebug() << "Found a Group";
                Group *group = static_cast<Group*>(item);
                for (int i = 0; i < group->children().count(); i++) {
                    Item *subItem = (Item *) group->children().at(i);
                    if (subItem->entryId() == entryId) {
                        kDebug() << "Found the Item";
                        subItem->setEntry(entry);
//                         emit dataChanged();TODO
                        itemInsertedOrChanged = true;
                        // break cause we already found the item
                        break; // better that we break here.
                    }
                }
                // if the item was NOT changed we can add it.
                if (!itemInsertedOrChanged) {
                    kDebug() << "It's a new Item";
                    beginInsertRows(createIndex(group->children().count(), 0, group),
                                    group->children().count(), group->children().count());
                    group->appendChild(new Item(entry));
                    endInsertRows();
                    itemInsertedOrChanged = true;
                }
                // break cause we already found the group
                break;
            }
        }

        if (itemInsertedOrChanged) {
            continue;
        }

        kDebug() << "It's a new group, roots counting: " << rootItem->children().count();
        //no previous groups was found let's addone
        beginInsertRows(QModelIndex(), rootItem->children().count(), rootItem->children().count());
        QString vendorName = polkit_policy_file_entry_get_action_vendor(entry);
        if (vendorName.isEmpty())
            vendorName = rootString;
        Group *group = new Group(vendorName, rootString, rootItem);
        endInsertRows();

        kDebug() << "It's a new Item";
        //now lets add the item
        //CHECKME :P this don't look right.. NOTE: maybe with children().count() is better
        beginInsertRows(createIndex(group->children().count(), 0, group),
                        group->children().count(), group->children().count()); // so that we append
        group->appendChild(new Item(entry));
        endInsertRows();

    }

    //TODO see which items were deleted (need to read all already inserted items)

    // beginInsertRows(QModelIndex(), m_entries.count(), m_entries.count());
    // m_entries.append(new Item(entry));
    // endInsertRows();
    /*if (!parent) {
        parent = invisibleRootItem();
    }
    //m_entries.append(entry);
    QString mypath = polkit_policy_file_entry_get_id(entry);
    QStringList ep = mypath.split('.');
    QString root = ep.at(0)+'.'+ep.at(1)+'.'+ep.at(2);

    QList<QStandardItem*> list = findItems(root);

    QStandardItem *ritm;

    if (list.isEmpty()) {
        ritm = new QStandardItem(root);
        ritm->setData(root, IdRole);
        ritm->setData(QVariant::fromValue(entry), PolkitEntryRole);
        ritm->setData(Group, EntryTypeRole);
        parent->appendRow(ritm);
    } else {
        ritm = list.at(0);
    }

    for (int i=3; i<ep.count(); ++i) {
        QStandardItem *previous = 0;

        // I know, this is nasty, but my time is short and this thing works :D

        int r = 0;

        while (true) {
            QStandardItem *child = ritm->child(r);

            if (!child) {
                break;
            }

            if (child->data(IdRole).toString() == root) {
                previous = child;
                break;
            }

            ++r;
        }

        if (!previous) {
            QStandardItem *itm = new QStandardItem(ep.at(i));
            itm->setData(QVariant::fromValue(entry), PolkitEntryRole);
            if (i == ep.count() - 1) {
                kDebug() << "Setting" << ep.at(i) << "as an action with id" << mypath;
                itm->setData(mypath, IdRole);
                itm->setData(Action, EntryTypeRole);
                itm->setText( polkit_policy_file_entry_get_action_description(entry) );
            } else {
                itm->setData(root, IdRole);
                itm->setData(Group, EntryTypeRole);
            }

            ritm->appendRow(itm);

            ritm = itm;
        } else {
            ritm = previous;
        }
    }

    emit reset();*/
    //kDebug() << "Added entry to tree:" << m_tree->tree()["org"]->tree();
}

PolKitPolicyFileEntry* AuthorizationsModel::polkitFileEntry(const QModelIndex &index)
{
    return index.data(PolkitEntryRole).value<PolKitPolicyFileEntry*>();
}

AuthorizationsModel::EntryType AuthorizationsModel::entryType(const QModelIndex &index)
{
    //TODO all these items are sending the decoration role, that's why i added a new method for
    // entries id
    // imo data definition of data(INT, role) is wrong..
    kDebug() << "Entry int" << index.data(EntryTypeRole);
    return (EntryType)index.data(EntryTypeRole).toInt();
}

QString AuthorizationsModel::polkitId(const QModelIndex &index) const
{
    return index.data(IdRole).toString();
}

PolKitPolicyFileEntry* AuthorizationsModel::getPFEFromActionId(const QString &action_id)
{
    foreach(Item * item, items(rootItem)) {
        if (item->entryId() == action_id)
            return item->fileEntry();
    }
    return 0;
}

QList<Item*> AuthorizationsModel::items(Group * group)
{
    QList<Item*> list;

    if (!group) {
        group = dynamic_cast<Group*>(rootItem);
    }

    foreach(AbstractItem* item, group->children()) {
        if (item->isGroup()) {
            list << items(dynamic_cast<Group*>(item));
        } else {
            list.append(dynamic_cast<Item*>(item));
        }
    }
    return list;
}

QList<Group*> AuthorizationsModel::groups(Group * group)
{
    QList<Group*> list;

    if (!group) {
        group = dynamic_cast<Group*>(rootItem);
    }

    foreach(AbstractItem* item, group->children()) {
        if (item->isGroup()) {
            list << dynamic_cast<Group*>(item);
            list << groups(dynamic_cast<Group*>(item));
        }
    }
    return list;
}

QModelIndex AuthorizationsModel::indexFromId(const QString &id) const
{
    foreach(const QModelIndex &index, itemsAsModelIndex(QModelIndex())) {
        kDebug() << index << index.data(IdRole);
        if (index.data(IdRole) == id) {
            return index;
        }
    }
    return QModelIndex();
}

QModelIndexList AuthorizationsModel::itemsAsModelIndex(const QModelIndex &parent) const
{
    kDebug() << parent;
    QModelIndexList list;
    for (int i = 0; i != rowCount(parent); ++i) {
        if (hasChildren(index(i, 0, parent))) {
            kDebug() << "Children found, iterating";
            list << itemsAsModelIndex(sibling(i, 0, parent));
        } else {
            kDebug() << "That should be an item";
            list << index(i, 0, parent);
        }
    }
    return list;
}

// YourType XXX::depthVisit(QList<YourType> &list, Node* rootNode)
// {
//         foreach(Node* child, rootNode->getChilds())
//         {
//                 list << printProperties(list, child);
//         }
//         return rootNode->myData();
// }

}

#include "AuthorizationsModel.moc"
