/*
 * KMLOFax
 *
 * A utility to process facsimiles received with a modem of the
 * ELSA MicroLink(tm) Office family.
 *
 * Copyright (C) 1999,2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen, Germany.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <qfont.h>
#include <qglobal.h>

#include <kapplication.h>
#include <klocale.h>
#include <kiconloader.h>
#include <ksimpleconfig.h>
#include <kconfig.h>

#include "faxview.h"
#include "folderview.h"
#include "senderaliases.h"
#include "global.h"



FaxViewItem::FaxViewItem(QListView *parent, const char *name): KListViewItem(parent, name)
{
	m_pages = 0;
	m_size = 0;
	m_new = false;
}


FaxViewItem::~FaxViewItem()
{
}


void FaxViewItem::setName(const QString &name_)
{
	m_name = name_;
}


void FaxViewItem::setTime(const QDateTime &time_)
{
	m_time = time_;
	setText(0, KGlobal::locale()->formatDate(time_.date(), true) + " (" + KGlobal::locale()->formatTime(time_.time(), true) + ")");
}


void FaxViewItem::setSender(const QString &sender_)
{
	m_sender = sender_;
	refreshAlias();
}


void FaxViewItem::setPages(int pages_)
{
	QString s;
	
	m_pages = pages_;
	setText(2, s.setNum(pages_));
}


void FaxViewItem::setSize(uint size_)
{
	m_size = size_;
	setText(3, KGlobal::locale()->formatNumber((double)size_, 0));
}


void FaxViewItem::setNew(bool new_)
{
	m_new = new_;
	repaint();
}


void FaxViewItem::setParams(const QString &name_, const QDateTime &time_, const QString &sender_, int pages_, uint size_, bool new_)
{
	setName(name_);
	setTime(time_);
	setSender(sender_);
	setPages(pages_);
	setSize(size_);
	setNew(new_);
}


void FaxViewItem::refreshAlias()
{
	setText(1, senderaliases->alias(m_sender));
}


QString FaxViewItem::key(int column, bool) const
{
	QString s;

	if (column == 0) {
		s.sprintf("%04d%02d%02d%02d%02d%02d", m_time.date().year(), m_time.date().month(), m_time.date().day(), m_time.time().hour(), m_time.time().minute(), m_time.time().second());
		return s;
	}
	if (column == 2) {
		s.sprintf("%10d", m_pages);
		return s;
	}
	if (column == 3) {
		s.sprintf("%10d", m_size);
		return s;
	}

	return text(column);
}


void FaxViewItem::paintCell(QPainter *p, const QColorGroup& cg, int column, int width, int alignment)
{
	QColorGroup cg2(cg);

	if (m_new) {
		QFont f = p->font();
		f.setBold(true);
		p->setFont(f);
	}

	KListViewItem::paintCell(p, cg2, column, width, alignment);
}



FaxDrag::FaxDrag(QWidget *parent, const char *name): QStoredDrag("faxdrag/magic", parent, name)
{
}


FaxDrag::~FaxDrag()
{
}


bool FaxDrag::canDecode(QDragMoveEvent *e)
{
	return e->provides("faxdrag/magic");
}



FaxView::FaxView(QWidget *parent, const char *name) : KListView(parent, name)
{
	has_selection = false;
	m_drag = false;
	m_dragpos = QPoint(0, 0);

	config = kapp->config();

	addColumn(i18n("Date (Time)"), 0);
	addColumn(i18n("Sender"), 0);
	addColumn(i18n("Pages"), 0);
	addColumn(i18n("Size"), 0);
	
	setColumnAlignment(0, AlignVCenter | AlignHCenter);
	setColumnAlignment(2, AlignVCenter | AlignHCenter);
	setColumnAlignment(3, AlignVCenter | AlignRight);
	
	setMinimumSize(sizeHint());
	setSelectionModeExt(Extended);
	setSorting(0, false);
	setAllColumnsShowFocus(true);
	setShowSortIndicator(true);

	context_menu = new QPopupMenu(this, "contextmenu");
	Q_CHECK_PTR(context_menu);
	context_menu->insertItem(SmallIcon("viewmag"), i18n("&View..."), ID_FILE_VIEW);
	context_menu->insertItem(SmallIcon("fileprint"), i18n("&Print..."), ID_FILE_PRINT);
	context_menu->insertItem(SmallIcon("filesave"), i18n("&Export..."), ID_FILE_EXPORT);
	context_menu->insertItem(SmallIcon("mail_send"), i18n("&Mail..."), ID_FILE_MAIL);
	context_menu->insertItem(SmallIcon("trashcan_empty"), i18n("Move to &Trash"), ID_FILE_TRASH);
	context_menu->insertItem(i18n("&Sender Alias..."), ID_FILE_SENDER_ALIAS);
	context_menu->insertSeparator();
	context_menu->insertItem(i18n("Move To"), moveMenu);
	context_menu->insertItem(i18n("Copy To"), copyMenu);
	
	connect(this, SIGNAL(currentChanged(QListViewItem *)), SLOT(currentChangedSlot(QListViewItem *)));
	connect(this, SIGNAL(selectionChanged()), SLOT(selectionChangedSlot()));
	connect(this, SIGNAL(doubleClicked(QListViewItem *)), SLOT(executedSlot(QListViewItem *)));
	connect(this, SIGNAL(returnPressed(QListViewItem *)), SLOT(executedSlot(QListViewItem *)));
	connect(this, SIGNAL(rightButtonPressed(QListViewItem *, const QPoint&, int)), SLOT(rightButtonPressedSlot(QListViewItem *, const QPoint&, int)));

	connect(context_menu, SIGNAL(activated(int)), SLOT(activatedSlot(int)));

	connect(senderaliases, SIGNAL(applyButtonPressed()), SLOT(refreshAliases()));

	setMinimumSize(40, 30);

	readOptions();
}


FaxView::~FaxView()
{
}


void FaxView::readOptions()
{
	config->setGroup("FaxWindow Options");
	
	setColumnWidth(0, config->readNumEntry("DateWidth", 140));
	setColumnWidth(1, config->readNumEntry("SenderWidth", 160));
	setColumnWidth(2, config->readNumEntry("PagesWidth", 50));
	setColumnWidth(3, config->readNumEntry("SizeWidth", 60));
}


void FaxView::saveOptions()
{
	config->setGroup("FaxWindow Options");
	
	config->writeEntry("DateWidth", columnWidth(0));
	config->writeEntry("SenderWidth", columnWidth(1));
	config->writeEntry("PagesWidth", columnWidth(2));
	config->writeEntry("SizeWidth", columnWidth(3));
}


void FaxView::insertFaxItem(const QString &name, const QDateTime &time, const QString &sender, int pages, uint size, bool isnew)
{
	FaxViewItem *item;
	
	item = new FaxViewItem(this);
	item->setParams(name, time, sender, pages, size, isnew);
}


void FaxView::takeFaxItem(const QString &name, const QDateTime &time, const QString &sender, int pages, uint size, bool isnew)
{
	FaxViewItem *item;

	for (item = (FaxViewItem *)firstChild(); (item); item = (FaxViewItem *)item->nextSibling())
		if (item->name() == name)
			break;
	
	if (item)
		item->setParams(name, time, sender, pages, size, isnew);
	else
		insertFaxItem(name, time, sender, pages, size, isnew);
}


void FaxView::deleteFaxItem(const QString &name)
{
	FaxViewItem *item;

	for (item = (FaxViewItem *)firstChild(); (item); item = (FaxViewItem *)item->nextSibling())
		if (item->name() == name)
			break;

	if (item)
		delete item;
}


void FaxView::setCurrentFaxNew(bool isnew)
{
	if (currentItem())
		((FaxViewItem *)currentItem())->setNew(isnew);
}


bool FaxView::currentFaxNew()
{
	if (currentItem())
		return ((FaxViewItem *)currentItem())->isNew();

	return false;
}


QString FaxView::currentFaxName()
{
	if (currentItem())
		return ((FaxViewItem *)currentItem())->name();

	return QString::null;
}


QString FaxView::currentFaxSender()
{
	if (currentItem())
		return ((FaxViewItem *)currentItem())->sender();

	return QString::null;
}


bool FaxView::faxNew(const QString &name)
{
	FaxViewItem *item;

	for (item = (FaxViewItem *)firstChild(); (item); item = (FaxViewItem *)item->nextSibling())
		if (item->name() == name)
			return item->isNew();

	return true;
}


void FaxView::rightButtonPressedSlot(QListViewItem *, const QPoint &point, int)
{
	if (has_selection)
		context_menu->popup(point);
}


void FaxView::currentChangedSlot(QListViewItem *item)
{
	emit hasCurrent(item != 0);
}


void FaxView::selectionChangedSlot()
{
	QListViewItem *item;

	has_selection = false;

	for (item = firstChild(); (item); item = item->nextSibling())
		if (item->isSelected()) {
			has_selection = true;
			break;
		}

	emit hasSelection(has_selection);
}


void FaxView::executedSlot(QListViewItem *)
{
	emit command(ID_FILE_VIEW);
}


void FaxView::activatedSlot(int id_)
{
	emit command(id_);
}


void FaxView::refreshAliases()
{
	FaxViewItem *item;

	for (item = (FaxViewItem *)firstChild(); (item); item = (FaxViewItem *)item->nextSibling())
		item->refreshAlias();
}


void FaxView::contentsMouseMoveEvent(QMouseEvent *e)
{
	if (m_drag && ((e->pos() - m_dragpos).manhattanLength() > 4)) {
		m_drag = false;
		FaxDrag *d = new FaxDrag(this);
		d->drag();
	}
}

void FaxView::contentsMousePressEvent(QMouseEvent *e)
{
	if ((e->button() == LeftButton) && !(e->state() & ControlButton) && !(e->state() & ShiftButton)) {
		if (itemAt(contentsToViewport(e->pos()))) {
			m_drag = true;
			m_dragpos = e->pos();
		}
	}
	KListView::contentsMousePressEvent(e);
}


void FaxView::contentsMouseReleaseEvent(QMouseEvent *e)
{
	KListView::contentsMouseReleaseEvent(e);
	m_drag = false;
}
