/*
 * KMLOFax
 *
 * A utility to process facsimiles received with a modem of the
 * ELSA MicroLink(tm) Office family.
 *
 * Copyright (C) 1999,2000,2001,2002 Oliver Gantz <Oliver.Gantz@epost.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ------
 * ELSA and MicroLink are trademarks of ELSA AG, Aachen, Germany.
 */

#ifndef FAXVIEW_H 
#define FAXVIEW_H 

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif 

#include <qstring.h>
#include <qdatetime.h>
#include <qevent.h>
#include <qdragobject.h>
#include <qwidget.h>
#include <qpoint.h>
#include <qpopupmenu.h>
#include <qpainter.h>
#include <qpalette.h>

#include <klistview.h>
#include <kconfig.h>


/**
 * This object represent a single item in the facsimile list.
 *
 * It is used with the @ref FaxView widget.
 */
class FaxViewItem: public KListViewItem
{
public:
	/**
	 * Constructor
	 *
	 * The parameters @p parent and @p name are handled by @ref QListViewItem, as usual.
	 */
	FaxViewItem(QListView *parent = 0, const char *name = 0);
	
	/**
	 * Destructor
	 */
	virtual ~FaxViewItem();
	
	/**
	 * Sets the item's name (the filename).
	 *
	 * @see name()
	 */
	void setName(const QString &name_);

	/**
	 * @return the item's name (the filename).
	 *
	 * @see setName()
	 */
	QString name();
	
	/**
	 * Sets the item's date and time.
	 *
	 * @see time()
	 */
	void setTime(const QDateTime &time_);
	
	/**
	 * @return the item's date and time.
	 *
	 * @see setTime()
	 */
	QDateTime time();

	/**
	 * Sets the item's sender id. The alias of the id is displayed instead.
	 *
	 * @see sender()
	 */
	void setSender(const QString &sender_);
	
	/**
	 * @return the item's sender id (not the alias!).
	 *
	 * @see setSender()
	 */
	QString sender();
	
	/**
	 * Sets the item's number of pages.
	 *
	 * @see pages()
	 */
	void setPages(int pages_);
	
	/**
	 * @return the item's number of pages.
	 *
	 * @see setPages()
	 */
	int pages();
	
	/**
	 * Sets the item's file size.
	 *
	 * @see size()
	 */
	void setSize(uint size_);
	
	/**
	 * @return the item's file size.
	 *
	 * @see setSize()
	 */
	uint size();

	/**
	 * Marks the item as new (with green tick) or old (with red cross) depending on @p new_.
	 *
	 * @see isNew()
	 */
	void setNew(bool new_);
	
	/**
	 * @return true if the item is marked as new.
	 *
	 * @see setNew()
	 */
	bool isNew();
	
	/**
	 * Sets all parameters of the item.
	 */
	void setParams(const QString &name_, const QDateTime &time_, const QString &sender_, int pages_, uint size_, bool new_);

	/**
	 * Reads and sets the current alias of the item's sender id.
	 *
	 * Sould be called every time the alias list has been modified.
	 */
	void refreshAlias();

	/**
	 * Reimplemation of the key function to get a correct sort order for all columns.
	 */
	virtual QString key(int column, bool) const;

	virtual void paintCell(QPainter *p, const QColorGroup& cg, int column, int width, int alignment);

private:
	QString m_name;
	QDateTime m_time;
	QString m_sender;
	int m_pages;
	uint m_size;
	bool m_new;
};


inline QString FaxViewItem::name()
{ return m_name; }

inline QDateTime FaxViewItem::time()
{ return m_time; }

inline QString FaxViewItem::sender()
{ return m_sender; }

inline int FaxViewItem::pages()
{ return m_pages; }

inline uint FaxViewItem::size()
{ return m_size; }

inline bool FaxViewItem::isNew()
{ return m_new; }



class FaxDrag: public QStoredDrag
{
public:
	FaxDrag(QWidget *parent = 0, const char *name = 0);
	~FaxDrag();

	static bool canDecode(QDragMoveEvent *e);
};



/**
 * This widgets implements the list of facsimiles.
 */
class FaxView: public KListView
{
	Q_OBJECT
public:
	/**
	 * Constructor
	 *
	 * The parameters @p parent and @p name are handled by @ref KListView, as usual.
	 */
	FaxView(QWidget *parent = 0, const char *name = 0);
	
	/**
	 * Destructor
	 */
	virtual ~FaxView();

	/**
	 * Reads the saved window settings for this widget.
	 */
	void readOptions();

	/**
	 * Saves the current window settings.
	 */
	void saveOptions();
	
	/**
	 * Inserts a new facsimile item into the list without considering if the name exist in the list.
	 *
	 * The resource file of this folder is untouched.
	 *
	 * @see takeFaxItem()
	 */
	void insertFaxItem(const QString &name, const QDateTime &time, const QString &sender, int pages, uint size, bool isnew);

	/**
	 * If an item with @p name already exists in the list, the parameters are replaced by the given ones.
	 * Otherwise a new item is inserted into the list.
	 *
	 * The resource file of this folder is untouched.
	 *
	 * @see insertFaxItem()
	 */
	void takeFaxItem(const QString &name, const QDateTime &time, const QString &sender, int pages, uint size, bool isnew);

	/**
	 * Removes an item from the list.
	 *
	 * The resource file of this folder is untouched.
	 */
	void deleteFaxItem(const QString &name);
	
	/**
	 * Sets the highlighted facsimile as new or not new.
	 *
	 * @see currentFaxNew()
	 */
	void setCurrentFaxNew(bool isnew);
	
	/**
	 * @return true if the highlighted facsimile is marked as new.
	 *
	 * @see setCurrentFaxNew()
	 */
	bool currentFaxNew();
	
	/**
	 * @return the name of the currently highlighted facsimile item.
	 */
	QString currentFaxName();
	
	/**
	 * @return the sender of the currently highlighted facsimile item.
	 */
	QString currentFaxSender();
	
	/**
	 * @return true if the facsimile with @p name is marked as new.
	 */
	bool faxNew(const QString &name);


private slots:
  /**
   * Open the context menu with right mouse button.
   */
	void rightButtonPressedSlot(QListViewItem *item, const QPoint& point, int);
	
	/**
	 * Changes the currently highlighted item and emits @ref hasCurrent().
	 */
	void currentChangedSlot(QListViewItem *);

	/**
	 * Changes the currently selected item and emits @ref hasSelection().
	 */
	void selectionChangedSlot();

	/**
	 * Emits @ref command() with ID_FILE_VIEW
	 */
	void executedSlot(QListViewItem *item);

	/**
	 * Emits @ref command() with @p id_.
	 */
	void activatedSlot(int id_);

	/**
	 * Reads and sets the current aliases of all items' sender ids.
	 */
	void refreshAliases();

protected:
	virtual void contentsMouseMoveEvent(QMouseEvent *e);
	virtual void contentsMousePressEvent(QMouseEvent *e);
	virtual void contentsMouseReleaseEvent(QMouseEvent *e);

signals:
	/**
	 * This signal is emitted when ever the selection of items changes.
	 * @p yesno is true if one or more items are selected.
	 */
	void hasCurrent(bool yesno);

	/**
	 * This signal is emitted when ever the selection of items changes.
	 * @p yesno is true if one or more items are selected.
	 */
	void hasSelection(bool yesno);

	/**
	 * This signal is emitted when ever a command is requested by clicking/doubleclicking on
	 * an item or by the context menu.
	 * @p id_ is the ID of the requested command.
	 */
	void command(int id_);
	
private:
	KConfig *config;
	QPopupMenu *context_menu;
	bool has_selection;
	bool m_drag;
	QPoint m_dragpos;
};


#endif // FAXVIEW_H
