/*
 *
 * Copyright (C) 2002 Richard Moore, <rich@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */


#include <qcheckbox.h>
#include <qfont.h>
#include <qpalette.h>
#include <qpoint.h>
#include <qslider.h>
#include <qspinbox.h>
#include <qtimer.h>
#include <qwidget.h>

#include <kconfig.h>
#include <kdebug.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kpopupmenu.h>

#include "elegant.h"
#include "elegant.moc"

KdetvElegant::KdetvElegant(Kdetv *ktv, QWidget *p, const char *name)
    : KdetvOSDPlugin( ktv, "elegant-osd", p, name ),
      parent(p),
      align(AlignBottom|AlignCenter), indent(20),
      iconSize(KIcon::SizeLarge), fontSize(48),
      fontBold(true), fontItalic(false), fontUnderline(false), autoFontSize(false)
{
    timer = new QTimer( this );
    popup = new KPopupTitle( p, "display" );
    popup->hide();

    // FIXME: Don't use system colors, they may not work or look bad as OSD
    popup->setPaletteBackgroundColor( QColor( 200, 200, 200 ) );

    readConfig();
    applySettings();

    connect( timer, SIGNAL( timeout() ), popup, SLOT( hide() ) );
    connect( parent, SIGNAL( resized(int,int) ), this, SLOT( viewResized(int,int) ) );

    viewResized(p->width(), p->height());
}

KdetvElegant::~KdetvElegant()
{
    kdDebug() << "Elegant: destroying plugin." << endl;
}

void KdetvElegant::applySettings()
{
    QFont font = popup->font();
    font.setPointSize( fontSize );
    font.setBold( fontBold );
    font.setItalic( fontItalic );
    font.setUnderline( fontUnderline );
    popup->setFont( font );

    updateGeometry();
}

void KdetvElegant::readConfig()
{
    kdDebug() << "Elegant: reading config" << endl;
    
    _cfg->setGroup( "Elegant OSD");
    timeout = _cfg->readNumEntry( "Timeout", 2000 );
    indent = _cfg->readNumEntry( "Indent", 20 );
    align = _cfg->readNumEntry( "Alignment", AlignBottom|AlignCenter );
    iconSize = _cfg->readNumEntry( "IconSize", KIcon::SizeLarge );
    fontSize = _cfg->readNumEntry( "FontSize", 48 );
    fontBold = _cfg->readBoolEntry( "FontBold", true );
    fontItalic = _cfg->readBoolEntry( "FontItalic", false );
    fontUnderline = _cfg->readBoolEntry( "FontUnderline", false );
    autoFontSize = _cfg->readBoolEntry( "AutoFontSize", true );
}

void KdetvElegant::saveConfig()
{
    applyConfigWidget();

    kdDebug() << "Elegant: writing config" << endl;
    
    _cfg->setGroup( "Elegant OSD");
    _cfg->writeEntry( "Timeout", timeout );
    _cfg->writeEntry( "Indent", indent );
    _cfg->writeEntry( "Alignment", align );
    _cfg->writeEntry( "IconSize", iconSize );
    _cfg->writeEntry( "FontSize", fontSize );
    _cfg->writeEntry( "FontBold", fontBold );
    _cfg->writeEntry( "FontItalic", fontItalic );
    _cfg->writeEntry( "FontUnderline", fontUnderline );
    _cfg->writeEntry( "AutoFontSize", autoFontSize );
    _cfg->sync();
}

void KdetvElegant::updateGeometry()
{
    popup->resize( popup->minimumSizeHint() );

    int x;
    if ( align & AlignLeft )
        x = indent;
    else if ( align & AlignRight )
        x = parent->width() - indent - popup->width();
    else
        x = parent->width()/2 - popup->width()/2;

    int y;
    if ( align & AlignTop )
        y = indent;
    else if ( align & AlignBottom )
        y = parent->height() - indent - popup->height();
    else
        y = parent->height()/2 - popup->height()/2;

    popup->move( QPoint(x,y) );
}

void KdetvElegant::clear()
{
    static QPixmap kwicon = KGlobal::iconLoader()->loadIcon("kdetv", KIcon::NoGroup, iconSize);
    popup->setTitle(i18n("Elegant OSD"), &kwicon);
}

void KdetvElegant::showPopup()
{
    if ( popup->isHidden() )
        popup->show();
    else {
        popup->update();
        popup->raise();
    }

    updateGeometry();
    timer->start(timeout, true);
}

void KdetvElegant::display( const QString &text, const QPixmap &icon )
{
    popup->setTitle(text, &icon);
    showPopup();
}

void KdetvElegant::display( const QString &text, const QString &icon )
{
    QString i = icon.isNull() ? QString("kdetv") : icon;
    display( text, KGlobal::iconLoader()->loadIcon( i, KIcon::NoGroup, iconSize ) );
}

void KdetvElegant::displayMisc( const QString &text )
{
    display( text, "kdetv" );
}

void KdetvElegant::displayChannel( int channel, const QString &name )
{
    display( i18n("%1 - %2").arg(channel).arg(name), "kdetv" );
}

void KdetvElegant::displayMuted( bool muted )
{
    if ( muted )
        display( i18n("Mute"), "kdetv_muteon" );
    else
        display( i18n("Unmute"), "kdetv_volup" );
}

void KdetvElegant::displayVolume( int vol )
{
    display( i18n("Volume: %1%").arg(vol), "kdetv_volup" );
}

void KdetvElegant::displayCC( const QString &text )
{
    if ( text.isEmpty() )
        popup->hide();
    else
        display( text );
}

void KdetvElegant::viewResized( int w, int /*h*/ )
{
    if (!autoFontSize)
        return;
    QFont font = popup->font();
    font.setPixelSize(w / 15);
    popup->setFont( font );
    kdDebug() << "elegant font resized to " << w / 15 << endl;
    if ( !popup->isHidden() ) {
        updateGeometry();
    }
}

ElegantConfigWidget *KdetvElegant::configWidget( QWidget *parent, const char *name )
{
    form = new ElegantConfigWidget( parent, name );

    if ( align & AlignLeft )
        form->HSlider->setValue(0);
    else if ( align & AlignRight )
        form->HSlider->setValue(2);
    else
        form->HSlider->setValue(1);

    if ( align & AlignTop )
        form->VSlider->setValue(0);
    else if ( align & AlignBottom )
        form->VSlider->setValue(2);
    else
        form->VSlider->setValue(1);

    form->TimeSpin->setValue( timeout );
    form->MarginSpin->setValue( indent );
    form->SizeSpin->setValue( fontSize );
    form->BoldCheck->setChecked( fontBold );
    form->ItalicCheck->setChecked( fontItalic );
    form->UnderlineCheck->setChecked( fontUnderline );
    form->AutoFontSizeCheck->setChecked( autoFontSize );

    return form;
}

void KdetvElegant::applyConfigWidget()
{
    timeout = form->TimeSpin->value();
    indent = form->MarginSpin->value();
    fontSize = form->SizeSpin->value();
    iconSize = form->SizeSpin->value();
    fontBold = form->BoldCheck->isChecked();
    fontItalic = form->ItalicCheck->isChecked();
    fontUnderline = form->UnderlineCheck->isChecked();
    autoFontSize = form->AutoFontSizeCheck->isChecked();

    switch ( form->HSlider->value() ) {
	case 0:
	    align = AlignLeft;
	    break;
	case 1:
	    align = AlignHCenter;
	    break;
	case 2:
	default:
	    align = AlignRight;
	    break;
    }
    switch ( form->VSlider->value() ) {
	case 0:
	    align = align | AlignTop;
	    break;
	case 1:
	    align = align | AlignVCenter;
	    break;
	case 2:
	default:
	    align = align | AlignBottom;
	    break;
    }

    applySettings();
}


extern "C" {
    KdetvElegant* create_elegant( Kdetv *ktv, QWidget *parent )
    {
        if ( !parent )
            return 0;

        return new KdetvElegant( ktv, parent );
    }
}

#ifdef INCLUDE_TESTS

static int test_chan_num = 0;
static int test_muted = false;
static int test_vol = 0;

void KdetvElegant::test_channel()
{
    displayChannel( test_chan_num, QString("Channel %1").arg(test_chan_num) );
    test_chan_num++;
}

void KdetvElegant::test_mute()
{
    displayMuted( test_muted );
    test_muted = !test_muted;
}

void KdetvElegant::test_volume()
{
    displayVolume( test_vol );
    test_vol = test_vol+10;
}

void KdetvElegant::test_config()
{
    QWidget *w = configWidget( 0, "conf" );
    w->show();
}
#endif // INCLUDE_TESTS
