package tijmp;

import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.lang.management.MemoryPoolMXBean;
import java.lang.management.ThreadMXBean;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.LinkedBlockingQueue;
import tijmp.UIHandler;
import tijmp.ui.InVMUI;

/** This is a class that can be used to control the tijmp profiler from within java code.
 */
public class TIJMPController {
    private static List<HeapWalkEntry> previousEntries;
    private static LinkedBlockingQueue<Runnable> lbq;
    private static UIHandler uiHandler;
    private static boolean showStrings = false;

    static {
	System.loadLibrary ("tijmp");
    }

    /** This will run the garbage collector. 
     *  According to the specification this type of gc will 
     *  actually be guaranteed to run the garbage collector.
     */
    static native void runGC ();

    /** Walk the heap and gather object information.
     */
    static native void walkHeap ();

    /** Find all entries of a given class and then call back into java to 
     *  show them.
     */
    static native void showInstances (Class<?> clz);

    /** Finds all strings and then call back to java to show them.
     */
    static void showStrings () {
	showStrings = true;
	showInstances (char[].class);
    }

    /** Find owners for all objects of the given class and show them.
     */
    static native void showOwners (Class<?> clz);
    
    /** Find all objects below the given object and show a summary.
     */
    static native void childObjectsSummary (Object o);

    /** Find the objects for a set of tags. 
     */
    static native Object[] getObjectsForTags (long[] tags);
    
    public static void init () {
	lbq = new LinkedBlockingQueue<Runnable> ();
	Thread runner = new Thread (new EventHandler (lbq));
	runner.start ();
	uiHandler = new InVMUI ();
	uiHandler.init (new InVMPH ());
    }

    /** Submit a long running task to the tijmp task handler.
     */
    static void submitTask (Runnable r) {
	try {
	    lbq.put (r);
	} catch (InterruptedException e) {
	    e.printStackTrace ();
	}
    }
    
    public static void heapWalkResult (Class<?>[] carr, long[] counts, 
				       long[] sizes) {
	Map<Class<?>, HeapWalkEntry> m = Collections.emptyMap ();
	if (previousEntries != null) {
	    m = new HashMap<Class<?>, HeapWalkEntry> (previousEntries.size ());
	    for (HeapWalkEntry e : previousEntries) 
		m.put (e.getEntryClass (), e);
	}
	final List<HeapWalkEntry> ls = new ArrayList<HeapWalkEntry> (carr.length);
	for (int i = 0; i < carr.length; i++) {
	    // for now we ignore null entries and also no-count entries.
	    if (carr[i] != null && counts[i] > 0) {
		HeapWalkEntry pe = m.get (carr[i]);
		long countChange = 0;
		long sizeChange = 0;
		if (pe != null) {
		    countChange = counts[i] - pe.getInstanceCount ();
		    sizeChange = sizes[i] - pe.getTotalSize ();
		} 
		ls.add (new HeapWalkEntry (carr[i], counts[i], countChange, 
					   sizes[i], sizeChange));
	    }
	}
	previousEntries = ls;
	uiHandler.showHeapWalkResult (ls);
    }

    public static void instances (Class<?> clz, Object[] objects,
				  long[] sizes, int[] lengths) {
	if (showStrings)
	    uiHandler.strings (objects);
	else
	    uiHandler.instances (clz, objects, sizes, lengths);
	showStrings = false;
    }

    public static void childObjects (Object[] childs) {
	uiHandler.childObjects (childs);
    }

    public static void owners (Map<Long, OwnerInfoHeader> owners, 
			       long[] startObjects) {
	uiHandler.owners (owners, startObjects);
    }

    public static void setStatus (String status) {
	uiHandler.showStatus (status);
    }
}

class InVMPH implements ProfilerHandler {
    public void submitTask (Runnable r) {
	TIJMPController.submitTask (r);
    }
    
    public void runGC () {
	TIJMPController.runGC ();
    }

    public void walkHeap () {
	TIJMPController.walkHeap ();
    }

    public void childObjectsSummary (Object o) {
	TIJMPController.childObjectsSummary (o);
    }

    public Object[] getObjectsForTags (long[] tags) {
	return TIJMPController.getObjectsForTags (tags);
    }

    public void showInstances (Class<?> clz) {
	TIJMPController.showInstances (clz);
    }

    public void showStrings () {
	TIJMPController.showStrings ();
    }

    public void showOwners (Class<?> clz) {
	TIJMPController.showOwners (clz);
    }

    public MemoryMXBean getMemoryMXBean () {
	return ManagementFactory.getMemoryMXBean ();
    }

    public List<MemoryPoolMXBean> getMemoryPoolMXBeans () {
	return ManagementFactory.getMemoryPoolMXBeans ();
    }

    public ThreadMXBean getThreadMXBean () {
	return ManagementFactory.getThreadMXBean ();
    }
}
