!{\src2tex{textfont=tt}}
!!****f* ABINIT/vhrtre
!! NAME
!! vhrtre
!!
!! FUNCTION
!! Finds the Hartree potential created by a radial electron density,
!! using Numerov's method to integrate the radial Poisson equation:
!!   d2(r*V)/dr2 = -4*pi*rho*r = -(4*pi*r2*rho)/r
!! Algorithm: see routine NUMOUT
!!
!! COPYRIGHT
!! Copyright (C) 2005-2007 ABINIT group (JJ)
!!
!! INPUTS
!!   real*8  R2RHO(NR) : 4*pi*r**2*rho, with rho the electron density
!!   real*8  R(NR)     : Logarithmic radial mesh R(i)=B*(exp(A*(i-1)-1)
!!   real*8  DRDI(NR)  : dr/di at the mesh points
!!   real*8  SRDRDI(NR): sqrt(dr/di) at the mesh points
!!   integer NR        : Number of radial mesh points, including r(1)=0
!!   real*8  A         : The parameter A in r(i)=B*(exp(A*(i-1)-1)
!!  (to be completed)
!!
!! OUTPUT
!!   real*8  V(NR)     : Electrostatic potential created by rho, in Ryd
!!                       The constants of integration are fixed so that
!!                       V=finite at the origin and V(NR)=Q/R(NR),
!!                       where Q is the integral of rho up to R(NR)
!!  (to be completed)
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine vhrtre(R2RHO,V,R,DRDI,SRDRDI,NR,A)

 use defs_basis

 implicit none

!Arguments ----------------------------------
!scalars
 integer,intent(in) :: nr
 real(dp),intent(in) :: a
!arrays
 real(dp),intent(in) :: drdi(nr),r(nr),r2rho(nr),srdrdi(nr)
 real(dp),intent(out) :: v(nr)

!Local variables ----------------------------
!scalars
 integer :: ir
 real(dp) :: a2by4,beta,dv,dy,dz,q,qbyy,qpartc,qt,t,v0,y,ybyq

! *************************************************************************

! Find some constants
 A2BY4 = A * A / 4.D0
 YBYQ  = 1.D0 - A * A / 48.D0
 QBYY  = 1.D0 / YBYQ

! Use Simpson's rule to find the total charge QT, and the
! potential at the origin V0:
! QT = Int(4*pi*r**2*rho*dr) = Int((4*pi*r**2*rho)*(dr/di)*di)
! V0 = Int(4*pi*r*rho*dr) =  Int((4*pi*r**2*rho)/r*(dr/di)*di)
 V0 = 0.D0
 QT = 0.D0
 DO IR = 2, NR-1, 2
   DZ = DRDI(IR) * R2RHO(IR)
   QT = QT + DZ
   V0 = V0 + DZ / R(IR)
 end do
 V0 = V0 + V0
 QT = QT + QT
 DO IR = 3, NR-2, 2
   DZ = DRDI(IR) * R2RHO(IR)
   QT = QT + DZ
   V0 = V0 + DZ / R(IR)
 end do
 DZ =DRDI(NR) * R2RHO(NR)
 QT =( QT + QT + DZ ) / 3.D0
 V0 =( V0 + V0 + DZ / R(NR) ) / 3.D0

! Fix V(1) and V(2) to start Numerov integration. To find a
! particular solution of the inhomog. eqn, V(2) is fixed by
! setting rV(2)=0. Notice that V=finite => rV=0 at r=0
 V(1)=2.D0*V0    ! Factor 2 because we use Rydbergs
 T    = SRDRDI(2) / R(2)
 BETA = DRDI(2) * T * R2RHO(2)
 DY   = 0.D0
 Y    = 0.D0
 Q    = ( Y - BETA / 12.D0 ) * QBYY
 V(2) = 2.D0 * T * Q

! Integrate Poisson's equation outwards, using Numerov's method
 DO IR = 3,NR
   DY    = DY + A2BY4 * Q - BETA
   Y     = Y + DY
   T     = SRDRDI(IR) / R(IR)
   BETA  = T * DRDI(IR) * R2RHO(IR)
   Q     = ( Y - BETA / 12.D0 ) * QBYY
   V(IR) = 2.D0 * T * Q
 END DO

! Add a solution (finite at r=0) of the homogeneous equation
! d2(r*V)/dr2=0 => rV=const*r => V=const, to ensure that
! V(NR)=Q/R(NR). Notice that V(1) is set independently
 QPARTC = R(NR) * V(NR) / 2.D0
 DZ = QT - QPARTC
 DV = 2.D0 * DZ / R(NR)
 DO IR = 2, NR
   V(IR) = V(IR) + DV
 end do

end subroutine vhrtre
!!***
