/* Copyright (C) 2007 One Laptop Per Child
 * Author: Marc Maurer <uwog@uwog.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <gsf/gsf-input-memory.h>
#include <libxml/tree.h>

#include "ut_assert.h"
#include "ut_debugmsg.h"
#include "ut_go_file.h"

#include "LanguageDefinition.h"
#include "LanguagePattern.h"

LanguageDefinition* LanguageDefinition::construct(const string& fname)
{
	xxx_UT_DEBUGMSG(("LanguageDefinition::construct(fname: %s)\n", fname.c_str()));

	LanguageDefinition* pLang = NULL;
	
	GsfInput* in = NULL;
	char *uri = UT_go_filename_to_uri(fname.c_str());
	if (uri)
		in = UT_go_file_open(uri, NULL); // TODO: shouldn't use NULL here, but check for errors

	if (in)
	{
		guint8 const* contents = gsf_input_read(in, gsf_input_size(in), NULL);
		if (contents)
		{
			xmlDocPtr reader = xmlReadMemory(reinterpret_cast<const char*>(contents), 
								strlen(reinterpret_cast<const char*>(contents)), 0, "UTF-8", 0);
			if (reader)
			{
				xmlNode* node = xmlDocGetRootElement(reader);
				if (node)
				{
					if (strcmp(reinterpret_cast<const char*>(node->name), "language") == 0)	
					{
						char* name = (char *)xmlGetProp(node, (const xmlChar *)"name");
						if (!name) // no worries, _name is allowed too, meaning it is eligable for translation
							name = (char *)xmlGetProp(node, (const xmlChar *)"_name");

						char* section = (char *)xmlGetProp(node, (const xmlChar *)"section");
						if (!section) // no worries, _section is allowed too, meaning it is eligable for translation
							section = (char *)xmlGetProp(node, (const xmlChar *)"_section");

						if (name)
						{
							pLang = new LanguageDefinition(name, section);
							UT_DEBUGMSG(("Parsing language definition - name: %s, section: %s\n", name, section));
							FREEP(name);
							FREEP(section);
							
							// now parse all language patterns
							for (xmlNode* pattern = node->children; pattern; pattern = pattern->next)
							{
								if (pattern->type == XML_ELEMENT_NODE)
								{
									LanguagePattern::construct(pattern, pLang->getPatterns());
								}
							}
						}
					}
				}
				xmlFreeDoc(reader);
			}
		}
		g_object_unref(G_OBJECT(in));
	}
	
	return pLang;
}

LanguageDefinition::LanguageDefinition(const string& language, const string& section)
	: m_sLanguage(language),
	m_sSection(section)
{
}

void LanguageDefinition::addPattern(LanguagePattern* pPattern)
{
	UT_return_if_fail(pPattern);
	m_vPatterns.push_back(pPattern);
}

