

#define LOCAL_DEBUG
#include "debug.h"

#include "acfg.h"
#include "meta.h"
#include "filereader.h"
#include "fileio.h"
#include "lockable.h"
#include "cleaner.h"

#include <regex.h>

#include <iostream>
#include <fstream>
#include <string>
#include <deque>
#include <map>
#include <algorithm>

using namespace MYSTD;


#define _iterPos(it, start) (it-start.begin())/sizeof(it)
#define sProblemLoc szPath<< ':'<< _iterPos(it, lines)

string sFilterSet(SPACECHARS "#");
#define IsValidButIrrelevantLine(x) (x.empty() || stmiss != sFilterSet.find(x[0]))
#define BARF(x) { cerr << x << endl; exit(EXIT_FAILURE); }

namespace rechecks
{
bool CompileExpressions();
bool CompileUncExpressions(const string & req, const string & tgt);
}


namespace acfg {


// internal stuff:
char alphabet[65] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
string sPopularPath("/debian/");

string dontcache, dontcacherq, dontcachetgt;

tStrMap localdirs;
NoCaseStringMap mimemap;

#ifndef MINIBUILD

MapNameToString n2sTbl[] = {
		{  "Port", 			&port , 0}
		,{ "CacheDir", 	&cachedir, 0 }
		,{ "LogDir", 	&logdir , 0}
		,{ "SocketPath", 	&fifopath, 0}
		,{ "PidFile", 	&pidfile, 0}
		//,{ "Proxy",		&proxy, 0}
		,{ "ReportPage",&reportpage, 0}
		,{ "VfilePattern", &vfilepat, 0}
		,{ "PfilePattern", &pfilepat, 0}
		,{ "WfilePattern", &wfilepat, 0}
		,{ "AdminAuth",  &adminauth, 0}
		,{ "BindAddress", &bindaddr, 0}
		,{ "UserAgent", &agentname, 0}
		,{ "DontCache",	&dontcache, 0}
		,{ "DontCacheRequested",	&dontcachetgt, 0}
		,{ "DontCacheResolved",	&dontcacherq, 0}
		,{ "PrecacheFor", &mirrorsrcs, 0}
		,{ "RequestAppendix", &requestapx, 0}
};

MapNameToInt n2iTbl[] = {
		{ "Debug", 		&debug, 0 , 10}
		,{ "OfflineMode", 	&offlinemode , 0, 10}
		,{ "ForeGround", 	&foreground , 0, 10}
		,{ "Verbose", 		0, "Option is deprecated, ignoring the value." , 10}
		,{ "ForceManaged", 	&forcemanaged , 0, 10}
		,{ "StupidFs", 		&stupidfs , 0, 10}
		,{ "VerboseLog",	&verboselog , 0, 10}
		,{ "ExTreshold",	&extreshhold, 0, 10}
		,{ "MaxSpareThreadSets",	&tpstandbymax, "Deprecated option name, mapped to MaxStandbyConThreads", 10}
		,{ "MaxStandbyConThreads",	&tpstandbymax, 0, 10}
		,{ "MaxConThreads",	&tpthreadmax, 0, 10}
		,{ "DnsCacheSeconds", &dnscachetime, 0, 10}
		,{ "UnbufferLogs", &debug , 0, 10}
		,{ "ExAbortOnProblems", &exfailabort, 0, 10}
		,{ "ExposeOrigin", &exporigin, 0, 10}
		,{ "LogSubmittedOrigin", &logxff, 0, 10}
		,{ "OldIndexUpdater", &oldupdate, "Option is deprecated, ignoring the value." , 10}
		,{ "RecompBz2", &recompbz2, 0, 10}
		,{ "NetworkTimeout", &nettimeout, 0, 10}
		,{ "MinUpdateInterval", &updinterval, 0, 10}
		,{ "ForwardBtsSoap", &forwardsoap, 0, 10}

		,{ "DirPerms", &dirperms, 0, 8}
		,{ "FilePerms", &fileperms, 0, 8}
};

void _ReadRewriteFiles(const string & sFile, const string & sRepName);
void _ReadBackendsFiles(const string & sFile, const string &sRepName);


struct fct_lt_host
{
  bool operator()(const tHttpUrl &a, const tHttpUrl &b) const
  {
    return strcasecmp(a.sHost.c_str(), b.sHost.c_str()) < 0;
  }
};
typedef multimap<tHttpUrl,const string*, fct_lt_host> tMapUrl2StringPtr; 
typedef tMapUrl2StringPtr::iterator tUrl2RepIter;
tMapUrl2StringPtr mapUrl2pVname;

typedef map<const string, tRepoData> tRepMap;
tRepMap repoparms;

string * _GetStringPtr(const string &key) {
	for(unsigned int i=0; i<_countof(n2sTbl); i++)
	{
		if(0==strcasecmp(key.c_str(), n2sTbl[i].name))
		{
			if(n2sTbl[i].warn)
				cerr << "Warning, " << key << ": " << n2sTbl[i].warn << endl;

			return n2sTbl[i].ptr;
		}
	}
	return NULL;
}

int * _GetIntPtr(const string &key, int &base) {
	for(unsigned int i=0; i<_countof(n2iTbl); i++)
	{
		if(0==strcasecmp(key.c_str(), n2iTbl[i].name))
		{
			if(n2iTbl[i].warn)
				cerr << "Warning, " << key << ": " << n2iTbl[i].warn << endl;
			base = n2iTbl[i].base;
			return n2iTbl[i].ptr;
		}
	}
	return NULL;
}

inline void _FixPostPreSlashes(string &val)
{
	// fix broken entries

	if (val.empty() || val.at(val.length()-1) != '/')
		val.append("/");
	if (val.at(0) != '/')
		val.insert(0, "/", 1);
}

inline bool _ReadMainConfiguration(const string & szFilename)
{
	filereader reader;
	reader.OpenFile(szFilename);
	reader.CheckGoodState(true);
	string sLine;
	for (bool bNotEof=true; bNotEof;)
	{
		bNotEof=reader.GetOneLine(sLine);
		if (IsValidButIrrelevantLine(sLine))
			continue;
#ifdef DEBUG
		cerr << sLine <<endl;
#endif
		tStrPos pos=sLine.find('#');
		if(stmiss != pos)
			sLine.erase(pos);

		if(! SetOption(sLine))
			BARF("Error reading main options, terminating.");
	}
	return true;
}

inline const string * _CheckBEentryGetNamePtr(const string & sRepName)
{
	// some memory saving using STL containers properties:
	// needs a reliably stored string for the pointer. Backend description may not exist,
	// then create a dummy one with no contents and point at this string. Iterators on map are
	// guaranteed to be stable so point at its key variable.

	pair<tRepMap::iterator,bool> addRes = repoparms.insert(make_pair(sRepName, tRepoData()));
	return & addRes.first->first;
}

inline void _AddRemapInfo(bool bAsBackend, const string & token,
		const string &repname)
{
	if (0!=token.compare(0, 5, "file:"))
	{
		tHttpUrl url;
		if(! url.SetHttpUrl(token))
			BARF(token + " <-- bad URL detected");
		_FixPostPreSlashes(url.sPath);
		
		if (bAsBackend)
			repoparms[repname].push_back(url);
		else
			mapUrl2pVname.insert(pair<tHttpUrl,const string*>(
					url, _CheckBEentryGetNamePtr(repname)));
	}
	else
	{
		string sPath=token.substr(5);
		if (sPath.empty())
			BARF("Bad file spec for repname, file:?");
		
		MakeAbsolutePath(sPath, confdir);
		
		tStrDeq srcs=ExpandFilePattern(sPath, true);
		for(tStrDeq::const_iterator it=srcs.begin(); it!=srcs.end(); it++)
		{
			if (bAsBackend)
				_ReadBackendsFiles(*it, repname);
			else
				_ReadRewriteFiles(*it, repname);
		}
	}
}

inline bool ParseOptionLine(const string &sLine, string &key, string &val, bool bQuiet)
{
	string::size_type posCol = sLine.find(":");
	string::size_type posEq = sLine.find("=");
	if (posEq==stmiss && posCol==stmiss)
	{
		if(!bQuiet)
			cerr << "Not a valid configuration directive: " << sLine <<endl;
		return false;
	}
	string::size_type pos;
	if (posEq!=stmiss && posCol!=stmiss)
		pos=min(posEq,posCol);
	else if (posEq!=stmiss)
		pos=posEq;
	else
		pos=posCol;

	key=sLine.substr(0, pos);
	val=sLine.substr(pos+1);
	trimString(key);
	trimString(val);
	if(key.empty())
		return false; // weird

	return true;
}


struct tHookHandler: public tRepoData::IHookHandler, public lockable
{
	string cmdRel, cmdCon;
	time_t downDuration, downTimeNext;

	int m_nRefCnt;

	tHookHandler(cmstring&) :
		downDuration(30), downTimeNext(0), m_nRefCnt(0)
	{
//		cmdRel = "logger JobRelease/" + name;
//		cmdCon = "logger JobConnect/" + name;
	}
	virtual void JobRelease()
	{
		setLockGuard;
		if (0 >= --m_nRefCnt)
		{
			//system(cmdRel.c_str());
			downTimeNext= time(0) + downDuration;
		}
	}
	virtual void JobConnect()
	{
		setLockGuard;
		if (0 == m_nRefCnt++)
		{
			if(downTimeNext) // huh, already ticking? reset
				downTimeNext=0;
			else
				system(cmdCon.c_str());
		}
	}
};

void _AddHooksFile(cmstring& vname)
{
	filereader reader;
	if(!reader.OpenFile(acfg::confdir+"/"+vname+".hooks"))
		return;

	string sLine, key, val;
	struct tHookHandler &hs = *(new tHookHandler(vname));

	for (bool bNotEof=true; bNotEof;)
	{
		bNotEof=reader.GetOneLine(sLine);
		if (IsValidButIrrelevantLine(sLine))
			continue;

		if(!ParseOptionLine(sLine, key, val, false))
			continue;
		const char *p=key.c_str();
		if(strcasecmp("PreUp", p) == 0)
		{
			hs.cmdCon = val;
		}
		else if(strcasecmp("Down", p) == 0)
		{
			hs.cmdRel = val;
		}
		else if(strcasecmp("DownTimeout", p) == 0)
		{
			errno=0;
			uint n = strtoul(val.c_str(), NULL, 10);
			if(!errno)
				hs.downDuration = n;
		}
	}
	repoparms[vname].m_pHooks = &hs;

}

string _GetBase64Auth(const string & s)
{
	int cols=0, bits=0, c=0, char_count=0;
	tStrPos pos=0;
	string out;
	while ( pos<s.size() )
	{
		c=s[pos++];
		if('%' == c && pos<s.length()-1
				&& hexmap[(unsigned)s[pos]]>=0
				&& hexmap[(unsigned)s[pos+1]]>=0)
		{
			c=char(16*hexmap[(unsigned)s[pos]]+hexmap[(unsigned)s[pos+1]]);
			pos+=2;
		}
		bits += c;
		char_count++;
		if (char_count == 3)
		{
			out+=(alphabet[bits >> 18]);
			out+=(alphabet[(bits >> 12) & 0x3f]);
			out+=(alphabet[(bits >> 6) & 0x3f]);
			out+=(alphabet[bits & 0x3f]);
			cols += 4;
			bits = 0;
			char_count = 0;
		}
		else
		{
			bits <<= 8;
		}
	}
	if (char_count != 0)
	{
		bits <<= 16 - (8 * char_count);
		out+=(alphabet[bits >> 18]);
		out+=(alphabet[(bits >> 12) & 0x3f]);
		if (char_count == 1)
		{
			out+=('=');
			out+=('=');
		}
		else
		{
			out+=(alphabet[(bits >> 6) & 0x3f]);
			out+=('=');
		}
	}
	return out;
}

inline void _ParseLocalDirs(cmstring &value)
{
	tStrVec tokens;
	Tokenize(value, ";", tokens);
	for(tStrVecIter it=tokens.begin(); it!=tokens.end(); it++)
	{
		trimString(*it);
		tStrPos pos = it->find_first_of(SPACECHARS);
		if(stmiss == pos)
		{
			cerr << "Cannot map " << *it << ", needed format: virtualdir realdir, ignoring it";
			continue;
		}
		string from=it->substr(0, pos);
		trimString(from, "/");
		string what=it->substr(pos);
		trimString(what, SPACECHARS "'\"");
		if(what.empty())
		{
			cerr << "Unsupported target of " << from << ": " << what << ", ignoring it" << endl;
			continue;
		}
		localdirs[from]=what;
	}
}

inline void _FetchMimeTypes()
{
	filereader mreader;
	tStrVec toks;
	string line;
	for(mreader.OpenFile("/etc/mime.types"); mreader.CheckGoodState(false); )
	{
		if(!mreader.GetOneLine(line))
			break;
		if(IsValidButIrrelevantLine(line))
			continue;

		if(Tokenize(line, SPACECHARS, toks, false)>1)
		{
			for(UINT i=1; i<toks.size();i++)
				mimemap[toks[i]]=toks[0];
		}
	}
}

bool SetOption(const string &sLine, bool bQuiet)
{

	string key, value;

	if(!ParseOptionLine(sLine, key, value, bQuiet))
		return false;

	string * psTarget;
	int * pnTarget;
	int nNumBase(10);

	if ( NULL != (psTarget = _GetStringPtr(key)))
		*psTarget=value;
	else if ( NULL != (pnTarget = _GetIntPtr(key, nNumBase)))
	{
		const char *pStart=value.c_str();
		if(! *pStart)
		{
			cerr << "Missing value for " << key << " option!" <<endl;
			return false;
		}
		
		errno=0;
		char *pEnd(0);
		long nVal = strtol(pStart, &pEnd, nNumBase);

		if(RESERVED_DEFVAL == nVal)
		{
			cerr << "Bad value for " << key << " (protected value, use another one)" <<endl;
			return false;
		}

		*pnTarget=nVal;

		if (errno)
		{
			cerr << "Invalid number for " << key << " ";
			perror("option");
			return false;
		}
		if(*pEnd)
		{
			cerr << "Bad value for " << key << " option or found trailing garbage: " << pEnd <<endl;
			return false;
		}
	}

	else if(0==strcasecmp(key.c_str(), "Proxy"))
	{
		if (!proxy_info.SetHttpUrl(value))
		{
			cerr << "Invalid proxy specification, aborting..." << endl;
			exit(EXIT_FAILURE);
		}
		tStrPos pos = proxy_info.sHost.rfind('@');
		if (stmiss != pos)
		{
			tStrPos pwStart= startsWithSz(value, "http://") ? 7 : 0;
			proxy_info.sPath = string("Proxy-Authorization: Basic ")
					+ _GetBase64Auth(value.substr(pwStart, pos)) + "\r\n";
			proxy_info.sHost.erase(0, pos + 1);
		}
		else
			proxy_info.sPath.clear(); // no auth at all, stop confusion

		if (proxy_info.sHost.empty())
		{
			cerr << "Invalid proxy specification, aborting..." << endl;
			exit(EXIT_FAILURE);
		}
		if (proxy_info.sPort.empty())
		{
			if (proxy_info.sPath.empty()) // guess unless there is any risk...
			{
				cerr << "Warning, unknown proxy port, assuming 80." << endl;
				proxy_info.sPort = "80";
			}
			else
			{
				cerr << "Error, unknown proxy port!" << endl;
				exit(EXIT_FAILURE);
			}
		}
	}
	else if(0==strcasecmp(key.c_str(), "LocalDirs"))
	{
		_ParseLocalDirs(value);
		_FetchMimeTypes();
		return !localdirs.empty();
	}
	else if(0==strncasecmp(key.c_str(), "Remap-", 6))
	{
		string vname=key.substr(6, key.npos);
		tStrVec tokens;
		
		Tokenize(value, SPACECHARS, tokens);
		if(tokens.empty() || vname.empty())
		{
			if(!bQuiet)
				cerr << "Found invalid entry, ignoring " << key << ": " << value <<endl;
			return false;
		}
		enum xtype { prefix, backend, flags } type = prefix;
		for(UINT i=0; i<tokens.size(); i++)
		{
			if(tokens[i]==";")
			{
				type=xtype(int(type)+1);
				continue;
			}
			if(startsWithSz(tokens[i], "#"))
				break;
			if(flags == type)
			{
#warning do something with the stuff
			}
			else
				_AddRemapInfo(backend == type, tokens[i], vname);
		}
		_AddHooksFile(vname);
	}
	else
	{
		if(!bQuiet)
			cerr << "Warning, unknown configuration directive: " << key <<endl;
		return false;
	}
	return true;
}


//const string * GetVnameForUrl(string path, string::size_type * nMatchLen)
const string * GetRepNameAndPathResidual(const tHttpUrl & in, string & sRetPathResidual)
{
	sRetPathResidual.clear();
	
	// get all the URLs matching THE HOSTNAME
	pair<tUrl2RepIter,tUrl2RepIter> range=mapUrl2pVname.equal_range(in);
	if(range.first==range.second)
		return NULL;
	
	tStrPos bestMatchLen(0);
	string const * psBestHit(NULL);
		
	// now find the longest directory part which is the suffix of requested URL's path
	for (tUrl2RepIter & it=range.first; it!=range.second; it++)
	{
		// rewrite rule path must be a real prefix
		// it's also surrounded by /, ensured during construction
		const string & prefix=it->first.sPath;
		tStrPos len=prefix.length();
		if (in.sPath.size() > len && 0==in.sPath.compare(0, len, prefix))
		{
			if (len>bestMatchLen)
			{
				bestMatchLen=len;
				psBestHit=it->second;
			}
		}
	}
		
	if(psBestHit) sRetPathResidual=in.sPath.substr(bestMatchLen);
	return psBestHit;
	
}

tRepoData * GetBackendVec(const string * vname)
{
	if(!vname)
		return NULL;
	tRepMap::iterator it=repoparms.find(*vname);
	if(it==repoparms.end() || it->second.empty())
		return NULL;
	return & it->second;
}


void _ReadBackendsFiles(const string & sFile, const string &sRepName)
{

	int nAddCount=0;
	string sLine, key, val;
	tHttpUrl entry;

	filereader reader;
	reader.OpenFile(sFile.c_str());

	if(debug>4)
		cout << "Reading backend file: " << sFile <<endl;
	if(!reader.CheckGoodState(false))
	{
		if(debug>4)
			cout << "No backend data found, file ignored."<<endl;
		goto try_read_default;
	}
	
	
	for(bool bNotEof=true;bNotEof;)
	{
		bNotEof=reader.GetOneLine(sLine);
		//if(debug)
		//	cerr << "backends, got line: " << sLine <<endl;
		

		if( (startsWithSz(sLine, "http://") && entry.SetHttpUrl(sLine) )
				||
			(IsValidButIrrelevantLine(sLine) 
					&& ! entry.sHost.empty() 
					&& ! entry.sPath.empty()
			)
		)
		{
			_FixPostPreSlashes(entry.sPath);
#ifdef DEBUG
			cerr << "Backend: " << sRepName << " <-- " << entry.ToURI() <<endl;
#endif		

			repoparms[sRepName].push_back(entry);
			nAddCount++;
			entry.clear();
		}
		else if(_ParseLine(sLine, key, val))
		{
			if(keyEq("Site", key))
				entry.sHost=val;
			/* TODO: not supported yet, maybe add later - push to a vector of hosts and add multiple later
			if(keyEq("Aliases", key))
			{
				val+=" ";
				for(string::size_type posA(0), posB(0);
					posA<val.length();
					posA=posB+1)
				{
					posB=val.find_first_of(" \t\r\n\f\v", posA);
					if(posB!=posA)
						hosts.push_back(val.substr(posA, posB-posA));
				}
			}
			*/
			else if(keyEq("Archive-http", key) || keyEq("X-Archive-http", key))
			{
				entry.sPath=val;
			}
		}
		else if(bNotEof && !IsValidButIrrelevantLine(sLine))
		{
			cerr << "Bad backend description, around line "
					<< reader.GetPositionDescription() << endl;
			exit(2);
		}
	}

try_read_default:
	if(nAddCount || endsWithSzAr(sFile, ".default"))
		return;
	if(debug>4)
		cout << "Trying to read replacement from " << sFile << ".default" <<endl;
	_ReadBackendsFiles(sFile+".default", sRepName);

}

void ShutDown()
{
	mapUrl2pVname.clear();
	repoparms.clear();
}

void _ReadRewriteFiles(const string & sFile, const string & sRepName)
{

	filereader reader;
	if(debug>4)
		cout << "Reading rewrite file: " << sFile <<endl;
	reader.OpenFile(sFile.c_str());
	reader.CheckGoodState(true);

	tStrVec hosts, paths;
	string sLine, key, val;
	tHttpUrl url;
	
	for(bool bNotEof=true; bNotEof; )
	{
		bNotEof=reader.GetOneLine(sLine);

		if (0==sLine.compare(0, 7, "http://"))
		{ // TODO: The check above is optional since SetHttpUrl does that too, but it's 
			// more userfriendly on errors (more exact error message)
			if (url.SetHttpUrl(sLine))
			{
				_FixPostPreSlashes(url.sPath);
				pair<tHttpUrl, const string*> info(url,
						_CheckBEentryGetNamePtr(sRepName));
				mapUrl2pVname.insert(info);
			}
			else
			{
				cout << "Parse error, invalid URL" << sLine << " on line "
						<< reader.GetPositionDescription() <<endl;
				exit(2);
			}
			continue;
		}
		else if (IsValidButIrrelevantLine(sLine)) // end of block, eof, ... -> commit it
		{
			if (hosts.empty() && paths.empty())
				continue; // dummy run
			if ( !hosts.empty() && paths.empty())
			{
				cerr << "Warning, missing path spec for the site " << hosts[0] <<", ignoring mirror."<< endl;
				continue;
			}
			if ( !paths.empty() && hosts.empty())
			{
				cout << "Parse error, missing Site: field around line "
						<< reader.GetPositionDescription() <<endl;
				exit(2);
			}
			for (tStrVecIterConst itHost=hosts.begin();
			itHost!=hosts.end(); 
			itHost++)
			{
				for (tStrVecIterConst itPath=paths.begin();
				itPath!=paths.end(); 
				itPath++)
				{
					//mapUrl2pVname[*itHost+*itPath]= &itHostiVec->first;
					tHttpUrl url;
					url.sHost=*itHost;
					url.sPath=*itPath;
					pair<tHttpUrl,const string*> info(url, _CheckBEentryGetNamePtr(sRepName));
					mapUrl2pVname.insert(info);

#ifdef DEBUG
						cout << "Mapping: "<< *itHost << *itPath 
						<< " -> "<< sRepName <<endl;
#endif
				}
			}
			hosts.clear();
			paths.clear();
			continue;
		}
		else if(!_ParseLine(sLine, key, val))
		{
			cerr << "Error parsing rewrite definitions, around line " << reader.GetPositionDescription() <<endl;
			exit(1);
		}
		
		// got something, intepret it...
		if( keyEq("Site", key) || keyEq("Alias", key) || keyEq("Aliases", key))
			Tokenize(val, SPACECHARS, hosts, true);
		
		if(keyEq("Archive-http", key) || keyEq("X-Archive-http", key))
		{
			// help STL saving some memory
			if(sPopularPath==val)
				paths.push_back(sPopularPath);
			else
			{
				_FixPostPreSlashes(val);
				paths.push_back(val);
			}
			continue;
		}
	}
}


tRepoData::~tRepoData()
{
	if(m_pHooks)
	{
		delete m_pHooks;
		m_pHooks=NULL;
	}
}

void ReadConfigDirectory(const char *szPath)
{
	// TODO: early abort when the dir does not exist!

	char buf[PATH_MAX];
	if(!realpath(szPath, buf))
  {
     cerr << "Failed to open config directory" << endl;
     exit(42);
  }
	confdir=buf; // pickup the last config directory

#if defined(HAVE_WORDEXP) || defined(HAVE_GLOB)
	tStrDeq srcs=ExpandFilePattern(confdir+SZPATHSEP"*.conf", true);
	for(tStrDeq::const_iterator it=srcs.begin(); it!=srcs.end(); it++)
		_ReadMainConfiguration(*it);
#else
	_ReadMainConfiguration(confdir+SZPATHSEP"acng.conf");
#endif
}

void PostProcConfig() 
{
	if(stupidfs)
		cerr << "Warning, support for primitive filesystems is not completely implemented!\n";
	
	// let's also apply the umask to the directory permissions
	{
		mode_t mask = umask(0);
		umask(mask); // restore it...
		dirperms &= ~mask;
		fileperms &= ~mask;
	}

    // postprocessing

#ifdef FORCE_CUSTOM_UMASK
	if(!sUmask.empty())
	{
		mode_t nUmask=0;
		if(sUmask.size()>4)
		{
			cerr << "Invalid umask length\n";
			exit(EXIT_FAILURE);
		}
		for(unsigned int i=0; i<sUmask.size(); i++)
		{
			unsigned int val = sUmask[sUmask.size()-i-1]-'0';
			if(val>7)
			{
				cerr << "Invalid umask value\n" <<endl;
				exit(EXIT_FAILURE);
			}
			nUmask |= (val<<(3*i));
		
		}
		//cerr << "Got umask: " << nUmask <<endl;
		umask(nUmask);
	}
#endif
	
   if(cachedir.empty() || cachedir[0] != CPATHSEP) {
      cerr << "Cache directory unknown or not absolute, terminating..." <<endl;
      exit(EXIT_FAILURE);
   }
   
   if(!rechecks::CompileExpressions())
   {
	   cerr << "An error occured while compiling file type regular expression!" <<endl;
	   exit(EXIT_FAILURE);
   }
   
   if(acfg::tpthreadmax < 0)
	   acfg::tpthreadmax = numeric_limits<int>::max();
	   
   // get rid of duplicated and trailing slash(es)
	for(tStrPos pos; stmiss != (pos = cachedir.find(SZPATHSEP SZPATHSEP )); )
		cachedir.erase(pos, 1);

	cacheDirSlash=cachedir+CPATHSEP;

   if(!pidfile.empty() && pidfile.at(0) != CPATHSEP)
   {
	   cerr << "Pid file path must be absolute, terminating..."  <<endl;
	         exit(EXIT_FAILURE);
   }
   
   if(!acfg::agentname.empty())
	   acfg::agentheader=string("User-Agent: ")+acfg::agentname + "\r\n";
   
   if(!adminauth.empty())
	   adminauth=string("Basic ")+_GetBase64Auth(adminauth);
   
   // create working paths before something else fails somewhere
   if(!fifopath.empty())
	   mkbasedir(acfg::fifopath);
   if(!cachedir.empty())
	   mkbasedir(acfg::cachedir);
   if(! pidfile.empty())
	   mkbasedir(acfg::pidfile);

   if(nettimeout < 5) {
	   cerr << "Warning, NetworkTimeout too small, assuming 5." << endl;
	   nettimeout = 5;
   }

   if(RESERVED_DEFVAL == forwardsoap)
	   forwardsoap = !forcemanaged;


   if(!rechecks::CompileUncExpressions(dontcachetgt.empty() ? dontcache : dontcachetgt,
		   dontcacherq.empty() ? dontcache : dontcacherq))
   {
	   cerr << "An error occured while compiling regular expression for non-cached paths!" <<endl;
	   exit(EXIT_FAILURE);
   }

   if (acfg::debug>3)
	{
		for (UINT i=0; i<_countof(n2sTbl); i++)
			if(n2sTbl[i].ptr)
				cout << n2sTbl[i].name << " = " << *(n2sTbl[i].ptr) <<endl;

		if (acfg::debug > 4)
		{
			cerr << "escaped version:" << endl;
			for (UINT i = 0; i < _countof(n2sTbl); i++)
				if (n2sTbl[i].ptr)
				{
					cout << n2sTbl[i].name << " = ";
					for (const char *p = n2sTbl[i].ptr->c_str(); *p; p++)
						if('\\' == *p)
							cout << "\\\\";
						else
							cout << *p;
					cout <<endl;
				}
		}

		for (UINT i=0; i<_countof(n2iTbl); i++)
			if(n2iTbl[i].ptr)
				cout << n2iTbl[i].name << " = \"" << *(n2iTbl[i].ptr) << "\"\n";
	}

#ifndef DEBUG
   if(acfg::debug>=5)
	   cout << "\n\nAdditional debugging information not compiled in.\n\n";
#endif
   
   
   /*
   // help STL saving some memory
   for(tUrl2RepIter it=mapUrl2pVname.begin(); it!=mapUrl2pVname.end(); it++)
   {
	   tUrl2RepIter suc=it;
	   suc++;
	   if(suc==mapUrl2pVname.end()) break;
	   if(suc->first.sPath==it->first.sPath)
		   const_cast<string&>(suc->first.sPath)=it->first.sPath;
   }
   */

#if 0 //def DEBUG
#warning adding hook control pins
   for(tMapString2Hostivec::iterator it = repoparms.begin();
		   it!=repoparms.end() ; ++it)
   {
	   tHookHandler *p = new tHookHandler(it->first);
	   p->downDuration=10;
	   p->cmdCon = "logger wanna/connect";
	   p->cmdRel = "logger wanna/disconnect";
	   it->second.m_pHooks = p;
   }
#endif

   if(debug == -42)
   {
	   /*
	   for(tMapString2Hostivec::const_iterator it=mapRepName2Backends.begin();
			   it!=mapRepName2Backends.end(); it++)
	   {
		   for(tRepoData::const_iterator jit=it->second.begin();
				   jit != it->second.end(); jit++)
		   {
			   cout << jit->ToURI() <<endl;
		   }
	   }
	   */
	   for(tUrl2RepIter it=mapUrl2pVname.begin(); it!=mapUrl2pVname.end(); it++)
	   {
		   cout << it->first.ToURI() << " ___" << *(it->second) << endl;
	   }

	   exit(1);
   }
} // PostProcConfig

time_t ExecutePostponed()
{
	time_t ret(cleaner::never), now(time(0));
	for (tRepMap::iterator it = repoparms.begin(); it != repoparms.end(); ++it)
	{
		if (!it->second.m_pHooks)
			continue;
		tHookHandler & hooks = *(static_cast<tHookHandler*> (it->second.m_pHooks));
		lockguard g(hooks);
		if (hooks.downTimeNext)
		{
			if (hooks.downTimeNext <= now) // time to execute
			{
				if(acfg::debug>1)
					aclog::misc(hooks.cmdRel, 'X');
				if(acfg::debug>3)
					aclog::flush();

				system(hooks.cmdRel.c_str());
				hooks.downTimeNext = 0;
			}
			else // in future, use the soonest time
				ret = min(ret, hooks.downTimeNext);
		}
	}
	return ret;
}

#endif // MINIBUILD

} // namespace acfg

namespace rechecks
{
	cmstring *psRex[] = {&acfg::vfilepat, &acfg::pfilepat, &acfg::wfilepat};
	regex_t rex[_countof(psRex)];

bool CompileExpressions()
{
	char buf[1024];
	for(UINT i=0; i<_countof(psRex); i++)
	{
		int nErr=regcomp(&rex[i], psRex[i]->c_str(), REG_EXTENDED);
		if(nErr)
		{
			regerror(nErr,  &rex[i], buf, sizeof(buf));
			buf[_countof(buf)-1]=0; // better be safe...
			std::cerr << buf << ": " << *(psRex[i]) << std::endl;
			return false;
		}
	}
	return true;
}

eFileKind GetFiletype(const string & in) {
	//LOGSTART("rechecks::getFiletype");
    //dbgline;
    if(!regexec(&rex[0], in.c_str(), 0, NULL, 0))
        return FILE_INDEX;
    //dbgline;
    if(!regexec(&rex[1], in.c_str(), 0, NULL, 0))
        return FILE_PKG;
    //dbgline;
    return FILE_INVALID;
}

bool MatchWhitelist(const string & in)
{
	LOGSTART("MatchWhiteList");
	bool bPersistent=!regexec(&rex[2], in.c_str(), 0, NULL, 0);
	LOG(in <<" is " << (bPersistent ? "persistent" : "mortal") <<  " file");
	return bPersistent;
}

deque<regex_t> vecReqPatters, vecTgtPatterns;

#ifndef MINIBUILD
inline bool CompileUncachedRex(const string & token, bool bIsTgtPattern, bool bRecursiveCall)
{
	deque<regex_t> & patvec = bIsTgtPattern ? vecTgtPatterns : vecReqPatters;

	if (0!=token.compare(0, 5, "file:")) // pure pattern
	{
		UINT pos = patvec.size();
		patvec.resize(pos+1);
		return 0==regcomp(&patvec[pos], token.c_str(), REG_EXTENDED);
	}
	else if(!bRecursiveCall) // don't go further than one level
	{
		string sPath=token.substr(5);
		if (sPath.empty())
			BARF("Bad file spec for repname, file:?");

		MakeAbsolutePath(sPath, acfg::confdir);

		tStrDeq srcs = ExpandFilePattern(sPath, true);
		for(tStrDeq::const_iterator it=srcs.begin(); it!=srcs.end(); it++)
		{
			filereader reader;
			if(!reader.OpenFile(*it))
			{
				cerr << "Error opening pattern file: " << *it <<endl;
				return false;
			}
			string line;
			while(reader.GetOneLine(line))
			{
				if(!CompileUncachedRex(*it, bIsTgtPattern, true))
					return false;
			}
		}
		return true;
	}

	cerr << token << " is not supported here" <<endl;
	return false;
}


bool CompileUncExpressions(const string & req, const string & tgt)
{
	tStrVec pats;
	Tokenize(req, SPACECHARS, pats, false);
	/*
	 *
	 if(acfg::debug>10)
			cerr << "req: " << req << ", tgt: " << tgt << ",#req: " << pats.size()<< endl;
			*/
	for (tStrVec::const_iterator it = pats.begin(); it != pats.end(); it++)
		if (!CompileUncachedRex(*it, false, false))
			return false;
	Tokenize(tgt, SPACECHARS, pats, false);
	for (tStrVec::const_iterator it = pats.begin(); it != pats.end(); it++)
		if (!CompileUncachedRex(*it, true, false))
			return false;
	return true;
}



bool MatchUncacheableRequest(const string & in)
{
	LOGSTART2("MatchUncacheableRequest", in << " against " << vecReqPatters.size() << " patterns");
	for(deque<regex_t>::const_iterator it=vecReqPatters.begin();
			it!=vecReqPatters.end(); it++)
	{
		if(!regexec(& (*it), in.c_str(), 0, NULL, 0))
			return true;
	}
	return false;
}

bool MatchUncacheableTarget(const string &in)
{
	for(deque<regex_t>::const_iterator it=vecTgtPatterns.begin();
			it!=vecTgtPatterns.end(); it++)
	{
		if(!regexec(& (*it), in.c_str(), 0, NULL, 0))
			return true;
	}
	return false;
}

#endif //MINIBUILD


} // namespace rechecks

mstring GetDirPart(const string &in)
{
	if(in.empty())
		return "";

	tStrPos end = in.find_last_of(CPATHSEP);
	if(end == stmiss) // none? don't care then
		return "";

	return in.substr(0, end+1);
}

void mkbasedir(const string & path)
{
	if(0==mkdir(GetDirPart(path).c_str(), acfg::dirperms) || EEXIST == errno)
		return; // should succeed in most cases

	UINT pos=0; // but skip the cache dir components, if possible
	if(startsWith(path, acfg::cacheDirSlash))
	{
		// pos=acfg::cachedir.size();
		pos=path.find("/", acfg::cachedir.size()+1);
	}
    for(; pos<path.size(); pos=path.find(SZPATHSEP, pos+1))
    {
        if(pos>0)
            mkdir(path.substr(0,pos).c_str(), acfg::dirperms);
    }
}


/*
int main(int argc, char **argv)
{
	if(argc<2)
		return -1;
	
	acfg::tHostInfo hi;
	cout << "Parsing " << argv[1] << ", result: " << hi.SetUrl(argv[1])<<endl;
	cout << "Host: " << hi.sHost <<", Port: " << hi.sPort << ", Path: " << hi.sPath<<endl;
	return 0;
}
*/

