(* File: utils.mli

    Copyright (C) 2007-

      Jane Street Holding, LLC
      Author: Markus Mottl
      email: mmottl\@janestcapital.com
      WWW: http://www.janestcapital.com/ocaml

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

(** Utils: utility functions for user convenience *)

open Bigarray
open Common
open Read_ml
open Sw_arg

val bin_dump : ?with_size : bool -> 'a sw_arg -> 'a -> buf
(** [bin_dump ?with_size sw_arg v] uses the [(sizer, writer)] pair
    [sw_arg] to first compute the size of [v] in the binary protocol,
    then allocates a buffer of exactly this size, and then writes out
    the value.  If [with_size] is [true], the size of the resulting
    binary string will be prefixed as an integer.  @return the buffer
    containing the written out value.

    @param with_size = false

    @raise Failure if the size of the value changes during writing,
    and any other exceptions that the binary writer in [sw_arg] can raise.
*)

(** Read buffers for efficient parsing of size-prefixed values *)
module Read_buf : sig
  type t (** Type of read buffer *)

  val create : unit -> t
  (** [create ()] @return a read buffer. *)

  val next : ?max_size : int -> 'a reader -> (t -> 'a option) reader
  (** [next ?max_size reader buf ~pos_ref rbuf] looks at read buffer
      [rbuf] and (ordinary) buffer [buf] starting at the position stored
      in [pos_ref] to parse OCaml-values encoded in the size-prefixed
      binary protocol.

      NOTE: [buf] must not contain trailing garbage after messages
      (including partial ones)!  Use {!Bigarray.Array1.sub} to constrain
      the buffer to the exact range covered by valid data.

      The function first attempts to read the size (binary protocol
      natural number) of a value to be parsed, then the value using
      [reader].  If [buf] does not contain enough data to read the size
      and/or the value, a partial read will be performed.  [rbuf] will
      then buffer all available data, thus always making it safe to store
      other data in [buf] as soon as {!next} returns.  Subsequent calls
      to {!next} with additional data allows one to continue parsing.

      [max_size] determines the maximum binary-encoded size of a value
      allowed.  [pos_ref] will contain the position after the last
      character read in [buf].

      @return [Some value] if a value was read, [None] otherwise,
      or raise any exceptions associated with parse errors in [reader].

      @raise Failure if [max_size] is exceeded or if the protocol lies
      about the binary-encoded size of the value.

      @param max_size default = None
  *)

  val enforce_buf_size : t -> int -> bool
  (** [enforce_buf_size rbuf n] enforces a maximum size of the internal
      buffer to be [n] bytes IF POSSIBLE.  If the currently buffered
      data (and hence the buffer) is larger than that, or if the buffer
      is smaller, or if [n] is smaller than the minimum internal buffer
      size, nothing will happen.  Use {!purge} to forget buffered data.
      @return [true] if the size of the internal buffer now fits into [n],
      [false] otherwise.
  *)

  val purge : t -> unit
  (** [purge rbuf] empties [rbuf], thus forgetting about previously
      buffered data.  The internal buffer may still remain large.
      Use {!enforce_buf_size} to change that. *)

  val get_size : t -> int
  (** [get_size rbuf] @return the size of the internal buffer of [rbuf]. *)

  val get_buffered_data_size : t -> int
  (** [get_buffered_data_size rbuf] @return the size of buffered data in
      [rbuf], including partially read length prefixes. *)
end


(** Conversion of binable types *)

module type Make_binable_spec = sig
  type t
  type binable

  val to_binable : t -> binable
  val of_binable : binable -> t

  val bin_size_binable : binable Size.sizer
  val bin_write_binable : binable Map_to_safe.writer
  val bin_write_binable_ : binable Unsafe_write_c.writer
  val bin_read_binable : binable Read_ml.reader
  val bin_read_binable_ : binable Unsafe_read_c.reader
  val bin_read_binable__ : (int -> binable) Unsafe_read_c.reader
  val bin_sw_arg_binable : binable Map_to_safe.sw_arg
end

module Make_binable (Bin_spec : Make_binable_spec) : sig
  open Bin_spec

  type t = Bin_spec.t

  val bin_size_t : t Size.sizer
  val bin_write_t : t Map_to_safe.writer
  val bin_write_t_ : t Unsafe_write_c.writer
  val bin_read_t : t Read_ml.reader
  val bin_read_t_ : t Unsafe_read_c.reader
  val bin_read_t__ : (int -> t) Unsafe_read_c.reader
  val bin_sw_arg_t : t Map_to_safe.sw_arg
end
