# Copyright (C) 2006 Canonical Ltd
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Test the CVSPS Parsing functionality"""

import cStringIO

from bzrlib import tests

import cvsps.parser


_SINGLE_CVSPS_DUMP = """\
---------------------
PatchSet 1 
Date: 2006/11/20 13:52:17
Author: jameinel
Branch: HEAD
Tag: (none) 
Log:
m

Members: 
	b:INITIAL->1.1 

"""


_COMPLEX_CVSPS_DUMP = """\
---------------------
PatchSet 1 
Date: 2006/11/20 13:52:17
Author: jameinel
Branch: HEAD
Tag: (none) 
Log:
m

Members: 
	b:INITIAL->1.1 

---------------------
PatchSet 2 
Date: 2006/11/20 13:53:35
Author: jameinel
Branch: HEAD
Tag: (none) 
Log:
add foo/c

Members: 
	foo/c:INITIAL->1.1 

---------------------
PatchSet 3 
Date: 2006/11/20 17:31:42
Author: jameinel
Branch: mytest
Ancestor branch: HEAD
Tag: (none) 
Log:
add c to b in mytest

Members: 
	b:1.1->1.1.2.1 

---------------------
PatchSet 4 
Date: 2006/11/20 17:37:19
Author: jameinel
Branch: HEAD
Tag: (none) 
Log:
Adding 'd' to 'b' on HEAD

Members: 
	b:1.1->1.2 

---------------------
PatchSet 5 
Date: 2006/11/20 17:38:15
Author: georgia
Branch: mytest
Tag: atag
Log:
fooc on mytest

Members: 
	foo/c:1.1->1.1.2.1 

---------------------
PatchSet 6 
Date: 2006/11/20 17:43:10
Author: jameinel
Branch: HEAD
Tag: (none) 
Log:
Adding 'e' on HEAD

Members: 
	b:1.2->1.3 

---------------------
PatchSet 7 
Date: 2006/11/20 17:43:39
Author: jameinel
Branch: other
Ancestor branch: mytest
Tag: (none) 
Log:
Adding 'other' to b on 'other'

Members: 
	b:1.1.2.1->1.1.2.1.2.1 

---------------------
PatchSet 8 
Date: 2006/11/20 18:03:29
Author: jameinel
Branch: other
Tag: (none) 
Log:
Multiline
commit describing a bogus
PatchSet 2
message for adding 'baz' to foo/c

Members: 
	foo/c:1.1.2.1->1.1.2.1.2.1 

---------------------
PatchSet 9 
Date: 2006/11/20 18:15:47
Author: jameinel
Branch: other
Tag: (none) 
Log:
multiline commit

with blank line

Members: 
	b:1.1.2.1.2.1->1.1.2.1.2.2 

"""

_ISO_8859_1_DUMP = """\
---------------------
PatchSet 1 
Date: 2006/11/20 13:52:17
Author: j\xe5meinel
Branch: HEAD
Tag: (none) 
Log:
message with '\xb5\xf1\xed\xe7\xf6d\xe9'

Members: 
	a:INITIAL->1.1 

"""

_UTF_8_DUMP = """\
---------------------
PatchSet 1 
Date: 2006/11/20 13:52:17
Author: j\xc3\xa5meinel
Branch: HEAD
Tag: (none) 
Log:
message with '\xc2\xb5\xc3\xb1\xc3\xad\xc3\xa7\xc3\xb6d\xc3\xa9'

Members: 
	a:INITIAL->1.1 

"""

class TestParser(tests.TestCase):
    """Direct tests of the CVSPS Parser"""

    def parse(self, txt, parser_kwargs={}):
        """Parse the given text."""
        sio = cStringIO.StringIO(txt)
        parser = cvsps.parser.Parser(sio, **parser_kwargs)
        self._patchsets = parser.parse()

    def test_simple(self):
        self.parse(_SINGLE_CVSPS_DUMP)

        self.assertEqual(1, len(self._patchsets))
        ps = self._patchsets[0]
        self.assertEqual(1, ps.num)
        self.assertEqual('jameinel', ps.author)
        self.assertEqual('HEAD', ps.branch)
        self.assertEqual(None, ps.ancestor_branch)
        self.assertEqual(None, ps.tag)
        self.assertEqual(['m\n'], ps.log)
        self.assertEqual([('b', '1.1')], ps.members)
        self.assertEqual('2006/11/20 13:52:17', ps.date)
        self.assertEqual(1164030737.000, ps.timestamp)
        self.assertEqual(0, ps.time_offset)

    def test_complex(self):
        self.parse(_COMPLEX_CVSPS_DUMP)
        self.assertEqual(9, len(self._patchsets))

        # 0 has already been checked by test_simple
        ps = self._patchsets[1]
        self.assertEqual(2, ps.num)
        self.assertEqual('jameinel', ps.author)
        self.assertEqual('HEAD', ps.branch)
        self.assertEqual(None, ps.ancestor_branch)
        self.assertEqual(None, ps.tag)
        self.assertEqual(['add foo/c\n'], ps.log)
        self.assertEqual([('foo/c', '1.1')], ps.members)
        self.assertEqual('2006/11/20 13:53:35', ps.date)
        self.assertEqual(1164030815.000, ps.timestamp)
        self.assertEqual(0, ps.time_offset)

        ps = self._patchsets[2]
        self.assertEqual(3, ps.num)
        self.assertEqual('jameinel', ps.author)
        self.assertEqual('mytest', ps.branch)
        self.assertEqual('HEAD', ps.ancestor_branch)
        self.assertEqual(None, ps.tag)
        self.assertEqual(['add c to b in mytest\n'], ps.log)
        self.assertEqual([('b', '1.1.2.1')], ps.members)
        self.assertEqual('2006/11/20 17:31:42', ps.date)
        self.assertEqual(1164043902.000, ps.timestamp)
        self.assertEqual(0, ps.time_offset)

        ps = self._patchsets[3]
        self.assertEqual(4, ps.num)
        self.assertEqual('jameinel', ps.author)
        self.assertEqual('HEAD', ps.branch)
        self.assertEqual(None, ps.ancestor_branch)
        self.assertEqual(None, ps.tag)
        self.assertEqual(["Adding 'd' to 'b' on HEAD\n"], ps.log)
        self.assertEqual([('b', '1.2')], ps.members)
        self.assertEqual('2006/11/20 17:37:19', ps.date)
        self.assertEqual(1164044239.000, ps.timestamp)
        self.assertEqual(0, ps.time_offset)

        ps = self._patchsets[4]
        self.assertEqual(5, ps.num)
        self.assertEqual('georgia', ps.author)
        self.assertEqual('mytest', ps.branch)
        self.assertEqual(None, ps.ancestor_branch)
        self.assertEqual('atag', ps.tag)
        self.assertEqual(["fooc on mytest\n"], ps.log)
        self.assertEqual([('foo/c', '1.1.2.1')], ps.members)
        self.assertEqual('2006/11/20 17:38:15', ps.date)
        self.assertEqual(1164044295.000, ps.timestamp)
        self.assertEqual(0, ps.time_offset)

        ps = self._patchsets[5]
        self.assertEqual(6, ps.num)
        self.assertEqual('jameinel', ps.author)
        self.assertEqual('HEAD', ps.branch)
        self.assertEqual(None, ps.ancestor_branch)
        self.assertEqual(None, ps.tag)
        self.assertEqual(["Adding 'e' on HEAD\n"], ps.log)
        self.assertEqual([('b', '1.3')], ps.members)
        self.assertEqual('2006/11/20 17:43:10', ps.date)
        self.assertEqual(1164044590.000, ps.timestamp)
        self.assertEqual(0, ps.time_offset)

        ps = self._patchsets[6]
        self.assertEqual(7, ps.num)
        self.assertEqual('jameinel', ps.author)
        self.assertEqual('other', ps.branch)
        self.assertEqual('mytest', ps.ancestor_branch)
        self.assertEqual(None, ps.tag)
        self.assertEqual(["Adding 'other' to b on 'other'\n"], ps.log)
        self.assertEqual([('b', '1.1.2.1.2.1')], ps.members)
        self.assertEqual('2006/11/20 17:43:39', ps.date)
        self.assertEqual(1164044619.000, ps.timestamp)
        self.assertEqual(0, ps.time_offset)

        ps = self._patchsets[7]
        self.assertEqual(8, ps.num)
        self.assertEqual('jameinel', ps.author)
        self.assertEqual('other', ps.branch)
        self.assertEqual(None, ps.ancestor_branch)
        self.assertEqual(None, ps.tag)
        self.assertEqual(["Multiline\n",
                          "commit describing a bogus\n",
                          "PatchSet 2\n",
                          "message for adding 'baz' to foo/c\n",
                         ], ps.log)
        self.assertEqual([('foo/c', '1.1.2.1.2.1')], ps.members)
        self.assertEqual('2006/11/20 18:03:29', ps.date)
        self.assertEqual(1164045809.000, ps.timestamp)
        self.assertEqual(0, ps.time_offset)

        ps = self._patchsets[8]
        self.assertEqual(9, ps.num)
        self.assertEqual('jameinel', ps.author)
        self.assertEqual('other', ps.branch)
        self.assertEqual(None, ps.ancestor_branch)
        self.assertEqual(None, ps.tag)
        self.assertEqual(["multiline commit\n",
                          "\n",
                          "with blank line\n",
                         ], ps.log)
        self.assertEqual([('b', '1.1.2.1.2.2')], ps.members)
        self.assertEqual('2006/11/20 18:15:47', ps.date)
        self.assertEqual(1164046547.000, ps.timestamp)
        self.assertEqual(0, ps.time_offset)

    def test_iso_8859_1(self):
        self.parse(_ISO_8859_1_DUMP)
        self.assertEqual(1, len(self._patchsets))

        ps = self._patchsets[0]
        self.assertEqual(1, ps.num)
        self.assertEqual(u'j\xe5meinel', ps.author)
        self.assertEqual('HEAD', ps.branch)
        self.assertEqual(None, ps.ancestor_branch)
        self.assertEqual(None, ps.tag)
        self.assertEqual([u"message with '\xb5\xf1\xed\xe7\xf6d\xe9'\n"],
                         ps.log)

    def test_utf_8(self):
        self.parse(_UTF_8_DUMP, parser_kwargs={'encoding':'utf-8'})
        self.assertEqual(1, len(self._patchsets))

        ps = self._patchsets[0]
        self.assertEqual(1, ps.num)
        self.assertEqual(u'j\xe5meinel', ps.author)
        self.assertEqual('HEAD', ps.branch)
        self.assertEqual(None, ps.ancestor_branch)
        self.assertEqual(None, ps.tag)
        self.assertEqual([u"message with '\xb5\xf1\xed\xe7\xf6d\xe9'\n"],
                         ps.log)
