# Written by Bram Cohen
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: CurrentRateMeasure.py 401 2008-07-26 00:36:45Z camrdale-guest $

"""Measuring rates of download and upload."""

from clock import clock

class Measure:
    """The measurement of one rate.

    This keeps track of both the current rate and the total
    amount of data sent or received.

    For DebTorrent, it can keep a subtotal of the amount of data
    transferred via HTTP.  Currently the HTTP rate is not
    calculated, just the total.

    @type max_rate_period: C{float}
    @ivar max_rate_period: maximum amount of time to guess the current rate 
        estimate represents
    @type ratesince: C{float}
    @ivar ratesince: the oldest time the rate estimate is for
    @type last: C{float}
    @ivar last: the last time the rate was updated
    @type rate: C{float}
    @ivar rate: the latest calculated rate
    @type total: C{long}
    @ivar total: the total amount that went in to calculating the rate
    @type httptotal: C{long}
    @ivar httptotal: the total amount of HTTP data (also included in the total)
    
    """
    
    def __init__(self, max_rate_period, fudge = 1, saved_total = 0L, http_total = 0L):
        """Initialize the measurement.
        
        @type max_rate_period: C{float}
        @param max_rate_period: maximum amount of time to guess the current 
            rate estimate represents
        @type fudge: C{int}
        @param fudge: time equivalent of writing to kernel-level TCP buffer, 
            for rate adjustment (optional, defaults to 1)
        @type saved_total: C{long}
        @param saved_total: the saved amount measured from a previous run
            (optional, defaults to 0)
        @param http_total: the saved amount measured from a previous run
            (optional, defaults to 0)
        
        """
        
        self.max_rate_period = max_rate_period
        self.ratesince = clock() - fudge
        self.last = self.ratesince
        self.rate = 0.0
        self.total = long(saved_total)
        self.httptotal = long(http_total)

    def update_rate(self, amount):
        """Update the rate with new data.
        
        @type amount: C{long}
        @param amount: the new data to add into the rate calculation
        
        """
        
        self.total += amount
        t = clock()
        self.rate = (self.rate * (self.last - self.ratesince) + 
            amount) / (t - self.ratesince + 0.0001)
        self.last = t
        if self.ratesince < t - self.max_rate_period:
            self.ratesince = t - self.max_rate_period

    def update_http_rate(self, amount):
        """Update the rate with new data.
        
        This new data is added to the figures returned by all of
        C{get_rate}, C{get_total} and C{get_http_subtotal}.

        @type amount: C{long}
        @param amount: the new data to add into the rate calculation

        """

        self.httptotal += amount
        self.update_rate(amount)
        
    def get_rate(self):
        """Get the current rate measurement.
        
        @rtype: C{float}
        @return: the current rate
        
        """
        
        self.update_rate(0)
        return self.rate

    def get_rate_noupdate(self):
        """Get the current rate measurement without updating it.
        
        @rtype: C{float}
        @return: the current rate
        
        """
        
        return self.rate

    def time_until_rate(self, newrate):
        """Calculate how long until the rate drops to the target.
        
        @type newrate: C{float}
        @param newrate: the target rate
        @rtype: C{float}
        @return: the number of seconds until the rate decreases to the target 
            rate, or 0 if it's already there (or below it)
        
        """
        
        if self.rate <= newrate:
            return 0
        t = clock() - self.ratesince
        return ((self.rate * t) / newrate) - t

    def get_total(self):
        """Get the total amount used to calculate the rate..
        
        @rtype: C{float}
        @return: the total amount
        
        """
        
        return self.total

    def get_http_subtotal(self):
        """Get the amount transferred via HTTP only
        
        @rtype: C{float}
        @return: the total amount
        
        """

        return self.httptotal
