# Written by Bram Cohen
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: RawServer.py 270 2007-08-19 07:33:13Z camrdale-guest $

"""Manage connections to and tasks to be run on the server.

@type logger: C{logging.Logger}
@var logger: the logger to send all log messages to for this module
@type READSIZE: C{int}
@var READSIZE: the maximum amount of data to read from any sockets

"""

from bisect import insort
from SocketHandler import SocketHandler
import socket
from cStringIO import StringIO
from traceback import print_exc
from select import error
from threading import Thread, Event
from time import sleep
from clock import clock
from signal import signal, SIGINT, SIG_DFL
import sys, logging

logger = logging.getLogger('DebTorrent.RawServer')

def autodetect_ipv6():
    """Detect whether IPv6 connections are supported (not used)."""
    try:
        assert sys.version_info >= (2,3)
        assert socket.has_ipv6
        socket.socket(socket.AF_INET6, socket.SOCK_STREAM)
    except:
        return 0
    return 1

def autodetect_socket_style():
    """Determine if an IPv6 server socket will also field IPv4 connections."""
    if sys.platform.find('linux') < 0:
        return 1
    else:
        try:
            f = open('/proc/sys/net/ipv6/bindv6only','r')
            dual_socket_style = int(f.read())
            f.close()
            return int(not dual_socket_style)
        except:
            return 0


READSIZE = 32768

class RawServer:
    """Manage connections and tasks like a server.
    
    Mostly just manages the tasks, any socket related duties are passed on to 
    the L{SocketHandler.SocketHandler} instance.
    
    @type timeout_check_interval: C{float}
    @ivar timeout_check_interval: seconds to wait between checking if any 
        connections have timed out
    @type timeout: C{float}
    @ivar timeout: seconds to wait between closing sockets on which 
        nothing has been received on
    @type servers: C{dictionary}
    @ivar servers: not used
    @type single_sockets: C{dictionary}
    @ivar single_sockets: not used
    @type dead_from_write: C{list}
    @ivar dead_from_write: not used
    @type doneflag: C{threading.Event}
    @ivar doneflag: flag to indicate the program is to be shutdown
    @type exccount: C{int}
    @ivar exccount: number of exceptions that have occurred
    @type funcs: C{list} of (C{float}, C{method}, unknown)
    @ivar funcs: the list of future methods to invoke, the time to invoke at,
        the method, and the ID to use to identify the method
    @type externally_added: C{list} of (C{float}, C{method}, unknown)
    @ivar externally_added: externally queued tasks to add to L{funcs}
    @type finished: C{threading.Event}
    @ivar finished: whether the server is done listening
    @type tasks_to_kill: C{list}
    @ivar tasks_to_kill: the IDs of tasks to remove from L{funcs}
    @type excflag: C{threading.Event}
    @ivar excflag: the flag to use to indicate an exception has occurred
    @type sockethandler: L{SocketHandler.SocketHandler}
    @ivar sockethandler: the handler to use to manage all open sockets
    
    """
    
    def __init__(self, doneflag, timeout_check_interval, timeout,
                 ipv6_enable = True, sockethandler = None, excflag = Event()):
        """Initialize the instance and start the socket handler.
        
        @type doneflag: C{threading.Event}
        @param doneflag: flag to indicate the program is to be shutdown
        @type timeout_check_interval: C{float}
        @param timeout_check_interval: seconds to wait between checking if any 
            connections have timed out
        @type timeout: C{float}
        @param timeout: seconds to wait between closing sockets on which 
            nothing has been received on
        @type ipv6_enable: C{boolean}
        @param ipv6_enable: allow the client to connect to peers via IPv6
            (optional, defaults to True)
        @type sockethandler: L{SocketHandler.SocketHandler}
        @param sockethandler: the handler to use to manage all open sockets
            (optional, defaults to creating a new one)
        @type excflag: C{threading.Event}
        @param excflag: the flag to use to indicate an exception has occurred
            (optional, defaults to using a new flag)
        
        """
        
        self.timeout_check_interval = timeout_check_interval
        self.timeout = timeout
        self.servers = {}
        self.single_sockets = {}
        self.dead_from_write = []
        self.doneflag = doneflag
        self.exccount = 0
        self.funcs = []
        self.externally_added = []
        self.finished = Event()
        self.tasks_to_kill = []
        self.excflag = excflag
        
        if sockethandler is None:
            sockethandler = SocketHandler(timeout, ipv6_enable, READSIZE)
        self.sockethandler = sockethandler
        self.add_task(self.scan_for_timeouts, timeout_check_interval)

    def get_exception_flag(self):
        """Get the flag used to indicate exceptions.
        
        @rtype: C{threading.Event}
        @return: the exception flag
        
        """
        
        return self.excflag

    def _add_task(self, func, delay, id = None):
        """Add the task to the sorted list of tasks to execute.
        
        @type func: C{method}
        @param func: the task to be run
        @type delay: C{int}
        @param delay: the number of seconds to delay before running the task
        @type id: unknown
        @param id: an identifier to later find the task by
        
        """
        
        assert float(delay) >= 0
        insort(self.funcs, (clock() + delay, func, id))

    def add_task(self, func, delay = 0, id = None):
        """Add the task to the list of tasks to schedule.
        
        @type func: C{method}
        @param func: the task to be run
        @type delay: C{int}
        @param delay: the number of seconds to delay before running the task
        @type id: unknown
        @param id: an identifier to later find the task by
        
        """
        
        assert float(delay) >= 0
        self.externally_added.append((func, delay, id))

    def scan_for_timeouts(self):
        """Scan the open sockets for any timeouts."""
        self.add_task(self.scan_for_timeouts, self.timeout_check_interval)
        self.sockethandler.scan_for_timeouts()

    def bind(self, port, bind = '', reuse = False,
                        ipv6_socket_style = 1):
        """Bind to listen on a single port.
        
        @type port: C{int}
        @param port: the port to listen on
        @type bind: C{string}
        @param bind: the IP address to bind to (optional, defaults to all)
        @type reuse: C{boolean}
        @param reuse: whether to use SO_REUSEADDR to bind (optional, defaults 
            to False). This allows the bind to work if the socket is still
            open in the TIME_WAIT state from a recently shutdown server.
        @type ipv6_socket_style: C{int}
        @param ipv6_socket_style: whether an IPv6 server socket will also 
            field IPv4 connections (optional, defaults to yes)
        
        """
        
        self.sockethandler.bind(port, bind, reuse, ipv6_socket_style)

    def find_and_bind(self, minport, maxport, bind = '', reuse = False,
                      ipv6_socket_style = 1, randomizer = False):
        """Bind to listen on a single port within a range.
        
        @type minport: C{int}
        @param minport: the minimum port to listen on
        @type maxport: C{int}
        @param maxport: the maximum port to listen on
        @type bind: C{string}
        @param bind: the addresses to bind to (optional, defaults to the 
            default IPv6 and IPv4 addreses). Parsed as a comma seperated 
            list (can be IP addresses or hostnames).
        @type reuse: C{boolean}
        @param reuse: whether to use SO_REUSEADDR to bind (optional, defaults 
            to False). This allows the bind to work if the socket is still
            open in the TIME_WAIT state from a recently shutdown server.
        @type ipv6_socket_style: C{int}
        @param ipv6_socket_style: whether an IPv6 server socket will also 
            field IPv4 connections (optional, defaults to yes)
        @type randomizer: C{boolean}
        @param randomizer: whether to randomize the range or use it sequentially
        @rtype: C{int}
        @return: the port that was bound to
        
        """
        
        return self.sockethandler.find_and_bind(minport, maxport, bind, reuse,
                                 ipv6_socket_style, randomizer)

    def start_connection_raw(self, dns, socktype, handler = None):
        """Initiate a new connection to a peer (setting the type of socket).
        
        @type dns: (C{string}, C{int})
        @param dns: the IP address and port number to contact the peer on
        @type socktype: C{int}
        @param socktype: the type of socket to open
        @type handler: unknown
        @param handler: the data handler to use to process data on the connection
            (optional, defaults to using the defualt handler)
        @rtype: L{SocketHandler.SingleSocket}
        @return: the new connection made to the peer
        
        """
        
        return self.sockethandler.start_connection_raw(dns, socktype, handler)

    def start_connection(self, dns, handler = None, randomize = False):
        """Initiate a new connection to a peer.
        
        @type dns: (C{string}, C{int})
        @param dns: the IP address and port number to contact the peer on
        @type handler: unknown
        @param handler: the data handler to use to process data on the connection
            (optional, defaults to using the defualt handler)
        @type randomize: C{boolean}
        @param randomize: whether to randomize the possible sockets or 
            choose one sequentially
        @rtype: L{SocketHandler.SingleSocket}
        @return: the new connection made to the peer
        
        """
        
        return self.sockethandler.start_connection(dns, handler, randomize)

    def get_stats(self):
        """Get some information about the bound interfaces and ports.
        
        @rtype: C{dictionary}
        @return: info about the bound interfaces
        
        """
        
        return self.sockethandler.get_stats()

    def pop_external(self):
        """Add the waiting externally added tasks to the list of tasks to process."""
        while self.externally_added:
            (a, b, c) = self.externally_added.pop(0)
            self._add_task(a, b, c)


    def sighandler(self, signalnum, frame):
        """Properly handle a SIGINT.
        
        @type signalnum: C{int}
        @param signalnum: the signal that was received
        @type frame: C{frame}
        @param frame: the current stack frame (not used)
        @raise KeyboardInterrupt: always
        
        """
        
        logger.info("Received signal: "+str(signalnum))
        raise KeyboardInterrupt

    def listen_forever(self, handler):
        """Start the server listening on sockets and processing tasks.
        
        @type handler: unknown
        @param handler: the default data handler to use to process data on connections
        @rtype: C{boolean}
        @return: whether the server should be restarted
        
        """
        
        self.sockethandler.set_handler(handler)
        signal(SIGINT, self.sighandler)
        try:
            while not self.doneflag.isSet():
                try:
                    self.pop_external()
                    self._kill_tasks()
                    if self.funcs:
                        period = self.funcs[0][0] + 0.001 - clock()
                    else:
                        period = 2 ** 30
                    if period < 0:
                        period = 0
                    events = self.sockethandler.do_poll(period)
                    if self.doneflag.isSet():
                        return True
                    while self.funcs and self.funcs[0][0] <= clock():
                        garbage1, func, id = self.funcs.pop(0)
                        if id in self.tasks_to_kill:
                            pass
                        try:
#                            logger.debug(func.func_name)
                            func()
                        except (SystemError, MemoryError), e:
                            logger.exception('Occurred while running '+func.__name__)
                            return True
                        except KeyboardInterrupt:
                            signal(SIGINT, SIG_DFL)
                            self.exception(True)
                            return False
                        except:
                            self.exception()
                    self.sockethandler.close_dead()
                    self.sockethandler.handle_events(events)
                    if self.doneflag.isSet():
                        return True
                    self.sockethandler.close_dead()
                except (SystemError, MemoryError), e:
                    logger.exception('Occurred while processing queued functions')
                    return True
                except error:
                    if self.doneflag.isSet():
                        return True
                except KeyboardInterrupt:
                    signal(SIGINT, SIG_DFL)
                    self.exception(True)
                    return False
                except:
                    self.exception()
                if self.exccount > 10:
                    return True
        finally:
#            self.sockethandler.shutdown()
            self.finished.set()

    def is_finished(self):
        """Check if the server is done listening.
        
        @rtype: C{boolean}
        @return: whether the server is done listeneing
        
        """
        
        return self.finished.isSet()

    def wait_until_finished(self):
        """Wait until the server is done listeneing."""
        self.finished.wait()

    def _kill_tasks(self):
        """Remove the pending list of tasks to remove from those tasks still pending."""
        if self.tasks_to_kill:
            new_funcs = []
            for (t, func, id) in self.funcs:
                if id not in self.tasks_to_kill:
                    new_funcs.append((t, func, id))
            self.funcs = new_funcs
            self.tasks_to_kill = []

    def kill_tasks(self, id):
        """Remove tasks from the list of those pending to execute.
        
        @type id: unknown
        @param id: an identifier find the tasks by
        
        """
        
        self.tasks_to_kill.append(id)

    def exception(self, kbint = False):
        """Print an exception that has occurred.
        
        @type kbint: C{boolean}
        @param kbint: whether the exception was from a KeyboardInterrupt 
            (optional, defaults to False)
        
        """
        
        if not kbint:
            self.excflag.set()
        self.exccount += 1
        logger.exception('RawServer exception occurred')

    def set_handler(self, handler, port = None):
        """Set the handler to use for a port (or the default handler).
        
        @type handler: unknown
        @param handler: the data handler to use to process data on the port
        @type port: C{int}
        @param port: the port to use the handler for
            (optional, defaults to setting the default handler)
        
        """
        
        self.sockethandler.set_handler(handler, port)

    def shutdown(self):
        """Shutdown the socket handler."""
        self.sockethandler.shutdown()
