# Written by Bram Cohen, Uoti Urpala, and John Hoffman
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: bitfield.py 333 2008-01-19 21:32:11Z camrdale-guest $

"""Manage creation and modification of bitfields.

@type lookup_table: C{list} of C{tuple} of C{boolean}
@var lookup_table: A lookup table for converting characters to a tuple of
    booleans. Each tuple has 8 booleans, which are true for all the bits
    in the character that are set.
@type reverse_lookup_table: C{dictionary}
@var reverse_lookup_table: A reverse mapping of lookup_table. The keys are 
    the tuples of 8 booleans, which map to values that are the characters
    the tuples were created from.

"""

negsum = lambda a: len(a)-sum(a)
    
def _int_to_booleans(x):
    """Convert an integer to a list of booleans.
    
    @type x: C{int}
    @param x: the integer to convert
    @rtype: C{tuple} of C{list} of C{boolean}
    @return: the list of booleans
    
    """
    
    r = []
    for i in range(8):
        r.append(bool(x & 0x80))
        x <<= 1
    return tuple(r)

lookup_table = []
reverse_lookup_table = {}
for i in xrange(256):
    x = _int_to_booleans(i)
    lookup_table.append(x)
    reverse_lookup_table[x] = chr(i)


class Bitfield:
    """ A bitfield, a indicating the pieces a peer has.
    
    @type length: C{int}
    @ivar length: the length of the bitfield
    @type array: C{list} of C{boolean}
    @ivar array: the bitfield
    @type numfalse: C{int}
    @ivar numfalse: the number of false entries in the bitfield
    
    """
    
    def __init__(self, length = None, bitstring = None, copyfrom = None):
        """
        
        @type length: C{int}
        @param length: the length of the bitfield to create
            (optional, if missing use length of copyfrom)
        @type bitstring: C{string}
        @param bitstring: the bitfield string to initialize the bitfield from
            (optional, default is to initialize all to false)
        @type copyfrom: L{Bitfield}
        @param copyfrom: another bitfield to make a copy of
            (optional, default is to create a new empty one)
        
        """
        
        if copyfrom is not None:
            self.length = copyfrom.length
            self.array = copyfrom.array[:]
            self.numfalse = copyfrom.numfalse
            return
        if length is None:
            raise ValueError, "length must be provided unless copying from another array"
        self.length = length
        if bitstring is not None:
            # Construct the bitfield
            t = lookup_table
            r = []
            for c in bitstring:
                r.extend(t[ord(c)])
                if len(r) > length:
                    # Stop if enough bits have been found
                    break
                
            if len(r) > length:
                # Remove any extra bits that were added
                del r[length:]
            elif len(r) < length:
                # Add any missing bits as all False
                r.extend([False]*(length - len(r)))
                
            self.array = r
            self.numfalse = negsum(r)
        else:
            self.array = [False] * length
            self.numfalse = length

    def __setitem__(self, index, val):
        """Set one of the bitfield entries.
        
        @type index: C{int}
        @param index: the index to set
        @type val: C{boolean}
        @param val: the value to set to
        
        """
        
        val = bool(val)
        self.numfalse += self.array[index]-val
        self.array[index] = val

    def __getitem__(self, index):
        """Get one of the bitfield entries.
        
        @type index: C{int}
        @param index: the index to get
        @rtype: C{boolean}
        @return: the value of the bitfield entry
        
        """
        
        return self.array[index]

    def __len__(self):
        """Get the length of the bitfield.
        
        @rtype:C{int}
        @return: the length of the bitfield
        
        """
        
        return self.length

    def tostring(self):
        """Convert the bitfield to a string
        
        @rtype: C{string}
        @return: the bitfield represented as a string
        
        """
        
        booleans = self.array
        t = reverse_lookup_table
        s = len(booleans) % 8
        r = [ t[tuple(booleans[x:x+8])] for x in xrange(0, len(booleans)-s, 8) ]
        if s:
            r += t[tuple(booleans[-s:] + ([0] * (8-s)))]
        return ''.join(r)

    def complete(self):
        """Check if the bitfield is all true.
        
        @rtype: C{boolean}
        @return: whether the bitfield is complete (all true)
        
        """
        
        return not self.numfalse


def test_bitfield():
    """Test the bitfield implementation."""
    try:
        x = Bitfield(7, 'ab')
        assert False
    except ValueError:
        pass
    try:
        x = Bitfield(7, 'ab')
        assert False
    except ValueError:
        pass
    try:
        x = Bitfield(9, 'abc')
        assert False
    except ValueError:
        pass
    try:
        x = Bitfield(0, 'a')
        assert False
    except ValueError:
        pass
    try:
        x = Bitfield(1, '')
        assert False
    except ValueError:
        pass
    try:
        x = Bitfield(7, '')
        assert False
    except ValueError:
        pass
    try:
        x = Bitfield(8, '')
        assert False
    except ValueError:
        pass
    try:
        x = Bitfield(9, 'a')
        assert False
    except ValueError:
        pass
    try:
        x = Bitfield(7, chr(1))
        assert False
    except ValueError:
        pass
    try:
        x = Bitfield(9, chr(0) + chr(0x40))
        assert False
    except ValueError:
        pass
    assert Bitfield(0, '').tostring() == ''
    assert Bitfield(1, chr(0x80)).tostring() == chr(0x80)
    assert Bitfield(7, chr(0x02)).tostring() == chr(0x02)
    assert Bitfield(8, chr(0xFF)).tostring() == chr(0xFF)
    assert Bitfield(9, chr(0) + chr(0x80)).tostring() == chr(0) + chr(0x80)
    x = Bitfield(1)
    assert x.numfalse == 1
    x[0] = 1
    assert x.numfalse == 0
    x[0] = 1
    assert x.numfalse == 0
    assert x.tostring() == chr(0x80)
    x = Bitfield(7)
    assert len(x) == 7
    x[6] = 1
    assert x.numfalse == 6
    assert x.tostring() == chr(0x02)
    x = Bitfield(8)
    x[7] = 1
    assert x.tostring() == chr(1)
    x = Bitfield(9)
    x[8] = 1
    assert x.numfalse == 8
    assert x.tostring() == chr(0) + chr(0x80)
    x = Bitfield(8, chr(0xC4))
    assert len(x) == 8
    assert x.numfalse == 5
    assert x.tostring() == chr(0xC4)
