// ---------------------------------------------------------------------
// $Id: Optimizer.hh 153 2008-06-04 05:49:38Z daaugusto $
//
//   Optimizer.hh (created on Thu Nov 17 18:25:35 BRT 2005)
// 
//   Genetic Algorithm File Fitter (gaffitter)
//
//   Copyright (C) 2005-2008 Douglas A. Augusto
// 
// This file is part of gaffitter.
// 
// gaffitter is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 3 of the License, or (at
// your option) any later version.
// 
// gaffitter is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with gaffitter; if not, see <http://www.gnu.org/licenses/>.
//
// ---------------------------------------------------------------------

#ifndef Optimizer_hh
#define Optimizer_hh

#include "Input.hh"

#include <vector>
#include <iostream>

using namespace std;

// ---------------------------------------------------------------------
/** Base class for the search algorithms. */
class Optimizer {
public:
   /** Initialize some variables */
   Optimizer(vector<SizeName>& sn, Params& p): 
             m_files(sn), m_params(p), m_best(0), m_cur_iter(0)
   {
      if (m_params.m_verbose) cout << "> Searching... " << flush << endl;
   }

private:
   /** Copies are not allowed. */
   Optimizer(const Optimizer&);
   /** Attributions are not allowed. */
   Optimizer& operator= (const Optimizer&);

public:
   /**
    * Frees memory pointed by m_best (vector)
    */
   virtual ~Optimizer() { delete m_best; }
   /**
    * Searches for a optimal solution (via Genetic Algorithm, Best
    * First or Brute Force)
    */
   virtual void Evolve() = 0;
   /**
    * Print the result to the standard output.
    *
    * Print the selected files, a brief summary (sum, diff and number of
    * selected files) and optionally do this for the unselected files
    * too.
    */
   bool Output();
   /**
    * This overloaded operator prints the name and parameters of the
    * search algorithms
    */
   friend ostream& operator<< (ostream&, const Optimizer&);

protected:
   /**
    * Write to streams the name and parameters of the search algorithms
    */
   virtual ostream& Write(ostream& s) const
   {
      s << "> Target: " << m_params.PrettySize(m_params.m_target) 
        << endl
        << "> Input size: " << m_files.size()
        << endl;

      return s;
   }

protected:
   /**
    * Calculate the total space used by the items for the given vector.
    */
   Params::Size_t CalcSum(const vector<bool>&) const;
   /**
    * Evaluate candidates.
    *
    * - Zero is the best fitness (perfect fit)
    * - Positive values indicate size sum less than target
    * - Negative values indicate size sum greater than target
    */
   Params::Size_t Evaluate(const vector<bool>& v) const { return Diff(v); }
   /**
    * Calculate the distance (diff) between the input target and a given size.
    */
   Params::Size_t Diff(Params::Size_t s) const { return m_params.m_target-s; }
   /**
    * Calculate the distance (diff) between the input target and a given
    * vector of items.
    */
   Params::Size_t Diff(const vector<bool>& v) const { return Diff(CalcSum(v)); }
   /**
    * Reset some variables
    */
   virtual void Reset() { delete m_best; m_best = 0; }
   /**
    * Vector of input files/sizes
    */
   vector<SizeName>& m_files;
   /**
    * Global parameters.
    */
   Params& m_params;

protected:
   /**
    * \var m_best
    *
    * Pointer to the best solution found
    */
   vector<bool>* m_best;

private:
   /**
    * This data member holds the current iteration. \sa Output()
    */
   int m_cur_iter;
};

// --------------------------------------------------------------------
inline ostream& operator<< (ostream& s, const Optimizer& optimizer)
{
   return optimizer.Write(s);
}

// --------------------------------------------------------------------
#endif
