/*
 *  Copyright (C) 2000, 2001, 2002 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon-embed-popup.h"
#include "galeon-embed-event.h"
#include "galeon-embed-shell.h"
#include "galeon-embed-utils.h"
#include "prefs-strings.h"
#include "eel-gconf-extensions.h"
#include "galeon-config.h"
#include "gul-bonobo-extensions.h"
#include "gul-string.h"
#include "galeon-embed-helper-list.h"
#include "galeon-marshal.h"

#include <string.h>
#include <glib/gi18n.h>
#include <bonobo/bonobo-ui-component.h>
#include <gtk/gtkclipboard.h>
#include <gtk/gtkmain.h>
#include <libgnome/gnome-exec.h>
#include <libgnomevfs/gnome-vfs-uri.h>

typedef enum
{
	EMBED_POPUP_INPUT,
	EMBED_POPUP_DOCUMENT,
	EMBED_POPUP_ELEMENT,
	EMBED_POPUP_SELECTION
} EmbedPopupType;

#define GALEON_EMBED_POPUP_GET_PRIVATE(object) (G_TYPE_INSTANCE_GET_PRIVATE ((object), \
				       GALEON_TYPE_EMBED_POPUP, GaleonEmbedPopupPrivate))


struct GaleonEmbedPopupPrivate
{
	GaleonEmbedHelperList *helper_list;
	GaleonEmbedEvent *event;
	GaleonEmbed *embed;
	EmbedEventContext context;
	BonoboUIComponent *ui_component;
	EmbedPopupType popup_type;
};

static void
galeon_embed_popup_class_init (GaleonEmbedPopupClass *klass);
static void
galeon_embed_popup_init (GaleonEmbedPopup *gep);
static void
galeon_embed_popup_finalize (GObject *object);
static void 
embed_popup_copy_location_cmd (BonoboUIComponent *uic, 
                               GaleonEmbedPopup *popup, 
                               const char* verbname);
static void 
embed_popup_copy_email_cmd (BonoboUIComponent *uic, 
                            GaleonEmbedPopup *popup, 
                            const char* verbname);
static void 
embed_popup_copy_link_location_cmd (BonoboUIComponent *uic, 
                      	            GaleonEmbedPopup *popup, 
                                    const char* verbname);
static void 
embed_popup_download_link_cmd (BonoboUIComponent *uic, 
                      	       GaleonEmbedPopup *popup, 
                               const char* verbname);
static void 
embed_popup_save_image_as_cmd (BonoboUIComponent *uic, 
                      	       GaleonEmbedPopup *popup, 
                               const char* verbname);
static void 
embed_popup_set_image_as_background_cmd (BonoboUIComponent *uic, 
                      	       		 GaleonEmbedPopup *popup, 
                               		 const char* verbname);
static void 
embed_popup_copy_image_location_cmd (BonoboUIComponent *uic, 
                      	       	     GaleonEmbedPopup *popup, 
                                     const char* verbname);
static void 
embed_popup_save_page_as_cmd (BonoboUIComponent *uic, 
                      	      GaleonEmbedPopup *popup, 
                              const char* verbname);
static void 
embed_popup_save_background_as_cmd (BonoboUIComponent *uic, 
                      	       	    GaleonEmbedPopup *popup, 
                                    const char* verbname);
static void 
embed_popup_open_frame_cmd (BonoboUIComponent *uic, 
                      	    GaleonEmbedPopup *popup, 
                            const char* verbname);
static void 
embed_popup_open_image_cmd (BonoboUIComponent *uic, 
                      	    GaleonEmbedPopup *popup, 
                            const char* verbname);
static void
embed_popup_copy_to_clipboard (GaleonEmbedPopup *popup, const char *text);

static GObjectClass *parent_class = NULL;

/**
 * Signals enums and ids
 */
enum GaleonEmbedPopupSignalsEnum {
	GALEON_EMBED_POPUP_BEFORE_SHOW,
	GALEON_EMBED_POPUP_LAST_SIGNAL
};
static gint GaleonEmbedPopupSignals[GALEON_EMBED_POPUP_LAST_SIGNAL];


#define DOCUMENT_POPUP_PATH "/popups/GaleonEmbedDocumentPopup"
#define ELEMENT_POPUP_PATH "/popups/GaleonEmbedElementPopup"
#define INPUT_POPUP_PATH "/popups/GaleonEmbedInputPopup"
#define SELECTION_POPUP_PATH "/popups/GaleonEmbedSelectionPopup"

#define GALEON_POPUP_NAVIGATION_ITEMS_PLACEHOLDER "/popups/GaleonEmbedDocumentPopup/NavigationItems"
#define GALEON_POPUP_LINK_ITEMS_PLACEHOLDER "/popups/GaleonEmbedElementPopup/LinkItems"
#define GALEON_POPUP_EMAIL_LINK_ITEMS_PLACEHOLDER "/popups/GaleonEmbedElementPopup/EmailLinkItems"
#define GALEON_POPUP_IMAGE_ITEMS_PLACEHOLDER "/popups/GaleonEmbedElementPopup/ImageItems"
#define GALEON_POPUP_FRAME_ITEMS_PLACEHOLDER "/popups/GaleonEmbedDocumentPopup/FrameItems"
#define GALEON_POPUP_BETWEEN_ELEMENTS1_PLACEHOLDER "/popups/GaleonEmbedElementPopup/BetweenElements1"
#define GALEON_POPUP_SAVE_BG_PATH "/commands/DPSaveBackgroundAs"
#define GALEON_POPUP_DOC_OPENWITH_PATH "/popups/GaleonEmbedDocumentPopup/DPOpenWith"
#define GALEON_POPUP_IMAGE_OPENWITH_PATH "/popups/GaleonEmbedElementPopup/ImageItems/EPOpenImageWith"
#define GALEON_POPUP_OPEN_IMAGE_PATH "/commands/EPOpenImage"

/*This command is from src/popup-commands.c but manipulating item label
 *text must be done here
 */
#define GALEON_POPUP_BLOCK_IMAGE_SITE_PATH "/commands/EPBlockImageSite"
#define GALEON_POPUP_ALLOW_IMAGE_SITE_PATH "/commands/EPAllowImageSite"

BonoboUIVerb embed_popup_verbs [] = {
	BONOBO_UI_VERB ("EPCopyLinkLocation", (BonoboUIVerbFn)embed_popup_copy_link_location_cmd),
	BONOBO_UI_VERB ("EPDownloadLink", (BonoboUIVerbFn)embed_popup_download_link_cmd),
	BONOBO_UI_VERB ("EPOpenImage", (BonoboUIVerbFn)embed_popup_open_image_cmd),
	BONOBO_UI_VERB ("EPSaveImageAs", (BonoboUIVerbFn)embed_popup_save_image_as_cmd),
	BONOBO_UI_VERB ("EPSetImageAsBackground", (BonoboUIVerbFn)embed_popup_set_image_as_background_cmd),
	BONOBO_UI_VERB ("EPCopyImageLocation", (BonoboUIVerbFn)embed_popup_copy_image_location_cmd),
	
	BONOBO_UI_VERB ("DPCopyLocation", (BonoboUIVerbFn)embed_popup_copy_location_cmd),
	BONOBO_UI_VERB ("EPCopyEmail", (BonoboUIVerbFn)embed_popup_copy_email_cmd),
        BONOBO_UI_VERB ("DPSavePageAs", (BonoboUIVerbFn)embed_popup_save_page_as_cmd),
	BONOBO_UI_VERB ("DPSaveBackgroundAs", (BonoboUIVerbFn)embed_popup_save_background_as_cmd),
	BONOBO_UI_VERB ("DPOpenFrame", (BonoboUIVerbFn)embed_popup_open_frame_cmd),
	
	BONOBO_UI_VERB_END
};

GType
galeon_embed_popup_get_type (void)
{
       static GType galeon_embed_popup_type = 0;

        if (galeon_embed_popup_type == 0)
        {
                static const GTypeInfo our_info =
                {
                        sizeof (GaleonEmbedPopupClass),
                        NULL, /* base_init */
                        NULL, /* base_finalize */
                        (GClassInitFunc) galeon_embed_popup_class_init,
                        NULL, /* class_finalize */
                        NULL, /* class_data */
                        sizeof (GaleonEmbedPopup),
                        0,    /* n_preallocs */
                        (GInstanceInitFunc) galeon_embed_popup_init
                };

		
                galeon_embed_popup_type = g_type_register_static (G_TYPE_OBJECT,
								  "GaleonEmbedPopup",
								  &our_info, 0);
        }

        return galeon_embed_popup_type;
}

static void
galeon_embed_popup_class_init (GaleonEmbedPopupClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = (GObjectClass *) g_type_class_peek_parent (klass);

        object_class->finalize = galeon_embed_popup_finalize;

	klass->show = NULL; /* abstract */

	GaleonEmbedPopupSignals[GALEON_EMBED_POPUP_BEFORE_SHOW] = g_signal_new (
		"before-show", G_OBJECT_CLASS_TYPE (klass),  
		G_SIGNAL_RUN_FIRST | G_SIGNAL_RUN_LAST | G_SIGNAL_RUN_CLEANUP,
                G_STRUCT_OFFSET (GaleonEmbedPopupClass, before_show), 
		NULL, NULL, 
		galeon_marshal_VOID__VOID,
		G_TYPE_NONE, 0);


	g_type_class_add_private (klass, sizeof (GaleonEmbedPopupPrivate));
}

static void
galeon_embed_popup_init (GaleonEmbedPopup *gep)
{
        gep->priv = GALEON_EMBED_POPUP_GET_PRIVATE (gep);
	gep->priv->embed = NULL;
	gep->priv->event = NULL;
	gep->priv->ui_component = NULL;
	gep->priv->helper_list = galeon_embed_helper_list_new ();
}

static void
galeon_embed_popup_finalize (GObject *object)
{
	GaleonEmbedPopup *gep;

        g_return_if_fail (object != NULL);
        g_return_if_fail (GALEON_IS_EMBED_POPUP (object));

        gep = GALEON_EMBED_POPUP (object);

        g_return_if_fail (gep->priv != NULL);

	if (gep->priv->event)
	{
		g_object_unref (G_OBJECT (gep->priv->event));
	}	
	
	if (gep->priv->helper_list)
	{
		g_object_unref (G_OBJECT (gep->priv->helper_list));
	}
	
	
        G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
setup_image_openwith (GaleonEmbedPopup *p)
{
	GaleonEmbedEvent *info;
	const char *url;
	const GValue *value;
	GaleonEmbedHelperList *hl;
	
	info = galeon_embed_popup_get_event (p);
	value = galeon_embed_event_get_property (info, "image");
	url = g_value_get_string (value);
		
	hl = p->priv->helper_list;
	if (!galeon_embed_helper_list_set_uri (hl, url))
	{
		/* it must be an image, so let's try with png */
		galeon_embed_helper_list_add_mime_type (hl, "image/png");
	}
	galeon_embed_helper_list_add_to_bonoboui_menu (hl, p->priv->ui_component,
						       GALEON_POPUP_IMAGE_OPENWITH_PATH);
}

static void 
setup_element_menu (GaleonEmbedPopup *p)
{
	gboolean is_link, is_image, is_email_link, is_sidebar;
	
	is_image = p->priv->context & EMBED_CONTEXT_IMAGE;
	is_email_link =  p->priv->context & EMBED_CONTEXT_EMAIL_LINK;
	is_link = (p->priv->context & EMBED_CONTEXT_LINK) && !is_email_link;
	is_sidebar = p->priv->context & EMBED_CONTEXT_SIDEBAR;

	gul_bonobo_set_hidden (p->priv->ui_component, 
			       GALEON_POPUP_LINK_ITEMS_PLACEHOLDER, 
			       !is_link);
	gul_bonobo_set_hidden (p->priv->ui_component, 
			       GALEON_POPUP_IMAGE_ITEMS_PLACEHOLDER, 
			       !is_image);
	gul_bonobo_set_hidden (p->priv->ui_component, 
			       GALEON_POPUP_IMAGE_OPENWITH_PATH,
			       !is_image);
	gul_bonobo_set_hidden (p->priv->ui_component, 
			       GALEON_POPUP_EMAIL_LINK_ITEMS_PLACEHOLDER, 
			       !is_email_link);
	gul_bonobo_set_hidden (p->priv->ui_component, 
			       GALEON_POPUP_BETWEEN_ELEMENTS1_PLACEHOLDER, 
			       !is_image || (!is_link && !is_email_link));

	/* Hide items that doesnt make sense to show in the sidebar */
	gul_bonobo_set_hidden (p->priv->ui_component, 
			       GALEON_POPUP_OPEN_IMAGE_PATH, 
			       is_sidebar);
	
	if (is_image)
	{
		GaleonEmbedEvent *info;
		const GValue *value;
		char *open_image_str, *block_host_str,
		     *esc_file = NULL, *file = NULL;
		const char *host = NULL;
		GnomeVFSURI *uri = NULL;
		const char *show_path, *hide_path;
		PermissionActionType perm_action;

		info = galeon_embed_popup_get_event(p);

		value = galeon_embed_event_get_property(info, "image");

		uri = gnome_vfs_uri_new(g_value_get_string(value));

		if (uri)
		{
			host = gnome_vfs_uri_get_host_name(uri);
			file = gnome_vfs_uri_extract_short_name(uri);
		}

		if (!host)
			host = _("This Site");

		if (file)
		{
			esc_file = gul_string_double_underscores (file);
			g_free(file);
		}

		perm_action = galeon_embed_shell_test_permission(embed_shell,
								 g_value_get_string(value),
								 IMAGES_PERMISSION);

		if (perm_action == DENY_ACTION)
		{
			block_host_str = g_strdup_printf(_("Allow Images From %s"),
							 host);
			show_path = GALEON_POPUP_ALLOW_IMAGE_SITE_PATH;
			hide_path = GALEON_POPUP_BLOCK_IMAGE_SITE_PATH;
		}
		else
		{
			block_host_str = g_strdup_printf(_("Block Images From %s"),
							 host);
			hide_path = GALEON_POPUP_ALLOW_IMAGE_SITE_PATH;
			show_path = GALEON_POPUP_BLOCK_IMAGE_SITE_PATH;
		}

		gul_bonobo_set_label(p->priv->ui_component,
				     show_path,
				     block_host_str);
		gul_bonobo_set_hidden(p->priv->ui_component,
				      show_path, FALSE);
		gul_bonobo_set_hidden(p->priv->ui_component,
				      hide_path, TRUE);

		open_image_str = g_strdup_printf(_("Open Image (%s)"),
						 esc_file);

		gul_bonobo_set_label(p->priv->ui_component,
				     GALEON_POPUP_OPEN_IMAGE_PATH,
				     open_image_str);

		if (esc_file)
			g_free(esc_file);

		if (uri)
			gnome_vfs_uri_unref(uri);

		g_free(block_host_str);
		g_free(open_image_str);

		setup_image_openwith(p);
	}
}

static void
setup_doc_openwith (GaleonEmbedPopup *popup)
{
	char *location;
	GaleonEmbedHelperList *hl = popup->priv->helper_list;

	g_return_if_fail (popup->priv->embed != NULL);

	location = galeon_embed_get_location (popup->priv->embed, FALSE, FALSE);

	if (!galeon_embed_helper_list_set_uri (hl, location))
	{
		/* well, it's very probably html */
		galeon_embed_helper_list_add_mime_type (hl, "text/html");
	}
	
	g_free (location);
	
	galeon_embed_helper_list_add_mime_type (hl, "text/plain");
	
	galeon_embed_helper_list_add_to_bonoboui_menu (hl, popup->priv->ui_component,
						       GALEON_POPUP_DOC_OPENWITH_PATH);
}

static void
setup_document_menu (GaleonEmbedPopup *p)
{	
	gboolean is_framed;
	const GValue *value;
	gboolean has_background;
	gboolean is_sidebar = p->priv->context & EMBED_CONTEXT_SIDEBAR;
	
	value = galeon_embed_event_get_property (p->priv->event,
						 "framed_page");
	is_framed = g_value_get_int (value);
	gul_bonobo_set_hidden (BONOBO_UI_COMPONENT(p->priv->ui_component), 
			       GALEON_POPUP_FRAME_ITEMS_PLACEHOLDER, !is_framed);

	has_background = galeon_embed_event_has_property (p->priv->event,
							  "background_image");
	gul_bonobo_set_hidden (BONOBO_UI_COMPONENT(p->priv->ui_component), 
			       GALEON_POPUP_SAVE_BG_PATH, !has_background);

	/* Hide items that doesnt make sense to show in the sidebar */
	gul_bonobo_set_hidden (p->priv->ui_component, 
			       GALEON_POPUP_NAVIGATION_ITEMS_PLACEHOLDER, 
			       is_sidebar);
}

void 
galeon_embed_popup_set_event (GaleonEmbedPopup *p,
			      GaleonEmbedEvent *event)
{
	EmbedEventContext context;

	if (p->priv->event)
	{
		g_object_unref (G_OBJECT (p->priv->event));
	}

	context = galeon_embed_event_get_context (event);
	
	p->priv->context = context;

	p->priv->event = event;
	g_object_ref (G_OBJECT(event));

	if ((p->priv->context & EMBED_CONTEXT_LINK) ||
	    (p->priv->context & EMBED_CONTEXT_EMAIL_LINK) ||
	    (p->priv->context & EMBED_CONTEXT_IMAGE))
	{
		setup_element_menu (p);
		p->priv->popup_type = EMBED_POPUP_ELEMENT;
	}
	else if (p->priv->context & EMBED_CONTEXT_INPUT)
	{
		p->priv->popup_type = EMBED_POPUP_INPUT;
	}
	else if (galeon_embed_event_has_property (event, "selection"))
	{
		/* TODO here we could set up some additional menu items that
		 * use the selected text, like smart bookmarks for example */
		p->priv->popup_type = EMBED_POPUP_SELECTION;
	}
	else
	{
		setup_document_menu (p);
		p->priv->popup_type = EMBED_POPUP_DOCUMENT;
	}
}

void
galeon_embed_popup_set_embed (GaleonEmbedPopup *p, 
			      GaleonEmbed *e)
{
	p->priv->embed = e;

	galeon_embed_helper_list_set_embed (p->priv->helper_list, e);

	if (p->priv->popup_type == EMBED_POPUP_DOCUMENT)
	{
		setup_doc_openwith (p);
	}
}

GaleonEmbed *
galeon_embed_popup_get_embed (GaleonEmbedPopup *p)
{
	return p->priv->embed;
}

void
galeon_embed_popup_show (GaleonEmbedPopup *p,
			 GaleonEmbed *embed)
{
	GaleonEmbedPopupClass *klass = GALEON_EMBED_POPUP_GET_CLASS (p);
	klass->show (p, embed);

}

void 
galeon_embed_popup_connect_verbs (GaleonEmbedPopup *p,
				  BonoboUIComponent *ui_component)
{

	p->priv->ui_component = BONOBO_UI_COMPONENT (ui_component);
	
	bonobo_ui_component_add_verb_list_with_data (BONOBO_UI_COMPONENT(ui_component), 
                                                     embed_popup_verbs, 
                                                     p);
}

GaleonEmbedEvent *
galeon_embed_popup_get_event (GaleonEmbedPopup *p)
{
	g_return_val_if_fail (GALEON_IS_EMBED_POPUP (p), NULL);

	return p->priv->event;
}

static void 
embed_popup_copy_location_cmd (BonoboUIComponent *uic, 
                               GaleonEmbedPopup *popup, 
                               const char* verbname)
{
	char *location;
	location = galeon_embed_get_location (popup->priv->embed, FALSE, FALSE);
	embed_popup_copy_to_clipboard (popup, location);
	g_free (location);
}

static void 
embed_popup_copy_email_cmd (BonoboUIComponent *uic, 
                            GaleonEmbedPopup *popup, 
                            const char* verbname)
{
	GaleonEmbedEvent *info;
	const char *location;
	const GValue *value;
	
	info = galeon_embed_popup_get_event (popup);
	value = galeon_embed_event_get_property (info, "email");
	location = g_value_get_string (value);
	embed_popup_copy_to_clipboard (popup, location);
}

static void 
embed_popup_copy_link_location_cmd (BonoboUIComponent *uic, 
                      	            GaleonEmbedPopup *popup, 
                                    const char* verbname)
{
	GaleonEmbedEvent *info;
	const char *location;
	const GValue *value;
	
	info = galeon_embed_popup_get_event (popup);
	value = galeon_embed_event_get_property (info, "link");
	location = g_value_get_string (value);
	embed_popup_copy_to_clipboard (popup, location);
}

const char *
galeon_embed_popup_get_popup_path (GaleonEmbedPopup *p)
{
	const char *result = NULL;

	switch (p->priv->popup_type)
	{
		case EMBED_POPUP_INPUT:
			result = INPUT_POPUP_PATH;
			break;
		case EMBED_POPUP_ELEMENT:
			result = ELEMENT_POPUP_PATH;
			break;
		case EMBED_POPUP_DOCUMENT:
			result = DOCUMENT_POPUP_PATH;
			break;
		case EMBED_POPUP_SELECTION:
			result = SELECTION_POPUP_PATH;
			break;
	}
	
	return result;
}

/* commands */

static void 
embed_popup_download_link_cmd (BonoboUIComponent *uic, 
                      	       GaleonEmbedPopup *popup, 
                               const char* verbname)
{
	GaleonEmbedEvent *event = galeon_embed_popup_get_event (popup);
	GaleonEmbed *embed = galeon_embed_popup_get_embed (popup);

	galeon_embed_utils_download_event_property (embed, event, FALSE, NULL, "link");
}

static void 
embed_popup_save_image_as_cmd (BonoboUIComponent *uic, 
                      	       GaleonEmbedPopup *popup, 
                               const char* verbname)
{
	GaleonEmbedEvent *event = galeon_embed_popup_get_event (popup);
	GaleonEmbed *embed = galeon_embed_popup_get_embed (popup);
	
	galeon_embed_utils_save_event_property (embed, event, TRUE, FALSE, 
						_("Save Image As..."), "image" );
}

#define CONF_DESKTOP_BG_PICTURE "/desktop/gnome/background/picture_filename"
#define CONF_DESKTOP_BG_TYPE "/desktop/gnome/background/picture_options"

static void
background_download_completed (GaleonEmbedPersist *persist,
			       gpointer data)
{
	const char *bg;
	char *type;
	
	bg = galeon_embed_persist_get_dest (persist);
	eel_gconf_set_string (CONF_DESKTOP_BG_PICTURE, bg);

	type = eel_gconf_get_string (CONF_DESKTOP_BG_TYPE);
	if (type && strcmp (type, "none") == 0)
	{
		eel_gconf_set_string (CONF_DESKTOP_BG_TYPE,
				      "wallpaper");
	}

	g_free (type);
}

static void 
embed_popup_set_image_as_background_cmd (BonoboUIComponent *uic, 
                      	       		 GaleonEmbedPopup *popup, 
                               		 const char* verbname)
{
	GaleonEmbedEvent *info;
	const char *location;
	char *dest, *base;
	const GValue *value;
	GaleonEmbedPersist *persist;
	
	info = galeon_embed_popup_get_event (popup);
	value = galeon_embed_event_get_property (info, "image");
	location = g_value_get_string (value);

	persist = galeon_embed_persist_new (popup->priv->embed);
	
	base = g_path_get_basename (location);
	dest = g_build_filename (g_get_home_dir (), 
				 GALEON_DIR, 
				 base, NULL);
	
	galeon_embed_persist_set_source (persist, location);
	galeon_embed_persist_set_dest (persist, dest);

	g_signal_connect (persist, "completed",
			  G_CALLBACK (background_download_completed),
			  NULL);
	
	galeon_embed_persist_save (persist);
	g_object_unref (persist);

	g_free (dest);
	g_free (base);
}

static void 
embed_popup_copy_image_location_cmd (BonoboUIComponent *uic, 
                      	       	     GaleonEmbedPopup *popup, 
                                     const char* verbname)
{
	GaleonEmbedEvent *info;
	const char *location;
	const GValue *value;
	
	info = galeon_embed_popup_get_event (popup);
	value = galeon_embed_event_get_property (info, "image");
	location = g_value_get_string (value);
	embed_popup_copy_to_clipboard (popup, location);
}

static void 
embed_popup_save_page_as_cmd (BonoboUIComponent *uic, 
                      	      GaleonEmbedPopup *popup, 
                              const char* verbname)
{
	GtkWidget *window;
	GaleonEmbedPersist *persist;

	window = gtk_widget_get_toplevel (GTK_WIDGET (popup->priv->embed));
	
	persist = galeon_embed_persist_new (popup->priv->embed);
	galeon_embed_persist_set_flags (persist, EMBED_PERSIST_MAINDOC
					| EMBED_PERSIST_SAVE_CONTENT
					| EMBED_PERSIST_ASK_DESTINATION
					| EMBED_PERSIST_ADD_TO_RECENT);

	galeon_embed_persist_set_fc_parent (persist, window);
	galeon_embed_persist_set_fc_title (persist, _("Save Page As..."));
	galeon_embed_persist_set_user_time (persist, gtk_get_current_event_time());
	galeon_embed_persist_save (persist);

	g_object_unref (persist);
}

static void 
embed_popup_save_background_as_cmd (BonoboUIComponent *uic, 
                      	       	    GaleonEmbedPopup *popup, 
                                    const char* verbname)
{	
	GaleonEmbedEvent *event = galeon_embed_popup_get_event (popup);
	GaleonEmbed *embed = galeon_embed_popup_get_embed (popup);
	
	galeon_embed_utils_save_event_property (embed, event, TRUE, FALSE,
						_("Save Background As..."),
						"background_image");
}

static void 
embed_popup_open_frame_cmd (BonoboUIComponent *uic, 
                      	    GaleonEmbedPopup *popup, 
                            const char* verbname)
{
	char *location;
	
	location = galeon_embed_get_location (popup->priv->embed, FALSE, FALSE);

	galeon_embed_load_url (popup->priv->embed, location);
}

static void 
embed_popup_open_image_cmd (BonoboUIComponent *uic, 
                      	    GaleonEmbedPopup *popup, 
                            const char* verbname)
{
	GaleonEmbedEvent *info;
	const char *location;
	const GValue *value;
	
	info = galeon_embed_popup_get_event (popup);
	value = galeon_embed_event_get_property (info, "image");
	location = g_value_get_string (value);

	galeon_embed_load_url (popup->priv->embed, location);	
}

static void
embed_popup_copy_to_clipboard (GaleonEmbedPopup *popup, const char *text)
{
	gtk_clipboard_set_text (gtk_clipboard_get (GDK_NONE),
				text, -1);
	gtk_clipboard_set_text (gtk_clipboard_get (GDK_SELECTION_PRIMARY),
				text, -1);
}
