/*
*  
*  $Id: controladorextensiones.cpp 4371 2011-11-04 12:22:54Z tovar $
*  Ginkgo CADx Project
*
*  Copyright 2008-10 MetaEmotion S.L. All rights reserved.
*  http://ginkgo-cadx.com
*
*  This file is licensed under LGPL v3 license.
*  See License.txt for details
*
*
*/
#include <main/entorno.h>
#include <main/controllers/controladorextensiones.h>
#include "controladorlog.h"

#include <string>
#include <map>
#include <cctype>
#include <algorithm>
#include <exception>
#include <wx/dir.h>
#include <wx/file.h>
#include <wx/filename.h>
#include <main/controllers/configurationcontroller.h>
#include <wx/dynlib.h>
#include <algorithm>
#include <set>
#include <prvext/prvext.h>

#if defined(__WXOSX__)
#include <dlfcn.h>
#elif defined(__WXGTK__)
#include <dlfcn.h>
#elif defined(_WINDOWS)
#else
#error Plataforma no soportada
#endif

#define EXT wxT("*.inf")


namespace GADVAPI {
	class PrivateExtensionWrapper : public IPrivateExtensionWrapper {

	public:

		PrivateExtensionWrapper() {
			this->version = 0;
			this->subversion = 0;
			this->release = 0;
			this->build = 0;
			this->core_version = 0;
			this->core_subversion = 0;

			this->privateExtension = NULL;
			
			this->loaded = false;
			this->dll = NULL;
		}

		virtual ~PrivateExtensionWrapper() {
			UnLoad();
		}

		bool operator==(const PrivateExtension* other) const {
			return other != NULL && (*this) == (*other);
		}

		bool operator==(const PrivateExtension& other) const {
			return 
				this->sid == other.GetSID() &&
				this->version == other.GetVersion() &&
				this->subversion == other.GetSubVersion() &&
				this->release == other.GetSubRelease() &&
				this->codename == other.GetCodeName();
		}

		virtual bool Load() {

			if (this->loaded || this->dll != NULL) {
				return this->loaded;
			}

			this->loaded = false;

			// Load library file
			this->dll = new wxDynamicLibrary();
			LOG_DEBUG("Extensions", _Std("Loading") << " " << this->sid << " [" << filePath << "]")
			if(!dll->Load(FROMPATH(filePath)))
			{
				LOG_ERROR("Extensions", _Std("Could not load extension library file") << "[" << filePath << "]");
				this->error = _Std("Could not load library file. Check log");
				//std::cout << "Lib not loaded!" << std::endl;
				delete this->dll;
				this->dll = NULL;
			}
			else {
#if defined(_DEBUG)
				//Create a valid function pointer using the function pointer type in plugin.h
				wxDYNLIB_FUNCTION(CreatePrivateExtension_function,CreatePrivateExtensionDebug,*dll);
				//check if the function is found
				if(pfnCreatePrivateExtensionDebug)
				{
					privateExtension = pfnCreatePrivateExtensionDebug();
#else
				//Create a valid function pointer using the function pointer type in header
				wxDYNLIB_FUNCTION(CreatePrivateExtension_function,CreatePrivateExtension,*dll);
				//check if the function is found
				if(pfnCreatePrivateExtension)
				{
					privateExtension = pfnCreatePrivateExtension();
#endif

					// We calls
					if (*this == privateExtension) {
						this->loaded = true;
					}
					else {
						if (this->privateExtension != NULL) {
							delete this->privateExtension;
							this->privateExtension = NULL;
						}
						if (this->dll != NULL) {
							delete this->dll;
							this->dll = NULL;
						}
						this->error = _Std("Extension descriptor does not match");
					}
				}
				else {
					this->error = _Std("Could not find entry point");
					delete this->dll;
					this->dll = NULL;
				}
			}

			return this->loaded;
		}

		virtual bool UnLoad() {
			
			if (this->loaded) {
				this->loaded = false;
				if (this->privateExtension != NULL) {
					delete this->privateExtension;
				}
				if (this->dll != NULL) {
					delete this->dll;
					this->dll = NULL;
				}
			}
			return !this->loaded;
		}

		virtual bool IsLoaded() const {
			return this->loaded;
		}

		void setSid(const std::string& sidStr) {
			this->sid = sidStr;
		}

		const std::string& getSid() const {
			return this->sid;
		}

		void setFilePath(const std::string& path) {
			this->filePath = path;
		}

		virtual int GetVersion() const {
			return this->version;
		}

		void setVersion(const std::string& versionStr) {
			this->version = parseInt(versionStr);
		}

		virtual int GetSubVersion() const {
			return this->subversion;
		}

		void setSubVersion(const std::string& subversionStr) {
			this->subversion = parseInt(subversionStr);
		}

		virtual int GetRelease() const {
			return this->release;
		}

		void setRelease(const std::string& releaseStr) {
			this->release = parseInt(releaseStr);
		}

		virtual int GetBuild() const {
			return this->build;
		}

		void setBuild(const std::string& buildStr) {
			this->build = parseInt(buildStr);
		}

		virtual const std::string& GetCodeName() const {
			return this->codename;
		}

		void setCodeName(const std::string& codeNameStr) {
			this->codename = codeNameStr;
		}

		void setCoreVersion(const std::string& coreVersionStr) {
			this->core_version = parseInt(coreVersionStr);
		}

		void setCoreSubVersion(const std::string& coreSubVersionStr) {
			this->core_subversion = parseInt(coreSubVersionStr);
		}

		bool isCompatibleWithCore(int coreVersion, int coreSubVersion) {
			if (this->core_version == coreVersion && this->core_subversion == coreSubVersion) {
				return true;
			} else {
				this->error = _Std("Extension is incompatible with current core. Please, check for upgrades.");
				return false;
			}
		}

		virtual const std::string& GetProvider() const {
			return this->provider;
		}

		void setProvider(const std::string& providerStr) {
			this->provider = providerStr;
		}

		virtual const std::string& GetDescription() const {
			return this->description;
		}

		void setDescription(const std::string& descriptionStr) {
			this->description = descriptionStr;
		}

		virtual const std::string& GetUpdateURL() const {
			return this->updateurl;
		}

		void setUpdateURL(const std::string& updateURLStr) {
			this->updateurl = updateURLStr;
		}

		virtual const std::string& GetError() const {
			return this->error;
		}

		virtual PrivateExtension* GetExtension() {
			return this->privateExtension;
		}

		bool good() {
			return this->sid.size() > 0 && this->filePath.size() > 0 && this->version != 0 && this->core_version != 0;
		}

		int parseInt(const std::string& str) {
			int val = -1;
			std::istringstream is(str);
			is >> val;
			return val;
		}

	private:
		bool loaded;

		std::string sid;

		std::string filePath;

		int version;
		int subversion;
		int release;
		int build;

		int core_version;
		int core_subversion;

		std::string codename;

		std::string provider;
		std::string description;

		std::string updateurl;
		std::string error;
		PrivateExtension* privateExtension;
		wxDynamicLibrary* dll;
	};
}



GNC::GCS::IExtensionException::IExtensionException(const std::string& msg, const std::string& component, bool fatal) throw() : GNC::GCS::IException(msg, component, fatal) {}
GNC::GCS::IExtensionException::~IExtensionException() throw() {}


GNC::GCS::IControladorExtensiones::IControladorExtensiones() {}
GNC::GCS::IControladorExtensiones::~IControladorExtensiones() {}


GNC::ControladorExtensiones* GNC::ControladorExtensiones::m_pInstance = 0;

GNC::ControladorExtensiones::ControladorExtensiones()
{
	m_Manager.Scan();
	for (GADVAPI::PrivateExtensionManager::iterator it= m_Manager.begin(); it != m_Manager.end(); ++it)
	{
		GADVAPI::IPrivateExtensionWrapper* iew = (*it).second;
		if (iew->IsLoaded()) {
			RegistrarModulo(iew->GetExtension()->InitializeLibrary(GNC::Entorno::Instance()));
		}
	}
}

GNC::ControladorExtensiones::~ControladorExtensiones()
{
	DesRegistrarModulos();
	m_Manager.UnLoadAll();
}

GNC::ControladorExtensiones* GNC::ControladorExtensiones::Instance()
{
	if (m_pInstance == NULL) {
		m_pInstance = new GNC::ControladorExtensiones();
	}
	return m_pInstance;
}

void GNC::ControladorExtensiones::FreeInstance()
{
	if (m_pInstance != NULL) {
		delete m_pInstance;
	}
	m_pInstance = 0;
}

void GNC::ControladorExtensiones::RegistrarModulo(GNC::GCS::IControladorModulo* pCtrlModulo)
{

	if (pCtrlModulo == NULL) {
		return;
	}

	pCtrlModulo->RegistrarConfiguracion();

	const std::string uid = pCtrlModulo->GetUID();
	m_Modulos[uid] = pCtrlModulo;
	NotificarRegistro(pCtrlModulo);
}

void GNC::ControladorExtensiones::DesRegistrarModulo(GNC::GCS::IControladorModulo* pCtrlModulo)
{
	if (pCtrlModulo == NULL) {
		return;
	}

	NotificarDesRegistro(pCtrlModulo);

	m_Modulos.erase(pCtrlModulo->GetUID());
	delete pCtrlModulo;
}

void GNC::ControladorExtensiones::DesRegistrarModulos()
{
	GNC::ControladorExtensiones::IteradorListaModulos it;

	for (it = m_Modulos.begin(); it != m_Modulos.end(); it++) {
		GNC::GCS::IControladorModulo* item = (*it).second;
		NotificarDesRegistro(item);
		delete item;
	}
	m_Modulos.clear();
}

GNC::GCS::IControladorModulo* GNC::ControladorExtensiones::ObtenerModulo(const std::string& idModulo)
{
	GNC::GCS::IControladorModulo* cm = NULL;
	IteradorListaModulos it = m_Modulos.find(idModulo);
	if (it != m_Modulos.end()) {
		cm = (*it).second;
	}
	return cm;
}



bool compareListaModulos(GNC::GCS::IControladorModulo* s1, GNC::GCS::IControladorModulo* s2)
{
	return s1->GetPriority()< s2->GetPriority();
}

std::list<GNC::GCS::IControladorModulo*> GNC::ControladorExtensiones::ModulosOrdenados()
{
	std::list<GNC::GCS::IControladorModulo*> listaModulosPriorizados;
	for(ListaModulos::iterator it = m_Modulos.begin(); it!= m_Modulos.end(); ++it)
	{
		listaModulosPriorizados.push_back((*it).second);
	}

	listaModulosPriorizados.sort(compareListaModulos);	

	return listaModulosPriorizados;
}

const GNC::ControladorExtensiones::ListaModulos& GNC::ControladorExtensiones::Modulos() const
{
	return m_Modulos;
}

// Realización de la interfaz IControladorExtensiones
GADVAPI::PrivateExtensionManager& GNC::ControladorExtensiones::GetPrivateExtensionsManager()
{
	return m_Manager;
}

void GNC::ControladorExtensiones::NotificarRegistro(GNC::GCS::IControladorModulo* pModulo)
{
	GNC::Entorno* pEntorno = GNC::Entorno::Instance();
	Entorno::ListaObservadoresExtensiones::const_iterator it;
	for (it = pEntorno->ObservadoresExtensiones.begin(); it != pEntorno->ObservadoresExtensiones.end(); it++) {
		GNC::GCS::IObservadorExtensiones* ne = *it;
		ne->OnModuloCargado(pModulo);
	}
}

void GNC::ControladorExtensiones::NotificarDesRegistro(GNC::GCS::IControladorModulo* pModulo)
{
	GNC::Entorno* pEntorno = GNC::Entorno::Instance();
	Entorno::ListaObservadoresExtensiones::const_iterator it;
	for (it = pEntorno->ObservadoresExtensiones.begin(); it != pEntorno->ObservadoresExtensiones.end(); it++) {
		GNC::GCS::IObservadorExtensiones* ne = *it;
		ne->OnModuloDescargado(pModulo);
	}
}

//-------------------------------------------------------------------------------------------
// Helpers

inline std::wstring StringToWString(const std::string& str) {
	std::wstring temp(str.length(),L' ');
	std::copy(str.begin(), str.end(), temp.begin());
	return temp;
}


//-------------------------------------------------------------------------------------------
// PrivateExtensionManager

GADVAPI::PrivateExtensionManager::PrivateExtensionManager() : GADVAPI::PrivateExtensionManager::Base()
{

}

GADVAPI::PrivateExtensionManager::~PrivateExtensionManager()
{
	UnLoadAll();
}

GADVAPI::IPrivateExtensionWrapper* GADVAPI::PrivateExtensionManager::GetExtension(const std::string& nombre)
{
	Base& base = *this;
	if (base.find(nombre) != base.end()) {
		return base[nombre];
	} else {
		return NULL;
	}
}

void GADVAPI::PrivateExtensionManager::Scan()
{

	UnLoadAll();

	int core_version = -1;
	int core_subversion = -1;
	GNC::Entorno::Instance()->GetGinkgoVersionAsTuple(&core_version, &core_subversion, NULL, NULL, NULL);

	wxDir dir;
	wxString rutas[1] = {
		FROMPATH(GNC::Entorno::Instance()->GetPluginsPath())
	};

	wxString descFileName;
	wxString descFilePath;
	std::string stdExtensionDir;

	for (int i = 0; i < 1; i++) {
		if (dir.Exists(rutas[i]) && dir.Open(rutas[i])) {
			bool cont = dir.GetFirst(&descFileName, EXT, wxDIR_FILES);
			while (cont) {
				stdExtensionDir = std::string( (rutas[i] + wxFileName::GetPathSeparator()).ToUTF8() );
				descFilePath = rutas[i] + wxFileName::GetPathSeparator() + descFileName;

				std::string descStdFilePath(descFilePath.ToUTF8());

				ifstream file (descStdFilePath.c_str());
				std::string line;
				std::vector<std::string> tokens(2);

				/*
				DLL=file.dll
				SID=TheSid
				PROVIDER=TheProvider
				DESCRIPTION=TheDescription
				COMPILATION=TheBuildDate
				VERSION=TheVersion
				SUBVERSION=TheSubVersion
				RELEASE=TheRelease
				BUILD=TheBuild
				CODENAME=TheCodeNameString
				UPDATEURL=TheUpdateURL
				CORE_VERSION=TheCoreExactCompatibleVersion
				CORE_SUBVERSION=TheCoreExactCompatibleSubVersion
				*/

				if (file.is_open()) {

					PrivateExtensionWrapper* pExt = new PrivateExtensionWrapper();

					while(std::getline(file, line)) {

						std::string::size_type sepPos = line.find("=");
						if (sepPos != std::string::npos) {
							std::string key = line.substr(0, sepPos);
							std::string value = line.substr(sepPos+1);

							if (key == "DLL") {
								pExt->setFilePath(stdExtensionDir + value);
							}
							else if (key == "SID") {
								pExt->setSid(value);
							}
							else if (key == "PROVIDER") {
								pExt->setProvider(value);
							}
							else if (key == "DESCRIPTION") {
								pExt->setDescription(value);
							}
							else if (key == "COMPILATION") {

							}
							else if (key == "VERSION") {
								pExt->setVersion(value);
							}
							else if (key == "SUBVERSION") {
								pExt->setSubVersion(value);
							}
							else if (key == "RELEASE") {
								pExt->setRelease(value);
							}
							else if (key == "BUILD") {
								pExt->setBuild(value);
							}
							else if (key == "CODENAME") {
								pExt->setCodeName(value);
							}
							else if (key == "UPDATEURL") {
								pExt->setUpdateURL(value);
							}
							else if (key == "CORE_VERSION") {
								pExt->setCoreVersion(value);
							}
							else if (key == "CORE_SUBVERSION") {
								pExt->setCoreSubVersion(value);
							}
						}
					}

					if (pExt->good()) {

						if (this->find(pExt->getSid()) != end()) {
							LOG_ERROR("Extensions", _Std("Skipping extension with duplicated SID:") << " " << pExt->getSid());
							delete pExt;
						}
						else {
							Base& base = *this;							
							base[pExt->getSid()] = pExt;

							if (pExt->isCompatibleWithCore(core_version, core_subversion)) {
								pExt->Load();
							} 
						}
					}
					else {
						LOG_ERROR("Extensions", _Std("Skipping invalid extension without required keys:") << " " << pExt->getSid());
						delete pExt;
					}

				}
				cont = dir.GetNext(&descFileName);
			}
		}
	}
}

void GADVAPI::PrivateExtensionManager::UnLoadAll() {
	for(iterator it = begin(); it != end(); it++)
	{
		(*it).second->UnLoad();
		delete (*it).second;
	}
	clear();
}

GADVAPI::PrivateExtensionManager::iterator GADVAPI::PrivateExtensionManager::begin() {
	return Base::begin();
}

GADVAPI::PrivateExtensionManager::iterator GADVAPI::PrivateExtensionManager::end() {
	return Base::end();
}
