/*
*  
*  $Id$
*  Ginkgo CADx Project
*
*  Copyright 2008-10 MetaEmotion S.L. All rights reserved.
*  http://ginkgo-cadx.com
*
*  This file is licensed under LGPL v3 license.
*  See License.txt for details
*
*/
#pragma once
#include <string>
#include <list>
#include <sstream>
#include <api/pathwildcards.h>

namespace GIL {
	class IModeloIdentificador
	{
	public:
		std::string tipoId;
		std::string valor;

		IModeloIdentificador()
		{
		}

		IModeloIdentificador(const std::string& tipoId, const std::string& valor)
		{
			this->tipoId=tipoId;
			this->valor = valor;
		}

		IModeloIdentificador& operator=(const IModeloIdentificador& otro)
		{
			this->tipoId= otro.tipoId;
			this->valor = otro.valor;
			return *this;
		}
	};

	typedef std::list<IModeloIdentificador> TListaIdentificadores;

	class IModeloModificador
	{
	public:
		std::string Etiqueta;

		IModeloModificador()
		{
		}

		IModeloModificador(const std::string& etiqueta)
		{
			this->Etiqueta = etiqueta;
		}

		IModeloModificador(const IModeloModificador& otro)
		{
			*this = otro;
		}

		IModeloModificador& operator=(const IModeloModificador& otro)
		{
			this->Etiqueta = otro.Etiqueta;

			return *this;
		}
	};

	typedef std::list<IModeloModificador> TListaModificadores;

	class IModeloMedico {
	public:
		TListaIdentificadores listaIdentificadores;
		std::string nombre;
		std::string apellido1;
		std::string apellido2;
		//	std::string numColegiado;
		//	std::string idServicio;
		//	std::string cargo;
		std::string idCentro;
		std::string nombreCentro;

		IModeloMedico()
		{
		}

		IModeloMedico(const std::string& nombre, const std::string& apellido1, const std::string& apellido2, const std::string& idCentro, const std::string& nombreCentro )
		{
			this->nombre = nombre;
			this->apellido1 = apellido1;
			this->apellido2 = apellido2;
			this->idCentro = idCentro;
			this->nombreCentro = nombreCentro;
		}

		IModeloIdentificador GetIdentificadorPreferido() {
			for(TListaIdentificadores::iterator it = listaIdentificadores.begin(); it!=listaIdentificadores.end(); it++) {
				if((*it).tipoId == "NNESP") return (*it);
			}
			for(TListaIdentificadores::iterator it = listaIdentificadores.begin(); it!=listaIdentificadores.end(); it++) {
				if((*it).tipoId == "MD") return (*it);
			}
			for(TListaIdentificadores::iterator it = listaIdentificadores.begin(); it!=listaIdentificadores.end(); it++) {
				if((*it).tipoId == "PPN") return (*it);
			}
			for(TListaIdentificadores::iterator it = listaIdentificadores.begin(); it!=listaIdentificadores.end(); it++) {
				if((*it).tipoId == "SS") return (*it);
			}
			for(TListaIdentificadores::iterator it = listaIdentificadores.begin(); it!=listaIdentificadores.end(); it++) {
				if((*it).tipoId == "RI") return (*it);
			}
			for(TListaIdentificadores::iterator it = listaIdentificadores.begin(); it!=listaIdentificadores.end(); it++) {
				if((*it).tipoId == "MCF") return (*it);
			}
			if(listaIdentificadores.size()>0) {
				return listaIdentificadores.front();
			}
			return IModeloIdentificador("Desconocido","Desconocido");
		}

		const std::string GetNIF() const
		{
			std::string nif = "";
			for(TListaIdentificadores::const_iterator it = listaIdentificadores.begin(); it!=listaIdentificadores.end(); it++) {
				if((*it).tipoId == "NNESP") return (*it).valor;
			}
			return nif;
		}

		IModeloMedico& operator=(const IModeloMedico& otro)
		{
			this->nombre = otro.nombre;
			this->apellido1 = otro.apellido1;
			this->apellido2 = otro.apellido2;
			this->idCentro = otro.idCentro;
			this->nombreCentro = otro.nombreCentro;
			this->listaIdentificadores = otro.listaIdentificadores;
			return *this;
		}
	};


	class IModeloPaciente
	{
	public:
		TListaIdentificadores listaIdentificadores;
		std::string nombre;
		std::string apellido1;
		std::string apellido2;
		std::string numEpisodio;

		IModeloPaciente()
		{
		}

		IModeloPaciente(const std::string& nombre, const std::string& apellido1, const std::string& apellido2)
		{
			this->nombre = nombre;
			this->apellido1 = apellido1;
			this->apellido2 = apellido2;
		}

		IModeloIdentificador GetIdentificadorPreferido() {
			for(TListaIdentificadores::iterator it = listaIdentificadores.begin(); it!=listaIdentificadores.end(); it++) {
				if((*it).tipoId == "PI") return (*it);
			}
			for(TListaIdentificadores::iterator it = listaIdentificadores.begin(); it!=listaIdentificadores.end(); it++) {
				if((*it).tipoId == "HC") return (*it);
			}
			for(TListaIdentificadores::iterator it = listaIdentificadores.begin(); it!=listaIdentificadores.end(); it++) {
				if((*it).tipoId == "SS") return (*it);
			}
			for(TListaIdentificadores::iterator it = listaIdentificadores.begin(); it!=listaIdentificadores.end(); it++) {
				if((*it).tipoId == "NNESP") return (*it);
			}
			for(TListaIdentificadores::iterator it = listaIdentificadores.begin(); it!=listaIdentificadores.end(); it++) {
				if((*it).tipoId == "JHN") return (*it);
			}
			for(TListaIdentificadores::iterator it = listaIdentificadores.begin(); it!=listaIdentificadores.end(); it++) {
				if((*it).tipoId == "PPN") return (*it);
			}
			if(listaIdentificadores.size()>0) {
				return listaIdentificadores.front();
			}
			return IModeloIdentificador("Desconocido","Desconocido");
		}

		IModeloPaciente& operator=(const IModeloPaciente& otro)
		{
			this->nombre = otro.nombre;
			this->apellido1 = otro.apellido1;
			this->apellido2 = otro.apellido2;
			this->listaIdentificadores = otro.listaIdentificadores;
			this->numEpisodio = otro.numEpisodio;
			return *this;
		}
	};

	class IModeloHCE
	{
	public:
		std::string Sid;
		std::string Hostname;
		std::string Puerto;

		std::string IdOrganizacionReceptora;
		std::string IdSistemaPeticionario;
		std::string IdAplicacionRellenadora;
		std::string IdAplicacionPACS;

		IModeloHCE()
		{
		}

		IModeloHCE(const std::string& sid, const std::string& hostname, const std::string& puerto, const std::string& idOrganizacionReceptora, const std::string& idSistemaPeticionario, const std::string& idAplicacionRellenadora, const std::string& idAplicacionPACS)
		{
			this->Sid                     = sid;
			this->Hostname                = hostname;
			this->Puerto                  = puerto;
			this->IdOrganizacionReceptora = idOrganizacionReceptora;
			this->IdSistemaPeticionario   = idSistemaPeticionario;
			this->IdAplicacionRellenadora = idAplicacionRellenadora;
			this->IdAplicacionPACS        = idAplicacionPACS;
		}

		IModeloHCE(const IModeloHCE& o)
		{
			*this = o;
		}

		IModeloHCE& operator=(const IModeloHCE& otro)
		{
			this->Sid                     = otro.Sid;
			this->Hostname                = otro.Hostname;
			this->Puerto                  = otro.Puerto;
			this->IdOrganizacionReceptora = otro.IdOrganizacionReceptora;
			this->IdSistemaPeticionario   = otro.IdSistemaPeticionario;
			this->IdAplicacionRellenadora = otro.IdAplicacionRellenadora;
			this->IdAplicacionPACS        = otro.IdAplicacionPACS;
			return *this;
		}
	};

	class IModeloTupla
	{
	public:
		std::string codigo;         // CODIGO puede ser DICOM o LOINC
		std::string clave;          // clave
		std::string valor;          // valor
		std::string descripcion;    // descripcion

		IModeloTupla()
		{
		}

		IModeloTupla(const std::string& codigo, const std::string& clave, const std::string& valor, const std::string& descripcion)
		{
			this->codigo      = codigo;
			this->clave       = clave;
			this->valor       = valor;
			this->descripcion = descripcion;
		}

		IModeloTupla& operator=(const IModeloTupla& otro)
		{
			this->codigo      = otro.codigo;
			this->clave       = otro.clave;
			this->valor       = otro.valor;
			this->descripcion = otro.descripcion;
			return *this;
		}
	};

	class IModeloPACSServer
	{
	public:

		typedef enum IMPS_Method {
			IMPS_GET,
			IMPS_MOVE
		} IMPS_Method;

		std::string sid;
		std::string AET;
		std::string hostname;
		std::string puerto;
		std::string pdu;
		std::string retrieveLevel;
		bool        tls;
		bool        verify;
		IMPS_Method metodo;
		std::string cert;
		std::string key;
		std::string user;
		std::string password;

		IModeloPACSServer()
		{
			this->tls = false;
			this->verify = false;
			this->metodo = IMPS_GET;
			user=password = "";
		}

		IModeloPACSServer(const IModeloPACSServer& otro)
		{
			*this = otro;
		}

		IModeloPACSServer(const std::string& sid, const std::string& AET, const std::string& hostname, const std::string& puerto, const std::string& pdu, IMPS_Method metodo, const std::string& retrieveLevel = "SERIES", bool tls = false, bool verify = false, const std::string& cert = "", const std::string& key = "")
		{
			this->sid      = sid;
			this->AET      = AET;
			this->hostname = hostname;
			this->puerto   = puerto;
			this->pdu      = pdu;
			this->tls      = tls;
			this->verify   = verify;
			this->metodo   = metodo;
			this->retrieveLevel = retrieveLevel;
			this->cert     = cert;
			this->key      = key;

		}

		IModeloPACSServer& operator=(const IModeloPACSServer& otro)
		{
			this->sid      = otro.sid;
			this->AET      = otro.AET;
			this->hostname = otro.hostname;
			this->puerto   = otro.puerto;
			this->pdu      = otro.pdu;
			this->tls      = otro.tls;
			this->verify   = verify;
			this->metodo   = otro.metodo;	
			this->retrieveLevel = otro.retrieveLevel;
			this->cert     = otro.cert;
			this->key      = otro.key;
			return *this;
		}
	};

	class IModeloEvidenciaRegistro {
	public:

		std::string        EtiquetaRegistro;
		std::string        Valor;

		TListaModificadores Modificadores; // Lo que seran OBX.17

		IModeloEvidenciaRegistro()
		{
		}

		IModeloEvidenciaRegistro(const std::string& etiqueta, const std::string& valor)
		{
			this->EtiquetaRegistro = etiqueta;
			this->Valor            = valor;
		}

		IModeloEvidenciaRegistro(const IModeloEvidenciaRegistro& o)
		{
			*this = o;
		}

		IModeloEvidenciaRegistro& operator=(const IModeloEvidenciaRegistro& otro)
		{
			this->EtiquetaRegistro   = otro.EtiquetaRegistro;
			this->Valor              = otro.Valor;
			this->Modificadores      = otro.Modificadores;
			return *this;
		}

		/** Anyade un elemento a la lista de modificadores (Lo que sera un campo OBX.17 en HL7) **/
		void AddModificador(const IModeloModificador& otro)
		{
			this->Modificadores.push_back(otro);
		}
	};

	class IModeloEvidenciaImagen {
	public:

		std::string          EtiquetaRegistro;
		std::string          SOPInstanceUID;
		std::string          SOPClassUID;

		TListaModificadores  Modificadores;

		IModeloEvidenciaImagen()
		{

		}

		IModeloEvidenciaImagen(const std::string& etiqueta, const std::string& sopInstanceUID, const std::string& sopClassUID)
		{
			this->EtiquetaRegistro        = etiqueta;			
			this->SOPInstanceUID          = sopInstanceUID;
			this->SOPClassUID					= sopClassUID;
		}

		IModeloEvidenciaImagen(const IModeloEvidenciaImagen& otro)
		{
			*this = otro;
		}

		IModeloEvidenciaImagen& operator=(const IModeloEvidenciaImagen& otro)
		{
			EtiquetaRegistro        = otro.EtiquetaRegistro;
			SOPInstanceUID				= otro.SOPInstanceUID;
			SOPClassUID					= otro.SOPClassUID;
			Modificadores           = otro.Modificadores;
			return *this;
		}

		/** Anyade un elemento a la lista de modificadores (Lo que sera un campo OBX.17 en HL7) **/
		void AddModificador(const IModeloModificador& otro)
		{
			this->Modificadores.push_back(otro);
		}
	};

	class IModeloEvidenciaSerie {
	public:
		typedef std::list<IModeloEvidenciaImagen>    ListaEvidenciasImagenes;


		ListaEvidenciasImagenes   ListaImagenes;

		std::string             SeriesInstanceUID;
		std::string             EtiquetaSerie;

		IModeloEvidenciaSerie()
		{

		}

		IModeloEvidenciaSerie(const std::string etiqueta, const std::string& siuid)
		{				
			this->EtiquetaSerie         = etiqueta;
			this->SeriesInstanceUID        = siuid;
		}

		IModeloEvidenciaSerie(const IModeloEvidenciaSerie& otro)
		{
			*this = otro;
		}

		IModeloEvidenciaSerie& operator=(const IModeloEvidenciaSerie& otro)
		{
			ListaImagenes				= otro.ListaImagenes;
			EtiquetaSerie         = otro.EtiquetaSerie;
			SeriesInstanceUID        = otro.SeriesInstanceUID;
			return *this;
		}

	};

	class IModeloEvidenciaEstudio {
	public:
		typedef std::list<IModeloEvidenciaSerie>    ListaEvidenciasSeries;


		ListaEvidenciasSeries   ListaSeries;

		std::string             StudyInstanceUID;
		std::string             AccessionNumber;
		std::string             EtiquetaEstudio;

		IModeloEvidenciaEstudio()
		{

		}

		IModeloEvidenciaEstudio(const std::string etiqueta, const std::string& siuid, const std::string& accnum)
		{				
			this->EtiquetaEstudio         = etiqueta;
			this->StudyInstanceUID        = siuid;
			this->AccessionNumber         = accnum;
		}

		IModeloEvidenciaEstudio(const IModeloEvidenciaEstudio& otro)
		{
			*this = otro;
		}

		IModeloEvidenciaEstudio& operator=(const IModeloEvidenciaEstudio& otro)
		{
			ListaSeries					= otro.ListaSeries;
			EtiquetaEstudio         = otro.EtiquetaEstudio;
			StudyInstanceUID        = otro.StudyInstanceUID;
			AccessionNumber         = otro.AccessionNumber;
			return *this;
		}

	};

	class IModeloIntegracion
	{
	public:
		typedef std::list<IModeloTupla> ListaTuplas;
		typedef std::list<IModeloEvidenciaEstudio> ListaEvidencias;
		typedef std::list<std::string> ListOfPaths;

		typedef enum TipoAccion {
			TA_Ninguna,
			TA_Dicomizar,
			TA_Obtener,
			TA_Imprimir,
			TA_Open
		} TipoAccion;

		typedef enum TipoLateralidad {
			TL_Derecha,
			TL_Izquierda,
			TL_NoEspecificada
		} TipoLateralidad;

		IModeloMedico     Medico;
		IModeloPaciente   Paciente;
		IModeloHCE        HCE;
		ListaTuplas       Metadatos;
		ListaEvidencias   Evidencias;
		ListOfPaths			Files;

		std::string       PACSAlmacenamiento;
		std::string       CodigoAplicacion;
		std::string       PACSObtencion;

		std::string       idPlantilla;

		TipoAccion        accion;
		std::string       idPeticion;
		std::string       idAmbitoPeticion;
		std::string       idEpisodio;
		std::string       accessionNumber;
		std::string       rawXmlData;
		TipoLateralidad   lateralidad;

		std::string       UIDModulo;

		std::string		   ambitoDIMSE;

		std::string       observaciones;

		bool              atendido;

		IModeloIntegracion()
		{
			accion      = TA_Ninguna;
			ambitoDIMSE = "";
			lateralidad = TL_NoEspecificada;
			atendido = false;
		}

		IModeloIntegracion(const IModeloIntegracion& otro)
		{
			*this = otro;
		}

		~IModeloIntegracion()
		{

		}

		IModeloIntegracion& operator=(const IModeloIntegracion& otro)
		{
			this->Medico             = otro.Medico;
			this->Paciente           = otro.Paciente;
			this->HCE                = otro.HCE;
			this->Metadatos          = otro.Metadatos;
			this->Evidencias         = otro.Evidencias;
			this->PACSAlmacenamiento = otro.PACSAlmacenamiento;
			this->CodigoAplicacion   = otro.CodigoAplicacion;
			this->PACSObtencion      = otro.PACSObtencion;
			this->idPlantilla        = otro.idPlantilla;
			this->accion             = otro.accion;
			this->idPeticion         = otro.idPeticion;
			this->idAmbitoPeticion   = otro.idAmbitoPeticion;
			this->idEpisodio         = otro.idEpisodio;
			this->ambitoDIMSE        = otro.ambitoDIMSE;
			this->accessionNumber    = otro.accessionNumber;
			this->UIDModulo          = otro.UIDModulo;
			this->observaciones      = otro.observaciones;
			this->lateralidad        = otro.lateralidad;

			return *this;
		}
	};

	class Ubicacion
	{
	public:

		std::string       Titulo;						
		std::string       Ruta;
		std::string       Descripcion;
		bool					Monitorize;
		bool					CleanBefore;
		bool					CleanAfter;

		Ubicacion()
		{

		}

		Ubicacion(const std::string& titulo, const std::string& ruta, const std::string& descripcion, bool monitorize, bool cleanbefore, bool cleanafter)
		{
			this->Titulo = titulo;
			Update(ruta, descripcion, monitorize, cleanbefore, cleanafter);

		}

		Ubicacion(const Ubicacion& otro)
		{
			*this = otro;
		}

		~Ubicacion()
		{

		}

		void Update(const std::string& ruta, const std::string& desc, bool monitorize, bool cleanbefore, bool cleanafter)
		{
			this->Ruta        = ruta;
			this->Descripcion = desc;
			this->Monitorize = monitorize;
			this->CleanBefore = cleanbefore;
			this->CleanAfter = cleanafter;
		}

		Ubicacion& operator=(const Ubicacion& otro)
		{
			this->Titulo      = otro.Titulo;
			this->Ruta        = otro.Ruta;
			this->Descripcion = otro.Descripcion;
			this->Monitorize	= otro.Monitorize;
			this->CleanBefore	= otro.CleanBefore;
			this->CleanAfter	= otro.CleanAfter;

			return *this;
		}

		std::string GetRutaFormateada(GnkPtr<IModeloIntegracion>& modeloIntegracion)
		{
			return GNC::GCS::PathWildcards::Parse(Ruta, modeloIntegracion);
		}			
	};
}
