/* $Id: freebsd.cc,v 1.6 2000/09/30 15:24:52 bergo Exp $ */

/*

    GPS - Graphical Process Statistics
    Copyright (C) 1999-2000 Felipe Paulo Guazzi Bergo
    bergo@seul.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "transient.h"

#ifdef HAVEFREEBSD

#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <pwd.h>
#include <errno.h>
#include <dirent.h>
#include <signal.h>
#include <fcntl.h>

#include <kvm.h>
#include <limits.h>

#include <osreldate.h>
#include <sys/conf.h>

#if __FreeBSD_version < 400000
#include <sys/rlist.h>
#endif

#include <sys/dkstat.h>
#include <sys/sysctl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/lock.h>
#include <sys/proc.h>

#if __FreeBSD_version > 399999
#include <vm/vm.h>
#include <vm/pmap.h>
#include "freebsd4_vmmap.h"
#endif

#include <sys/user.h>
#include <sys/vmmeter.h>
#include <nlist.h>
#include <vm/vm.h>
#include <vm/vm_map.h>
#include <vm/vm_param.h>

#include <gtk/gtk.h>

#include "gps.h"
#include "polling.h"
#include "freebsd.h"
#include "tstring.h"
#include "importglobals.h"


#define FREEBSD_FLAG_COUNT	22
static char *flag_text[]={"ADVLOCK" ,"CONTROLT","INMEM"   ,"NOCLDSTOP",
                          "PPWAIT"  ,"PROFIL"  ,"SELECT"  ,"SINTR",
                          "SUGID"   ,"SYSTEM"  ,"TIMEOUT" ,"TRACED",
                          "WAITED"  ,"WEXIT"   ,"EXEC"    ,"NOSWAP",
                          "PHYSIO"  ,"OWEUPC"  ,"SWAPPING","SWAPINREQ",
                          "KTHREADP","NOCLDWAIT"};

#if  __FreeBSD_version >= 400000
 #define P_NOCLDSTOP 0
 #define P_NOSWAP    0
 #define P_PHYSIO    0
 #define P_NOCLDWAIT 0
#endif

static long flag_value[]={P_ADVLOCK  ,P_CONTROLT,P_INMEM   ,P_NOCLDSTOP,
                          P_PPWAIT   ,P_PROFIL  ,P_SELECT  ,P_SINTR,
                          P_SUGID    ,P_SYSTEM  ,P_TIMEOUT ,P_TRACED,
                          P_WAITED   ,P_WEXIT   ,P_EXEC    ,P_NOSWAP,
                          P_PHYSIO   ,P_OWEUPC  ,P_SWAPPING,P_SWAPINREQ,
                          P_KTHREADP ,P_NOCLDWAIT};

Kvm GlobalKvm;

/* KVM */
Kvm::Kvm() {
  refcount=0;
}

void * Kvm::getp() {
  return(kvmp);
}

void * Kvm::reference() {
  refcount++;
  if (refcount==1)
	Open();
  return(kvmp);
}

void Kvm::dereference() {
  refcount--;
  if (refcount==0)
	Close();
}

void Kvm::Open() {
	kvm_t *p;

	p=NULL;
	while(p==NULL) {
		p=kvm_open(NULL,NULL,NULL,O_RDONLY,NULL);
		if (p==NULL)
			usleep(300000);
	}
	kvmp=(void *)p;
}

void Kvm::Close() {
	kvm_close( (kvm_t *)kvmp );
}

/* ********************************************************
 * 
 *
 *                 FreeBSDProcessListPoller
 *	      
 *
 * ********************************************************* */

FreeBSDProcessListPoller::FreeBSDProcessListPoller() : ProcessListPoller() {
	GlobalKvm.reference();
}

void FreeBSDProcessListPoller::terminate() {
	GlobalKvm.dereference();
}

void FreeBSDProcessListPoller::poll() {
	ProcessListItem *pli;
	kvm_t *kt;
	struct kinfo_proc * kip;
	struct passwd *pw;
	double cpuv;
	struct nlist mylist[2];
	int fscale; /* thank you BSD guys for documenting this one */
	int mv,mem_unit,nicety;
	time_t born;
	struct pstats mypstats;
	struct user *u_addr = (struct user *)USRSTACK;
	long mem[2];
	char **argv;

	int cnt,i,j;
	char b2[1024];

	reset();

	kt=(kvm_t *)(GlobalKvm.getp());

	kip=kvm_getprocs(kt,KERN_PROC_ALL,0,&cnt);
	if (kip==NULL)
		return;

	mylist[0].n_name="_fscale";
	mylist[1].n_name=NULL;
	kvm_nlist(kt,mylist);
	fscale=1;
	if (mylist[0].n_value)
		kvm_read(kt,mylist[0].n_value,&fscale,sizeof(fscale));
	
	for(i=0;i<cnt;i++) {
		pli=new ProcessListItem();
		sprintf(b2,"%d",(int)(kip[i].kp_proc.p_pid));
		pli->setField(PID,b2);

		sprintf(b2,"%s",kip[i].kp_proc.p_comm);
		pli->setField(NAME,b2);

		if (!(kip[i].kp_proc.p_flag&P_SYSTEM)) {
			argv=kvm_getargv(kt,&(kip[i]),512);
			if (argv==NULL)
				b2[0]=0;
			else
				for(b2[0]=0,j=0;argv[j]!=NULL;j++) {
					strcat(b2,argv[j]);
					if (argv[j+1])
						strcat(b2," ");
					if (strlen(b2)>500)
						break;
				}
		} else {
			strcpy(b2,kip[i].kp_proc.p_comm);
		}

		b2[511]=0;
		pli->setField(LONGNAME,b2);

		pw=getpwuid(kip[i].kp_eproc.e_pcred.p_ruid);
		if (pw!=NULL)
			sprintf(b2,"%s",pw->pw_name);
		else
			sprintf(b2,"%d",kip[i].kp_eproc.e_pcred.p_ruid);
		pli->setField(OWNER,b2);

		pli->setField(MACHINE,get_local_hostname());

		mem[0]=kip[i].kp_eproc.e_vm.vm_rssize;
		mem[1]=kip[i].kp_eproc.e_vm.vm_map.size;

		mem[0]*=(long)getpagesize();

		mem_unit=0;
  		while(mem[0]>(99UL<<10)) {
			++mem_unit;
			mem[0]>>=10;
		}
		sprintf(b2,"%lu",mem[0]);
		strcat(b2,power_of_two_suffix(mem_unit));
		pli->setField(RSS,b2);

		mem_unit=0;
  		while(mem[1]>(99UL<<10)) {
			++mem_unit;
			mem[1]>>=10;
		}
		sprintf(b2,"%lu",mem[1]);
		strcat(b2,power_of_two_suffix(mem_unit));
		pli->setField(SIZE,b2);

		b2[1]=b2[2]=b2[3]=0;
		switch(kip[i].kp_proc.p_stat) {
			case SSTOP:
				b2[0]='T';
				break;
			case SSLEEP:
				if (kip[i].kp_proc.p_flag&P_SINTR)
					b2[0]='S'; /* FreeBSD dudes may wish an 'I' state here, but no */
				else
					b2[0]='D';
				break;
			case SRUN:
			case SIDL:
				b2[0]='R';
				break;
			case SZOMB:
				b2[0]='Z';
				break;
			default:
				b2[0]='U';
		}

		nicety=(int)(kip[i].kp_proc.p_nice-NZERO);
		if (mem[0]==0)
			b2[1]='W';
		if (nicety>0)
			if (strlen(b2)==1)
				strcat(b2," N");
			else
				strcat(b2,"N");
		
		pli->setField(STATE,b2);

		cpuv=(double)(kip[i].kp_proc.p_pctcpu);
		cpuv=cpuv*100.0/fscale;
		sprintf(b2,"%.2f",cpuv);
		pli->setField(CPU,b2);

		sprintf(b2,"%d",nicety);
		pli->setField(NICE,b2);

		mv=kip[i].kp_proc.p_priority-PZERO;
		sprintf(b2,"%d",mv);
		pli->setField(PRIORITY,b2);

#if __FreeBSD_version < 400000
		if (kvm_uread(kt,&(kip[i].kp_proc),
				(unsigned long)&u_addr->u_stats,
				(char *)&mypstats, sizeof(mypstats))==sizeof(mypstats)) {
			born=mypstats.p_start.tv_sec;
			time_to_gps(born,b2);
		} else
			sprintf(b2,"<not available>");
#else
		sprintf(b2,"<not stated in 4.x>");
#endif

		pli->setField(START,b2);

		process_list=g_list_append(process_list,(gpointer)pli);
	}
}

/* ********************************************************
 * 
 *
 *                 FreeBSDProcessDetailsPoller
 *	      
 *
 * ********************************************************* */

FreeBSDProcessDetailsPoller::FreeBSDProcessDetailsPoller()
  : ProcessDetailsPoller() {
	GlobalKvm.reference();
}

void FreeBSDProcessDetailsPoller::terminate() {
	GlobalKvm.dereference();
}

void FreeBSDProcessDetailsPoller::poll(int whichpid) {
	char b2[1024];
	int i,j,cnt;
	kvm_t *kt;
	struct kinfo_proc * kip;
	struct passwd *pw;
	int mv,mem_unit,nicety;
	time_t born;
	struct pstats mypstats;
	struct user *u_addr = (struct user *)USRSTACK;
	struct session mysession;
	long mem[2],sv[2];
	char **argv,*ttname;
	dev_t ttyd;

	reset();

	kt=(kvm_t *)(GlobalKvm.getp());

	kip=kvm_getprocs(kt,KERN_PROC_PID,whichpid,&cnt);
	if ((kip==NULL)||(!cnt))
		return;

	item=new ProcessItem();
	item->setField(MACHINE,get_local_hostname());

	sprintf(b2,"%d",(int)(kip[0].kp_proc.p_pid));
	item->setField(PID,b2);

	sprintf(b2,"%s",kip[0].kp_proc.p_comm);
	item->setField(NAME,b2);

	if (!(kip[0].kp_proc.p_flag&P_SYSTEM)) {
		argv=kvm_getargv(kt,&(kip[0]),512);
		if (argv==NULL)
			b2[0]=0;
		else
			for(b2[0]=0,j=0;argv[j]!=NULL;j++) {
				strcat(b2,argv[j]);
				if (argv[j+1])
					strcat(b2," ");
				if (strlen(b2)>500)
					break;
			}
	} else {
		strcpy(b2,kip[0].kp_proc.p_comm);
	}

	b2[511]=0;
	item->setField(LONGNAME,b2);

	pw=getpwuid(kip[0].kp_eproc.e_pcred.p_ruid);
	if (pw!=NULL)
		sprintf(b2,"%s",pw->pw_name);
	else
		sprintf(b2,"%d",kip[0].kp_eproc.e_pcred.p_ruid);
	item->setField(OWNER,b2);

	mem[0]=kip[0].kp_eproc.e_vm.vm_rssize;
	mem[1]=kip[0].kp_eproc.e_vm.vm_map.size;

	mem[0]*=(long)getpagesize();

	mem_unit=0;
	while(mem[0]>(99UL<<10)) {
		++mem_unit;
		mem[0]>>=10;
	}
	sprintf(b2,"%lu",mem[0]);
	strcat(b2,power_of_two_suffix(mem_unit));
	item->setField(RSS,b2);

	mem_unit=0;
	while(mem[1]>(99UL<<10)) {
		++mem_unit;
		mem[1]>>=10;
	}
	sprintf(b2,"%lu",mem[1]);
	strcat(b2,power_of_two_suffix(mem_unit));
	item->setField(SIZE,b2);

	b2[1]=b2[2]=b2[3]=0;
	switch(kip[0].kp_proc.p_stat) {
		case SSTOP:
			b2[0]='T';
			break;
		case SSLEEP:
			if (kip[0].kp_proc.p_flag&P_SINTR)
				b2[0]='S'; /* FreeBSD dudes may wish an 'I' state here, but no */
			else
				b2[0]='D';
			break;
		case SRUN:
		case SIDL:
			b2[0]='R';
			break;
		case SZOMB:
			b2[0]='Z';
			break;
		default:
			b2[0]='U';
	}
	nicety=(int)(kip[0].kp_proc.p_nice-NZERO);
	if (mem[0]==0)
		b2[1]='W';
	if (nicety>0)
		if (strlen(b2)==1)
			strcat(b2," N");
		else
			strcat(b2,"N");
	
	item->setField(STATE,b2);

	sprintf(b2,"%d",nicety);
	item->setField(NICE,b2);

	mv=kip[0].kp_proc.p_priority-PZERO;
	sprintf(b2,"%d",mv);
	item->setField(PRIORITY,b2);

	memset(&mypstats,0,sizeof(mypstats));
#if __FreeBSD_version < 400000
		if (kvm_uread(kt,&(kip[0].kp_proc),
				(unsigned long)&u_addr->u_stats,
				(char *)&mypstats, sizeof(mypstats))==sizeof(mypstats)) {
			born=mypstats.p_start.tv_sec;
			time_to_gps(born,b2);
		} else
			sprintf(b2,"<not available>");
#else
	sprintf(b2,"<not stated in 4.x>");
#endif

		item->setField(START,b2);

	sprintf(b2,"%d",kip[0].kp_eproc.e_ppid);
	item->setField(PPID,b2);
	sprintf(b2,"%d",kip[0].kp_eproc.e_pgid);
	item->setField(PGID,b2);
	sprintf(b2,"%d",kip[0].kp_eproc.e_tpgid);
	item->setField(TPGID,b2);

	if (kvm_read(kt,(unsigned long)kip[0].kp_eproc.e_sess,
			(char *)&mysession, sizeof(mysession))==sizeof(mysession)) {
		sprintf(b2,"%d",mysession.s_sid);
	} else
		sprintf(b2,"<not available>");

	item->setField(SID,b2);

	ttyd=kip[0].kp_eproc.e_tdev;
	if (ttyd==NODEV) {
		strcpy(b2,"none");
	} else {
		ttname=devname(ttyd,S_IFCHR);
		if (ttname==NULL) {
			strcpy(b2,"none");
		} else {
			strcpy(b2,ttname);		
		}
	}
	item->setField(TTY,b2);

	parse_process_flags((unsigned long)(kip[0].kp_proc.p_flag),b2);
	item->setField(FLAGS,b2);

	sprintf(b2,"%lu",mypstats.p_ru.ru_minflt);
	item->setField(MINFLT,b2);
	sprintf(b2,"%lu",mypstats.p_ru.ru_majflt);
	item->setField(MAJFLT,b2);
	sprintf(b2,"%lu",mypstats.p_cru.ru_minflt);
	item->setField(CMINFLT,b2);
	sprintf(b2,"%lu",mypstats.p_cru.ru_majflt);
	item->setField(CMAJFLT,b2);

	sv[0]=mypstats.p_ru.ru_utime.tv_sec*100+mypstats.p_ru.ru_utime.tv_usec/10000;
	sprintf(b2,"%lu",sv[0]);
	item->setField(USRJIFFIES,b2);
	sv[0]=mypstats.p_ru.ru_stime.tv_sec*100+mypstats.p_ru.ru_stime.tv_usec/10000;
	sprintf(b2,"%lu",sv[0]);
	item->setField(KRNJIFFIES,b2);

	sv[0]=mypstats.p_cru.ru_utime.tv_sec*100+mypstats.p_cru.ru_utime.tv_usec/10000;
	sprintf(b2,"%lu",sv[0]);
	item->setField(CUSRJIFFIES,b2);
	sv[0]=mypstats.p_cru.ru_stime.tv_sec*100+mypstats.p_cru.ru_stime.tv_usec/10000;
	sprintf(b2,"%lu",sv[0]);
	item->setField(CKRNJIFFIES,b2);

	sprintf(b2,"n/a");
	item->setField(RSSLIM,b2);

#if __FreeBSD_version > 399999
	item->setField(MINFLT,"not stated in 4.x");
	item->setField(MAJFLT,"not stated in 4.x");
	item->setField(CMINFLT,"not stated in 4.x");
	item->setField(CMAJFLT,"not stated in 4.x");
	item->setField(USRJIFFIES,"not stated in 4.x");
	item->setField(KRNJIFFIES,"not stated in 4.x");
	item->setField(CUSRJIFFIES,"not stated in 4.x");
	item->setField(CKRNJIFFIES,"not stated in 4.x");
#endif

#if __FreeBSD_version < 400000
	sv[0]=kip[0].kp_eproc.e_procsig.ps_sigignore;
	make_signal_string(sv[0],b2);
	item->setField(SIGIGNORED,b2);
	sv[0]=kip[0].kp_eproc.e_procsig.ps_sigcatch;
	make_signal_string(sv[0],b2);
	item->setField(SIGCAUGHT,b2);
#endif

	strcpy(b2,"not stated");
#if __FreeBSD_version >= 400000
	item->setField(SIGIGNORED,"not stated in 4.x");
	item->setField(SIGCAUGHT,"not stated in 4.x");
#endif
	item->setField(SIGPENDING,b2);
	item->setField(SIGBLOCKED,b2);
}

void FreeBSDProcessDetailsPoller::
	parse_process_flags(unsigned long flags,
					char *dest)
{
	int i;
	dest[0]=0;
	for(i=0;i<FREEBSD_FLAG_COUNT;i++)
		if (flags&flag_value[i]) {
			strcat(dest,flag_text[i]);
			strcat(dest,",");
		}
	if (!strlen(dest)) {
		strcpy(dest,"(no flags set)");
	}
	if (dest[strlen(dest)-1]==',')
		dest[strlen(dest)-1]=0;
}

void FreeBSDProcessDetailsPoller::
make_signal_string(unsigned long sigs,char *dest)
{
	int i;
	dest[0]=0;
	for(i=0;i<POSIXSIG;i++) {
		if (sigs&sigmask(posixvalues[i])) {
			strcat(dest,posixnames[i]);
			strcat(dest,",");			
		}
	}
	if (dest[strlen(dest)-1]==',')
		dest[strlen(dest)-1]=0;
}


FreeBSDSystemInfoProvider::FreeBSDSystemInfoProvider() {
  prevTotal=0;
  prevUsed=0;
  GlobalKvm.reference();
  CpuCount=1; // no SMP support yet
}

void FreeBSDSystemInfoProvider::terminate() {
  GlobalKvm.dereference();
}

void FreeBSDSystemInfoProvider::update() {
  update_cpu();
  SMP_faker();
  update_memory();
}

/* this method's code comes mostly from gkrellm 0.9.6 www.gkrellm.net */
void FreeBSDSystemInfoProvider::update_cpu() {
	#define N_CP_TIME	0
	static struct nlist nl[] = {
		{ "_cp_time" },{ "" }
	};

	long        cp_time[CPUSTATES];
	gulong      cpu[4]; /* user,nice,sys,idle */
	gulong      lused,ltotal;
	kvm_t *kt;

	kt=(kvm_t *)(GlobalKvm.getp());

	if (cpu_usage>1.0)
		cpu_usage=0.0;

	if (kvm_nlist(kt, nl) < 0 || nl[0].n_type == 0)
		return;

	if (kvm_read(kt, nl[N_CP_TIME].n_value,
		     (char *)&cp_time, sizeof(cp_time)) == sizeof(cp_time))
	{
		cpu[0] = cp_time[CP_USER];
		cpu[1] = cp_time[CP_NICE];
		cpu[2] = cp_time[CP_SYS];
		cpu[3] = cp_time[CP_IDLE];
		lused=(cpu[0]+cpu[1]+cpu[2]);
		ltotal=lused+cpu[3];
		lused-=prevUsed;
		ltotal-=prevTotal;
		prevUsed+=lused;
		prevTotal+=ltotal;
		cpu_usage=((double)lused)/((double)ltotal);
		if (cpu_usage>1.0)
			cpu_usage=1.0;
		if (cpu_usage<0.0)
			cpu_usage=0.0;
	}
}

void FreeBSDSystemInfoProvider::update_memory() {
	static unsigned long cnt_offset;
	static struct nlist mylist[2];
	static struct vmmeter sum;
	long pagesize;
	kvm_t *kt;

	memset(&sum,0,sizeof(sum));

	kt=(kvm_t *)(GlobalKvm.getp());

	mylist[0].n_name="_cnt";
	mylist[1].n_name=NULL;

	kvm_nlist(kt,mylist);
	cnt_offset=mylist[0].n_value;
	kvm_read(kt,(unsigned long)cnt_offset,&sum,sizeof(sum));
	
	pagesize=sum.v_page_size;
	memory_free=(sum.v_free_count+sum.v_inactive_count)*pagesize;
	memory_used=(sum.v_active_count+sum.v_wire_count)*pagesize;
	memory_total=memory_used+memory_free;
	
	swapmode(&swap_total, &swap_free);
	swap_used=swap_total-swap_free;

	swap_used<<=10;
	swap_free<<=10;
	swap_total<<=10;
}

/* After reading the next function answer these ( 5 - 1 ) questions:

   1) Why does Linux have a proc filesystem with all this information in ?

   2) Why did the Operating System cross the road ?

   3) Is it of any relevance to know why an inanimated piece of software
      crossed a road ?

   4) Is the IOCCC taking sole functions as entries ?

*/

int FreeBSDSystemInfoProvider::swapmode(unsigned long *retavail, unsigned long *retfree)
{
	kvm_t *kvmd;

static struct nlist nl[] = {
#define N_CNT		0
	{ "_cnt" },
#define N_BUFSPACE	1
	{ "_bufspace" },
#if __FreeBSD_version < 400000
#define VM_SWAPLIST	2
	{ "_swaplist" },
#define VM_SWDEVT	3
	{ "_swdevt" },
#define VM_NSWAP	4
	{ "_nswap" },
#define VM_NSWDEV	5
	{ "_nswdev" },
#define VM_DMMAX	6
	{ "_dmmax" },
#endif
	{ "" }
};

	int used, avail;
#if  __FreeBSD_version >= 400000
	struct kvm_swap kvmswap;
#else
	char *header;
	int hlen, nswap, nswdev, dmmax;
	int i, div, nfree, npfree;
	struct swdevt *sw;
	long blocksize, *perdev;
	u_long ptr;
	struct rlist head;
#  if __FreeBSD_version >= 220000
	struct rlisthdr swaplist;
#  else 
	struct rlist *swaplist;
#  endif
	struct rlist *swapptr;
#endif

	/*
	 * Counter for error messages. If we reach the limit,
	 * stop reading information from swap devices and
	 * return zero. This prevent endless 'bad address'
	 * messages.
	 */
	static int warning = 10;

	if (warning <= 0)
		{
		/* a single warning */
		if (!warning)
	    		{
			warning--;
			fprintf(stderr, "Too much errors, stop reading swap devices ...\n");
			(void)sleep(3);
			}
		return(0);
		}
	warning--;		/* decrease counter, see end of function */

	kvmd=(kvm_t *)(GlobalKvm.getp());
	if (kvmd == NULL)
		return(0);

	if (kvm_nlist(kvmd,nl)<0) {
		return(0);
	}

#if  __FreeBSD_version >= 400000
	if (kvm_getswapinfo(kvmd, &kvmswap, 1, 0) < 0)
		{
		fprintf(stderr, "kvm_getswapinfo failed\n");
		return(0);
		}

	*retavail = avail = kvmswap.ksw_total;
	used = kvmswap.ksw_used;
	*retfree = kvmswap.ksw_total - used;
#else
	if (kvm_read(kvmd, nl[VM_NSWAP].n_value,
		     &nswap, sizeof(nswap)) != sizeof(nswap))
		return(0);
	if (!nswap)
		{
		fprintf(stderr, "No swap space available\n");
		return(0);
		}

	if (kvm_read(kvmd, nl[VM_NSWDEV].n_value,
		     &nswdev, sizeof(nswdev)) != sizeof(nswdev))
		return(0);
	if (kvm_read(kvmd, nl[VM_DMMAX].n_value,
		     &dmmax, sizeof(dmmax)) != sizeof(dmmax))
		return(0);
	if (kvm_read(kvmd, nl[VM_SWAPLIST].n_value,
		     &swaplist, sizeof(swaplist)) != sizeof(swaplist))
		return(0);

	if ((sw = (struct swdevt *)malloc(nswdev * sizeof(*sw))) == NULL ||
	    (perdev = (long *)malloc(nswdev * sizeof(*perdev))) == NULL)
		{
		perror("malloc");
		exit(1);
		}
	if (kvm_read(kvmd, nl[VM_SWDEVT].n_value,
		     &ptr, sizeof ptr) != sizeof ptr)
		return(0);
	if (kvm_read(kvmd, ptr,
		     sw, nswdev * sizeof(*sw)) != nswdev * sizeof(*sw))
		return(0);

	/* Count up swap space. */
	nfree = 0;
	memset(perdev, 0, nswdev * sizeof(*perdev));
#if  __FreeBSD_version >= 220000
	swapptr = swaplist.rlh_list;
	while (swapptr)
#else
	while (swaplist)
#endif
		{
		int	top, bottom, next_block;
#if  __FreeBSD_version >= 220000
		if (kvm_read(kvmd, (u_long)swapptr, &head,
			     sizeof(struct rlist)) != sizeof(struct rlist))
			return (0);
#else
		if (kvm_read(kvmd, (u_long)swaplist, &head,
			     sizeof(struct rlist)) != sizeof(struct rlist))
			return (0);
#endif

		top = head.rl_end;
		bottom = head.rl_start;

		nfree += top - bottom + 1;

		/*
		 * Swap space is split up among the configured disks.
		 *
		 * For interleaved swap devices, the first dmmax blocks
		 * of swap space some from the first disk, the next dmmax
		 * blocks from the next, and so on up to nswap blocks.
		 *
		 * The list of free space joins adjacent free blocks,
		 * ignoring device boundries.  If we want to keep track
		 * of this information per device, we'll just have to
		 * extract it ourselves.
		 */

		while (top / dmmax != bottom / dmmax)
			{
			next_block = ((bottom + dmmax) / dmmax);
			perdev[(bottom / dmmax) % nswdev] +=
				next_block * dmmax - bottom;
			bottom = next_block * dmmax;
			}
		perdev[(bottom / dmmax) % nswdev] +=
			top - bottom + 1;

#if  __FreeBSD_version >= 220000
		swapptr = head.rl_next;
#else
		swaplist = head.rl_next;
#endif
		}

	header = getbsize(&hlen, &blocksize);
	div = blocksize / 512;
	avail = npfree = 0;
	for (i = 0; i < nswdev; i++)
		{
		int xsize, xfree;

		/*
		 * Don't report statistics for partitions which have not
		 * yet been activated via swapon(8).
		 */

		xsize = sw[i].sw_nblks;
		xfree = perdev[i];
		used = xsize - xfree;
		npfree++;
		avail += xsize;
		}

	/* 
	 * If only one partition has been set up via swapon(8), we don't
	 * need to bother with totals.
	 */
	*retavail = avail / 2;
	*retfree = nfree / 2;
	used = avail - nfree;
	free(sw); free(perdev);
#endif /* __FreeBSD_version >= 400000 */

	/* increase counter, no errors occurs */
	warning++; 

	return  (int)(((double)used / (double)avail * 100.0) + 0.5);

}

#endif /* HAVEFREEBSD */
