/*
 *  Hamlib Interface - calibration routines
 *  Copyright (c) 2000-2004 by Stephane Fillod and Frank Singleton
 *
 *		$Id: cal.c,v 1.7 2007/11/05 03:48:14 aa6e Exp $
 *
 *   This library is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License as
 *   published by the Free Software Foundation; either version 2 of
 *   the License, or (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/**
 * \addtogroup rig_internal
 * @{
 */

/**
 * \file cal.c
 * \brief Calibration routines
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <hamlib/rig.h>
#include "cal.h"

/* add rig_set_cal(cal_table), rig_get_calstat(rawmin,rawmax,cal_table), */

/**
 * \brief Convert raw S-meter data to calibated value, according to table
 * \param rawval input value
 * \param cal calibration table
 * \return calibrated value

 * cal_table_t is a data type suited to hold linear calibration
 * cal_table_t.size tell the number of plot cal_table_t.table contains
 * If a value is below or equal to cal_table_t.table[0].raw, 
 * rig_raw2val() will return cal_table_t.table[0].val
 * If a value is greater or equal to cal_table_t.table[cal_table_t.size-1].raw, 
 * rig_raw2val() will return cal_table_t.table[cal_table_t.size-1].val
 */

float HAMLIB_API rig_raw2val(int rawval, const cal_table_t *cal)
{
#ifdef WANT_CHEAP_WNO_FP
	int interpolation;
#else
	float interpolation;
#endif
	int i;

	/* ASSERT(cal != NULL) */
	/* ASSERT(cal->size <= MAX_CAL_LENGTH) */

	if (cal->size == 0)
			return rawval;

	for (i=0; i<cal->size; i++)
		if (rawval < cal->table[i].raw)
			break;

	if (i==0)
		return cal->table[0].val;

	if (i>=cal->size)
		return cal->table[i-1].val;

	if (cal->table[i].raw == cal->table[i-1].raw)	/* catch divide by 0 error */
		return cal->table[i].val;

#ifdef WANT_CHEAP_WNO_FP
	/* cheap, less accurate, but no fp needed */
	interpolation = ((cal->table[i].raw - rawval) * 
			 (cal->table[i].val - cal->table[i-1].val)) /
			  (cal->table[i].raw - cal->table[i-1].raw);

	return cal->table[i].val - interpolation;
#else
	interpolation = ((cal->table[i].raw - rawval) * 
			 (float)(cal->table[i].val - cal->table[i-1].val)) /
			  (float)(cal->table[i].raw - cal->table[i-1].raw);
#endif
	return cal->table[i].val - interpolation;
}

/** @} */
