
/****************************************************************************
 *
 * All portions copyright their respective authors.  All rights reserved.
 *
 * This file is part of IVMan (ivm).
 *
 * This file may be distributed under the terms of the Q Public License
 * as defined by Troll Tech AS of Norway and appearing in the file
 * LICENSE.QPL included in the packaging of this file.
 * 
 * See http://www.troll.no/qpl for QPL licensing information.
 *
 * $Id: dbus_interface.c,v 1.8 2007/01/29 11:52:31 makoenig Exp $
 *****************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib-lowlevel.h>

#include "dbus_interface.h"
#include "hal_interface.h"
#include "log.h"
#include "common.h"

char *ivm_dbus_get_name_owner(const char *name)
{
	DBusMessage *	message;
	DBusMessage *	reply;
	char *		owner = NULL;

	g_return_val_if_fail (name != NULL, NULL);

	if (dbus_connection == NULL)
		return NULL;

	if ((message = dbus_message_new_method_call(DBUS_SERVICE_DBUS,
						    DBUS_PATH_DBUS,
						    DBUS_INTERFACE_DBUS, "GetNameOwner")))
	{
		dbus_message_append_args (message, DBUS_TYPE_STRING, &name, DBUS_TYPE_INVALID);
		if ((reply = dbus_connection_send_with_reply_and_block (dbus_connection, message, -1, NULL)))
		{
			DBusError error;
			const char *tmp_name = NULL;

			dbus_error_init(&error);
			if (dbus_message_get_args(reply,
						  &error,
						  DBUS_TYPE_STRING, &tmp_name,
						  DBUS_TYPE_INVALID)) {
				if (dbus_error_is_set(&error)) {
					log_error("dbus error: %s", error.message);
					dbus_error_free(&error);
					return NULL;
				}
				owner = g_strdup(tmp_name);
			}
			dbus_message_unref(reply);
		}
		dbus_message_unref(message);
	}

	return owner;
}

void ivm_check_dbus_error(DBusError * error)
{
    if (dbus_error_is_set(error)) {
        DEBUG(_("DBus Error! %s: %s"), error->name, error->message);
	dbus_error_free(error);
    }
}

void ivm_dbus_init(void)
{
	DBusError error;

	dbus_error_init(&error);
	dbus_connection = dbus_bus_get(DBUS_BUS_SYSTEM, &error);
	if ((dbus_connection == NULL) || dbus_error_is_set(&error))
	{
		log_error("Could not get the system bus.  Make sure dbus is running!");
		dbus_error_free(&error);
		return;
	}

	dbus_connection_set_exit_on_disconnect(dbus_connection, FALSE);
	dbus_connection_setup_with_g_main(dbus_connection, NULL);

	if (!dbus_connection_add_filter(dbus_connection, ivm_dbus_signal_filter, NULL, NULL))
	{
		log_error("could not attach a dbus message filter.");
		dbus_connection = NULL;
		return;
	}

	dbus_bus_add_match(dbus_connection,
			   "type='signal',"
			   "interface='" DBUS_INTERFACE_DBUS "',"
			   "sender='" DBUS_SERVICE_DBUS "'",
			   &error);
	if (dbus_error_is_set(&error))
		log_error("dbus error: %s", error.message);

	dbus_error_free(&error);
	DEBUG("Connected to dbus");
}

void *ivm_dbus_reinit(void)
{
	do {
		ivm_dbus_init();
		g_usleep (G_USEC_PER_SEC * 3);
	} while (dbus_connection == NULL);

	/* if HAL was quick it is already back on the bus. Thus, we do not receive NameOwnerChanged */
	if (ivm_dbus_get_name_owner("org.freedesktop.Hal"))
		hal_init();

	return NULL;
}

DBusHandlerResult ivm_dbus_signal_filter(DBusConnection *connection, DBusMessage *message, void *user_data)
{
	DBusError error;
	gboolean handled = FALSE;

	dbus_error_init(&error);

	if (dbus_message_get_type(message) != DBUS_MESSAGE_TYPE_SIGNAL)
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

	/* Disconnected from DBus */
	if (dbus_message_is_signal(message, DBUS_INTERFACE_LOCAL, "Disconnected")) {
		DEBUG("disconnected from DBus");
		hal_deinit();
		dbus_connection_unref(dbus_connection);
		dbus_connection = NULL;
		g_thread_create((GThreadFunc)ivm_dbus_reinit, NULL, FALSE, NULL);
		handled = TRUE;
	}

	/* Check for signal NameOwnerChange */
	if (dbus_message_is_signal(message, DBUS_INTERFACE_DBUS, "NameOwnerChanged")) {
		char 	*service;
		char	*old_owner;
		char	*new_owner;

		if (dbus_message_get_args(message, &error,
					  DBUS_TYPE_STRING, &service,
					  DBUS_TYPE_STRING, &old_owner,
					  DBUS_TYPE_STRING, &new_owner,
					  DBUS_TYPE_INVALID)) {

			gboolean old_owner_good = (old_owner && (strlen(old_owner) > 0));
			gboolean new_owner_good = (new_owner && (strlen(new_owner) > 0));

			if (strcmp(service, "org.freedesktop.Hal") == 0) {
				if (!old_owner_good && new_owner_good) {
					/* Hal just appeared */
					hal_init();
					handled = TRUE;
				}
				else if (old_owner_good && !new_owner_good) {
					/* Hal went away */
					hal_deinit();
					handled = TRUE;
				}
			}
		} else {
			log_error("dbus error: %s", error.message);
			handled = TRUE;
		}
	}

	dbus_error_free(&error);
	return (handled ? DBUS_HANDLER_RESULT_HANDLED : DBUS_HANDLER_RESULT_NOT_YET_HANDLED);
}
