(******************************************************************************
 *                             Core                                           *
 *                                                                            *
 * Copyright (C) 2008- Jane Street Holding, LLC                               *
 *    Contact: opensource@janestreet.com                                      *
 *    WWW: http://www.janestreet.com/ocaml                                    *
 *                                                                            *
 *                                                                            *
 * This library is free software; you can redistribute it and/or              *
 * modify it under the terms of the GNU Lesser General Public                 *
 * License as published by the Free Software Foundation; either               *
 * version 2 of the License, or (at your option) any later version.           *
 *                                                                            *
 * This library is distributed in the hope that it will be useful,            *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          *
 * Lesser General Public License for more details.                            *
 *                                                                            *
 * You should have received a copy of the GNU Lesser General Public           *
 * License along with this library; if not, write to the Free Software        *
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  *
 *                                                                            *
 ******************************************************************************)

(** Module for easily generating unit tests.  Based on code posted by
    padiolea\@irisa.fr to the caml mailing list. *)


(** the type of a random ['a]-generator *)
type 'a gen = unit -> 'a

(** positive float generator *)
val pfg : float gen

(** float generator (no nan, inf, etc.) *)
val fg : float gen

(** natural number generator *)
val nng : int gen

(** unsigned int generator (uniform random in range min_int, max_int) *)
val uig : int gen

(** char generator *)
val cg : char gen

(** pair generator *)
val pg : 'a gen -> 'b gen -> ('a * 'b) gen

(** triple generator *)
val tg : 'a gen -> 'b gen -> 'c gen -> ('a * 'b * 'c) gen

(** list generator *)
val lg : 'a gen -> ?size_gen:int gen -> 'a list gen

(** string generator *)
val sg : ?char_gen : char gen -> ?size_gen : int gen -> string gen


(** Given a list of generators, returns generator that randomly uses one of the
    generators from the list *)
val oneof : 'a gen list -> 'a gen

(** generator that always returns given value *)
val always : 'a -> 'a gen


(** Given list of [(frequency,value)] pairs, returns value with probability
    proportional to given frequency *)
val frequency : (int * 'a) list -> 'a

(** a [gen]-returning version of [frequency] *)
val frequencyl : (int * 'a) list -> 'a gen

(** [repeat n test t] tests that the first [n] values generated by [t]
    all pass [test] *)
val repeat : int -> ('a -> unit) -> 'a gen -> unit

(** [laws iter gen func] applies [func] repeatedly ([iter] times) on output
    of [gen], and if [func] ever returns false, then the input that caused
    the failure is returned optionally. *)
val laws : int -> 'a gen -> ('a -> bool) -> 'a option

(** Like laws, but throws an exception instead of returning an option. *)
val laws_exn : string -> int -> 'a gen -> ('a -> bool) -> unit
