/*----------------------------------------------------------------------------------
Name               : acpi.c
Author             : Marvin Raaijmakers
Description        : For reading ACPI events
Date of last change: 02-Sep-2007
History            : 02-Sep-2007 read_event() now only cuts off a line when the line
                                 starts with "hotkey" or "button"

    Copyright (C) 2006-2007 Marvin Raaijmakers

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or any later version.
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

    You can contact me at: marvinr(at)users(dot)sf(dot)net
    (replace (at) by @ and (dot) by .)
------------------------------------------------------------------------------------*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <unistd.h>

#include <keytouch-editor.h>
#include <acpi.h>

#include "ud_socket.h"

static char *read_line (int fd);


Boolean
check_acpi (void)
/*
Returns:
	TRUE if ACPI_SOCKETFILE is a valid and accesible ACPI socket, otherwise FALSE
	will be returned.
*/
{
	int fd;
	
	fd = ud_connect (ACPI_SOCKETFILE);
	if (fd >= 0)
	{
		close (fd);
	}
	return fd >= 0;
}


char
*read_eventdescr (int fd)
/*
Input:
	fd	- The socket to read from
Output:
	-
Returns:
	A pointer to the readen event description or NULL if reading the event
	failed.
Description:
	read_eventdescr() reads a line from fd and puts it in a string. The last
	space character in this string will be replaced by '\0' if the string does
	not begin with "ibm". A pointer to the resulting string will be returned.
	The string may not be modified and may not be used after calling this
	function again.
*/
{
	char *line, *last_space, *c;
	
	line = read_line(fd);
	
	/* If the line starts with "hotkey" or "button", then the last part of
	 * the line will be the value of a count. This value will be cut off.
	 * Lines that start with one of the following strings are known to not
	 * have such counter value:
	 * "Hotkey", "ibm", "video", "sony/hotkey", "hkey"
	 */
	if (line && (strncmp ("hotkey", line, 6) == 0 || strncmp("button", line, 6) == 0))
	{
		last_space = NULL;
		for (c = line; *c != '\0'; c++)
		{
			if (isspace(*c))
			{
				last_space = c;
			}
		}
		if (last_space != NULL)
		{
			*last_space = '\0';
		}
	}
	return (line);
}


/* The code below was borrowed from:
 *  acpi_listen.c - ACPI client for acpid's UNIX socket
 *
 *  Portions Copyright (C) 2003 Sun Microsystems (thockin@sun.com)
 *  Some parts (C) 2003 - Gismo / Luca Capello <luca.pca.it> http://luca.pca.it
 */
#define MAX_BUFLEN	1024
static char *
read_line(int fd)
{
	static char *buf;
	int buflen = 64;
	int i = 0;
	int r;
	int searching = 1;

	while (searching) {
		buf = realloc(buf, buflen);
		if (!buf) {
			fprintf(stderr, "ERR: malloc(%d): %s\n",
				buflen, strerror(errno));
			return NULL;
		}
		memset(buf+i, 0, buflen-i);

		while (i < buflen) {
			r = read(fd, buf+i, 1);
			if (r < 0 && errno != EINTR) {
				/* we should do something with the data */
				fprintf(stderr, "ERR: read(): %s\n",
					strerror(errno));
				return NULL;
			} else if (r == 0) {
				/* signal this in an almost standard way */
				errno = EPIPE;
				return NULL;
			} else if (r == 1) {
				/* scan for a newline */
				if (buf[i] == '\n') {
					searching = 0;
					buf[i] = '\0';
					break;
				}
				i++;
			}
		}
		if (buflen >= MAX_BUFLEN) {
			break;
		}
		buflen *= 2;
	}

	return buf;
}
