/*************************************************/
/* methods for class XDevice                     */
/*                                               */
/* visible Device                                */
/*                                               */
/* Andreas Rostin                                */
/* 15.03.99                                      */
/*************************************************/
#include <qpainter.h>

#include <stdio.h>
#include <string.h>
#include <math.h>

#include "mainw.h"
#include "xwire.h"
#include "xdevice.h"
#include "klogic.h"
#include "calculator.h"
#include "value.h"
#include "deviceTypes.h"
#include "busOutputArray.h"
#include "deviceText.h"
#include "grid.h"

/***************************************************/
/* static methods of XDevice class                 */
/***************************************************/
int XDevice::instance = 0;

BasicImageCache XDevice::m_oBasicImageCache;
ImageCache XDevice::m_oImageCache;

const QString XDevice::INPUT_TRISTATE_CONTROL = "EN";
const QString BusOutputArray::IO_BUS_DATA[8] = {"D0", "D1", "D2", "D3", "D4", "D5", "D6", "D7"};

/***************************************************/
/* methods of XDevice instance                     */
/***************************************************/
XDevice::XDevice(int iFunction, const QPoint& oPos, int iSize)
	:XObject(oPos), DeviceInterface(iFunction)
{
	instance++;

	if (iSize > 0) {
		setSize(iSize);
	} else {
		if (Device::IMPORT_IGNORE_GLOBAL) {
			setSize(Global::Device::DEFAULT_SIZE);
		} else {
			setSize(Global::Device::getSize());
		}
	}

	if (Device::IMPORT_IGNORE_GLOBAL)
		deviceOutputInverted = false;
	else
		deviceOutputInverted = Global::Device::isInverted();

	deviceMaxInput = Global::Device::MAX_INPUT_CONNECTORS;
	deviceMaxOutput = Global::Device::MAX_OUTPUT_CONNECTORS;
	m_bGraphEnabled = true;
	m_poTextDev = 0;
	m_iTristateInputMoved = 0;
	m_iTristateImportPos = 0;

	m_oImageKey = ImageKey(type(), size(), width(), clock(), hasMaster(), Global::Device::getSymbolSet());
}

// remove device: remove all inputs and outputs first
XDevice::~XDevice()
{
	instance--;

	int idx;

	// remove all inputs
	while (m_oIncomingWires.First()) {
		idx = m_oIncomingWires.First()->Get()->getInputIndex(this);
		m_oIncomingWires.First()->Get()->disconnectInput(idx);
	}
	m_oIncomingWires.Destroy();

	// remove all outputs
	while (m_oOutgoingWires.First()) {
		idx = m_oOutgoingWires.First()->Get()->getOutputIndex(this);
		if (XWire::WFAIL == idx)
			fatal("XDevice::~XDevice() this is unknown in a connected wire !?");
		m_oOutgoingWires.First()->Get()->disconnectOutput(idx);
	}
	m_oOutgoingWires.Destroy();

	if (m_poTextDev) delete m_poTextDev;
}

int XDevice::size()
{	return m_iSize;
}

bool XDevice::setSize(int iSize)
{	KlogicList<XWire> *lw;
	XWire *w;
	int iMinSize = Global::Device::MINSIZE;
	int iMaxY = 0;
	bool retval;

	lw = m_oIncomingWires.First();
	while(lw) {
		w = lw->Get();
		int idx = w->getInputIndex(this);
		if (w->getInputPosition(idx).y() - getPos().y() > iMaxY)
			iMaxY = w->getInputPosition(idx).y() - getPos().y();
		lw = lw->Next();
	}
	lw = m_oOutgoingWires.First();
	while(lw) {
		w = lw->Get();
		int idx = w->getOutputIndex(this);
		if (XWire::WFAIL == idx)
			fatal("XDevice::setSize() this is unknown in a connected wire !?");
		if (w->getOutputPosition(idx).y() - getPos().y() > iMaxY)
			iMaxY = w->getOutputPosition(idx).y() - getPos().y();
		lw = lw->Next();
	}
	iMaxY = iMaxY / Grid::GRID;
	if (iMaxY > iMinSize) iMinSize = iMaxY;

	if (iSize > iMinSize) {
		if (size() != iSize) retval = true;
		else retval = false;
		m_iSize = iSize;
	} else {
		if (size() != iMinSize) retval = true;
		else retval = false;
		m_iSize = iMinSize;
	}
	m_oImageKey.setSize(iSize);
	return retval;
}

int XDevice::neededSize()
{	int iSize = Global::Device::MINSIZE;

	for (ConnectorList::iterator iter = m_oNamedOutput.begin(); iter != m_oNamedOutput.end(); iter++) {
		const Connector& oConnector = *iter;
		if (oConnector.getPosition() + 1 > iSize)
			iSize = oConnector.getPosition();
	}
	for (ConnectorList::iterator iter = m_oNamedInput.begin(); iter != m_oNamedInput.end(); iter++) {
		const Connector& oConnector = *iter;
		if (oConnector.getPosition() + 1 > iSize)
			iSize = oConnector.getPosition();
	}

	if (iSize < Global::Device::MINSIZE) iSize = Global::Device::MINSIZE;
	return iSize;
}

int XDevice::maxI()
{	return deviceMaxInput;
}

int XDevice::currI()
{       return m_oIncomingWires.counter();
}

void XDevice::setMaxI(int newmax)
{	if (newmax < currI()) deviceMaxInput = currI();
	else deviceMaxInput = newmax;
}

int XDevice::maxO()
{	return deviceMaxOutput;
}

int XDevice::currO()
{       return m_oOutgoingWires.counter();
}

void XDevice::setMaxO(int newmax)
{	if (newmax < currO()) deviceMaxOutput = currO();
	else deviceMaxOutput = newmax;
}

// output inverted by default?
bool XDevice::isInverted()
{	return deviceOutputInverted;
}

// output inverted by default?
// 1=invert always
void XDevice::setInverted(bool bEnable)
{
	deviceOutputInverted = bEnable;
}

// virtual
void XDevice::setName(const QString& sName)
{
	Device::setName(sName);

	if (nameDisplayed())
		m_poTextDev->setName(sName);
}

// show/hide the name of the display
void XDevice::displayName(bool bEnable)
{
	if (!getID())
		return;

	if (bEnable) {
		QString sText = getName();
		if (sText.isEmpty()) sText = " ";

		bool bWasEnabled = false;
		if (!m_poTextDev)
			m_poTextDev = new TextDevice(getPos());
		else
			bWasEnabled = true;

		m_poTextDev->setName(sText);

		if (!bWasEnabled) {
			QRect oDimensions = m_poTextDev->getGeometry();
			m_poTextDev->moveBy(QPoint(5, -1 * oDimensions.height() - 2));
		}
	} else {
		delete m_poTextDev;
		m_poTextDev = 0;
	}
}

// is the name of the device shown/hidden
bool XDevice::nameDisplayed()
{
	return (m_poTextDev != 0);
}

// set all output to tristate enabled
// add/remove "EN" input
void XDevice::setTristate(bool bEnable)
{
	if (!m_iHasTristateOutput && !bEnable || m_iHasTristateOutput && bEnable) return;
	Device::setTristate(bEnable);

	// add/remove "EN" input if nessesary
	if (m_iHasTristateOutput) {
		setSize(neededSize() + 1);

		// if possible, add it at top position
		if (type() != DeviceType::fRAM && !m_oIncomingWires.counter() && !m_iTristateInputMoved) {
			m_iTristateInputMoved++;
			// move all inputs down
			for (ConnectorList::iterator iter = m_oNamedInput.begin(); iter != m_oNamedInput.end(); iter++)
				(*iter).setPosition((*iter).getPosition() + 1);

			m_iTristateEnableID = addInputName(INPUT_TRISTATE_CONTROL, 0);
		} else {
			m_iTristateEnableID = addInputName(INPUT_TRISTATE_CONTROL);
		}
	} else {
		// if possible and needed, move all inputs up again
		if (!m_oNamedInput.find(m_iTristateEnableID))
			return;
		if (!m_oIncomingWires.counter() && m_iTristateInputMoved) {
			m_iTristateInputMoved--;
			removeInputName(m_iTristateEnableID);
			for (ConnectorList::iterator iter = m_oNamedInput.begin(); iter != m_oNamedInput.end(); iter++)
				(*iter).setPosition((*iter).getPosition() - 1);
		} else {
			removeInputName(m_iTristateEnableID);
		}
		m_iTristateEnableID = 0;
		setSize(neededSize());
	}
	if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
}

// check if all outputs are inverted
int XDevice::invertGraph(int value, int iOutputID)
{	KlogicList<XWire> *lw = m_oOutgoingWires.First();
	int idx;

	if (!iOutputID) {
		while (lw) {
			if (!lw->Get()->outputIsInverted(this)) return value;
			lw = lw->Next();
		}
		if (value == 0) return 1;
		else return 0;
	} else {
		while (lw) {
			idx = lw->Get()->getOutputIndex(this);
			if (XWire::WFAIL == idx)
				fatal("XDevice::invertGraph() this is unknown in a connected wire !?");
			if (lw->Get()->outputID(idx) == iOutputID) {
				if (!lw->Get()->outputIsInverted(this)) return value;
				if (value == 0) return 1;
				else return 0;
			}
			lw = lw->Next();
		}
		// not found??
		return value;
	}
}

// draw a graph in the simulation graph widget?
bool XDevice::drawGraph()
{
	return maxO();
}

// returns if device should be shown in the simulation graph
bool XDevice::graphEnabled()
{
	return m_bGraphEnabled;
}

// show device in the simulation graph
void XDevice::enableGraph(bool bGraphEnabled)
{
	m_bGraphEnabled = bGraphEnabled;
}

// update positions of device-connections
void XDevice::erase(QPainter *p)
{
	if (m_poTextDev)
		m_poTextDev->erase(p);

	KlogicList<XWire> *l = m_oIncomingWires.First();
	while(l) {
		l->Get()->erase(p);
		l = l->Next();
	}
	l = m_oOutgoingWires.First();
	while(l) {
		l->Get()->erase(p);
		l = l->Next();
	}
	XObject::erase(p);
}

// garbage collection for connected wires
void XDevice::garbageCollection()
{	KlogicList<XWire> *lw;

	lw = m_oIncomingWires.First();
	while(lw) {
		lw->Get()->Wire::garbageCollection();
		lw = lw->Next();
	}
	lw = m_oOutgoingWires.First();
	while(lw) {
		lw->Get()->Wire::garbageCollection();
		lw = lw->Next();
	}
} 

// update nodes of wires which are connected to the device
void XDevice::updateWires()
{
	const QPoint& p1 = getPos();
	const QPoint& p2 = getOldPos();
	int dx = p1.x() - p2.x();
	int dy = p1.y() - p2.y();
	QPoint oNewPos(dx, dy);

	KlogicList<XWire> *lw = m_oIncomingWires.First();
	while(lw) {
		XWire *w = lw->Get();
		int idx = w->getInputIndex(this);
		w->setInputPosition(w->getInputPosition(idx) + oNewPos, idx);
		lw = lw->Next();
	}
	lw = m_oOutgoingWires.First();
	while(lw) {
		XWire *w = lw->Get();
		int idx = w->getOutputIndex(this);
		if (XWire::WFAIL == idx)
			fatal("XDevice::updateWires() this is unknown in a connected wire !?");
		w->setOutputPosition(w->getOutputPosition(idx) + oNewPos, idx);
		lw = lw->Next();
	}
}

// checks, if actice-node of wire is input or output
// connects if given, or disconnects if not given
// wire must be locked!
int XDevice::checkConnection(XWire *poWire, bool bInvert)
{	int found;
	int ret1, ret2;

	// active node of poWire must be the first or last node of poWire
	if (!poWire->activeIsEnd()) return DOK;

	// check, if wire is already input
	if (m_oIncomingWires.Get(poWire)) found = 1;
	else found = 0;
	// new connected, connection lost or not connected?
	ret1 = checkInput(poWire, found, bInvert);
	if (ret1 != DOK && ret1 != DDISC) return ret1;

	// check, if wire is already output
	if (m_oOutgoingWires.Get(poWire)) found = 1;
	else found = 0;
	// new connected, connection lost or not connected?
	ret2 = checkOutput(poWire, found, bInvert);
	if (ret2 != DOK && ret2 != DDISC) return ret2;

	if (ret1 == DDISC || ret2 == DDISC) return DDISC;
	return DOK;
}

int XDevice::checkInput(XWire *poWire, bool bFound, bool bInvert)
{
	if (!bFound) {
		QPoint oActiveNode = poWire->getActive();

		// check input connection
		if (!m_oInputRegion.contains(oActiveNode)) return DOK;

		// connections in all points allowed
		if (!hasNamedInput()) {
			return connectInput(poWire, oActiveNode, bInvert);
		} else {
			// connections only in some points allowed (named inputs)
			for (ConnectorList::iterator iter = m_oNamedInput.begin(); iter != m_oNamedInput.end(); iter++) {
				int iNewID = (*iter).getConnectorID();
				int iPosition = (*iter).getPosition();
				QPoint oAllowedPt(getPos().x(), iPosition * Grid::GRID + getPos().y() + getVOffset());
				if (oAllowedPt == oActiveNode)
					return connectInput(poWire, oActiveNode, bInvert, iNewID);
			}
		}
	} else {
		// check input disconnection
		int idx = poWire->getInputIndex(this);
		QPoint oActiveNode = poWire->getInputPosition(idx);

		if (!hasNamedInput()) {
			if (!m_oInputRegion.contains(oActiveNode)) {
				poWire->disconnectInput(idx);
				return DDISC;
			}
		}
		else {
			int iNewID = 0;
			bool bIsConnected = false;
			ConnectorList::iterator iter = m_oNamedInput.begin();
			while (!bIsConnected && iter != m_oNamedInput.end()) {
				iNewID = (*iter).getConnectorID();
				int iPosition = (*iter).getPosition();
				QPoint oAllowedPt(getPos().x(), iPosition * Grid::GRID + getPos().y() + getVOffset());
				if (oAllowedPt == oActiveNode)
					bIsConnected = true;
				iter++;
			}
			if (!bIsConnected) {
				poWire->disconnectInput(idx);
				return DDISC;
			} else {
				int iConnectionID = poWire->getConnectionID(this);
				if (iConnectionID != iNewID) {
					poWire->disconnectInput(idx);
					connectInput(poWire, oActiveNode, bInvert, iNewID);
				}
			}
		}
	}
	return DOK;
}

// give wire the information to be an input to this device
// input-id: id of the named input of this device
int XDevice::connectInput(XWire *poWire, QPoint oActiveNode, bool bInvert, int iInputID)
{
	if (XWire::WFAIL != poWire->activeIsInput()) return DOK;

	// check that another input is allowed
	if (maxI() <= currI())
		return DFAILMAXI;

	if (XWire::WFAIL == poWire->connectInput(this, bInvert, iInputID))
		return DFAIL;

	// set the position of the input-node of the wire exactly to the connection-point of this device
	oActiveNode.setX(getPos().x());
	poWire->setInputPosition(oActiveNode);

	return DCONN;
}

int XDevice::checkOutput(XWire *poWire, bool bFound, bool bInvert)
{
	if (!bFound) {
		QPoint oActiveNode = poWire->getActive();
		// check output connection
		if (!m_oOutputRegion.contains(oActiveNode)) {
			// fSWI became smaller .. workaround in order to load old files!
			if (type() == DeviceType::fSWI && Device::IMPORT_IGNORE_GLOBAL) {
				QRect oOutputRegion(m_oOutputRegion);
				oOutputRegion.setWidth(oOutputRegion.width() + 10);
				if (!oOutputRegion.contains(oActiveNode)) {
					oOutputRegion.setHeight(oOutputRegion.height() + 10);
					if (!oOutputRegion.contains(oActiveNode)) {
						return DOK;
					} else {
						oActiveNode.setY(oActiveNode.y() - 10);
						poWire->updateNode(oActiveNode);
					}
				} else {
					oActiveNode.setX(oActiveNode.y() - 10);
					poWire->updateNode(oActiveNode);
				}
			} else {
				return DOK;
			}
		}

		if (!hasNamedOutput()) {
			if (XWire::WFAIL != poWire->activeIsOutput()) return DOK;
			// if the wire is not connected already, connect it now
			return connectOutput(poWire, oActiveNode, bInvert);
		}
		// connections in some points allowed only
		else {
			ConnectorList::iterator iter = m_oNamedOutput.begin();
			iter++;
			while (iter != m_oNamedOutput.end()) {
				const Connector& oConnector = *iter;
				const Calculator *poCalculator = *oConnector;
				if (!poCalculator->isInternal()) {
					int iNewID = oConnector.getConnectorID();
					const QRect& oGeo = getGeometry();
					QPoint oAllowedPt(oGeo.x() + oGeo.width() - 1,
						oConnector.getPosition() * Grid::GRID + oGeo.y() + getVOffset());
					if (oAllowedPt == oActiveNode)
						return connectOutput(poWire, oActiveNode, bInvert, iNewID);
				}
				iter++;
			}
		}
	}
	// check output disconnection
	// if wire is connected, disconnect it now
	else {
		int idx = poWire->getOutputIndex(this);
		QPoint oActiveNode = poWire->getOutputPosition(idx);
		if (!hasNamedOutput()) {
			if (!m_oOutputRegion.contains(oActiveNode)) {
				poWire->disconnectOutput(idx);
				return DDISC;
			}
		}
		else {
			bool bIsConnected = false;
			int iNewID = 0;
			ConnectorList::iterator iter = m_oNamedOutput.begin();
			iter++;
			while (iter != m_oNamedOutput.end() && !bIsConnected) {
				const Connector& oConnector = *iter;
				const Calculator *poCalculator = *oConnector;
				if (!poCalculator->isInternal()) {
					iNewID = oConnector.getConnectorID();
					const QRect& oGeo = getGeometry();
					QPoint oAllowedPt(oGeo.x() + oGeo.width() - 1,
						oConnector.getPosition() * Grid::GRID + oGeo.y() + getVOffset());
					if (oAllowedPt == oActiveNode)
						bIsConnected = true;
				}
				iter++;
			}
			if (!bIsConnected) {
				poWire->disconnectOutput(idx);
				return DDISC;
			} else {
				int iConnectionID = poWire->getConnectionID(this);
				if (iConnectionID != iNewID) {
					poWire->disconnectOutput(idx);
					return connectOutput(poWire, oActiveNode, bInvert, iNewID); 
				}
			}
		}
	}
	return DOK;
}

// give wire the information to get input from this (output-)device
// output-id: id of the named output of this device
int XDevice::connectOutput(XWire *poWire, QPoint oPos, bool bInvert, int iOutputID)
{
	if (XWire::WFAIL != poWire->activeIsInput())
		return DOK;

	// check that another output is allowed
	if (maxO() <= currO())
		return DFAILMAXO;

	int ret = poWire->connectOutput(this, bInvert, iOutputID);
	if (ret != WOK)
		return ret;

	oPos.setX(getGeometry().x() + getGeometry().width() - 1);
	poWire->setOutputPosition(oPos);

	return DCONN;
}

// called by XWire
// returns invertation of another wire connected in the same point
int XDevice::inputIsInvertedAt(QPoint conn_pt)
{
	KlogicList<XWire> *lw = m_oIncomingWires.First();
	while(lw) {
		if (conn_pt == lw->Get()->getInputPosition(0) || conn_pt == lw->Get()->getInputPosition(1)) {
			return lw->Get()->inputIsInverted(this);
		}
		lw = lw->Next();
	}
	return 0;
}

// called by XWire
// add wire to input-wire-list
// draw wire-connection into image of device
void XDevice::addInputWire(XWire *w)
{
	m_oIncomingWires.Append(w);
}

// called by XWire
// remove wire from input-wire-list
void XDevice::removeInputWire(XWire *w)
{
	m_oIncomingWires.Destroy(w);
}

// called by XWire
// returns invertation of another wire connected in the same point
int XDevice::outputIsInvertedAt(QPoint conn_pt)
{
	KlogicList<XWire> *lw = m_oOutgoingWires.First();
	while(lw) {
		if (conn_pt == lw->Get()->getOutputPosition(0) || conn_pt == lw->Get()->getOutputPosition(1)) {
			return lw->Get()->outputIsInverted(this);
		}
		lw = lw->Next();
	}
	return 0;
}

// called by XWire
// add wire to output-wire-list
void XDevice::addOutputWire(XWire *w)
{
	m_oOutgoingWires.Append(w);
}

// called by XWire
// remove wire from output-wire-list
void XDevice::removeOutputWire(XWire *w)
{
	m_oOutgoingWires.Destroy(w);
}

// export method
// get tristate control input position
int XDevice::getTristateControlPos()
{
	const Connector *poConnector = m_oNamedInput.find(m_iTristateEnableID);
	if (poConnector) {
		return poConnector->getPosition();
	}
	return 0;
}

void XDevice::setTristateControlPos(int pos)
{
	Device::setTristate(true);
	m_iTristateImportPos = pos;
}

void XDevice::setUndefinedValue(int iUndef)
{
	Device::setUndefinedValue(iUndef);
}

// private, used for import
// shift existing inputs around that the tristate input "EN" can be inserted
void XDevice::prepareTristateInputImport()
{
	// look if given entry is free
	bool moved = false;
	for (ConnectorList::iterator iter = m_oNamedInput.begin(); iter != m_oNamedInput.end(); iter++) {
		Connector& oConnector = *iter;
		if (oConnector.getPosition() >= m_iTristateImportPos) {
			if (!moved) {
				moved = true;
				m_iTristateInputMoved++;
			}
			oConnector.setPosition(oConnector.getPosition() + 1);
		}
	}
}

// import method
void XDevice::setTristateInputImport()
{
	const Connector *poConnector = m_oNamedInput.find("EN");
	if (poConnector) {
		m_iTristateEnableID = poConnector->getConnectorID();
	} else {
		setSize(neededSize() + 1);
		prepareTristateInputImport();	// shift input
		m_iTristateEnableID = addInputName("EN", m_iTristateImportPos);
	}
}

// called by device-net(-device)
// add an possible named input (no connection!)
int XDevice::addInputName(const QString& sName, int iPosition, int iDeviceID)
{
	bool bExist = false;
	int iTakePosition = iPosition;
	bool bSearch = (iPosition < 0);

	if (bSearch)
		iTakePosition = 0;

	// search for a free entry
	int iMaxPosition = size();
	do {
		bExist = false;
		for (ConnectorList::iterator iter = m_oNamedInput.begin(); iter != m_oNamedInput.end(); iter++) {
			const Connector& oConnector = *iter;
			if (oConnector.getPosition() == iTakePosition)
				bExist = true;
		}
		if (bExist)
			iTakePosition++;
	} while (bSearch && bExist && iTakePosition < iMaxPosition);

	if (bExist)
		return -1;

	// add input
	int iInputID = Device::addInputName(sName, iTakePosition, iDeviceID);
	if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
	return iInputID;
}

// change an input name
// if the input does not exist, create a new named input
int XDevice::changeInputName(int iInputID, const QString& sName)
{
	if (!iInputID || !m_oNamedInput.find(iInputID))
		iInputID = addInputName(sName);
	else
		Device::changeInputName(iInputID, sName);
	if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
	return iInputID;
}

// remove a named input
void XDevice::removeInputName(int iInputID)
{	KlogicList<XWire> *lw = m_oIncomingWires.First();

	// remove all connections to this input
	while(lw) {
		if (lw->Get()->isNamedInput(this, iInputID)) {
			int idx = lw->Get()->getInputIndex(this);
			lw->Get()->disconnectInput(idx);
			lw = m_oIncomingWires.First();
		} else lw = lw->Next();
	}

	// remove input name
	Device::removeInputName(iInputID);
	if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
}

// called by device-net(-device) and equation-device properties
// add an possible named output (no connection!)
int XDevice::addOutputName(const QString& sName, int iPosition, int iDeviceID)
{
	bool bExist = false;
	int iTakePosition = iPosition;
	bool bSearch = (iPosition < 0);

	if (bSearch)
		iTakePosition = 0;

	// ignore ...
	if (!sName.length() || !sName.compare("(null)"))
		return 0;

	// try to find an empty entry
	int iMaxPosition = size();
	do {
		bExist = false;
		ConnectorList::iterator iter = m_oNamedOutput.begin();
		while (iter != m_oNamedOutput.end()) {
			if (!(**iter)->isInternal() && (*iter).getPosition() == iTakePosition)
				bExist = true;
			iter++;
		}
		if (bExist)
			iTakePosition++;
	} while (bSearch && bExist && (iTakePosition < iMaxPosition));

	if (bExist)
		return -1;

	// add output
	int iOutputID = Device::addOutputName(sName, iTakePosition, iDeviceID);
	if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
	return iOutputID;
}

int XDevice::changeOutputName(int iOutputID, const QString& sName)
{
	if (!iOutputID || !m_oNamedOutput[iOutputID])
		iOutputID = addOutputName(sName);
	else
		Device::changeOutputName(iOutputID, sName);
	if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
	return iOutputID;
}

// called by device-net(-device) and equation-devices properties
// remove output name
void XDevice::removeOutputName(int iOutputID)
{	KlogicList<XWire> *lw = m_oOutgoingWires.First();

	// remove all connections to this output
	while(lw) {
		if (lw->Get()->isNamedOutput(this, iOutputID)) {
			int idx = lw->Get()->getOutputIndex(this);
			if (XWire::WFAIL == idx)
				fatal("XDevice::removeOutputName() this is unknown in a connected wire !?");
			lw->Get()->disconnectOutput(idx);
			lw = m_oOutgoingWires.First();
		} else lw = lw->Next();
	}

	// remove output name
	Device::removeOutputName(iOutputID);
	if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
}

KlogicList<XWire> * XDevice::getIRef()
{
	return &m_oIncomingWires;
}

KlogicList<XWire> * XDevice::getORef()
{
	return &m_oOutgoingWires;
}

// get device image from image cache
// if not inside cache, create a new one
void XDevice::setImage()
{
	setImageKey();
	const Image *poCachedImage = m_oImageCache.getImage(m_oImageKey);
	if (poCachedImage) {
		XObject::setImage(poCachedImage);
		if (m_poTextDev)
			m_poTextDev->setImage();
		return;
	}

	Image *poImage = createImage(m_oImageKey);
	m_oImageCache.addImage(m_oImageKey, poImage);
	XObject::setImage(poImage);

	if (m_poTextDev)
		m_poTextDev->setImage();
}

Image* XDevice::createImage(const ImageKey&)
{
	warning("XDevice::createImage called - overload XDevice::setImage or XDevice::createImage !!");
	exit(0);
	return 0;
}

// absolute move of device
bool XDevice::moveTo(const QPoint& newpos)
{
	QPoint oOldPos = getPos();
	bool ret = XObject::moveTo(newpos);
	if (ret) {
		if (m_poTextDev) {
			QPoint oDelta = getPos() - oOldPos;
			m_poTextDev->moveBy(oDelta);
		}
		updateWires();
	}
	return ret;
}

// relative move of device
void XDevice::moveBy(const QPoint& newpos)
{
	XObject::moveBy(newpos);
	if (m_poTextDev)
		m_poTextDev->moveBy(newpos);
	updateWires();
}

// draw device to the screen
void XDevice::drawImage(QPaintDevice *paintd, QPainter *p)
{
	XObject::drawImage(paintd, p);
	if (m_poTextDev)
		m_poTextDev->drawImage(paintd, p);

	KlogicList<XWire> *lw = m_oIncomingWires.First();
	while(lw) {
		XWire *w = lw->Get();
		int idx = w->getInputIndex(this);
		w->drawConnections(p, Wire::INPUT, idx);
		lw = lw->Next();
	}
	lw = m_oOutgoingWires.First();
	while(lw) {
		XWire *w = lw->Get();
		int idx = w->getOutputIndex(this);
		w->drawConnections(p, Wire::OUTPUT, idx);
		lw = lw->Next();
	}
}

// draw device as lines and text to a printer device
void XDevice::drawImage(QPainter *p)
{
	const Image *poImage = getImage();

	// position
	int x = getPos().x();
	int y = getPos().y();

	x += XObject::IOREG;
	y -= getVOffset() / 2;

	// draw device rect
	QRect r;
	r.setX(x);
	r.setY(y);
	r.setWidth(poImage->width() - (XObject::IOREG * 2) - 1);
	r.setHeight(poImage->height());
	p->setPen(Qt::black);
	if (type() != DeviceType::fPWR && type() != DeviceType::fSWI) {
		p->drawRect(r);
	}
	printImage(p, r);
}

void XDevice::printImage(QPainter *p, const QRect& r)
{
	int picwidth = getImage()->width();
	// draw io-names and type specific symbol
	switch(type()) {
		case DeviceType::fAND:
			p->setFont( QFont( "helvetica", 16, QFont::Bold ) );
			p->drawText(r.x() + (picwidth / 2) - 8, r.y() + (r.height() / 2) + 5, "&");
			break;
		case DeviceType::fOR:
			p->setFont( QFont( "helvetica", 16, QFont::Bold ) );
			p->drawText(r.x() + (picwidth / 2) - 8, r.y() + (r.height() / 2) + 1, ">");
			p->drawText(r.x() + (picwidth / 2) - 8, r.y() + (r.height() / 2) + 1, "_");
			break;
		case DeviceType::fXOR:
			p->setFont( QFont( "helvetica", 16, QFont::Bold ) );
			p->drawText(r.x() + (picwidth / 2) - 8, r.y() + (r.height() / 2) + 3, "=");
			break;
		case DeviceType::fONE:
			p->setFont( QFont( "helvetica", 16, QFont::Bold ) );
			p->drawText(r.x() + (picwidth / 2) - 8, r.y() + (r.height() / 2) + 5, "1");
			break;
		default:
			break;
	}
}

void XDevice::init()
{
	switch (type()) {
		case DeviceType::fONE:
			deviceOutputInverted = true;
			setSize(Global::Device::MINSIZE);
			deviceMaxInput = 1;
			break;
	}

	if (!Device::IMPORT_IGNORE_GLOBAL) setImage();
	setEquation();
	parseEquation();
	if (Global::Device::isDisplayNames() && type() != DeviceType::fTXT)
		displayName(true);
}

