#!/usr/bin/perl

# $Id: hypua2jamo.pl,v 1.2 2007/07/22 04:12:15 karnes Exp $
#
### Copyrihgt (C) 2003-2007 Dohyun Kim <nomos at ktug.or.kr>
###
### -------------------------------------------------------------------
### part of ko.TeX
###
### This work may be distributed and/or modified under the
### conditions of the LaTeX Project Public License, either version 1.3c
### of this license or (at your option) any later version.
### The latest version of this license is in
###  http://www.latex-project.org/lppl.txt
### and version 1.3c or later is part of all distributions of LaTeX
### version 2006/05/20 or later.
### -------------------------------------------------------------------

#use strict;
use Getopt::Std;

my %args;
getopts('sxctah',\%args);
if($args{a}) {
    $args{s} = $args{x} = $args{c} = $args{t} = 1;
}

if($args{h}) {
    print STDERR
    "Usage: $0 [-s] [-x] [-c] [-t] [-a] < IN_FILE > OUT_FILE\n",
    "       $0 -h\n\n",
    "Translate Hanyang PUA pre-1933 orthography Korean Syllables to Hangul Jamo\n\n",
    "  -s  also translate Hangul Syllables to Hangul Jamo\n",
    "  -x  do not translate TeX Control Sequences\n",
    "  -c  also translate Hangul Compatibility Jamo to Hangul Jamo\n",
    "  -t  also translate U+00B7 or U+003A to Hangul Tone Mark\n",
    "  -a  same as `-s -x -c -t'\n",
    "  -h  print this message\n";
    exit 1;
}

my @cjamo_to_jamo = qw/
    1100 1101 11aa 1102 11ac 11ad 1103 1104
    1105 11b0 11b1 11b2 11b3 11b4 11b5 111a
    1106 1107 1108 1121 1109 110a 110b 110c
    110d 110e 110f 1110 1111 1112 1161 1162
    1163 1164 1165 1166 1167 1168 1169 116a
    116b 116c 116d 116e 116f 1170 1171 1172
    1173 1174 1175 1160 1114 1115 11c7 11c8
    11cc 11ce 11d3 11d7 11d9 111c 11dd 11df
    111d 111e 1120 1122 1123 1127 1129 112b
    112c 112d 112e 112f 1132 1136 1140 1147
    114c 11f1 11f2 1157 1158 1159 1184 1185
    1188 1191 1192 1194 119e 11a1
/;

binmode STDIN,":utf8";
binmode STDOUT,":utf8";

while(<>) {
    if($args{x} and /\\[^\s{\[]+/) {
	my(@cs,@noncs,$i);
	while(/(\\[^\s{\[]+)/gc) {
	    push @cs,$1;
	}
	for(@noncs = split /\\[^\s{\[]+/,$_) {
	    &to_jamo_main;
	}
	for($i=0;$i<@noncs;$i++) {
	    print $noncs[$i];
	    print $cs[$i] if $i < @cs;
	}
    }
    else {
	&to_jamo_main;
	print;
    }
}

sub to_jamo_main {
    &insertfiller;
    &tonemark if $args{t};
    &pua2jamo;
    &syllable2jamo if $args{s};
    &cjamo2jamo if $args{c};
}

sub insertfiller {
    # intentionally disregard `jamo + syllable' or
    # `syllable + jamo' in spite of unicode standard!
    s/([\x{1100}-\x{115F}])([\x{11A8}-\x{11F9}])/$1\x{1160}\x{115F}\x{1160}$2/g;
    s/([\x{1100}-\x{115F}])([^\x{1100}-\x{11A2}])/$1\x{1160}$2/g;
    s/([^\x{1100}-\x{11A2}])([\x{1160}-\x{11A2}])/$1\x{115F}$2/g;
    s/([^\x{1160}-\x{11A2}])([\x{11A8}-\x{11F9}])/$1\x{115F}\x{1160}$2/g;
    s/([\x{1100}-\x{115F}])$/$1\x{1160}/g;
    s/^([\x{1160}-\x{11A2}])/\x{115F}$1/;
    s/^([\x{11A8}-\x{11F9}])/\x{115F}\x{1160}$1/;
}

sub cjamo2jamo {
    s/([\x{3131}-\x{318E}])/cjamo2jamoreal($1)/ge;
}

sub cjamo2jamoreal {
    my $cj = ord shift;
    $cj = hex $cjamo_to_jamo[$cj - 0x3131];
    if($cj <= 0x115f) { ### choseong
	return chr($cj)."\x{1160}";
    }
    elsif($cj >= 0x1160 and $cj <= 0x11a2) { ### jungseong
	return "\x{115f}".chr($cj);
    }
    else { ### jongseong
	return "\x{115f}\x{1160}".chr($cj);
    }
}

sub syllable2jamo {
    s/([\x{AC00}-\x{D7A3}])/syllable2jamoreal($1)/ge;
}

sub syllable2jamoreal {
    my $syl = ord shift;
    my $cho = ($syl - 0xac00) / (21 * 28) + 0x1100;
    my $jung = (($syl - 0xac00) % (21 * 28)) / 28 + 0x1161;
    my $jong = ($syl - 0xac00) % 28 + 0x11a7;
    my $result = chr($cho).chr($jung);
    $result .= chr($jong) unless $jong == 0x11a7;
    return $result;
}

sub tonemark {
    s/\x{B7}([\x{1100}-\x{115F}]+[\x{1160}-\x{11A2}]+[\x{11A8}-\x{11F9}]*)/$1\x{302E}/g;
    s/\x{B7}([\x{F784}-\x{F800}][\x{F806}-\x{F864}][\x{F86A}-\x{F8F7}]?)/$1\x{302E}/g;
    s/\x{B7}([\x{AC00}-\x{D7A3}]|[\x{E0BC}-\x{F66E}])/$1\x{302E}/g;
    s/\x{3A}([\x{1100}-\x{115F}]+[\x{1160}-\x{11A2}]+[\x{11A8}-\x{11F9}]*)/$1\x{302F}/g;
    s/\x{3A}([\x{F784}-\x{F800}][\x{F806}-\x{F864}][\x{F86A}-\x{F8F7}]?)/$1\x{302F}/g;
    s/\x{3A}([\x{AC00}-\x{D7A3}]|[\x{E0BC}-\x{F66E}])/$1\x{302F}/g;
}

sub pua2jamo {
    s/\x{E0BC}/\x{115F}\x{1161}\x{11AE}/g;
    s/\x{E0BD}/\x{115F}\x{1161}\x{11AE}\x{11AE}/g;
    s/\x{E0BE}/\x{115F}\x{1161}\x{11AF}/g;
    s/\x{E0BF}/\x{115F}\x{1161}\x{11B0}/g;
    s/\x{E0C0}/\x{115F}\x{1161}\x{11B1}/g;
    s/\x{E0C1}/\x{115F}\x{1161}\x{11B2}/g;
    s/\x{E0C2}/\x{115F}\x{1161}\x{11BA}/g;
    s/\x{E0C3}/\x{115F}\x{1161}\x{11BB}/g;
    s/\x{E0C4}/\x{115F}\x{1161}\x{11BD}/g;
    s/\x{E0C5}/\x{115F}\x{1161}\x{11BD}\x{11BD}/g;
    s/\x{E0C6}/\x{115F}\x{1161}\x{1173}/g;
    s/\x{E0C7}/\x{115F}\x{1161}\x{1173}\x{11AE}/g;
    s/\x{E0C8}/\x{115F}\x{1163}\x{11AB}/g;
    s/\x{E0C9}/\x{115F}\x{1165}\x{11AF}/g;
    s/\x{E0CA}/\x{115F}\x{1165}\x{11B8}/g;
    s/\x{E0CB}/\x{115F}\x{116E}\x{11AB}/g;
    s/\x{E0CC}/\x{115F}\x{1175}\x{11A8}/g;
    s/\x{E0CD}/\x{115F}\x{1175}\x{11AB}/g;
    s/\x{E0CE}/\x{115F}\x{1175}\x{11AF}/g;
    s/\x{E0CF}/\x{115F}\x{1175}\x{11BA}/g;
    s/\x{E0D0}/\x{115F}\x{1175}\x{11BC}/g;
    s/\x{E0D1}/\x{115F}\x{119E}\x{11AF}/g;
    s/\x{E0D2}/\x{1100}\x{1161}\x{11AB}\x{11AB}/g;
    s/\x{E0D3}/\x{1100}\x{1161}\x{11C7}/g;
    s/\x{E0D4}/\x{1100}\x{1161}\x{11CA}/g;
    s/\x{E0D5}/\x{1100}\x{1161}\x{11CD}/g;
    s/\x{E0D6}/\x{1100}\x{1161}\x{11CE}/g;
    s/\x{E0D7}/\x{1100}\x{1161}\x{11D0}/g;
    s/\x{E0D8}/\x{1100}\x{1161}\x{11AF}\x{11F0}/g;
    s/\x{E0D9}/\x{1100}\x{1161}\x{11D9}/g;
    s/\x{E0DA}/\x{1100}\x{1161}\x{11DD}/g;
    s/\x{E0DB}/\x{1100}\x{1161}\x{11E1}/g;
    s/\x{E0DC}/\x{1100}\x{1161}\x{11E2}/g;
    s/\x{E0DD}/\x{1100}\x{1161}\x{11E4}/g;
    s/\x{E0DE}/\x{1100}\x{1161}\x{11E5}/g;
    s/\x{E0DF}/\x{1100}\x{1161}\x{11E6}/g;
    s/\x{E0E0}/\x{1100}\x{1161}\x{11E7}/g;
    s/\x{E0E1}/\x{1100}\x{1161}\x{11EA}/g;
    s/\x{E0E2}/\x{1100}\x{1161}\x{11BA}\x{11BE}/g;
    s/\x{E0E3}/\x{1100}\x{1161}\x{11BA}\x{11C0}/g;
    s/\x{E0E4}/\x{1100}\x{1161}\x{11EB}/g;
    s/\x{E0E5}/\x{1100}\x{1161}\x{11BC}\x{11BA}/g;
    s/\x{E0E6}/\x{1100}\x{1161}\x{11F0}/g;
    s/\x{E0E7}/\x{1100}\x{1161}\x{11F1}/g;
    s/\x{E0E8}/\x{1100}\x{1161}\x{11F9}/g;
    s/\x{E0E9}/\x{1100}\x{1176}/g;
    s/\x{E0EA}/\x{1100}\x{1177}/g;
    s/\x{E0EB}/\x{1100}\x{1162}\x{11CD}/g;
    s/\x{E0EC}/\x{1100}\x{1162}\x{11AF}\x{11F0}/g;
    s/\x{E0ED}/\x{1100}\x{1162}\x{11F0}/g;
    s/\x{E0EE}/\x{1100}\x{1163}\x{11E2}/g;
    s/\x{E0EF}/\x{1100}\x{1163}\x{11E6}/g;
    s/\x{E0F0}/\x{1100}\x{1163}\x{11F0}/g;
    s/\x{E0F1}/\x{1100}\x{1163}\x{11F9}/g;
    s/\x{E0F2}/\x{1100}\x{1178}/g;
    s/\x{E0F3}/\x{1100}\x{1165}\x{11C7}/g;
    s/\x{E0F4}/\x{1100}\x{1165}\x{11D0}/g;
    s/\x{E0F5}/\x{1100}\x{1165}\x{11D9}/g;
    s/\x{E0F6}/\x{1100}\x{1165}\x{11DD}/g;
    s/\x{E0F7}/\x{1100}\x{1165}\x{11E2}/g;
    s/\x{E0F8}/\x{1100}\x{1165}\x{11E3}/g;
    s/\x{E0F9}/\x{1100}\x{1165}\x{11E6}/g;
    s/\x{E0FA}/\x{1100}\x{1165}\x{11E7}/g;
    s/\x{E0FB}/\x{1100}\x{1165}\x{11BA}\x{11BE}/g;
    s/\x{E0FC}/\x{1100}\x{1165}\x{11BA}\x{11C2}/g;
    s/\x{E0FD}/\x{1100}\x{1165}\x{11EB}/g;
    s/\x{E0FE}/\x{1100}\x{1165}\x{11F0}/g;
    s/\x{E0FF}/\x{1100}\x{1165}\x{11F9}/g;
    s/\x{E100}/\x{1100}\x{117C}/g;
    s/\x{E101}/\x{1100}\x{117C}\x{11AB}/g;
    s/\x{E102}/\x{1100}\x{117C}\x{11AF}/g;
    s/\x{E103}/\x{1100}\x{117C}\x{11B7}/g;
    s/\x{E104}/\x{1100}\x{1166}\x{11F0}/g;
    s/\x{E105}/\x{1100}\x{1167}\x{11AE}\x{11C0}/g;
    s/\x{E106}/\x{1100}\x{1167}\x{11D9}/g;
    s/\x{E107}/\x{1100}\x{1167}\x{11E2}/g;
    s/\x{E108}/\x{1100}\x{1167}\x{11B8}\x{11BE}/g;
    s/\x{E109}/\x{1100}\x{1167}\x{11E7}/g;
    s/\x{E10A}/\x{1100}\x{1167}\x{11BA}\x{11BE}/g;
    s/\x{E10B}/\x{1100}\x{1167}\x{11BA}\x{11C0}/g;
    s/\x{E10C}/\x{1100}\x{1167}\x{11BA}\x{11C2}/g;
    s/\x{E10D}/\x{1100}\x{1167}\x{11EB}/g;
    s/\x{E10E}/\x{1100}\x{1167}\x{11F0}/g;
    s/\x{E10F}/\x{1100}\x{1167}\x{11F1}/g;
    s/\x{E110}/\x{1100}\x{1167}\x{11F9}/g;
    s/\x{E111}/\x{1100}\x{1168}\x{11F0}/g;
    s/\x{E112}/\x{1100}\x{1169}\x{11D0}/g;
    s/\x{E113}/\x{1100}\x{1169}\x{11B2}\x{11C1}/g;
    s/\x{E114}/\x{1100}\x{1169}\x{11D6}/g;
    s/\x{E115}/\x{1100}\x{1169}\x{11D9}/g;
    s/\x{E116}/\x{1100}\x{1169}\x{11DA}/g;
    s/\x{E117}/\x{1100}\x{1169}\x{11DD}/g;
    s/\x{E118}/\x{1100}\x{1169}\x{11E2}/g;
    s/\x{E119}/\x{1100}\x{1169}\x{11E6}/g;
    s/\x{E11A}/\x{1100}\x{1169}\x{11BA}\x{11BE}/g;
    s/\x{E11B}/\x{1100}\x{1169}\x{11EB}/g;
    s/\x{E11C}/\x{1100}\x{1169}\x{11F0}/g;
    s/\x{E11D}/\x{1100}\x{1169}\x{11F1}/g;
    s/\x{E11E}/\x{1100}\x{1169}\x{11F7}/g;
    s/\x{E11F}/\x{1100}\x{1169}\x{11F8}/g;
    s/\x{E120}/\x{1100}\x{116A}\x{11C7}/g;
    s/\x{E121}/\x{1100}\x{116A}\x{11D9}/g;
    s/\x{E122}/\x{1100}\x{116A}\x{11F0}/g;
    s/\x{E123}/\x{1100}\x{116A}\x{11F9}/g;
    s/\x{E124}/\x{1100}\x{1169}\x{1163}/g;
    s/\x{E125}/\x{1100}\x{1169}\x{1163}\x{11BC}/g;
    s/\x{E126}/\x{1100}\x{117F}/g;
    s/\x{E127}/\x{1100}\x{117F}\x{11AB}/g;
    s/\x{E128}/\x{1100}\x{117F}\x{11AF}/g;
    s/\x{E129}/\x{1100}\x{1182}/g;
    s/\x{E12A}/\x{1100}\x{1182}\x{11AF}/g;
    s/\x{E12B}/\x{1100}\x{116C}\x{11F0}/g;
    s/\x{E12C}/\x{1100}\x{116D}\x{11D9}/g;
    s/\x{E12D}/\x{1100}\x{116D}\x{11E2}/g;
    s/\x{E12E}/\x{1100}\x{116D}\x{11F0}/g;
    s/\x{E12F}/\x{1100}\x{116D}\x{1161}/g;
    s/\x{E130}/\x{1100}\x{116D}\x{1162}/g;
    s/\x{E131}/\x{1100}\x{116D}\x{1162}\x{11AF}/g;
    s/\x{E132}/\x{1100}\x{1184}/g;
    s/\x{E133}/\x{1100}\x{1185}/g;
    s/\x{E134}/\x{1100}\x{1185}\x{11AF}/g;
    s/\x{E135}/\x{1100}\x{1185}\x{11B0}/g;
    s/\x{E136}/\x{1100}\x{1188}/g;
    s/\x{E137}/\x{1100}\x{1188}\x{11AB}/g;
    s/\x{E138}/\x{1100}\x{116E}\x{11A8}\x{11BF}/g;
    s/\x{E139}/\x{1100}\x{116E}\x{11C7}/g;
    s/\x{E13A}/\x{1100}\x{116E}\x{11CD}/g;
    s/\x{E13B}/\x{1100}\x{116E}\x{11D0}/g;
    s/\x{E13C}/\x{1100}\x{116E}\x{11D9}/g;
    s/\x{E13D}/\x{1100}\x{116E}\x{11DA}/g;
    s/\x{E13E}/\x{1100}\x{116E}\x{11DD}/g;
    s/\x{E13F}/\x{1100}\x{116E}\x{11E2}/g;
    s/\x{E140}/\x{1100}\x{116E}\x{11E6}/g;
    s/\x{E141}/\x{1100}\x{116E}\x{11EC}/g;
    s/\x{E142}/\x{1100}\x{116E}\x{11F0}/g;
    s/\x{E143}/\x{1100}\x{116E}\x{11F0}\x{11A8}/g;
    s/\x{E144}/\x{1100}\x{116E}\x{11F6}/g;
    s/\x{E145}/\x{1100}\x{116E}\x{11F9}/g;
    s/\x{E146}/\x{1100}\x{1189}/g;
    s/\x{E147}/\x{1100}\x{1189}\x{11AB}/g;
    s/\x{E148}/\x{1100}\x{1189}\x{11B7}/g;
    s/\x{E149}/\x{1100}\x{1189}\x{11BC}/g;
    s/\x{E14A}/\x{1100}\x{118A}/g;
    s/\x{E14B}/\x{1100}\x{116F}\x{11C7}/g;
    s/\x{E14C}/\x{1100}\x{116F}\x{11D9}/g;
    s/\x{E14D}/\x{1100}\x{116F}\x{11F9}/g;
    s/\x{E14E}/\x{1100}\x{118B}/g;
    s/\x{E14F}/\x{1100}\x{118B}\x{11AB}/g;
    s/\x{E150}/\x{1100}\x{116E}\x{1167}/g;
    s/\x{E151}/\x{1100}\x{118C}/g;
    s/\x{E152}/\x{1100}\x{118D}/g;
    s/\x{E153}/\x{1100}\x{118D}\x{11AF}/g;
    s/\x{E154}/\x{1100}\x{1171}\x{11F0}/g;
    s/\x{E155}/\x{1100}\x{1171}\x{11F9}/g;
    s/\x{E156}/\x{1100}\x{1172}\x{11D9}/g;
    s/\x{E157}/\x{1100}\x{1172}\x{11E2}/g;
    s/\x{E158}/\x{1100}\x{1172}\x{11F0}/g;
    s/\x{E159}/\x{1100}\x{1172}\x{11F9}/g;
    s/\x{E15A}/\x{1100}\x{118E}/g;
    s/\x{E15B}/\x{1100}\x{118E}\x{11AB}/g;
    s/\x{E15C}/\x{1100}\x{118F}/g;
    s/\x{E15D}/\x{1100}\x{118F}\x{11AB}/g;
    s/\x{E15E}/\x{1100}\x{1191}/g;
    s/\x{E15F}/\x{1100}\x{1191}\x{11A8}/g;
    s/\x{E160}/\x{1100}\x{1191}\x{11AB}/g;
    s/\x{E161}/\x{1100}\x{1191}\x{11AF}/g;
    s/\x{E162}/\x{1100}\x{1191}\x{11D9}/g;
    s/\x{E163}/\x{1100}\x{1191}\x{11BC}/g;
    s/\x{E164}/\x{1100}\x{1191}\x{11F0}/g;
    s/\x{E165}/\x{1100}\x{1191}\x{11F9}/g;
    s/\x{E166}/\x{1100}\x{1192}/g;
    s/\x{E167}/\x{1100}\x{1192}\x{11AF}/g;
    s/\x{E168}/\x{1100}\x{1192}\x{11B0}/g;
    s/\x{E169}/\x{1100}\x{1192}\x{11BC}/g;
    s/\x{E16A}/\x{1100}\x{1194}/g;
    s/\x{E16B}/\x{1100}\x{1194}\x{11A8}/g;
    s/\x{E16C}/\x{1100}\x{1194}\x{11AF}/g;
    s/\x{E16D}/\x{1100}\x{1194}\x{11D9}/g;
    s/\x{E16E}/\x{1100}\x{1194}\x{11BC}/g;
    s/\x{E16F}/\x{1100}\x{1194}\x{11F0}/g;
    s/\x{E170}/\x{1100}\x{1173}\x{11C7}/g;
    s/\x{E171}/\x{1100}\x{1173}\x{11AB}\x{11BE}/g;
    s/\x{E172}/\x{1100}\x{1173}\x{11C9}/g;
    s/\x{E173}/\x{1100}\x{1173}\x{11AE}\x{11BE}/g;
    s/\x{E174}/\x{1100}\x{1173}\x{11AE}\x{11C0}/g;
    s/\x{E175}/\x{1100}\x{1173}\x{11CD}/g;
    s/\x{E176}/\x{1100}\x{1173}\x{11D0}/g;
    s/\x{E177}/\x{1100}\x{1173}\x{11D9}/g;
    s/\x{E178}/\x{1100}\x{1173}\x{11DA}/g;
    s/\x{E179}/\x{1100}\x{1173}\x{11DD}/g;
    s/\x{E17A}/\x{1100}\x{1173}\x{11DF}/g;
    s/\x{E17B}/\x{1100}\x{1173}\x{11E1}/g;
    s/\x{E17C}/\x{1100}\x{1173}\x{11E2}/g;
    s/\x{E17D}/\x{1100}\x{1173}\x{11BA}\x{11BE}/g;
    s/\x{E17E}/\x{1100}\x{1173}\x{11EB}/g;
    s/\x{E17F}/\x{1100}\x{1173}\x{11F0}/g;
    s/\x{E180}/\x{1100}\x{1173}\x{11F9}/g;
    s/\x{E181}/\x{1100}\x{1173}\x{1165}/g;
    s/\x{E182}/\x{1100}\x{1173}\x{1166}/g;
    s/\x{E183}/\x{1100}\x{1195}/g;
    s/\x{E184}/\x{1100}\x{1196}/g;
    s/\x{E185}/\x{1100}\x{1174}\x{11D9}/g;
    s/\x{E186}/\x{1100}\x{1174}\x{11F9}/g;
    s/\x{E187}/\x{1100}\x{1175}\x{11CA}/g;
    s/\x{E188}/\x{1100}\x{1175}\x{11AE}\x{11AE}/g;
    s/\x{E189}/\x{1100}\x{1175}\x{11CD}/g;
    s/\x{E18A}/\x{1100}\x{1175}\x{11D0}/g;
    s/\x{E18B}/\x{1100}\x{1175}\x{11D9}/g;
    s/\x{E18C}/\x{1100}\x{1175}\x{11D9}\x{11C2}/g;
    s/\x{E18D}/\x{1100}\x{1175}\x{11E0}/g;
    s/\x{E18E}/\x{1100}\x{1175}\x{11E2}/g;
    s/\x{E18F}/\x{1100}\x{1175}\x{11B8}\x{11B8}/g;
    s/\x{E190}/\x{1100}\x{1175}\x{11B8}\x{11BD}/g;
    s/\x{E191}/\x{1100}\x{1175}\x{11E4}/g;
    s/\x{E192}/\x{1100}\x{1175}\x{11E5}/g;
    s/\x{E193}/\x{1100}\x{1175}\x{11E6}/g;
    s/\x{E194}/\x{1100}\x{1175}\x{11E7}/g;
    s/\x{E195}/\x{1100}\x{1175}\x{11BA}\x{11E7}/g;
    s/\x{E196}/\x{1100}\x{1175}\x{11BA}\x{11BE}/g;
    s/\x{E197}/\x{1100}\x{1175}\x{11EB}/g;
    s/\x{E198}/\x{1100}\x{1175}\x{11F0}/g;
    s/\x{E199}/\x{1100}\x{1175}\x{11F9}/g;
    s/\x{E19A}/\x{1100}\x{1198}/g;
    s/\x{E19B}/\x{1100}\x{1199}/g;
    s/\x{E19C}/\x{1100}\x{1199}\x{11AB}/g;
    s/\x{E19D}/\x{1100}\x{1199}\x{11BC}/g;
    s/\x{E19E}/\x{1100}\x{1175}\x{1167}/g;
    s/\x{E19F}/\x{1100}\x{1175}\x{1168}/g;
    s/\x{E1A0}/\x{1100}\x{1175}\x{116D}/g;
    s/\x{E1A1}/\x{1100}\x{119B}/g;
    s/\x{E1A2}/\x{1100}\x{1175}\x{1172}/g;
    s/\x{E1A3}/\x{1100}\x{1175}\x{1172}\x{11AB}/g;
    s/\x{E1A4}/\x{1100}\x{119C}/g;
    s/\x{E1A5}/\x{1100}\x{1175}\x{1175}/g;
    s/\x{E1A6}/\x{1100}\x{119D}/g;
    s/\x{E1A7}/\x{1100}\x{119E}/g;
    s/\x{E1A8}/\x{1100}\x{119E}\x{11A8}/g;
    s/\x{E1A9}/\x{1100}\x{119E}\x{11A9}/g;
    s/\x{E1AA}/\x{1100}\x{119E}\x{11AB}/g;
    s/\x{E1AB}/\x{1100}\x{119E}\x{11AE}/g;
    s/\x{E1AC}/\x{1100}\x{119E}\x{11AE}\x{11B8}/g;
    s/\x{E1AD}/\x{1100}\x{119E}\x{11AF}/g;
    s/\x{E1AE}/\x{1100}\x{119E}\x{11B0}/g;
    s/\x{E1AF}/\x{1100}\x{119E}\x{11CD}/g;
    s/\x{E1B0}/\x{1100}\x{119E}\x{11D0}/g;
    s/\x{E1B1}/\x{1100}\x{119E}\x{11B1}/g;
    s/\x{E1B2}/\x{1100}\x{119E}\x{11B2}/g;
    s/\x{E1B3}/\x{1100}\x{119E}\x{11D3}/g;
    s/\x{E1B4}/\x{1100}\x{119E}\x{11B2}\x{11C1}/g;
    s/\x{E1B5}/\x{1100}\x{119E}\x{11D5}/g;
    s/\x{E1B6}/\x{1100}\x{119E}\x{11B3}/g;
    s/\x{E1B7}/\x{1100}\x{119E}\x{11B5}/g;
    s/\x{E1B8}/\x{1100}\x{119E}\x{11B6}/g;
    s/\x{E1B9}/\x{1100}\x{119E}\x{11D9}/g;
    s/\x{E1BA}/\x{1100}\x{119E}\x{11B7}/g;
    s/\x{E1BB}/\x{1100}\x{119E}\x{11E2}/g;
    s/\x{E1BC}/\x{1100}\x{119E}\x{11B8}/g;
    s/\x{E1BD}/\x{1100}\x{119E}\x{11E3}/g;
    s/\x{E1BE}/\x{1100}\x{119E}\x{11E6}/g;
    s/\x{E1BF}/\x{1100}\x{119E}\x{11BA}/g;
    s/\x{E1C0}/\x{1100}\x{119E}\x{11E7}/g;
    s/\x{E1C1}/\x{1100}\x{119E}\x{11EA}/g;
    s/\x{E1C2}/\x{1100}\x{119E}\x{11EB}/g;
    s/\x{E1C3}/\x{1100}\x{119E}\x{11BC}/g;
    s/\x{E1C4}/\x{1100}\x{119E}\x{11F0}/g;
    s/\x{E1C5}/\x{1100}\x{119E}\x{11F1}/g;
    s/\x{E1C6}/\x{1100}\x{119E}\x{11BD}/g;
    s/\x{E1C7}/\x{1100}\x{119E}\x{11BE}/g;
    s/\x{E1C8}/\x{1100}\x{119E}\x{11C0}/g;
    s/\x{E1C9}/\x{1100}\x{119E}\x{11C2}/g;
    s/\x{E1CA}/\x{1100}\x{119F}/g;
    s/\x{E1CB}/\x{1100}\x{119F}\x{11AB}/g;
    s/\x{E1CC}/\x{1100}\x{119F}\x{11BC}/g;
    s/\x{E1CD}/\x{1100}\x{11A1}/g;
    s/\x{E1CE}/\x{1100}\x{11A1}\x{11A8}/g;
    s/\x{E1CF}/\x{1100}\x{11A1}\x{11AB}/g;
    s/\x{E1D0}/\x{1100}\x{11A1}\x{11AF}/g;
    s/\x{E1D1}/\x{1100}\x{11A1}\x{11B7}/g;
    s/\x{E1D2}/\x{1100}\x{11A1}\x{11B8}/g;
    s/\x{E1D3}/\x{1100}\x{11A1}\x{11BA}/g;
    s/\x{E1D4}/\x{1100}\x{11A1}\x{11BC}/g;
    s/\x{E1D5}/\x{1100}\x{11A1}\x{11F0}/g;
    s/\x{E1D6}/\x{1100}\x{11A1}\x{11F1}/g;
    s/\x{E1D7}/\x{1101}\x{1161}\x{11F0}/g;
    s/\x{E1D8}/\x{1101}\x{1176}/g;
    s/\x{E1D9}/\x{1101}\x{1163}\x{11E2}/g;
    s/\x{E1DA}/\x{1101}\x{1165}\x{11D9}/g;
    s/\x{E1DB}/\x{1101}\x{1165}\x{11E2}/g;
    s/\x{E1DC}/\x{1101}\x{1165}\x{11E6}/g;
    s/\x{E1DD}/\x{1101}\x{1165}\x{11EC}/g;
    s/\x{E1DE}/\x{1101}\x{1165}\x{11ED}/g;
    s/\x{E1DF}/\x{1101}\x{117B}/g;
    s/\x{E1E0}/\x{1101}\x{1167}\x{11F0}/g;
    s/\x{E1E1}/\x{1101}\x{1169}\x{11E2}/g;
    s/\x{E1E2}/\x{1101}\x{1169}\x{11F0}/g;
    s/\x{E1E3}/\x{1101}\x{116A}\x{11F0}/g;
    s/\x{E1E4}/\x{1101}\x{116D}\x{11E2}/g;
    s/\x{E1E5}/\x{1101}\x{116D}\x{11F0}/g;
    s/\x{E1E6}/\x{1101}\x{1188}/g;
    s/\x{E1E7}/\x{1101}\x{116E}\x{11D9}/g;
    s/\x{E1E8}/\x{1101}\x{116E}\x{11DA}/g;
    s/\x{E1E9}/\x{1101}\x{116E}\x{11E2}/g;
    s/\x{E1EA}/\x{1101}\x{116E}\x{11F0}/g;
    s/\x{E1EB}/\x{1101}\x{1189}/g;
    s/\x{E1EC}/\x{1101}\x{116F}\x{11D9}/g;
    s/\x{E1ED}/\x{1101}\x{116F}\x{11F0}/g;
    s/\x{E1EE}/\x{1101}\x{116E}\x{1167}/g;
    s/\x{E1EF}/\x{1101}\x{118C}/g;
    s/\x{E1F0}/\x{1101}\x{1172}\x{11D9}/g;
    s/\x{E1F1}/\x{1101}\x{1172}\x{11E2}/g;
    s/\x{E1F2}/\x{1101}\x{1172}\x{11F0}/g;
    s/\x{E1F3}/\x{1101}\x{1191}/g;
    s/\x{E1F4}/\x{1101}\x{1191}\x{11AB}/g;
    s/\x{E1F5}/\x{1101}\x{1191}\x{11BC}/g;
    s/\x{E1F6}/\x{1101}\x{1191}\x{11F0}/g;
    s/\x{E1F7}/\x{1101}\x{1191}\x{11F9}/g;
    s/\x{E1F8}/\x{1101}\x{1194}/g;
    s/\x{E1F9}/\x{1101}\x{1194}\x{11BC}/g;
    s/\x{E1FA}/\x{1101}\x{1173}\x{11D9}/g;
    s/\x{E1FB}/\x{1101}\x{1173}\x{11F0}/g;
    s/\x{E1FC}/\x{1101}\x{1173}\x{11F6}/g;
    s/\x{E1FD}/\x{1101}\x{1174}\x{11F0}/g;
    s/\x{E1FE}/\x{1101}\x{1175}\x{11E2}/g;
    s/\x{E1FF}/\x{1101}\x{1175}\x{11F0}/g;
    s/\x{E200}/\x{1101}\x{1175}\x{11F9}/g;
    s/\x{E201}/\x{1101}\x{1198}/g;
    s/\x{E202}/\x{1101}\x{119E}/g;
    s/\x{E203}/\x{1101}\x{11A1}/g;
    s/\x{E204}/\x{1100}\x{1103}\x{1165}/g;
    s/\x{E205}/\x{1100}\x{1103}\x{1173}/g;
    s/\x{E206}/\x{1102}\x{1161}\x{11C4}/g;
    s/\x{E207}/\x{1102}\x{1161}\x{11C7}/g;
    s/\x{E208}/\x{1102}\x{1161}\x{11AE}\x{11B8}/g;
    s/\x{E209}/\x{1102}\x{1161}\x{11CD}/g;
    s/\x{E20A}/\x{1102}\x{1161}\x{11D0}/g;
    s/\x{E20B}/\x{1102}\x{1161}\x{11D9}/g;
    s/\x{E20C}/\x{1102}\x{1161}\x{11DA}/g;
    s/\x{E20D}/\x{1102}\x{1161}\x{11DC}/g;
    s/\x{E20E}/\x{1102}\x{1161}\x{11DD}/g;
    s/\x{E20F}/\x{1102}\x{1161}\x{11E2}/g;
    s/\x{E210}/\x{1102}\x{1161}\x{11E6}/g;
    s/\x{E211}/\x{1102}\x{1161}\x{11E7}/g;
    s/\x{E212}/\x{1102}\x{1161}\x{11EA}/g;
    s/\x{E213}/\x{1102}\x{1161}\x{11BA}\x{11BD}/g;
    s/\x{E214}/\x{1102}\x{1161}\x{11BA}\x{11C2}/g;
    s/\x{E215}/\x{1102}\x{1161}\x{11EB}/g;
    s/\x{E216}/\x{1102}\x{1161}\x{11EC}/g;
    s/\x{E217}/\x{1102}\x{1161}\x{11F0}/g;
    s/\x{E218}/\x{1102}\x{1161}\x{11BD}\x{11B8}/g;
    s/\x{E219}/\x{1102}\x{1161}\x{11BD}\x{11B8}\x{11B8}/g;
    s/\x{E21A}/\x{1102}\x{1161}\x{11F5}/g;
    s/\x{E21B}/\x{1102}\x{1161}\x{11F9}/g;
    s/\x{E21C}/\x{1102}\x{1176}/g;
    s/\x{E21D}/\x{1102}\x{1177}/g;
    s/\x{E21E}/\x{1102}\x{1162}\x{11AE}\x{11B8}/g;
    s/\x{E21F}/\x{1102}\x{1162}\x{11D9}/g;
    s/\x{E220}/\x{1102}\x{1162}\x{11B8}\x{11AE}/g;
    s/\x{E221}/\x{1102}\x{1162}\x{11E3}/g;
    s/\x{E222}/\x{1102}\x{1163}\x{11E2}/g;
    s/\x{E223}/\x{1102}\x{1163}\x{11F0}/g;
    s/\x{E224}/\x{1102}\x{1178}/g;
    s/\x{E225}/\x{1102}\x{1165}\x{11D0}/g;
    s/\x{E226}/\x{1102}\x{1165}\x{11DA}/g;
    s/\x{E227}/\x{1102}\x{1165}\x{11DB}/g;
    s/\x{E228}/\x{1102}\x{1165}\x{11DD}/g;
    s/\x{E229}/\x{1102}\x{1165}\x{11E3}/g;
    s/\x{E22A}/\x{1102}\x{1165}\x{11E5}/g;
    s/\x{E22B}/\x{1102}\x{1165}\x{11E9}/g;
    s/\x{E22C}/\x{1102}\x{1165}\x{11BA}\x{11C2}/g;
    s/\x{E22D}/\x{1102}\x{1165}\x{11F0}/g;
    s/\x{E22E}/\x{1102}\x{117B}/g;
    s/\x{E22F}/\x{1102}\x{117B}\x{11B7}/g;
    s/\x{E230}/\x{1102}\x{117C}/g;
    s/\x{E231}/\x{1102}\x{117C}\x{11A8}/g;
    s/\x{E232}/\x{1102}\x{117C}\x{11C2}/g;
    s/\x{E233}/\x{1102}\x{1167}\x{11A8}\x{11BF}/g;
    s/\x{E234}/\x{1102}\x{1167}\x{11A8}\x{11C2}/g;
    s/\x{E235}/\x{1102}\x{1167}\x{11C5}/g;
    s/\x{E236}/\x{1102}\x{1167}\x{11AB}\x{11AB}/g;
    s/\x{E237}/\x{1102}\x{1167}\x{11C7}/g;
    s/\x{E238}/\x{1102}\x{1167}\x{11D9}/g;
    s/\x{E239}/\x{1102}\x{1167}\x{11E7}/g;
    s/\x{E23A}/\x{1102}\x{1167}\x{11BA}\x{11C2}/g;
    s/\x{E23B}/\x{1102}\x{1167}\x{11F0}/g;
    s/\x{E23C}/\x{1102}\x{1167}\x{11F9}/g;
    s/\x{E23D}/\x{1102}\x{1169}\x{11CD}/g;
    s/\x{E23E}/\x{1102}\x{1169}\x{11D0}/g;
    s/\x{E23F}/\x{1102}\x{1169}\x{11D9}/g;
    s/\x{E240}/\x{1102}\x{1169}\x{11E2}/g;
    s/\x{E241}/\x{1102}\x{1169}\x{11E4}/g;
    s/\x{E242}/\x{1102}\x{1169}\x{11E5}/g;
    s/\x{E243}/\x{1102}\x{1169}\x{11E6}/g;
    s/\x{E244}/\x{1102}\x{1169}\x{11BA}\x{11C2}/g;
    s/\x{E245}/\x{1102}\x{1169}\x{11EB}/g;
    s/\x{E246}/\x{1102}\x{1169}\x{11F0}/g;
    s/\x{E247}/\x{1102}\x{1169}\x{11F0}\x{11C2}/g;
    s/\x{E248}/\x{1102}\x{1169}\x{11F3}/g;
    s/\x{E249}/\x{1102}\x{116A}\x{11D9}/g;
    s/\x{E24A}/\x{1102}\x{117F}/g;
    s/\x{E24B}/\x{1102}\x{116C}\x{11F0}/g;
    s/\x{E24C}/\x{1102}\x{116D}\x{11D9}/g;
    s/\x{E24D}/\x{1102}\x{116D}\x{11E2}/g;
    s/\x{E24E}/\x{1102}\x{116D}\x{11F0}/g;
    s/\x{E24F}/\x{1102}\x{116D}\x{1161}/g;
    s/\x{E250}/\x{1102}\x{116D}\x{1161}\x{11BC}/g;
    s/\x{E251}/\x{1102}\x{1188}/g;
    s/\x{E252}/\x{1102}\x{116E}\x{11C6}/g;
    s/\x{E253}/\x{1102}\x{116E}\x{11C7}/g;
    s/\x{E254}/\x{1102}\x{116E}\x{11C8}/g;
    s/\x{E255}/\x{1102}\x{116E}\x{11D0}/g;
    s/\x{E256}/\x{1102}\x{116E}\x{11DD}/g;
    s/\x{E257}/\x{1102}\x{116E}\x{11E2}/g;
    s/\x{E258}/\x{1102}\x{116E}\x{11E6}/g;
    s/\x{E259}/\x{1102}\x{116E}\x{11F5}/g;
    s/\x{E25A}/\x{1102}\x{1189}/g;
    s/\x{E25B}/\x{1102}\x{1189}\x{11AB}/g;
    s/\x{E25C}/\x{1102}\x{116E}\x{1167}/g;
    s/\x{E25D}/\x{1102}\x{118C}/g;
    s/\x{E25E}/\x{1102}\x{1171}\x{11EB}/g;
    s/\x{E25F}/\x{1102}\x{1171}\x{1175}/g;
    s/\x{E260}/\x{1102}\x{1172}\x{11D9}/g;
    s/\x{E261}/\x{1102}\x{1172}\x{11E2}/g;
    s/\x{E262}/\x{1102}\x{1172}\x{11F0}/g;
    s/\x{E263}/\x{1102}\x{118F}/g;
    s/\x{E264}/\x{1102}\x{1194}/g;
    s/\x{E265}/\x{1102}\x{1194}\x{11AB}/g;
    s/\x{E266}/\x{1102}\x{1194}\x{11BC}/g;
    s/\x{E267}/\x{1102}\x{1173}\x{11C3}/g;
    s/\x{E268}/\x{1102}\x{1173}\x{11CD}/g;
    s/\x{E269}/\x{1102}\x{1173}\x{11D9}/g;
    s/\x{E26A}/\x{1102}\x{1173}\x{11F0}/g;
    s/\x{E26B}/\x{1102}\x{1173}\x{11F0}\x{11C2}/g;
    s/\x{E26C}/\x{1102}\x{1196}/g;
    s/\x{E26D}/\x{1102}\x{1174}\x{11F0}/g;
    s/\x{E26E}/\x{1102}\x{1197}/g;
    s/\x{E26F}/\x{1102}\x{1175}\x{11C3}/g;
    s/\x{E270}/\x{1102}\x{1175}\x{11CE}/g;
    s/\x{E271}/\x{1102}\x{1175}\x{11D0}/g;
    s/\x{E272}/\x{1102}\x{1175}\x{11B2}\x{11AE}/g;
    s/\x{E273}/\x{1102}\x{1175}\x{11D9}/g;
    s/\x{E274}/\x{1102}\x{1175}\x{11DD}/g;
    s/\x{E275}/\x{1102}\x{1175}\x{11E2}/g;
    s/\x{E276}/\x{1102}\x{1175}\x{11E4}/g;
    s/\x{E277}/\x{1102}\x{1175}\x{11E5}/g;
    s/\x{E278}/\x{1102}\x{1175}\x{11E6}/g;
    s/\x{E279}/\x{1102}\x{1175}\x{11EB}/g;
    s/\x{E27A}/\x{1102}\x{1175}\x{11F0}/g;
    s/\x{E27B}/\x{1102}\x{1175}\x{11F9}/g;
    s/\x{E27C}/\x{1102}\x{1198}/g;
    s/\x{E27D}/\x{1102}\x{1199}/g;
    s/\x{E27E}/\x{1102}\x{1199}\x{11AB}/g;
    s/\x{E27F}/\x{1102}\x{1199}\x{11BC}/g;
    s/\x{E280}/\x{1102}\x{1175}\x{1167}/g;
    s/\x{E281}/\x{1102}\x{1175}\x{1168}/g;
    s/\x{E282}/\x{1102}\x{119B}/g;
    s/\x{E283}/\x{1102}\x{119E}/g;
    s/\x{E284}/\x{1102}\x{119E}\x{11A8}/g;
    s/\x{E285}/\x{1102}\x{119E}\x{11AB}/g;
    s/\x{E286}/\x{1102}\x{119E}\x{11AE}/g;
    s/\x{E287}/\x{1102}\x{119E}\x{11AE}\x{11BE}/g;
    s/\x{E288}/\x{1102}\x{119E}\x{11AF}/g;
    s/\x{E289}/\x{1102}\x{119E}\x{11B0}/g;
    s/\x{E28A}/\x{1102}\x{119E}\x{11B1}/g;
    s/\x{E28B}/\x{1102}\x{119E}\x{11B2}/g;
    s/\x{E28C}/\x{1102}\x{119E}\x{11B3}/g;
    s/\x{E28D}/\x{1102}\x{119E}\x{11D7}/g;
    s/\x{E28E}/\x{1102}\x{119E}\x{11B6}/g;
    s/\x{E28F}/\x{1102}\x{119E}\x{11D9}/g;
    s/\x{E290}/\x{1102}\x{119E}\x{11B7}/g;
    s/\x{E291}/\x{1102}\x{119E}\x{11B8}/g;
    s/\x{E292}/\x{1102}\x{119E}\x{11BA}/g;
    s/\x{E293}/\x{1102}\x{119E}\x{11BA}\x{11BE}/g;
    s/\x{E294}/\x{1102}\x{119E}\x{11EB}/g;
    s/\x{E295}/\x{1102}\x{119E}\x{11BC}/g;
    s/\x{E296}/\x{1102}\x{119E}\x{11F0}/g;
    s/\x{E297}/\x{1102}\x{119E}\x{11BD}/g;
    s/\x{E298}/\x{1102}\x{119E}\x{11BE}/g;
    s/\x{E299}/\x{1102}\x{11A1}/g;
    s/\x{E29A}/\x{1102}\x{11A1}\x{11A8}/g;
    s/\x{E29B}/\x{1102}\x{11A1}\x{11AB}/g;
    s/\x{E29C}/\x{1102}\x{11A1}\x{11AE}/g;
    s/\x{E29D}/\x{1102}\x{11A1}\x{11AF}/g;
    s/\x{E29E}/\x{1102}\x{11A1}\x{11B7}/g;
    s/\x{E29F}/\x{1102}\x{11A1}\x{11B8}/g;
    s/\x{E2A0}/\x{1102}\x{11A1}\x{11BA}/g;
    s/\x{E2A1}/\x{1102}\x{11A1}\x{11E8}/g;
    s/\x{E2A2}/\x{1102}\x{11A1}\x{11BC}/g;
    s/\x{E2A3}/\x{1102}\x{11A1}\x{11F0}/g;
    s/\x{E2A4}/\x{1102}\x{11A1}\x{11C2}/g;
    s/\x{E2A5}/\x{1113}\x{1161}/g;
    s/\x{E2A6}/\x{1113}\x{1162}/g;
    s/\x{E2A7}/\x{1113}\x{1163}/g;
    s/\x{E2A8}/\x{1113}\x{1165}/g;
    s/\x{E2A9}/\x{1113}\x{1166}/g;
    s/\x{E2AA}/\x{1113}\x{1167}/g;
    s/\x{E2AB}/\x{1113}\x{1168}/g;
    s/\x{E2AC}/\x{1113}\x{1168}\x{11AB}/g;
    s/\x{E2AD}/\x{1113}\x{1169}/g;
    s/\x{E2AE}/\x{1113}\x{116A}/g;
    s/\x{E2AF}/\x{1113}\x{116B}/g;
    s/\x{E2B0}/\x{1113}\x{116C}/g;
    s/\x{E2B1}/\x{1113}\x{116D}/g;
    s/\x{E2B2}/\x{1113}\x{116E}/g;
    s/\x{E2B3}/\x{1113}\x{116F}/g;
    s/\x{E2B4}/\x{1113}\x{1170}/g;
    s/\x{E2B5}/\x{1113}\x{116E}\x{1167}/g;
    s/\x{E2B6}/\x{1113}\x{118C}/g;
    s/\x{E2B7}/\x{1113}\x{1171}/g;
    s/\x{E2B8}/\x{1113}\x{1172}/g;
    s/\x{E2B9}/\x{1113}\x{1194}/g;
    s/\x{E2BA}/\x{1113}\x{1173}/g;
    s/\x{E2BB}/\x{1113}\x{1174}/g;
    s/\x{E2BC}/\x{1113}\x{1175}/g;
    s/\x{E2BD}/\x{1113}\x{1198}/g;
    s/\x{E2BE}/\x{1113}\x{119E}/g;
    s/\x{E2BF}/\x{1113}\x{11A1}/g;
    s/\x{E2C0}/\x{1114}\x{1161}/g;
    s/\x{E2C1}/\x{1114}\x{1163}/g;
    s/\x{E2C2}/\x{1114}\x{1163}\x{11BC}/g;
    s/\x{E2C3}/\x{1114}\x{1165}/g;
    s/\x{E2C4}/\x{1114}\x{1166}/g;
    s/\x{E2C5}/\x{1114}\x{1169}/g;
    s/\x{E2C6}/\x{1114}\x{116E}/g;
    s/\x{E2C7}/\x{1114}\x{1173}/g;
    s/\x{E2C8}/\x{1114}\x{1173}\x{11AB}/g;
    s/\x{E2C9}/\x{1114}\x{119E}/g;
    s/\x{E2CA}/\x{1114}\x{119E}\x{11AB}/g;
    s/\x{E2CB}/\x{1115}\x{1161}/g;
    s/\x{E2CC}/\x{1115}\x{1161}\x{11AB}/g;
    s/\x{E2CD}/\x{1115}\x{1162}/g;
    s/\x{E2CE}/\x{1115}\x{1163}/g;
    s/\x{E2CF}/\x{1115}\x{1165}/g;
    s/\x{E2D0}/\x{1115}\x{1166}/g;
    s/\x{E2D1}/\x{1115}\x{1166}\x{11AB}/g;
    s/\x{E2D2}/\x{1115}\x{1167}/g;
    s/\x{E2D3}/\x{1115}\x{1168}/g;
    s/\x{E2D4}/\x{1115}\x{1169}/g;
    s/\x{E2D5}/\x{1115}\x{1169}\x{11AB}/g;
    s/\x{E2D6}/\x{1115}\x{116A}/g;
    s/\x{E2D7}/\x{1115}\x{116B}/g;
    s/\x{E2D8}/\x{1115}\x{116C}/g;
    s/\x{E2D9}/\x{1115}\x{116D}/g;
    s/\x{E2DA}/\x{1115}\x{116E}/g;
    s/\x{E2DB}/\x{1115}\x{116F}/g;
    s/\x{E2DC}/\x{1115}\x{1170}/g;
    s/\x{E2DD}/\x{1115}\x{116E}\x{1167}/g;
    s/\x{E2DE}/\x{1115}\x{118C}/g;
    s/\x{E2DF}/\x{1115}\x{1171}/g;
    s/\x{E2E0}/\x{1115}\x{1172}/g;
    s/\x{E2E1}/\x{1115}\x{1194}/g;
    s/\x{E2E2}/\x{1115}\x{1173}/g;
    s/\x{E2E3}/\x{1115}\x{1174}/g;
    s/\x{E2E4}/\x{1115}\x{1175}/g;
    s/\x{E2E5}/\x{1115}\x{1198}/g;
    s/\x{E2E6}/\x{1115}\x{119E}/g;
    s/\x{E2E7}/\x{1115}\x{11A1}/g;
    s/\x{E2E8}/\x{1116}\x{1161}/g;
    s/\x{E2E9}/\x{1116}\x{1162}/g;
    s/\x{E2EA}/\x{1116}\x{1163}/g;
    s/\x{E2EB}/\x{1116}\x{1165}/g;
    s/\x{E2EC}/\x{1116}\x{1166}/g;
    s/\x{E2ED}/\x{1116}\x{1167}/g;
    s/\x{E2EE}/\x{1116}\x{1168}/g;
    s/\x{E2EF}/\x{1116}\x{1169}/g;
    s/\x{E2F0}/\x{1116}\x{116A}/g;
    s/\x{E2F1}/\x{1116}\x{116B}/g;
    s/\x{E2F2}/\x{1116}\x{116C}/g;
    s/\x{E2F3}/\x{1116}\x{116D}/g;
    s/\x{E2F4}/\x{1116}\x{116E}/g;
    s/\x{E2F5}/\x{1116}\x{116F}/g;
    s/\x{E2F6}/\x{1116}\x{1170}/g;
    s/\x{E2F7}/\x{1116}\x{116E}\x{1167}/g;
    s/\x{E2F8}/\x{1116}\x{118C}/g;
    s/\x{E2F9}/\x{1116}\x{1171}/g;
    s/\x{E2FA}/\x{1116}\x{1172}/g;
    s/\x{E2FB}/\x{1116}\x{1194}/g;
    s/\x{E2FC}/\x{1116}\x{1173}/g;
    s/\x{E2FD}/\x{1116}\x{1174}/g;
    s/\x{E2FE}/\x{1116}\x{1175}/g;
    s/\x{E2FF}/\x{1116}\x{1198}/g;
    s/\x{E300}/\x{1116}\x{119E}/g;
    s/\x{E301}/\x{1116}\x{11A1}/g;
    s/\x{E302}/\x{1102}\x{1109}\x{1175}/g;
    s/\x{E303}/\x{1102}\x{1109}\x{1198}/g;
    s/\x{E304}/\x{1102}\x{110C}\x{1173}/g;
    s/\x{E305}/\x{1102}\x{110C}\x{1175}/g;
    s/\x{E306}/\x{1102}\x{1112}\x{1163}/g;
    s/\x{E307}/\x{1102}\x{1112}\x{1163}\x{11BC}/g;
    s/\x{E308}/\x{1102}\x{1112}\x{1169}/g;
    s/\x{E309}/\x{1102}\x{1112}\x{1172}/g;
    s/\x{E30A}/\x{1102}\x{1112}\x{1172}\x{11AB}/g;
    s/\x{E30B}/\x{1102}\x{1112}\x{1175}/g;
    s/\x{E30C}/\x{1102}\x{1112}\x{1175}\x{11BC}/g;
    s/\x{E30D}/\x{1103}\x{1161}\x{11C7}/g;
    s/\x{E30E}/\x{1103}\x{1161}\x{11CA}/g;
    s/\x{E30F}/\x{1103}\x{1161}\x{11AE}\x{11AE}/g;
    s/\x{E310}/\x{1103}\x{1161}\x{11CD}/g;
    s/\x{E311}/\x{1103}\x{1161}\x{11D0}/g;
    s/\x{E312}/\x{1103}\x{1161}\x{11D9}/g;
    s/\x{E313}/\x{1103}\x{1161}\x{11DB}/g;
    s/\x{E314}/\x{1103}\x{1161}\x{11DD}/g;
    s/\x{E315}/\x{1103}\x{1161}\x{11E2}/g;
    s/\x{E316}/\x{1103}\x{1161}\x{11E5}/g;
    s/\x{E317}/\x{1103}\x{1161}\x{11E6}/g;
    s/\x{E318}/\x{1103}\x{1161}\x{11E7}/g;
    s/\x{E319}/\x{1103}\x{1161}\x{11BA}\x{11C2}/g;
    s/\x{E31A}/\x{1103}\x{1161}\x{11EB}/g;
    s/\x{E31B}/\x{1103}\x{1161}\x{11F0}/g;
    s/\x{E31C}/\x{1103}\x{1161}\x{11F1}/g;
    s/\x{E31D}/\x{1103}\x{1161}\x{11F0}\x{11C2}/g;
    s/\x{E31E}/\x{1103}\x{1161}\x{11F9}/g;
    s/\x{E31F}/\x{1103}\x{1176}/g;
    s/\x{E320}/\x{1103}\x{1176}\x{11AF}/g;
    s/\x{E321}/\x{1103}\x{1162}\x{11F0}/g;
    s/\x{E322}/\x{1103}\x{1163}\x{11E2}/g;
    s/\x{E323}/\x{1103}\x{1163}\x{11F0}/g;
    s/\x{E324}/\x{1103}\x{1163}\x{11F1}/g;
    s/\x{E325}/\x{1103}\x{1178}/g;
    s/\x{E326}/\x{1103}\x{1165}\x{11DD}/g;
    s/\x{E327}/\x{1103}\x{1165}\x{11E4}/g;
    s/\x{E328}/\x{1103}\x{1165}\x{11E6}/g;
    s/\x{E329}/\x{1103}\x{1165}\x{11EB}/g;
    s/\x{E32A}/\x{1103}\x{1165}\x{11F0}/g;
    s/\x{E32B}/\x{1103}\x{117A}/g;
    s/\x{E32C}/\x{1103}\x{117B}/g;
    s/\x{E32D}/\x{1103}\x{117B}\x{11AF}/g;
    s/\x{E32E}/\x{1103}\x{117C}/g;
    s/\x{E32F}/\x{1103}\x{117C}\x{11AF}/g;
    s/\x{E330}/\x{1103}\x{117C}\x{11B7}/g;
    s/\x{E331}/\x{1103}\x{1166}\x{11C6}/g;
    s/\x{E332}/\x{1103}\x{1166}\x{11B8}\x{11AE}/g;
    s/\x{E333}/\x{1103}\x{1167}\x{11D9}/g;
    s/\x{E334}/\x{1103}\x{1167}\x{11E2}/g;
    s/\x{E335}/\x{1103}\x{1167}\x{11F0}/g;
    s/\x{E336}/\x{1103}\x{1167}\x{11F9}/g;
    s/\x{E337}/\x{1103}\x{1169}\x{11C6}/g;
    s/\x{E338}/\x{1103}\x{1169}\x{11CA}/g;
    s/\x{E339}/\x{1103}\x{1169}\x{11AE}\x{11C0}/g;
    s/\x{E33A}/\x{1103}\x{1169}\x{11D0}/g;
    s/\x{E33B}/\x{1103}\x{1169}\x{11D6}/g;
    s/\x{E33C}/\x{1103}\x{1169}\x{11D8}/g;
    s/\x{E33D}/\x{1103}\x{1169}\x{11D9}/g;
    s/\x{E33E}/\x{1103}\x{1169}\x{11E2}/g;
    s/\x{E33F}/\x{1103}\x{1169}\x{11E6}/g;
    s/\x{E340}/\x{1103}\x{1169}\x{11E7}/g;
    s/\x{E341}/\x{1103}\x{1169}\x{11BA}\x{11E7}/g;
    s/\x{E342}/\x{1103}\x{1169}\x{11BA}\x{11C2}/g;
    s/\x{E343}/\x{1103}\x{1169}\x{11EB}/g;
    s/\x{E344}/\x{1103}\x{1169}\x{11F0}/g;
    s/\x{E345}/\x{1103}\x{1169}\x{11F1}/g;
    s/\x{E346}/\x{1103}\x{116A}\x{11D9}/g;
    s/\x{E347}/\x{1103}\x{116A}\x{11F0}/g;
    s/\x{E348}/\x{1103}\x{1182}/g;
    s/\x{E349}/\x{1103}\x{1182}\x{11B7}/g;
    s/\x{E34A}/\x{1103}\x{116D}\x{11E2}/g;
    s/\x{E34B}/\x{1103}\x{116D}\x{11BC}\x{11B7}/g;
    s/\x{E34C}/\x{1103}\x{116D}\x{11F0}/g;
    s/\x{E34D}/\x{1103}\x{1188}/g;
    s/\x{E34E}/\x{1103}\x{1188}\x{11AB}/g;
    s/\x{E34F}/\x{1103}\x{1188}\x{11AF}/g;
    s/\x{E350}/\x{1103}\x{116E}\x{11CE}/g;
    s/\x{E351}/\x{1103}\x{116E}\x{11D0}/g;
    s/\x{E352}/\x{1103}\x{116E}\x{11D9}/g;
    s/\x{E353}/\x{1103}\x{116E}\x{11DA}/g;
    s/\x{E354}/\x{1103}\x{116E}\x{11E2}/g;
    s/\x{E355}/\x{1103}\x{116E}\x{11F0}/g;
    s/\x{E356}/\x{1103}\x{1189}/g;
    s/\x{E357}/\x{1103}\x{1189}\x{11AB}/g;
    s/\x{E358}/\x{1103}\x{116F}\x{11F9}/g;
    s/\x{E359}/\x{1103}\x{116E}\x{1167}/g;
    s/\x{E35A}/\x{1103}\x{118C}/g;
    s/\x{E35B}/\x{1103}\x{1171}\x{11E6}/g;
    s/\x{E35C}/\x{1103}\x{1171}\x{11F9}/g;
    s/\x{E35D}/\x{1103}\x{1172}\x{11D9}/g;
    s/\x{E35E}/\x{1103}\x{1172}\x{11E2}/g;
    s/\x{E35F}/\x{1103}\x{1172}\x{11F0}/g;
    s/\x{E360}/\x{1103}\x{1172}\x{11F1}/g;
    s/\x{E361}/\x{1103}\x{1191}/g;
    s/\x{E362}/\x{1103}\x{1191}\x{11AB}/g;
    s/\x{E363}/\x{1103}\x{1191}\x{11AF}/g;
    s/\x{E364}/\x{1103}\x{1191}\x{11D9}/g;
    s/\x{E365}/\x{1103}\x{1191}\x{11BC}/g;
    s/\x{E366}/\x{1103}\x{1192}/g;
    s/\x{E367}/\x{1103}\x{1192}\x{11BC}/g;
    s/\x{E368}/\x{1103}\x{1194}/g;
    s/\x{E369}/\x{1103}\x{1194}\x{11AB}/g;
    s/\x{E36A}/\x{1103}\x{1194}\x{11BC}/g;
    s/\x{E36B}/\x{1103}\x{1173}\x{11CD}/g;
    s/\x{E36C}/\x{1103}\x{1173}\x{11CE}/g;
    s/\x{E36D}/\x{1103}\x{1173}\x{11D0}/g;
    s/\x{E36E}/\x{1103}\x{1173}\x{11D9}/g;
    s/\x{E36F}/\x{1103}\x{1173}\x{11DA}/g;
    s/\x{E370}/\x{1103}\x{1173}\x{11E2}/g;
    s/\x{E371}/\x{1103}\x{1173}\x{11E3}/g;
    s/\x{E372}/\x{1103}\x{1173}\x{11E6}/g;
    s/\x{E373}/\x{1103}\x{1173}\x{11EB}/g;
    s/\x{E374}/\x{1103}\x{1173}\x{11F0}/g;
    s/\x{E375}/\x{1103}\x{1173}\x{11F1}/g;
    s/\x{E376}/\x{1103}\x{1195}/g;
    s/\x{E377}/\x{1103}\x{1196}/g;
    s/\x{E378}/\x{1103}\x{1174}\x{11D9}/g;
    s/\x{E379}/\x{1103}\x{1173}\x{11F9}/g;
    s/\x{E37A}/\x{1103}\x{1175}\x{11D0}/g;
    s/\x{E37B}/\x{1103}\x{1175}\x{11D9}/g;
    s/\x{E37C}/\x{1103}\x{1175}\x{11E2}/g;
    s/\x{E37D}/\x{1103}\x{1175}\x{11E4}/g;
    s/\x{E37E}/\x{1103}\x{1175}\x{11E5}/g;
    s/\x{E37F}/\x{1103}\x{1175}\x{11BA}\x{11C0}/g;
    s/\x{E380}/\x{1103}\x{1175}\x{11EB}/g;
    s/\x{E381}/\x{1103}\x{1175}\x{11F0}/g;
    s/\x{E382}/\x{1103}\x{1175}\x{11F9}/g;
    s/\x{E383}/\x{1103}\x{1198}/g;
    s/\x{E384}/\x{1103}\x{1199}/g;
    s/\x{E385}/\x{1103}\x{1199}\x{11AB}/g;
    s/\x{E386}/\x{1103}\x{1199}\x{11BC}/g;
    s/\x{E387}/\x{1103}\x{1175}\x{1167}/g;
    s/\x{E388}/\x{1103}\x{1175}\x{1168}/g;
    s/\x{E389}/\x{1103}\x{119B}/g;
    s/\x{E38A}/\x{1103}\x{119E}/g;
    s/\x{E38B}/\x{1103}\x{119E}\x{11A8}/g;
    s/\x{E38C}/\x{1103}\x{119E}\x{11C3}/g;
    s/\x{E38D}/\x{1103}\x{119E}\x{11AB}/g;
    s/\x{E38E}/\x{1103}\x{119E}\x{11AE}/g;
    s/\x{E38F}/\x{1103}\x{119E}\x{11AF}/g;
    s/\x{E390}/\x{1103}\x{119E}\x{11B0}/g;
    s/\x{E391}/\x{1103}\x{119E}\x{11CC}/g;
    s/\x{E392}/\x{1103}\x{119E}\x{11B1}/g;
    s/\x{E393}/\x{1103}\x{119E}\x{11B2}/g;
    s/\x{E394}/\x{1103}\x{119E}\x{11B3}/g;
    s/\x{E395}/\x{1103}\x{119E}\x{11B6}/g;
    s/\x{E396}/\x{1103}\x{119E}\x{11D9}/g;
    s/\x{E397}/\x{1103}\x{119E}\x{11B7}/g;
    s/\x{E398}/\x{1103}\x{119E}\x{11DA}/g;
    s/\x{E399}/\x{1103}\x{119E}\x{11B7}\x{11AB}/g;
    s/\x{E39A}/\x{1103}\x{119E}\x{11B7}\x{11AB}\x{11AB}/g;
    s/\x{E39B}/\x{1103}\x{119E}\x{11B8}/g;
    s/\x{E39C}/\x{1103}\x{119E}\x{11E3}/g;
    s/\x{E39D}/\x{1103}\x{119E}\x{11E6}/g;
    s/\x{E39E}/\x{1103}\x{119E}\x{11BA}/g;
    s/\x{E39F}/\x{1103}\x{119E}\x{11E7}/g;
    s/\x{E3A0}/\x{1103}\x{119E}\x{11E8}/g;
    s/\x{E3A1}/\x{1103}\x{119E}\x{11E9}/g;
    s/\x{E3A2}/\x{1103}\x{119E}\x{11EA}/g;
    s/\x{E3A3}/\x{1103}\x{119E}\x{11EB}/g;
    s/\x{E3A4}/\x{1103}\x{119E}\x{11BC}/g;
    s/\x{E3A5}/\x{1103}\x{119E}\x{11F0}/g;
    s/\x{E3A6}/\x{1103}\x{119E}\x{11BD}/g;
    s/\x{E3A7}/\x{1103}\x{119E}\x{11BE}/g;
    s/\x{E3A8}/\x{1103}\x{11A1}/g;
    s/\x{E3A9}/\x{1103}\x{11A1}\x{11A8}/g;
    s/\x{E3AA}/\x{1103}\x{11A1}\x{11AB}/g;
    s/\x{E3AB}/\x{1103}\x{11A1}\x{11AE}/g;
    s/\x{E3AC}/\x{1103}\x{11A1}\x{11AF}/g;
    s/\x{E3AD}/\x{1103}\x{11A1}\x{11D9}/g;
    s/\x{E3AE}/\x{1103}\x{11A1}\x{11B7}/g;
    s/\x{E3AF}/\x{1103}\x{11A1}\x{11B8}/g;
    s/\x{E3B0}/\x{1103}\x{11A1}\x{11BA}/g;
    s/\x{E3B1}/\x{1103}\x{11A1}\x{11EB}/g;
    s/\x{E3B2}/\x{1103}\x{11A1}\x{11BC}/g;
    s/\x{E3B3}/\x{1103}\x{11A1}\x{11F0}/g;
    s/\x{E3B4}/\x{1117}\x{1173}/g;
    s/\x{E3B5}/\x{1117}\x{1173}\x{11AF}/g;
    s/\x{E3B6}/\x{1117}\x{1175}/g;
    s/\x{E3B7}/\x{1117}\x{1175}\x{11AF}/g;
    s/\x{E3B8}/\x{1104}\x{1161}\x{11D0}/g;
    s/\x{E3B9}/\x{1104}\x{1161}\x{11D8}/g;
    s/\x{E3BA}/\x{1104}\x{1161}\x{11D9}/g;
    s/\x{E3BB}/\x{1104}\x{1161}\x{11E2}/g;
    s/\x{E3BC}/\x{1104}\x{1161}\x{11E6}/g;
    s/\x{E3BD}/\x{1104}\x{1161}\x{11F0}/g;
    s/\x{E3BE}/\x{1104}\x{1161}\x{11F9}/g;
    s/\x{E3BF}/\x{1104}\x{1163}\x{11E2}/g;
    s/\x{E3C0}/\x{1104}\x{1163}\x{11F0}/g;
    s/\x{E3C1}/\x{1104}\x{117B}/g;
    s/\x{E3C2}/\x{1104}\x{117C}/g;
    s/\x{E3C3}/\x{1104}\x{117C}\x{11AF}/g;
    s/\x{E3C4}/\x{1104}\x{117C}\x{11B2}/g;
    s/\x{E3C5}/\x{1104}\x{1167}\x{11D9}/g;
    s/\x{E3C6}/\x{1104}\x{1167}\x{11F0}/g;
    s/\x{E3C7}/\x{1104}\x{1169}\x{11D9}/g;
    s/\x{E3C8}/\x{1104}\x{1169}\x{11E2}/g;
    s/\x{E3C9}/\x{1104}\x{1169}\x{11F0}/g;
    s/\x{E3CA}/\x{1104}\x{1169}\x{11F9}/g;
    s/\x{E3CB}/\x{1104}\x{116A}\x{11D9}/g;
    s/\x{E3CC}/\x{1104}\x{116A}\x{11F0}/g;
    s/\x{E3CD}/\x{1104}\x{117F}/g;
    s/\x{E3CE}/\x{1104}\x{116D}\x{11E2}/g;
    s/\x{E3CF}/\x{1104}\x{116E}\x{11E2}/g;
    s/\x{E3D0}/\x{1104}\x{116E}\x{11F0}/g;
    s/\x{E3D1}/\x{1104}\x{116E}\x{11F9}/g;
    s/\x{E3D2}/\x{1104}\x{116E}\x{1167}/g;
    s/\x{E3D3}/\x{1104}\x{118C}/g;
    s/\x{E3D4}/\x{1104}\x{1171}\x{11F9}/g;
    s/\x{E3D5}/\x{1104}\x{1172}\x{11D9}/g;
    s/\x{E3D6}/\x{1104}\x{1172}\x{11E2}/g;
    s/\x{E3D7}/\x{1104}\x{1172}\x{11F0}/g;
    s/\x{E3D8}/\x{1104}\x{1191}/g;
    s/\x{E3D9}/\x{1104}\x{1191}\x{11AB}/g;
    s/\x{E3DA}/\x{1104}\x{1194}/g;
    s/\x{E3DB}/\x{1104}\x{1194}\x{11BC}/g;
    s/\x{E3DC}/\x{1104}\x{1173}\x{11AE}\x{11AE}/g;
    s/\x{E3DD}/\x{1104}\x{1173}\x{11E2}/g;
    s/\x{E3DE}/\x{1104}\x{1173}\x{11F0}/g;
    s/\x{E3DF}/\x{1104}\x{1173}\x{11F9}/g;
    s/\x{E3E0}/\x{1104}\x{1174}\x{11F9}/g;
    s/\x{E3E1}/\x{1104}\x{1175}\x{11D9}/g;
    s/\x{E3E2}/\x{1104}\x{1175}\x{11F0}/g;
    s/\x{E3E3}/\x{1104}\x{1175}\x{11F9}/g;
    s/\x{E3E4}/\x{1104}\x{1198}/g;
    s/\x{E3E5}/\x{1104}\x{119E}/g;
    s/\x{E3E6}/\x{1104}\x{119E}\x{11A8}/g;
    s/\x{E3E7}/\x{1104}\x{119E}\x{11AB}/g;
    s/\x{E3E8}/\x{1104}\x{119E}\x{11AF}/g;
    s/\x{E3E9}/\x{1104}\x{119E}\x{11B7}/g;
    s/\x{E3EA}/\x{1104}\x{119E}\x{11BA}/g;
    s/\x{E3EB}/\x{1104}\x{11A1}/g;
    s/\x{E3EC}/\x{1104}\x{11A1}\x{11A8}/g;
    s/\x{E3ED}/\x{1104}\x{11A1}\x{11AF}/g;
    s/\x{E3EE}/\x{1104}\x{11A1}\x{11BC}/g;
    s/\x{E3EF}/\x{1104}\x{11A1}\x{11F0}/g;
    s/\x{E3F0}/\x{1103}\x{1105}\x{1175}/g;
    s/\x{E3F1}/\x{1103}\x{1105}\x{1175}\x{11BC}/g;
    s/\x{E3F2}/\x{1103}\x{1105}\x{1175}\x{11EF}/g;
    s/\x{E3F3}/\x{1103}\x{1106}\x{1161}/g;
    s/\x{E3F4}/\x{1103}\x{1106}\x{1161}\x{11AF}/g;
    s/\x{E3F5}/\x{1103}\x{1107}\x{1161}/g;
    s/\x{E3F6}/\x{1103}\x{1107}\x{1162}/g;
    s/\x{E3F7}/\x{1103}\x{1107}\x{116E}/g;
    s/\x{E3F8}/\x{1103}\x{1107}\x{116E}\x{11AF}/g;
    s/\x{E3F9}/\x{1103}\x{1109}\x{1172}/g;
    s/\x{E3FA}/\x{1103}\x{1109}\x{1172}\x{11AF}/g;
    s/\x{E3FB}/\x{1103}\x{1109}\x{1172}\x{11BC}/g;
    s/\x{E3FC}/\x{1103}\x{1109}\x{1173}/g;
    s/\x{E3FD}/\x{1103}\x{1109}\x{1175}/g;
    s/\x{E3FE}/\x{1103}\x{1109}\x{1175}\x{11A8}/g;
    s/\x{E3FF}/\x{1103}\x{110C}\x{1163}/g;
    s/\x{E400}/\x{1103}\x{110C}\x{1163}\x{11BC}/g;
    s/\x{E401}/\x{1103}\x{110C}\x{119E}/g;
    s/\x{E402}/\x{1105}\x{1161}\x{11D0}/g;
    s/\x{E403}/\x{1105}\x{1161}\x{11D9}/g;
    s/\x{E404}/\x{1105}\x{1161}\x{11B7}\x{11B7}/g;
    s/\x{E405}/\x{1105}\x{1161}\x{11DD}/g;
    s/\x{E406}/\x{1105}\x{1161}\x{11E2}/g;
    s/\x{E407}/\x{1105}\x{1161}\x{11E6}/g;
    s/\x{E408}/\x{1105}\x{1161}\x{11E7}/g;
    s/\x{E409}/\x{1105}\x{1161}\x{11BA}\x{11E7}/g;
    s/\x{E40A}/\x{1105}\x{1161}\x{11EB}/g;
    s/\x{E40B}/\x{1105}\x{1161}\x{11F0}/g;
    s/\x{E40C}/\x{1105}\x{1161}\x{11F0}\x{11A8}/g;
    s/\x{E40D}/\x{1105}\x{1161}\x{11F5}/g;
    s/\x{E40E}/\x{1105}\x{1161}\x{11F9}/g;
    s/\x{E40F}/\x{1105}\x{1176}/g;
    s/\x{E410}/\x{1105}\x{1176}\x{11AF}/g;
    s/\x{E411}/\x{1105}\x{1177}/g;
    s/\x{E412}/\x{1105}\x{1162}\x{11F0}/g;
    s/\x{E413}/\x{1105}\x{1163}\x{11E2}/g;
    s/\x{E414}/\x{1105}\x{1163}\x{11F0}/g;
    s/\x{E415}/\x{1105}\x{1178}/g;
    s/\x{E416}/\x{1105}\x{1165}\x{11B7}\x{11B7}/g;
    s/\x{E417}/\x{1105}\x{1165}\x{11E6}/g;
    s/\x{E418}/\x{1105}\x{1165}\x{11BA}\x{11C2}/g;
    s/\x{E419}/\x{1105}\x{1165}\x{11F0}/g;
    s/\x{E41A}/\x{1105}\x{1165}\x{11F5}/g;
    s/\x{E41B}/\x{1105}\x{117A}/g;
    s/\x{E41C}/\x{1105}\x{117B}/g;
    s/\x{E41D}/\x{1105}\x{117B}\x{11AF}/g;
    s/\x{E41E}/\x{1105}\x{1166}\x{11F0}/g;
    s/\x{E41F}/\x{1105}\x{1167}\x{11C7}/g;
    s/\x{E420}/\x{1105}\x{1167}\x{11D9}/g;
    s/\x{E421}/\x{1105}\x{1167}\x{11E2}/g;
    s/\x{E422}/\x{1105}\x{1167}\x{11E6}/g;
    s/\x{E423}/\x{1105}\x{1167}\x{11F0}/g;
    s/\x{E424}/\x{1105}\x{1167}\x{11F9}/g;
    s/\x{E425}/\x{1105}\x{1169}\x{11AE}\x{11BA}/g;
    s/\x{E426}/\x{1105}\x{1169}\x{11D9}/g;
    s/\x{E427}/\x{1105}\x{1169}\x{11DD}/g;
    s/\x{E428}/\x{1105}\x{1169}\x{11E2}/g;
    s/\x{E429}/\x{1105}\x{1169}\x{11B8}\x{11B7}/g;
    s/\x{E42A}/\x{1105}\x{1169}\x{11E6}/g;
    s/\x{E42B}/\x{1105}\x{1169}\x{11BA}\x{11BE}/g;
    s/\x{E42C}/\x{1105}\x{1169}\x{11F0}/g;
    s/\x{E42D}/\x{1105}\x{1169}\x{11F1}/g;
    s/\x{E42E}/\x{1105}\x{116A}\x{11D9}/g;
    s/\x{E42F}/\x{1105}\x{116A}\x{11F0}/g;
    s/\x{E430}/\x{1105}\x{1182}/g;
    s/\x{E431}/\x{1105}\x{116D}\x{11D9}/g;
    s/\x{E432}/\x{1105}\x{116D}\x{11DD}/g;
    s/\x{E433}/\x{1105}\x{116D}\x{11E2}/g;
    s/\x{E434}/\x{1105}\x{116D}\x{11F0}/g;
    s/\x{E435}/\x{1105}\x{1186}/g;
    s/\x{E436}/\x{1105}\x{1187}/g;
    s/\x{E437}/\x{1105}\x{1188}/g;
    s/\x{E438}/\x{1105}\x{116E}\x{11D9}/g;
    s/\x{E439}/\x{1105}\x{116E}\x{11DD}/g;
    s/\x{E43A}/\x{1105}\x{116E}\x{11E2}/g;
    s/\x{E43B}/\x{1105}\x{116E}\x{11E4}/g;
    s/\x{E43C}/\x{1105}\x{116E}\x{11F0}/g;
    s/\x{E43D}/\x{1105}\x{116E}\x{11F9}/g;
    s/\x{E43E}/\x{1105}\x{1189}/g;
    s/\x{E43F}/\x{1105}\x{1189}\x{11AB}/g;
    s/\x{E440}/\x{1105}\x{116E}\x{1167}/g;
    s/\x{E441}/\x{1105}\x{118C}/g;
    s/\x{E442}/\x{1105}\x{118D}/g;
    s/\x{E443}/\x{1105}\x{1172}\x{11D9}/g;
    s/\x{E444}/\x{1105}\x{1172}\x{11E2}/g;
    s/\x{E445}/\x{1105}\x{1172}\x{11F0}/g;
    s/\x{E446}/\x{1105}\x{1172}\x{11F9}/g;
    s/\x{E447}/\x{1105}\x{1191}/g;
    s/\x{E448}/\x{1105}\x{1191}\x{11AB}/g;
    s/\x{E449}/\x{1105}\x{1191}\x{11AF}/g;
    s/\x{E44A}/\x{1105}\x{1191}\x{11D9}/g;
    s/\x{E44B}/\x{1105}\x{1194}/g;
    s/\x{E44C}/\x{1105}\x{1194}\x{11BC}/g;
    s/\x{E44D}/\x{1105}\x{1173}\x{11AE}\x{11BE}/g;
    s/\x{E44E}/\x{1105}\x{1173}\x{11D0}/g;
    s/\x{E44F}/\x{1105}\x{1173}\x{11D9}/g;
    s/\x{E450}/\x{1105}\x{1173}\x{11B7}\x{11B7}/g;
    s/\x{E451}/\x{1105}\x{1173}\x{11DD}/g;
    s/\x{E452}/\x{1105}\x{1173}\x{11E2}/g;
    s/\x{E453}/\x{1105}\x{1173}\x{11E6}/g;
    s/\x{E454}/\x{1105}\x{1173}\x{11F0}/g;
    s/\x{E455}/\x{1105}\x{1195}/g;
    s/\x{E456}/\x{1105}\x{1196}/g;
    s/\x{E457}/\x{1105}\x{1174}\x{11F9}/g;
    s/\x{E458}/\x{1105}\x{1175}\x{11C7}/g;
    s/\x{E459}/\x{1105}\x{1175}\x{11D9}/g;
    s/\x{E45A}/\x{1105}\x{1175}\x{11DD}/g;
    s/\x{E45B}/\x{1105}\x{1175}\x{11E2}/g;
    s/\x{E45C}/\x{1105}\x{1175}\x{11E6}/g;
    s/\x{E45D}/\x{1105}\x{1175}\x{11F0}/g;
    s/\x{E45E}/\x{1105}\x{1175}\x{11F9}/g;
    s/\x{E45F}/\x{1105}\x{1198}/g;
    s/\x{E460}/\x{1105}\x{1199}/g;
    s/\x{E461}/\x{1105}\x{1199}\x{11AB}/g;
    s/\x{E462}/\x{1105}\x{1199}\x{11BC}/g;
    s/\x{E463}/\x{1105}\x{1175}\x{1178}/g;
    s/\x{E464}/\x{1105}\x{1175}\x{1167}/g;
    s/\x{E465}/\x{1105}\x{1175}\x{1168}/g;
    s/\x{E466}/\x{1105}\x{119A}/g;
    s/\x{E467}/\x{1105}\x{119A}\x{11AF}/g;
    s/\x{E468}/\x{1105}\x{1175}\x{116D}/g;
    s/\x{E469}/\x{1105}\x{119B}/g;
    s/\x{E46A}/\x{1105}\x{119B}\x{11AF}/g;
    s/\x{E46B}/\x{1105}\x{1175}\x{1175}/g;
    s/\x{E46C}/\x{1105}\x{119E}/g;
    s/\x{E46D}/\x{1105}\x{119E}\x{11A8}/g;
    s/\x{E46E}/\x{1105}\x{119E}\x{11AB}/g;
    s/\x{E46F}/\x{1105}\x{119E}\x{11AE}/g;
    s/\x{E470}/\x{1105}\x{119E}\x{11AF}/g;
    s/\x{E471}/\x{1105}\x{119E}\x{11B0}/g;
    s/\x{E472}/\x{1105}\x{119E}\x{11B1}/g;
    s/\x{E473}/\x{1105}\x{119E}\x{11B3}/g;
    s/\x{E474}/\x{1105}\x{119E}\x{11D7}/g;
    s/\x{E475}/\x{1105}\x{119E}\x{11B6}/g;
    s/\x{E476}/\x{1105}\x{119E}\x{11D9}/g;
    s/\x{E477}/\x{1105}\x{119E}\x{11B7}/g;
    s/\x{E478}/\x{1105}\x{119E}\x{11B7}\x{11B7}/g;
    s/\x{E479}/\x{1105}\x{119E}\x{11DC}/g;
    s/\x{E47A}/\x{1105}\x{119E}\x{11DD}/g;
    s/\x{E47B}/\x{1105}\x{119E}\x{11B8}/g;
    s/\x{E47C}/\x{1105}\x{119E}\x{11B9}/g;
    s/\x{E47D}/\x{1105}\x{119E}\x{11E6}/g;
    s/\x{E47E}/\x{1105}\x{119E}\x{11BA}/g;
    s/\x{E47F}/\x{1105}\x{119E}\x{11BB}/g;
    s/\x{E480}/\x{1105}\x{119E}\x{11EB}/g;
    s/\x{E481}/\x{1105}\x{119E}\x{11F0}/g;
    s/\x{E482}/\x{1105}\x{119E}\x{11BD}/g;
    s/\x{E483}/\x{1105}\x{119E}\x{11BE}/g;
    s/\x{E484}/\x{1105}\x{119E}\x{11C0}/g;
    s/\x{E485}/\x{1105}\x{119E}\x{11C2}/g;
    s/\x{E486}/\x{1105}\x{119E}\x{1161}/g;
    s/\x{E487}/\x{1105}\x{11A1}/g;
    s/\x{E488}/\x{1105}\x{11A1}\x{11A8}/g;
    s/\x{E489}/\x{1105}\x{11A1}\x{11AB}/g;
    s/\x{E48A}/\x{1105}\x{11A1}\x{11AE}/g;
    s/\x{E48B}/\x{1105}\x{11A1}\x{11AF}/g;
    s/\x{E48C}/\x{1105}\x{11A1}\x{11B7}/g;
    s/\x{E48D}/\x{1105}\x{11A1}\x{11BA}/g;
    s/\x{E48E}/\x{1105}\x{11A1}\x{11BC}/g;
    s/\x{E48F}/\x{1105}\x{11A1}\x{11F0}/g;
    s/\x{E490}/\x{1105}\x{1100}\x{1161}/g;
    s/\x{E491}/\x{1105}\x{1100}\x{1169}/g;
    s/\x{E492}/\x{1105}\x{1100}\x{1169}\x{11BA}/g;
    s/\x{E493}/\x{1105}\x{1101}\x{1161}/g;
    s/\x{E494}/\x{1118}\x{1161}/g;
    s/\x{E495}/\x{1118}\x{1161}\x{11A8}/g;
    s/\x{E496}/\x{1118}\x{1161}\x{11AB}/g;
    s/\x{E497}/\x{1118}\x{1161}\x{11BC}/g;
    s/\x{E498}/\x{1118}\x{1165}/g;
    s/\x{E499}/\x{1118}\x{1166}/g;
    s/\x{E49A}/\x{1118}\x{1166}\x{11AF}/g;
    s/\x{E49B}/\x{1118}\x{1166}\x{11B8}/g;
    s/\x{E49C}/\x{1118}\x{1169}/g;
    s/\x{E49D}/\x{1118}\x{1169}\x{11A8}/g;
    s/\x{E49E}/\x{1118}\x{1169}\x{11AB}/g;
    s/\x{E49F}/\x{1118}\x{1169}\x{11B7}/g;
    s/\x{E4A0}/\x{1118}\x{1169}\x{11BA}/g;
    s/\x{E4A1}/\x{1118}\x{116D}/g;
    s/\x{E4A2}/\x{1118}\x{116D}\x{11AB}/g;
    s/\x{E4A3}/\x{1118}\x{116E}/g;
    s/\x{E4A4}/\x{1118}\x{1172}/g;
    s/\x{E4A5}/\x{1118}\x{1172}\x{11A8}/g;
    s/\x{E4A6}/\x{1118}\x{1173}/g;
    s/\x{E4A7}/\x{1118}\x{1174}/g;
    s/\x{E4A8}/\x{1118}\x{1174}\x{11A8}/g;
    s/\x{E4A9}/\x{1118}\x{1174}\x{11AB}/g;
    s/\x{E4AA}/\x{1118}\x{1174}\x{11B8}/g;
    s/\x{E4AB}/\x{1118}\x{119E}/g;
    s/\x{E4AC}/\x{1118}\x{119E}\x{11AB}/g;
    s/\x{E4AD}/\x{1118}\x{119E}\x{11BC}/g;
    s/\x{E4AE}/\x{1118}\x{11A1}/g;
    s/\x{E4AF}/\x{1118}\x{11A1}\x{11A8}/g;
    s/\x{E4B0}/\x{1118}\x{11A1}\x{11B7}/g;
    s/\x{E4B1}/\x{1105}\x{1104}\x{1161}/g;
    s/\x{E4B2}/\x{1119}\x{1161}/g;
    s/\x{E4B3}/\x{1119}\x{1161}\x{11A8}/g;
    s/\x{E4B4}/\x{1119}\x{1161}\x{11AB}/g;
    s/\x{E4B5}/\x{1119}\x{1161}\x{11BC}/g;
    s/\x{E4B6}/\x{1119}\x{1176}/g;
    s/\x{E4B7}/\x{1119}\x{1162}/g;
    s/\x{E4B8}/\x{1119}\x{1162}\x{11AB}/g;
    s/\x{E4B9}/\x{1119}\x{1163}/g;
    s/\x{E4BA}/\x{1119}\x{1163}\x{11BC}/g;
    s/\x{E4BB}/\x{1119}\x{1178}/g;
    s/\x{E4BC}/\x{1119}\x{1165}/g;
    s/\x{E4BD}/\x{1119}\x{1165}\x{11AB}/g;
    s/\x{E4BE}/\x{1119}\x{1165}\x{11BC}/g;
    s/\x{E4BF}/\x{1119}\x{117B}/g;
    s/\x{E4C0}/\x{1119}\x{1166}/g;
    s/\x{E4C1}/\x{1119}\x{1166}\x{11A8}/g;
    s/\x{E4C2}/\x{1119}\x{1167}/g;
    s/\x{E4C3}/\x{1119}\x{1167}\x{11A8}/g;
    s/\x{E4C4}/\x{1119}\x{1168}/g;
    s/\x{E4C5}/\x{1119}\x{1168}\x{11AB}/g;
    s/\x{E4C6}/\x{1119}\x{1169}/g;
    s/\x{E4C7}/\x{1119}\x{1169}\x{11BC}/g;
    s/\x{E4C8}/\x{1119}\x{116D}/g;
    s/\x{E4C9}/\x{1119}\x{116E}/g;
    s/\x{E4CA}/\x{1119}\x{116E}\x{11AB}/g;
    s/\x{E4CB}/\x{1119}\x{116E}\x{11BC}/g;
    s/\x{E4CC}/\x{1119}\x{1189}/g;
    s/\x{E4CD}/\x{1119}\x{1189}\x{11AB}/g;
    s/\x{E4CE}/\x{1119}\x{116F}/g;
    s/\x{E4CF}/\x{1119}\x{1170}/g;
    s/\x{E4D0}/\x{1119}\x{1171}/g;
    s/\x{E4D1}/\x{1119}\x{1172}/g;
    s/\x{E4D2}/\x{1119}\x{118F}/g;
    s/\x{E4D3}/\x{1119}\x{1190}/g;
    s/\x{E4D4}/\x{1119}\x{1173}/g;
    s/\x{E4D5}/\x{1119}\x{1173}\x{11AF}/g;
    s/\x{E4D6}/\x{1119}\x{1173}\x{11D0}/g;
    s/\x{E4D7}/\x{1119}\x{1173}\x{11BC}/g;
    s/\x{E4D8}/\x{1119}\x{1175}/g;
    s/\x{E4D9}/\x{1119}\x{1175}\x{11AB}/g;
    s/\x{E4DA}/\x{1119}\x{1175}\x{11BC}/g;
    s/\x{E4DB}/\x{1119}\x{119B}/g;
    s/\x{E4DC}/\x{1105}\x{1106}\x{1161}/g;
    s/\x{E4DD}/\x{1105}\x{1106}\x{1161}\x{11AB}/g;
    s/\x{E4DE}/\x{1105}\x{1107}\x{1161}/g;
    s/\x{E4DF}/\x{1105}\x{1108}\x{1161}/g;
    s/\x{E4E0}/\x{1105}\x{112B}\x{119E}/g;
    s/\x{E4E1}/\x{1105}\x{112B}\x{11A1}/g;
    s/\x{E4E2}/\x{1105}\x{1109}\x{1169}/g;
    s/\x{E4E3}/\x{1105}\x{1109}\x{1169}\x{11A8}/g;
    s/\x{E4E4}/\x{1105}\x{1109}\x{1173}/g;
    s/\x{E4E5}/\x{1105}\x{110C}\x{1165}/g;
    s/\x{E4E6}/\x{1105}\x{110C}\x{1166}/g;
    s/\x{E4E7}/\x{1105}\x{110C}\x{1172}/g;
    s/\x{E4E8}/\x{1105}\x{110C}\x{1172}\x{11AF}/g;
    s/\x{E4E9}/\x{1105}\x{110F}\x{1161}/g;
    s/\x{E4EA}/\x{111A}\x{1161}/g;
    s/\x{E4EB}/\x{111A}\x{1161}\x{11AB}/g;
    s/\x{E4EC}/\x{111A}\x{1163}/g;
    s/\x{E4ED}/\x{111A}\x{1163}\x{11BC}/g;
    s/\x{E4EE}/\x{111A}\x{1169}/g;
    s/\x{E4EF}/\x{111A}\x{116A}/g;
    s/\x{E4F0}/\x{111A}\x{1175}/g;
    s/\x{E4F1}/\x{1106}\x{1161}\x{11A8}\x{11C2}/g;
    s/\x{E4F2}/\x{1106}\x{1161}\x{11C6}/g;
    s/\x{E4F3}/\x{1106}\x{1161}\x{11C7}/g;
    s/\x{E4F4}/\x{1106}\x{1161}\x{11AE}\x{11C0}/g;
    s/\x{E4F5}/\x{1106}\x{1161}\x{11CD}/g;
    s/\x{E4F6}/\x{1106}\x{1161}\x{11D0}/g;
    s/\x{E4F7}/\x{1106}\x{1161}\x{11D8}/g;
    s/\x{E4F8}/\x{1106}\x{1161}\x{11D9}/g;
    s/\x{E4F9}/\x{1106}\x{1161}\x{11E2}/g;
    s/\x{E4FA}/\x{1106}\x{1161}\x{11E6}/g;
    s/\x{E4FB}/\x{1106}\x{1161}\x{11E8}/g;
    s/\x{E4FC}/\x{1106}\x{1161}\x{11BA}\x{11E8}/g;
    s/\x{E4FD}/\x{1106}\x{1161}\x{11BA}\x{11BE}/g;
    s/\x{E4FE}/\x{1106}\x{1161}\x{11BA}\x{11C0}/g;
    s/\x{E4FF}/\x{1106}\x{1161}\x{11EB}/g;
    s/\x{E500}/\x{1106}\x{1161}\x{11F0}/g;
    s/\x{E501}/\x{1106}\x{1161}\x{11F9}/g;
    s/\x{E502}/\x{1106}\x{1176}/g;
    s/\x{E503}/\x{1106}\x{1177}/g;
    s/\x{E504}/\x{1106}\x{1162}\x{11F0}/g;
    s/\x{E505}/\x{1106}\x{1163}\x{11E2}/g;
    s/\x{E506}/\x{1106}\x{1178}/g;
    s/\x{E507}/\x{1106}\x{1165}\x{11C6}/g;
    s/\x{E508}/\x{1106}\x{1165}\x{11CD}/g;
    s/\x{E509}/\x{1106}\x{1165}\x{11D0}/g;
    s/\x{E50A}/\x{1106}\x{1165}\x{11D9}/g;
    s/\x{E50B}/\x{1106}\x{1165}\x{11F0}/g;
    s/\x{E50C}/\x{1106}\x{1165}\x{11F9}/g;
    s/\x{E50D}/\x{1106}\x{117A}/g;
    s/\x{E50E}/\x{1106}\x{117C}/g;
    s/\x{E50F}/\x{1106}\x{117C}\x{11AF}/g;
    s/\x{E510}/\x{1106}\x{1166}\x{11D9}/g;
    s/\x{E511}/\x{1106}\x{1167}\x{11D9}/g;
    s/\x{E512}/\x{1106}\x{1167}\x{11E2}/g;
    s/\x{E513}/\x{1106}\x{1167}\x{11EB}/g;
    s/\x{E514}/\x{1106}\x{1167}\x{11F0}/g;
    s/\x{E515}/\x{1106}\x{1167}\x{11F0}\x{11C2}/g;
    s/\x{E516}/\x{1106}\x{1167}\x{11F9}/g;
    s/\x{E517}/\x{1106}\x{1169}\x{11C6}/g;
    s/\x{E518}/\x{1106}\x{1169}\x{11D0}/g;
    s/\x{E519}/\x{1106}\x{1169}\x{11D9}/g;
    s/\x{E51A}/\x{1106}\x{1169}\x{11DD}/g;
    s/\x{E51B}/\x{1106}\x{1169}\x{11E2}/g;
    s/\x{E51C}/\x{1106}\x{1169}\x{11BA}\x{11BE}/g;
    s/\x{E51D}/\x{1106}\x{1169}\x{11EB}/g;
    s/\x{E51E}/\x{1106}\x{1169}\x{11F0}/g;
    s/\x{E51F}/\x{1106}\x{117F}/g;
    s/\x{E520}/\x{1106}\x{117F}\x{11AB}/g;
    s/\x{E521}/\x{1106}\x{1182}/g;
    s/\x{E522}/\x{1106}\x{1182}\x{116E}/g;
    s/\x{E523}/\x{1106}\x{1183}/g;
    s/\x{E524}/\x{1106}\x{116D}\x{11E2}/g;
    s/\x{E525}/\x{1106}\x{1187}/g;
    s/\x{E526}/\x{1106}\x{1188}/g;
    s/\x{E527}/\x{1106}\x{116E}\x{11C6}/g;
    s/\x{E528}/\x{1106}\x{116E}\x{11C7}/g;
    s/\x{E529}/\x{1106}\x{116E}\x{11D0}/g;
    s/\x{E52A}/\x{1106}\x{116E}\x{11D9}/g;
    s/\x{E52B}/\x{1106}\x{116E}\x{11DD}/g;
    s/\x{E52C}/\x{1106}\x{116E}\x{11E2}/g;
    s/\x{E52D}/\x{1106}\x{116E}\x{11E7}/g;
    s/\x{E52E}/\x{1106}\x{116E}\x{11BA}\x{11E7}/g;
    s/\x{E52F}/\x{1106}\x{116E}\x{11BA}\x{11C0}/g;
    s/\x{E530}/\x{1106}\x{116E}\x{11EB}/g;
    s/\x{E531}/\x{1106}\x{116E}\x{11EC}/g;
    s/\x{E532}/\x{1106}\x{116E}\x{11ED}/g;
    s/\x{E533}/\x{1106}\x{116E}\x{11F0}/g;
    s/\x{E534}/\x{1106}\x{116E}\x{11F9}/g;
    s/\x{E535}/\x{1106}\x{116F}\x{11F9}/g;
    s/\x{E536}/\x{1106}\x{116E}\x{1167}/g;
    s/\x{E537}/\x{1106}\x{118C}/g;
    s/\x{E538}/\x{1106}\x{118D}/g;
    s/\x{E539}/\x{1106}\x{1171}\x{11F9}/g;
    s/\x{E53A}/\x{1106}\x{1172}\x{11E2}/g;
    s/\x{E53B}/\x{1106}\x{1194}/g;
    s/\x{E53C}/\x{1106}\x{1173}\x{11C4}/g;
    s/\x{E53D}/\x{1106}\x{1173}\x{11D0}/g;
    s/\x{E53E}/\x{1106}\x{1173}\x{11D9}/g;
    s/\x{E53F}/\x{1106}\x{1173}\x{11E2}/g;
    s/\x{E540}/\x{1106}\x{1173}\x{11E7}/g;
    s/\x{E541}/\x{1106}\x{1173}\x{11BA}\x{11C2}/g;
    s/\x{E542}/\x{1106}\x{1173}\x{11EB}/g;
    s/\x{E543}/\x{1106}\x{1173}\x{11F0}/g;
    s/\x{E544}/\x{1106}\x{1195}/g;
    s/\x{E545}/\x{1106}\x{1196}/g;
    s/\x{E546}/\x{1106}\x{1196}\x{11B7}/g;
    s/\x{E547}/\x{1106}\x{1174}\x{11D9}/g;
    s/\x{E548}/\x{1106}\x{1174}\x{11E6}/g;
    s/\x{E549}/\x{1106}\x{1174}\x{11F9}/g;
    s/\x{E54A}/\x{1106}\x{1175}\x{11AE}\x{11B8}/g;
    s/\x{E54B}/\x{1106}\x{1175}\x{11AE}\x{11BE}/g;
    s/\x{E54C}/\x{1106}\x{1175}\x{11AE}\x{11C0}/g;
    s/\x{E54D}/\x{1106}\x{1175}\x{11CD}/g;
    s/\x{E54E}/\x{1106}\x{1175}\x{11D0}/g;
    s/\x{E54F}/\x{1106}\x{1175}\x{11D9}/g;
    s/\x{E550}/\x{1106}\x{1175}\x{11E6}/g;
    s/\x{E551}/\x{1106}\x{1175}\x{11E7}/g;
    s/\x{E552}/\x{1106}\x{1175}\x{11BA}\x{11BE}/g;
    s/\x{E553}/\x{1106}\x{1175}\x{11BA}\x{11C0}/g;
    s/\x{E554}/\x{1106}\x{1175}\x{11BA}\x{11C2}/g;
    s/\x{E555}/\x{1106}\x{1175}\x{11EB}/g;
    s/\x{E556}/\x{1106}\x{1175}\x{11F0}/g;
    s/\x{E557}/\x{1106}\x{1175}\x{11F6}/g;
    s/\x{E558}/\x{1106}\x{1175}\x{11F9}/g;
    s/\x{E559}/\x{1106}\x{1198}/g;
    s/\x{E55A}/\x{1106}\x{1199}/g;
    s/\x{E55B}/\x{1106}\x{1199}\x{11AB}/g;
    s/\x{E55C}/\x{1106}\x{1175}\x{1167}/g;
    s/\x{E55D}/\x{1106}\x{1175}\x{1168}/g;
    s/\x{E55E}/\x{1106}\x{119C}/g;
    s/\x{E55F}/\x{1106}\x{119C}\x{11BC}/g;
    s/\x{E560}/\x{1106}\x{119E}/g;
    s/\x{E561}/\x{1106}\x{119E}\x{11A8}/g;
    s/\x{E562}/\x{1106}\x{119E}\x{11C3}/g;
    s/\x{E563}/\x{1106}\x{119E}\x{11AB}/g;
    s/\x{E564}/\x{1106}\x{119E}\x{11AE}/g;
    s/\x{E565}/\x{1106}\x{119E}\x{11AE}\x{11BE}/g;
    s/\x{E566}/\x{1106}\x{119E}\x{11AF}/g;
    s/\x{E567}/\x{1106}\x{119E}\x{11B0}/g;
    s/\x{E568}/\x{1106}\x{119E}\x{11D0}/g;
    s/\x{E569}/\x{1106}\x{119E}\x{11B2}/g;
    s/\x{E56A}/\x{1106}\x{119E}\x{11D5}/g;
    s/\x{E56B}/\x{1106}\x{119E}\x{11B3}/g;
    s/\x{E56C}/\x{1106}\x{119E}\x{11B6}/g;
    s/\x{E56D}/\x{1106}\x{119E}\x{11D9}/g;
    s/\x{E56E}/\x{1106}\x{119E}\x{11B7}/g;
    s/\x{E56F}/\x{1106}\x{119E}\x{11B8}/g;
    s/\x{E570}/\x{1106}\x{119E}\x{11BA}/g;
    s/\x{E571}/\x{1106}\x{119E}\x{11BA}\x{11BE}/g;
    s/\x{E572}/\x{1106}\x{119E}\x{11EB}/g;
    s/\x{E573}/\x{1106}\x{119E}\x{11BC}/g;
    s/\x{E574}/\x{1106}\x{119E}\x{11F0}/g;
    s/\x{E575}/\x{1106}\x{119E}\x{11BD}/g;
    s/\x{E576}/\x{1106}\x{119E}\x{11BE}/g;
    s/\x{E577}/\x{1106}\x{119E}\x{11C0}/g;
    s/\x{E578}/\x{1106}\x{119F}/g;
    s/\x{E579}/\x{1106}\x{119F}\x{11AB}/g;
    s/\x{E57A}/\x{1106}\x{119F}\x{11BC}/g;
    s/\x{E57B}/\x{1106}\x{11A1}/g;
    s/\x{E57C}/\x{1106}\x{11A1}\x{11A8}/g;
    s/\x{E57D}/\x{1106}\x{11A1}\x{11A9}/g;
    s/\x{E57E}/\x{1106}\x{11A1}\x{11AB}/g;
    s/\x{E57F}/\x{1106}\x{11A1}\x{11AE}/g;
    s/\x{E580}/\x{1106}\x{11A1}\x{11AF}/g;
    s/\x{E581}/\x{1106}\x{11A1}\x{11B3}/g;
    s/\x{E582}/\x{1106}\x{11A1}\x{11D9}/g;
    s/\x{E583}/\x{1106}\x{11A1}\x{11B7}/g;
    s/\x{E584}/\x{1106}\x{11A1}\x{11B8}/g;
    s/\x{E585}/\x{1106}\x{11A1}\x{11E6}/g;
    s/\x{E586}/\x{1106}\x{11A1}\x{11BA}/g;
    s/\x{E587}/\x{1106}\x{11A1}\x{11BC}/g;
    s/\x{E588}/\x{1106}\x{11A1}\x{11F0}/g;
    s/\x{E589}/\x{1106}\x{11A1}\x{11BD}/g;
    s/\x{E58A}/\x{1106}\x{11A1}\x{11BE}/g;
    s/\x{E58B}/\x{1106}\x{11A1}\x{11C2}/g;
    s/\x{E58C}/\x{1106}\x{1100}\x{1163}/g;
    s/\x{E58D}/\x{1106}\x{1100}\x{1178}/g;
    s/\x{E58E}/\x{1106}\x{1100}\x{1167}/g;
    s/\x{E58F}/\x{1106}\x{1100}\x{1168}/g;
    s/\x{E590}/\x{1106}\x{1103}\x{1167}/g;
    s/\x{E591}/\x{1106}\x{1103}\x{1167}\x{11AF}/g;
    s/\x{E592}/\x{1106}\x{1103}\x{1169}/g;
    s/\x{E593}/\x{111C}\x{1161}/g;
    s/\x{E594}/\x{111C}\x{1161}\x{11AB}/g;
    s/\x{E595}/\x{111C}\x{1161}\x{11AE}/g;
    s/\x{E596}/\x{111C}\x{1162}/g;
    s/\x{E597}/\x{111C}\x{1163}/g;
    s/\x{E598}/\x{111C}\x{1165}/g;
    s/\x{E599}/\x{111C}\x{1166}/g;
    s/\x{E59A}/\x{111C}\x{1167}/g;
    s/\x{E59B}/\x{111C}\x{1168}/g;
    s/\x{E59C}/\x{111C}\x{1168}\x{11AB}/g;
    s/\x{E59D}/\x{111C}\x{1168}\x{11AE}/g;
    s/\x{E59E}/\x{111C}\x{1169}/g;
    s/\x{E59F}/\x{111C}\x{1169}\x{11A8}/g;
    s/\x{E5A0}/\x{111C}\x{1169}\x{11AB}/g;
    s/\x{E5A1}/\x{111C}\x{1169}\x{11AE}/g;
    s/\x{E5A2}/\x{111C}\x{116A}/g;
    s/\x{E5A3}/\x{111C}\x{116B}/g;
    s/\x{E5A4}/\x{111C}\x{116C}/g;
    s/\x{E5A5}/\x{111C}\x{116D}/g;
    s/\x{E5A6}/\x{111C}\x{116E}/g;
    s/\x{E5A7}/\x{111C}\x{116E}\x{11AB}/g;
    s/\x{E5A8}/\x{111C}\x{116E}\x{11BC}/g;
    s/\x{E5A9}/\x{111C}\x{116F}/g;
    s/\x{E5AA}/\x{111C}\x{1170}/g;
    s/\x{E5AB}/\x{111C}\x{116E}\x{1167}/g;
    s/\x{E5AC}/\x{111C}\x{118C}/g;
    s/\x{E5AD}/\x{111C}\x{1171}/g;
    s/\x{E5AE}/\x{111C}\x{1172}/g;
    s/\x{E5AF}/\x{111C}\x{1194}/g;
    s/\x{E5B0}/\x{111C}\x{1173}/g;
    s/\x{E5B1}/\x{111C}\x{1174}/g;
    s/\x{E5B2}/\x{111C}\x{1175}/g;
    s/\x{E5B3}/\x{111C}\x{1175}\x{11AB}/g;
    s/\x{E5B4}/\x{111C}\x{1198}/g;
    s/\x{E5B5}/\x{111C}\x{119E}/g;
    s/\x{E5B6}/\x{111C}\x{11A1}/g;
    s/\x{E5B7}/\x{1106}\x{1109}\x{1169}/g;
    s/\x{E5B8}/\x{1106}\x{1109}\x{1169}\x{11A8}/g;
    s/\x{E5B9}/\x{111D}\x{1161}/g;
    s/\x{E5BA}/\x{111D}\x{1161}\x{11AB}/g;
    s/\x{E5BB}/\x{111D}\x{1161}\x{11BC}/g;
    s/\x{E5BC}/\x{111D}\x{1161}\x{11F0}/g;
    s/\x{E5BD}/\x{111D}\x{1169}/g;
    s/\x{E5BE}/\x{111D}\x{116A}/g;
    s/\x{E5BF}/\x{111D}\x{116A}\x{11AB}/g;
    s/\x{E5C0}/\x{111D}\x{116A}\x{11F9}/g;
    s/\x{E5C1}/\x{111D}\x{116E}/g;
    s/\x{E5C2}/\x{111D}\x{116E}\x{11AB}/g;
    s/\x{E5C3}/\x{111D}\x{116E}\x{11BC}/g;
    s/\x{E5C4}/\x{111D}\x{116E}\x{11F9}/g;
    s/\x{E5C5}/\x{111D}\x{1171}/g;
    s/\x{E5C6}/\x{111D}\x{1172}/g;
    s/\x{E5C7}/\x{111D}\x{1172}\x{11AB}/g;
    s/\x{E5C8}/\x{111D}\x{1173}/g;
    s/\x{E5C9}/\x{111D}\x{1173}\x{11AB}/g;
    s/\x{E5CA}/\x{111D}\x{1173}\x{11F9}/g;
    s/\x{E5CB}/\x{111D}\x{1173}\x{1161}\x{11AB}/g;
    s/\x{E5CC}/\x{111D}\x{1175}/g;
    s/\x{E5CD}/\x{111D}\x{1175}\x{11BC}/g;
    s/\x{E5CE}/\x{1107}\x{1161}\x{11C3}/g;
    s/\x{E5CF}/\x{1107}\x{1161}\x{11A8}\x{11BE}/g;
    s/\x{E5D0}/\x{1107}\x{1161}\x{11AB}\x{11AF}/g;
    s/\x{E5D1}/\x{1107}\x{1161}\x{11C7}/g;
    s/\x{E5D2}/\x{1107}\x{1161}\x{11C9}/g;
    s/\x{E5D3}/\x{1107}\x{1161}\x{11CA}/g;
    s/\x{E5D4}/\x{1107}\x{1161}\x{11AE}\x{11BA}/g;
    s/\x{E5D5}/\x{1107}\x{1161}\x{11AE}\x{11E7}/g;
    s/\x{E5D6}/\x{1107}\x{1161}\x{11AE}\x{11C0}/g;
    s/\x{E5D7}/\x{1107}\x{1161}\x{11D0}/g;
    s/\x{E5D8}/\x{1107}\x{1161}\x{11D9}/g;
    s/\x{E5D9}/\x{1107}\x{1161}\x{11B7}\x{11B7}/g;
    s/\x{E5DA}/\x{1107}\x{1161}\x{11DC}/g;
    s/\x{E5DB}/\x{1107}\x{1161}\x{11DD}/g;
    s/\x{E5DC}/\x{1107}\x{1161}\x{11E2}/g;
    s/\x{E5DD}/\x{1107}\x{1161}\x{11E3}/g;
    s/\x{E5DE}/\x{1107}\x{1161}\x{11E6}/g;
    s/\x{E5DF}/\x{1107}\x{1161}\x{11E7}/g;
    s/\x{E5E0}/\x{1107}\x{1161}\x{11E8}/g;
    s/\x{E5E1}/\x{1107}\x{1161}\x{11EA}/g;
    s/\x{E5E2}/\x{1107}\x{1161}\x{11BA}\x{11BE}/g;
    s/\x{E5E3}/\x{1107}\x{1161}\x{11EB}/g;
    s/\x{E5E4}/\x{1107}\x{1161}\x{11BC}\x{11BA}/g;
    s/\x{E5E5}/\x{1107}\x{1161}\x{11F0}/g;
    s/\x{E5E6}/\x{1107}\x{1161}\x{11F1}/g;
    s/\x{E5E7}/\x{1107}\x{1161}\x{11F6}/g;
    s/\x{E5E8}/\x{1107}\x{1161}\x{11F9}/g;
    s/\x{E5E9}/\x{1107}\x{1176}/g;
    s/\x{E5EA}/\x{1107}\x{1176}\x{11AF}/g;
    s/\x{E5EB}/\x{1107}\x{1177}/g;
    s/\x{E5EC}/\x{1107}\x{1177}\x{11AF}/g;
    s/\x{E5ED}/\x{1107}\x{1162}\x{11F0}/g;
    s/\x{E5EE}/\x{1107}\x{1163}\x{11E2}/g;
    s/\x{E5EF}/\x{1107}\x{1178}/g;
    s/\x{E5F0}/\x{1107}\x{1165}\x{11C7}/g;
    s/\x{E5F1}/\x{1107}\x{1165}\x{11CD}/g;
    s/\x{E5F2}/\x{1107}\x{1165}\x{11D9}/g;
    s/\x{E5F3}/\x{1107}\x{1165}\x{11EB}/g;
    s/\x{E5F4}/\x{1107}\x{1165}\x{11F0}/g;
    s/\x{E5F5}/\x{1107}\x{117B}/g;
    s/\x{E5F6}/\x{1107}\x{117B}\x{11AF}/g;
    s/\x{E5F7}/\x{1107}\x{117C}/g;
    s/\x{E5F8}/\x{1107}\x{117C}\x{11A8}/g;
    s/\x{E5F9}/\x{1107}\x{117C}\x{11AF}/g;
    s/\x{E5FA}/\x{1107}\x{117C}\x{11B7}/g;
    s/\x{E5FB}/\x{1107}\x{117C}\x{11BC}/g;
    s/\x{E5FC}/\x{1107}\x{1167}\x{11D9}/g;
    s/\x{E5FD}/\x{1107}\x{1167}\x{11E2}/g;
    s/\x{E5FE}/\x{1107}\x{1167}\x{11BA}\x{11C0}/g;
    s/\x{E5FF}/\x{1107}\x{1167}\x{11F0}/g;
    s/\x{E600}/\x{1107}\x{1167}\x{11F1}/g;
    s/\x{E601}/\x{1107}\x{1167}\x{11F9}/g;
    s/\x{E602}/\x{1107}\x{1169}\x{11D0}/g;
    s/\x{E603}/\x{1107}\x{1169}\x{11D9}/g;
    s/\x{E604}/\x{1107}\x{1169}\x{11B7}\x{11B7}/g;
    s/\x{E605}/\x{1107}\x{1169}\x{11DC}/g;
    s/\x{E606}/\x{1107}\x{1169}\x{11DD}/g;
    s/\x{E607}/\x{1107}\x{1169}\x{11E2}/g;
    s/\x{E608}/\x{1107}\x{1169}\x{11E7}/g;
    s/\x{E609}/\x{1107}\x{1169}\x{11BA}\x{11E7}/g;
    s/\x{E60A}/\x{1107}\x{1169}\x{11EB}/g;
    s/\x{E60B}/\x{1107}\x{1169}\x{11F0}/g;
    s/\x{E60C}/\x{1107}\x{1182}/g;
    s/\x{E60D}/\x{1107}\x{1182}\x{116E}/g;
    s/\x{E60E}/\x{1107}\x{116C}\x{11D9}/g;
    s/\x{E60F}/\x{1107}\x{116C}\x{11BA}\x{11BE}/g;
    s/\x{E610}/\x{1107}\x{116D}\x{11D9}/g;
    s/\x{E611}/\x{1107}\x{116D}\x{11E2}/g;
    s/\x{E612}/\x{1107}\x{116D}\x{11F0}/g;
    s/\x{E613}/\x{1107}\x{1187}/g;
    s/\x{E614}/\x{1107}\x{1188}/g;
    s/\x{E615}/\x{1107}\x{116E}\x{11C7}/g;
    s/\x{E616}/\x{1107}\x{116E}\x{11D0}/g;
    s/\x{E617}/\x{1107}\x{116E}\x{11D1}/g;
    s/\x{E618}/\x{1107}\x{116E}\x{11D2}/g;
    s/\x{E619}/\x{1107}\x{116E}\x{11D9}/g;
    s/\x{E61A}/\x{1107}\x{116E}\x{11E2}/g;
    s/\x{E61B}/\x{1107}\x{116E}\x{11E6}/g;
    s/\x{E61C}/\x{1107}\x{116E}\x{11E7}/g;
    s/\x{E61D}/\x{1107}\x{116E}\x{11BA}\x{11BE}/g;
    s/\x{E61E}/\x{1107}\x{116E}\x{11EB}/g;
    s/\x{E61F}/\x{1107}\x{116E}\x{11EF}/g;
    s/\x{E620}/\x{1107}\x{116E}\x{11F0}/g;
    s/\x{E621}/\x{1107}\x{116E}\x{11F9}/g;
    s/\x{E622}/\x{1107}\x{116F}\x{11F9}/g;
    s/\x{E623}/\x{1107}\x{116E}\x{1167}/g;
    s/\x{E624}/\x{1107}\x{118C}/g;
    s/\x{E625}/\x{1107}\x{118D}/g;
    s/\x{E626}/\x{1107}\x{1171}\x{11D9}/g;
    s/\x{E627}/\x{1107}\x{1171}\x{11F9}/g;
    s/\x{E628}/\x{1107}\x{1172}\x{11D9}/g;
    s/\x{E629}/\x{1107}\x{1172}\x{11E2}/g;
    s/\x{E62A}/\x{1107}\x{1194}/g;
    s/\x{E62B}/\x{1107}\x{1173}\x{11C3}/g;
    s/\x{E62C}/\x{1107}\x{1173}\x{11AE}\x{11C0}/g;
    s/\x{E62D}/\x{1107}\x{1173}\x{11D0}/g;
    s/\x{E62E}/\x{1107}\x{1173}\x{11D9}/g;
    s/\x{E62F}/\x{1107}\x{1173}\x{11DD}/g;
    s/\x{E630}/\x{1107}\x{1173}\x{11E7}/g;
    s/\x{E631}/\x{1107}\x{1173}\x{11BA}\x{11C0}/g;
    s/\x{E632}/\x{1107}\x{1173}\x{11EB}/g;
    s/\x{E633}/\x{1107}\x{1173}\x{11F0}/g;
    s/\x{E634}/\x{1107}\x{1173}\x{11F9}/g;
    s/\x{E635}/\x{1107}\x{1196}/g;
    s/\x{E636}/\x{1107}\x{1174}\x{11F9}/g;
    s/\x{E637}/\x{1107}\x{1175}\x{11CA}/g;
    s/\x{E638}/\x{1107}\x{1175}\x{11AE}\x{11BA}/g;
    s/\x{E639}/\x{1107}\x{1175}\x{11D9}/g;
    s/\x{E63A}/\x{1107}\x{1175}\x{11E2}/g;
    s/\x{E63B}/\x{1107}\x{1175}\x{11E7}/g;
    s/\x{E63C}/\x{1107}\x{1175}\x{11BA}\x{11EB}/g;
    s/\x{E63D}/\x{1107}\x{1175}\x{11BA}\x{11BE}/g;
    s/\x{E63E}/\x{1107}\x{1175}\x{11BA}\x{11C2}/g;
    s/\x{E63F}/\x{1107}\x{1175}\x{11EB}/g;
    s/\x{E640}/\x{1107}\x{1175}\x{11F0}/g;
    s/\x{E641}/\x{1107}\x{1175}\x{11F1}/g;
    s/\x{E642}/\x{1107}\x{1175}\x{11F9}/g;
    s/\x{E643}/\x{1107}\x{1198}/g;
    s/\x{E644}/\x{1107}\x{1199}/g;
    s/\x{E645}/\x{1107}\x{1199}\x{11AB}/g;
    s/\x{E646}/\x{1107}\x{1175}\x{1167}/g;
    s/\x{E647}/\x{1107}\x{1175}\x{1168}/g;
    s/\x{E648}/\x{1107}\x{119A}/g;
    s/\x{E649}/\x{1107}\x{119B}/g;
    s/\x{E64A}/\x{1107}\x{119E}/g;
    s/\x{E64B}/\x{1107}\x{119E}\x{11A8}/g;
    s/\x{E64C}/\x{1107}\x{119E}\x{11C3}/g;
    s/\x{E64D}/\x{1107}\x{119E}\x{11AB}/g;
    s/\x{E64E}/\x{1107}\x{119E}\x{11AE}/g;
    s/\x{E64F}/\x{1107}\x{119E}\x{11AF}/g;
    s/\x{E650}/\x{1107}\x{119E}\x{11B0}/g;
    s/\x{E651}/\x{1107}\x{119E}\x{11D0}/g;
    s/\x{E652}/\x{1107}\x{119E}\x{11B1}/g;
    s/\x{E653}/\x{1107}\x{119E}\x{11B2}/g;
    s/\x{E654}/\x{1107}\x{119E}\x{11D5}/g;
    s/\x{E655}/\x{1107}\x{119E}\x{11B3}/g;
    s/\x{E656}/\x{1107}\x{119E}\x{11B4}/g;
    s/\x{E657}/\x{1107}\x{119E}\x{11B6}/g;
    s/\x{E658}/\x{1107}\x{119E}\x{11D9}/g;
    s/\x{E659}/\x{1107}\x{119E}\x{11B7}/g;
    s/\x{E65A}/\x{1107}\x{119E}\x{11B8}/g;
    s/\x{E65B}/\x{1107}\x{119E}\x{11E3}/g;
    s/\x{E65C}/\x{1107}\x{119E}\x{11BA}/g;
    s/\x{E65D}/\x{1107}\x{119E}\x{11BA}\x{11C0}/g;
    s/\x{E65E}/\x{1107}\x{119E}\x{11EB}/g;
    s/\x{E65F}/\x{1107}\x{119E}\x{11F0}/g;
    s/\x{E660}/\x{1107}\x{119E}\x{11BD}/g;
    s/\x{E661}/\x{1107}\x{119E}\x{11BE}/g;
    s/\x{E662}/\x{1107}\x{119E}\x{11C0}/g;
    s/\x{E663}/\x{1107}\x{119F}/g;
    s/\x{E664}/\x{1107}\x{119F}\x{11AB}/g;
    s/\x{E665}/\x{1107}\x{119F}\x{11AF}/g;
    s/\x{E666}/\x{1107}\x{119F}\x{11BC}/g;
    s/\x{E667}/\x{1107}\x{119E}\x{1166}/g;
    s/\x{E668}/\x{1107}\x{11A1}/g;
    s/\x{E669}/\x{1107}\x{11A1}\x{11A8}/g;
    s/\x{E66A}/\x{1107}\x{11A1}\x{11AB}/g;
    s/\x{E66B}/\x{1107}\x{11A1}\x{11AE}/g;
    s/\x{E66C}/\x{1107}\x{11A1}\x{11AF}/g;
    s/\x{E66D}/\x{1107}\x{11A1}\x{11B0}/g;
    s/\x{E66E}/\x{1107}\x{11A1}\x{11D9}/g;
    s/\x{E66F}/\x{1107}\x{11A1}\x{11B7}/g;
    s/\x{E670}/\x{1107}\x{11A1}\x{11B8}/g;
    s/\x{E671}/\x{1107}\x{11A1}\x{11BA}/g;
    s/\x{E672}/\x{1107}\x{11A1}\x{11BC}/g;
    s/\x{E673}/\x{1107}\x{11A1}\x{11F0}/g;
    s/\x{E674}/\x{1107}\x{11A1}\x{11F1}/g;
    s/\x{E675}/\x{1107}\x{11A1}\x{11BD}/g;
    s/\x{E676}/\x{1107}\x{11A1}\x{11BE}/g;
    s/\x{E677}/\x{1107}\x{11A1}\x{11C0}/g;
    s/\x{E678}/\x{1107}\x{11A1}\x{11C2}/g;
    s/\x{E679}/\x{111E}\x{1161}/g;
    s/\x{E67A}/\x{111E}\x{1161}\x{11A8}/g;
    s/\x{E67B}/\x{111E}\x{1161}\x{11B7}/g;
    s/\x{E67C}/\x{111E}\x{1162}/g;
    s/\x{E67D}/\x{111E}\x{1162}\x{11AB}/g;
    s/\x{E67E}/\x{111E}\x{1165}/g;
    s/\x{E67F}/\x{111E}\x{1165}\x{11BA}/g;
    s/\x{E680}/\x{111E}\x{1166}/g;
    s/\x{E681}/\x{111E}\x{1166}\x{11AF}/g;
    s/\x{E682}/\x{111E}\x{1167}/g;
    s/\x{E683}/\x{111E}\x{1167}\x{11AB}/g;
    s/\x{E684}/\x{111E}\x{1167}\x{11BA}/g;
    s/\x{E685}/\x{111E}\x{1168}/g;
    s/\x{E686}/\x{111E}\x{1168}\x{11AF}/g;
    s/\x{E687}/\x{111E}\x{1169}/g;
    s/\x{E688}/\x{111E}\x{1169}\x{11BA}/g;
    s/\x{E689}/\x{111E}\x{116C}/g;
    s/\x{E68A}/\x{111E}\x{116C}\x{11BA}/g;
    s/\x{E68B}/\x{111E}\x{116E}/g;
    s/\x{E68C}/\x{111E}\x{116E}\x{11AF}/g;
    s/\x{E68D}/\x{111E}\x{116E}\x{11B7}/g;
    s/\x{E68E}/\x{111E}\x{116E}\x{11BA}/g;
    s/\x{E68F}/\x{111E}\x{116F}/g;
    s/\x{E690}/\x{111E}\x{1170}/g;
    s/\x{E691}/\x{111E}\x{1171}/g;
    s/\x{E692}/\x{111E}\x{1173}/g;
    s/\x{E693}/\x{111E}\x{1173}\x{11AB}/g;
    s/\x{E694}/\x{111E}\x{1173}\x{11AE}/g;
    s/\x{E695}/\x{111E}\x{1173}\x{11AF}/g;
    s/\x{E696}/\x{111E}\x{1173}\x{11BA}/g;
    s/\x{E697}/\x{111E}\x{1174}/g;
    s/\x{E698}/\x{111E}\x{1174}\x{11BA}/g;
    s/\x{E699}/\x{111E}\x{1175}/g;
    s/\x{E69A}/\x{111E}\x{1175}\x{11AF}/g;
    s/\x{E69B}/\x{111E}\x{1175}\x{11B7}/g;
    s/\x{E69C}/\x{111E}\x{1175}\x{11BA}/g;
    s/\x{E69D}/\x{111E}\x{119E}/g;
    s/\x{E69E}/\x{111E}\x{119E}\x{11AF}/g;
    s/\x{E69F}/\x{111E}\x{11A1}/g;
    s/\x{E6A0}/\x{111E}\x{11A1}\x{11AB}/g;
    s/\x{E6A1}/\x{111E}\x{11A1}\x{11BA}/g;
    s/\x{E6A2}/\x{111F}\x{1167}/g;
    s/\x{E6A3}/\x{1120}\x{1161}/g;
    s/\x{E6A4}/\x{1120}\x{1161}\x{11A8}/g;
    s/\x{E6A5}/\x{1120}\x{1161}\x{11AB}/g;
    s/\x{E6A6}/\x{1120}\x{1161}\x{11AF}/g;
    s/\x{E6A7}/\x{1120}\x{1161}\x{11B7}/g;
    s/\x{E6A8}/\x{1120}\x{1161}\x{11B8}/g;
    s/\x{E6A9}/\x{1120}\x{1161}\x{11BA}/g;
    s/\x{E6AA}/\x{1120}\x{1162}/g;
    s/\x{E6AB}/\x{1120}\x{1163}/g;
    s/\x{E6AC}/\x{1120}\x{1165}/g;
    s/\x{E6AD}/\x{1120}\x{1165}\x{11A8}/g;
    s/\x{E6AE}/\x{1120}\x{1165}\x{11AB}/g;
    s/\x{E6AF}/\x{1120}\x{1165}\x{11AE}/g;
    s/\x{E6B0}/\x{1120}\x{1165}\x{11AF}/g;
    s/\x{E6B1}/\x{1120}\x{1165}\x{11B2}/g;
    s/\x{E6B2}/\x{1120}\x{1165}\x{11D5}/g;
    s/\x{E6B3}/\x{1120}\x{1165}\x{11B7}/g;
    s/\x{E6B4}/\x{1120}\x{1165}\x{11BA}/g;
    s/\x{E6B5}/\x{1120}\x{1166}/g;
    s/\x{E6B6}/\x{1120}\x{1166}\x{11AB}/g;
    s/\x{E6B7}/\x{1120}\x{1166}\x{11BA}/g;
    s/\x{E6B8}/\x{1120}\x{1167}/g;
    s/\x{E6B9}/\x{1120}\x{1167}\x{11BA}/g;
    s/\x{E6BA}/\x{1120}\x{1168}/g;
    s/\x{E6BB}/\x{1120}\x{1168}\x{11BA}/g;
    s/\x{E6BC}/\x{1120}\x{1169}/g;
    s/\x{E6BD}/\x{1120}\x{1169}\x{11A8}/g;
    s/\x{E6BE}/\x{1120}\x{1169}\x{11AF}/g;
    s/\x{E6BF}/\x{1120}\x{1169}\x{11B7}/g;
    s/\x{E6C0}/\x{1120}\x{1169}\x{11BC}/g;
    s/\x{E6C1}/\x{1120}\x{116A}/g;
    s/\x{E6C2}/\x{1120}\x{116B}/g;
    s/\x{E6C3}/\x{1120}\x{116C}/g;
    s/\x{E6C4}/\x{1120}\x{116C}\x{11BA}/g;
    s/\x{E6C5}/\x{1120}\x{116D}/g;
    s/\x{E6C6}/\x{1120}\x{116E}/g;
    s/\x{E6C7}/\x{1120}\x{116E}\x{11A8}/g;
    s/\x{E6C8}/\x{1120}\x{116E}\x{11AF}/g;
    s/\x{E6C9}/\x{1120}\x{116E}\x{11B2}/g;
    s/\x{E6CA}/\x{1120}\x{116E}\x{11B7}/g;
    s/\x{E6CB}/\x{1120}\x{116E}\x{11BA}/g;
    s/\x{E6CC}/\x{1120}\x{116E}\x{11BC}/g;
    s/\x{E6CD}/\x{1120}\x{116E}\x{11F0}/g;
    s/\x{E6CE}/\x{1120}\x{116F}/g;
    s/\x{E6CF}/\x{1120}\x{1170}/g;
    s/\x{E6D0}/\x{1120}\x{116E}\x{1167}/g;
    s/\x{E6D1}/\x{1120}\x{118C}/g;
    s/\x{E6D2}/\x{1120}\x{1171}/g;
    s/\x{E6D3}/\x{1120}\x{1171}\x{11AF}/g;
    s/\x{E6D4}/\x{1120}\x{1171}\x{11BA}/g;
    s/\x{E6D5}/\x{1120}\x{1172}/g;
    s/\x{E6D6}/\x{1120}\x{1194}/g;
    s/\x{E6D7}/\x{1120}\x{1173}/g;
    s/\x{E6D8}/\x{1120}\x{1173}\x{11A8}/g;
    s/\x{E6D9}/\x{1120}\x{1173}\x{11AB}/g;
    s/\x{E6DA}/\x{1120}\x{1173}\x{11AE}/g;
    s/\x{E6DB}/\x{1120}\x{1173}\x{11AF}/g;
    s/\x{E6DC}/\x{1120}\x{1173}\x{11B0}/g;
    s/\x{E6DD}/\x{1120}\x{1173}\x{11B2}/g;
    s/\x{E6DE}/\x{1120}\x{1173}\x{11B3}/g;
    s/\x{E6DF}/\x{1120}\x{1173}\x{11B6}/g;
    s/\x{E6E0}/\x{1120}\x{1173}\x{11B7}/g;
    s/\x{E6E1}/\x{1120}\x{1173}\x{11BA}/g;
    s/\x{E6E2}/\x{1120}\x{1173}\x{11BC}/g;
    s/\x{E6E3}/\x{1120}\x{1173}\x{11BD}/g;
    s/\x{E6E4}/\x{1120}\x{1174}/g;
    s/\x{E6E5}/\x{1120}\x{1174}\x{11AF}/g;
    s/\x{E6E6}/\x{1120}\x{1174}\x{11BA}/g;
    s/\x{E6E7}/\x{1120}\x{1175}/g;
    s/\x{E6E8}/\x{1120}\x{1175}\x{11A8}/g;
    s/\x{E6E9}/\x{1120}\x{1175}\x{11AB}/g;
    s/\x{E6EA}/\x{1120}\x{1175}\x{11AF}/g;
    s/\x{E6EB}/\x{1120}\x{1175}\x{11D0}/g;
    s/\x{E6EC}/\x{1120}\x{1175}\x{11B7}/g;
    s/\x{E6ED}/\x{1120}\x{1175}\x{11BA}/g;
    s/\x{E6EE}/\x{1120}\x{1198}/g;
    s/\x{E6EF}/\x{1120}\x{119E}/g;
    s/\x{E6F0}/\x{1120}\x{119E}\x{11AB}/g;
    s/\x{E6F1}/\x{1120}\x{119E}\x{11AE}/g;
    s/\x{E6F2}/\x{1120}\x{119E}\x{11AF}/g;
    s/\x{E6F3}/\x{1120}\x{119E}\x{11B7}/g;
    s/\x{E6F4}/\x{1120}\x{119E}\x{11BA}/g;
    s/\x{E6F5}/\x{1120}\x{11A1}/g;
    s/\x{E6F6}/\x{1120}\x{11A1}\x{11AB}/g;
    s/\x{E6F7}/\x{1120}\x{11A1}\x{11BA}/g;
    s/\x{E6F8}/\x{1108}\x{1161}\x{11D0}/g;
    s/\x{E6F9}/\x{1108}\x{1161}\x{11D9}/g;
    s/\x{E6FA}/\x{1108}\x{1161}\x{11E2}/g;
    s/\x{E6FB}/\x{1108}\x{1161}\x{11E6}/g;
    s/\x{E6FC}/\x{1108}\x{1161}\x{11F0}/g;
    s/\x{E6FD}/\x{1108}\x{1161}\x{11F9}/g;
    s/\x{E6FE}/\x{1108}\x{1163}\x{11E2}/g;
    s/\x{E6FF}/\x{1108}\x{1163}\x{11F0}/g;
    s/\x{E700}/\x{1108}\x{1165}\x{11D9}/g;
    s/\x{E701}/\x{1108}\x{1165}\x{11F9}/g;
    s/\x{E702}/\x{1108}\x{1167}\x{11D9}/g;
    s/\x{E703}/\x{1108}\x{1167}\x{11E2}/g;
    s/\x{E704}/\x{1108}\x{1167}\x{11F0}/g;
    s/\x{E705}/\x{1108}\x{1167}\x{11F9}/g;
    s/\x{E706}/\x{1108}\x{1169}\x{11D9}/g;
    s/\x{E707}/\x{1108}\x{1169}\x{11E2}/g;
    s/\x{E708}/\x{1108}\x{1169}\x{11F0}/g;
    s/\x{E709}/\x{1108}\x{116D}\x{11E2}/g;
    s/\x{E70A}/\x{1108}\x{116E}\x{11D9}/g;
    s/\x{E70B}/\x{1108}\x{116E}\x{11E2}/g;
    s/\x{E70C}/\x{1108}\x{116E}\x{11F0}/g;
    s/\x{E70D}/\x{1108}\x{116E}\x{11F9}/g;
    s/\x{E70E}/\x{1108}\x{116F}\x{11F9}/g;
    s/\x{E70F}/\x{1108}\x{116E}\x{1167}/g;
    s/\x{E710}/\x{1108}\x{116E}\x{1167}\x{11F9}/g;
    s/\x{E711}/\x{1108}\x{1172}\x{11E2}/g;
    s/\x{E712}/\x{1108}\x{1173}\x{11D9}/g;
    s/\x{E713}/\x{1108}\x{1173}\x{11E2}/g;
    s/\x{E714}/\x{1108}\x{1173}\x{11B8}\x{11B8}/g;
    s/\x{E715}/\x{1108}\x{1173}\x{11F0}/g;
    s/\x{E716}/\x{1108}\x{1173}\x{11F9}/g;
    s/\x{E717}/\x{1108}\x{1174}\x{11F0}/g;
    s/\x{E718}/\x{1108}\x{1174}\x{11F9}/g;
    s/\x{E719}/\x{1108}\x{1175}\x{11D9}/g;
    s/\x{E71A}/\x{1108}\x{1175}\x{11E2}/g;
    s/\x{E71B}/\x{1108}\x{1175}\x{11F0}/g;
    s/\x{E71C}/\x{1108}\x{1175}\x{11F9}/g;
    s/\x{E71D}/\x{1108}\x{119E}/g;
    s/\x{E71E}/\x{1108}\x{119E}\x{11AF}/g;
    s/\x{E71F}/\x{1108}\x{119E}\x{11B3}/g;
    s/\x{E720}/\x{1108}\x{119E}\x{11D9}/g;
    s/\x{E721}/\x{1108}\x{11A1}/g;
    s/\x{E722}/\x{1108}\x{11A1}\x{11A8}/g;
    s/\x{E723}/\x{1108}\x{11A1}\x{11BC}/g;
    s/\x{E724}/\x{1108}\x{11A1}\x{11F0}/g;
    s/\x{E725}/\x{1121}\x{1161}/g;
    s/\x{E726}/\x{1121}\x{1161}\x{11A8}/g;
    s/\x{E727}/\x{1121}\x{1161}\x{11AB}/g;
    s/\x{E728}/\x{1121}\x{1161}\x{11AE}/g;
    s/\x{E729}/\x{1121}\x{1161}\x{11AF}/g;
    s/\x{E72A}/\x{1121}\x{1161}\x{11B2}/g;
    s/\x{E72B}/\x{1121}\x{1161}\x{11B7}/g;
    s/\x{E72C}/\x{1121}\x{1161}\x{11B8}/g;
    s/\x{E72D}/\x{1121}\x{1161}\x{11BA}/g;
    s/\x{E72E}/\x{1121}\x{1161}\x{11BA}\x{11C0}/g;
    s/\x{E72F}/\x{1121}\x{1161}\x{11BA}\x{11C2}/g;
    s/\x{E730}/\x{1121}\x{1161}\x{11BC}/g;
    s/\x{E731}/\x{1121}\x{1161}\x{11F0}/g;
    s/\x{E732}/\x{1121}\x{1161}\x{11C2}/g;
    s/\x{E733}/\x{1121}\x{1162}/g;
    s/\x{E734}/\x{1121}\x{1162}\x{11B8}/g;
    s/\x{E735}/\x{1121}\x{1163}/g;
    s/\x{E736}/\x{1121}\x{1163}\x{11BC}/g;
    s/\x{E737}/\x{1121}\x{1163}\x{11F0}/g;
    s/\x{E738}/\x{1121}\x{1163}\x{11C1}/g;
    s/\x{E739}/\x{1121}\x{1163}\x{11C2}/g;
    s/\x{E73A}/\x{1121}\x{1164}/g;
    s/\x{E73B}/\x{1121}\x{1165}/g;
    s/\x{E73C}/\x{1121}\x{1165}\x{11A8}/g;
    s/\x{E73D}/\x{1121}\x{1165}\x{11AB}/g;
    s/\x{E73E}/\x{1121}\x{1165}\x{11AF}/g;
    s/\x{E73F}/\x{1121}\x{1165}\x{11B7}/g;
    s/\x{E740}/\x{1121}\x{1165}\x{11BA}/g;
    s/\x{E741}/\x{1121}\x{1166}/g;
    s/\x{E742}/\x{1121}\x{1166}\x{11AB}/g;
    s/\x{E743}/\x{1121}\x{1166}\x{11AF}/g;
    s/\x{E744}/\x{1121}\x{1167}/g;
    s/\x{E745}/\x{1121}\x{1167}\x{11A8}/g;
    s/\x{E746}/\x{1121}\x{1167}\x{11AB}/g;
    s/\x{E747}/\x{1121}\x{1167}\x{11AF}/g;
    s/\x{E748}/\x{1121}\x{1167}\x{11B7}/g;
    s/\x{E749}/\x{1121}\x{1167}\x{11BA}/g;
    s/\x{E74A}/\x{1121}\x{1168}/g;
    s/\x{E74B}/\x{1121}\x{1168}\x{11AB}/g;
    s/\x{E74C}/\x{1121}\x{1169}/g;
    s/\x{E74D}/\x{1121}\x{1169}\x{11A8}/g;
    s/\x{E74E}/\x{1121}\x{1169}\x{11AB}/g;
    s/\x{E74F}/\x{1121}\x{1169}\x{11AE}/g;
    s/\x{E750}/\x{1121}\x{1169}\x{11AF}/g;
    s/\x{E751}/\x{1121}\x{1169}\x{11B7}/g;
    s/\x{E752}/\x{1121}\x{1169}\x{11BA}/g;
    s/\x{E753}/\x{1121}\x{1169}\x{11BC}/g;
    s/\x{E754}/\x{1121}\x{1169}\x{11F0}/g;
    s/\x{E755}/\x{1121}\x{116A}/g;
    s/\x{E756}/\x{1121}\x{116A}\x{11BC}/g;
    s/\x{E757}/\x{1121}\x{116A}\x{11F0}/g;
    s/\x{E758}/\x{1121}\x{116A}\x{11F9}/g;
    s/\x{E759}/\x{1121}\x{116B}/g;
    s/\x{E75A}/\x{1121}\x{116B}\x{11BA}/g;
    s/\x{E75B}/\x{1121}\x{116C}/g;
    s/\x{E75C}/\x{1121}\x{116D}/g;
    s/\x{E75D}/\x{1121}\x{116E}/g;
    s/\x{E75E}/\x{1121}\x{116E}\x{11A8}/g;
    s/\x{E75F}/\x{1121}\x{116E}\x{11AB}/g;
    s/\x{E760}/\x{1121}\x{116E}\x{11AF}/g;
    s/\x{E761}/\x{1121}\x{116E}\x{11D9}/g;
    s/\x{E762}/\x{1121}\x{116E}\x{11B7}/g;
    s/\x{E763}/\x{1121}\x{116E}\x{11BA}/g;
    s/\x{E764}/\x{1121}\x{116E}\x{11BD}/g;
    s/\x{E765}/\x{1121}\x{116E}\x{11BE}/g;
    s/\x{E766}/\x{1121}\x{116F}/g;
    s/\x{E767}/\x{1121}\x{1170}/g;
    s/\x{E768}/\x{1121}\x{116E}\x{1167}/g;
    s/\x{E769}/\x{1121}\x{118C}/g;
    s/\x{E76A}/\x{1121}\x{1171}/g;
    s/\x{E76B}/\x{1121}\x{1172}/g;
    s/\x{E76C}/\x{1121}\x{1172}\x{11B8}/g;
    s/\x{E76D}/\x{1121}\x{1172}\x{11BA}/g;
    s/\x{E76E}/\x{1121}\x{1172}\x{11BD}/g;
    s/\x{E76F}/\x{1121}\x{1194}/g;
    s/\x{E770}/\x{1121}\x{1173}/g;
    s/\x{E771}/\x{1121}\x{1173}\x{11A8}/g;
    s/\x{E772}/\x{1121}\x{1173}\x{11AB}/g;
    s/\x{E773}/\x{1121}\x{1173}\x{11AE}/g;
    s/\x{E774}/\x{1121}\x{1173}\x{11AF}/g;
    s/\x{E775}/\x{1121}\x{1173}\x{11B6}/g;
    s/\x{E776}/\x{1121}\x{1173}\x{11D9}/g;
    s/\x{E777}/\x{1121}\x{1173}\x{11B7}/g;
    s/\x{E778}/\x{1121}\x{1173}\x{11BA}/g;
    s/\x{E779}/\x{1121}\x{1173}\x{11BE}/g;
    s/\x{E77A}/\x{1121}\x{1174}/g;
    s/\x{E77B}/\x{1121}\x{1174}\x{11A8}/g;
    s/\x{E77C}/\x{1121}\x{1175}/g;
    s/\x{E77D}/\x{1121}\x{1175}\x{11A8}/g;
    s/\x{E77E}/\x{1121}\x{1175}\x{11AB}/g;
    s/\x{E77F}/\x{1121}\x{1175}\x{11AF}/g;
    s/\x{E780}/\x{1121}\x{1175}\x{11B7}/g;
    s/\x{E781}/\x{1121}\x{1175}\x{11B8}/g;
    s/\x{E782}/\x{1121}\x{1175}\x{11BA}/g;
    s/\x{E783}/\x{1121}\x{1175}\x{11BC}/g;
    s/\x{E784}/\x{1121}\x{1175}\x{11F0}/g;
    s/\x{E785}/\x{1121}\x{1198}/g;
    s/\x{E786}/\x{1121}\x{119E}/g;
    s/\x{E787}/\x{1121}\x{119E}\x{11AB}/g;
    s/\x{E788}/\x{1121}\x{119E}\x{11AE}/g;
    s/\x{E789}/\x{1121}\x{119E}\x{11AF}/g;
    s/\x{E78A}/\x{1121}\x{119E}\x{11B3}/g;
    s/\x{E78B}/\x{1121}\x{119E}\x{11D9}/g;
    s/\x{E78C}/\x{1121}\x{119E}\x{11B7}/g;
    s/\x{E78D}/\x{1121}\x{119E}\x{11B8}/g;
    s/\x{E78E}/\x{1121}\x{119E}\x{11BA}/g;
    s/\x{E78F}/\x{1121}\x{119E}\x{11BA}\x{11C2}/g;
    s/\x{E790}/\x{1121}\x{119E}\x{11F0}/g;
    s/\x{E791}/\x{1121}\x{119E}\x{11C2}/g;
    s/\x{E792}/\x{1121}\x{11A1}/g;
    s/\x{E793}/\x{1121}\x{11A1}\x{11BA}/g;
    s/\x{E794}/\x{1122}\x{1161}/g;
    s/\x{E795}/\x{1122}\x{1162}/g;
    s/\x{E796}/\x{1122}\x{1162}\x{11AB}/g;
    s/\x{E797}/\x{1122}\x{1162}\x{11BA}/g;
    s/\x{E798}/\x{1122}\x{1165}/g;
    s/\x{E799}/\x{1122}\x{1165}\x{11B8}/g;
    s/\x{E79A}/\x{1122}\x{1165}\x{11E6}/g;
    s/\x{E79B}/\x{1122}\x{1165}\x{11BA}/g;
    s/\x{E79C}/\x{1122}\x{1166}/g;
    s/\x{E79D}/\x{1122}\x{1166}\x{11AF}/g;
    s/\x{E79E}/\x{1122}\x{1167}/g;
    s/\x{E79F}/\x{1122}\x{1167}\x{11A8}/g;
    s/\x{E7A0}/\x{1122}\x{1167}\x{11BA}/g;
    s/\x{E7A1}/\x{1122}\x{1168}/g;
    s/\x{E7A2}/\x{1122}\x{1168}\x{11BA}/g;
    s/\x{E7A3}/\x{1122}\x{1169}/g;
    s/\x{E7A4}/\x{1122}\x{116C}/g;
    s/\x{E7A5}/\x{1122}\x{116C}\x{11BA}/g;
    s/\x{E7A6}/\x{1122}\x{116E}/g;
    s/\x{E7A7}/\x{1122}\x{116E}\x{11AB}/g;
    s/\x{E7A8}/\x{1122}\x{116E}\x{11AF}/g;
    s/\x{E7A9}/\x{1122}\x{116E}\x{11B7}/g;
    s/\x{E7AA}/\x{1122}\x{116E}\x{11BA}/g;
    s/\x{E7AB}/\x{1122}\x{116F}/g;
    s/\x{E7AC}/\x{1122}\x{1170}/g;
    s/\x{E7AD}/\x{1122}\x{1171}/g;
    s/\x{E7AE}/\x{1122}\x{1171}\x{11AB}/g;
    s/\x{E7AF}/\x{1122}\x{1173}/g;
    s/\x{E7B0}/\x{1122}\x{1173}\x{11AB}/g;
    s/\x{E7B1}/\x{1122}\x{1173}\x{11AF}/g;
    s/\x{E7B2}/\x{1122}\x{1173}\x{11B7}/g;
    s/\x{E7B3}/\x{1122}\x{1173}\x{11BA}/g;
    s/\x{E7B4}/\x{1122}\x{1174}/g;
    s/\x{E7B5}/\x{1122}\x{1174}\x{11AB}/g;
    s/\x{E7B6}/\x{1122}\x{1174}\x{11AF}/g;
    s/\x{E7B7}/\x{1122}\x{1174}\x{11B7}/g;
    s/\x{E7B8}/\x{1122}\x{1174}\x{11BA}/g;
    s/\x{E7B9}/\x{1122}\x{1175}/g;
    s/\x{E7BA}/\x{1122}\x{1175}\x{11AF}/g;
    s/\x{E7BB}/\x{1122}\x{1175}\x{11B7}/g;
    s/\x{E7BC}/\x{1122}\x{119E}/g;
    s/\x{E7BD}/\x{1122}\x{119E}\x{11AB}/g;
    s/\x{E7BE}/\x{1123}\x{1161}/g;
    s/\x{E7BF}/\x{1123}\x{1161}\x{11B8}/g;
    s/\x{E7C0}/\x{1123}\x{1162}/g;
    s/\x{E7C1}/\x{1123}\x{1165}/g;
    s/\x{E7C2}/\x{1123}\x{1165}\x{11AF}/g;
    s/\x{E7C3}/\x{1123}\x{1165}\x{11B2}/g;
    s/\x{E7C4}/\x{1123}\x{1165}\x{11BA}/g;
    s/\x{E7C5}/\x{1123}\x{1167}/g;
    s/\x{E7C6}/\x{1123}\x{1167}\x{11BA}/g;
    s/\x{E7C7}/\x{1123}\x{1169}/g;
    s/\x{E7C8}/\x{1123}\x{116C}/g;
    s/\x{E7C9}/\x{1123}\x{116E}/g;
    s/\x{E7CA}/\x{1123}\x{116E}\x{11AF}/g;
    s/\x{E7CB}/\x{1123}\x{116E}\x{11B7}/g;
    s/\x{E7CC}/\x{1123}\x{1172}/g;
    s/\x{E7CD}/\x{1123}\x{1172}\x{11B7}/g;
    s/\x{E7CE}/\x{1123}\x{1173}/g;
    s/\x{E7CF}/\x{1123}\x{1173}\x{11AB}/g;
    s/\x{E7D0}/\x{1123}\x{1173}\x{11BA}/g;
    s/\x{E7D1}/\x{1123}\x{1174}/g;
    s/\x{E7D2}/\x{1123}\x{1175}/g;
    s/\x{E7D3}/\x{1123}\x{1175}\x{11AF}/g;
    s/\x{E7D4}/\x{1123}\x{119E}/g;
    s/\x{E7D5}/\x{1123}\x{119E}\x{11AF}/g;
    s/\x{E7D6}/\x{1123}\x{11A1}/g;
    s/\x{E7D7}/\x{1124}\x{1167}/g;
    s/\x{E7D8}/\x{1124}\x{1168}/g;
    s/\x{E7D9}/\x{1125}\x{1165}/g;
    s/\x{E7DA}/\x{1125}\x{1169}/g;
    s/\x{E7DB}/\x{1125}\x{1173}/g;
    s/\x{E7DC}/\x{1126}\x{119E}/g;
    s/\x{E7DD}/\x{1126}\x{119E}\x{11AB}/g;
    s/\x{E7DE}/\x{1121}\x{1110}\x{1161}/g;
    s/\x{E7DF}/\x{1127}\x{1161}/g;
    s/\x{E7E0}/\x{1127}\x{1161}\x{11A8}/g;
    s/\x{E7E1}/\x{1127}\x{1161}\x{11AB}/g;
    s/\x{E7E2}/\x{1127}\x{1161}\x{11B7}/g;
    s/\x{E7E3}/\x{1127}\x{1161}\x{11BA}/g;
    s/\x{E7E4}/\x{1127}\x{1161}\x{11BC}/g;
    s/\x{E7E5}/\x{1127}\x{1161}\x{11F0}/g;
    s/\x{E7E6}/\x{1127}\x{1162}/g;
    s/\x{E7E7}/\x{1127}\x{1163}/g;
    s/\x{E7E8}/\x{1127}\x{1163}\x{11A8}/g;
    s/\x{E7E9}/\x{1127}\x{1163}\x{11BC}/g;
    s/\x{E7EA}/\x{1127}\x{1163}\x{11F0}/g;
    s/\x{E7EB}/\x{1127}\x{1165}/g;
    s/\x{E7EC}/\x{1127}\x{1165}\x{11A8}/g;
    s/\x{E7ED}/\x{1127}\x{1166}/g;
    s/\x{E7EE}/\x{1127}\x{1167}/g;
    s/\x{E7EF}/\x{1127}\x{1169}/g;
    s/\x{E7F0}/\x{1127}\x{1169}\x{11A8}/g;
    s/\x{E7F1}/\x{1127}\x{1169}\x{11AB}/g;
    s/\x{E7F2}/\x{1127}\x{1169}\x{11B7}/g;
    s/\x{E7F3}/\x{1127}\x{1169}\x{11BA}/g;
    s/\x{E7F4}/\x{1127}\x{1169}\x{11BC}/g;
    s/\x{E7F5}/\x{1127}\x{1169}\x{11F0}/g;
    s/\x{E7F6}/\x{1127}\x{1169}\x{11BD}/g;
    s/\x{E7F7}/\x{1127}\x{1169}\x{11BE}/g;
    s/\x{E7F8}/\x{1127}\x{116A}/g;
    s/\x{E7F9}/\x{1127}\x{116A}\x{11BC}/g;
    s/\x{E7FA}/\x{1127}\x{116A}\x{11F0}/g;
    s/\x{E7FB}/\x{1127}\x{116B}/g;
    s/\x{E7FC}/\x{1127}\x{116C}/g;
    s/\x{E7FD}/\x{1127}\x{116C}\x{11AB}/g;
    s/\x{E7FE}/\x{1127}\x{116C}\x{11AF}/g;
    s/\x{E7FF}/\x{1127}\x{116D}/g;
    s/\x{E800}/\x{1127}\x{116E}/g;
    s/\x{E801}/\x{1127}\x{116E}\x{11A8}/g;
    s/\x{E802}/\x{1127}\x{116E}\x{11AF}/g;
    s/\x{E803}/\x{1127}\x{116E}\x{11B7}/g;
    s/\x{E804}/\x{1127}\x{1171}/g;
    s/\x{E805}/\x{1127}\x{1173}/g;
    s/\x{E806}/\x{1127}\x{1173}\x{11A8}/g;
    s/\x{E807}/\x{1127}\x{1173}\x{11AB}/g;
    s/\x{E808}/\x{1127}\x{1173}\x{11AE}/g;
    s/\x{E809}/\x{1127}\x{1173}\x{11AF}/g;
    s/\x{E80A}/\x{1127}\x{1173}\x{11B7}/g;
    s/\x{E80B}/\x{1127}\x{1173}\x{11BA}/g;
    s/\x{E80C}/\x{1127}\x{1173}\x{11BD}/g;
    s/\x{E80D}/\x{1127}\x{1174}/g;
    s/\x{E80E}/\x{1127}\x{1174}\x{11BC}/g;
    s/\x{E80F}/\x{1127}\x{1174}\x{11F0}/g;
    s/\x{E810}/\x{1127}\x{1175}/g;
    s/\x{E811}/\x{1127}\x{1175}\x{11A8}/g;
    s/\x{E812}/\x{1127}\x{1175}\x{11AB}/g;
    s/\x{E813}/\x{1127}\x{1175}\x{11AE}/g;
    s/\x{E814}/\x{1127}\x{1175}\x{11AF}/g;
    s/\x{E815}/\x{1127}\x{1175}\x{11B7}/g;
    s/\x{E816}/\x{1127}\x{1175}\x{11BA}/g;
    s/\x{E817}/\x{1127}\x{1175}\x{11BA}\x{11C2}/g;
    s/\x{E818}/\x{1127}\x{1175}\x{11BC}/g;
    s/\x{E819}/\x{1127}\x{1175}\x{11F0}/g;
    s/\x{E81A}/\x{1127}\x{119E}/g;
    s/\x{E81B}/\x{1127}\x{119E}\x{11A8}/g;
    s/\x{E81C}/\x{1127}\x{119E}\x{11AB}/g;
    s/\x{E81D}/\x{1127}\x{119E}\x{11AF}/g;
    s/\x{E81E}/\x{1127}\x{119E}\x{11B7}/g;
    s/\x{E81F}/\x{1127}\x{11A1}/g;
    s/\x{E820}/\x{1127}\x{11A1}\x{11AB}/g;
    s/\x{E821}/\x{1127}\x{11A1}\x{11AF}/g;
    s/\x{E822}/\x{1127}\x{11A1}\x{11BC}/g;
    s/\x{E823}/\x{1128}\x{1169}/g;
    s/\x{E824}/\x{1128}\x{1169}\x{11BC}/g;
    s/\x{E825}/\x{1128}\x{119E}/g;
    s/\x{E826}/\x{1128}\x{119E}\x{11AF}/g;
    s/\x{E827}/\x{1107}\x{110F}\x{1169}/g;
    s/\x{E828}/\x{1107}\x{110F}\x{116A}/g;
    s/\x{E829}/\x{1107}\x{110F}\x{116B}/g;
    s/\x{E82A}/\x{1107}\x{110F}\x{1173}/g;
    s/\x{E82B}/\x{1129}\x{1161}/g;
    s/\x{E82C}/\x{1129}\x{1165}/g;
    s/\x{E82D}/\x{1129}\x{1165}\x{11A8}/g;
    s/\x{E82E}/\x{1129}\x{1165}\x{11AF}/g;
    s/\x{E82F}/\x{1129}\x{1165}\x{11B2}/g;
    s/\x{E830}/\x{1129}\x{1165}\x{11D5}/g;
    s/\x{E831}/\x{1129}\x{1165}\x{11B7}/g;
    s/\x{E832}/\x{1129}\x{1165}\x{11BA}/g;
    s/\x{E833}/\x{1129}\x{1167}/g;
    s/\x{E834}/\x{1129}\x{1169}/g;
    s/\x{E835}/\x{1129}\x{1169}\x{11B7}/g;
    s/\x{E836}/\x{1129}\x{1169}\x{11B8}/g;
    s/\x{E837}/\x{1129}\x{116E}/g;
    s/\x{E838}/\x{1129}\x{116E}\x{11A8}/g;
    s/\x{E839}/\x{1129}\x{116E}\x{11A8}\x{11B8}/g;
    s/\x{E83A}/\x{1129}\x{1171}/g;
    s/\x{E83B}/\x{1129}\x{1171}\x{11AF}/g;
    s/\x{E83C}/\x{1129}\x{1171}\x{11B7}/g;
    s/\x{E83D}/\x{1129}\x{1173}/g;
    s/\x{E83E}/\x{1129}\x{1173}\x{11AB}/g;
    s/\x{E83F}/\x{1129}\x{1173}\x{11AE}/g;
    s/\x{E840}/\x{1129}\x{1173}\x{11BA}/g;
    s/\x{E841}/\x{1129}\x{1174}/g;
    s/\x{E842}/\x{1129}\x{1174}\x{11B7}/g;
    s/\x{E843}/\x{1129}\x{1175}/g;
    s/\x{E844}/\x{1129}\x{1175}\x{11AB}/g;
    s/\x{E845}/\x{1129}\x{1175}\x{11AF}/g;
    s/\x{E846}/\x{1129}\x{1175}\x{11B7}/g;
    s/\x{E847}/\x{1129}\x{119E}/g;
    s/\x{E848}/\x{1129}\x{119E}\x{11AB}/g;
    s/\x{E849}/\x{1129}\x{119E}\x{11AF}/g;
    s/\x{E84A}/\x{1129}\x{11A1}/g;
    s/\x{E84B}/\x{112A}\x{1161}/g;
    s/\x{E84C}/\x{112A}\x{1161}\x{11AB}/g;
    s/\x{E84D}/\x{112A}\x{1169}/g;
    s/\x{E84E}/\x{112A}\x{1169}\x{11AB}/g;
    s/\x{E84F}/\x{112A}\x{116D}/g;
    s/\x{E850}/\x{112A}\x{1173}/g;
    s/\x{E851}/\x{112A}\x{1173}\x{11AF}/g;
    s/\x{E852}/\x{112A}\x{1175}/g;
    s/\x{E853}/\x{112A}\x{119E}/g;
    s/\x{E854}/\x{112A}\x{119E}\x{11AF}/g;
    s/\x{E855}/\x{1107}\x{1112}\x{1161}/g;
    s/\x{E856}/\x{1107}\x{1112}\x{1161}\x{11AB}/g;
    s/\x{E857}/\x{112B}\x{1161}/g;
    s/\x{E858}/\x{112B}\x{1161}\x{11A8}/g;
    s/\x{E859}/\x{112B}\x{1161}\x{11AB}/g;
    s/\x{E85A}/\x{112B}\x{1161}\x{11AE}/g;
    s/\x{E85B}/\x{112B}\x{1161}\x{11AF}/g;
    s/\x{E85C}/\x{112B}\x{1161}\x{11B0}/g;
    s/\x{E85D}/\x{112B}\x{1161}\x{11D9}/g;
    s/\x{E85E}/\x{112B}\x{1161}\x{11B7}/g;
    s/\x{E85F}/\x{112B}\x{1161}\x{11BA}/g;
    s/\x{E860}/\x{112B}\x{1161}\x{11EB}/g;
    s/\x{E861}/\x{112B}\x{1161}\x{11BC}/g;
    s/\x{E862}/\x{112B}\x{1161}\x{11F0}/g;
    s/\x{E863}/\x{112B}\x{1161}\x{11C0}/g;
    s/\x{E864}/\x{112B}\x{1161}\x{11F9}/g;
    s/\x{E865}/\x{112B}\x{1162}/g;
    s/\x{E866}/\x{112B}\x{1162}\x{11AB}/g;
    s/\x{E867}/\x{112B}\x{1162}\x{11BA}/g;
    s/\x{E868}/\x{112B}\x{1165}/g;
    s/\x{E869}/\x{112B}\x{1165}\x{11AB}/g;
    s/\x{E86A}/\x{112B}\x{1165}\x{11B7}/g;
    s/\x{E86B}/\x{112B}\x{1165}\x{11B8}/g;
    s/\x{E86C}/\x{112B}\x{1165}\x{11BC}/g;
    s/\x{E86D}/\x{112B}\x{1166}/g;
    s/\x{E86E}/\x{112B}\x{1166}\x{11BA}/g;
    s/\x{E86F}/\x{112B}\x{1167}/g;
    s/\x{E870}/\x{112B}\x{1167}\x{11A8}/g;
    s/\x{E871}/\x{112B}\x{1167}\x{11AF}/g;
    s/\x{E872}/\x{112B}\x{1167}\x{11D9}/g;
    s/\x{E873}/\x{112B}\x{1167}\x{11BA}/g;
    s/\x{E874}/\x{112B}\x{1168}/g;
    s/\x{E875}/\x{112B}\x{1168}\x{11BA}/g;
    s/\x{E876}/\x{112B}\x{1169}/g;
    s/\x{E877}/\x{112B}\x{1169}\x{11AB}/g;
    s/\x{E878}/\x{112B}\x{1169}\x{11AF}/g;
    s/\x{E879}/\x{112B}\x{1169}\x{11B3}/g;
    s/\x{E87A}/\x{112B}\x{1169}\x{11D9}/g;
    s/\x{E87B}/\x{112B}\x{1169}\x{11B7}/g;
    s/\x{E87C}/\x{112B}\x{1169}\x{11DD}/g;
    s/\x{E87D}/\x{112B}\x{1169}\x{11B8}/g;
    s/\x{E87E}/\x{112B}\x{1169}\x{11BA}/g;
    s/\x{E87F}/\x{112B}\x{1169}\x{11BC}/g;
    s/\x{E880}/\x{112B}\x{116A}/g;
    s/\x{E881}/\x{112B}\x{116A}\x{11AB}/g;
    s/\x{E882}/\x{112B}\x{116D}/g;
    s/\x{E883}/\x{112B}\x{1187}/g;
    s/\x{E884}/\x{112B}\x{1188}/g;
    s/\x{E885}/\x{112B}\x{116E}/g;
    s/\x{E886}/\x{112B}\x{116E}\x{11AB}/g;
    s/\x{E887}/\x{112B}\x{116E}\x{11AF}/g;
    s/\x{E888}/\x{112B}\x{116E}\x{11B7}/g;
    s/\x{E889}/\x{112B}\x{116E}\x{11B8}/g;
    s/\x{E88A}/\x{112B}\x{116E}\x{11BA}/g;
    s/\x{E88B}/\x{112B}\x{116E}\x{11BC}/g;
    s/\x{E88C}/\x{112B}\x{116E}\x{11F0}/g;
    s/\x{E88D}/\x{112B}\x{116E}\x{11C2}/g;
    s/\x{E88E}/\x{112B}\x{116E}\x{11F9}/g;
    s/\x{E88F}/\x{112B}\x{116F}/g;
    s/\x{E890}/\x{112B}\x{1171}/g;
    s/\x{E891}/\x{112B}\x{1171}\x{11AF}/g;
    s/\x{E892}/\x{112B}\x{1171}\x{11D9}/g;
    s/\x{E893}/\x{112B}\x{1172}/g;
    s/\x{E894}/\x{112B}\x{1172}\x{11F9}/g;
    s/\x{E895}/\x{112B}\x{1173}/g;
    s/\x{E896}/\x{112B}\x{1173}\x{11AB}/g;
    s/\x{E897}/\x{112B}\x{1173}\x{11C7}/g;
    s/\x{E898}/\x{112B}\x{1173}\x{11AF}/g;
    s/\x{E899}/\x{112B}\x{1173}\x{11B3}/g;
    s/\x{E89A}/\x{112B}\x{1173}\x{11D9}/g;
    s/\x{E89B}/\x{112B}\x{1173}\x{11B7}/g;
    s/\x{E89C}/\x{112B}\x{1173}\x{11E2}/g;
    s/\x{E89D}/\x{112B}\x{1173}\x{11BA}/g;
    s/\x{E89E}/\x{112B}\x{1173}\x{11BC}/g;
    s/\x{E89F}/\x{112B}\x{1195}/g;
    s/\x{E8A0}/\x{112B}\x{1174}/g;
    s/\x{E8A1}/\x{112B}\x{1175}/g;
    s/\x{E8A2}/\x{112B}\x{1175}\x{11A8}/g;
    s/\x{E8A3}/\x{112B}\x{1175}\x{11AB}/g;
    s/\x{E8A4}/\x{112B}\x{1175}\x{11AF}/g;
    s/\x{E8A5}/\x{112B}\x{1175}\x{11D9}/g;
    s/\x{E8A6}/\x{112B}\x{1175}\x{11BC}/g;
    s/\x{E8A7}/\x{112B}\x{119E}/g;
    s/\x{E8A8}/\x{112B}\x{119E}\x{11A8}/g;
    s/\x{E8A9}/\x{112B}\x{119E}\x{11AB}/g;
    s/\x{E8AA}/\x{112B}\x{119E}\x{11AF}/g;
    s/\x{E8AB}/\x{112B}\x{119E}\x{11B0}/g;
    s/\x{E8AC}/\x{112B}\x{119E}\x{11B2}/g;
    s/\x{E8AD}/\x{112B}\x{119E}\x{11B3}/g;
    s/\x{E8AE}/\x{112B}\x{119E}\x{11D9}/g;
    s/\x{E8AF}/\x{112B}\x{119E}\x{11B7}/g;
    s/\x{E8B0}/\x{112B}\x{119E}\x{11EB}/g;
    s/\x{E8B1}/\x{112B}\x{119E}\x{11BC}/g;
    s/\x{E8B2}/\x{112B}\x{11A1}/g;
    s/\x{E8B3}/\x{112B}\x{11A1}\x{11A8}/g;
    s/\x{E8B4}/\x{112B}\x{11A1}\x{11AB}/g;
    s/\x{E8B5}/\x{112B}\x{11A1}\x{11AF}/g;
    s/\x{E8B6}/\x{112C}\x{1161}/g;
    s/\x{E8B7}/\x{112C}\x{1161}\x{11AB}/g;
    s/\x{E8B8}/\x{112C}\x{1161}\x{11BC}/g;
    s/\x{E8B9}/\x{112C}\x{1161}\x{11F0}/g;
    s/\x{E8BA}/\x{112C}\x{1161}\x{11F9}/g;
    s/\x{E8BB}/\x{112C}\x{1169}/g;
    s/\x{E8BC}/\x{112C}\x{1169}\x{11AB}/g;
    s/\x{E8BD}/\x{112C}\x{116A}/g;
    s/\x{E8BE}/\x{112C}\x{116A}\x{11AB}/g;
    s/\x{E8BF}/\x{112C}\x{116E}/g;
    s/\x{E8C0}/\x{112C}\x{116E}\x{11AB}/g;
    s/\x{E8C1}/\x{112C}\x{116E}\x{11BC}/g;
    s/\x{E8C2}/\x{112C}\x{116E}\x{11F0}/g;
    s/\x{E8C3}/\x{112C}\x{116E}\x{11F9}/g;
    s/\x{E8C4}/\x{112C}\x{1172}/g;
    s/\x{E8C5}/\x{112C}\x{1172}\x{11AB}/g;
    s/\x{E8C6}/\x{112C}\x{1173}/g;
    s/\x{E8C7}/\x{112C}\x{1173}\x{11AB}/g;
    s/\x{E8C8}/\x{112C}\x{1173}\x{11B7}/g;
    s/\x{E8C9}/\x{112C}\x{1173}\x{11E2}/g;
    s/\x{E8CA}/\x{112C}\x{1173}\x{11BC}/g;
    s/\x{E8CB}/\x{112C}\x{1174}/g;
    s/\x{E8CC}/\x{112C}\x{1175}/g;
    s/\x{E8CD}/\x{1109}\x{1161}\x{11C7}/g;
    s/\x{E8CE}/\x{1109}\x{1161}\x{11CA}/g;
    s/\x{E8CF}/\x{1109}\x{1161}\x{11AE}\x{11C0}/g;
    s/\x{E8D0}/\x{1109}\x{1161}\x{11CD}/g;
    s/\x{E8D1}/\x{1109}\x{1161}\x{11D0}/g;
    s/\x{E8D2}/\x{1109}\x{1161}\x{11D9}/g;
    s/\x{E8D3}/\x{1109}\x{1161}\x{11DD}/g;
    s/\x{E8D4}/\x{1109}\x{1161}\x{11E2}/g;
    s/\x{E8D5}/\x{1109}\x{1161}\x{11E7}/g;
    s/\x{E8D6}/\x{1109}\x{1161}\x{11E8}/g;
    s/\x{E8D7}/\x{1109}\x{1161}\x{11BA}\x{11C2}/g;
    s/\x{E8D8}/\x{1109}\x{1161}\x{11EB}/g;
    s/\x{E8D9}/\x{1109}\x{1161}\x{11BC}\x{11BA}/g;
    s/\x{E8DA}/\x{1109}\x{1161}\x{11F0}/g;
    s/\x{E8DB}/\x{1109}\x{1161}\x{11F1}/g;
    s/\x{E8DC}/\x{1109}\x{1161}\x{11F9}/g;
    s/\x{E8DD}/\x{1109}\x{1176}/g;
    s/\x{E8DE}/\x{1109}\x{1177}/g;
    s/\x{E8DF}/\x{1109}\x{1161}\x{1173}/g;
    s/\x{E8E0}/\x{1109}\x{1161}\x{1173}\x{11B7}/g;
    s/\x{E8E1}/\x{1109}\x{1162}\x{11E6}/g;
    s/\x{E8E2}/\x{1109}\x{1162}\x{11F0}/g;
    s/\x{E8E3}/\x{1109}\x{1163}\x{11D9}/g;
    s/\x{E8E4}/\x{1109}\x{1163}\x{11E2}/g;
    s/\x{E8E5}/\x{1109}\x{1163}\x{11EB}/g;
    s/\x{E8E6}/\x{1109}\x{1163}\x{11F0}/g;
    s/\x{E8E7}/\x{1109}\x{1163}\x{11F1}/g;
    s/\x{E8E8}/\x{1109}\x{1178}/g;
    s/\x{E8E9}/\x{1109}\x{1165}\x{11C3}/g;
    s/\x{E8EA}/\x{1109}\x{1165}\x{11D9}/g;
    s/\x{E8EB}/\x{1109}\x{1165}\x{11E5}/g;
    s/\x{E8EC}/\x{1109}\x{1165}\x{11E7}/g;
    s/\x{E8ED}/\x{1109}\x{1165}\x{11EB}/g;
    s/\x{E8EE}/\x{1109}\x{1165}\x{11EC}/g;
    s/\x{E8EF}/\x{1109}\x{1165}\x{11ED}/g;
    s/\x{E8F0}/\x{1109}\x{1165}\x{11F0}/g;
    s/\x{E8F1}/\x{1109}\x{1165}\x{11F9}/g;
    s/\x{E8F2}/\x{1109}\x{117A}/g;
    s/\x{E8F3}/\x{1109}\x{117B}/g;
    s/\x{E8F4}/\x{1109}\x{117C}/g;
    s/\x{E8F5}/\x{1109}\x{117C}\x{11A8}/g;
    s/\x{E8F6}/\x{1109}\x{117C}\x{11AF}/g;
    s/\x{E8F7}/\x{1109}\x{117C}\x{11B7}/g;
    s/\x{E8F8}/\x{1109}\x{117C}\x{11BA}/g;
    s/\x{E8F9}/\x{1109}\x{117C}\x{11BB}/g;
    s/\x{E8FA}/\x{1109}\x{117C}\x{11BC}/g;
    s/\x{E8FB}/\x{1109}\x{1166}\x{11C1}\x{11C0}/g;
    s/\x{E8FC}/\x{1109}\x{1167}\x{11D5}/g;
    s/\x{E8FD}/\x{1109}\x{1167}\x{11D9}/g;
    s/\x{E8FE}/\x{1109}\x{1167}\x{11DD}/g;
    s/\x{E8FF}/\x{1109}\x{1167}\x{11E2}/g;
    s/\x{E900}/\x{1109}\x{1167}\x{11E3}/g;
    s/\x{E901}/\x{1109}\x{1167}\x{11E4}/g;
    s/\x{E902}/\x{1109}\x{1167}\x{11E5}/g;
    s/\x{E903}/\x{1109}\x{1167}\x{11E7}/g;
    s/\x{E904}/\x{1109}\x{1167}\x{11F0}/g;
    s/\x{E905}/\x{1109}\x{1167}\x{11F1}/g;
    s/\x{E906}/\x{1109}\x{1167}\x{11F9}/g;
    s/\x{E907}/\x{1109}\x{117D}/g;
    s/\x{E908}/\x{1109}\x{117E}/g;
    s/\x{E909}/\x{1109}\x{1168}\x{11F0}/g;
    s/\x{E90A}/\x{1109}\x{1169}\x{11C7}/g;
    s/\x{E90B}/\x{1109}\x{1169}\x{11D9}/g;
    s/\x{E90C}/\x{1109}\x{1169}\x{11E2}/g;
    s/\x{E90D}/\x{1109}\x{1169}\x{11B9}\x{11AE}/g;
    s/\x{E90E}/\x{1109}\x{1169}\x{11E8}/g;
    s/\x{E90F}/\x{1109}\x{1169}\x{11BA}\x{11BE}/g;
    s/\x{E910}/\x{1109}\x{1169}\x{11EB}/g;
    s/\x{E911}/\x{1109}\x{1169}\x{11F0}/g;
    s/\x{E912}/\x{1109}\x{116A}\x{11D9}/g;
    s/\x{E913}/\x{1109}\x{116A}\x{11F0}/g;
    s/\x{E914}/\x{1109}\x{116A}\x{11F9}/g;
    s/\x{E915}/\x{1109}\x{116B}\x{11F9}/g;
    s/\x{E916}/\x{1109}\x{1169}\x{1163}/g;
    s/\x{E917}/\x{1109}\x{1169}\x{1164}/g;
    s/\x{E918}/\x{1109}\x{1182}/g;
    s/\x{E919}/\x{1109}\x{116C}\x{11F0}/g;
    s/\x{E91A}/\x{1109}\x{116D}\x{11D9}/g;
    s/\x{E91B}/\x{1109}\x{116D}\x{11E2}/g;
    s/\x{E91C}/\x{1109}\x{116D}\x{11F0}/g;
    s/\x{E91D}/\x{1109}\x{116D}\x{11F1}/g;
    s/\x{E91E}/\x{1109}\x{1187}/g;
    s/\x{E91F}/\x{1109}\x{1187}\x{11AF}/g;
    s/\x{E920}/\x{1109}\x{1188}/g;
    s/\x{E921}/\x{1109}\x{1188}\x{11AB}/g;
    s/\x{E922}/\x{1109}\x{1188}\x{11AF}/g;
    s/\x{E923}/\x{1109}\x{1188}\x{11BA}/g;
    s/\x{E924}/\x{1109}\x{116E}\x{11D0}/g;
    s/\x{E925}/\x{1109}\x{116E}\x{11D9}/g;
    s/\x{E926}/\x{1109}\x{116E}\x{11E2}/g;
    s/\x{E927}/\x{1109}\x{116E}\x{11E4}/g;
    s/\x{E928}/\x{1109}\x{116E}\x{11E6}/g;
    s/\x{E929}/\x{1109}\x{116E}\x{11E7}/g;
    s/\x{E92A}/\x{1109}\x{116E}\x{11BA}\x{11BE}/g;
    s/\x{E92B}/\x{1109}\x{116E}\x{11EB}/g;
    s/\x{E92C}/\x{1109}\x{116E}\x{11F0}/g;
    s/\x{E92D}/\x{1109}\x{116E}\x{11F9}/g;
    s/\x{E92E}/\x{1109}\x{1189}/g;
    s/\x{E92F}/\x{1109}\x{1189}\x{11AB}/g;
    s/\x{E930}/\x{1109}\x{1189}\x{11B7}/g;
    s/\x{E931}/\x{1109}\x{116E}\x{1167}/g;
    s/\x{E932}/\x{1109}\x{116E}\x{1167}\x{11AB}/g;
    s/\x{E933}/\x{1109}\x{118C}/g;
    s/\x{E934}/\x{1109}\x{1171}\x{11D9}/g;
    s/\x{E935}/\x{1109}\x{1171}\x{11E6}/g;
    s/\x{E936}/\x{1109}\x{1171}\x{11E7}/g;
    s/\x{E937}/\x{1109}\x{1172}\x{11D9}/g;
    s/\x{E938}/\x{1109}\x{1172}\x{11E2}/g;
    s/\x{E939}/\x{1109}\x{1172}\x{11E6}/g;
    s/\x{E93A}/\x{1109}\x{1172}\x{11F0}/g;
    s/\x{E93B}/\x{1109}\x{118E}/g;
    s/\x{E93C}/\x{1109}\x{118E}\x{11AB}/g;
    s/\x{E93D}/\x{1109}\x{118E}\x{11F0}/g;
    s/\x{E93E}/\x{1109}\x{1172}\x{1162}/g;
    s/\x{E93F}/\x{1109}\x{118F}/g;
    s/\x{E940}/\x{1109}\x{1190}/g;
    s/\x{E941}/\x{1109}\x{1191}/g;
    s/\x{E942}/\x{1109}\x{1191}\x{11AB}/g;
    s/\x{E943}/\x{1109}\x{1191}\x{11AF}/g;
    s/\x{E944}/\x{1109}\x{1191}\x{11D9}/g;
    s/\x{E945}/\x{1109}\x{1191}\x{11BC}/g;
    s/\x{E946}/\x{1109}\x{1191}\x{11F9}/g;
    s/\x{E947}/\x{1109}\x{1192}/g;
    s/\x{E948}/\x{1109}\x{1192}\x{11BC}/g;
    s/\x{E949}/\x{1109}\x{1172}\x{1169}/g;
    s/\x{E94A}/\x{1109}\x{1172}\x{1169}\x{11AF}/g;
    s/\x{E94B}/\x{1109}\x{1193}/g;
    s/\x{E94C}/\x{1109}\x{1193}\x{11AF}/g;
    s/\x{E94D}/\x{1109}\x{1194}/g;
    s/\x{E94E}/\x{1109}\x{1194}\x{11AB}/g;
    s/\x{E94F}/\x{1109}\x{1194}\x{11AF}/g;
    s/\x{E950}/\x{1109}\x{1194}\x{11B7}/g;
    s/\x{E951}/\x{1109}\x{1194}\x{11B8}/g;
    s/\x{E952}/\x{1109}\x{1194}\x{11BC}/g;
    s/\x{E953}/\x{1109}\x{1173}\x{11D9}/g;
    s/\x{E954}/\x{1109}\x{1173}\x{11E2}/g;
    s/\x{E955}/\x{1109}\x{1173}\x{11E3}/g;
    s/\x{E956}/\x{1109}\x{1173}\x{11BA}\x{11BE}/g;
    s/\x{E957}/\x{1109}\x{1173}\x{11BA}\x{11C2}/g;
    s/\x{E958}/\x{1109}\x{1173}\x{11EB}/g;
    s/\x{E959}/\x{1109}\x{1173}\x{11BC}\x{11BA}/g;
    s/\x{E95A}/\x{1109}\x{1173}\x{11BC}\x{11C2}/g;
    s/\x{E95B}/\x{1109}\x{1173}\x{11F0}/g;
    s/\x{E95C}/\x{1109}\x{1173}\x{11F1}/g;
    s/\x{E95D}/\x{1109}\x{1173}\x{11F9}/g;
    s/\x{E95E}/\x{1109}\x{1195}/g;
    s/\x{E95F}/\x{1109}\x{1196}/g;
    s/\x{E960}/\x{1109}\x{1174}\x{11F0}/g;
    s/\x{E961}/\x{1109}\x{1174}\x{11F9}/g;
    s/\x{E962}/\x{1109}\x{1175}\x{11A8}\x{11AB}/g;
    s/\x{E963}/\x{1109}\x{1175}\x{11AB}\x{11AB}/g;
    s/\x{E964}/\x{1109}\x{1175}\x{11C7}/g;
    s/\x{E965}/\x{1109}\x{1175}\x{11CE}/g;
    s/\x{E966}/\x{1109}\x{1175}\x{11D0}/g;
    s/\x{E967}/\x{1109}\x{1175}\x{11D9}/g;
    s/\x{E968}/\x{1109}\x{1175}\x{11DA}/g;
    s/\x{E969}/\x{1109}\x{1175}\x{11E1}/g;
    s/\x{E96A}/\x{1109}\x{1175}\x{11E2}/g;
    s/\x{E96B}/\x{1109}\x{1175}\x{11E6}/g;
    s/\x{E96C}/\x{1109}\x{1175}\x{11EB}/g;
    s/\x{E96D}/\x{1109}\x{1175}\x{11F0}/g;
    s/\x{E96E}/\x{1109}\x{1175}\x{11F9}/g;
    s/\x{E96F}/\x{1109}\x{1198}/g;
    s/\x{E970}/\x{1109}\x{1199}/g;
    s/\x{E971}/\x{1109}\x{1199}\x{11AB}/g;
    s/\x{E972}/\x{1109}\x{1199}\x{11BC}/g;
    s/\x{E973}/\x{1109}\x{1175}\x{1178}/g;
    s/\x{E974}/\x{1109}\x{1175}\x{1164}/g;
    s/\x{E975}/\x{1109}\x{1175}\x{1167}/g;
    s/\x{E976}/\x{1109}\x{1175}\x{1168}/g;
    s/\x{E977}/\x{1109}\x{119A}/g;
    s/\x{E978}/\x{1109}\x{1175}\x{1183}/g;
    s/\x{E979}/\x{1109}\x{1175}\x{116D}/g;
    s/\x{E97A}/\x{1109}\x{119B}/g;
    s/\x{E97B}/\x{1109}\x{1175}\x{1172}/g;
    s/\x{E97C}/\x{1109}\x{1175}\x{1172}\x{11AB}/g;
    s/\x{E97D}/\x{1109}\x{119E}/g;
    s/\x{E97E}/\x{1109}\x{119E}\x{11A8}/g;
    s/\x{E97F}/\x{1109}\x{119E}\x{11AB}/g;
    s/\x{E980}/\x{1109}\x{119E}\x{11AE}/g;
    s/\x{E981}/\x{1109}\x{119E}\x{11CA}/g;
    s/\x{E982}/\x{1109}\x{119E}\x{11AF}/g;
    s/\x{E983}/\x{1109}\x{119E}\x{11B0}/g;
    s/\x{E984}/\x{1109}\x{119E}\x{11D0}/g;
    s/\x{E985}/\x{1109}\x{119E}\x{11B1}/g;
    s/\x{E986}/\x{1109}\x{119E}\x{11B2}/g;
    s/\x{E987}/\x{1109}\x{119E}\x{11D5}/g;
    s/\x{E988}/\x{1109}\x{119E}\x{11B3}/g;
    s/\x{E989}/\x{1109}\x{119E}\x{11B6}/g;
    s/\x{E98A}/\x{1109}\x{119E}\x{11D9}/g;
    s/\x{E98B}/\x{1109}\x{119E}\x{11B7}/g;
    s/\x{E98C}/\x{1109}\x{119E}\x{11B7}\x{11B7}/g;
    s/\x{E98D}/\x{1109}\x{119E}\x{11DD}/g;
    s/\x{E98E}/\x{1109}\x{119E}\x{11B8}/g;
    s/\x{E98F}/\x{1109}\x{119E}\x{11E3}/g;
    s/\x{E990}/\x{1109}\x{119E}\x{11E6}/g;
    s/\x{E991}/\x{1109}\x{119E}\x{11BA}/g;
    s/\x{E992}/\x{1109}\x{119E}\x{11BA}\x{11BE}/g;
    s/\x{E993}/\x{1109}\x{119E}\x{11EB}/g;
    s/\x{E994}/\x{1109}\x{119E}\x{11BC}/g;
    s/\x{E995}/\x{1109}\x{119E}\x{11F0}/g;
    s/\x{E996}/\x{1109}\x{119E}\x{11BD}/g;
    s/\x{E997}/\x{1109}\x{119E}\x{11BE}/g;
    s/\x{E998}/\x{1109}\x{119E}\x{11C0}/g;
    s/\x{E999}/\x{1109}\x{11A1}/g;
    s/\x{E99A}/\x{1109}\x{11A1}\x{11A8}/g;
    s/\x{E99B}/\x{1109}\x{11A1}\x{11AB}/g;
    s/\x{E99C}/\x{1109}\x{11A1}\x{11AF}/g;
    s/\x{E99D}/\x{1109}\x{11A1}\x{11D9}/g;
    s/\x{E99E}/\x{1109}\x{11A1}\x{11B7}/g;
    s/\x{E99F}/\x{1109}\x{11A1}\x{11DD}/g;
    s/\x{E9A0}/\x{1109}\x{11A1}\x{11B8}/g;
    s/\x{E9A1}/\x{1109}\x{11A1}\x{11BA}/g;
    s/\x{E9A2}/\x{1109}\x{11A1}\x{11BC}/g;
    s/\x{E9A3}/\x{1109}\x{11A1}\x{11F0}/g;
    s/\x{E9A4}/\x{1109}\x{11A1}\x{11F1}/g;
    s/\x{E9A5}/\x{112D}\x{1161}/g;
    s/\x{E9A6}/\x{112D}\x{1161}\x{11A8}/g;
    s/\x{E9A7}/\x{112D}\x{1161}\x{11A9}/g;
    s/\x{E9A8}/\x{112D}\x{1161}\x{11AA}/g;
    s/\x{E9A9}/\x{112D}\x{1161}\x{11AB}/g;
    s/\x{E9AA}/\x{112D}\x{1161}\x{11AF}/g;
    s/\x{E9AB}/\x{112D}\x{1161}\x{11B6}/g;
    s/\x{E9AC}/\x{112D}\x{1161}\x{11B7}/g;
    s/\x{E9AD}/\x{112D}\x{1161}\x{11B8}/g;
    s/\x{E9AE}/\x{112D}\x{1161}\x{11B9}/g;
    s/\x{E9AF}/\x{112D}\x{1161}\x{11E6}/g;
    s/\x{E9B0}/\x{112D}\x{1161}\x{11BA}/g;
    s/\x{E9B1}/\x{112D}\x{1161}\x{11E7}/g;
    s/\x{E9B2}/\x{112D}\x{1161}\x{11BC}/g;
    s/\x{E9B3}/\x{112D}\x{1161}\x{11F0}/g;
    s/\x{E9B4}/\x{112D}\x{1162}/g;
    s/\x{E9B5}/\x{112D}\x{1162}\x{11A8}/g;
    s/\x{E9B6}/\x{112D}\x{1162}\x{11AB}/g;
    s/\x{E9B7}/\x{112D}\x{1162}\x{11AF}/g;
    s/\x{E9B8}/\x{112D}\x{1162}\x{11B7}/g;
    s/\x{E9B9}/\x{112D}\x{1162}\x{11BA}/g;
    s/\x{E9BA}/\x{112D}\x{1162}\x{11F0}/g;
    s/\x{E9BB}/\x{112D}\x{1163}/g;
    s/\x{E9BC}/\x{112D}\x{1163}\x{11A8}/g;
    s/\x{E9BD}/\x{112D}\x{1164}/g;
    s/\x{E9BE}/\x{112D}\x{1164}\x{11BA}/g;
    s/\x{E9BF}/\x{112D}\x{1165}/g;
    s/\x{E9C0}/\x{112D}\x{1165}\x{11A8}/g;
    s/\x{E9C1}/\x{112D}\x{1165}\x{11A9}/g;
    s/\x{E9C2}/\x{112D}\x{1165}\x{11AB}/g;
    s/\x{E9C3}/\x{112D}\x{1165}\x{11AF}/g;
    s/\x{E9C4}/\x{112D}\x{1165}\x{11B2}/g;
    s/\x{E9C5}/\x{112D}\x{1165}\x{11D5}/g;
    s/\x{E9C6}/\x{112D}\x{1165}\x{11B7}/g;
    s/\x{E9C7}/\x{112D}\x{1165}\x{11B8}/g;
    s/\x{E9C8}/\x{112D}\x{1165}\x{11E6}/g;
    s/\x{E9C9}/\x{112D}\x{1165}\x{11BA}/g;
    s/\x{E9CA}/\x{112D}\x{1165}\x{11E7}/g;
    s/\x{E9CB}/\x{112D}\x{1165}\x{11EB}/g;
    s/\x{E9CC}/\x{112D}\x{1165}\x{11F0}/g;
    s/\x{E9CD}/\x{112D}\x{1166}/g;
    s/\x{E9CE}/\x{112D}\x{1166}\x{11A8}/g;
    s/\x{E9CF}/\x{112D}\x{1166}\x{11AB}/g;
    s/\x{E9D0}/\x{112D}\x{1166}\x{11AF}/g;
    s/\x{E9D1}/\x{112D}\x{1166}\x{11B7}/g;
    s/\x{E9D2}/\x{112D}\x{1166}\x{11B8}/g;
    s/\x{E9D3}/\x{112D}\x{1166}\x{11BA}/g;
    s/\x{E9D4}/\x{112D}\x{1166}\x{11F0}/g;
    s/\x{E9D5}/\x{112D}\x{1167}/g;
    s/\x{E9D6}/\x{112D}\x{1167}\x{11A8}/g;
    s/\x{E9D7}/\x{112D}\x{1167}\x{11AB}/g;
    s/\x{E9D8}/\x{112D}\x{1167}\x{11AE}/g;
    s/\x{E9D9}/\x{112D}\x{1167}\x{11AF}/g;
    s/\x{E9DA}/\x{112D}\x{1167}\x{11B7}/g;
    s/\x{E9DB}/\x{112D}\x{1167}\x{11B8}/g;
    s/\x{E9DC}/\x{112D}\x{1167}\x{11BA}/g;
    s/\x{E9DD}/\x{112D}\x{1167}\x{11BC}/g;
    s/\x{E9DE}/\x{112D}\x{1167}\x{11F0}/g;
    s/\x{E9DF}/\x{112D}\x{1167}\x{11C0}/g;
    s/\x{E9E0}/\x{112D}\x{1168}/g;
    s/\x{E9E1}/\x{112D}\x{1169}/g;
    s/\x{E9E2}/\x{112D}\x{1169}\x{11A8}/g;
    s/\x{E9E3}/\x{112D}\x{1169}\x{11AA}/g;
    s/\x{E9E4}/\x{112D}\x{1169}\x{11AB}/g;
    s/\x{E9E5}/\x{112D}\x{1169}\x{11AE}/g;
    s/\x{E9E6}/\x{112D}\x{1169}\x{11AF}/g;
    s/\x{E9E7}/\x{112D}\x{1169}\x{11B3}/g;
    s/\x{E9E8}/\x{112D}\x{1169}\x{11B7}/g;
    s/\x{E9E9}/\x{112D}\x{1169}\x{11B8}/g;
    s/\x{E9EA}/\x{112D}\x{1169}\x{11BA}/g;
    s/\x{E9EB}/\x{112D}\x{1169}\x{11BA}\x{11BE}/g;
    s/\x{E9EC}/\x{112D}\x{1169}\x{11BC}/g;
    s/\x{E9ED}/\x{112D}\x{1169}\x{11F0}/g;
    s/\x{E9EE}/\x{112D}\x{1169}\x{11BD}/g;
    s/\x{E9EF}/\x{112D}\x{1169}\x{11BE}/g;
    s/\x{E9F0}/\x{112D}\x{116A}/g;
    s/\x{E9F1}/\x{112D}\x{116A}\x{11A8}/g;
    s/\x{E9F2}/\x{112D}\x{116A}\x{11AB}/g;
    s/\x{E9F3}/\x{112D}\x{116A}\x{11AF}/g;
    s/\x{E9F4}/\x{112D}\x{116A}\x{11BC}/g;
    s/\x{E9F5}/\x{112D}\x{116A}\x{11F0}/g;
    s/\x{E9F6}/\x{112D}\x{116B}/g;
    s/\x{E9F7}/\x{112D}\x{116B}\x{11AF}/g;
    s/\x{E9F8}/\x{112D}\x{116B}\x{11B0}/g;
    s/\x{E9F9}/\x{112D}\x{116B}\x{11BC}/g;
    s/\x{E9FA}/\x{112D}\x{116C}/g;
    s/\x{E9FB}/\x{112D}\x{116C}\x{11A8}/g;
    s/\x{E9FC}/\x{112D}\x{116C}\x{11AB}/g;
    s/\x{E9FD}/\x{112D}\x{116C}\x{11AF}/g;
    s/\x{E9FE}/\x{112D}\x{116C}\x{11B7}/g;
    s/\x{E9FF}/\x{112D}\x{116C}\x{11BA}/g;
    s/\x{EA00}/\x{112D}\x{116C}\x{11F0}/g;
    s/\x{EA01}/\x{112D}\x{116D}/g;
    s/\x{EA02}/\x{112D}\x{116D}\x{11AB}/g;
    s/\x{EA03}/\x{112D}\x{116D}\x{11AF}/g;
    s/\x{EA04}/\x{112D}\x{116D}\x{11B7}/g;
    s/\x{EA05}/\x{112D}\x{116E}/g;
    s/\x{EA06}/\x{112D}\x{116E}\x{11A8}/g;
    s/\x{EA07}/\x{112D}\x{116E}\x{11AB}/g;
    s/\x{EA08}/\x{112D}\x{116E}\x{11AE}/g;
    s/\x{EA09}/\x{112D}\x{116E}\x{11AF}/g;
    s/\x{EA0A}/\x{112D}\x{116E}\x{11D9}/g;
    s/\x{EA0B}/\x{112D}\x{116E}\x{11B7}/g;
    s/\x{EA0C}/\x{112D}\x{116E}\x{11DA}/g;
    s/\x{EA0D}/\x{112D}\x{116E}\x{11B7}\x{11B7}/g;
    s/\x{EA0E}/\x{112D}\x{116E}\x{11DD}/g;
    s/\x{EA0F}/\x{112D}\x{116E}\x{11B8}/g;
    s/\x{EA10}/\x{112D}\x{116E}\x{11BA}/g;
    s/\x{EA11}/\x{112D}\x{116E}\x{11BC}/g;
    s/\x{EA12}/\x{112D}\x{116E}\x{11EC}/g;
    s/\x{EA13}/\x{112D}\x{116E}\x{11F0}/g;
    s/\x{EA14}/\x{112D}\x{116E}\x{11BD}/g;
    s/\x{EA15}/\x{112D}\x{116E}\x{11BE}/g;
    s/\x{EA16}/\x{112D}\x{116F}/g;
    s/\x{EA17}/\x{112D}\x{116F}\x{11A8}/g;
    s/\x{EA18}/\x{112D}\x{116F}\x{11AB}/g;
    s/\x{EA19}/\x{112D}\x{116F}\x{11AF}/g;
    s/\x{EA1A}/\x{112D}\x{116F}\x{11B7}/g;
    s/\x{EA1B}/\x{112D}\x{116F}\x{11BC}/g;
    s/\x{EA1C}/\x{112D}\x{116F}\x{11F0}/g;
    s/\x{EA1D}/\x{112D}\x{1170}/g;
    s/\x{EA1E}/\x{112D}\x{1170}\x{11AB}/g;
    s/\x{EA1F}/\x{112D}\x{1170}\x{11AF}/g;
    s/\x{EA20}/\x{112D}\x{116E}\x{1167}/g;
    s/\x{EA21}/\x{112D}\x{118C}/g;
    s/\x{EA22}/\x{112D}\x{1171}/g;
    s/\x{EA23}/\x{112D}\x{1171}\x{11AB}/g;
    s/\x{EA24}/\x{112D}\x{1171}\x{11AF}/g;
    s/\x{EA25}/\x{112D}\x{1171}\x{11B7}/g;
    s/\x{EA26}/\x{112D}\x{1171}\x{11F0}/g;
    s/\x{EA27}/\x{112D}\x{1172}/g;
    s/\x{EA28}/\x{112D}\x{1172}\x{11AB}/g;
    s/\x{EA29}/\x{112D}\x{1172}\x{11AF}/g;
    s/\x{EA2A}/\x{112D}\x{1172}\x{11B7}/g;
    s/\x{EA2B}/\x{112D}\x{118F}/g;
    s/\x{EA2C}/\x{112D}\x{118F}\x{11BC}/g;
    s/\x{EA2D}/\x{112D}\x{1194}/g;
    s/\x{EA2E}/\x{112D}\x{1173}/g;
    s/\x{EA2F}/\x{112D}\x{1173}\x{11A8}/g;
    s/\x{EA30}/\x{112D}\x{1173}\x{11AB}/g;
    s/\x{EA31}/\x{112D}\x{1173}\x{11AB}\x{11BE}/g;
    s/\x{EA32}/\x{112D}\x{1173}\x{11AD}/g;
    s/\x{EA33}/\x{112D}\x{1173}\x{11AE}/g;
    s/\x{EA34}/\x{112D}\x{1173}\x{11AE}\x{11BE}/g;
    s/\x{EA35}/\x{112D}\x{1173}\x{11AF}/g;
    s/\x{EA36}/\x{112D}\x{1173}\x{11D0}/g;
    s/\x{EA37}/\x{112D}\x{1173}\x{11B3}/g;
    s/\x{EA38}/\x{112D}\x{1173}\x{11B4}/g;
    s/\x{EA39}/\x{112D}\x{1173}\x{11B6}/g;
    s/\x{EA3A}/\x{112D}\x{1173}\x{11D9}/g;
    s/\x{EA3B}/\x{112D}\x{1173}\x{11B7}/g;
    s/\x{EA3C}/\x{112D}\x{1173}\x{11BA}/g;
    s/\x{EA3D}/\x{112D}\x{1173}\x{11BA}\x{11BE}/g;
    s/\x{EA3E}/\x{112D}\x{1173}\x{11BA}\x{11C0}/g;
    s/\x{EA3F}/\x{112D}\x{1173}\x{11BA}\x{11C2}/g;
    s/\x{EA40}/\x{112D}\x{1173}\x{11EB}/g;
    s/\x{EA41}/\x{112D}\x{1173}\x{11BC}/g;
    s/\x{EA42}/\x{112D}\x{1173}\x{11F0}/g;
    s/\x{EA43}/\x{112D}\x{1173}\x{11BE}/g;
    s/\x{EA44}/\x{112D}\x{1173}\x{11C0}/g;
    s/\x{EA45}/\x{112D}\x{1174}/g;
    s/\x{EA46}/\x{112D}\x{1174}\x{11AB}/g;
    s/\x{EA47}/\x{112D}\x{1174}\x{11AF}/g;
    s/\x{EA48}/\x{112D}\x{1174}\x{11D0}/g;
    s/\x{EA49}/\x{112D}\x{1174}\x{11B7}/g;
    s/\x{EA4A}/\x{112D}\x{1174}\x{11BA}/g;
    s/\x{EA4B}/\x{112D}\x{1175}/g;
    s/\x{EA4C}/\x{112D}\x{1175}\x{11A8}/g;
    s/\x{EA4D}/\x{112D}\x{1175}\x{11AB}/g;
    s/\x{EA4E}/\x{112D}\x{1175}\x{11AD}/g;
    s/\x{EA4F}/\x{112D}\x{1175}\x{11AE}/g;
    s/\x{EA50}/\x{112D}\x{1175}\x{11AF}/g;
    s/\x{EA51}/\x{112D}\x{1175}\x{11B6}/g;
    s/\x{EA52}/\x{112D}\x{1175}\x{11B7}/g;
    s/\x{EA53}/\x{112D}\x{1175}\x{11B8}/g;
    s/\x{EA54}/\x{112D}\x{1175}\x{11BA}/g;
    s/\x{EA55}/\x{112D}\x{1175}\x{11BC}/g;
    s/\x{EA56}/\x{112D}\x{1175}\x{11C2}/g;
    s/\x{EA57}/\x{112D}\x{1198}/g;
    s/\x{EA58}/\x{112D}\x{119E}/g;
    s/\x{EA59}/\x{112D}\x{119E}\x{11A8}/g;
    s/\x{EA5A}/\x{112D}\x{119E}\x{11AB}/g;
    s/\x{EA5B}/\x{112D}\x{119E}\x{11AE}/g;
    s/\x{EA5C}/\x{112D}\x{119E}\x{11AF}/g;
    s/\x{EA5D}/\x{112D}\x{119E}\x{11B7}/g;
    s/\x{EA5E}/\x{112D}\x{119E}\x{11B8}/g;
    s/\x{EA5F}/\x{112D}\x{119E}\x{11BA}/g;
    s/\x{EA60}/\x{112D}\x{119E}\x{11BC}/g;
    s/\x{EA61}/\x{112D}\x{11A1}/g;
    s/\x{EA62}/\x{112D}\x{11A1}\x{11A8}/g;
    s/\x{EA63}/\x{112D}\x{11A1}\x{11AB}/g;
    s/\x{EA64}/\x{112D}\x{11A1}\x{11AF}/g;
    s/\x{EA65}/\x{112D}\x{11A1}\x{11D9}/g;
    s/\x{EA66}/\x{112D}\x{11A1}\x{11B7}/g;
    s/\x{EA67}/\x{112D}\x{11A1}\x{11B8}/g;
    s/\x{EA68}/\x{112D}\x{11A1}\x{11BA}/g;
    s/\x{EA69}/\x{112D}\x{11A1}\x{11BC}/g;
    s/\x{EA6A}/\x{112E}\x{1161}/g;
    s/\x{EA6B}/\x{112E}\x{1161}\x{11AF}/g;
    s/\x{EA6C}/\x{112E}\x{1167}/g;
    s/\x{EA6D}/\x{112E}\x{1167}\x{11A8}/g;
    s/\x{EA6E}/\x{112E}\x{1169}/g;
    s/\x{EA6F}/\x{112E}\x{1175}/g;
    s/\x{EA70}/\x{112E}\x{119E}/g;
    s/\x{EA71}/\x{112F}\x{1161}/g;
    s/\x{EA72}/\x{112F}\x{1161}\x{11A8}/g;
    s/\x{EA73}/\x{112F}\x{1161}\x{11AB}/g;
    s/\x{EA74}/\x{112F}\x{1161}\x{11AE}/g;
    s/\x{EA75}/\x{112F}\x{1161}\x{11AF}/g;
    s/\x{EA76}/\x{112F}\x{1161}\x{11B6}/g;
    s/\x{EA77}/\x{112F}\x{1161}\x{11B7}/g;
    s/\x{EA78}/\x{112F}\x{1161}\x{11B8}/g;
    s/\x{EA79}/\x{112F}\x{1161}\x{11BA}/g;
    s/\x{EA7A}/\x{112F}\x{1161}\x{11BC}/g;
    s/\x{EA7B}/\x{112F}\x{1161}\x{11BC}\x{11C2}/g;
    s/\x{EA7C}/\x{112F}\x{1161}\x{11F0}/g;
    s/\x{EA7D}/\x{112F}\x{1161}\x{11F0}\x{11C2}/g;
    s/\x{EA7E}/\x{112F}\x{1161}\x{11C2}/g;
    s/\x{EA7F}/\x{112F}\x{1162}/g;
    s/\x{EA80}/\x{112F}\x{1162}\x{11A8}/g;
    s/\x{EA81}/\x{112F}\x{1162}\x{11AB}/g;
    s/\x{EA82}/\x{112F}\x{1162}\x{11AF}/g;
    s/\x{EA83}/\x{112F}\x{1162}\x{11B7}/g;
    s/\x{EA84}/\x{112F}\x{1162}\x{11B8}/g;
    s/\x{EA85}/\x{112F}\x{1162}\x{11BA}/g;
    s/\x{EA86}/\x{112F}\x{1162}\x{11BC}/g;
    s/\x{EA87}/\x{112F}\x{1162}\x{11F0}/g;
    s/\x{EA88}/\x{112F}\x{1163}/g;
    s/\x{EA89}/\x{112F}\x{1163}\x{11A8}/g;
    s/\x{EA8A}/\x{112F}\x{1163}\x{11BC}/g;
    s/\x{EA8B}/\x{112F}\x{1163}\x{11F0}/g;
    s/\x{EA8C}/\x{112F}\x{1164}/g;
    s/\x{EA8D}/\x{112F}\x{1165}/g;
    s/\x{EA8E}/\x{112F}\x{1165}\x{11A8}/g;
    s/\x{EA8F}/\x{112F}\x{1165}\x{11AB}/g;
    s/\x{EA90}/\x{112F}\x{1165}\x{11AE}/g;
    s/\x{EA91}/\x{112F}\x{1165}\x{11AF}/g;
    s/\x{EA92}/\x{112F}\x{1165}\x{11B0}/g;
    s/\x{EA93}/\x{112F}\x{1165}\x{11B2}/g;
    s/\x{EA94}/\x{112F}\x{1165}\x{11B7}/g;
    s/\x{EA95}/\x{112F}\x{1165}\x{11B8}/g;
    s/\x{EA96}/\x{112F}\x{1165}\x{11BA}/g;
    s/\x{EA97}/\x{112F}\x{1165}\x{11BC}/g;
    s/\x{EA98}/\x{112F}\x{1165}\x{11F0}/g;
    s/\x{EA99}/\x{112F}\x{1165}\x{11BF}/g;
    s/\x{EA9A}/\x{112F}\x{117C}/g;
    s/\x{EA9B}/\x{112F}\x{117C}\x{11BA}/g;
    s/\x{EA9C}/\x{112F}\x{117C}\x{11BB}/g;
    s/\x{EA9D}/\x{112F}\x{117C}\x{11BF}/g;
    s/\x{EA9E}/\x{112F}\x{1166}/g;
    s/\x{EA9F}/\x{112F}\x{1166}\x{11A8}/g;
    s/\x{EAA0}/\x{112F}\x{1166}\x{11AB}/g;
    s/\x{EAA1}/\x{112F}\x{1166}\x{11AF}/g;
    s/\x{EAA2}/\x{112F}\x{1166}\x{11B7}/g;
    s/\x{EAA3}/\x{112F}\x{1166}\x{11BA}/g;
    s/\x{EAA4}/\x{112F}\x{1166}\x{11BC}/g;
    s/\x{EAA5}/\x{112F}\x{1167}/g;
    s/\x{EAA6}/\x{112F}\x{1167}\x{11A8}/g;
    s/\x{EAA7}/\x{112F}\x{1167}\x{11AB}/g;
    s/\x{EAA8}/\x{112F}\x{1167}\x{11BA}/g;
    s/\x{EAA9}/\x{112F}\x{1167}\x{11BC}/g;
    s/\x{EAAA}/\x{112F}\x{1168}/g;
    s/\x{EAAB}/\x{112F}\x{1169}/g;
    s/\x{EAAC}/\x{112F}\x{1169}\x{11A8}/g;
    s/\x{EAAD}/\x{112F}\x{1169}\x{11AA}/g;
    s/\x{EAAE}/\x{112F}\x{1169}\x{11AB}/g;
    s/\x{EAAF}/\x{112F}\x{1169}\x{11AF}/g;
    s/\x{EAB0}/\x{112F}\x{1169}\x{11B7}/g;
    s/\x{EAB1}/\x{112F}\x{1169}\x{11B8}/g;
    s/\x{EAB2}/\x{112F}\x{1169}\x{11BA}/g;
    s/\x{EAB3}/\x{112F}\x{1169}\x{11BC}/g;
    s/\x{EAB4}/\x{112F}\x{1169}\x{11F0}/g;
    s/\x{EAB5}/\x{112F}\x{1169}\x{11F1}/g;
    s/\x{EAB6}/\x{112F}\x{116A}/g;
    s/\x{EAB7}/\x{112F}\x{116B}/g;
    s/\x{EAB8}/\x{112F}\x{116C}/g;
    s/\x{EAB9}/\x{112F}\x{116C}\x{11AB}/g;
    s/\x{EABA}/\x{112F}\x{116C}\x{11AF}/g;
    s/\x{EABB}/\x{112F}\x{116D}/g;
    s/\x{EABC}/\x{112F}\x{116D}\x{11A8}/g;
    s/\x{EABD}/\x{112F}\x{116D}\x{11BA}/g;
    s/\x{EABE}/\x{112F}\x{116D}\x{11BC}/g;
    s/\x{EABF}/\x{112F}\x{116E}/g;
    s/\x{EAC0}/\x{112F}\x{116E}\x{11A8}/g;
    s/\x{EAC1}/\x{112F}\x{116E}\x{11AB}/g;
    s/\x{EAC2}/\x{112F}\x{116E}\x{11AF}/g;
    s/\x{EAC3}/\x{112F}\x{116E}\x{11B0}/g;
    s/\x{EAC4}/\x{112F}\x{116E}\x{11B2}/g;
    s/\x{EAC5}/\x{112F}\x{116E}\x{11B7}/g;
    s/\x{EAC6}/\x{112F}\x{116E}\x{11BA}/g;
    s/\x{EAC7}/\x{112F}\x{116E}\x{11BC}/g;
    s/\x{EAC8}/\x{112F}\x{116E}\x{11F0}/g;
    s/\x{EAC9}/\x{112F}\x{116F}/g;
    s/\x{EACA}/\x{112F}\x{1170}/g;
    s/\x{EACB}/\x{112F}\x{116E}\x{1167}/g;
    s/\x{EACC}/\x{112F}\x{118C}/g;
    s/\x{EACD}/\x{112F}\x{1171}/g;
    s/\x{EACE}/\x{112F}\x{1171}\x{11AB}/g;
    s/\x{EACF}/\x{112F}\x{1171}\x{11AF}/g;
    s/\x{EAD0}/\x{112F}\x{1171}\x{11B7}/g;
    s/\x{EAD1}/\x{112F}\x{1171}\x{11BA}/g;
    s/\x{EAD2}/\x{112F}\x{1172}/g;
    s/\x{EAD3}/\x{112F}\x{1172}\x{11A8}/g;
    s/\x{EAD4}/\x{112F}\x{1172}\x{11AF}/g;
    s/\x{EAD5}/\x{112F}\x{1172}\x{11B7}/g;
    s/\x{EAD6}/\x{112F}\x{1172}\x{11BC}/g;
    s/\x{EAD7}/\x{112F}\x{1172}\x{11F0}/g;
    s/\x{EAD8}/\x{112F}\x{1194}/g;
    s/\x{EAD9}/\x{112F}\x{1173}/g;
    s/\x{EADA}/\x{112F}\x{1173}\x{11A8}/g;
    s/\x{EADB}/\x{112F}\x{1173}\x{11AB}/g;
    s/\x{EADC}/\x{112F}\x{1173}\x{11AE}/g;
    s/\x{EADD}/\x{112F}\x{1173}\x{11AF}/g;
    s/\x{EADE}/\x{112F}\x{1173}\x{11B2}/g;
    s/\x{EADF}/\x{112F}\x{1173}\x{11B6}/g;
    s/\x{EAE0}/\x{112F}\x{1173}\x{11B7}/g;
    s/\x{EAE1}/\x{112F}\x{1173}\x{11B8}/g;
    s/\x{EAE2}/\x{112F}\x{1173}\x{11BA}/g;
    s/\x{EAE3}/\x{112F}\x{1173}\x{11BC}/g;
    s/\x{EAE4}/\x{112F}\x{1173}\x{11F0}/g;
    s/\x{EAE5}/\x{112F}\x{1173}\x{11F1}/g;
    s/\x{EAE6}/\x{112F}\x{1173}\x{11BD}/g;
    s/\x{EAE7}/\x{112F}\x{1174}/g;
    s/\x{EAE8}/\x{112F}\x{1174}\x{11AB}/g;
    s/\x{EAE9}/\x{112F}\x{1174}\x{11AF}/g;
    s/\x{EAEA}/\x{112F}\x{1174}\x{11BA}/g;
    s/\x{EAEB}/\x{112F}\x{1174}\x{11EB}/g;
    s/\x{EAEC}/\x{112F}\x{1174}\x{11BE}/g;
    s/\x{EAED}/\x{112F}\x{1175}/g;
    s/\x{EAEE}/\x{112F}\x{1175}\x{11A8}/g;
    s/\x{EAEF}/\x{112F}\x{1175}\x{11AB}/g;
    s/\x{EAF0}/\x{112F}\x{1175}\x{11AE}/g;
    s/\x{EAF1}/\x{112F}\x{1175}\x{11AF}/g;
    s/\x{EAF2}/\x{112F}\x{1175}\x{11B7}/g;
    s/\x{EAF3}/\x{112F}\x{1175}\x{11BA}/g;
    s/\x{EAF4}/\x{112F}\x{1175}\x{11BA}\x{11C2}/g;
    s/\x{EAF5}/\x{112F}\x{1175}\x{11BC}/g;
    s/\x{EAF6}/\x{112F}\x{1175}\x{11C2}/g;
    s/\x{EAF7}/\x{112F}\x{1198}/g;
    s/\x{EAF8}/\x{112F}\x{119E}/g;
    s/\x{EAF9}/\x{112F}\x{119E}\x{11A8}/g;
    s/\x{EAFA}/\x{112F}\x{119E}\x{11AB}/g;
    s/\x{EAFB}/\x{112F}\x{119E}\x{11AE}/g;
    s/\x{EAFC}/\x{112F}\x{119E}\x{11AF}/g;
    s/\x{EAFD}/\x{112F}\x{119E}\x{11B6}/g;
    s/\x{EAFE}/\x{112F}\x{119E}\x{11B7}/g;
    s/\x{EAFF}/\x{112F}\x{119E}\x{11B8}/g;
    s/\x{EB00}/\x{112F}\x{119E}\x{11BA}/g;
    s/\x{EB01}/\x{112F}\x{119E}\x{11BC}/g;
    s/\x{EB02}/\x{112F}\x{119E}\x{11F0}/g;
    s/\x{EB03}/\x{112F}\x{119E}\x{11C2}/g;
    s/\x{EB04}/\x{112F}\x{11A1}/g;
    s/\x{EB05}/\x{112F}\x{11A1}\x{11A8}/g;
    s/\x{EB06}/\x{112F}\x{11A1}\x{11AB}/g;
    s/\x{EB07}/\x{112F}\x{11A1}\x{11AF}/g;
    s/\x{EB08}/\x{112F}\x{11A1}\x{11B7}/g;
    s/\x{EB09}/\x{112F}\x{11A1}\x{11B8}/g;
    s/\x{EB0A}/\x{112F}\x{11A1}\x{11BA}/g;
    s/\x{EB0B}/\x{112F}\x{11A1}\x{11BC}/g;
    s/\x{EB0C}/\x{112F}\x{11A1}\x{11F0}/g;
    s/\x{EB0D}/\x{1130}\x{1161}/g;
    s/\x{EB0E}/\x{1130}\x{1162}/g;
    s/\x{EB0F}/\x{1130}\x{1162}\x{11AB}/g;
    s/\x{EB10}/\x{1130}\x{1163}/g;
    s/\x{EB11}/\x{1130}\x{1165}/g;
    s/\x{EB12}/\x{1130}\x{1165}\x{11BA}/g;
    s/\x{EB13}/\x{1130}\x{1166}/g;
    s/\x{EB14}/\x{1130}\x{1167}/g;
    s/\x{EB15}/\x{1130}\x{1167}\x{11BA}/g;
    s/\x{EB16}/\x{1130}\x{116E}/g;
    s/\x{EB17}/\x{1130}\x{1172}/g;
    s/\x{EB18}/\x{1130}\x{1172}\x{11AB}/g;
    s/\x{EB19}/\x{1130}\x{1172}\x{11BC}/g;
    s/\x{EB1A}/\x{1130}\x{1173}/g;
    s/\x{EB1B}/\x{1130}\x{1173}\x{11AB}/g;
    s/\x{EB1C}/\x{1130}\x{1173}\x{11AF}/g;
    s/\x{EB1D}/\x{1130}\x{1173}\x{11B7}/g;
    s/\x{EB1E}/\x{1130}\x{1175}/g;
    s/\x{EB1F}/\x{1130}\x{1175}\x{11AB}/g;
    s/\x{EB20}/\x{1131}\x{1161}/g;
    s/\x{EB21}/\x{1131}\x{1161}\x{11AF}/g;
    s/\x{EB22}/\x{1131}\x{1169}/g;
    s/\x{EB23}/\x{1131}\x{1173}/g;
    s/\x{EB24}/\x{1131}\x{1173}\x{11AF}/g;
    s/\x{EB25}/\x{1131}\x{119E}/g;
    s/\x{EB26}/\x{1131}\x{119E}\x{11AF}/g;
    s/\x{EB27}/\x{1132}\x{1161}/g;
    s/\x{EB28}/\x{1132}\x{1161}\x{11A8}/g;
    s/\x{EB29}/\x{1132}\x{1161}\x{11AB}/g;
    s/\x{EB2A}/\x{1132}\x{1161}\x{11AE}/g;
    s/\x{EB2B}/\x{1132}\x{1161}\x{11AF}/g;
    s/\x{EB2C}/\x{1132}\x{1161}\x{11B0}/g;
    s/\x{EB2D}/\x{1132}\x{1161}\x{11D0}/g;
    s/\x{EB2E}/\x{1132}\x{1161}\x{11B7}/g;
    s/\x{EB2F}/\x{1132}\x{1161}\x{11B8}/g;
    s/\x{EB30}/\x{1132}\x{1161}\x{11BA}/g;
    s/\x{EB31}/\x{1132}\x{1161}\x{11BC}/g;
    s/\x{EB32}/\x{1132}\x{1161}\x{11F0}/g;
    s/\x{EB33}/\x{1132}\x{1161}\x{11C2}/g;
    s/\x{EB34}/\x{1132}\x{1162}/g;
    s/\x{EB35}/\x{1132}\x{1162}\x{11A8}/g;
    s/\x{EB36}/\x{1132}\x{1162}\x{11AB}/g;
    s/\x{EB37}/\x{1132}\x{1162}\x{11AF}/g;
    s/\x{EB38}/\x{1132}\x{1162}\x{11B7}/g;
    s/\x{EB39}/\x{1132}\x{1162}\x{11BA}/g;
    s/\x{EB3A}/\x{1132}\x{1162}\x{11F0}/g;
    s/\x{EB3B}/\x{1132}\x{1163}/g;
    s/\x{EB3C}/\x{1132}\x{1163}\x{11B7}/g;
    s/\x{EB3D}/\x{1132}\x{1165}/g;
    s/\x{EB3E}/\x{1132}\x{1165}\x{11A8}/g;
    s/\x{EB3F}/\x{1132}\x{1165}\x{11AB}/g;
    s/\x{EB40}/\x{1132}\x{1165}\x{11AE}/g;
    s/\x{EB41}/\x{1132}\x{1165}\x{11AF}/g;
    s/\x{EB42}/\x{1132}\x{1165}\x{11B7}/g;
    s/\x{EB43}/\x{1132}\x{1165}\x{11B8}/g;
    s/\x{EB44}/\x{1132}\x{1165}\x{11BA}/g;
    s/\x{EB45}/\x{1132}\x{1165}\x{11BC}/g;
    s/\x{EB46}/\x{1132}\x{1165}\x{11F0}/g;
    s/\x{EB47}/\x{1132}\x{1166}/g;
    s/\x{EB48}/\x{1132}\x{1166}\x{11AB}/g;
    s/\x{EB49}/\x{1132}\x{1166}\x{11AF}/g;
    s/\x{EB4A}/\x{1132}\x{1166}\x{11B7}/g;
    s/\x{EB4B}/\x{1132}\x{1166}\x{11BA}/g;
    s/\x{EB4C}/\x{1132}\x{1167}/g;
    s/\x{EB4D}/\x{1132}\x{1167}\x{11A8}/g;
    s/\x{EB4E}/\x{1132}\x{1167}\x{11AB}/g;
    s/\x{EB4F}/\x{1132}\x{1167}\x{11AF}/g;
    s/\x{EB50}/\x{1132}\x{1167}\x{11B7}/g;
    s/\x{EB51}/\x{1132}\x{1167}\x{11BA}/g;
    s/\x{EB52}/\x{1132}\x{1167}\x{11BA}\x{11C2}/g;
    s/\x{EB53}/\x{1132}\x{1167}\x{11F0}/g;
    s/\x{EB54}/\x{1132}\x{1167}\x{11C2}/g;
    s/\x{EB55}/\x{1132}\x{1168}/g;
    s/\x{EB56}/\x{1132}\x{1169}/g;
    s/\x{EB57}/\x{1132}\x{1169}\x{11A8}/g;
    s/\x{EB58}/\x{1132}\x{1169}\x{11AB}/g;
    s/\x{EB59}/\x{1132}\x{1169}\x{11AF}/g;
    s/\x{EB5A}/\x{1132}\x{1169}\x{11B7}/g;
    s/\x{EB5B}/\x{1132}\x{1169}\x{11B8}/g;
    s/\x{EB5C}/\x{1132}\x{1169}\x{11BA}/g;
    s/\x{EB5D}/\x{1132}\x{1169}\x{11E7}/g;
    s/\x{EB5E}/\x{1132}\x{1169}\x{11BC}/g;
    s/\x{EB5F}/\x{1132}\x{1169}\x{11F0}/g;
    s/\x{EB60}/\x{1132}\x{1169}\x{11F1}/g;
    s/\x{EB61}/\x{1132}\x{1169}\x{11C2}/g;
    s/\x{EB62}/\x{1132}\x{116A}/g;
    s/\x{EB63}/\x{1132}\x{116B}/g;
    s/\x{EB64}/\x{1132}\x{116C}/g;
    s/\x{EB65}/\x{1132}\x{116D}/g;
    s/\x{EB66}/\x{1132}\x{116E}/g;
    s/\x{EB67}/\x{1132}\x{116E}\x{11A8}/g;
    s/\x{EB68}/\x{1132}\x{116E}\x{11AB}/g;
    s/\x{EB69}/\x{1132}\x{116E}\x{11AE}/g;
    s/\x{EB6A}/\x{1132}\x{116E}\x{11AF}/g;
    s/\x{EB6B}/\x{1132}\x{116E}\x{11B1}/g;
    s/\x{EB6C}/\x{1132}\x{116E}\x{11B7}/g;
    s/\x{EB6D}/\x{1132}\x{116E}\x{11BA}/g;
    s/\x{EB6E}/\x{1132}\x{116E}\x{11BC}/g;
    s/\x{EB6F}/\x{1132}\x{116E}\x{11BD}/g;
    s/\x{EB70}/\x{1132}\x{116E}\x{11BE}/g;
    s/\x{EB71}/\x{1132}\x{116F}/g;
    s/\x{EB72}/\x{1132}\x{1170}/g;
    s/\x{EB73}/\x{1132}\x{116E}\x{1167}/g;
    s/\x{EB74}/\x{1132}\x{118C}/g;
    s/\x{EB75}/\x{1132}\x{1171}/g;
    s/\x{EB76}/\x{1132}\x{1172}/g;
    s/\x{EB77}/\x{1132}\x{1194}/g;
    s/\x{EB78}/\x{1132}\x{1173}/g;
    s/\x{EB79}/\x{1132}\x{1173}\x{11A8}/g;
    s/\x{EB7A}/\x{1132}\x{1173}\x{11AB}/g;
    s/\x{EB7B}/\x{1132}\x{1173}\x{11AF}/g;
    s/\x{EB7C}/\x{1132}\x{1173}\x{11B3}/g;
    s/\x{EB7D}/\x{1132}\x{1173}\x{11B7}/g;
    s/\x{EB7E}/\x{1132}\x{1173}\x{11BA}/g;
    s/\x{EB7F}/\x{1132}\x{1173}\x{11BC}/g;
    s/\x{EB80}/\x{1132}\x{1174}/g;
    s/\x{EB81}/\x{1132}\x{1174}\x{11AF}/g;
    s/\x{EB82}/\x{1132}\x{1175}/g;
    s/\x{EB83}/\x{1132}\x{1175}\x{11A8}/g;
    s/\x{EB84}/\x{1132}\x{1175}\x{11AB}/g;
    s/\x{EB85}/\x{1132}\x{1175}\x{11AF}/g;
    s/\x{EB86}/\x{1132}\x{1175}\x{11B7}/g;
    s/\x{EB87}/\x{1132}\x{1175}\x{11BA}/g;
    s/\x{EB88}/\x{1132}\x{1175}\x{11E7}/g;
    s/\x{EB89}/\x{1132}\x{1175}\x{11BA}\x{11C2}/g;
    s/\x{EB8A}/\x{1132}\x{1175}\x{11F0}/g;
    s/\x{EB8B}/\x{1132}\x{1175}\x{11BE}/g;
    s/\x{EB8C}/\x{1132}\x{1175}\x{11C2}/g;
    s/\x{EB8D}/\x{1132}\x{1198}/g;
    s/\x{EB8E}/\x{1132}\x{119E}/g;
    s/\x{EB8F}/\x{1132}\x{119E}\x{11A8}/g;
    s/\x{EB90}/\x{1132}\x{119E}\x{11AB}/g;
    s/\x{EB91}/\x{1132}\x{119E}\x{11AF}/g;
    s/\x{EB92}/\x{1132}\x{119E}\x{11D0}/g;
    s/\x{EB93}/\x{1132}\x{119E}\x{11B7}/g;
    s/\x{EB94}/\x{1132}\x{119E}\x{11B8}/g;
    s/\x{EB95}/\x{1132}\x{119E}\x{11BA}/g;
    s/\x{EB96}/\x{1132}\x{119E}\x{11BC}/g;
    s/\x{EB97}/\x{1132}\x{11A1}/g;
    s/\x{EB98}/\x{1132}\x{11A1}\x{11A8}/g;
    s/\x{EB99}/\x{1132}\x{11A1}\x{11AB}/g;
    s/\x{EB9A}/\x{1132}\x{11A1}\x{11AF}/g;
    s/\x{EB9B}/\x{1132}\x{11A1}\x{11B7}/g;
    s/\x{EB9C}/\x{1132}\x{11A1}\x{11BA}/g;
    s/\x{EB9D}/\x{1132}\x{11A1}\x{11E7}/g;
    s/\x{EB9E}/\x{1132}\x{11A1}\x{11BC}/g;
    s/\x{EB9F}/\x{1132}\x{11A1}\x{11F0}/g;
    s/\x{EBA0}/\x{1133}\x{1161}/g;
    s/\x{EBA1}/\x{1133}\x{1162}/g;
    s/\x{EBA2}/\x{1133}\x{116E}/g;
    s/\x{EBA3}/\x{1133}\x{116E}\x{11AF}/g;
    s/\x{EBA4}/\x{1133}\x{1173}/g;
    s/\x{EBA5}/\x{1133}\x{1173}\x{11AF}/g;
    s/\x{EBA6}/\x{1133}\x{1174}/g;
    s/\x{EBA7}/\x{1133}\x{1175}/g;
    s/\x{EBA8}/\x{110A}\x{1161}\x{11E2}/g;
    s/\x{EBA9}/\x{110A}\x{1161}\x{11BA}\x{11C2}/g;
    s/\x{EBAA}/\x{110A}\x{1161}\x{11F0}/g;
    s/\x{EBAB}/\x{110A}\x{1163}\x{11E2}/g;
    s/\x{EBAC}/\x{110A}\x{1163}\x{11F0}/g;
    s/\x{EBAD}/\x{110A}\x{1165}\x{11F0}/g;
    s/\x{EBAE}/\x{110A}\x{1167}\x{11D9}/g;
    s/\x{EBAF}/\x{110A}\x{1167}\x{11F0}/g;
    s/\x{EBB0}/\x{110A}\x{1167}\x{11F9}/g;
    s/\x{EBB1}/\x{110A}\x{1169}\x{11E2}/g;
    s/\x{EBB2}/\x{110A}\x{1169}\x{11F0}/g;
    s/\x{EBB3}/\x{110A}\x{116D}\x{11E2}/g;
    s/\x{EBB4}/\x{110A}\x{116D}\x{11F0}/g;
    s/\x{EBB5}/\x{110A}\x{1188}/g;
    s/\x{EBB6}/\x{110A}\x{116E}\x{11D9}/g;
    s/\x{EBB7}/\x{110A}\x{116E}\x{11E2}/g;
    s/\x{EBB8}/\x{110A}\x{116E}\x{11F9}/g;
    s/\x{EBB9}/\x{110A}\x{1189}/g;
    s/\x{EBBA}/\x{110A}\x{1189}\x{11AB}/g;
    s/\x{EBBB}/\x{110A}\x{1189}\x{11BC}/g;
    s/\x{EBBC}/\x{110A}\x{116E}\x{1167}/g;
    s/\x{EBBD}/\x{110A}\x{116E}\x{1167}\x{11AB}/g;
    s/\x{EBBE}/\x{110A}\x{118C}/g;
    s/\x{EBBF}/\x{110A}\x{1172}\x{11D9}/g;
    s/\x{EBC0}/\x{110A}\x{1172}\x{11E2}/g;
    s/\x{EBC1}/\x{110A}\x{1172}\x{11F0}/g;
    s/\x{EBC2}/\x{110A}\x{1172}\x{11F9}/g;
    s/\x{EBC3}/\x{110A}\x{1191}/g;
    s/\x{EBC4}/\x{110A}\x{1191}\x{11AB}/g;
    s/\x{EBC5}/\x{110A}\x{1191}\x{11BC}/g;
    s/\x{EBC6}/\x{110A}\x{1192}/g;
    s/\x{EBC7}/\x{110A}\x{1192}\x{11BC}/g;
    s/\x{EBC8}/\x{110A}\x{1194}/g;
    s/\x{EBC9}/\x{110A}\x{1194}\x{11BC}/g;
    s/\x{EBCA}/\x{110A}\x{1194}\x{11F0}/g;
    s/\x{EBCB}/\x{110A}\x{1173}\x{11D9}/g;
    s/\x{EBCC}/\x{110A}\x{1173}\x{11EB}/g;
    s/\x{EBCD}/\x{110A}\x{1173}\x{11F0}/g;
    s/\x{EBCE}/\x{110A}\x{1175}\x{11D9}/g;
    s/\x{EBCF}/\x{110A}\x{1175}\x{11E2}/g;
    s/\x{EBD0}/\x{110A}\x{1175}\x{11EB}/g;
    s/\x{EBD1}/\x{110A}\x{1175}\x{11F0}/g;
    s/\x{EBD2}/\x{110A}\x{1175}\x{11F9}/g;
    s/\x{EBD3}/\x{110A}\x{1198}/g;
    s/\x{EBD4}/\x{110A}\x{119E}/g;
    s/\x{EBD5}/\x{110A}\x{119E}\x{11A8}/g;
    s/\x{EBD6}/\x{110A}\x{119E}\x{11AB}/g;
    s/\x{EBD7}/\x{110A}\x{119E}\x{11AE}/g;
    s/\x{EBD8}/\x{110A}\x{119E}\x{11AF}/g;
    s/\x{EBD9}/\x{110A}\x{119E}\x{11B7}/g;
    s/\x{EBDA}/\x{110A}\x{119E}\x{11DD}/g;
    s/\x{EBDB}/\x{110A}\x{119E}\x{11B8}/g;
    s/\x{EBDC}/\x{110A}\x{119E}\x{11E6}/g;
    s/\x{EBDD}/\x{110A}\x{119E}\x{11BA}/g;
    s/\x{EBDE}/\x{110A}\x{119E}\x{11BC}/g;
    s/\x{EBDF}/\x{110A}\x{119E}\x{11F0}/g;
    s/\x{EBE0}/\x{110A}\x{119E}\x{11C1}/g;
    s/\x{EBE1}/\x{110A}\x{11A1}/g;
    s/\x{EBE2}/\x{110A}\x{11A1}\x{11AB}/g;
    s/\x{EBE3}/\x{110A}\x{11A1}\x{11F0}/g;
    s/\x{EBE4}/\x{1109}\x{1132}\x{1169}/g;
    s/\x{EBE5}/\x{1134}\x{1173}/g;
    s/\x{EBE6}/\x{1135}\x{1173}/g;
    s/\x{EBE7}/\x{1135}\x{1173}\x{11AF}/g;
    s/\x{EBE8}/\x{1135}\x{1173}\x{11B7}/g;
    s/\x{EBE9}/\x{1135}\x{1175}/g;
    s/\x{EBEA}/\x{1135}\x{1175}\x{11AF}/g;
    s/\x{EBEB}/\x{1136}\x{1161}/g;
    s/\x{EBEC}/\x{1136}\x{1161}\x{11A8}/g;
    s/\x{EBED}/\x{1136}\x{1161}\x{11AB}/g;
    s/\x{EBEE}/\x{1136}\x{1161}\x{11AF}/g;
    s/\x{EBEF}/\x{1136}\x{1161}\x{11B2}/g;
    s/\x{EBF0}/\x{1136}\x{1161}\x{11B7}/g;
    s/\x{EBF1}/\x{1136}\x{1161}\x{11B8}/g;
    s/\x{EBF2}/\x{1136}\x{1161}\x{11BA}/g;
    s/\x{EBF3}/\x{1136}\x{1161}\x{11E9}/g;
    s/\x{EBF4}/\x{1136}\x{1161}\x{11BC}/g;
    s/\x{EBF5}/\x{1136}\x{1161}\x{11F0}/g;
    s/\x{EBF6}/\x{1136}\x{1162}/g;
    s/\x{EBF7}/\x{1136}\x{1162}\x{11A8}/g;
    s/\x{EBF8}/\x{1136}\x{1162}\x{11AB}/g;
    s/\x{EBF9}/\x{1136}\x{1162}\x{11AF}/g;
    s/\x{EBFA}/\x{1136}\x{1162}\x{11BA}/g;
    s/\x{EBFB}/\x{1136}\x{1162}\x{11BC}/g;
    s/\x{EBFC}/\x{1136}\x{1162}\x{11F0}/g;
    s/\x{EBFD}/\x{1136}\x{1163}/g;
    s/\x{EBFE}/\x{1136}\x{1163}\x{11A8}/g;
    s/\x{EBFF}/\x{1136}\x{1163}\x{11BC}/g;
    s/\x{EC00}/\x{1136}\x{1163}\x{11F0}/g;
    s/\x{EC01}/\x{1136}\x{1165}/g;
    s/\x{EC02}/\x{1136}\x{1165}\x{11A8}/g;
    s/\x{EC03}/\x{1136}\x{1165}\x{11AB}/g;
    s/\x{EC04}/\x{1136}\x{1165}\x{11AF}/g;
    s/\x{EC05}/\x{1136}\x{1165}\x{11B7}/g;
    s/\x{EC06}/\x{1136}\x{1165}\x{11B8}/g;
    s/\x{EC07}/\x{1136}\x{1165}\x{11BA}/g;
    s/\x{EC08}/\x{1136}\x{1165}\x{11BC}/g;
    s/\x{EC09}/\x{1136}\x{1166}/g;
    s/\x{EC0A}/\x{1136}\x{1166}\x{11AB}/g;
    s/\x{EC0B}/\x{1136}\x{1166}\x{11AF}/g;
    s/\x{EC0C}/\x{1136}\x{1166}\x{11B8}/g;
    s/\x{EC0D}/\x{1136}\x{1166}\x{11BA}/g;
    s/\x{EC0E}/\x{1136}\x{1167}/g;
    s/\x{EC0F}/\x{1136}\x{1167}\x{11A8}/g;
    s/\x{EC10}/\x{1136}\x{1167}\x{11AB}/g;
    s/\x{EC11}/\x{1136}\x{1167}\x{11AF}/g;
    s/\x{EC12}/\x{1136}\x{1167}\x{11BA}/g;
    s/\x{EC13}/\x{1136}\x{1167}\x{11F0}/g;
    s/\x{EC14}/\x{1136}\x{1168}/g;
    s/\x{EC15}/\x{1136}\x{1168}\x{11AB}/g;
    s/\x{EC16}/\x{1136}\x{1169}/g;
    s/\x{EC17}/\x{1136}\x{1169}\x{11A8}/g;
    s/\x{EC18}/\x{1136}\x{1169}\x{11AB}/g;
    s/\x{EC19}/\x{1136}\x{1169}\x{11AF}/g;
    s/\x{EC1A}/\x{1136}\x{1169}\x{11B7}/g;
    s/\x{EC1B}/\x{1136}\x{1169}\x{11B8}/g;
    s/\x{EC1C}/\x{1136}\x{1169}\x{11BA}/g;
    s/\x{EC1D}/\x{1136}\x{1169}\x{11BC}/g;
    s/\x{EC1E}/\x{1136}\x{1169}\x{11BD}/g;
    s/\x{EC1F}/\x{1136}\x{1169}\x{11BE}/g;
    s/\x{EC20}/\x{1136}\x{116A}/g;
    s/\x{EC21}/\x{1136}\x{116A}\x{11A8}/g;
    s/\x{EC22}/\x{1136}\x{116C}/g;
    s/\x{EC23}/\x{1136}\x{116C}\x{11A8}/g;
    s/\x{EC24}/\x{1136}\x{116C}\x{11AB}/g;
    s/\x{EC25}/\x{1136}\x{116C}\x{11AF}/g;
    s/\x{EC26}/\x{1136}\x{116C}\x{11B7}/g;
    s/\x{EC27}/\x{1136}\x{116D}/g;
    s/\x{EC28}/\x{1136}\x{116D}\x{11A8}/g;
    s/\x{EC29}/\x{1136}\x{116D}\x{11AE}/g;
    s/\x{EC2A}/\x{1136}\x{116D}\x{11B7}/g;
    s/\x{EC2B}/\x{1136}\x{116D}\x{11BA}/g;
    s/\x{EC2C}/\x{1136}\x{116D}\x{11BC}/g;
    s/\x{EC2D}/\x{1136}\x{116E}/g;
    s/\x{EC2E}/\x{1136}\x{116E}\x{11A8}/g;
    s/\x{EC2F}/\x{1136}\x{116E}\x{11AB}/g;
    s/\x{EC30}/\x{1136}\x{116E}\x{11AF}/g;
    s/\x{EC31}/\x{1136}\x{116E}\x{11B7}/g;
    s/\x{EC32}/\x{1136}\x{116E}\x{11B8}/g;
    s/\x{EC33}/\x{1136}\x{116E}\x{11BC}/g;
    s/\x{EC34}/\x{1136}\x{116F}/g;
    s/\x{EC35}/\x{1136}\x{1170}/g;
    s/\x{EC36}/\x{1136}\x{1171}/g;
    s/\x{EC37}/\x{1136}\x{1171}\x{11A8}/g;
    s/\x{EC38}/\x{1136}\x{1172}/g;
    s/\x{EC39}/\x{1136}\x{1172}\x{11A8}/g;
    s/\x{EC3A}/\x{1136}\x{1172}\x{11AF}/g;
    s/\x{EC3B}/\x{1136}\x{1172}\x{11B7}/g;
    s/\x{EC3C}/\x{1136}\x{1173}/g;
    s/\x{EC3D}/\x{1136}\x{1173}\x{11A8}/g;
    s/\x{EC3E}/\x{1136}\x{1173}\x{11AB}/g;
    s/\x{EC3F}/\x{1136}\x{1173}\x{11AF}/g;
    s/\x{EC40}/\x{1136}\x{1173}\x{11B7}/g;
    s/\x{EC41}/\x{1136}\x{1173}\x{11BA}/g;
    s/\x{EC42}/\x{1136}\x{1173}\x{11BC}/g;
    s/\x{EC43}/\x{1136}\x{1174}/g;
    s/\x{EC44}/\x{1136}\x{1174}\x{11AF}/g;
    s/\x{EC45}/\x{1136}\x{1175}/g;
    s/\x{EC46}/\x{1136}\x{1175}\x{11A8}/g;
    s/\x{EC47}/\x{1136}\x{1175}\x{11AB}/g;
    s/\x{EC48}/\x{1136}\x{1175}\x{11AF}/g;
    s/\x{EC49}/\x{1136}\x{1175}\x{11D9}/g;
    s/\x{EC4A}/\x{1136}\x{1175}\x{11B7}/g;
    s/\x{EC4B}/\x{1136}\x{1175}\x{11B8}/g;
    s/\x{EC4C}/\x{1136}\x{1175}\x{11BA}/g;
    s/\x{EC4D}/\x{1136}\x{1175}\x{11BA}\x{11C2}/g;
    s/\x{EC4E}/\x{1136}\x{1175}\x{11BC}/g;
    s/\x{EC4F}/\x{1136}\x{1175}\x{11F0}/g;
    s/\x{EC50}/\x{1136}\x{1175}\x{11C2}/g;
    s/\x{EC51}/\x{1136}\x{119E}/g;
    s/\x{EC52}/\x{1136}\x{119E}\x{11A8}/g;
    s/\x{EC53}/\x{1136}\x{119E}\x{11AB}/g;
    s/\x{EC54}/\x{1136}\x{119E}\x{11AF}/g;
    s/\x{EC55}/\x{1136}\x{119E}\x{11B7}/g;
    s/\x{EC56}/\x{1136}\x{119E}\x{11B8}/g;
    s/\x{EC57}/\x{1136}\x{119E}\x{11BA}/g;
    s/\x{EC58}/\x{1136}\x{119E}\x{11BC}/g;
    s/\x{EC59}/\x{1136}\x{11A1}/g;
    s/\x{EC5A}/\x{1136}\x{11A1}\x{11A8}/g;
    s/\x{EC5B}/\x{1136}\x{11A1}\x{11AF}/g;
    s/\x{EC5C}/\x{1136}\x{11A1}\x{11BC}/g;
    s/\x{EC5D}/\x{1136}\x{11A1}\x{11F0}/g;
    s/\x{EC5E}/\x{1137}\x{1169}/g;
    s/\x{EC5F}/\x{1137}\x{1175}/g;
    s/\x{EC60}/\x{1137}\x{119E}/g;
    s/\x{EC61}/\x{1137}\x{119E}\x{11AF}/g;
    s/\x{EC62}/\x{1137}\x{119E}\x{11B8}/g;
    s/\x{EC63}/\x{1138}\x{1161}/g;
    s/\x{EC64}/\x{1138}\x{1161}\x{11BC}/g;
    s/\x{EC65}/\x{1138}\x{1165}/g;
    s/\x{EC66}/\x{1138}\x{1165}\x{11AB}/g;
    s/\x{EC67}/\x{1138}\x{1165}\x{11AF}/g;
    s/\x{EC68}/\x{1138}\x{1167}/g;
    s/\x{EC69}/\x{1138}\x{1169}/g;
    s/\x{EC6A}/\x{1138}\x{1169}\x{11AB}/g;
    s/\x{EC6B}/\x{1138}\x{1169}\x{11BA}/g;
    s/\x{EC6C}/\x{1138}\x{1169}\x{11BC}/g;
    s/\x{EC6D}/\x{1138}\x{1169}\x{11F0}/g;
    s/\x{EC6E}/\x{1138}\x{116A}/g;
    s/\x{EC6F}/\x{1138}\x{116B}/g;
    s/\x{EC70}/\x{1138}\x{116C}/g;
    s/\x{EC71}/\x{1138}\x{116C}\x{11BC}/g;
    s/\x{EC72}/\x{1138}\x{116C}\x{11F0}/g;
    s/\x{EC73}/\x{1138}\x{116E}/g;
    s/\x{EC74}/\x{1138}\x{116F}/g;
    s/\x{EC75}/\x{1138}\x{1173}/g;
    s/\x{EC76}/\x{1138}\x{1173}\x{11AB}/g;
    s/\x{EC77}/\x{1138}\x{1174}/g;
    s/\x{EC78}/\x{1138}\x{1175}/g;
    s/\x{EC79}/\x{1138}\x{119E}/g;
    s/\x{EC7A}/\x{1139}\x{1161}/g;
    s/\x{EC7B}/\x{1139}\x{1161}\x{11AF}/g;
    s/\x{EC7C}/\x{1139}\x{1165}/g;
    s/\x{EC7D}/\x{1139}\x{1165}\x{11AF}/g;
    s/\x{EC7E}/\x{1139}\x{1166}/g;
    s/\x{EC7F}/\x{1139}\x{1169}/g;
    s/\x{EC80}/\x{1139}\x{1169}\x{11B8}/g;
    s/\x{EC81}/\x{1139}\x{116E}/g;
    s/\x{EC82}/\x{1139}\x{116E}\x{11A8}/g;
    s/\x{EC83}/\x{1139}\x{1171}/g;
    s/\x{EC84}/\x{1139}\x{1173}/g;
    s/\x{EC85}/\x{1139}\x{1173}\x{11AF}/g;
    s/\x{EC86}/\x{1139}\x{1173}\x{11B7}/g;
    s/\x{EC87}/\x{1139}\x{1173}\x{11BA}/g;
    s/\x{EC88}/\x{1139}\x{1175}/g;
    s/\x{EC89}/\x{1139}\x{1175}\x{11BF}/g;
    s/\x{EC8A}/\x{1139}\x{119E}/g;
    s/\x{EC8B}/\x{1139}\x{119E}\x{11AF}/g;
    s/\x{EC8C}/\x{1139}\x{119E}\x{11B7}/g;
    s/\x{EC8D}/\x{113A}\x{1167}/g;
    s/\x{EC8E}/\x{113A}\x{1167}\x{11AB}/g;
    s/\x{EC8F}/\x{113A}\x{116E}/g;
    s/\x{EC90}/\x{113A}\x{116E}\x{11AB}/g;
    s/\x{EC91}/\x{113A}\x{1175}/g;
    s/\x{EC92}/\x{113A}\x{119E}/g;
    s/\x{EC93}/\x{113A}\x{119E}\x{11AF}/g;
    s/\x{EC94}/\x{113B}\x{1165}/g;
    s/\x{EC95}/\x{113B}\x{1165}\x{11AF}/g;
    s/\x{EC96}/\x{113B}\x{1167}/g;
    s/\x{EC97}/\x{113B}\x{1173}/g;
    s/\x{EC98}/\x{113B}\x{1173}\x{11AF}/g;
    s/\x{EC99}/\x{113C}\x{1161}/g;
    s/\x{EC9A}/\x{113C}\x{1161}\x{11AB}/g;
    s/\x{EC9B}/\x{113C}\x{1161}\x{11F0}/g;
    s/\x{EC9C}/\x{113C}\x{1162}/g;
    s/\x{EC9D}/\x{113C}\x{1163}/g;
    s/\x{EC9E}/\x{113C}\x{1163}\x{11B7}/g;
    s/\x{EC9F}/\x{113C}\x{1163}\x{11E2}/g;
    s/\x{ECA0}/\x{113C}\x{1163}\x{11F0}/g;
    s/\x{ECA1}/\x{113C}\x{1165}/g;
    s/\x{ECA2}/\x{113C}\x{1165}\x{11AB}/g;
    s/\x{ECA3}/\x{113C}\x{1167}/g;
    s/\x{ECA4}/\x{113C}\x{1167}\x{11AB}/g;
    s/\x{ECA5}/\x{113C}\x{1168}/g;
    s/\x{ECA6}/\x{113C}\x{116E}/g;
    s/\x{ECA7}/\x{113C}\x{116E}\x{11AB}/g;
    s/\x{ECA8}/\x{113C}\x{116E}\x{11BC}/g;
    s/\x{ECA9}/\x{113C}\x{1171}/g;
    s/\x{ECAA}/\x{113C}\x{1172}/g;
    s/\x{ECAB}/\x{113C}\x{1172}\x{11AB}/g;
    s/\x{ECAC}/\x{113C}\x{1191}/g;
    s/\x{ECAD}/\x{113C}\x{1191}\x{11AB}/g;
    s/\x{ECAE}/\x{113C}\x{1173}/g;
    s/\x{ECAF}/\x{113C}\x{1173}\x{11AB}/g;
    s/\x{ECB0}/\x{113C}\x{1173}\x{11EB}/g;
    s/\x{ECB1}/\x{113C}\x{1175}/g;
    s/\x{ECB2}/\x{113C}\x{1175}\x{11BC}/g;
    s/\x{ECB3}/\x{113D}\x{1163}/g;
    s/\x{ECB4}/\x{113D}\x{1163}\x{11BC}/g;
    s/\x{ECB5}/\x{113D}\x{1163}\x{11F0}/g;
    s/\x{ECB6}/\x{113D}\x{1167}/g;
    s/\x{ECB7}/\x{113D}\x{1167}\x{11AB}/g;
    s/\x{ECB8}/\x{113D}\x{116E}/g;
    s/\x{ECB9}/\x{113D}\x{1171}/g;
    s/\x{ECBA}/\x{113D}\x{1172}/g;
    s/\x{ECBB}/\x{113D}\x{1172}\x{11AB}/g;
    s/\x{ECBC}/\x{113D}\x{1172}\x{11F0}/g;
    s/\x{ECBD}/\x{113D}\x{1191}/g;
    s/\x{ECBE}/\x{113D}\x{1191}\x{11AB}/g;
    s/\x{ECBF}/\x{113D}\x{1194}/g;
    s/\x{ECC0}/\x{113D}\x{1173}/g;
    s/\x{ECC1}/\x{113D}\x{1173}\x{11EB}/g;
    s/\x{ECC2}/\x{113D}\x{1175}/g;
    s/\x{ECC3}/\x{113D}\x{1175}\x{11AB}/g;
    s/\x{ECC4}/\x{113D}\x{1175}\x{11EB}/g;
    s/\x{ECC5}/\x{113D}\x{1175}\x{11F9}/g;
    s/\x{ECC6}/\x{113E}\x{1161}/g;
    s/\x{ECC7}/\x{113E}\x{1161}\x{11AB}/g;
    s/\x{ECC8}/\x{113E}\x{1161}\x{11AE}/g;
    s/\x{ECC9}/\x{113E}\x{1162}/g;
    s/\x{ECCA}/\x{113E}\x{1163}/g;
    s/\x{ECCB}/\x{113E}\x{1163}\x{11B7}/g;
    s/\x{ECCC}/\x{113E}\x{1163}\x{11E2}/g;
    s/\x{ECCD}/\x{113E}\x{1167}/g;
    s/\x{ECCE}/\x{113E}\x{1167}\x{11AB}/g;
    s/\x{ECCF}/\x{113E}\x{1169}/g;
    s/\x{ECD0}/\x{113E}\x{116A}/g;
    s/\x{ECD1}/\x{113E}\x{116B}/g;
    s/\x{ECD2}/\x{113E}\x{116E}/g;
    s/\x{ECD3}/\x{113E}\x{116E}\x{11A8}/g;
    s/\x{ECD4}/\x{113E}\x{116E}\x{11BC}/g;
    s/\x{ECD5}/\x{113E}\x{1171}/g;
    s/\x{ECD6}/\x{113E}\x{1172}/g;
    s/\x{ECD7}/\x{113E}\x{1172}\x{11AB}/g;
    s/\x{ECD8}/\x{113E}\x{1191}/g;
    s/\x{ECD9}/\x{113E}\x{1191}\x{11AB}/g;
    s/\x{ECDA}/\x{113E}\x{1173}/g;
    s/\x{ECDB}/\x{113E}\x{1173}\x{11EB}/g;
    s/\x{ECDC}/\x{113E}\x{1173}\x{11BC}/g;
    s/\x{ECDD}/\x{113E}\x{1175}/g;
    s/\x{ECDE}/\x{113E}\x{1175}\x{11AB}/g;
    s/\x{ECDF}/\x{113E}\x{1175}\x{11B7}/g;
    s/\x{ECE0}/\x{113E}\x{1175}\x{11E2}/g;
    s/\x{ECE1}/\x{113F}\x{1163}/g;
    s/\x{ECE2}/\x{113F}\x{1163}\x{11F0}/g;
    s/\x{ECE3}/\x{113F}\x{1167}/g;
    s/\x{ECE4}/\x{113F}\x{1167}\x{11AB}/g;
    s/\x{ECE5}/\x{113F}\x{116E}/g;
    s/\x{ECE6}/\x{113F}\x{1171}/g;
    s/\x{ECE7}/\x{113F}\x{1172}/g;
    s/\x{ECE8}/\x{113F}\x{1172}\x{11A8}/g;
    s/\x{ECE9}/\x{113F}\x{1172}\x{11AB}/g;
    s/\x{ECEA}/\x{113F}\x{1172}\x{11F0}/g;
    s/\x{ECEB}/\x{113F}\x{1173}/g;
    s/\x{ECEC}/\x{113F}\x{1173}\x{11EB}/g;
    s/\x{ECED}/\x{113F}\x{1175}/g;
    s/\x{ECEE}/\x{113F}\x{1175}\x{11AB}/g;
    s/\x{ECEF}/\x{113F}\x{1175}\x{11F9}/g;
    s/\x{ECF0}/\x{1140}\x{1161}/g;
    s/\x{ECF1}/\x{1140}\x{1161}\x{11A8}/g;
    s/\x{ECF2}/\x{1140}\x{1161}\x{11AB}/g;
    s/\x{ECF3}/\x{1140}\x{1161}\x{11AE}/g;
    s/\x{ECF4}/\x{1140}\x{1161}\x{11AF}/g;
    s/\x{ECF5}/\x{1140}\x{1161}\x{11B7}/g;
    s/\x{ECF6}/\x{1140}\x{1161}\x{11DD}/g;
    s/\x{ECF7}/\x{1140}\x{1161}\x{11B8}/g;
    s/\x{ECF8}/\x{1140}\x{1161}\x{11BA}/g;
    s/\x{ECF9}/\x{1140}\x{1161}\x{11BC}/g;
    s/\x{ECFA}/\x{1140}\x{1161}\x{11F0}/g;
    s/\x{ECFB}/\x{1140}\x{1162}/g;
    s/\x{ECFC}/\x{1140}\x{1162}\x{11AB}/g;
    s/\x{ECFD}/\x{1140}\x{1162}\x{11BA}/g;
    s/\x{ECFE}/\x{1140}\x{1162}\x{11EB}/g;
    s/\x{ECFF}/\x{1140}\x{1162}\x{11BC}/g;
    s/\x{ED00}/\x{1140}\x{1162}\x{11F0}/g;
    s/\x{ED01}/\x{1140}\x{1163}/g;
    s/\x{ED02}/\x{1140}\x{1163}\x{11A8}/g;
    s/\x{ED03}/\x{1140}\x{1163}\x{11AB}/g;
    s/\x{ED04}/\x{1140}\x{1163}\x{11B7}/g;
    s/\x{ED05}/\x{1140}\x{1163}\x{11E2}/g;
    s/\x{ED06}/\x{1140}\x{1163}\x{11B8}/g;
    s/\x{ED07}/\x{1140}\x{1163}\x{11E6}/g;
    s/\x{ED08}/\x{1140}\x{1163}\x{11BC}/g;
    s/\x{ED09}/\x{1140}\x{1163}\x{11F0}/g;
    s/\x{ED0A}/\x{1140}\x{1163}\x{11F9}/g;
    s/\x{ED0B}/\x{1140}\x{1178}/g;
    s/\x{ED0C}/\x{1140}\x{1165}/g;
    s/\x{ED0D}/\x{1140}\x{1165}\x{11A8}/g;
    s/\x{ED0E}/\x{1140}\x{1165}\x{11AB}/g;
    s/\x{ED0F}/\x{1140}\x{1165}\x{11AF}/g;
    s/\x{ED10}/\x{1140}\x{1165}\x{11B7}/g;
    s/\x{ED11}/\x{1140}\x{1165}\x{11DD}/g;
    s/\x{ED12}/\x{1140}\x{1165}\x{11B8}/g;
    s/\x{ED13}/\x{1140}\x{1165}\x{11BA}/g;
    s/\x{ED14}/\x{1140}\x{1165}\x{11EB}/g;
    s/\x{ED15}/\x{1140}\x{1165}\x{11BC}/g;
    s/\x{ED16}/\x{1140}\x{1165}\x{11F0}/g;
    s/\x{ED17}/\x{1140}\x{1165}\x{11C2}/g;
    s/\x{ED18}/\x{1140}\x{117B}/g;
    s/\x{ED19}/\x{1140}\x{1166}/g;
    s/\x{ED1A}/\x{1140}\x{1166}\x{11AB}/g;
    s/\x{ED1B}/\x{1140}\x{1166}\x{11BA}/g;
    s/\x{ED1C}/\x{1140}\x{1167}/g;
    s/\x{ED1D}/\x{1140}\x{1167}\x{11A8}/g;
    s/\x{ED1E}/\x{1140}\x{1167}\x{11AB}/g;
    s/\x{ED1F}/\x{1140}\x{1167}\x{11AF}/g;
    s/\x{ED20}/\x{1140}\x{1167}\x{11B3}/g;
    s/\x{ED21}/\x{1140}\x{1167}\x{11D9}/g;
    s/\x{ED22}/\x{1140}\x{1167}\x{11B7}/g;
    s/\x{ED23}/\x{1140}\x{1167}\x{11B8}/g;
    s/\x{ED24}/\x{1140}\x{1167}\x{11BC}/g;
    s/\x{ED25}/\x{1140}\x{1167}\x{11F0}/g;
    s/\x{ED26}/\x{1140}\x{1167}\x{11F9}/g;
    s/\x{ED27}/\x{1140}\x{1168}/g;
    s/\x{ED28}/\x{1140}\x{1168}\x{11AB}/g;
    s/\x{ED29}/\x{1140}\x{1168}\x{11AE}/g;
    s/\x{ED2A}/\x{1140}\x{1168}\x{11B7}/g;
    s/\x{ED2B}/\x{1140}\x{1168}\x{11BA}/g;
    s/\x{ED2C}/\x{1140}\x{1168}\x{11BC}/g;
    s/\x{ED2D}/\x{1140}\x{1169}/g;
    s/\x{ED2E}/\x{1140}\x{1169}\x{11AB}/g;
    s/\x{ED2F}/\x{1140}\x{1169}\x{11AF}/g;
    s/\x{ED30}/\x{1140}\x{1169}\x{11D9}/g;
    s/\x{ED31}/\x{1140}\x{1169}\x{11B7}/g;
    s/\x{ED32}/\x{1140}\x{1169}\x{11B8}/g;
    s/\x{ED33}/\x{1140}\x{1169}\x{11EB}/g;
    s/\x{ED34}/\x{1140}\x{1169}\x{11BC}/g;
    s/\x{ED35}/\x{1140}\x{1169}\x{11F0}/g;
    s/\x{ED36}/\x{1140}\x{116A}/g;
    s/\x{ED37}/\x{1140}\x{116B}/g;
    s/\x{ED38}/\x{1140}\x{116C}/g;
    s/\x{ED39}/\x{1140}\x{116D}/g;
    s/\x{ED3A}/\x{1140}\x{116D}\x{11A8}/g;
    s/\x{ED3B}/\x{1140}\x{116D}\x{11B7}/g;
    s/\x{ED3C}/\x{1140}\x{116D}\x{11E2}/g;
    s/\x{ED3D}/\x{1140}\x{116D}\x{11BA}/g;
    s/\x{ED3E}/\x{1140}\x{116D}\x{11BC}/g;
    s/\x{ED3F}/\x{1140}\x{116D}\x{11F0}/g;
    s/\x{ED40}/\x{1140}\x{116D}\x{11C2}/g;
    s/\x{ED41}/\x{1140}\x{116E}/g;
    s/\x{ED42}/\x{1140}\x{116E}\x{11A8}/g;
    s/\x{ED43}/\x{1140}\x{116E}\x{11AB}/g;
    s/\x{ED44}/\x{1140}\x{116E}\x{11AF}/g;
    s/\x{ED45}/\x{1140}\x{116E}\x{11D9}/g;
    s/\x{ED46}/\x{1140}\x{116E}\x{11B7}/g;
    s/\x{ED47}/\x{1140}\x{116E}\x{11DD}/g;
    s/\x{ED48}/\x{1140}\x{116E}\x{11B8}/g;
    s/\x{ED49}/\x{1140}\x{116E}\x{11BA}/g;
    s/\x{ED4A}/\x{1140}\x{116E}\x{11BC}/g;
    s/\x{ED4B}/\x{1140}\x{116E}\x{11F0}/g;
    s/\x{ED4C}/\x{1140}\x{116E}\x{11BE}/g;
    s/\x{ED4D}/\x{1140}\x{116E}\x{11C2}/g;
    s/\x{ED4E}/\x{1140}\x{1189}/g;
    s/\x{ED4F}/\x{1140}\x{1189}\x{11AB}/g;
    s/\x{ED50}/\x{1140}\x{116F}/g;
    s/\x{ED51}/\x{1140}\x{1170}/g;
    s/\x{ED52}/\x{1140}\x{116E}\x{1167}/g;
    s/\x{ED53}/\x{1140}\x{118C}/g;
    s/\x{ED54}/\x{1140}\x{1171}/g;
    s/\x{ED55}/\x{1140}\x{1171}\x{11AB}/g;
    s/\x{ED56}/\x{1140}\x{1172}/g;
    s/\x{ED57}/\x{1140}\x{1172}\x{11A8}/g;
    s/\x{ED58}/\x{1140}\x{1172}\x{11AB}/g;
    s/\x{ED59}/\x{1140}\x{1172}\x{11B7}/g;
    s/\x{ED5A}/\x{1140}\x{1172}\x{11E2}/g;
    s/\x{ED5B}/\x{1140}\x{1172}\x{11BA}/g;
    s/\x{ED5C}/\x{1140}\x{1172}\x{11BC}/g;
    s/\x{ED5D}/\x{1140}\x{1172}\x{11F0}/g;
    s/\x{ED5E}/\x{1140}\x{1172}\x{11BD}/g;
    s/\x{ED5F}/\x{1140}\x{1172}\x{11BE}/g;
    s/\x{ED60}/\x{1140}\x{1172}\x{11F9}/g;
    s/\x{ED61}/\x{1140}\x{118F}/g;
    s/\x{ED62}/\x{1140}\x{118F}\x{11AB}/g;
    s/\x{ED63}/\x{1140}\x{1191}/g;
    s/\x{ED64}/\x{1140}\x{1191}\x{11AB}/g;
    s/\x{ED65}/\x{1140}\x{1191}\x{11AF}/g;
    s/\x{ED66}/\x{1140}\x{1191}\x{11D9}/g;
    s/\x{ED67}/\x{1140}\x{1192}/g;
    s/\x{ED68}/\x{1140}\x{1192}\x{11BC}/g;
    s/\x{ED69}/\x{1140}\x{1194}/g;
    s/\x{ED6A}/\x{1140}\x{1194}\x{11BC}/g;
    s/\x{ED6B}/\x{1140}\x{1173}/g;
    s/\x{ED6C}/\x{1140}\x{1173}\x{11A8}/g;
    s/\x{ED6D}/\x{1140}\x{1173}\x{11AB}/g;
    s/\x{ED6E}/\x{1140}\x{1173}\x{11AF}/g;
    s/\x{ED6F}/\x{1140}\x{1173}\x{11B1}/g;
    s/\x{ED70}/\x{1140}\x{1173}\x{11B3}/g;
    s/\x{ED71}/\x{1140}\x{1173}\x{11B6}/g;
    s/\x{ED72}/\x{1140}\x{1173}\x{11D9}/g;
    s/\x{ED73}/\x{1140}\x{1173}\x{11B7}/g;
    s/\x{ED74}/\x{1140}\x{1173}\x{11DD}/g;
    s/\x{ED75}/\x{1140}\x{1173}\x{11E0}/g;
    s/\x{ED76}/\x{1140}\x{1173}\x{11B8}/g;
    s/\x{ED77}/\x{1140}\x{1173}\x{11BA}/g;
    s/\x{ED78}/\x{1140}\x{1173}\x{11BA}\x{11BE}/g;
    s/\x{ED79}/\x{1140}\x{1173}\x{11EB}/g;
    s/\x{ED7A}/\x{1140}\x{1173}\x{11F0}/g;
    s/\x{ED7B}/\x{1140}\x{1173}\x{11BE}/g;
    s/\x{ED7C}/\x{1140}\x{1174}/g;
    s/\x{ED7D}/\x{1140}\x{1174}\x{11AB}/g;
    s/\x{ED7E}/\x{1140}\x{1174}\x{11BA}/g;
    s/\x{ED7F}/\x{1140}\x{1175}/g;
    s/\x{ED80}/\x{1140}\x{1175}\x{11A8}/g;
    s/\x{ED81}/\x{1140}\x{1175}\x{11AB}/g;
    s/\x{ED82}/\x{1140}\x{1175}\x{11AE}/g;
    s/\x{ED83}/\x{1140}\x{1175}\x{11AF}/g;
    s/\x{ED84}/\x{1140}\x{1175}\x{11B3}/g;
    s/\x{ED85}/\x{1140}\x{1175}\x{11D9}/g;
    s/\x{ED86}/\x{1140}\x{1175}\x{11B7}/g;
    s/\x{ED87}/\x{1140}\x{1175}\x{11B8}/g;
    s/\x{ED88}/\x{1140}\x{1175}\x{11E6}/g;
    s/\x{ED89}/\x{1140}\x{1175}\x{11BA}/g;
    s/\x{ED8A}/\x{1140}\x{1175}\x{11EB}/g;
    s/\x{ED8B}/\x{1140}\x{1175}\x{11BC}/g;
    s/\x{ED8C}/\x{1140}\x{1175}\x{11F0}/g;
    s/\x{ED8D}/\x{1140}\x{1175}\x{11F9}/g;
    s/\x{ED8E}/\x{1140}\x{1198}/g;
    s/\x{ED8F}/\x{1140}\x{119E}/g;
    s/\x{ED90}/\x{1140}\x{119E}\x{11A8}/g;
    s/\x{ED91}/\x{1140}\x{119E}\x{11AB}/g;
    s/\x{ED92}/\x{1140}\x{119E}\x{11AE}/g;
    s/\x{ED93}/\x{1140}\x{119E}\x{11AF}/g;
    s/\x{ED94}/\x{1140}\x{119E}\x{11B0}/g;
    s/\x{ED95}/\x{1140}\x{119E}\x{11B3}/g;
    s/\x{ED96}/\x{1140}\x{119E}\x{11B6}/g;
    s/\x{ED97}/\x{1140}\x{119E}\x{11D9}/g;
    s/\x{ED98}/\x{1140}\x{119E}\x{11B7}/g;
    s/\x{ED99}/\x{1140}\x{119E}\x{11DD}/g;
    s/\x{ED9A}/\x{1140}\x{119E}\x{11B8}/g;
    s/\x{ED9B}/\x{1140}\x{119E}\x{11E6}/g;
    s/\x{ED9C}/\x{1140}\x{119E}\x{11BA}/g;
    s/\x{ED9D}/\x{1140}\x{119E}\x{11EB}/g;
    s/\x{ED9E}/\x{1140}\x{119E}\x{11BC}/g;
    s/\x{ED9F}/\x{1140}\x{119E}\x{11BD}/g;
    s/\x{EDA0}/\x{1140}\x{11A1}/g;
    s/\x{EDA1}/\x{1140}\x{11A1}\x{11AB}/g;
    s/\x{EDA2}/\x{1140}\x{11A1}\x{11AF}/g;
    s/\x{EDA3}/\x{1140}\x{11A1}\x{11B8}/g;
    s/\x{EDA4}/\x{1140}\x{11A1}\x{11BA}/g;
    s/\x{EDA5}/\x{110B}\x{1161}\x{11C7}/g;
    s/\x{EDA6}/\x{110B}\x{1161}\x{11C9}/g;
    s/\x{EDA7}/\x{110B}\x{1161}\x{11CD}/g;
    s/\x{EDA8}/\x{110B}\x{1161}\x{11D0}/g;
    s/\x{EDA9}/\x{110B}\x{1161}\x{11D3}/g;
    s/\x{EDAA}/\x{110B}\x{1161}\x{11B2}\x{11C1}/g;
    s/\x{EDAB}/\x{110B}\x{1161}\x{11D4}/g;
    s/\x{EDAC}/\x{110B}\x{1161}\x{11D6}/g;
    s/\x{EDAD}/\x{110B}\x{1161}\x{11D9}/g;
    s/\x{EDAE}/\x{110B}\x{1161}\x{11DD}/g;
    s/\x{EDAF}/\x{110B}\x{1161}\x{11E1}/g;
    s/\x{EDB0}/\x{110B}\x{1161}\x{11E2}/g;
    s/\x{EDB1}/\x{110B}\x{1161}\x{11E3}/g;
    s/\x{EDB2}/\x{110B}\x{1161}\x{11B8}\x{11B5}/g;
    s/\x{EDB3}/\x{110B}\x{1161}\x{11E4}/g;
    s/\x{EDB4}/\x{110B}\x{1161}\x{11E5}/g;
    s/\x{EDB5}/\x{110B}\x{1161}\x{11E6}/g;
    s/\x{EDB6}/\x{110B}\x{1161}\x{11EA}/g;
    s/\x{EDB7}/\x{110B}\x{1161}\x{11EB}/g;
    s/\x{EDB8}/\x{110B}\x{1161}\x{11F0}/g;
    s/\x{EDB9}/\x{110B}\x{1161}\x{11F5}/g;
    s/\x{EDBA}/\x{110B}\x{1176}/g;
    s/\x{EDBB}/\x{110B}\x{1176}\x{11AF}/g;
    s/\x{EDBC}/\x{110B}\x{1162}\x{11EB}/g;
    s/\x{EDBD}/\x{110B}\x{1162}\x{11F0}/g;
    s/\x{EDBE}/\x{110B}\x{1163}\x{11D9}/g;
    s/\x{EDBF}/\x{110B}\x{1163}\x{11DC}/g;
    s/\x{EDC0}/\x{110B}\x{1163}\x{11DD}/g;
    s/\x{EDC1}/\x{110B}\x{1163}\x{11E2}/g;
    s/\x{EDC2}/\x{110B}\x{1163}\x{11E6}/g;
    s/\x{EDC3}/\x{110B}\x{1163}\x{11EA}/g;
    s/\x{EDC4}/\x{110B}\x{1163}\x{11BC}\x{11BA}/g;
    s/\x{EDC5}/\x{110B}\x{1163}\x{11F0}/g;
    s/\x{EDC6}/\x{110B}\x{1163}\x{11F5}/g;
    s/\x{EDC7}/\x{110B}\x{1163}\x{11F9}/g;
    s/\x{EDC8}/\x{110B}\x{1178}/g;
    s/\x{EDC9}/\x{110B}\x{1163}\x{116E}/g;
    s/\x{EDCA}/\x{110B}\x{1165}\x{11C3}/g;
    s/\x{EDCB}/\x{110B}\x{1165}\x{11C6}/g;
    s/\x{EDCC}/\x{110B}\x{1165}\x{11C7}/g;
    s/\x{EDCD}/\x{110B}\x{1165}\x{11CD}/g;
    s/\x{EDCE}/\x{110B}\x{1165}\x{11D0}/g;
    s/\x{EDCF}/\x{110B}\x{1165}\x{11D9}/g;
    s/\x{EDD0}/\x{110B}\x{1165}\x{11DD}/g;
    s/\x{EDD1}/\x{110B}\x{1165}\x{11DE}/g;
    s/\x{EDD2}/\x{110B}\x{1165}\x{11E4}/g;
    s/\x{EDD3}/\x{110B}\x{1165}\x{11E6}/g;
    s/\x{EDD4}/\x{110B}\x{1165}\x{11BA}\x{11BD}/g;
    s/\x{EDD5}/\x{110B}\x{1165}\x{11EB}/g;
    s/\x{EDD6}/\x{110B}\x{1165}\x{11F0}/g;
    s/\x{EDD7}/\x{110B}\x{1165}\x{11C1}\x{11BA}/g;
    s/\x{EDD8}/\x{110B}\x{117A}/g;
    s/\x{EDD9}/\x{110B}\x{117C}/g;
    s/\x{EDDA}/\x{110B}\x{117C}\x{11AB}/g;
    s/\x{EDDB}/\x{110B}\x{117C}\x{11AF}/g;
    s/\x{EDDC}/\x{110B}\x{117C}\x{11B7}/g;
    s/\x{EDDD}/\x{110B}\x{117C}\x{11B8}/g;
    s/\x{EDDE}/\x{110B}\x{117C}\x{11B9}/g;
    s/\x{EDDF}/\x{110B}\x{117C}\x{11BA}/g;
    s/\x{EDE0}/\x{110B}\x{117C}\x{11BC}/g;
    s/\x{EDE1}/\x{110B}\x{1166}\x{11C6}/g;
    s/\x{EDE2}/\x{110B}\x{1166}\x{11EB}/g;
    s/\x{EDE3}/\x{110B}\x{1166}\x{11F0}/g;
    s/\x{EDE4}/\x{110B}\x{1167}\x{11C4}/g;
    s/\x{EDE5}/\x{110B}\x{1167}\x{11A8}\x{11BF}/g;
    s/\x{EDE6}/\x{110B}\x{1167}\x{11C6}/g;
    s/\x{EDE7}/\x{110B}\x{1167}\x{11C7}/g;
    s/\x{EDE8}/\x{110B}\x{1167}\x{11C9}/g;
    s/\x{EDE9}/\x{110B}\x{1167}\x{11AE}\x{11C0}/g;
    s/\x{EDEA}/\x{110B}\x{1167}\x{11D0}/g;
    s/\x{EDEB}/\x{110B}\x{1167}\x{11D5}/g;
    s/\x{EDEC}/\x{110B}\x{1167}\x{11D9}/g;
    s/\x{EDED}/\x{110B}\x{1167}\x{11E2}/g;
    s/\x{EDEE}/\x{110B}\x{1167}\x{11E3}/g;
    s/\x{EDEF}/\x{110B}\x{1167}\x{11B8}\x{11B8}/g;
    s/\x{EDF0}/\x{110B}\x{1167}\x{11E6}/g;
    s/\x{EDF1}/\x{110B}\x{1167}\x{11E7}/g;
    s/\x{EDF2}/\x{110B}\x{1167}\x{11EA}/g;
    s/\x{EDF3}/\x{110B}\x{1167}\x{11BA}\x{11C0}/g;
    s/\x{EDF4}/\x{110B}\x{1167}\x{11BA}\x{11C2}/g;
    s/\x{EDF5}/\x{110B}\x{1167}\x{11EB}/g;
    s/\x{EDF6}/\x{110B}\x{1167}\x{11EC}/g;
    s/\x{EDF7}/\x{110B}\x{1167}\x{11ED}/g;
    s/\x{EDF8}/\x{110B}\x{1167}\x{11F0}/g;
    s/\x{EDF9}/\x{110B}\x{1167}\x{11F9}/g;
    s/\x{EDFA}/\x{110B}\x{117D}/g;
    s/\x{EDFB}/\x{110B}\x{117E}/g;
    s/\x{EDFC}/\x{110B}\x{1169}\x{11C7}/g;
    s/\x{EDFD}/\x{110B}\x{1169}\x{11AF}\x{11A9}/g;
    s/\x{EDFE}/\x{110B}\x{1169}\x{11B0}\x{11C2}/g;
    s/\x{EDFF}/\x{110B}\x{1169}\x{11CD}/g;
    s/\x{EE00}/\x{110B}\x{1169}\x{11D0}/g;
    s/\x{EE01}/\x{110B}\x{1169}\x{11D6}/g;
    s/\x{EE02}/\x{110B}\x{1169}\x{11D9}/g;
    s/\x{EE03}/\x{110B}\x{1169}\x{11DD}/g;
    s/\x{EE04}/\x{110B}\x{1169}\x{11E2}/g;
    s/\x{EE05}/\x{110B}\x{1169}\x{11BA}\x{11BE}/g;
    s/\x{EE06}/\x{110B}\x{1169}\x{11EB}/g;
    s/\x{EE07}/\x{110B}\x{1169}\x{11F0}/g;
    s/\x{EE08}/\x{110B}\x{1169}\x{11F1}/g;
    s/\x{EE09}/\x{110B}\x{116A}\x{11AE}\x{11B8}/g;
    s/\x{EE0A}/\x{110B}\x{116A}\x{11AE}\x{11C0}/g;
    s/\x{EE0B}/\x{110B}\x{116A}\x{11DD}/g;
    s/\x{EE0C}/\x{110B}\x{116A}\x{11EA}/g;
    s/\x{EE0D}/\x{110B}\x{116A}\x{11F0}/g;
    s/\x{EE0E}/\x{110B}\x{116A}\x{11F9}/g;
    s/\x{EE0F}/\x{110B}\x{117F}/g;
    s/\x{EE10}/\x{110B}\x{117F}\x{11AB}/g;
    s/\x{EE11}/\x{110B}\x{117F}\x{11AF}/g;
    s/\x{EE12}/\x{110B}\x{1182}/g;
    s/\x{EE13}/\x{110B}\x{1182}\x{11AF}/g;
    s/\x{EE14}/\x{110B}\x{116C}\x{11D0}/g;
    s/\x{EE15}/\x{110B}\x{116C}\x{11D9}/g;
    s/\x{EE16}/\x{110B}\x{116C}\x{11F0}/g;
    s/\x{EE17}/\x{110B}\x{116D}\x{11D9}/g;
    s/\x{EE18}/\x{110B}\x{116D}\x{11E2}/g;
    s/\x{EE19}/\x{110B}\x{116D}\x{11F0}/g;
    s/\x{EE1A}/\x{110B}\x{116D}\x{1161}/g;
    s/\x{EE1B}/\x{110B}\x{116D}\x{1162}/g;
    s/\x{EE1C}/\x{110B}\x{116D}\x{1162}\x{11AB}/g;
    s/\x{EE1D}/\x{110B}\x{116D}\x{1165}/g;
    s/\x{EE1E}/\x{110B}\x{116D}\x{1165}\x{11AB}/g;
    s/\x{EE1F}/\x{110B}\x{1187}/g;
    s/\x{EE20}/\x{110B}\x{1188}/g;
    s/\x{EE21}/\x{110B}\x{1188}\x{11AF}/g;
    s/\x{EE22}/\x{110B}\x{116E}\x{11C9}/g;
    s/\x{EE23}/\x{110B}\x{116E}\x{11AE}\x{11BE}/g;
    s/\x{EE24}/\x{110B}\x{116E}\x{11CD}/g;
    s/\x{EE25}/\x{110B}\x{116E}\x{11D0}/g;
    s/\x{EE26}/\x{110B}\x{116E}\x{11D9}/g;
    s/\x{EE27}/\x{110B}\x{116E}\x{11DD}/g;
    s/\x{EE28}/\x{110B}\x{116E}\x{11E0}/g;
    s/\x{EE29}/\x{110B}\x{116E}\x{11E1}/g;
    s/\x{EE2A}/\x{110B}\x{116E}\x{11E2}/g;
    s/\x{EE2B}/\x{110B}\x{116E}\x{11E4}/g;
    s/\x{EE2C}/\x{110B}\x{116E}\x{11EA}/g;
    s/\x{EE2D}/\x{110B}\x{116E}\x{11BA}\x{11E6}/g;
    s/\x{EE2E}/\x{110B}\x{116E}\x{11BA}\x{11BE}/g;
    s/\x{EE2F}/\x{110B}\x{116E}\x{11EB}/g;
    s/\x{EE30}/\x{110B}\x{116E}\x{11EB}\x{11B8}/g;
    s/\x{EE31}/\x{110B}\x{116E}\x{11EB}\x{11E6}/g;
    s/\x{EE32}/\x{110B}\x{116E}\x{11F0}/g;
    s/\x{EE33}/\x{110B}\x{116E}\x{11F9}/g;
    s/\x{EE34}/\x{110B}\x{1189}/g;
    s/\x{EE35}/\x{110B}\x{1189}\x{11AB}/g;
    s/\x{EE36}/\x{110B}\x{1189}\x{11BC}/g;
    s/\x{EE37}/\x{110B}\x{1189}\x{11F0}/g;
    s/\x{EE38}/\x{110B}\x{118A}/g;
    s/\x{EE39}/\x{110B}\x{116F}\x{11D8}/g;
    s/\x{EE3A}/\x{110B}\x{116F}\x{11D9}/g;
    s/\x{EE3B}/\x{110B}\x{116F}\x{11EB}/g;
    s/\x{EE3C}/\x{110B}\x{116F}\x{11F0}/g;
    s/\x{EE3D}/\x{110B}\x{116F}\x{11F1}/g;
    s/\x{EE3E}/\x{110B}\x{118B}/g;
    s/\x{EE3F}/\x{110B}\x{118B}\x{11AB}/g;
    s/\x{EE40}/\x{110B}\x{116E}\x{1167}/g;
    s/\x{EE41}/\x{110B}\x{118C}/g;
    s/\x{EE42}/\x{110B}\x{1171}\x{11D9}/g;
    s/\x{EE43}/\x{110B}\x{1171}\x{11F0}/g;
    s/\x{EE44}/\x{110B}\x{1172}\x{11D9}/g;
    s/\x{EE45}/\x{110B}\x{1172}\x{11E2}/g;
    s/\x{EE46}/\x{110B}\x{1172}\x{11F0}/g;
    s/\x{EE47}/\x{110B}\x{1172}\x{11F9}/g;
    s/\x{EE48}/\x{110B}\x{118E}/g;
    s/\x{EE49}/\x{110B}\x{118E}\x{11AB}/g;
    s/\x{EE4A}/\x{110B}\x{118E}\x{11B7}/g;
    s/\x{EE4B}/\x{110B}\x{118F}/g;
    s/\x{EE4C}/\x{110B}\x{1190}/g;
    s/\x{EE4D}/\x{110B}\x{1190}\x{11AB}/g;
    s/\x{EE4E}/\x{110B}\x{1191}/g;
    s/\x{EE4F}/\x{110B}\x{1191}\x{11A8}/g;
    s/\x{EE50}/\x{110B}\x{1191}\x{11AB}/g;
    s/\x{EE51}/\x{110B}\x{1191}\x{11AF}/g;
    s/\x{EE52}/\x{110B}\x{1191}\x{11D9}/g;
    s/\x{EE53}/\x{110B}\x{1191}\x{11F0}/g;
    s/\x{EE54}/\x{110B}\x{1191}\x{11F9}/g;
    s/\x{EE55}/\x{110B}\x{1192}/g;
    s/\x{EE56}/\x{110B}\x{1192}\x{11BC}/g;
    s/\x{EE57}/\x{110B}\x{1192}\x{11F0}/g;
    s/\x{EE58}/\x{110B}\x{1193}/g;
    s/\x{EE59}/\x{110B}\x{1194}/g;
    s/\x{EE5A}/\x{110B}\x{1194}\x{11AB}/g;
    s/\x{EE5B}/\x{110B}\x{1194}\x{11BC}/g;
    s/\x{EE5C}/\x{110B}\x{1194}\x{11F0}/g;
    s/\x{EE5D}/\x{110B}\x{1173}\x{11AB}\x{11AB}/g;
    s/\x{EE5E}/\x{110B}\x{1173}\x{11C7}/g;
    s/\x{EE5F}/\x{110B}\x{1173}\x{11AE}\x{11AE}/g;
    s/\x{EE60}/\x{110B}\x{1173}\x{11D0}/g;
    s/\x{EE61}/\x{110B}\x{1173}\x{11B2}\x{11C1}/g;
    s/\x{EE62}/\x{110B}\x{1173}\x{11D4}/g;
    s/\x{EE63}/\x{110B}\x{1173}\x{11D9}/g;
    s/\x{EE64}/\x{110B}\x{1173}\x{11DD}/g;
    s/\x{EE65}/\x{110B}\x{1173}\x{11E2}/g;
    s/\x{EE66}/\x{110B}\x{1173}\x{11E3}/g;
    s/\x{EE67}/\x{110B}\x{1173}\x{11B8}\x{11B8}/g;
    s/\x{EE68}/\x{110B}\x{1173}\x{11E4}/g;
    s/\x{EE69}/\x{110B}\x{1173}\x{11BA}\x{11BE}/g;
    s/\x{EE6A}/\x{110B}\x{1173}\x{11EB}/g;
    s/\x{EE6B}/\x{110B}\x{1173}\x{11F0}/g;
    s/\x{EE6C}/\x{110B}\x{1173}\x{11BD}\x{11BD}/g;
    s/\x{EE6D}/\x{110B}\x{1173}\x{11F9}/g;
    s/\x{EE6E}/\x{110B}\x{1195}/g;
    s/\x{EE6F}/\x{110B}\x{1196}/g;
    s/\x{EE70}/\x{110B}\x{1196}\x{11AB}/g;
    s/\x{EE71}/\x{110B}\x{1196}\x{11BC}/g;
    s/\x{EE72}/\x{110B}\x{1196}\x{11F0}/g;
    s/\x{EE73}/\x{110B}\x{1196}\x{11F9}/g;
    s/\x{EE74}/\x{110B}\x{1174}\x{11D9}/g;
    s/\x{EE75}/\x{110B}\x{1174}\x{11F9}/g;
    s/\x{EE76}/\x{110B}\x{1175}\x{11C7}/g;
    s/\x{EE77}/\x{110B}\x{1175}\x{11CD}/g;
    s/\x{EE78}/\x{110B}\x{1175}\x{11D0}/g;
    s/\x{EE79}/\x{110B}\x{1175}\x{11D9}/g;
    s/\x{EE7A}/\x{110B}\x{1175}\x{11E2}/g;
    s/\x{EE7B}/\x{110B}\x{1175}\x{11E5}/g;
    s/\x{EE7C}/\x{110B}\x{1175}\x{11E6}/g;
    s/\x{EE7D}/\x{110B}\x{1175}\x{11E7}/g;
    s/\x{EE7E}/\x{110B}\x{1175}\x{11E9}/g;
    s/\x{EE7F}/\x{110B}\x{1175}\x{11EA}/g;
    s/\x{EE80}/\x{110B}\x{1175}\x{11F0}/g;
    s/\x{EE81}/\x{110B}\x{1175}\x{11F9}/g;
    s/\x{EE82}/\x{110B}\x{1198}/g;
    s/\x{EE83}/\x{110B}\x{119A}/g;
    s/\x{EE84}/\x{110B}\x{1175}\x{1183}/g;
    s/\x{EE85}/\x{110B}\x{119B}/g;
    s/\x{EE86}/\x{110B}\x{119C}/g;
    s/\x{EE87}/\x{110B}\x{119C}\x{11AB}/g;
    s/\x{EE88}/\x{110B}\x{119E}/g;
    s/\x{EE89}/\x{110B}\x{119E}\x{11A8}/g;
    s/\x{EE8A}/\x{110B}\x{119E}\x{11AB}/g;
    s/\x{EE8B}/\x{110B}\x{119E}\x{11AE}/g;
    s/\x{EE8C}/\x{110B}\x{119E}\x{11AF}/g;
    s/\x{EE8D}/\x{110B}\x{119E}\x{11B0}/g;
    s/\x{EE8E}/\x{110B}\x{119E}\x{11B1}/g;
    s/\x{EE8F}/\x{110B}\x{119E}\x{11B2}/g;
    s/\x{EE90}/\x{110B}\x{119E}\x{11B3}/g;
    s/\x{EE91}/\x{110B}\x{119E}\x{11B6}/g;
    s/\x{EE92}/\x{110B}\x{119E}\x{11D9}/g;
    s/\x{EE93}/\x{110B}\x{119E}\x{11B7}/g;
    s/\x{EE94}/\x{110B}\x{119E}\x{11DD}/g;
    s/\x{EE95}/\x{110B}\x{119E}\x{11B8}/g;
    s/\x{EE96}/\x{110B}\x{119E}\x{11B9}/g;
    s/\x{EE97}/\x{110B}\x{119E}\x{11BA}/g;
    s/\x{EE98}/\x{110B}\x{119E}\x{11EB}/g;
    s/\x{EE99}/\x{110B}\x{119E}\x{11BC}/g;
    s/\x{EE9A}/\x{110B}\x{119E}\x{11F0}/g;
    s/\x{EE9B}/\x{110B}\x{119E}\x{11BD}/g;
    s/\x{EE9C}/\x{110B}\x{119E}\x{11BE}/g;
    s/\x{EE9D}/\x{110B}\x{119E}\x{11C0}/g;
    s/\x{EE9E}/\x{110B}\x{119F}/g;
    s/\x{EE9F}/\x{110B}\x{119F}\x{11AB}/g;
    s/\x{EEA0}/\x{110B}\x{119F}\x{11AF}/g;
    s/\x{EEA1}/\x{110B}\x{119F}\x{11B2}/g;
    s/\x{EEA2}/\x{110B}\x{119F}\x{11B7}/g;
    s/\x{EEA3}/\x{110B}\x{119F}\x{11BC}/g;
    s/\x{EEA4}/\x{110B}\x{11A1}/g;
    s/\x{EEA5}/\x{110B}\x{11A1}\x{11A8}/g;
    s/\x{EEA6}/\x{110B}\x{11A1}\x{11AB}/g;
    s/\x{EEA7}/\x{110B}\x{11A1}\x{11AF}/g;
    s/\x{EEA8}/\x{110B}\x{11A1}\x{11B7}/g;
    s/\x{EEA9}/\x{110B}\x{11A1}\x{11B8}/g;
    s/\x{EEAA}/\x{110B}\x{11A1}\x{11BA}/g;
    s/\x{EEAB}/\x{110B}\x{11A1}\x{11EB}/g;
    s/\x{EEAC}/\x{110B}\x{11A1}\x{11BC}/g;
    s/\x{EEAD}/\x{110B}\x{11A1}\x{11F0}/g;
    s/\x{EEAE}/\x{110B}\x{11A2}/g;
    s/\x{EEAF}/\x{110B}\x{11A2}\x{11AF}/g;
    s/\x{EEB0}/\x{1141}\x{1161}/g;
    s/\x{EEB1}/\x{1141}\x{1161}\x{11A8}/g;
    s/\x{EEB2}/\x{1141}\x{1162}/g;
    s/\x{EEB3}/\x{1141}\x{1163}/g;
    s/\x{EEB4}/\x{1141}\x{1163}\x{11AB}/g;
    s/\x{EEB5}/\x{1141}\x{1165}/g;
    s/\x{EEB6}/\x{1141}\x{1166}/g;
    s/\x{EEB7}/\x{1141}\x{1167}/g;
    s/\x{EEB8}/\x{1141}\x{1168}/g;
    s/\x{EEB9}/\x{1141}\x{1168}\x{11AB}/g;
    s/\x{EEBA}/\x{1141}\x{1169}/g;
    s/\x{EEBB}/\x{1141}\x{1169}\x{11A8}/g;
    s/\x{EEBC}/\x{1141}\x{1169}\x{11AB}/g;
    s/\x{EEBD}/\x{1141}\x{116A}/g;
    s/\x{EEBE}/\x{1141}\x{116A}\x{11AB}/g;
    s/\x{EEBF}/\x{1141}\x{116B}/g;
    s/\x{EEC0}/\x{1141}\x{116C}/g;
    s/\x{EEC1}/\x{1141}\x{116D}/g;
    s/\x{EEC2}/\x{1141}\x{116E}/g;
    s/\x{EEC3}/\x{1141}\x{116E}\x{11AB}/g;
    s/\x{EEC4}/\x{1141}\x{116F}/g;
    s/\x{EEC5}/\x{1141}\x{1170}/g;
    s/\x{EEC6}/\x{1141}\x{116E}\x{1167}/g;
    s/\x{EEC7}/\x{1141}\x{118C}/g;
    s/\x{EEC8}/\x{1141}\x{1171}/g;
    s/\x{EEC9}/\x{1141}\x{1172}/g;
    s/\x{EECA}/\x{1141}\x{1194}/g;
    s/\x{EECB}/\x{1141}\x{1173}/g;
    s/\x{EECC}/\x{1141}\x{1174}/g;
    s/\x{EECD}/\x{1141}\x{1175}/g;
    s/\x{EECE}/\x{1141}\x{1175}\x{11AB}/g;
    s/\x{EECF}/\x{1141}\x{1198}/g;
    s/\x{EED0}/\x{1141}\x{119E}/g;
    s/\x{EED1}/\x{1141}\x{11A1}/g;
    s/\x{EED2}/\x{1142}\x{1161}/g;
    s/\x{EED3}/\x{1142}\x{1161}\x{11AB}/g;
    s/\x{EED4}/\x{1142}\x{1165}/g;
    s/\x{EED5}/\x{1142}\x{1165}\x{11AF}/g;
    s/\x{EED6}/\x{1142}\x{1166}/g;
    s/\x{EED7}/\x{1142}\x{1166}\x{11AB}/g;
    s/\x{EED8}/\x{1142}\x{1169}/g;
    s/\x{EED9}/\x{1142}\x{1169}\x{11AB}/g;
    s/\x{EEDA}/\x{1142}\x{1169}\x{11AF}/g;
    s/\x{EEDB}/\x{1142}\x{1173}/g;
    s/\x{EEDC}/\x{1142}\x{1173}\x{11B7}/g;
    s/\x{EEDD}/\x{1142}\x{1174}/g;
    s/\x{EEDE}/\x{1142}\x{1175}/g;
    s/\x{EEDF}/\x{1142}\x{1175}\x{11BC}/g;
    s/\x{EEE0}/\x{1142}\x{119E}/g;
    s/\x{EEE1}/\x{1142}\x{119E}\x{11B7}/g;
    s/\x{EEE2}/\x{110B}\x{1105}\x{1165}/g;
    s/\x{EEE3}/\x{110B}\x{1105}\x{1169}/g;
    s/\x{EEE4}/\x{110B}\x{1105}\x{1175}/g;
    s/\x{EEE5}/\x{1143}\x{116E}/g;
    s/\x{EEE6}/\x{1143}\x{119E}/g;
    s/\x{EEE7}/\x{1143}\x{11A1}/g;
    s/\x{EEE8}/\x{1144}\x{1161}/g;
    s/\x{EEE9}/\x{1144}\x{1161}\x{11AB}/g;
    s/\x{EEEA}/\x{1144}\x{1161}\x{11AF}/g;
    s/\x{EEEB}/\x{1144}\x{1161}\x{11B7}/g;
    s/\x{EEEC}/\x{1144}\x{1162}/g;
    s/\x{EEED}/\x{1144}\x{1162}\x{11AF}/g;
    s/\x{EEEE}/\x{1144}\x{1162}\x{11B7}/g;
    s/\x{EEEF}/\x{1144}\x{1165}/g;
    s/\x{EEF0}/\x{1144}\x{1165}\x{11AF}/g;
    s/\x{EEF1}/\x{1144}\x{1165}\x{11C0}/g;
    s/\x{EEF2}/\x{1144}\x{1166}/g;
    s/\x{EEF3}/\x{1144}\x{1166}\x{11AB}/g;
    s/\x{EEF4}/\x{1144}\x{1166}\x{11AF}/g;
    s/\x{EEF5}/\x{1144}\x{1166}\x{11C0}/g;
    s/\x{EEF6}/\x{1144}\x{1167}/g;
    s/\x{EEF7}/\x{1144}\x{1168}/g;
    s/\x{EEF8}/\x{1144}\x{1168}\x{11AB}/g;
    s/\x{EEF9}/\x{1144}\x{1169}/g;
    s/\x{EEFA}/\x{1144}\x{1169}\x{11AB}/g;
    s/\x{EEFB}/\x{1144}\x{1169}\x{11AF}/g;
    s/\x{EEFC}/\x{1144}\x{1169}\x{11C0}/g;
    s/\x{EEFD}/\x{1144}\x{116A}/g;
    s/\x{EEFE}/\x{1144}\x{116A}\x{11B7}/g;
    s/\x{EEFF}/\x{1144}\x{116D}/g;
    s/\x{EF00}/\x{1144}\x{116E}/g;
    s/\x{EF01}/\x{1144}\x{116E}\x{11AB}/g;
    s/\x{EF02}/\x{1144}\x{116E}\x{11AF}/g;
    s/\x{EF03}/\x{1144}\x{116F}/g;
    s/\x{EF04}/\x{1144}\x{1170}/g;
    s/\x{EF05}/\x{1144}\x{1170}\x{11AF}/g;
    s/\x{EF06}/\x{1144}\x{1171}/g;
    s/\x{EF07}/\x{1144}\x{1171}\x{11BC}/g;
    s/\x{EF08}/\x{1144}\x{1172}/g;
    s/\x{EF09}/\x{1144}\x{1173}/g;
    s/\x{EF0A}/\x{1144}\x{1173}\x{11AB}/g;
    s/\x{EF0B}/\x{1144}\x{1173}\x{11AF}/g;
    s/\x{EF0C}/\x{1144}\x{1174}/g;
    s/\x{EF0D}/\x{1144}\x{1175}/g;
    s/\x{EF0E}/\x{1144}\x{1175}\x{11A8}/g;
    s/\x{EF0F}/\x{1144}\x{1175}\x{11AB}/g;
    s/\x{EF10}/\x{1144}\x{1175}\x{11AF}/g;
    s/\x{EF11}/\x{1144}\x{1175}\x{11BC}/g;
    s/\x{EF12}/\x{1144}\x{1175}\x{11BF}/g;
    s/\x{EF13}/\x{1144}\x{119E}/g;
    s/\x{EF14}/\x{1144}\x{11A1}/g;
    s/\x{EF15}/\x{1145}\x{1161}/g;
    s/\x{EF16}/\x{1145}\x{1161}\x{11AB}/g;
    s/\x{EF17}/\x{1145}\x{1161}\x{11BC}/g;
    s/\x{EF18}/\x{1145}\x{1176}/g;
    s/\x{EF19}/\x{1145}\x{1162}/g;
    s/\x{EF1A}/\x{1145}\x{1165}/g;
    s/\x{EF1B}/\x{1145}\x{1165}\x{11AB}/g;
    s/\x{EF1C}/\x{1145}\x{1165}\x{11AF}/g;
    s/\x{EF1D}/\x{1145}\x{1165}\x{11B7}/g;
    s/\x{EF1E}/\x{1145}\x{1165}\x{11BC}/g;
    s/\x{EF1F}/\x{1145}\x{117B}/g;
    s/\x{EF20}/\x{1145}\x{1166}/g;
    s/\x{EF21}/\x{1145}\x{1167}/g;
    s/\x{EF22}/\x{1145}\x{1168}/g;
    s/\x{EF23}/\x{1145}\x{1168}\x{11AB}/g;
    s/\x{EF24}/\x{1145}\x{1169}/g;
    s/\x{EF25}/\x{1145}\x{1169}\x{11AB}/g;
    s/\x{EF26}/\x{1145}\x{116E}/g;
    s/\x{EF27}/\x{1145}\x{116E}\x{11AB}/g;
    s/\x{EF28}/\x{1145}\x{1189}/g;
    s/\x{EF29}/\x{1145}\x{1189}\x{11AB}/g;
    s/\x{EF2A}/\x{1145}\x{1189}\x{11BC}/g;
    s/\x{EF2B}/\x{1145}\x{118A}/g;
    s/\x{EF2C}/\x{1145}\x{116F}/g;
    s/\x{EF2D}/\x{1145}\x{1170}/g;
    s/\x{EF2E}/\x{1145}\x{1172}/g;
    s/\x{EF2F}/\x{1145}\x{118E}/g;
    s/\x{EF30}/\x{1145}\x{118E}\x{11AB}/g;
    s/\x{EF31}/\x{1145}\x{1191}/g;
    s/\x{EF32}/\x{1145}\x{1192}/g;
    s/\x{EF33}/\x{1145}\x{1194}/g;
    s/\x{EF34}/\x{1145}\x{1173}/g;
    s/\x{EF35}/\x{1145}\x{1173}\x{11AB}/g;
    s/\x{EF36}/\x{1145}\x{1174}/g;
    s/\x{EF37}/\x{1145}\x{1174}\x{11AB}/g;
    s/\x{EF38}/\x{1145}\x{1175}/g;
    s/\x{EF39}/\x{1145}\x{1175}\x{11AF}/g;
    s/\x{EF3A}/\x{1145}\x{119E}/g;
    s/\x{EF3B}/\x{1145}\x{119E}\x{11AF}/g;
    s/\x{EF3C}/\x{1146}\x{1161}/g;
    s/\x{EF3D}/\x{1146}\x{1162}/g;
    s/\x{EF3E}/\x{1146}\x{1163}/g;
    s/\x{EF3F}/\x{1146}\x{1165}/g;
    s/\x{EF40}/\x{1146}\x{1166}/g;
    s/\x{EF41}/\x{1146}\x{1167}/g;
    s/\x{EF42}/\x{1146}\x{1168}/g;
    s/\x{EF43}/\x{1146}\x{1169}/g;
    s/\x{EF44}/\x{1146}\x{116A}/g;
    s/\x{EF45}/\x{1146}\x{116B}/g;
    s/\x{EF46}/\x{1146}\x{116C}/g;
    s/\x{EF47}/\x{1146}\x{116D}/g;
    s/\x{EF48}/\x{1146}\x{116E}/g;
    s/\x{EF49}/\x{1146}\x{116F}/g;
    s/\x{EF4A}/\x{1146}\x{1170}/g;
    s/\x{EF4B}/\x{1146}\x{116E}\x{1167}/g;
    s/\x{EF4C}/\x{1146}\x{118C}/g;
    s/\x{EF4D}/\x{1146}\x{1171}/g;
    s/\x{EF4E}/\x{1146}\x{1172}/g;
    s/\x{EF4F}/\x{1146}\x{1194}/g;
    s/\x{EF50}/\x{1146}\x{1173}/g;
    s/\x{EF51}/\x{1146}\x{1174}/g;
    s/\x{EF52}/\x{1146}\x{1175}/g;
    s/\x{EF53}/\x{1146}\x{1198}/g;
    s/\x{EF54}/\x{1146}\x{119E}/g;
    s/\x{EF55}/\x{1146}\x{11A1}/g;
    s/\x{EF56}/\x{1147}\x{1163}/g;
    s/\x{EF57}/\x{1147}\x{1163}\x{11AB}/g;
    s/\x{EF58}/\x{1147}\x{1167}/g;
    s/\x{EF59}/\x{1147}\x{1167}\x{11AB}/g;
    s/\x{EF5A}/\x{1147}\x{1167}\x{11BA}/g;
    s/\x{EF5B}/\x{1147}\x{1168}/g;
    s/\x{EF5C}/\x{1147}\x{1168}\x{11BA}/g;
    s/\x{EF5D}/\x{1147}\x{116D}/g;
    s/\x{EF5E}/\x{1147}\x{116D}\x{11AB}/g;
    s/\x{EF5F}/\x{1147}\x{116D}\x{11B7}/g;
    s/\x{EF60}/\x{1147}\x{116D}\x{11E2}/g;
    s/\x{EF61}/\x{1147}\x{116D}\x{11BA}/g;
    s/\x{EF62}/\x{1147}\x{116E}/g;
    s/\x{EF63}/\x{1147}\x{116F}/g;
    s/\x{EF64}/\x{1147}\x{1172}/g;
    s/\x{EF65}/\x{1147}\x{1172}\x{11AB}/g;
    s/\x{EF66}/\x{1147}\x{1172}\x{11B7}/g;
    s/\x{EF67}/\x{1147}\x{118E}/g;
    s/\x{EF68}/\x{1147}\x{118E}\x{11AB}/g;
    s/\x{EF69}/\x{1147}\x{1175}/g;
    s/\x{EF6A}/\x{1147}\x{1175}\x{11AB}/g;
    s/\x{EF6B}/\x{1148}\x{1161}/g;
    s/\x{EF6C}/\x{1148}\x{1161}\x{11AB}/g;
    s/\x{EF6D}/\x{1148}\x{1161}\x{11BC}/g;
    s/\x{EF6E}/\x{1148}\x{1176}/g;
    s/\x{EF6F}/\x{1148}\x{1162}/g;
    s/\x{EF70}/\x{1148}\x{1165}/g;
    s/\x{EF71}/\x{1148}\x{1165}\x{11AB}/g;
    s/\x{EF72}/\x{1148}\x{1165}\x{11B7}/g;
    s/\x{EF73}/\x{1148}\x{1165}\x{11BC}/g;
    s/\x{EF74}/\x{1148}\x{117B}/g;
    s/\x{EF75}/\x{1148}\x{1169}/g;
    s/\x{EF76}/\x{1148}\x{116D}/g;
    s/\x{EF77}/\x{1148}\x{116E}/g;
    s/\x{EF78}/\x{1148}\x{116E}\x{11AB}/g;
    s/\x{EF79}/\x{1148}\x{116E}\x{11BC}/g;
    s/\x{EF7A}/\x{1148}\x{1189}/g;
    s/\x{EF7B}/\x{1148}\x{1189}\x{11AB}/g;
    s/\x{EF7C}/\x{1148}\x{116F}/g;
    s/\x{EF7D}/\x{1148}\x{1170}/g;
    s/\x{EF7E}/\x{1148}\x{1171}/g;
    s/\x{EF7F}/\x{1148}\x{1172}/g;
    s/\x{EF80}/\x{1148}\x{1194}/g;
    s/\x{EF81}/\x{1148}\x{1173}/g;
    s/\x{EF82}/\x{1148}\x{1175}/g;
    s/\x{EF83}/\x{1149}\x{1161}/g;
    s/\x{EF84}/\x{1149}\x{1161}\x{11AB}/g;
    s/\x{EF85}/\x{1149}\x{1161}\x{11BC}/g;
    s/\x{EF86}/\x{1149}\x{1162}/g;
    s/\x{EF87}/\x{1149}\x{1165}/g;
    s/\x{EF88}/\x{1149}\x{1165}\x{11AB}/g;
    s/\x{EF89}/\x{1149}\x{1165}\x{11BC}/g;
    s/\x{EF8A}/\x{1149}\x{116E}/g;
    s/\x{EF8B}/\x{1149}\x{116E}\x{11AB}/g;
    s/\x{EF8C}/\x{1149}\x{116E}\x{11BC}/g;
    s/\x{EF8D}/\x{1149}\x{1189}/g;
    s/\x{EF8E}/\x{1149}\x{1189}\x{11AB}/g;
    s/\x{EF8F}/\x{1149}\x{116F}/g;
    s/\x{EF90}/\x{1149}\x{1170}/g;
    s/\x{EF91}/\x{1149}\x{1172}/g;
    s/\x{EF92}/\x{1149}\x{118E}/g;
    s/\x{EF93}/\x{1149}\x{118E}\x{11AB}/g;
    s/\x{EF94}/\x{1149}\x{1194}/g;
    s/\x{EF95}/\x{1149}\x{1173}/g;
    s/\x{EF96}/\x{114A}\x{1173}/g;
    s/\x{EF97}/\x{114B}\x{1161}/g;
    s/\x{EF98}/\x{114B}\x{1161}\x{11A8}/g;
    s/\x{EF99}/\x{114B}\x{1161}\x{11AB}/g;
    s/\x{EF9A}/\x{114B}\x{1161}\x{11AF}/g;
    s/\x{EF9B}/\x{114B}\x{1161}\x{11D8}/g;
    s/\x{EF9C}/\x{114B}\x{1161}\x{11BF}/g;
    s/\x{EF9D}/\x{114B}\x{1162}/g;
    s/\x{EF9E}/\x{114B}\x{1162}\x{11AB}/g;
    s/\x{EF9F}/\x{114B}\x{1162}\x{11BF}/g;
    s/\x{EFA0}/\x{114B}\x{1165}/g;
    s/\x{EFA1}/\x{114B}\x{1165}\x{11AB}/g;
    s/\x{EFA2}/\x{114B}\x{1165}\x{11AF}/g;
    s/\x{EFA3}/\x{114B}\x{1165}\x{11B7}/g;
    s/\x{EFA4}/\x{114B}\x{1165}\x{11BC}/g;
    s/\x{EFA5}/\x{114B}\x{1166}/g;
    s/\x{EFA6}/\x{114B}\x{1166}\x{11A8}/g;
    s/\x{EFA7}/\x{114B}\x{1166}\x{11AB}/g;
    s/\x{EFA8}/\x{114B}\x{1166}\x{11AF}/g;
    s/\x{EFA9}/\x{114B}\x{1166}\x{11BF}/g;
    s/\x{EFAA}/\x{114B}\x{1169}/g;
    s/\x{EFAB}/\x{114B}\x{1169}\x{11AB}/g;
    s/\x{EFAC}/\x{114B}\x{1169}\x{11AF}/g;
    s/\x{EFAD}/\x{114B}\x{1169}\x{11B1}/g;
    s/\x{EFAE}/\x{114B}\x{1169}\x{11D8}/g;
    s/\x{EFAF}/\x{114B}\x{1169}\x{11B7}/g;
    s/\x{EFB0}/\x{114B}\x{1169}\x{11BF}/g;
    s/\x{EFB1}/\x{114B}\x{116A}/g;
    s/\x{EFB2}/\x{114B}\x{116B}/g;
    s/\x{EFB3}/\x{114B}\x{116B}\x{11AB}/g;
    s/\x{EFB4}/\x{114B}\x{116E}/g;
    s/\x{EFB5}/\x{114B}\x{116E}\x{11AB}/g;
    s/\x{EFB6}/\x{114B}\x{116E}\x{11AF}/g;
    s/\x{EFB7}/\x{114B}\x{116E}\x{11BF}/g;
    s/\x{EFB8}/\x{114B}\x{116E}\x{11C0}/g;
    s/\x{EFB9}/\x{114B}\x{1189}/g;
    s/\x{EFBA}/\x{114B}\x{1189}\x{11AB}/g;
    s/\x{EFBB}/\x{114B}\x{116F}/g;
    s/\x{EFBC}/\x{114B}\x{116F}\x{11AB}/g;
    s/\x{EFBD}/\x{114B}\x{1170}/g;
    s/\x{EFBE}/\x{114B}\x{1170}\x{11AB}/g;
    s/\x{EFBF}/\x{114B}\x{1171}/g;
    s/\x{EFC0}/\x{114B}\x{1171}\x{11A8}/g;
    s/\x{EFC1}/\x{114B}\x{1171}\x{11AB}/g;
    s/\x{EFC2}/\x{114B}\x{1171}\x{11AF}/g;
    s/\x{EFC3}/\x{114B}\x{1172}/g;
    s/\x{EFC4}/\x{114B}\x{1172}\x{11B7}/g;
    s/\x{EFC5}/\x{114B}\x{1173}/g;
    s/\x{EFC6}/\x{114B}\x{1173}\x{11AF}/g;
    s/\x{EFC7}/\x{114B}\x{1175}/g;
    s/\x{EFC8}/\x{114B}\x{1175}\x{11AB}/g;
    s/\x{EFC9}/\x{114B}\x{1175}\x{11AF}/g;
    s/\x{EFCA}/\x{114B}\x{1175}\x{11BC}/g;
    s/\x{EFCB}/\x{114B}\x{1175}\x{11BF}/g;
    s/\x{EFCC}/\x{114B}\x{1175}\x{11C0}/g;
    s/\x{EFCD}/\x{114B}\x{1175}\x{11C1}/g;
    s/\x{EFCE}/\x{114B}\x{119E}/g;
    s/\x{EFCF}/\x{114B}\x{11A1}/g;
    s/\x{EFD0}/\x{110B}\x{1112}\x{1169}/g;
    s/\x{EFD1}/\x{110B}\x{1112}\x{116E}/g;
    s/\x{EFD2}/\x{114C}\x{1161}/g;
    s/\x{EFD3}/\x{114C}\x{1161}\x{11A8}/g;
    s/\x{EFD4}/\x{114C}\x{1161}\x{11AB}/g;
    s/\x{EFD5}/\x{114C}\x{1161}\x{11AF}/g;
    s/\x{EFD6}/\x{114C}\x{1161}\x{11D9}/g;
    s/\x{EFD7}/\x{114C}\x{1161}\x{11B7}/g;
    s/\x{EFD8}/\x{114C}\x{1161}\x{11B8}/g;
    s/\x{EFD9}/\x{114C}\x{1161}\x{11EB}/g;
    s/\x{EFDA}/\x{114C}\x{1161}\x{11BC}/g;
    s/\x{EFDB}/\x{114C}\x{1161}\x{11F0}/g;
    s/\x{EFDC}/\x{114C}\x{1161}\x{11F9}/g;
    s/\x{EFDD}/\x{114C}\x{1162}/g;
    s/\x{EFDE}/\x{114C}\x{1162}\x{11BA}/g;
    s/\x{EFDF}/\x{114C}\x{1162}\x{11BC}/g;
    s/\x{EFE0}/\x{114C}\x{1163}/g;
    s/\x{EFE1}/\x{114C}\x{1163}\x{11A8}/g;
    s/\x{EFE2}/\x{114C}\x{1163}\x{11AB}/g;
    s/\x{EFE3}/\x{114C}\x{1163}\x{11BC}/g;
    s/\x{EFE4}/\x{114C}\x{1163}\x{11F0}/g;
    s/\x{EFE5}/\x{114C}\x{1164}/g;
    s/\x{EFE6}/\x{114C}\x{1164}\x{11BA}/g;
    s/\x{EFE7}/\x{114C}\x{1165}/g;
    s/\x{EFE8}/\x{114C}\x{1165}\x{11A8}/g;
    s/\x{EFE9}/\x{114C}\x{1165}\x{11AB}/g;
    s/\x{EFEA}/\x{114C}\x{1165}\x{11AF}/g;
    s/\x{EFEB}/\x{114C}\x{1165}\x{11D9}/g;
    s/\x{EFEC}/\x{114C}\x{1165}\x{11B7}/g;
    s/\x{EFED}/\x{114C}\x{1165}\x{11B8}/g;
    s/\x{EFEE}/\x{114C}\x{1165}\x{11BA}/g;
    s/\x{EFEF}/\x{114C}\x{1165}\x{11EB}/g;
    s/\x{EFF0}/\x{114C}\x{1165}\x{11BC}/g;
    s/\x{EFF1}/\x{114C}\x{1165}\x{11F0}/g;
    s/\x{EFF2}/\x{114C}\x{1166}/g;
    s/\x{EFF3}/\x{114C}\x{1166}\x{11AB}/g;
    s/\x{EFF4}/\x{114C}\x{1166}\x{11BA}/g;
    s/\x{EFF5}/\x{114C}\x{1166}\x{11EB}/g;
    s/\x{EFF6}/\x{114C}\x{1166}\x{11BC}/g;
    s/\x{EFF7}/\x{114C}\x{1166}\x{11F0}/g;
    s/\x{EFF8}/\x{114C}\x{1167}/g;
    s/\x{EFF9}/\x{114C}\x{1167}\x{11A8}/g;
    s/\x{EFFA}/\x{114C}\x{1167}\x{11AB}/g;
    s/\x{EFFB}/\x{114C}\x{1167}\x{11AF}/g;
    s/\x{EFFC}/\x{114C}\x{1167}\x{11B7}/g;
    s/\x{EFFD}/\x{114C}\x{1167}\x{11E2}/g;
    s/\x{EFFE}/\x{114C}\x{1167}\x{11B8}/g;
    s/\x{EFFF}/\x{114C}\x{1167}\x{11BA}/g;
    s/\x{F100}/\x{114C}\x{1167}\x{11BC}/g;
    s/\x{F101}/\x{114C}\x{1167}\x{11F0}/g;
    s/\x{F102}/\x{114C}\x{1167}\x{11C1}/g;
    s/\x{F103}/\x{114C}\x{1167}\x{11F4}/g;
    s/\x{F104}/\x{114C}\x{1168}/g;
    s/\x{F105}/\x{114C}\x{1168}\x{11BA}/g;
    s/\x{F106}/\x{114C}\x{1168}\x{11BC}/g;
    s/\x{F107}/\x{114C}\x{1169}/g;
    s/\x{F108}/\x{114C}\x{1169}\x{11A8}/g;
    s/\x{F109}/\x{114C}\x{1169}\x{11AB}/g;
    s/\x{F10A}/\x{114C}\x{1169}\x{11AF}/g;
    s/\x{F10B}/\x{114C}\x{1169}\x{11B3}/g;
    s/\x{F10C}/\x{114C}\x{1169}\x{11D9}/g;
    s/\x{F10D}/\x{114C}\x{1169}\x{11B7}/g;
    s/\x{F10E}/\x{114C}\x{1169}\x{11E2}/g;
    s/\x{F10F}/\x{114C}\x{1169}\x{11BA}/g;
    s/\x{F110}/\x{114C}\x{1169}\x{11BC}/g;
    s/\x{F111}/\x{114C}\x{1169}\x{11F0}/g;
    s/\x{F112}/\x{114C}\x{116A}/g;
    s/\x{F113}/\x{114C}\x{116A}\x{11A8}/g;
    s/\x{F114}/\x{114C}\x{116A}\x{11AB}/g;
    s/\x{F115}/\x{114C}\x{116A}\x{11AF}/g;
    s/\x{F116}/\x{114C}\x{116A}\x{11D9}/g;
    s/\x{F117}/\x{114C}\x{116A}\x{11B7}/g;
    s/\x{F118}/\x{114C}\x{116A}\x{11DD}/g;
    s/\x{F119}/\x{114C}\x{116A}\x{11BC}/g;
    s/\x{F11A}/\x{114C}\x{116A}\x{11F0}/g;
    s/\x{F11B}/\x{114C}\x{116B}/g;
    s/\x{F11C}/\x{114C}\x{116B}\x{11BC}/g;
    s/\x{F11D}/\x{114C}\x{116C}/g;
    s/\x{F11E}/\x{114C}\x{116C}\x{11AB}/g;
    s/\x{F11F}/\x{114C}\x{116C}\x{11BC}/g;
    s/\x{F120}/\x{114C}\x{116C}\x{11F0}/g;
    s/\x{F121}/\x{114C}\x{116D}/g;
    s/\x{F122}/\x{114C}\x{116D}\x{11B7}/g;
    s/\x{F123}/\x{114C}\x{116D}\x{11E2}/g;
    s/\x{F124}/\x{114C}\x{116D}\x{11F0}/g;
    s/\x{F125}/\x{114C}\x{1184}/g;
    s/\x{F126}/\x{114C}\x{1185}/g;
    s/\x{F127}/\x{114C}\x{1185}\x{11BC}/g;
    s/\x{F128}/\x{114C}\x{116E}/g;
    s/\x{F129}/\x{114C}\x{116E}\x{11A8}/g;
    s/\x{F12A}/\x{114C}\x{116E}\x{11AB}/g;
    s/\x{F12B}/\x{114C}\x{116E}\x{11AF}/g;
    s/\x{F12C}/\x{114C}\x{116E}\x{11D9}/g;
    s/\x{F12D}/\x{114C}\x{116E}\x{11B7}/g;
    s/\x{F12E}/\x{114C}\x{116E}\x{11E2}/g;
    s/\x{F12F}/\x{114C}\x{116E}\x{11BC}/g;
    s/\x{F130}/\x{114C}\x{116F}/g;
    s/\x{F131}/\x{114C}\x{116F}\x{11AB}/g;
    s/\x{F132}/\x{114C}\x{116F}\x{11AF}/g;
    s/\x{F133}/\x{114C}\x{116F}\x{11D9}/g;
    s/\x{F134}/\x{114C}\x{116F}\x{11BC}/g;
    s/\x{F135}/\x{114C}\x{116F}\x{11F0}/g;
    s/\x{F136}/\x{114C}\x{1170}/g;
    s/\x{F137}/\x{114C}\x{1171}/g;
    s/\x{F138}/\x{114C}\x{1171}\x{11AF}/g;
    s/\x{F139}/\x{114C}\x{1171}\x{11D9}/g;
    s/\x{F13A}/\x{114C}\x{1171}\x{11BC}/g;
    s/\x{F13B}/\x{114C}\x{1171}\x{11F0}/g;
    s/\x{F13C}/\x{114C}\x{1172}/g;
    s/\x{F13D}/\x{114C}\x{1172}\x{11A8}/g;
    s/\x{F13E}/\x{114C}\x{1172}\x{11AB}/g;
    s/\x{F13F}/\x{114C}\x{1172}\x{11AF}/g;
    s/\x{F140}/\x{114C}\x{1172}\x{11D9}/g;
    s/\x{F141}/\x{114C}\x{1172}\x{11B7}/g;
    s/\x{F142}/\x{114C}\x{1172}\x{11E2}/g;
    s/\x{F143}/\x{114C}\x{1191}/g;
    s/\x{F144}/\x{114C}\x{1191}\x{11A8}/g;
    s/\x{F145}/\x{114C}\x{1191}\x{11AB}/g;
    s/\x{F146}/\x{114C}\x{1191}\x{11F0}/g;
    s/\x{F147}/\x{114C}\x{1191}\x{11F9}/g;
    s/\x{F148}/\x{114C}\x{1194}/g;
    s/\x{F149}/\x{114C}\x{1173}/g;
    s/\x{F14A}/\x{114C}\x{1173}\x{11A8}/g;
    s/\x{F14B}/\x{114C}\x{1173}\x{11AB}/g;
    s/\x{F14C}/\x{114C}\x{1173}\x{11AF}/g;
    s/\x{F14D}/\x{114C}\x{1173}\x{11D9}/g;
    s/\x{F14E}/\x{114C}\x{1173}\x{11B7}/g;
    s/\x{F14F}/\x{114C}\x{1173}\x{11B8}/g;
    s/\x{F150}/\x{114C}\x{1173}\x{11BC}/g;
    s/\x{F151}/\x{114C}\x{1173}\x{11F0}/g;
    s/\x{F152}/\x{114C}\x{1174}/g;
    s/\x{F153}/\x{114C}\x{1174}\x{11BC}/g;
    s/\x{F154}/\x{114C}\x{1175}/g;
    s/\x{F155}/\x{114C}\x{1175}\x{11AB}/g;
    s/\x{F156}/\x{114C}\x{1175}\x{11AE}/g;
    s/\x{F157}/\x{114C}\x{1175}\x{11AF}/g;
    s/\x{F158}/\x{114C}\x{1175}\x{11BA}/g;
    s/\x{F159}/\x{114C}\x{1175}\x{11EB}/g;
    s/\x{F15A}/\x{114C}\x{1175}\x{11BC}/g;
    s/\x{F15B}/\x{114C}\x{1175}\x{11F0}/g;
    s/\x{F15C}/\x{114C}\x{119E}/g;
    s/\x{F15D}/\x{114C}\x{119E}\x{11A8}/g;
    s/\x{F15E}/\x{114C}\x{119E}\x{11AB}/g;
    s/\x{F15F}/\x{114C}\x{119E}\x{11AE}/g;
    s/\x{F160}/\x{114C}\x{119E}\x{11AF}/g;
    s/\x{F161}/\x{114C}\x{119E}\x{11BA}/g;
    s/\x{F162}/\x{114C}\x{11A1}/g;
    s/\x{F163}/\x{114C}\x{11A1}\x{11A8}/g;
    s/\x{F164}/\x{110C}\x{1161}\x{11C7}/g;
    s/\x{F165}/\x{110C}\x{1161}\x{11CD}/g;
    s/\x{F166}/\x{110C}\x{1161}\x{11D0}/g;
    s/\x{F167}/\x{110C}\x{1161}\x{11D9}/g;
    s/\x{F168}/\x{110C}\x{1161}\x{11DA}/g;
    s/\x{F169}/\x{110C}\x{1161}\x{11DD}/g;
    s/\x{F16A}/\x{110C}\x{1161}\x{11E2}/g;
    s/\x{F16B}/\x{110C}\x{1161}\x{11B8}\x{11B8}/g;
    s/\x{F16C}/\x{110C}\x{1161}\x{11E6}/g;
    s/\x{F16D}/\x{110C}\x{1161}\x{11EB}/g;
    s/\x{F16E}/\x{110C}\x{1161}\x{11F0}/g;
    s/\x{F16F}/\x{110C}\x{1161}\x{11F0}\x{11A8}/g;
    s/\x{F170}/\x{110C}\x{1161}\x{11F1}/g;
    s/\x{F171}/\x{110C}\x{1161}\x{11F9}/g;
    s/\x{F172}/\x{110C}\x{1176}/g;
    s/\x{F173}/\x{110C}\x{1162}\x{11F0}/g;
    s/\x{F174}/\x{110C}\x{1163}\x{11C7}/g;
    s/\x{F175}/\x{110C}\x{1163}\x{11CD}/g;
    s/\x{F176}/\x{110C}\x{1163}\x{11D0}/g;
    s/\x{F177}/\x{110C}\x{1163}\x{11E2}/g;
    s/\x{F178}/\x{110C}\x{1163}\x{11EB}/g;
    s/\x{F179}/\x{110C}\x{1163}\x{11BC}\x{11BA}/g;
    s/\x{F17A}/\x{110C}\x{1163}\x{11F0}/g;
    s/\x{F17B}/\x{110C}\x{1163}\x{11F1}/g;
    s/\x{F17C}/\x{110C}\x{1178}/g;
    s/\x{F17D}/\x{110C}\x{1163}\x{116E}/g;
    s/\x{F17E}/\x{110C}\x{1163}\x{116E}\x{11AF}/g;
    s/\x{F17F}/\x{110C}\x{1165}\x{11A8}\x{11C2}/g;
    s/\x{F180}/\x{110C}\x{1165}\x{11CD}/g;
    s/\x{F181}/\x{110C}\x{1165}\x{11D0}/g;
    s/\x{F182}/\x{110C}\x{1165}\x{11D9}/g;
    s/\x{F183}/\x{110C}\x{1165}\x{11DD}/g;
    s/\x{F184}/\x{110C}\x{1165}\x{11E4}/g;
    s/\x{F185}/\x{110C}\x{1165}\x{11BA}\x{11C2}/g;
    s/\x{F186}/\x{110C}\x{1165}\x{11EB}/g;
    s/\x{F187}/\x{110C}\x{1165}\x{11F0}/g;
    s/\x{F188}/\x{110C}\x{1165}\x{11F0}\x{11C2}/g;
    s/\x{F189}/\x{110C}\x{1165}\x{11F9}/g;
    s/\x{F18A}/\x{110C}\x{117B}/g;
    s/\x{F18B}/\x{110C}\x{117C}/g;
    s/\x{F18C}/\x{110C}\x{117C}\x{11A8}/g;
    s/\x{F18D}/\x{110C}\x{117C}\x{11AB}/g;
    s/\x{F18E}/\x{110C}\x{117C}\x{11BC}/g;
    s/\x{F18F}/\x{110C}\x{117C}\x{11F9}/g;
    s/\x{F190}/\x{110C}\x{1167}\x{11C7}/g;
    s/\x{F191}/\x{110C}\x{1167}\x{11CA}/g;
    s/\x{F192}/\x{110C}\x{1167}\x{11AE}\x{11AE}/g;
    s/\x{F193}/\x{110C}\x{1167}\x{11AE}\x{11AE}\x{11B8}/g;
    s/\x{F194}/\x{110C}\x{1167}\x{11AE}\x{11BD}/g;
    s/\x{F195}/\x{110C}\x{1167}\x{11D0}/g;
    s/\x{F196}/\x{110C}\x{1167}\x{11D9}/g;
    s/\x{F197}/\x{110C}\x{1167}\x{11E2}/g;
    s/\x{F198}/\x{110C}\x{1167}\x{11EB}/g;
    s/\x{F199}/\x{110C}\x{1167}\x{11F0}/g;
    s/\x{F19A}/\x{110C}\x{1167}\x{11F9}/g;
    s/\x{F19B}/\x{110C}\x{1167}\x{1163}/g;
    s/\x{F19C}/\x{110C}\x{1167}\x{1163}\x{11BC}/g;
    s/\x{F19D}/\x{110C}\x{117D}/g;
    s/\x{F19E}/\x{110C}\x{117E}/g;
    s/\x{F19F}/\x{110C}\x{1169}\x{11C7}/g;
    s/\x{F1A0}/\x{110C}\x{1169}\x{11D9}/g;
    s/\x{F1A1}/\x{110C}\x{1169}\x{11DD}/g;
    s/\x{F1A2}/\x{110C}\x{1169}\x{11E2}/g;
    s/\x{F1A3}/\x{110C}\x{1169}\x{11EB}/g;
    s/\x{F1A4}/\x{110C}\x{1169}\x{11F0}/g;
    s/\x{F1A5}/\x{110C}\x{116A}\x{11D9}/g;
    s/\x{F1A6}/\x{110C}\x{116A}\x{11E2}/g;
    s/\x{F1A7}/\x{110C}\x{116A}\x{11F0}/g;
    s/\x{F1A8}/\x{110C}\x{116D}\x{11D9}/g;
    s/\x{F1A9}/\x{110C}\x{116D}\x{11E2}/g;
    s/\x{F1AA}/\x{110C}\x{116D}\x{11E6}/g;
    s/\x{F1AB}/\x{110C}\x{116D}\x{11BC}\x{11BA}/g;
    s/\x{F1AC}/\x{110C}\x{116D}\x{11F0}/g;
    s/\x{F1AD}/\x{110C}\x{116D}\x{11F1}/g;
    s/\x{F1AE}/\x{110C}\x{1187}/g;
    s/\x{F1AF}/\x{110C}\x{1188}/g;
    s/\x{F1B0}/\x{110C}\x{1188}\x{11AF}/g;
    s/\x{F1B1}/\x{110C}\x{1188}\x{11D9}/g;
    s/\x{F1B2}/\x{110C}\x{116E}\x{11C7}/g;
    s/\x{F1B3}/\x{110C}\x{116E}\x{11CD}/g;
    s/\x{F1B4}/\x{110C}\x{116E}\x{11D9}/g;
    s/\x{F1B5}/\x{110C}\x{116E}\x{11DD}/g;
    s/\x{F1B6}/\x{110C}\x{116E}\x{11E2}/g;
    s/\x{F1B7}/\x{110C}\x{116E}\x{11EB}/g;
    s/\x{F1B8}/\x{110C}\x{116E}\x{11F0}/g;
    s/\x{F1B9}/\x{110C}\x{116E}\x{11F9}/g;
    s/\x{F1BA}/\x{110C}\x{1189}/g;
    s/\x{F1BB}/\x{110C}\x{1189}\x{11BC}/g;
    s/\x{F1BC}/\x{110C}\x{118A}/g;
    s/\x{F1BD}/\x{110C}\x{116E}\x{1167}/g;
    s/\x{F1BE}/\x{110C}\x{116E}\x{1167}\x{11AB}/g;
    s/\x{F1BF}/\x{110C}\x{116E}\x{1167}\x{11AF}/g;
    s/\x{F1C0}/\x{110C}\x{116E}\x{1167}\x{11D9}/g;
    s/\x{F1C1}/\x{110C}\x{118C}/g;
    s/\x{F1C2}/\x{110C}\x{1172}\x{11D9}/g;
    s/\x{F1C3}/\x{110C}\x{1172}\x{11E2}/g;
    s/\x{F1C4}/\x{110C}\x{1172}\x{11F0}/g;
    s/\x{F1C5}/\x{110C}\x{1172}\x{11F0}\x{11A8}/g;
    s/\x{F1C6}/\x{110C}\x{1172}\x{11F9}/g;
    s/\x{F1C7}/\x{110C}\x{118E}/g;
    s/\x{F1C8}/\x{110C}\x{118E}\x{11AB}/g;
    s/\x{F1C9}/\x{110C}\x{118F}/g;
    s/\x{F1CA}/\x{110C}\x{118F}\x{11AB}/g;
    s/\x{F1CB}/\x{110C}\x{118F}\x{11AF}/g;
    s/\x{F1CC}/\x{110C}\x{1190}/g;
    s/\x{F1CD}/\x{110C}\x{1191}/g;
    s/\x{F1CE}/\x{110C}\x{1191}\x{11AB}/g;
    s/\x{F1CF}/\x{110C}\x{1191}\x{11AF}/g;
    s/\x{F1D0}/\x{110C}\x{1191}\x{11D9}/g;
    s/\x{F1D1}/\x{110C}\x{1191}\x{11F9}/g;
    s/\x{F1D2}/\x{110C}\x{1192}/g;
    s/\x{F1D3}/\x{110C}\x{1192}\x{11AB}/g;
    s/\x{F1D4}/\x{110C}\x{1192}\x{11B8}/g;
    s/\x{F1D5}/\x{110C}\x{1192}\x{11BC}/g;
    s/\x{F1D6}/\x{110C}\x{1194}/g;
    s/\x{F1D7}/\x{110C}\x{1194}\x{11A8}/g;
    s/\x{F1D8}/\x{110C}\x{1194}\x{11AB}/g;
    s/\x{F1D9}/\x{110C}\x{1194}\x{11AF}/g;
    s/\x{F1DA}/\x{110C}\x{1194}\x{11D9}/g;
    s/\x{F1DB}/\x{110C}\x{1194}\x{11BA}/g;
    s/\x{F1DC}/\x{110C}\x{1194}\x{11BC}/g;
    s/\x{F1DD}/\x{110C}\x{1173}\x{11D0}/g;
    s/\x{F1DE}/\x{110C}\x{1173}\x{11D9}/g;
    s/\x{F1DF}/\x{110C}\x{1173}\x{11E2}/g;
    s/\x{F1E0}/\x{110C}\x{1173}\x{11E6}/g;
    s/\x{F1E1}/\x{110C}\x{1173}\x{11EB}/g;
    s/\x{F1E2}/\x{110C}\x{1173}\x{11F0}/g;
    s/\x{F1E3}/\x{110C}\x{1195}/g;
    s/\x{F1E4}/\x{110C}\x{1196}/g;
    s/\x{F1E5}/\x{110C}\x{1174}\x{11F0}/g;
    s/\x{F1E6}/\x{110C}\x{1174}\x{11F9}/g;
    s/\x{F1E7}/\x{110C}\x{1175}\x{11A8}\x{11C2}/g;
    s/\x{F1E8}/\x{110C}\x{1175}\x{11AE}\x{11BE}/g;
    s/\x{F1E9}/\x{110C}\x{1175}\x{11CD}/g;
    s/\x{F1EA}/\x{110C}\x{1175}\x{11D9}/g;
    s/\x{F1EB}/\x{110C}\x{1175}\x{11DD}/g;
    s/\x{F1EC}/\x{110C}\x{1175}\x{11E2}/g;
    s/\x{F1ED}/\x{110C}\x{1175}\x{11E6}/g;
    s/\x{F1EE}/\x{110C}\x{1175}\x{11BA}\x{11BE}/g;
    s/\x{F1EF}/\x{110C}\x{1175}\x{11BA}\x{11C0}/g;
    s/\x{F1F0}/\x{110C}\x{1175}\x{11BA}\x{11C2}/g;
    s/\x{F1F1}/\x{110C}\x{1175}\x{11EB}/g;
    s/\x{F1F2}/\x{110C}\x{1175}\x{11F0}/g;
    s/\x{F1F3}/\x{110C}\x{1175}\x{11F9}/g;
    s/\x{F1F4}/\x{110C}\x{1198}/g;
    s/\x{F1F5}/\x{110C}\x{1199}/g;
    s/\x{F1F6}/\x{110C}\x{1199}\x{11AB}/g;
    s/\x{F1F7}/\x{110C}\x{1175}\x{1167}/g;
    s/\x{F1F8}/\x{110C}\x{1175}\x{1168}/g;
    s/\x{F1F9}/\x{110C}\x{119A}/g;
    s/\x{F1FA}/\x{110C}\x{119A}\x{11AF}/g;
    s/\x{F1FB}/\x{110C}\x{119B}/g;
    s/\x{F1FC}/\x{110C}\x{119E}/g;
    s/\x{F1FD}/\x{110C}\x{119E}\x{11A8}/g;
    s/\x{F1FE}/\x{110C}\x{119E}\x{11AB}/g;
    s/\x{F1FF}/\x{110C}\x{119E}\x{11AE}/g;
    s/\x{F200}/\x{110C}\x{119E}\x{11AF}/g;
    s/\x{F201}/\x{110C}\x{119E}\x{11CD}/g;
    s/\x{F202}/\x{110C}\x{119E}\x{11D0}/g;
    s/\x{F203}/\x{110C}\x{119E}\x{11D9}/g;
    s/\x{F204}/\x{110C}\x{119E}\x{11B7}/g;
    s/\x{F205}/\x{110C}\x{119E}\x{11DA}/g;
    s/\x{F206}/\x{110C}\x{119E}\x{11DD}/g;
    s/\x{F207}/\x{110C}\x{119E}\x{11B8}/g;
    s/\x{F208}/\x{110C}\x{119E}\x{11E6}/g;
    s/\x{F209}/\x{110C}\x{119E}\x{11BA}/g;
    s/\x{F20A}/\x{110C}\x{119E}\x{11EB}/g;
    s/\x{F20B}/\x{110C}\x{119E}\x{11BC}/g;
    s/\x{F20C}/\x{110C}\x{119E}\x{11F0}/g;
    s/\x{F20D}/\x{110C}\x{119E}\x{11BD}/g;
    s/\x{F20E}/\x{110C}\x{119E}\x{11BE}/g;
    s/\x{F20F}/\x{110C}\x{119E}\x{11C0}/g;
    s/\x{F210}/\x{110C}\x{119E}\x{1161}/g;
    s/\x{F211}/\x{110C}\x{119E}\x{1161}\x{11BC}/g;
    s/\x{F212}/\x{110C}\x{11A1}/g;
    s/\x{F213}/\x{110C}\x{11A1}\x{11A8}/g;
    s/\x{F214}/\x{110C}\x{11A1}\x{11AB}/g;
    s/\x{F215}/\x{110C}\x{11A1}\x{11AE}/g;
    s/\x{F216}/\x{110C}\x{11A1}\x{11AF}/g;
    s/\x{F217}/\x{110C}\x{11A1}\x{11B7}/g;
    s/\x{F218}/\x{110C}\x{11A1}\x{11B8}/g;
    s/\x{F219}/\x{110C}\x{11A1}\x{11BA}/g;
    s/\x{F21A}/\x{110C}\x{11A1}\x{11BC}/g;
    s/\x{F21B}/\x{110C}\x{11A1}\x{11F0}/g;
    s/\x{F21C}/\x{114D}\x{1173}/g;
    s/\x{F21D}/\x{114D}\x{1173}\x{11AF}/g;
    s/\x{F21E}/\x{114D}\x{1175}/g;
    s/\x{F21F}/\x{114D}\x{1175}\x{11AF}/g;
    s/\x{F220}/\x{110D}\x{1161}\x{11D0}/g;
    s/\x{F221}/\x{110D}\x{1161}\x{11D9}/g;
    s/\x{F222}/\x{110D}\x{1161}\x{11E2}/g;
    s/\x{F223}/\x{110D}\x{1161}\x{11B8}\x{11B8}/g;
    s/\x{F224}/\x{110D}\x{1161}\x{11F0}/g;
    s/\x{F225}/\x{110D}\x{1161}\x{11F9}/g;
    s/\x{F226}/\x{110D}\x{1163}\x{11E2}/g;
    s/\x{F227}/\x{110D}\x{1163}\x{11E6}/g;
    s/\x{F228}/\x{110D}\x{1163}\x{11F0}/g;
    s/\x{F229}/\x{110D}\x{1165}\x{11F0}/g;
    s/\x{F22A}/\x{110D}\x{1165}\x{11F9}/g;
    s/\x{F22B}/\x{110D}\x{1167}\x{11D9}/g;
    s/\x{F22C}/\x{110D}\x{1167}\x{11F0}/g;
    s/\x{F22D}/\x{110D}\x{1169}\x{11D9}/g;
    s/\x{F22E}/\x{110D}\x{1169}\x{11E2}/g;
    s/\x{F22F}/\x{110D}\x{1169}\x{11F0}/g;
    s/\x{F230}/\x{110D}\x{1169}\x{11BD}\x{11BD}/g;
    s/\x{F231}/\x{110D}\x{1169}\x{11F9}/g;
    s/\x{F232}/\x{110D}\x{116A}\x{11F0}/g;
    s/\x{F233}/\x{110D}\x{117F}/g;
    s/\x{F234}/\x{110D}\x{116C}\x{11F9}/g;
    s/\x{F235}/\x{110D}\x{116D}\x{11E2}/g;
    s/\x{F236}/\x{110D}\x{116D}\x{11F0}/g;
    s/\x{F237}/\x{110D}\x{116E}\x{11D9}/g;
    s/\x{F238}/\x{110D}\x{116E}\x{11E2}/g;
    s/\x{F239}/\x{110D}\x{116E}\x{11F0}/g;
    s/\x{F23A}/\x{110D}\x{116E}\x{11F9}/g;
    s/\x{F23B}/\x{110D}\x{1171}\x{11F9}/g;
    s/\x{F23C}/\x{110D}\x{1172}\x{11D9}/g;
    s/\x{F23D}/\x{110D}\x{1172}\x{11E2}/g;
    s/\x{F23E}/\x{110D}\x{1191}/g;
    s/\x{F23F}/\x{110D}\x{1191}\x{11AB}/g;
    s/\x{F240}/\x{110D}\x{1191}\x{11AF}/g;
    s/\x{F241}/\x{110D}\x{1191}\x{11D9}/g;
    s/\x{F242}/\x{110D}\x{1191}\x{11F9}/g;
    s/\x{F243}/\x{110D}\x{1194}/g;
    s/\x{F244}/\x{110D}\x{1194}\x{11BC}/g;
    s/\x{F245}/\x{110D}\x{1173}\x{11E2}/g;
    s/\x{F246}/\x{110D}\x{1173}\x{11EB}/g;
    s/\x{F247}/\x{110D}\x{1173}\x{11F0}/g;
    s/\x{F248}/\x{110D}\x{1173}\x{11BD}\x{11BD}/g;
    s/\x{F249}/\x{110D}\x{1174}\x{11F0}/g;
    s/\x{F24A}/\x{110D}\x{1174}\x{11F9}/g;
    s/\x{F24B}/\x{110D}\x{1175}\x{11D9}/g;
    s/\x{F24C}/\x{110D}\x{1175}\x{11E2}/g;
    s/\x{F24D}/\x{110D}\x{1175}\x{11EB}/g;
    s/\x{F24E}/\x{110D}\x{1175}\x{11F0}/g;
    s/\x{F24F}/\x{110D}\x{1175}\x{11F9}/g;
    s/\x{F250}/\x{110D}\x{119E}/g;
    s/\x{F251}/\x{110D}\x{119E}\x{11A8}/g;
    s/\x{F252}/\x{110D}\x{119E}\x{11AB}/g;
    s/\x{F253}/\x{110D}\x{119E}\x{11AF}/g;
    s/\x{F254}/\x{110D}\x{119E}\x{11B7}/g;
    s/\x{F255}/\x{110D}\x{119E}\x{11B8}/g;
    s/\x{F256}/\x{110D}\x{119E}\x{11E6}/g;
    s/\x{F257}/\x{110D}\x{119E}\x{11BA}/g;
    s/\x{F258}/\x{110D}\x{119E}\x{11BC}/g;
    s/\x{F259}/\x{110D}\x{119E}\x{11F0}/g;
    s/\x{F25A}/\x{110D}\x{11A1}/g;
    s/\x{F25B}/\x{110D}\x{11A1}\x{11A8}/g;
    s/\x{F25C}/\x{110D}\x{11A1}\x{11AB}/g;
    s/\x{F25D}/\x{110D}\x{11A1}\x{11AF}/g;
    s/\x{F25E}/\x{110D}\x{11A1}\x{11D9}/g;
    s/\x{F25F}/\x{110D}\x{11A1}\x{11B7}/g;
    s/\x{F260}/\x{110D}\x{11A1}\x{11BC}/g;
    s/\x{F261}/\x{110D}\x{11A1}\x{11F0}/g;
    s/\x{F262}/\x{110D}\x{11A1}\x{11F9}/g;
    s/\x{F263}/\x{110D}\x{1112}\x{1165}/g;
    s/\x{F264}/\x{110D}\x{1112}\x{1166}/g;
    s/\x{F265}/\x{110D}\x{1112}\x{1166}\x{11AF}/g;
    s/\x{F266}/\x{110D}\x{1112}\x{1166}\x{11D0}/g;
    s/\x{F267}/\x{110D}\x{1112}\x{1166}\x{11D0}\x{11BF}/g;
    s/\x{F268}/\x{114E}\x{1161}/g;
    s/\x{F269}/\x{114E}\x{1161}\x{11AB}/g;
    s/\x{F26A}/\x{114E}\x{1161}\x{11B7}/g;
    s/\x{F26B}/\x{114E}\x{1161}\x{11E2}/g;
    s/\x{F26C}/\x{114E}\x{1162}/g;
    s/\x{F26D}/\x{114E}\x{1163}/g;
    s/\x{F26E}/\x{114E}\x{1163}\x{11BC}/g;
    s/\x{F26F}/\x{114E}\x{1165}/g;
    s/\x{F270}/\x{114E}\x{1165}\x{11AB}/g;
    s/\x{F271}/\x{114E}\x{1167}/g;
    s/\x{F272}/\x{114E}\x{1167}\x{11AB}/g;
    s/\x{F273}/\x{114E}\x{1167}\x{11F9}/g;
    s/\x{F274}/\x{114E}\x{1168}/g;
    s/\x{F275}/\x{114E}\x{1169}/g;
    s/\x{F276}/\x{114E}\x{116A}/g;
    s/\x{F277}/\x{114E}\x{116D}/g;
    s/\x{F278}/\x{114E}\x{1184}/g;
    s/\x{F279}/\x{114E}\x{116E}/g;
    s/\x{F27A}/\x{114E}\x{116E}\x{11AB}/g;
    s/\x{F27B}/\x{114E}\x{116E}\x{11BC}/g;
    s/\x{F27C}/\x{114E}\x{1171}/g;
    s/\x{F27D}/\x{114E}\x{1172}/g;
    s/\x{F27E}/\x{114E}\x{1172}\x{11AB}/g;
    s/\x{F27F}/\x{114E}\x{1172}\x{11BC}/g;
    s/\x{F280}/\x{114E}\x{1191}/g;
    s/\x{F281}/\x{114E}\x{1191}\x{11AB}/g;
    s/\x{F282}/\x{114E}\x{1173}/g;
    s/\x{F283}/\x{114E}\x{1173}\x{11AB}/g;
    s/\x{F284}/\x{114E}\x{1173}\x{11BA}/g;
    s/\x{F285}/\x{114E}\x{1173}\x{11EB}/g;
    s/\x{F286}/\x{114E}\x{1175}/g;
    s/\x{F287}/\x{114E}\x{1175}\x{11AB}/g;
    s/\x{F288}/\x{114E}\x{1175}\x{11B7}/g;
    s/\x{F289}/\x{114E}\x{1175}\x{11E2}/g;
    s/\x{F28A}/\x{114E}\x{1175}\x{11EB}/g;
    s/\x{F28B}/\x{114F}\x{1161}/g;
    s/\x{F28C}/\x{114F}\x{1161}\x{11AB}/g;
    s/\x{F28D}/\x{114F}\x{1161}\x{11B7}/g;
    s/\x{F28E}/\x{114F}\x{1161}\x{11E2}/g;
    s/\x{F28F}/\x{114F}\x{1161}\x{11F0}/g;
    s/\x{F290}/\x{114F}\x{1162}/g;
    s/\x{F291}/\x{114F}\x{1163}/g;
    s/\x{F292}/\x{114F}\x{1163}\x{11B7}/g;
    s/\x{F293}/\x{114F}\x{1163}\x{11F0}/g;
    s/\x{F294}/\x{114F}\x{1165}/g;
    s/\x{F295}/\x{114F}\x{1165}\x{11AB}/g;
    s/\x{F296}/\x{114F}\x{1167}/g;
    s/\x{F297}/\x{114F}\x{1167}\x{11AB}/g;
    s/\x{F298}/\x{114F}\x{1168}/g;
    s/\x{F299}/\x{114F}\x{116E}/g;
    s/\x{F29A}/\x{114F}\x{1171}/g;
    s/\x{F29B}/\x{114F}\x{1172}/g;
    s/\x{F29C}/\x{114F}\x{1172}\x{11F0}/g;
    s/\x{F29D}/\x{114F}\x{1191}/g;
    s/\x{F29E}/\x{114F}\x{1191}\x{11AB}/g;
    s/\x{F29F}/\x{114F}\x{1173}/g;
    s/\x{F2A0}/\x{114F}\x{1173}\x{11EB}/g;
    s/\x{F2A1}/\x{114F}\x{1173}\x{11F0}/g;
    s/\x{F2A2}/\x{114F}\x{1175}/g;
    s/\x{F2A3}/\x{114F}\x{1175}\x{11AB}/g;
    s/\x{F2A4}/\x{114F}\x{119E}/g;
    s/\x{F2A5}/\x{114F}\x{11A1}/g;
    s/\x{F2A6}/\x{1150}\x{1161}/g;
    s/\x{F2A7}/\x{1150}\x{1161}\x{11AB}/g;
    s/\x{F2A8}/\x{1150}\x{1161}\x{11AE}/g;
    s/\x{F2A9}/\x{1150}\x{1161}\x{11B7}/g;
    s/\x{F2AA}/\x{1150}\x{1161}\x{11E2}/g;
    s/\x{F2AB}/\x{1150}\x{1161}\x{11B8}/g;
    s/\x{F2AC}/\x{1150}\x{1162}/g;
    s/\x{F2AD}/\x{1150}\x{1163}/g;
    s/\x{F2AE}/\x{1150}\x{1163}\x{11B7}/g;
    s/\x{F2AF}/\x{1150}\x{1163}\x{11E2}/g;
    s/\x{F2B0}/\x{1150}\x{1163}\x{11F0}/g;
    s/\x{F2B1}/\x{1150}\x{1167}/g;
    s/\x{F2B2}/\x{1150}\x{1167}\x{11AB}/g;
    s/\x{F2B3}/\x{1150}\x{1169}/g;
    s/\x{F2B4}/\x{1150}\x{116A}/g;
    s/\x{F2B5}/\x{1150}\x{116A}\x{11BC}/g;
    s/\x{F2B6}/\x{1150}\x{116E}/g;
    s/\x{F2B7}/\x{1150}\x{116E}\x{11BC}/g;
    s/\x{F2B8}/\x{1150}\x{116E}\x{11F9}/g;
    s/\x{F2B9}/\x{1150}\x{1172}/g;
    s/\x{F2BA}/\x{1150}\x{1172}\x{11A8}/g;
    s/\x{F2BB}/\x{1150}\x{1172}\x{11AB}/g;
    s/\x{F2BC}/\x{1150}\x{1172}\x{11F0}/g;
    s/\x{F2BD}/\x{1150}\x{1191}/g;
    s/\x{F2BE}/\x{1150}\x{1191}\x{11AB}/g;
    s/\x{F2BF}/\x{1150}\x{1173}/g;
    s/\x{F2C0}/\x{1150}\x{1173}\x{11B7}/g;
    s/\x{F2C1}/\x{1150}\x{1173}\x{11EB}/g;
    s/\x{F2C2}/\x{1150}\x{1175}/g;
    s/\x{F2C3}/\x{1150}\x{1175}\x{11AB}/g;
    s/\x{F2C4}/\x{1150}\x{1175}\x{11EB}/g;
    s/\x{F2C5}/\x{1150}\x{1175}\x{11F0}/g;
    s/\x{F2C6}/\x{1151}\x{1161}/g;
    s/\x{F2C7}/\x{1151}\x{1161}\x{11AB}/g;
    s/\x{F2C8}/\x{1151}\x{1161}\x{11C2}/g;
    s/\x{F2C9}/\x{1151}\x{1162}/g;
    s/\x{F2CA}/\x{1151}\x{1163}/g;
    s/\x{F2CB}/\x{1151}\x{1163}\x{11B7}/g;
    s/\x{F2CC}/\x{1151}\x{1163}\x{11E2}/g;
    s/\x{F2CD}/\x{1151}\x{1163}\x{11B8}/g;
    s/\x{F2CE}/\x{1151}\x{1163}\x{11F0}/g;
    s/\x{F2CF}/\x{1151}\x{1167}/g;
    s/\x{F2D0}/\x{1151}\x{1167}\x{11AB}/g;
    s/\x{F2D1}/\x{1151}\x{116E}/g;
    s/\x{F2D2}/\x{1151}\x{1171}/g;
    s/\x{F2D3}/\x{1151}\x{1172}/g;
    s/\x{F2D4}/\x{1151}\x{1172}\x{11F0}/g;
    s/\x{F2D5}/\x{1151}\x{1191}/g;
    s/\x{F2D6}/\x{1151}\x{1191}\x{11AB}/g;
    s/\x{F2D7}/\x{1151}\x{1173}/g;
    s/\x{F2D8}/\x{1151}\x{1173}\x{11B7}/g;
    s/\x{F2D9}/\x{1151}\x{1173}\x{11E2}/g;
    s/\x{F2DA}/\x{1151}\x{1175}/g;
    s/\x{F2DB}/\x{1151}\x{1175}\x{11AB}/g;
    s/\x{F2DC}/\x{1151}\x{1175}\x{11B7}/g;
    s/\x{F2DD}/\x{1151}\x{1175}\x{11EB}/g;
    s/\x{F2DE}/\x{1151}\x{1175}\x{11F0}/g;
    s/\x{F2DF}/\x{110E}\x{1161}\x{11D9}/g;
    s/\x{F2E0}/\x{110E}\x{1161}\x{11E2}/g;
    s/\x{F2E1}/\x{110E}\x{1161}\x{11E6}/g;
    s/\x{F2E2}/\x{110E}\x{1161}\x{11F0}/g;
    s/\x{F2E3}/\x{110E}\x{1161}\x{11F9}/g;
    s/\x{F2E4}/\x{110E}\x{1176}/g;
    s/\x{F2E5}/\x{110E}\x{1163}\x{11D9}/g;
    s/\x{F2E6}/\x{110E}\x{1163}\x{11E6}/g;
    s/\x{F2E7}/\x{110E}\x{1163}\x{11F0}/g;
    s/\x{F2E8}/\x{110E}\x{1163}\x{11F1}/g;
    s/\x{F2E9}/\x{110E}\x{1178}/g;
    s/\x{F2EA}/\x{110E}\x{1178}\x{11AF}/g;
    s/\x{F2EB}/\x{110E}\x{1163}\x{116E}/g;
    s/\x{F2EC}/\x{110E}\x{1165}\x{11EB}/g;
    s/\x{F2ED}/\x{110E}\x{1165}\x{11F0}/g;
    s/\x{F2EE}/\x{110E}\x{117C}/g;
    s/\x{F2EF}/\x{110E}\x{117C}\x{11AB}/g;
    s/\x{F2F0}/\x{110E}\x{1167}\x{11AE}\x{11AE}/g;
    s/\x{F2F1}/\x{110E}\x{1167}\x{11AE}\x{11AE}\x{11B8}/g;
    s/\x{F2F2}/\x{110E}\x{1167}\x{11AE}\x{11B8}/g;
    s/\x{F2F3}/\x{110E}\x{1167}\x{11D9}/g;
    s/\x{F2F4}/\x{110E}\x{1167}\x{11E2}/g;
    s/\x{F2F5}/\x{110E}\x{1167}\x{11F0}/g;
    s/\x{F2F6}/\x{110E}\x{1167}\x{11F1}/g;
    s/\x{F2F7}/\x{110E}\x{1167}\x{11F9}/g;
    s/\x{F2F8}/\x{110E}\x{117D}/g;
    s/\x{F2F9}/\x{110E}\x{117E}/g;
    s/\x{F2FA}/\x{110E}\x{1169}\x{11D9}/g;
    s/\x{F2FB}/\x{110E}\x{1169}\x{11E2}/g;
    s/\x{F2FC}/\x{110E}\x{1169}\x{11B8}\x{11BD}/g;
    s/\x{F2FD}/\x{110E}\x{1169}\x{11EB}/g;
    s/\x{F2FE}/\x{110E}\x{1169}\x{11F0}/g;
    s/\x{F2FF}/\x{110E}\x{116A}\x{11D9}/g;
    s/\x{F300}/\x{110E}\x{116A}\x{11F0}/g;
    s/\x{F301}/\x{110E}\x{1182}/g;
    s/\x{F302}/\x{110E}\x{116D}\x{11E2}/g;
    s/\x{F303}/\x{110E}\x{116D}\x{11F0}/g;
    s/\x{F304}/\x{110E}\x{1187}/g;
    s/\x{F305}/\x{110E}\x{1188}/g;
    s/\x{F306}/\x{110E}\x{1188}\x{11AB}/g;
    s/\x{F307}/\x{110E}\x{116E}\x{11D9}/g;
    s/\x{F308}/\x{110E}\x{116E}\x{11E0}/g;
    s/\x{F309}/\x{110E}\x{116E}\x{11E2}/g;
    s/\x{F30A}/\x{110E}\x{116E}\x{11F0}/g;
    s/\x{F30B}/\x{110E}\x{116E}\x{11F9}/g;
    s/\x{F30C}/\x{110E}\x{1189}/g;
    s/\x{F30D}/\x{110E}\x{1189}\x{11AB}/g;
    s/\x{F30E}/\x{110E}\x{1189}\x{11BC}/g;
    s/\x{F30F}/\x{110E}\x{118A}/g;
    s/\x{F310}/\x{110E}\x{116F}\x{11F9}/g;
    s/\x{F311}/\x{110E}\x{116E}\x{1167}/g;
    s/\x{F312}/\x{110E}\x{118C}/g;
    s/\x{F313}/\x{110E}\x{1171}\x{11D9}/g;
    s/\x{F314}/\x{110E}\x{1171}\x{11F9}/g;
    s/\x{F315}/\x{110E}\x{1172}\x{11D9}/g;
    s/\x{F316}/\x{110E}\x{1172}\x{11E2}/g;
    s/\x{F317}/\x{110E}\x{1172}\x{11F0}/g;
    s/\x{F318}/\x{110E}\x{1172}\x{11F9}/g;
    s/\x{F319}/\x{110E}\x{118E}/g;
    s/\x{F31A}/\x{110E}\x{118E}\x{11AB}/g;
    s/\x{F31B}/\x{110E}\x{118F}/g;
    s/\x{F31C}/\x{110E}\x{118F}\x{11B7}/g;
    s/\x{F31D}/\x{110E}\x{1190}/g;
    s/\x{F31E}/\x{110E}\x{1191}/g;
    s/\x{F31F}/\x{110E}\x{1191}\x{11AB}/g;
    s/\x{F320}/\x{110E}\x{1191}\x{11AF}/g;
    s/\x{F321}/\x{110E}\x{1191}\x{11D9}/g;
    s/\x{F322}/\x{110E}\x{1191}\x{11B7}/g;
    s/\x{F323}/\x{110E}\x{1192}/g;
    s/\x{F324}/\x{110E}\x{1192}\x{11BC}/g;
    s/\x{F325}/\x{110E}\x{1194}/g;
    s/\x{F326}/\x{110E}\x{1194}\x{11AB}/g;
    s/\x{F327}/\x{110E}\x{1194}\x{11AF}/g;
    s/\x{F328}/\x{110E}\x{1194}\x{11BC}/g;
    s/\x{F329}/\x{110E}\x{1173}\x{11AF}\x{11F0}/g;
    s/\x{F32A}/\x{110E}\x{1173}\x{11D9}/g;
    s/\x{F32B}/\x{110E}\x{1173}\x{11AF}\x{11BC}/g;
    s/\x{F32C}/\x{110E}\x{1173}\x{11E2}/g;
    s/\x{F32D}/\x{110E}\x{1173}\x{11EB}/g;
    s/\x{F32E}/\x{110E}\x{1173}\x{11F0}/g;
    s/\x{F32F}/\x{110E}\x{1195}/g;
    s/\x{F330}/\x{110E}\x{1196}/g;
    s/\x{F331}/\x{110E}\x{1174}\x{11F9}/g;
    s/\x{F332}/\x{110E}\x{1175}\x{11D9}/g;
    s/\x{F333}/\x{110E}\x{1175}\x{11E2}/g;
    s/\x{F334}/\x{110E}\x{1175}\x{11B8}\x{11AE}/g;
    s/\x{F335}/\x{110E}\x{1175}\x{11E6}/g;
    s/\x{F336}/\x{110E}\x{1175}\x{11EB}/g;
    s/\x{F337}/\x{110E}\x{1175}\x{11F0}/g;
    s/\x{F338}/\x{110E}\x{1175}\x{11F9}/g;
    s/\x{F339}/\x{110E}\x{1198}/g;
    s/\x{F33A}/\x{110E}\x{1199}/g;
    s/\x{F33B}/\x{110E}\x{1199}\x{11AB}/g;
    s/\x{F33C}/\x{110E}\x{1199}\x{11BC}/g;
    s/\x{F33D}/\x{110E}\x{1175}\x{1167}/g;
    s/\x{F33E}/\x{110E}\x{1175}\x{1168}/g;
    s/\x{F33F}/\x{110E}\x{1175}\x{116D}/g;
    s/\x{F340}/\x{110E}\x{119B}/g;
    s/\x{F341}/\x{110E}\x{119E}/g;
    s/\x{F342}/\x{110E}\x{119E}\x{11A8}/g;
    s/\x{F343}/\x{110E}\x{119E}\x{11AB}/g;
    s/\x{F344}/\x{110E}\x{119E}\x{11AE}/g;
    s/\x{F345}/\x{110E}\x{119E}\x{11AF}/g;
    s/\x{F346}/\x{110E}\x{119E}\x{11CE}/g;
    s/\x{F347}/\x{110E}\x{119E}\x{11B1}/g;
    s/\x{F348}/\x{110E}\x{119E}\x{11B3}/g;
    s/\x{F349}/\x{110E}\x{119E}\x{11B6}/g;
    s/\x{F34A}/\x{110E}\x{119E}\x{11D9}/g;
    s/\x{F34B}/\x{110E}\x{119E}\x{11B7}/g;
    s/\x{F34C}/\x{110E}\x{119E}\x{11DD}/g;
    s/\x{F34D}/\x{110E}\x{119E}\x{11B8}/g;
    s/\x{F34E}/\x{110E}\x{119E}\x{11BA}/g;
    s/\x{F34F}/\x{110E}\x{119E}\x{11BB}/g;
    s/\x{F350}/\x{110E}\x{119E}\x{11BC}/g;
    s/\x{F351}/\x{110E}\x{119E}\x{11BD}/g;
    s/\x{F352}/\x{110E}\x{119E}\x{11BE}/g;
    s/\x{F353}/\x{110E}\x{11A1}/g;
    s/\x{F354}/\x{110E}\x{11A1}\x{11A8}/g;
    s/\x{F355}/\x{110E}\x{11A1}\x{11AB}/g;
    s/\x{F356}/\x{110E}\x{11A1}\x{11AF}/g;
    s/\x{F357}/\x{110E}\x{11A1}\x{11B7}/g;
    s/\x{F358}/\x{110E}\x{11A1}\x{11BA}/g;
    s/\x{F359}/\x{110E}\x{11A1}\x{11BC}/g;
    s/\x{F35A}/\x{110E}\x{11A1}\x{11F0}/g;
    s/\x{F35B}/\x{1152}\x{116E}/g;
    s/\x{F35C}/\x{1152}\x{116F}/g;
    s/\x{F35D}/\x{1153}\x{1175}/g;
    s/\x{F35E}/\x{1153}\x{1175}\x{11AF}/g;
    s/\x{F35F}/\x{1154}\x{1161}/g;
    s/\x{F360}/\x{1154}\x{1161}\x{11AB}/g;
    s/\x{F361}/\x{1154}\x{1176}/g;
    s/\x{F362}/\x{1154}\x{1162}/g;
    s/\x{F363}/\x{1154}\x{1165}/g;
    s/\x{F364}/\x{1154}\x{1165}\x{11AB}/g;
    s/\x{F365}/\x{1154}\x{1167}/g;
    s/\x{F366}/\x{1154}\x{1167}\x{11AB}/g;
    s/\x{F367}/\x{1154}\x{1168}/g;
    s/\x{F368}/\x{1154}\x{1169}/g;
    s/\x{F369}/\x{1154}\x{116D}/g;
    s/\x{F36A}/\x{1154}\x{116D}\x{1161}/g;
    s/\x{F36B}/\x{1154}\x{1184}/g;
    s/\x{F36C}/\x{1154}\x{116E}/g;
    s/\x{F36D}/\x{1154}\x{116E}\x{11AB}/g;
    s/\x{F36E}/\x{1154}\x{116E}\x{11BC}/g;
    s/\x{F36F}/\x{1154}\x{1171}/g;
    s/\x{F370}/\x{1154}\x{1172}/g;
    s/\x{F371}/\x{1154}\x{1172}\x{11AB}/g;
    s/\x{F372}/\x{1154}\x{1191}/g;
    s/\x{F373}/\x{1154}\x{1191}\x{11AB}/g;
    s/\x{F374}/\x{1154}\x{1173}/g;
    s/\x{F375}/\x{1154}\x{1175}/g;
    s/\x{F376}/\x{1154}\x{1175}\x{11AB}/g;
    s/\x{F377}/\x{1154}\x{1175}\x{11EB}/g;
    s/\x{F378}/\x{1154}\x{1175}\x{11F0}/g;
    s/\x{F379}/\x{1154}\x{1175}\x{11F9}/g;
    s/\x{F37A}/\x{1155}\x{1161}/g;
    s/\x{F37B}/\x{1155}\x{1161}\x{11AB}/g;
    s/\x{F37C}/\x{1155}\x{1161}\x{11B7}/g;
    s/\x{F37D}/\x{1155}\x{1161}\x{11E2}/g;
    s/\x{F37E}/\x{1155}\x{1161}\x{11F9}/g;
    s/\x{F37F}/\x{1155}\x{1162}/g;
    s/\x{F380}/\x{1155}\x{1163}/g;
    s/\x{F381}/\x{1155}\x{1163}\x{11F0}/g;
    s/\x{F382}/\x{1155}\x{1167}/g;
    s/\x{F383}/\x{1155}\x{1167}\x{11AB}/g;
    s/\x{F384}/\x{1155}\x{1169}/g;
    s/\x{F385}/\x{1155}\x{116A}/g;
    s/\x{F386}/\x{1155}\x{116A}\x{11AB}/g;
    s/\x{F387}/\x{1155}\x{116B}/g;
    s/\x{F388}/\x{1155}\x{116E}/g;
    s/\x{F389}/\x{1155}\x{1171}/g;
    s/\x{F38A}/\x{1155}\x{1172}/g;
    s/\x{F38B}/\x{1155}\x{1172}\x{11A8}/g;
    s/\x{F38C}/\x{1155}\x{1172}\x{11AB}/g;
    s/\x{F38D}/\x{1155}\x{1172}\x{11BC}/g;
    s/\x{F38E}/\x{1155}\x{1172}\x{11F0}/g;
    s/\x{F38F}/\x{1155}\x{1191}/g;
    s/\x{F390}/\x{1155}\x{1191}\x{11AB}/g;
    s/\x{F391}/\x{1155}\x{1173}/g;
    s/\x{F392}/\x{1155}\x{1173}\x{11AB}/g;
    s/\x{F393}/\x{1155}\x{1173}\x{11EB}/g;
    s/\x{F394}/\x{1155}\x{1175}/g;
    s/\x{F395}/\x{1155}\x{1175}\x{11AB}/g;
    s/\x{F396}/\x{1155}\x{1175}\x{11B7}/g;
    s/\x{F397}/\x{1155}\x{1175}\x{11E2}/g;
    s/\x{F398}/\x{1155}\x{1175}\x{11EB}/g;
    s/\x{F399}/\x{1155}\x{1175}\x{11F9}/g;
    s/\x{F39A}/\x{1155}\x{119B}/g;
    s/\x{F39B}/\x{110F}\x{1161}\x{11D0}/g;
    s/\x{F39C}/\x{110F}\x{1161}\x{11D9}/g;
    s/\x{F39D}/\x{110F}\x{1161}\x{11E2}/g;
    s/\x{F39E}/\x{110F}\x{1161}\x{11E6}/g;
    s/\x{F39F}/\x{110F}\x{1161}\x{11F0}/g;
    s/\x{F3A0}/\x{110F}\x{1176}/g;
    s/\x{F3A1}/\x{110F}\x{1163}\x{11E2}/g;
    s/\x{F3A2}/\x{110F}\x{1163}\x{11F0}/g;
    s/\x{F3A3}/\x{110F}\x{1163}\x{11F9}/g;
    s/\x{F3A4}/\x{110F}\x{1178}/g;
    s/\x{F3A5}/\x{110F}\x{1165}\x{11D9}/g;
    s/\x{F3A6}/\x{110F}\x{1165}\x{11F9}/g;
    s/\x{F3A7}/\x{110F}\x{117A}/g;
    s/\x{F3A8}/\x{110F}\x{117B}/g;
    s/\x{F3A9}/\x{110F}\x{1167}\x{11D9}/g;
    s/\x{F3AA}/\x{110F}\x{1167}\x{11E2}/g;
    s/\x{F3AB}/\x{110F}\x{1167}\x{11F0}/g;
    s/\x{F3AC}/\x{110F}\x{1167}\x{11F9}/g;
    s/\x{F3AD}/\x{110F}\x{117E}/g;
    s/\x{F3AE}/\x{110F}\x{1169}\x{11D9}/g;
    s/\x{F3AF}/\x{110F}\x{1169}\x{11E2}/g;
    s/\x{F3B0}/\x{110F}\x{1169}\x{11F0}/g;
    s/\x{F3B1}/\x{110F}\x{116A}\x{11D9}/g;
    s/\x{F3B2}/\x{110F}\x{116A}\x{11F0}/g;
    s/\x{F3B3}/\x{110F}\x{1182}/g;
    s/\x{F3B4}/\x{110F}\x{116C}\x{11F0}/g;
    s/\x{F3B5}/\x{110F}\x{116D}\x{11D9}/g;
    s/\x{F3B6}/\x{110F}\x{116D}\x{11E2}/g;
    s/\x{F3B7}/\x{110F}\x{1187}/g;
    s/\x{F3B8}/\x{110F}\x{116E}\x{11D9}/g;
    s/\x{F3B9}/\x{110F}\x{116E}\x{11E2}/g;
    s/\x{F3BA}/\x{110F}\x{116E}\x{11F0}/g;
    s/\x{F3BB}/\x{110F}\x{116E}\x{11F9}/g;
    s/\x{F3BC}/\x{110F}\x{1189}/g;
    s/\x{F3BD}/\x{110F}\x{1189}\x{11AB}/g;
    s/\x{F3BE}/\x{110F}\x{1189}\x{11BC}/g;
    s/\x{F3BF}/\x{110F}\x{118A}/g;
    s/\x{F3C0}/\x{110F}\x{116F}\x{11D9}/g;
    s/\x{F3C1}/\x{110F}\x{116F}\x{11F9}/g;
    s/\x{F3C2}/\x{110F}\x{116E}\x{1167}/g;
    s/\x{F3C3}/\x{110F}\x{116E}\x{1167}\x{11AB}/g;
    s/\x{F3C4}/\x{110F}\x{118C}/g;
    s/\x{F3C5}/\x{110F}\x{1172}\x{11F0}/g;
    s/\x{F3C6}/\x{110F}\x{1172}\x{11F9}/g;
    s/\x{F3C7}/\x{110F}\x{118F}/g;
    s/\x{F3C8}/\x{110F}\x{1190}/g;
    s/\x{F3C9}/\x{110F}\x{1191}/g;
    s/\x{F3CA}/\x{110F}\x{1191}\x{11A8}/g;
    s/\x{F3CB}/\x{110F}\x{1191}\x{11AB}/g;
    s/\x{F3CC}/\x{110F}\x{1191}\x{11AF}/g;
    s/\x{F3CD}/\x{110F}\x{1191}\x{11D9}/g;
    s/\x{F3CE}/\x{110F}\x{1191}\x{11F0}/g;
    s/\x{F3CF}/\x{110F}\x{1191}\x{11F9}/g;
    s/\x{F3D0}/\x{110F}\x{1192}/g;
    s/\x{F3D1}/\x{110F}\x{1192}\x{11BC}/g;
    s/\x{F3D2}/\x{110F}\x{1194}/g;
    s/\x{F3D3}/\x{110F}\x{1194}\x{11BC}/g;
    s/\x{F3D4}/\x{110F}\x{1173}\x{11D9}/g;
    s/\x{F3D5}/\x{110F}\x{1173}\x{11E2}/g;
    s/\x{F3D6}/\x{110F}\x{1173}\x{11F0}/g;
    s/\x{F3D7}/\x{110F}\x{1195}/g;
    s/\x{F3D8}/\x{110F}\x{1196}/g;
    s/\x{F3D9}/\x{110F}\x{1174}\x{11F9}/g;
    s/\x{F3DA}/\x{110F}\x{1175}\x{11D9}/g;
    s/\x{F3DB}/\x{110F}\x{1175}\x{11F0}/g;
    s/\x{F3DC}/\x{110F}\x{1175}\x{11F9}/g;
    s/\x{F3DD}/\x{110F}\x{1198}/g;
    s/\x{F3DE}/\x{110F}\x{1199}/g;
    s/\x{F3DF}/\x{110F}\x{1199}\x{11AB}/g;
    s/\x{F3E0}/\x{110F}\x{1199}\x{11BC}/g;
    s/\x{F3E1}/\x{110F}\x{1175}\x{1178}/g;
    s/\x{F3E2}/\x{110F}\x{1175}\x{1167}/g;
    s/\x{F3E3}/\x{110F}\x{1175}\x{1168}/g;
    s/\x{F3E4}/\x{110F}\x{119A}/g;
    s/\x{F3E5}/\x{110F}\x{1175}\x{116D}/g;
    s/\x{F3E6}/\x{110F}\x{119B}/g;
    s/\x{F3E7}/\x{110F}\x{119B}\x{11AF}/g;
    s/\x{F3E8}/\x{110F}\x{1175}\x{1172}/g;
    s/\x{F3E9}/\x{110F}\x{1175}\x{1172}\x{11AB}/g;
    s/\x{F3EA}/\x{110F}\x{119E}/g;
    s/\x{F3EB}/\x{110F}\x{119E}\x{11A8}/g;
    s/\x{F3EC}/\x{110F}\x{119E}\x{11AB}/g;
    s/\x{F3ED}/\x{110F}\x{119E}\x{11AE}/g;
    s/\x{F3EE}/\x{110F}\x{119E}\x{11AF}/g;
    s/\x{F3EF}/\x{110F}\x{119E}\x{11B7}/g;
    s/\x{F3F0}/\x{110F}\x{119E}\x{11B8}/g;
    s/\x{F3F1}/\x{110F}\x{119E}\x{11BA}/g;
    s/\x{F3F2}/\x{110F}\x{119E}\x{11C0}/g;
    s/\x{F3F3}/\x{110F}\x{11A1}/g;
    s/\x{F3F4}/\x{110F}\x{11A1}\x{11A8}/g;
    s/\x{F3F5}/\x{110F}\x{11A1}\x{11AB}/g;
    s/\x{F3F6}/\x{110F}\x{11A1}\x{11AF}/g;
    s/\x{F3F7}/\x{110F}\x{11A1}\x{11B7}/g;
    s/\x{F3F8}/\x{110F}\x{11A1}\x{11BA}/g;
    s/\x{F3F9}/\x{110F}\x{11A1}\x{11BC}/g;
    s/\x{F3FA}/\x{110F}\x{11A1}\x{11F0}/g;
    s/\x{F3FB}/\x{1110}\x{1161}\x{11D9}/g;
    s/\x{F3FC}/\x{1110}\x{1161}\x{11E2}/g;
    s/\x{F3FD}/\x{1110}\x{1161}\x{11F0}/g;
    s/\x{F3FE}/\x{1110}\x{1161}\x{11F9}/g;
    s/\x{F3FF}/\x{1110}\x{1176}/g;
    s/\x{F400}/\x{1110}\x{1177}/g;
    s/\x{F401}/\x{1110}\x{1163}\x{11E2}/g;
    s/\x{F402}/\x{1110}\x{1163}\x{11F0}/g;
    s/\x{F403}/\x{1110}\x{1178}/g;
    s/\x{F404}/\x{1110}\x{1165}\x{11DD}/g;
    s/\x{F405}/\x{1110}\x{1165}\x{11E6}/g;
    s/\x{F406}/\x{1110}\x{1165}\x{11F0}/g;
    s/\x{F407}/\x{1110}\x{1165}\x{11F1}/g;
    s/\x{F408}/\x{1110}\x{117B}/g;
    s/\x{F409}/\x{1110}\x{117B}\x{11AE}/g;
    s/\x{F40A}/\x{1110}\x{117B}\x{11AF}/g;
    s/\x{F40B}/\x{1110}\x{1167}\x{11D9}/g;
    s/\x{F40C}/\x{1110}\x{1167}\x{11F0}/g;
    s/\x{F40D}/\x{1110}\x{1167}\x{11F9}/g;
    s/\x{F40E}/\x{1110}\x{1168}\x{11F0}/g;
    s/\x{F40F}/\x{1110}\x{1169}\x{11D0}/g;
    s/\x{F410}/\x{1110}\x{1169}\x{11D9}/g;
    s/\x{F411}/\x{1110}\x{1169}\x{11E2}/g;
    s/\x{F412}/\x{1110}\x{1169}\x{11B8}\x{11B7}/g;
    s/\x{F413}/\x{1110}\x{1169}\x{11E7}/g;
    s/\x{F414}/\x{1110}\x{1169}\x{11F0}/g;
    s/\x{F415}/\x{1110}\x{116A}\x{11D9}/g;
    s/\x{F416}/\x{1110}\x{116A}\x{11F0}/g;
    s/\x{F417}/\x{1110}\x{1182}/g;
    s/\x{F418}/\x{1110}\x{1182}\x{11AF}/g;
    s/\x{F419}/\x{1110}\x{116D}\x{11D9}/g;
    s/\x{F41A}/\x{1110}\x{116D}\x{11E2}/g;
    s/\x{F41B}/\x{1110}\x{116D}\x{11F0}/g;
    s/\x{F41C}/\x{1110}\x{1188}/g;
    s/\x{F41D}/\x{1110}\x{116E}\x{11D0}/g;
    s/\x{F41E}/\x{1110}\x{116E}\x{11D9}/g;
    s/\x{F41F}/\x{1110}\x{116E}\x{11E2}/g;
    s/\x{F420}/\x{1110}\x{116E}\x{11F0}/g;
    s/\x{F421}/\x{1110}\x{116E}\x{11F9}/g;
    s/\x{F422}/\x{1110}\x{1189}/g;
    s/\x{F423}/\x{1110}\x{1189}\x{11AB}/g;
    s/\x{F424}/\x{1110}\x{116F}\x{11F9}/g;
    s/\x{F425}/\x{1110}\x{116E}\x{1167}/g;
    s/\x{F426}/\x{1110}\x{118C}/g;
    s/\x{F427}/\x{1110}\x{118D}/g;
    s/\x{F428}/\x{1110}\x{1172}\x{11D9}/g;
    s/\x{F429}/\x{1110}\x{1172}\x{11E2}/g;
    s/\x{F42A}/\x{1110}\x{1172}\x{11F0}/g;
    s/\x{F42B}/\x{1110}\x{1194}/g;
    s/\x{F42C}/\x{1110}\x{1173}\x{11D0}/g;
    s/\x{F42D}/\x{1110}\x{1173}\x{11D7}/g;
    s/\x{F42E}/\x{1110}\x{1173}\x{11D9}/g;
    s/\x{F42F}/\x{1110}\x{1173}\x{11E2}/g;
    s/\x{F430}/\x{1110}\x{1173}\x{11F0}/g;
    s/\x{F431}/\x{1110}\x{1195}/g;
    s/\x{F432}/\x{1110}\x{1196}/g;
    s/\x{F433}/\x{1110}\x{1174}\x{11F9}/g;
    s/\x{F434}/\x{1110}\x{1175}\x{11D9}/g;
    s/\x{F435}/\x{1110}\x{1175}\x{11E6}/g;
    s/\x{F436}/\x{1110}\x{1175}\x{11F0}/g;
    s/\x{F437}/\x{1110}\x{1198}/g;
    s/\x{F438}/\x{1110}\x{1199}/g;
    s/\x{F439}/\x{1110}\x{1199}\x{11AB}/g;
    s/\x{F43A}/\x{1110}\x{1175}\x{1167}/g;
    s/\x{F43B}/\x{1110}\x{1175}\x{1168}/g;
    s/\x{F43C}/\x{1110}\x{119E}/g;
    s/\x{F43D}/\x{1110}\x{119E}\x{11A8}/g;
    s/\x{F43E}/\x{1110}\x{119E}\x{11AB}/g;
    s/\x{F43F}/\x{1110}\x{119E}\x{11AE}/g;
    s/\x{F440}/\x{1110}\x{119E}\x{11AF}/g;
    s/\x{F441}/\x{1110}\x{119E}\x{11B0}/g;
    s/\x{F442}/\x{1110}\x{119E}\x{11B3}/g;
    s/\x{F443}/\x{1110}\x{119E}\x{11B7}/g;
    s/\x{F444}/\x{1110}\x{119E}\x{11B8}/g;
    s/\x{F445}/\x{1110}\x{119E}\x{11BA}/g;
    s/\x{F446}/\x{1110}\x{119E}\x{11BC}/g;
    s/\x{F447}/\x{1110}\x{119E}\x{11BD}/g;
    s/\x{F448}/\x{1110}\x{11A1}/g;
    s/\x{F449}/\x{1110}\x{11A1}\x{11A8}/g;
    s/\x{F44A}/\x{1110}\x{11A1}\x{11AB}/g;
    s/\x{F44B}/\x{1110}\x{11A1}\x{11AF}/g;
    s/\x{F44C}/\x{1110}\x{11A1}\x{11B7}/g;
    s/\x{F44D}/\x{1110}\x{11A1}\x{11BA}/g;
    s/\x{F44E}/\x{1110}\x{11A1}\x{11BC}/g;
    s/\x{F44F}/\x{1110}\x{11A1}\x{11F0}/g;
    s/\x{F450}/\x{1110}\x{11A1}\x{11F9}/g;
    s/\x{F451}/\x{1110}\x{1110}\x{116E}/g;
    s/\x{F452}/\x{1110}\x{1110}\x{119E}/g;
    s/\x{F453}/\x{1110}\x{1110}\x{119E}\x{11AF}/g;
    s/\x{F454}/\x{1111}\x{1161}\x{11D0}/g;
    s/\x{F455}/\x{1111}\x{1161}\x{11D8}/g;
    s/\x{F456}/\x{1111}\x{1161}\x{11D9}/g;
    s/\x{F457}/\x{1111}\x{1161}\x{11E2}/g;
    s/\x{F458}/\x{1111}\x{1161}\x{11E6}/g;
    s/\x{F459}/\x{1111}\x{1161}\x{11E7}/g;
    s/\x{F45A}/\x{1111}\x{1161}\x{11F0}/g;
    s/\x{F45B}/\x{1111}\x{1161}\x{11F9}/g;
    s/\x{F45C}/\x{1111}\x{1176}/g;
    s/\x{F45D}/\x{1111}\x{1162}\x{11F9}/g;
    s/\x{F45E}/\x{1111}\x{1163}\x{11E2}/g;
    s/\x{F45F}/\x{1111}\x{1178}/g;
    s/\x{F460}/\x{1111}\x{1165}\x{11D0}/g;
    s/\x{F461}/\x{1111}\x{1165}\x{11F0}/g;
    s/\x{F462}/\x{1111}\x{117A}/g;
    s/\x{F463}/\x{1111}\x{117C}/g;
    s/\x{F464}/\x{1111}\x{117C}\x{11AF}/g;
    s/\x{F465}/\x{1111}\x{1166}\x{11C7}/g;
    s/\x{F466}/\x{1111}\x{1167}\x{11C7}/g;
    s/\x{F467}/\x{1111}\x{1167}\x{11D9}/g;
    s/\x{F468}/\x{1111}\x{1167}\x{11E2}/g;
    s/\x{F469}/\x{1111}\x{1167}\x{11F0}/g;
    s/\x{F46A}/\x{1111}\x{1167}\x{11F9}/g;
    s/\x{F46B}/\x{1111}\x{1169}\x{11D9}/g;
    s/\x{F46C}/\x{1111}\x{1169}\x{11E2}/g;
    s/\x{F46D}/\x{1111}\x{1169}\x{11F0}/g;
    s/\x{F46E}/\x{1111}\x{1182}/g;
    s/\x{F46F}/\x{1111}\x{116D}\x{11E2}/g;
    s/\x{F470}/\x{1111}\x{116D}\x{11E6}/g;
    s/\x{F471}/\x{1111}\x{1188}/g;
    s/\x{F472}/\x{1111}\x{116E}\x{11D0}/g;
    s/\x{F473}/\x{1111}\x{116E}\x{11D9}/g;
    s/\x{F474}/\x{1111}\x{116E}\x{11E2}/g;
    s/\x{F475}/\x{1111}\x{116E}\x{11F0}/g;
    s/\x{F476}/\x{1111}\x{116E}\x{11F1}/g;
    s/\x{F477}/\x{1111}\x{116F}\x{11F9}/g;
    s/\x{F478}/\x{1111}\x{116E}\x{1167}/g;
    s/\x{F479}/\x{1111}\x{118C}/g;
    s/\x{F47A}/\x{1111}\x{1171}\x{11F9}/g;
    s/\x{F47B}/\x{1111}\x{1172}\x{11E2}/g;
    s/\x{F47C}/\x{1111}\x{1172}\x{11F0}/g;
    s/\x{F47D}/\x{1111}\x{118E}/g;
    s/\x{F47E}/\x{1111}\x{118E}\x{11AB}/g;
    s/\x{F47F}/\x{1111}\x{1194}/g;
    s/\x{F480}/\x{1111}\x{1173}\x{11D9}/g;
    s/\x{F481}/\x{1111}\x{1173}\x{11F0}/g;
    s/\x{F482}/\x{1111}\x{1196}/g;
    s/\x{F483}/\x{1111}\x{1174}\x{11F0}/g;
    s/\x{F484}/\x{1111}\x{1174}\x{11F9}/g;
    s/\x{F485}/\x{1111}\x{1175}\x{11D9}/g;
    s/\x{F486}/\x{1111}\x{1175}\x{11F0}/g;
    s/\x{F487}/\x{1111}\x{1175}\x{11F9}/g;
    s/\x{F488}/\x{1111}\x{1198}/g;
    s/\x{F489}/\x{1111}\x{1199}/g;
    s/\x{F48A}/\x{1111}\x{1199}\x{11AB}/g;
    s/\x{F48B}/\x{1111}\x{1175}\x{1178}/g;
    s/\x{F48C}/\x{1111}\x{1175}\x{1167}/g;
    s/\x{F48D}/\x{1111}\x{1175}\x{1168}/g;
    s/\x{F48E}/\x{1111}\x{119E}/g;
    s/\x{F48F}/\x{1111}\x{119E}\x{11A8}/g;
    s/\x{F490}/\x{1111}\x{119E}\x{11AB}/g;
    s/\x{F491}/\x{1111}\x{119E}\x{11AE}/g;
    s/\x{F492}/\x{1111}\x{119E}\x{11AF}/g;
    s/\x{F493}/\x{1111}\x{119E}\x{11B0}/g;
    s/\x{F494}/\x{1111}\x{119E}\x{11B3}/g;
    s/\x{F495}/\x{1111}\x{119E}\x{11B6}/g;
    s/\x{F496}/\x{1111}\x{119E}\x{11B7}/g;
    s/\x{F497}/\x{1111}\x{119E}\x{11B8}/g;
    s/\x{F498}/\x{1111}\x{119E}\x{11BA}/g;
    s/\x{F499}/\x{1111}\x{119E}\x{11E7}/g;
    s/\x{F49A}/\x{1111}\x{119E}\x{11BC}/g;
    s/\x{F49B}/\x{1111}\x{119E}\x{11BD}/g;
    s/\x{F49C}/\x{1111}\x{119E}\x{11BE}/g;
    s/\x{F49D}/\x{1111}\x{119E}\x{11C0}/g;
    s/\x{F49E}/\x{1111}\x{119F}/g;
    s/\x{F49F}/\x{1111}\x{119F}\x{11AB}/g;
    s/\x{F4A0}/\x{1111}\x{11A1}/g;
    s/\x{F4A1}/\x{1111}\x{11A1}\x{11A8}/g;
    s/\x{F4A2}/\x{1111}\x{11A1}\x{11AB}/g;
    s/\x{F4A3}/\x{1111}\x{11A1}\x{11AF}/g;
    s/\x{F4A4}/\x{1111}\x{11A1}\x{11B7}/g;
    s/\x{F4A5}/\x{1111}\x{11A1}\x{11B8}/g;
    s/\x{F4A6}/\x{1111}\x{11A1}\x{11BA}/g;
    s/\x{F4A7}/\x{1111}\x{11A1}\x{11BC}/g;
    s/\x{F4A8}/\x{1111}\x{11A1}\x{11F0}/g;
    s/\x{F4A9}/\x{1156}\x{1161}/g;
    s/\x{F4AA}/\x{1156}\x{1161}\x{11AB}/g;
    s/\x{F4AB}/\x{1111}\x{1112}\x{1175}/g;
    s/\x{F4AC}/\x{1111}\x{1112}\x{1175}\x{11AB}/g;
    s/\x{F4AD}/\x{1157}\x{116E}/g;
    s/\x{F4AE}/\x{1157}\x{116E}\x{11AB}/g;
    s/\x{F4AF}/\x{1112}\x{1161}\x{11CE}/g;
    s/\x{F4B0}/\x{1112}\x{1161}\x{11CF}/g;
    s/\x{F4B1}/\x{1112}\x{1161}\x{11D0}/g;
    s/\x{F4B2}/\x{1112}\x{1161}\x{11B1}\x{11C2}/g;
    s/\x{F4B3}/\x{1112}\x{1161}\x{11D9}/g;
    s/\x{F4B4}/\x{1112}\x{1161}\x{11B7}\x{11BD}/g;
    s/\x{F4B5}/\x{1112}\x{1161}\x{11E2}/g;
    s/\x{F4B6}/\x{1112}\x{1161}\x{11F0}/g;
    s/\x{F4B7}/\x{1112}\x{1161}\x{11F1}/g;
    s/\x{F4B8}/\x{1112}\x{1176}/g;
    s/\x{F4B9}/\x{1112}\x{1177}/g;
    s/\x{F4BA}/\x{1112}\x{1162}\x{11EB}/g;
    s/\x{F4BB}/\x{1112}\x{1162}\x{11F0}/g;
    s/\x{F4BC}/\x{1112}\x{1162}\x{11F0}\x{11C2}/g;
    s/\x{F4BD}/\x{1112}\x{1163}\x{11C7}/g;
    s/\x{F4BE}/\x{1112}\x{1163}\x{11E2}/g;
    s/\x{F4BF}/\x{1112}\x{1163}\x{11F0}/g;
    s/\x{F4C0}/\x{1112}\x{1163}\x{11F9}/g;
    s/\x{F4C1}/\x{1112}\x{1178}/g;
    s/\x{F4C2}/\x{1112}\x{1164}\x{11F0}/g;
    s/\x{F4C3}/\x{1112}\x{1165}\x{11D0}/g;
    s/\x{F4C4}/\x{1112}\x{1165}\x{11D9}/g;
    s/\x{F4C5}/\x{1112}\x{1165}\x{11F0}/g;
    s/\x{F4C6}/\x{1112}\x{1165}\x{11F1}/g;
    s/\x{F4C7}/\x{1112}\x{1165}\x{11F9}/g;
    s/\x{F4C8}/\x{1112}\x{117A}/g;
    s/\x{F4C9}/\x{1112}\x{117B}/g;
    s/\x{F4CA}/\x{1112}\x{117C}/g;
    s/\x{F4CB}/\x{1112}\x{117C}\x{11AB}/g;
    s/\x{F4CC}/\x{1112}\x{117C}\x{11B7}/g;
    s/\x{F4CD}/\x{1112}\x{1167}\x{11D9}/g;
    s/\x{F4CE}/\x{1112}\x{1167}\x{11E2}/g;
    s/\x{F4CF}/\x{1112}\x{1167}\x{11F0}/g;
    s/\x{F4D0}/\x{1112}\x{1167}\x{11F9}/g;
    s/\x{F4D1}/\x{1112}\x{1168}\x{11D9}/g;
    s/\x{F4D2}/\x{1112}\x{1169}\x{11C7}/g;
    s/\x{F4D3}/\x{1112}\x{1169}\x{11CD}/g;
    s/\x{F4D4}/\x{1112}\x{1169}\x{11D9}/g;
    s/\x{F4D5}/\x{1112}\x{1169}\x{11DA}/g;
    s/\x{F4D6}/\x{1112}\x{1169}\x{11DD}/g;
    s/\x{F4D7}/\x{1112}\x{1169}\x{11E2}/g;
    s/\x{F4D8}/\x{1112}\x{1169}\x{11EB}/g;
    s/\x{F4D9}/\x{1112}\x{1169}\x{11F0}/g;
    s/\x{F4DA}/\x{1112}\x{1169}\x{11F1}/g;
    s/\x{F4DB}/\x{1112}\x{116A}\x{11D9}/g;
    s/\x{F4DC}/\x{1112}\x{116A}\x{11F0}/g;
    s/\x{F4DD}/\x{1112}\x{116A}\x{11F1}/g;
    s/\x{F4DE}/\x{1112}\x{116B}\x{11F9}/g;
    s/\x{F4DF}/\x{1112}\x{1169}\x{1163}/g;
    s/\x{F4E0}/\x{1112}\x{117F}/g;
    s/\x{F4E1}/\x{1112}\x{117F}\x{11AB}/g;
    s/\x{F4E2}/\x{1112}\x{117F}\x{11F9}/g;
    s/\x{F4E3}/\x{1112}\x{1180}/g;
    s/\x{F4E4}/\x{1112}\x{1182}/g;
    s/\x{F4E5}/\x{1112}\x{116C}\x{11F0}/g;
    s/\x{F4E6}/\x{1112}\x{116C}\x{11F9}/g;
    s/\x{F4E7}/\x{1112}\x{116D}\x{11D9}/g;
    s/\x{F4E8}/\x{1112}\x{116D}\x{11E2}/g;
    s/\x{F4E9}/\x{1112}\x{116D}\x{1161}/g;
    s/\x{F4EA}/\x{1112}\x{116D}\x{1161}\x{11AB}/g;
    s/\x{F4EB}/\x{1112}\x{1184}/g;
    s/\x{F4EC}/\x{1112}\x{1184}\x{11BC}/g;
    s/\x{F4ED}/\x{1112}\x{1187}/g;
    s/\x{F4EE}/\x{1112}\x{1188}/g;
    s/\x{F4EF}/\x{1112}\x{1188}\x{11AB}/g;
    s/\x{F4F0}/\x{1112}\x{116E}\x{11D9}/g;
    s/\x{F4F1}/\x{1112}\x{116E}\x{11DD}/g;
    s/\x{F4F2}/\x{1112}\x{116E}\x{11E2}/g;
    s/\x{F4F3}/\x{1112}\x{116E}\x{11EB}/g;
    s/\x{F4F4}/\x{1112}\x{116E}\x{11F0}/g;
    s/\x{F4F5}/\x{1112}\x{116E}\x{11F9}/g;
    s/\x{F4F6}/\x{1112}\x{1189}/g;
    s/\x{F4F7}/\x{1112}\x{1189}\x{11AB}/g;
    s/\x{F4F8}/\x{1112}\x{1189}\x{11BC}/g;
    s/\x{F4F9}/\x{1112}\x{118A}/g;
    s/\x{F4FA}/\x{1112}\x{116F}\x{11D9}/g;
    s/\x{F4FB}/\x{1112}\x{116F}\x{11F0}/g;
    s/\x{F4FC}/\x{1112}\x{116F}\x{11F9}/g;
    s/\x{F4FD}/\x{1112}\x{118B}/g;
    s/\x{F4FE}/\x{1112}\x{118B}\x{11AB}/g;
    s/\x{F4FF}/\x{1112}\x{118B}\x{11AF}/g;
    s/\x{F500}/\x{1112}\x{116E}\x{1167}/g;
    s/\x{F501}/\x{1112}\x{118C}/g;
    s/\x{F502}/\x{1112}\x{1171}\x{11F0}/g;
    s/\x{F503}/\x{1112}\x{1171}\x{11F9}/g;
    s/\x{F504}/\x{1112}\x{1172}\x{11D9}/g;
    s/\x{F505}/\x{1112}\x{1172}\x{11E2}/g;
    s/\x{F506}/\x{1112}\x{1172}\x{11F0}/g;
    s/\x{F507}/\x{1112}\x{118E}/g;
    s/\x{F508}/\x{1112}\x{118E}\x{11AB}/g;
    s/\x{F509}/\x{1112}\x{1191}/g;
    s/\x{F50A}/\x{1112}\x{1191}\x{11A8}/g;
    s/\x{F50B}/\x{1112}\x{1191}\x{11AB}/g;
    s/\x{F50C}/\x{1112}\x{1191}\x{11AF}/g;
    s/\x{F50D}/\x{1112}\x{1191}\x{11D9}/g;
    s/\x{F50E}/\x{1112}\x{1191}\x{11BC}/g;
    s/\x{F50F}/\x{1112}\x{1191}\x{11F0}/g;
    s/\x{F510}/\x{1112}\x{1191}\x{11F9}/g;
    s/\x{F511}/\x{1112}\x{1192}/g;
    s/\x{F512}/\x{1112}\x{1192}\x{11AF}/g;
    s/\x{F513}/\x{1112}\x{1192}\x{11BC}/g;
    s/\x{F514}/\x{1112}\x{1194}/g;
    s/\x{F515}/\x{1112}\x{1194}\x{11BC}/g;
    s/\x{F516}/\x{1112}\x{1173}\x{11C3}/g;
    s/\x{F517}/\x{1112}\x{1173}\x{11AE}\x{11C0}/g;
    s/\x{F518}/\x{1112}\x{1173}\x{11D0}/g;
    s/\x{F519}/\x{1112}\x{1173}\x{11D9}/g;
    s/\x{F51A}/\x{1112}\x{1173}\x{11E2}/g;
    s/\x{F51B}/\x{1112}\x{1173}\x{11F0}/g;
    s/\x{F51C}/\x{1112}\x{1173}\x{1161}/g;
    s/\x{F51D}/\x{1112}\x{1173}\x{1169}/g;
    s/\x{F51E}/\x{1112}\x{1195}/g;
    s/\x{F51F}/\x{1112}\x{1196}/g;
    s/\x{F520}/\x{1112}\x{1174}\x{11EB}/g;
    s/\x{F521}/\x{1112}\x{1174}\x{11F0}/g;
    s/\x{F522}/\x{1112}\x{1174}\x{11F9}/g;
    s/\x{F523}/\x{1112}\x{1175}\x{11AB}\x{11AB}/g;
    s/\x{F524}/\x{1112}\x{1175}\x{11D9}/g;
    s/\x{F525}/\x{1112}\x{1175}\x{11DC}/g;
    s/\x{F526}/\x{1112}\x{1175}\x{11DC}\x{11BA}/g;
    s/\x{F527}/\x{1112}\x{1175}\x{11DD}/g;
    s/\x{F528}/\x{1112}\x{1175}\x{11DE}/g;
    s/\x{F529}/\x{1112}\x{1175}\x{11E2}/g;
    s/\x{F52A}/\x{1112}\x{1175}\x{11EE}/g;
    s/\x{F52B}/\x{1112}\x{1175}\x{11F0}/g;
    s/\x{F52C}/\x{1112}\x{1175}\x{11F1}/g;
    s/\x{F52D}/\x{1112}\x{1198}/g;
    s/\x{F52E}/\x{1112}\x{1199}/g;
    s/\x{F52F}/\x{1112}\x{1199}\x{11AB}/g;
    s/\x{F530}/\x{1112}\x{1199}\x{11BC}/g;
    s/\x{F531}/\x{1112}\x{1175}\x{1167}/g;
    s/\x{F532}/\x{1112}\x{1175}\x{1168}/g;
    s/\x{F533}/\x{1112}\x{1175}\x{116D}/g;
    s/\x{F534}/\x{1112}\x{119B}/g;
    s/\x{F535}/\x{1112}\x{1175}\x{1172}/g;
    s/\x{F536}/\x{1112}\x{1175}\x{1172}\x{11AB}/g;
    s/\x{F537}/\x{1112}\x{119E}/g;
    s/\x{F538}/\x{1112}\x{119E}\x{11A8}/g;
    s/\x{F539}/\x{1112}\x{119E}\x{11C3}/g;
    s/\x{F53A}/\x{1112}\x{119E}\x{11AB}/g;
    s/\x{F53B}/\x{1112}\x{119E}\x{11AE}/g;
    s/\x{F53C}/\x{1112}\x{119E}\x{11AF}/g;
    s/\x{F53D}/\x{1112}\x{119E}\x{11B0}/g;
    s/\x{F53E}/\x{1112}\x{119E}\x{11CD}/g;
    s/\x{F53F}/\x{1112}\x{119E}\x{11CE}/g;
    s/\x{F540}/\x{1112}\x{119E}\x{11D0}/g;
    s/\x{F541}/\x{1112}\x{119E}\x{11B1}/g;
    s/\x{F542}/\x{1112}\x{119E}\x{11B3}/g;
    s/\x{F543}/\x{1112}\x{119E}\x{11B6}/g;
    s/\x{F544}/\x{1112}\x{119E}\x{11D9}/g;
    s/\x{F545}/\x{1112}\x{119E}\x{11B7}/g;
    s/\x{F546}/\x{1112}\x{119E}\x{11B8}/g;
    s/\x{F547}/\x{1112}\x{119E}\x{11E6}/g;
    s/\x{F548}/\x{1112}\x{119E}\x{11BA}/g;
    s/\x{F549}/\x{1112}\x{119E}\x{11BC}/g;
    s/\x{F54A}/\x{1112}\x{119E}\x{11F0}/g;
    s/\x{F54B}/\x{1112}\x{119E}\x{11C0}/g;
    s/\x{F54C}/\x{1112}\x{119E}\x{11C2}/g;
    s/\x{F54D}/\x{1112}\x{119E}\x{11F9}/g;
    s/\x{F54E}/\x{1112}\x{119F}/g;
    s/\x{F54F}/\x{1112}\x{119F}\x{11AB}/g;
    s/\x{F550}/\x{1112}\x{11A1}/g;
    s/\x{F551}/\x{1112}\x{11A1}\x{11A8}/g;
    s/\x{F552}/\x{1112}\x{11A1}\x{11AB}/g;
    s/\x{F553}/\x{1112}\x{11A1}\x{11AE}/g;
    s/\x{F554}/\x{1112}\x{11A1}\x{11AF}/g;
    s/\x{F555}/\x{1112}\x{11A1}\x{11B0}/g;
    s/\x{F556}/\x{1112}\x{11A1}\x{11D9}/g;
    s/\x{F557}/\x{1112}\x{11A1}\x{11B7}/g;
    s/\x{F558}/\x{1112}\x{11A1}\x{11B8}/g;
    s/\x{F559}/\x{1112}\x{11A1}\x{11B8}\x{11BD}/g;
    s/\x{F55A}/\x{1112}\x{11A1}\x{11BA}/g;
    s/\x{F55B}/\x{1112}\x{11A1}\x{11BC}/g;
    s/\x{F55C}/\x{1112}\x{11A1}\x{11F0}/g;
    s/\x{F55D}/\x{1112}\x{1109}\x{1163}/g;
    s/\x{F55E}/\x{1112}\x{1109}\x{1163}\x{11BC}/g;
    s/\x{F55F}/\x{1112}\x{1109}\x{1169}/g;
    s/\x{F560}/\x{1112}\x{1109}\x{116A}/g;
    s/\x{F561}/\x{1112}\x{1109}\x{116A}\x{11AB}/g;
    s/\x{F562}/\x{1112}\x{1109}\x{116E}/g;
    s/\x{F563}/\x{1112}\x{1109}\x{116F}/g;
    s/\x{F564}/\x{1112}\x{1109}\x{1170}/g;
    s/\x{F565}/\x{1112}\x{1109}\x{1171}/g;
    s/\x{F566}/\x{1112}\x{1109}\x{1172}/g;
    s/\x{F567}/\x{1112}\x{1109}\x{1172}\x{11BC}/g;
    s/\x{F568}/\x{1112}\x{1109}\x{118E}/g;
    s/\x{F569}/\x{1112}\x{1109}\x{118E}\x{11AB}/g;
    s/\x{F56A}/\x{1112}\x{1109}\x{118F}/g;
    s/\x{F56B}/\x{1112}\x{1109}\x{1190}/g;
    s/\x{F56C}/\x{1112}\x{1109}\x{1194}/g;
    s/\x{F56D}/\x{1112}\x{1109}\x{1194}\x{11AB}/g;
    s/\x{F56E}/\x{1112}\x{1109}\x{1175}/g;
    s/\x{F56F}/\x{1112}\x{1109}\x{1175}\x{11AB}/g;
    s/\x{F570}/\x{1112}\x{1109}\x{1175}\x{11BC}/g;
    s/\x{F571}/\x{1158}\x{1161}/g;
    s/\x{F572}/\x{1158}\x{1161}\x{11A8}/g;
    s/\x{F573}/\x{1158}\x{1161}\x{11AB}/g;
    s/\x{F574}/\x{1158}\x{1161}\x{11AF}/g;
    s/\x{F575}/\x{1158}\x{1161}\x{11D9}/g;
    s/\x{F576}/\x{1158}\x{1161}\x{11B7}/g;
    s/\x{F577}/\x{1158}\x{1161}\x{11E2}/g;
    s/\x{F578}/\x{1158}\x{1161}\x{11B8}/g;
    s/\x{F579}/\x{1158}\x{1161}\x{11E6}/g;
    s/\x{F57A}/\x{1158}\x{1161}\x{11BC}/g;
    s/\x{F57B}/\x{1158}\x{1161}\x{11F0}/g;
    s/\x{F57C}/\x{1158}\x{1162}/g;
    s/\x{F57D}/\x{1158}\x{1162}\x{11BC}/g;
    s/\x{F57E}/\x{1158}\x{1162}\x{11F0}/g;
    s/\x{F57F}/\x{1158}\x{1163}/g;
    s/\x{F580}/\x{1158}\x{1163}\x{11AB}/g;
    s/\x{F581}/\x{1158}\x{1163}\x{11B7}/g;
    s/\x{F582}/\x{1158}\x{1163}\x{11E2}/g;
    s/\x{F583}/\x{1158}\x{1163}\x{11B8}/g;
    s/\x{F584}/\x{1158}\x{1163}\x{11E6}/g;
    s/\x{F585}/\x{1158}\x{1163}\x{11F0}/g;
    s/\x{F586}/\x{1158}\x{1163}\x{11F9}/g;
    s/\x{F587}/\x{1158}\x{1164}/g;
    s/\x{F588}/\x{1158}\x{1165}/g;
    s/\x{F589}/\x{1158}\x{1165}\x{11AB}/g;
    s/\x{F58A}/\x{1158}\x{1165}\x{11AF}/g;
    s/\x{F58B}/\x{1158}\x{1165}\x{11B7}/g;
    s/\x{F58C}/\x{1158}\x{1165}\x{11E2}/g;
    s/\x{F58D}/\x{1158}\x{1165}\x{11F9}/g;
    s/\x{F58E}/\x{1158}\x{1166}/g;
    s/\x{F58F}/\x{1158}\x{1166}\x{11BC}/g;
    s/\x{F590}/\x{1158}\x{1167}/g;
    s/\x{F591}/\x{1158}\x{1167}\x{11A8}/g;
    s/\x{F592}/\x{1158}\x{1167}\x{11AB}/g;
    s/\x{F593}/\x{1158}\x{1167}\x{11AF}/g;
    s/\x{F594}/\x{1158}\x{1167}\x{11D9}/g;
    s/\x{F595}/\x{1158}\x{1167}\x{11B7}/g;
    s/\x{F596}/\x{1158}\x{1167}\x{11B8}/g;
    s/\x{F597}/\x{1158}\x{1167}\x{11F0}/g;
    s/\x{F598}/\x{1158}\x{1167}\x{11F9}/g;
    s/\x{F599}/\x{1158}\x{1168}/g;
    s/\x{F59A}/\x{1158}\x{1168}\x{11BC}/g;
    s/\x{F59B}/\x{1158}\x{1169}/g;
    s/\x{F59C}/\x{1158}\x{1169}\x{11A8}/g;
    s/\x{F59D}/\x{1158}\x{1169}\x{11AB}/g;
    s/\x{F59E}/\x{1158}\x{1169}\x{11AF}/g;
    s/\x{F59F}/\x{1158}\x{1169}\x{11D9}/g;
    s/\x{F5A0}/\x{1158}\x{1169}\x{11B7}/g;
    s/\x{F5A1}/\x{1158}\x{1169}\x{11E2}/g;
    s/\x{F5A2}/\x{1158}\x{1169}\x{11BC}/g;
    s/\x{F5A3}/\x{1158}\x{1169}\x{11F0}/g;
    s/\x{F5A4}/\x{1158}\x{116A}/g;
    s/\x{F5A5}/\x{1158}\x{116A}\x{11A8}/g;
    s/\x{F5A6}/\x{1158}\x{116A}\x{11AB}/g;
    s/\x{F5A7}/\x{1158}\x{116A}\x{11AF}/g;
    s/\x{F5A8}/\x{1158}\x{116A}\x{11D9}/g;
    s/\x{F5A9}/\x{1158}\x{116A}\x{11BC}/g;
    s/\x{F5AA}/\x{1158}\x{116A}\x{11F0}/g;
    s/\x{F5AB}/\x{1158}\x{116A}\x{11F9}/g;
    s/\x{F5AC}/\x{1158}\x{116B}/g;
    s/\x{F5AD}/\x{1158}\x{116B}\x{11BC}/g;
    s/\x{F5AE}/\x{1158}\x{116C}/g;
    s/\x{F5AF}/\x{1158}\x{116C}\x{11A8}/g;
    s/\x{F5B0}/\x{1158}\x{116C}\x{11BC}/g;
    s/\x{F5B1}/\x{1158}\x{116C}\x{11F0}/g;
    s/\x{F5B2}/\x{1158}\x{116D}/g;
    s/\x{F5B3}/\x{1158}\x{116D}\x{11B7}/g;
    s/\x{F5B4}/\x{1158}\x{116D}\x{11E2}/g;
    s/\x{F5B5}/\x{1158}\x{116E}/g;
    s/\x{F5B6}/\x{1158}\x{116E}\x{11AB}/g;
    s/\x{F5B7}/\x{1158}\x{116E}\x{11B7}/g;
    s/\x{F5B8}/\x{1158}\x{116E}\x{11E2}/g;
    s/\x{F5B9}/\x{1158}\x{116E}\x{11B8}/g;
    s/\x{F5BA}/\x{1158}\x{116E}\x{11E6}/g;
    s/\x{F5BB}/\x{1158}\x{116E}\x{11BC}/g;
    s/\x{F5BC}/\x{1158}\x{116E}\x{11F0}/g;
    s/\x{F5BD}/\x{1158}\x{116E}\x{11F9}/g;
    s/\x{F5BE}/\x{1158}\x{116F}/g;
    s/\x{F5BF}/\x{1158}\x{116F}\x{11AB}/g;
    s/\x{F5C0}/\x{1158}\x{116F}\x{11F9}/g;
    s/\x{F5C1}/\x{1158}\x{1171}/g;
    s/\x{F5C2}/\x{1158}\x{1171}\x{11F9}/g;
    s/\x{F5C3}/\x{1158}\x{1172}/g;
    s/\x{F5C4}/\x{1158}\x{1172}\x{11B7}/g;
    s/\x{F5C5}/\x{1158}\x{118F}/g;
    s/\x{F5C6}/\x{1158}\x{118F}\x{11AB}/g;
    s/\x{F5C7}/\x{1158}\x{1191}/g;
    s/\x{F5C8}/\x{1158}\x{1191}\x{11AB}/g;
    s/\x{F5C9}/\x{1158}\x{1191}\x{11AF}/g;
    s/\x{F5CA}/\x{1158}\x{1191}\x{11D9}/g;
    s/\x{F5CB}/\x{1158}\x{1191}\x{11BC}/g;
    s/\x{F5CC}/\x{1158}\x{1191}\x{11F0}/g;
    s/\x{F5CD}/\x{1158}\x{1192}/g;
    s/\x{F5CE}/\x{1158}\x{1192}\x{11BC}/g;
    s/\x{F5CF}/\x{1158}\x{1173}/g;
    s/\x{F5D0}/\x{1158}\x{1173}\x{11A8}/g;
    s/\x{F5D1}/\x{1158}\x{1173}\x{11AB}/g;
    s/\x{F5D2}/\x{1158}\x{1173}\x{11B7}/g;
    s/\x{F5D3}/\x{1158}\x{1173}\x{11E2}/g;
    s/\x{F5D4}/\x{1158}\x{1173}\x{11BC}/g;
    s/\x{F5D5}/\x{1158}\x{1173}\x{11F0}/g;
    s/\x{F5D6}/\x{1158}\x{1195}/g;
    s/\x{F5D7}/\x{1158}\x{1174}/g;
    s/\x{F5D8}/\x{1158}\x{1174}\x{11A8}/g;
    s/\x{F5D9}/\x{1158}\x{1174}\x{11F0}/g;
    s/\x{F5DA}/\x{1158}\x{1174}\x{11C2}/g;
    s/\x{F5DB}/\x{1158}\x{1174}\x{11F9}/g;
    s/\x{F5DC}/\x{1158}\x{1175}/g;
    s/\x{F5DD}/\x{1158}\x{1175}\x{11AB}/g;
    s/\x{F5DE}/\x{1158}\x{1175}\x{11B7}/g;
    s/\x{F5DF}/\x{1158}\x{1175}\x{11E2}/g;
    s/\x{F5E0}/\x{1158}\x{1175}\x{11F0}/g;
    s/\x{F5E1}/\x{1158}\x{1175}\x{11F9}/g;
    s/\x{F5E2}/\x{1158}\x{119E}/g;
    s/\x{F5E3}/\x{1158}\x{119E}\x{11AB}/g;
    s/\x{F5E4}/\x{1158}\x{11A1}/g;
    s/\x{F5E5}/\x{1158}\x{11A1}\x{11A8}/g;
    s/\x{F5E6}/\x{1158}\x{11A1}\x{11BC}/g;
    s/\x{F5E7}/\x{1158}\x{11A1}\x{11F0}/g;
    s/\x{F5E8}/\x{1159}\x{1161}/g;
    s/\x{F5E9}/\x{1159}\x{1161}\x{11A8}/g;
    s/\x{F5EA}/\x{1159}\x{1161}\x{11AB}/g;
    s/\x{F5EB}/\x{1159}\x{1161}\x{11AF}/g;
    s/\x{F5EC}/\x{1159}\x{1161}\x{11D9}/g;
    s/\x{F5ED}/\x{1159}\x{1161}\x{11B7}/g;
    s/\x{F5EE}/\x{1159}\x{1161}\x{11E2}/g;
    s/\x{F5EF}/\x{1159}\x{1161}\x{11B8}/g;
    s/\x{F5F0}/\x{1159}\x{1161}\x{11E6}/g;
    s/\x{F5F1}/\x{1159}\x{1161}\x{11BC}/g;
    s/\x{F5F2}/\x{1159}\x{1161}\x{11F0}/g;
    s/\x{F5F3}/\x{1159}\x{1162}/g;
    s/\x{F5F4}/\x{1159}\x{1162}\x{11BC}/g;
    s/\x{F5F5}/\x{1159}\x{1162}\x{11F0}/g;
    s/\x{F5F6}/\x{1159}\x{1163}/g;
    s/\x{F5F7}/\x{1159}\x{1163}\x{11A8}/g;
    s/\x{F5F8}/\x{1159}\x{1163}\x{11AB}/g;
    s/\x{F5F9}/\x{1159}\x{1163}\x{11B7}/g;
    s/\x{F5FA}/\x{1159}\x{1163}\x{11E2}/g;
    s/\x{F5FB}/\x{1159}\x{1163}\x{11B8}/g;
    s/\x{F5FC}/\x{1159}\x{1163}\x{11E6}/g;
    s/\x{F5FD}/\x{1159}\x{1163}\x{11F0}/g;
    s/\x{F5FE}/\x{1159}\x{1163}\x{11F9}/g;
    s/\x{F5FF}/\x{1159}\x{1164}/g;
    s/\x{F600}/\x{1159}\x{1165}/g;
    s/\x{F601}/\x{1159}\x{1165}\x{11AB}/g;
    s/\x{F602}/\x{1159}\x{1165}\x{11AF}/g;
    s/\x{F603}/\x{1159}\x{1165}\x{11D9}/g;
    s/\x{F604}/\x{1159}\x{1165}\x{11B7}/g;
    s/\x{F605}/\x{1159}\x{1165}\x{11B8}/g;
    s/\x{F606}/\x{1159}\x{1165}\x{11BC}/g;
    s/\x{F607}/\x{1159}\x{1166}/g;
    s/\x{F608}/\x{1159}\x{1166}\x{11BC}/g;
    s/\x{F609}/\x{1159}\x{1167}/g;
    s/\x{F60A}/\x{1159}\x{1167}\x{11A8}/g;
    s/\x{F60B}/\x{1159}\x{1167}\x{11AB}/g;
    s/\x{F60C}/\x{1159}\x{1167}\x{11AF}/g;
    s/\x{F60D}/\x{1159}\x{1167}\x{11D9}/g;
    s/\x{F60E}/\x{1159}\x{1167}\x{11B7}/g;
    s/\x{F60F}/\x{1159}\x{1167}\x{11B8}/g;
    s/\x{F610}/\x{1159}\x{1167}\x{11F0}/g;
    s/\x{F611}/\x{1159}\x{1167}\x{11C2}/g;
    s/\x{F612}/\x{1159}\x{1168}/g;
    s/\x{F613}/\x{1159}\x{1168}\x{11BC}/g;
    s/\x{F614}/\x{1159}\x{1169}/g;
    s/\x{F615}/\x{1159}\x{1169}\x{11A8}/g;
    s/\x{F616}/\x{1159}\x{1169}\x{11AB}/g;
    s/\x{F617}/\x{1159}\x{1169}\x{11B7}/g;
    s/\x{F618}/\x{1159}\x{1169}\x{11E2}/g;
    s/\x{F619}/\x{1159}\x{1169}\x{11BC}/g;
    s/\x{F61A}/\x{1159}\x{1169}\x{11F0}/g;
    s/\x{F61B}/\x{1159}\x{116A}/g;
    s/\x{F61C}/\x{1159}\x{116A}\x{11A8}/g;
    s/\x{F61D}/\x{1159}\x{116A}\x{11AB}/g;
    s/\x{F61E}/\x{1159}\x{116A}\x{11AF}/g;
    s/\x{F61F}/\x{1159}\x{116A}\x{11D9}/g;
    s/\x{F620}/\x{1159}\x{116A}\x{11B8}/g;
    s/\x{F621}/\x{1159}\x{116A}\x{11E6}/g;
    s/\x{F622}/\x{1159}\x{116A}\x{11BC}/g;
    s/\x{F623}/\x{1159}\x{116A}\x{11F0}/g;
    s/\x{F624}/\x{1159}\x{116A}\x{11F9}/g;
    s/\x{F625}/\x{1159}\x{116B}/g;
    s/\x{F626}/\x{1159}\x{116B}\x{11BC}/g;
    s/\x{F627}/\x{1159}\x{116C}/g;
    s/\x{F628}/\x{1159}\x{116C}\x{11A8}/g;
    s/\x{F629}/\x{1159}\x{116C}\x{11BC}/g;
    s/\x{F62A}/\x{1159}\x{116C}\x{11F0}/g;
    s/\x{F62B}/\x{1159}\x{116D}/g;
    s/\x{F62C}/\x{1159}\x{116D}\x{11A8}/g;
    s/\x{F62D}/\x{1159}\x{116D}\x{11B7}/g;
    s/\x{F62E}/\x{1159}\x{116D}\x{11E2}/g;
    s/\x{F62F}/\x{1159}\x{116D}\x{11BC}/g;
    s/\x{F630}/\x{1159}\x{116E}/g;
    s/\x{F631}/\x{1159}\x{116E}\x{11A8}/g;
    s/\x{F632}/\x{1159}\x{116E}\x{11AB}/g;
    s/\x{F633}/\x{1159}\x{116E}\x{11AF}/g;
    s/\x{F634}/\x{1159}\x{116E}\x{11D9}/g;
    s/\x{F635}/\x{1159}\x{116E}\x{11B7}/g;
    s/\x{F636}/\x{1159}\x{116E}\x{11E2}/g;
    s/\x{F637}/\x{1159}\x{116E}\x{11BC}/g;
    s/\x{F638}/\x{1159}\x{116E}\x{11F0}/g;
    s/\x{F639}/\x{1159}\x{116E}\x{11F9}/g;
    s/\x{F63A}/\x{1159}\x{116F}/g;
    s/\x{F63B}/\x{1159}\x{116F}\x{11AB}/g;
    s/\x{F63C}/\x{1159}\x{116F}\x{11AF}/g;
    s/\x{F63D}/\x{1159}\x{116F}\x{11D9}/g;
    s/\x{F63E}/\x{1159}\x{116F}\x{11BC}/g;
    s/\x{F63F}/\x{1159}\x{1171}/g;
    s/\x{F640}/\x{1159}\x{1171}\x{11BC}/g;
    s/\x{F641}/\x{1159}\x{1172}/g;
    s/\x{F642}/\x{1159}\x{1172}\x{11AB}/g;
    s/\x{F643}/\x{1159}\x{1172}\x{11B7}/g;
    s/\x{F644}/\x{1159}\x{1172}\x{11E2}/g;
    s/\x{F645}/\x{1159}\x{1172}\x{11F9}/g;
    s/\x{F646}/\x{1159}\x{1191}/g;
    s/\x{F647}/\x{1159}\x{1191}\x{11A8}/g;
    s/\x{F648}/\x{1159}\x{1191}\x{11AB}/g;
    s/\x{F649}/\x{1159}\x{1191}\x{11AF}/g;
    s/\x{F64A}/\x{1159}\x{1191}\x{11D9}/g;
    s/\x{F64B}/\x{1159}\x{1191}\x{11BC}/g;
    s/\x{F64C}/\x{1159}\x{1191}\x{11F0}/g;
    s/\x{F64D}/\x{1159}\x{1192}/g;
    s/\x{F64E}/\x{1159}\x{1192}\x{11BC}/g;
    s/\x{F64F}/\x{1159}\x{1194}/g;
    s/\x{F650}/\x{1159}\x{1194}\x{11BC}/g;
    s/\x{F651}/\x{1159}\x{1173}/g;
    s/\x{F652}/\x{1159}\x{1173}\x{11A8}/g;
    s/\x{F653}/\x{1159}\x{1173}\x{11AB}/g;
    s/\x{F654}/\x{1159}\x{1173}\x{11B7}/g;
    s/\x{F655}/\x{1159}\x{1173}\x{11E2}/g;
    s/\x{F656}/\x{1159}\x{1173}\x{11B8}/g;
    s/\x{F657}/\x{1159}\x{1173}\x{11F0}/g;
    s/\x{F658}/\x{1159}\x{1173}\x{11F9}/g;
    s/\x{F659}/\x{1159}\x{1174}/g;
    s/\x{F65A}/\x{1159}\x{1174}\x{11AB}/g;
    s/\x{F65B}/\x{1159}\x{1174}\x{11BC}/g;
    s/\x{F65C}/\x{1159}\x{1175}/g;
    s/\x{F65D}/\x{1159}\x{1175}\x{11AB}/g;
    s/\x{F65E}/\x{1159}\x{1175}\x{11AF}/g;
    s/\x{F65F}/\x{1159}\x{1175}\x{11D9}/g;
    s/\x{F660}/\x{1159}\x{1175}\x{11B7}/g;
    s/\x{F661}/\x{1159}\x{1175}\x{11E2}/g;
    s/\x{F662}/\x{1159}\x{1175}\x{11BC}/g;
    s/\x{F663}/\x{1159}\x{1175}\x{11F0}/g;
    s/\x{F664}/\x{1159}\x{1175}\x{11F9}/g;
    s/\x{F665}/\x{1159}\x{119E}/g;
    s/\x{F666}/\x{1159}\x{119E}\x{11AB}/g;
    s/\x{F667}/\x{1159}\x{119E}\x{11B7}/g;
    s/\x{F668}/\x{1159}\x{11A1}/g;
    s/\x{F669}/\x{1159}\x{11A1}\x{11A8}/g;
    s/\x{F66A}/\x{1159}\x{11A1}\x{11AF}/g;
    s/\x{F66B}/\x{1159}\x{11A1}\x{11CD}/g;
    s/\x{F66C}/\x{1159}\x{11A1}\x{11B1}/g;
    s/\x{F66D}/\x{1159}\x{11A1}\x{11BC}/g;
    s/\x{F66E}/\x{1159}\x{11A1}\x{11F0}/g;
    s/\x{F784}/\x{115F}/g;
    s/\x{F785}/\x{1100}/g;
    s/\x{F786}/\x{1101}/g;
    s/\x{F787}/\x{1100}\x{1103}/g;
    s/\x{F788}/\x{1102}/g;
    s/\x{F789}/\x{1113}/g;
    s/\x{F78A}/\x{1114}/g;
    s/\x{F78B}/\x{1115}/g;
    s/\x{F78C}/\x{1116}/g;
    s/\x{F78D}/\x{1102}\x{1109}/g;
    s/\x{F78E}/\x{1102}\x{110C}/g;
    s/\x{F78F}/\x{1102}\x{1112}/g;
    s/\x{F790}/\x{1103}/g;
    s/\x{F791}/\x{1117}/g;
    s/\x{F792}/\x{1104}/g;
    s/\x{F793}/\x{1103}\x{1105}/g;
    s/\x{F794}/\x{1103}\x{1106}/g;
    s/\x{F795}/\x{1103}\x{1107}/g;
    s/\x{F796}/\x{1103}\x{1109}/g;
    s/\x{F797}/\x{1103}\x{110C}/g;
    s/\x{F798}/\x{1105}/g;
    s/\x{F799}/\x{1105}\x{1100}/g;
    s/\x{F79A}/\x{1105}\x{1101}/g;
    s/\x{F79B}/\x{1118}/g;
    s/\x{F79C}/\x{1105}\x{1103}/g;
    s/\x{F79D}/\x{1105}\x{1104}/g;
    s/\x{F79E}/\x{1119}/g;
    s/\x{F79F}/\x{1105}\x{1106}/g;
    s/\x{F7A0}/\x{1105}\x{1107}/g;
    s/\x{F7A1}/\x{1105}\x{1108}/g;
    s/\x{F7A2}/\x{1105}\x{112B}/g;
    s/\x{F7A3}/\x{1105}\x{1109}/g;
    s/\x{F7A4}/\x{1105}\x{110C}/g;
    s/\x{F7A5}/\x{1105}\x{110F}/g;
    s/\x{F7A6}/\x{111A}/g;
    s/\x{F7A7}/\x{111B}/g;
    s/\x{F7A8}/\x{1106}/g;
    s/\x{F7A9}/\x{1106}\x{1100}/g;
    s/\x{F7AA}/\x{1106}\x{1103}/g;
    s/\x{F7AB}/\x{111C}/g;
    s/\x{F7AC}/\x{1106}\x{1109}/g;
    s/\x{F7AD}/\x{111D}/g;
    s/\x{F7AE}/\x{1107}/g;
    s/\x{F7AF}/\x{111E}/g;
    s/\x{F7B0}/\x{111F}/g;
    s/\x{F7B1}/\x{1120}/g;
    s/\x{F7B2}/\x{1108}/g;
    s/\x{F7B3}/\x{1121}/g;
    s/\x{F7B4}/\x{1122}/g;
    s/\x{F7B5}/\x{1123}/g;
    s/\x{F7B6}/\x{1124}/g;
    s/\x{F7B7}/\x{1125}/g;
    s/\x{F7B8}/\x{1126}/g;
    s/\x{F7B9}/\x{1121}\x{1110}/g;
    s/\x{F7BA}/\x{1127}/g;
    s/\x{F7BB}/\x{1128}/g;
    s/\x{F7BC}/\x{1107}\x{110F}/g;
    s/\x{F7BD}/\x{1129}/g;
    s/\x{F7BE}/\x{112A}/g;
    s/\x{F7BF}/\x{1107}\x{1112}/g;
    s/\x{F7C0}/\x{112B}/g;
    s/\x{F7C1}/\x{112C}/g;
    s/\x{F7C2}/\x{1109}/g;
    s/\x{F7C3}/\x{112D}/g;
    s/\x{F7C4}/\x{112E}/g;
    s/\x{F7C5}/\x{112F}/g;
    s/\x{F7C6}/\x{1130}/g;
    s/\x{F7C7}/\x{1131}/g;
    s/\x{F7C8}/\x{1132}/g;
    s/\x{F7C9}/\x{1133}/g;
    s/\x{F7CA}/\x{110A}/g;
    s/\x{F7CB}/\x{1109}\x{1132}/g;
    s/\x{F7CC}/\x{1134}/g;
    s/\x{F7CD}/\x{1135}/g;
    s/\x{F7CE}/\x{1136}/g;
    s/\x{F7CF}/\x{1137}/g;
    s/\x{F7D0}/\x{1138}/g;
    s/\x{F7D1}/\x{1139}/g;
    s/\x{F7D2}/\x{113A}/g;
    s/\x{F7D3}/\x{113B}/g;
    s/\x{F7D4}/\x{113C}/g;
    s/\x{F7D5}/\x{113D}/g;
    s/\x{F7D6}/\x{113E}/g;
    s/\x{F7D7}/\x{113F}/g;
    s/\x{F7D8}/\x{1140}/g;
    s/\x{F7D9}/\x{110B}/g;
    s/\x{F7DA}/\x{1141}/g;
    s/\x{F7DB}/\x{1142}/g;
    s/\x{F7DC}/\x{110B}\x{1105}/g;
    s/\x{F7DD}/\x{1143}/g;
    s/\x{F7DE}/\x{1144}/g;
    s/\x{F7DF}/\x{1145}/g;
    s/\x{F7E0}/\x{1146}/g;
    s/\x{F7E1}/\x{1147}/g;
    s/\x{F7E2}/\x{1148}/g;
    s/\x{F7E3}/\x{1149}/g;
    s/\x{F7E4}/\x{114A}/g;
    s/\x{F7E5}/\x{114B}/g;
    s/\x{F7E6}/\x{110B}\x{1112}/g;
    s/\x{F7E7}/\x{114C}/g;
    s/\x{F7E8}/\x{110C}/g;
    s/\x{F7E9}/\x{114D}/g;
    s/\x{F7EA}/\x{110D}/g;
    s/\x{F7EB}/\x{110D}\x{1112}/g;
    s/\x{F7EC}/\x{114E}/g;
    s/\x{F7ED}/\x{114F}/g;
    s/\x{F7EE}/\x{1150}/g;
    s/\x{F7EF}/\x{1151}/g;
    s/\x{F7F0}/\x{110E}/g;
    s/\x{F7F1}/\x{1152}/g;
    s/\x{F7F2}/\x{1153}/g;
    s/\x{F7F3}/\x{1154}/g;
    s/\x{F7F4}/\x{1155}/g;
    s/\x{F7F5}/\x{110F}/g;
    s/\x{F7F6}/\x{1110}/g;
    s/\x{F7F7}/\x{1110}\x{1110}/g;
    s/\x{F7F8}/\x{1111}/g;
    s/\x{F7F9}/\x{1156}/g;
    s/\x{F7FA}/\x{1111}\x{1112}/g;
    s/\x{F7FB}/\x{1157}/g;
    s/\x{F7FC}/\x{1112}/g;
    s/\x{F7FD}/\x{1112}\x{1109}/g;
    s/\x{F7FE}/\x{1158}/g;
    s/\x{F7FF}/\x{1159}/g;
    s/\x{F800}/\x{1159}\x{1159}/g;
    s/\x{F806}/\x{1160}/g;
    s/\x{F807}/\x{1161}/g;
    s/\x{F808}/\x{1176}/g;
    s/\x{F809}/\x{1177}/g;
    s/\x{F80A}/\x{1161}\x{1173}/g;
    s/\x{F80B}/\x{1162}/g;
    s/\x{F80C}/\x{1163}/g;
    s/\x{F80D}/\x{1178}/g;
    s/\x{F80E}/\x{1179}/g;
    s/\x{F80F}/\x{1163}\x{116E}/g;
    s/\x{F810}/\x{1164}/g;
    s/\x{F811}/\x{1165}/g;
    s/\x{F812}/\x{117A}/g;
    s/\x{F813}/\x{117B}/g;
    s/\x{F814}/\x{117C}/g;
    s/\x{F815}/\x{1166}/g;
    s/\x{F816}/\x{1167}/g;
    s/\x{F817}/\x{1167}\x{1163}/g;
    s/\x{F818}/\x{117D}/g;
    s/\x{F819}/\x{117E}/g;
    s/\x{F81A}/\x{1168}/g;
    s/\x{F81B}/\x{1169}/g;
    s/\x{F81C}/\x{116A}/g;
    s/\x{F81D}/\x{116B}/g;
    s/\x{F81E}/\x{1169}\x{1163}/g;
    s/\x{F81F}/\x{1169}\x{1164}/g;
    s/\x{F820}/\x{117F}/g;
    s/\x{F821}/\x{1180}/g;
    s/\x{F822}/\x{1169}\x{1167}/g;
    s/\x{F823}/\x{1181}/g;
    s/\x{F824}/\x{1182}/g;
    s/\x{F825}/\x{1182}\x{116E}/g;
    s/\x{F826}/\x{1183}/g;
    s/\x{F827}/\x{116C}/g;
    s/\x{F828}/\x{116D}/g;
    s/\x{F829}/\x{116D}\x{1161}/g;
    s/\x{F82A}/\x{116D}\x{1162}/g;
    s/\x{F82B}/\x{1184}/g;
    s/\x{F82C}/\x{1185}/g;
    s/\x{F82D}/\x{116D}\x{1165}/g;
    s/\x{F82E}/\x{1186}/g;
    s/\x{F82F}/\x{1187}/g;
    s/\x{F830}/\x{1188}/g;
    s/\x{F831}/\x{116E}/g;
    s/\x{F832}/\x{1189}/g;
    s/\x{F833}/\x{118A}/g;
    s/\x{F834}/\x{116F}/g;
    s/\x{F835}/\x{118B}/g;
    s/\x{F836}/\x{1170}/g;
    s/\x{F837}/\x{116E}\x{1167}/g;
    s/\x{F838}/\x{118C}/g;
    s/\x{F839}/\x{118D}/g;
    s/\x{F83A}/\x{1171}/g;
    s/\x{F83B}/\x{1171}\x{1175}/g;
    s/\x{F83C}/\x{1172}/g;
    s/\x{F83D}/\x{118E}/g;
    s/\x{F83E}/\x{1172}\x{1162}/g;
    s/\x{F83F}/\x{118F}/g;
    s/\x{F840}/\x{1190}/g;
    s/\x{F841}/\x{1191}/g;
    s/\x{F842}/\x{1192}/g;
    s/\x{F843}/\x{1172}\x{1169}/g;
    s/\x{F844}/\x{1193}/g;
    s/\x{F845}/\x{1194}/g;
    s/\x{F846}/\x{1173}/g;
    s/\x{F847}/\x{1173}\x{1161}/g;
    s/\x{F848}/\x{1173}\x{1165}/g;
    s/\x{F849}/\x{1173}\x{1166}/g;
    s/\x{F84A}/\x{1173}\x{1169}/g;
    s/\x{F84B}/\x{1195}/g;
    s/\x{F84C}/\x{1196}/g;
    s/\x{F84D}/\x{1174}/g;
    s/\x{F84E}/\x{1197}/g;
    s/\x{F84F}/\x{1175}/g;
    s/\x{F850}/\x{1198}/g;
    s/\x{F851}/\x{1199}/g;
    s/\x{F852}/\x{1175}\x{1178}/g;
    s/\x{F853}/\x{1175}\x{1164}/g;
    s/\x{F854}/\x{1175}\x{1167}/g;
    s/\x{F855}/\x{1175}\x{1168}/g;
    s/\x{F856}/\x{119A}/g;
    s/\x{F857}/\x{1175}\x{1183}/g;
    s/\x{F858}/\x{1175}\x{116D}/g;
    s/\x{F859}/\x{119B}/g;
    s/\x{F85A}/\x{1175}\x{1172}/g;
    s/\x{F85B}/\x{119C}/g;
    s/\x{F85C}/\x{1175}\x{1175}/g;
    s/\x{F85D}/\x{119D}/g;
    s/\x{F85E}/\x{119E}/g;
    s/\x{F85F}/\x{119E}\x{1161}/g;
    s/\x{F860}/\x{119F}/g;
    s/\x{F861}/\x{119E}\x{1166}/g;
    s/\x{F862}/\x{11A0}/g;
    s/\x{F863}/\x{11A1}/g;
    s/\x{F864}/\x{11A2}/g;
    s/\x{F86B}/\x{11A8}/g;
    s/\x{F86C}/\x{11A9}/g;
    s/\x{F86D}/\x{11A8}\x{11AB}/g;
    s/\x{F86E}/\x{11C3}/g;
    s/\x{F86F}/\x{11A8}\x{11B8}/g;
    s/\x{F870}/\x{11AA}/g;
    s/\x{F871}/\x{11C4}/g;
    s/\x{F872}/\x{11A8}\x{11BE}/g;
    s/\x{F873}/\x{11A8}\x{11BF}/g;
    s/\x{F874}/\x{11A8}\x{11C2}/g;
    s/\x{F875}/\x{11AB}/g;
    s/\x{F876}/\x{11C5}/g;
    s/\x{F877}/\x{11AB}\x{11AB}/g;
    s/\x{F878}/\x{11C6}/g;
    s/\x{F879}/\x{11AB}\x{11AF}/g;
    s/\x{F87A}/\x{11C7}/g;
    s/\x{F87B}/\x{11C8}/g;
    s/\x{F87C}/\x{11AC}/g;
    s/\x{F87D}/\x{11AB}\x{11BE}/g;
    s/\x{F87E}/\x{11C9}/g;
    s/\x{F87F}/\x{11AD}/g;
    s/\x{F880}/\x{11AE}/g;
    s/\x{F881}/\x{11CA}/g;
    s/\x{F882}/\x{11AE}\x{11AE}/g;
    s/\x{F883}/\x{11AE}\x{11AE}\x{11B8}/g;
    s/\x{F884}/\x{11CB}/g;
    s/\x{F885}/\x{11AE}\x{11B8}/g;
    s/\x{F886}/\x{11AE}\x{11BA}/g;
    s/\x{F887}/\x{11AE}\x{11E7}/g;
    s/\x{F888}/\x{11AE}\x{11BD}/g;
    s/\x{F889}/\x{11AE}\x{11BE}/g;
    s/\x{F88A}/\x{11AE}\x{11C0}/g;
    s/\x{F88B}/\x{11AF}/g;
    s/\x{F88C}/\x{11B0}/g;
    s/\x{F88D}/\x{11AF}\x{11A9}/g;
    s/\x{F88E}/\x{11CC}/g;
    s/\x{F88F}/\x{11B0}\x{11C2}/g;
    s/\x{F890}/\x{11CD}/g;
    s/\x{F891}/\x{11CE}/g;
    s/\x{F892}/\x{11CF}/g;
    s/\x{F893}/\x{11D0}/g;
    s/\x{F894}/\x{11D0}\x{11BF}/g;
    s/\x{F895}/\x{11B1}/g;
    s/\x{F896}/\x{11D1}/g;
    s/\x{F897}/\x{11D2}/g;
    s/\x{F898}/\x{11B1}\x{11C2}/g;
    s/\x{F899}/\x{11B2}/g;
    s/\x{F89A}/\x{11B2}\x{11AE}/g;
    s/\x{F89B}/\x{11D3}/g;
    s/\x{F89C}/\x{11B2}\x{11C1}/g;
    s/\x{F89D}/\x{11D4}/g;
    s/\x{F89E}/\x{11D5}/g;
    s/\x{F89F}/\x{11B3}/g;
    s/\x{F8A0}/\x{11D6}/g;
    s/\x{F8A1}/\x{11D7}/g;
    s/\x{F8A2}/\x{11AF}\x{11F0}/g;
    s/\x{F8A3}/\x{11D8}/g;
    s/\x{F8A4}/\x{11B4}/g;
    s/\x{F8A5}/\x{11B5}/g;
    s/\x{F8A6}/\x{11B6}/g;
    s/\x{F8A7}/\x{11D9}/g;
    s/\x{F8A8}/\x{11D9}\x{11C2}/g;
    s/\x{F8A9}/\x{11AF}\x{11BC}/g;
    s/\x{F8AA}/\x{11B7}/g;
    s/\x{F8AB}/\x{11DA}/g;
    s/\x{F8AC}/\x{11B7}\x{11AB}/g;
    s/\x{F8AD}/\x{11B7}\x{11AB}\x{11AB}/g;
    s/\x{F8AE}/\x{11DB}/g;
    s/\x{F8AF}/\x{11B7}\x{11B7}/g;
    s/\x{F8B0}/\x{11DC}/g;
    s/\x{F8B1}/\x{11DC}\x{11BA}/g;
    s/\x{F8B2}/\x{11DD}/g;
    s/\x{F8B3}/\x{11DE}/g;
    s/\x{F8B4}/\x{11DF}/g;
    s/\x{F8B5}/\x{11B7}\x{11BD}/g;
    s/\x{F8B6}/\x{11E0}/g;
    s/\x{F8B7}/\x{11E1}/g;
    s/\x{F8B8}/\x{11E2}/g;
    s/\x{F8B9}/\x{11B8}/g;
    s/\x{F8BA}/\x{11B8}\x{11AE}/g;
    s/\x{F8BB}/\x{11E3}/g;
    s/\x{F8BC}/\x{11B8}\x{11B5}/g;
    s/\x{F8BD}/\x{11B8}\x{11B7}/g;
    s/\x{F8BE}/\x{11B8}\x{11B8}/g;
    s/\x{F8BF}/\x{11B9}/g;
    s/\x{F8C0}/\x{11B9}\x{11AE}/g;
    s/\x{F8C1}/\x{11B8}\x{11BD}/g;
    s/\x{F8C2}/\x{11B8}\x{11BE}/g;
    s/\x{F8C3}/\x{11E4}/g;
    s/\x{F8C4}/\x{11E5}/g;
    s/\x{F8C5}/\x{11E6}/g;
    s/\x{F8C6}/\x{11BA}/g;
    s/\x{F8C7}/\x{11E7}/g;
    s/\x{F8C8}/\x{11E8}/g;
    s/\x{F8C9}/\x{11E9}/g;
    s/\x{F8CA}/\x{11BA}\x{11B7}/g;
    s/\x{F8CB}/\x{11EA}/g;
    s/\x{F8CC}/\x{11BA}\x{11E6}/g;
    s/\x{F8CD}/\x{11BB}/g;
    s/\x{F8CE}/\x{11BA}\x{11E7}/g;
    s/\x{F8CF}/\x{11BA}\x{11E8}/g;
    s/\x{F8D0}/\x{11BA}\x{11EB}/g;
    s/\x{F8D1}/\x{11BA}\x{11BD}/g;
    s/\x{F8D2}/\x{11BA}\x{11BE}/g;
    s/\x{F8D3}/\x{11BA}\x{11C0}/g;
    s/\x{F8D4}/\x{11BA}\x{11C2}/g;
    s/\x{F8D5}/\x{11EB}/g;
    s/\x{F8D6}/\x{11EB}\x{11B8}/g;
    s/\x{F8D7}/\x{11EB}\x{11E6}/g;
    s/\x{F8D8}/\x{11BC}/g;
    s/\x{F8D9}/\x{11EC}/g;
    s/\x{F8DA}/\x{11ED}/g;
    s/\x{F8DB}/\x{11BC}\x{11B7}/g;
    s/\x{F8DC}/\x{11BC}\x{11BA}/g;
    s/\x{F8DD}/\x{11EE}/g;
    s/\x{F8DE}/\x{11EF}/g;
    s/\x{F8DF}/\x{11BC}\x{11C2}/g;
    s/\x{F8E0}/\x{11F0}/g;
    s/\x{F8E1}/\x{11F0}\x{11A8}/g;
    s/\x{F8E2}/\x{11F1}/g;
    s/\x{F8E3}/\x{11F2}/g;
    s/\x{F8E4}/\x{11F0}\x{11BF}/g;
    s/\x{F8E5}/\x{11F0}\x{11C2}/g;
    s/\x{F8E6}/\x{11BD}/g;
    s/\x{F8E7}/\x{11BD}\x{11B8}/g;
    s/\x{F8E8}/\x{11BD}\x{11B8}\x{11B8}/g;
    s/\x{F8E9}/\x{11BD}\x{11BD}/g;
    s/\x{F8EA}/\x{11BE}/g;
    s/\x{F8EB}/\x{11BF}/g;
    s/\x{F8EC}/\x{11C0}/g;
    s/\x{F8ED}/\x{11C1}/g;
    s/\x{F8EE}/\x{11F3}/g;
    s/\x{F8EF}/\x{11C1}\x{11BA}/g;
    s/\x{F8F0}/\x{11C1}\x{11C0}/g;
    s/\x{F8F1}/\x{11F4}/g;
    s/\x{F8F2}/\x{11C2}/g;
    s/\x{F8F3}/\x{11F5}/g;
    s/\x{F8F4}/\x{11F6}/g;
    s/\x{F8F5}/\x{11F7}/g;
    s/\x{F8F6}/\x{11F8}/g;
    s/\x{F8F7}/\x{11F9}/g;
    s/\x{F86A}//g;
}
