/*******************************************************************************************************************************************
 nservices.h
 Tools namespace :
 template <class T> class TBuffer; generic buffer.
 template <class T, T Min, T Max> class TSet; traitement d'un {ensemble} de valeurs comprises entre Min et Max.
 template <class K, class V> class TKeyBuffer; generic key / value association
 template <class T> class TNode; hierarchical pointers.
*******************************************************************************************************************************************/

#ifndef __NSERVICES_H__
#define __NSERVICES_H__

#include <stdio.h>
#include <stdarg.h>

//------------------------------------------------------------------------------------------------------------------------------------------
// simple types
//------------------------------------------------------------------------------------------------------------------------------------------
typedef char			SInt8;
typedef unsigned char		UInt8;
typedef short			SInt16;
typedef unsigned short		UInt16;
typedef long			SInt32;
typedef unsigned long		UInt32;
typedef long long		SInt64;
typedef unsigned long long	UInt64;
typedef float			Float32;
typedef double			Float64;
typedef void *			PVoid;
typedef bool			Bool;

const static SInt8		SINT8MIN	=0x80;
const static SInt8		SINT8MAX	=0x7F;
const static UInt8		UINT8MIN	=0x00;
const static UInt8		UINT8MAX	=0xFF;
const static SInt16		SINT16MIN	=0x8000;
const static SInt16		SINT16MAX	=0x7FFF;
const static UInt16		UINT16MIN	=0x0000;
const static UInt16		UINT16MAX	=0xFFFF;
const static SInt32		SINT32MIN	=0x80000000;
const static SInt32		SINT32MAX	=0x7FFFFFFF;
const static UInt32		UINT32MIN	=0x00000000;
const static UInt32		UINT32MAX	=0xFFFFFFFF;
/*
const static SInt64		SINT64MIN	=0x8000000000000000;
const static SInt64		SINT64MAX	=0x7FFFFFFFFFFFFFFF;
const static UInt64		UINT64MIN	=0x0000000000000000;
const static UInt64		UINT64MAX	=0xFFFFFFFFFFFFFFFF;
*/


//------------------------------------------------------------------------------------------------------------------------------------------
// macros, commodity 
//------------------------------------------------------------------------------------------------------------------------------------------
#ifndef min
#define min(x,y) ((x) < (y)) ? (x) : (y)
#endif
#ifndef max
#define max(x,y) ((x) > (y)) ? (x) : (y)
#endif

//------------------------------------------------------------------------------------------------------------------------------------------
// macros, handle and derived handles definition
//------------------------------------------------------------------------------------------------------------------------------------------
#define DEFINE_HANDLE(name)      typedef struct __##name##Handle__ { long unused; }; 		    typedef __##name##Handle__ * name;
#define DERIVE_HANDLE(base,name) typedef struct __##name##Handle__ : public __##base##Handle__ { }; typedef __##name##Handle__ * name;

// namespace
namespace NServices
{
	//----------------------------------------------------------------------------------------------------------------------------------
	// TBuffer template
	//----------------------------------------------------------------------------------------------------------------------------------
	template <class T> class TBuffer
	{
		// instanciation section
		public :

			TBuffer			(const T *inBuffer=NULL, const size_t inLength=0);
			TBuffer			(const size_t inLength, const T inFirst, ... );
			TBuffer			(const TBuffer &inBuffer);
			virtual ~TBuffer	();

		// general functions
		public :

			size_t			GetLength	() const;
			T *			Get		(const size_t inIndex=0) const;
			
			bool			Insert		(const T &inItem, const size_t toIndex);
			bool			Insert		(const TBuffer &inBuffer, const size_t toIndex);
			bool			InsertEmpty	(const size_t toIndex);
			
			void			Delete		(const T &inItem);
			void			Delete		(const TBuffer &inBuffer);
			void			Delete		(const size_t inIndex, const size_t inLength);

			bool			Find		(const T &inItem, const size_t inStart=0, size_t *outIndex=NULL) const;
			bool			Find		(const TBuffer &inBuffer, const size_t inStart=0, size_t *outIndex=NULL) const;
			
			void			Sort		();

			void			Reset		();

		// operators
		public :

			T *			operator []	(const size_t) const;

			bool 			operator ==	(const TBuffer &) const;
			bool 			operator !=	(const TBuffer &) const;

			TBuffer &		operator =	(const TBuffer &);
			
			TBuffer			operator +	(const T &) const;
			TBuffer			operator +	(const TBuffer &) const;
			TBuffer &		operator +=	(const T &);
			TBuffer &		operator +=	(const TBuffer &);
			
			TBuffer			operator -	(const T &) const;
			TBuffer			operator -	(const TBuffer &) const;
			TBuffer &		operator -=	(const T &);
			TBuffer &		operator -=	(const TBuffer &);

		// attributes section
		protected :

			size_t			Length;
			T *			Buffer;
	};
	
	//----------------------------------------------------------------------------------------------------------------------------------
	// TSet template
	//----------------------------------------------------------------------------------------------------------------------------------
	template <class T, T min, T max> class TSet
	{
		// instanciation section
		public :

			TSet			(const TBuffer <T> &inBuffer=TBuffer <T> ());
			TSet			(const TSet &);
			virtual ~TSet		();

		// general functions
		public :

			TBuffer <T>		Get		() const;

			bool			Contains	(const T) const;
			bool			Contains	(const TSet &) const;

			void			Reset		();

		// operators
		public :

			bool			operator []	(const T) const;
			bool			operator []	(const TSet &) const;

			bool			operator ==	(const TSet &) const;
			bool			operator !=	(const TSet &) const;

			TSet &			operator =	(const TSet &);
			
			TSet &			operator +=	(const T);
			TSet &			operator +=	(const TSet &);
			TSet &			operator -=	(const T);
			TSet &			operator -=	(const TSet &);
			TSet &			operator *=	(const T);
			TSet &			operator *=	(const TSet &);

			TSet			operator +	(const T) const;
			TSet 			operator +	(const TSet &) const;
			TSet			operator -	(const T) const;
			TSet 			operator -	(const TSet &) const;
			TSet			operator *	(const T) const;
			TSet 			operator *	(const TSet &) const;
			
		// protected section
		protected :

			TBuffer <T>		Set;
	};

	//----------------------------------------------------------------------------------------------------------------------------------
	// TKeyBuffer template
	//----------------------------------------------------------------------------------------------------------------------------------
	template <class K, class V> class TKeyBuffer
	{
		// instanciation
		public :

			TKeyBuffer		(const TBuffer <K> &inKeys=TBuffer <K> (), const TBuffer <V> &inValues=TBuffer <V> ());
			TKeyBuffer		(const TKeyBuffer &);
			virtual ~TKeyBuffer	();

		// operators
		public :

			TBuffer <K>		GetKeys		() const;
			TBuffer <V>		GetValues	() const;

			bool			Contains	(const K &) const;

			void			Reset		();

			V &			operator []	(const K &);
			TKeyBuffer &		operator -=	(const K &);

		// protected section
		protected :

			TBuffer <K>		Keys;
			TBuffer <V>		Values;
	};

	//----------------------------------------------------------------------------------------------------------------------------------
	// TNode template 
	//----------------------------------------------------------------------------------------------------------------------------------
	template <class T> class TNode
	{
		// instanciation section
		public :

			TNode			(TNode * =NULL, T * =NULL, const bool =true);
			TNode			(const TNode &);
			virtual ~TNode		();

		// protected fuctions
		public :

			TNode *			GetParent	() const;
			bool			SetParent	(TNode *);
			TNode *			RemoveParent	();
			
			TBuffer <TNode <T> *>	GetChildren	() const;
			bool			InsertChild	(T *, const size_t);
			bool			InsertChild	(TNode *, const size_t);
			
			T *			GetValue	() const;
			void			SetValue	(T *);
			T *			RemoveValue	();
			
			TNode *			Find		(const T *) const;
			
			TBuffer <TNode <T> *>	GetTree		() const;

			void			Delete		();

		// protected section
		protected :

			TNode *		 	Parent;
			TBuffer <TNode <T> *>	Children;
			T *			Value;
			bool			AutoDelete;

		public :

			bool			Destructor;
	};
	
	//----------------------------------------------------------------------------------------------------------------------------------
	// templates code resolution
	//----------------------------------------------------------------------------------------------------------------------------------
	
	template <class T> TBuffer <T> :: TBuffer (const T *inBuffer, const size_t inLength)
					: Buffer  (NULL),
					  Length  (0)
	{
		if (inBuffer != NULL && inLength > 0)
		{
			Length = inLength;
			Buffer = new T [Length];
			for (size_t i=Length; i>0; i--) Buffer[i-1] = inBuffer[i-1];
		}
	}

	template <class T> TBuffer <T> :: TBuffer (const size_t inLength, const T inFirst, ... )
					: Buffer  (NULL),
					  Length  (0)
	{
		if (inLength > 0)
		{
			Length = inLength;
			Buffer = new T [Length];
			Buffer[0] = inFirst;
			va_list args;
			va_start (args, inFirst);
			for (size_t i=1; i<Length; i++) Buffer[i] = va_arg (args, T);
			va_end (args);
		}
 	}

	template <class T> TBuffer <T> :: TBuffer (const TBuffer &inBuffer)
					: Buffer  (NULL),
					  Length  (0)
	{
		if (&inBuffer != this && inBuffer.Buffer != NULL)
		{
			Length = inBuffer.Length;
			Buffer = new T [Length];
			for (size_t i=Length; i>0; i--) Buffer[i-1] = inBuffer.Buffer[i-1];
		}
	}

	template <class T> TBuffer <T> :: ~TBuffer ()
	{
		if (Buffer != NULL) delete [] Buffer;
	}

	template <class T> size_t TBuffer <T> :: GetLength () const
	{
		return Length;
	}

	template <class T> void TBuffer <T> :: Reset ()
	{
		if (Buffer != NULL) delete [] Buffer;
		Buffer = NULL;
		Length = 0;
	}
	
	template <class T> T * TBuffer <T> :: Get (const size_t inIndex) const
	{
		return (inIndex < Length) ? &Buffer[inIndex] : NULL;
	}

	template <class T> bool TBuffer <T> :: Insert (const T &inItem, const size_t inIndex)
	{
		if (inIndex <= Length)
		{
			T *newBuffer = new T [++Length];
			for (size_t i=0; i<inIndex; i++) newBuffer[i] = Buffer[i];
			newBuffer[inIndex] = inItem;
			for (size_t i=inIndex+1; i<Length; i++) newBuffer[i] = Buffer[i-1];
			if (Buffer != NULL) delete [] Buffer;
			Buffer = newBuffer;
			return true;
		}
		return false;
	}

	template <class T> bool TBuffer <T> :: Insert (const TBuffer &inBuffer, const size_t inIndex)
	{
		if (&inBuffer != this && inIndex <= Length && inBuffer.Buffer != NULL)
		{
			Length += inBuffer.Length;
			T *newBuffer = new T [Length];
			size_t i, j, k;
			for (i=0, k=0; i<inIndex; i++, k++) newBuffer[k] = Buffer[i];
			for (j=0; j<inBuffer.Length; j++, k++) newBuffer[k] = inBuffer.Buffer[j];
			for (; k<Length; i++, k++) newBuffer[k] = Buffer[i];
			if (Buffer != NULL) delete [] Buffer;
			Buffer = newBuffer;
			return true;
		}
		return false;
	}

	template <class T> bool TBuffer <T> :: InsertEmpty (const size_t inIndex)
	{
		if (inIndex <= Length)
		{
			T *newBuffer = new T [++Length];
			for (size_t i=0; i<inIndex; i++) newBuffer[i] = Buffer[i];
			for (size_t i=inIndex+1; i<Length; i++) newBuffer[i] = Buffer[i-1];
			if (Buffer != NULL) delete [] Buffer;
			Buffer = newBuffer;
			return true;
		}
		return false;
	}

	template <class T> void TBuffer <T> :: Delete (const T &inItem)
	{
		size_t inIndex;
		while (Find (inItem, 0, &inIndex))
		{
			T *newBuffer = new T [Length-1];
			for (size_t i=0; i<inIndex; i++) newBuffer[i] = Buffer[i];
			for (size_t i=inIndex+1; i<Length; i++) newBuffer[i-1] = Buffer[i];
			Length--;
			delete [] Buffer;
			Buffer = newBuffer;
		}
	}

	template <class T> void TBuffer <T> :: Delete (const TBuffer &inBuffer)
	{
		if (&inBuffer == this) return;
		size_t inIndex;
		while (Find (inBuffer, 0, &inIndex))
		{
			T *newBuffer = new T [Length-inBuffer.Length];
			for (size_t i=0; i<inIndex; i++) newBuffer[i] = Buffer[i];
			for (size_t i=inIndex+inBuffer.Length; i<Length; i++) newBuffer[i-inBuffer.Length] = Buffer[i];
			delete [] Buffer;
			Length -= inBuffer.Length;
			Buffer = newBuffer;
		}
	}

	template <class T> void TBuffer <T> :: Delete (const size_t inStartIndex, const size_t inLength)
	{
		if (inStartIndex < Length && (inStartIndex+inLength) <= Length)
		{
			T *newBuffer = new T [Length-inLength];
			for (size_t i=0; i<inStartIndex; i++) newBuffer[i] = Buffer[i];
			for (size_t i=inStartIndex+inLength; i<Length; i++) newBuffer[i-inLength] = Buffer[i];
			delete [] Buffer;
			Buffer = newBuffer;
			Length -= inLength;
		}
	}

	template <class T> bool TBuffer <T> :: Find (const T &inItem, const size_t inStart, size_t *outIndex) const
	{
		if (Buffer == NULL || Length == 0) return false;
		for (size_t i=inStart; i<Length; i++)
		{
			if (Buffer[i] == inItem)
			{
				if (outIndex != NULL) *outIndex = i;
				return true;
			}
		}
		return false;
	}
	
	template <class T> bool TBuffer <T> :: Find (const TBuffer &inBuffer, const size_t inStart, size_t *outIndex) const
	{
		if (&inBuffer == this || Buffer == NULL || inBuffer.Buffer == NULL || Length == 0 || inBuffer.Length == 0) return false;
		for (size_t i=inStart; i<Length; i++)
		{
			bool found = true;
			for (size_t j=0, k=i; j<inBuffer.Length && found; j++, k++)
				if (Buffer[k] != inBuffer.Buffer[j]) found = false;
			if (found)
			{
				if (outIndex != NULL) *outIndex = i;
				return true;
			}
		}
		return false;
	}

	template <class T> void TBuffer <T> :: Sort ()
	{
		for (size_t i=0; i<Length-1; i++)
		{
			for (size_t j=i+1; j<Length; j++)
			{
				if (Buffer[j] < Buffer[i])
				{
					T tmp = Buffer[i];
					Buffer[i] = Buffer[j];
					Buffer[j] = tmp;
				}
			}
		}
	}
	
	template <class T> T * TBuffer <T> :: operator [] (const size_t inIndex) const
	{
		return (inIndex < Length) ? &Buffer[inIndex] : NULL;
	}

	template <class T> bool TBuffer <T> :: operator == (const TBuffer &inBuffer) const
	{
		if (&inBuffer == this) return true;
		if (Length != inBuffer.Length) return false;
		for (size_t i=Length; i>0; i--)
			if (Buffer[i-1] != inBuffer.Buffer[i])
				return false;
		return true;
	}
	
	template <class T> bool TBuffer <T> :: operator != (const TBuffer &inBuffer) const
	{
		return !operator == (inBuffer);
	}

	template <class T> TBuffer <T> & TBuffer <T> :: operator = (const TBuffer &inBuffer)
	{
		if (&inBuffer == this) return *this;
		if (Buffer != NULL) delete [] Buffer;
		Buffer = NULL;
		Length = 0;
		if (inBuffer.Buffer != NULL)
		{
			Length = inBuffer.Length;
			Buffer = new T [Length];
			for (size_t i=Length; i>0; i--) Buffer[i-1] = inBuffer.Buffer[i-1];
		}
		return *this;
	}
			
	template <class T> TBuffer <T> TBuffer <T> :: operator + (const T &inItem) const
	{
		TBuffer <T> outBuffer (*this);
		return outBuffer += inItem;
	}
	
	template <class T> TBuffer <T> TBuffer <T> :: operator + (const TBuffer &inBuffer) const
	{
		TBuffer <T> outBuffer (*this);
		return outBuffer += inBuffer;
	}
	
	template <class T> TBuffer <T> & TBuffer <T> :: operator += (const T & inItem)
	{
		Insert (inItem, Length);
		return *this;
	}
	
	template <class T> TBuffer <T> & TBuffer <T> :: operator += (const TBuffer &inBuffer)
	{
		Insert (inBuffer, Length);
		return *this;
	}
			
	template <class T> TBuffer <T> TBuffer <T> :: operator - (const T &inItem) const
	{
		TBuffer <T> outBuffer (*this);
		return outBuffer -= inItem;
	}
	
	template <class T> TBuffer <T> TBuffer <T> :: operator - (const TBuffer &inBuffer) const
	{
		TBuffer <T> outBuffer (*this);
		return outBuffer -= inBuffer;
	}
	
	template <class T> TBuffer <T> & TBuffer <T> :: operator -= (const T &inItem)
	{
		Delete (inItem);
		return *this;
	}
	
	template <class T> TBuffer <T> & TBuffer <T> :: operator -= (const TBuffer &inBuffer)
	{
		Delete (inBuffer);
		return *this;
	}

	template <class T, T min, T max> TSet <T, min, max> :: TSet (const TBuffer <T> &inBuffer)
	{
		for (size_t i=inBuffer.GetLength(), j=0; i>0; i--, j++)
			if (*inBuffer[j] >= min && *inBuffer[j] <= max && !Contains(*inBuffer[j]))
				Set += *inBuffer[j];
	}
	
	template <class T, T min, T max> TSet <T, min, max> :: TSet (const TSet &inSet)
							     : Set  (inSet.Set)
	{ }
	
	template <class T, T min, T max> TSet <T, min, max> :: ~TSet ()
	{ }

	template <class T, T min, T max> void TSet <T, min, max> :: Reset ()
	{
		Set.Reset();
	}
	
	template <class T, T min, T max> TBuffer <T> TSet <T, min, max> :: Get () const
	{
		return Set;
	}

	template <class T, T min, T max> bool TSet <T, min, max> :: Contains (const T inItem) const
	{
		for (size_t i=Set.GetLength(); i>0; i--)
			if (*Set[i-1] == inItem)
				return true;
		return false;
	}

	template <class T, T min, T max> bool TSet <T, min, max> :: Contains (const TSet &inSet) const
	{
		for (size_t i=inSet.Set.GetLength(); i>0; i--)
			if (!Contains (*inSet.Set[i-1]))
				return false;
		return true;
	}

	template <class T, T min, T max> bool TSet <T, min, max> :: operator []	(const T inValue) const
	{
		return Contains (inValue);
	}

	template <class T, T min, T max> bool TSet <T, min, max> :: operator [] (const TSet &inSet) const
	{
		return Contains (inSet);
	}
	
	template <class T, T min, T max> bool TSet <T, min, max> :: operator ==	(const TSet &inSet) const
	{
		if (Set.GetLength() != inSet.Set.GetLength()) return false;
		for (size_t i=inSet.Set.GetLength(); i>0; i--)
			if (!Contains (*inSet.Set[i-1]))
				return false;
		return true;
	}
	
	template <class T, T min, T max> bool TSet <T, min, max> :: operator !=	(const TSet &inSet) const
	{
		return ! operator == (inSet);
	}

	template <class T, T min, T max> TSet <T, min, max> & TSet <T, min, max> :: operator = (const TSet &inSet)
	{
		Set = inSet.Set;
		return *this;
	}
			
	template <class T, T min, T max> TSet <T, min, max> & TSet <T, min, max> :: operator += (const T inItem)
	{
		if (inItem >= min && inItem <= max && !Contains (inItem)) Set += inItem;
		return *this;
	}
	
	template <class T, T min, T max> TSet <T, min, max> & TSet <T, min, max> :: operator += (const TSet &inSet)
	{
		for (size_t i=inSet.Set.GetLength(), j=0; i>0; i--, j++)
			if (!Contains (*inSet.Set[j]))
				Set += *inSet.Set[j];
		return *this;
	}
	
	template <class T, T min, T max> TSet <T, min, max> & TSet <T, min, max> :: operator -= (const T inItem)
	{
		Set -= inItem;
		return *this;
	}
	
	template <class T, T min, T max> TSet <T, min, max> & TSet <T, min, max> :: operator -= (const TSet &inSet)
	{
		Set -= inSet.Set;
		return *this;
	}
	
	template <class T, T min, T max> TSet <T, min, max> & TSet <T, min, max> :: operator *= (const T inItem)
	{
		if (Contains (inItem)) Set = inItem;
		return *this;
	}
	
	template <class T, T min, T max> TSet <T, min, max> & TSet <T, min, max> :: operator *= (const TSet &inSet)
	{
		TSet <T, min, max> theSet;
		for (size_t i=Set.GetLength(), j=0; i>0; i--, j++)
			if (inSet.Contains (*Set[j]))
				theSet += *Set[j];
		for (size_t i=inSet.Set.GetLength(), j=0; i>0; i--, j++)
			if (Set.Contains (*inSet.Set[j]))
				theSet += *inSet.Set[j];
		Set = theSet.Set;
		return *this;
	}

	template <class T, T min, T max> TSet <T, min, max> TSet <T, min, max> :: operator + (const T inItem) const
	{
		return TSet <T, min, max> (*this) += inItem;
	}
	
	template <class T, T min, T max> TSet <T, min, max> TSet <T, min, max> :: operator + (const TSet &inSet) const
	{
		return TSet <T, min, max> (*this) += inSet;
	}
	
	template <class T, T min, T max> TSet <T, min, max> TSet <T, min, max> :: operator - (const T inItem) const
	{
		return TSet <T, min, max> (*this) -= inItem;
	}
	
	template <class T, T min, T max> TSet <T, min, max> TSet <T, min, max> :: operator - (const TSet &inSet) const
	{
		return TSet <T, min, max> (*this) -= inSet;
	}
			
	template <class T, T min, T max> TSet <T, min, max> TSet <T, min, max> :: operator * (const T inItem) const
	{
		return TSet <T, min, max> (*this) *= inItem;
	}
	
	template <class T, T min, T max> TSet <T, min, max> TSet <T, min, max> :: operator * (const TSet &inSet) const
	{
		return TSet <T, min, max> (*this) *= inSet;
	}

	template <class K, class V> TKeyBuffer <K, V> :: TKeyBuffer(const TBuffer <K> &inKeys, const TBuffer <V> &inValues)
	{
		Keys   = inKeys;
		Values = inValues;
	}
	
	template <class K, class V> TKeyBuffer <K, V> :: TKeyBuffer (const TKeyBuffer &inKey)
	{
		Keys   = inKey.Keys;
		Values = inKey.Values; 
	}

	template <class K, class V> TKeyBuffer <K, V> :: ~TKeyBuffer ()
	{ }

	template <class K, class V> void TKeyBuffer <K, V> :: Reset ()
	{
		Keys.Reset();
		Values.Reset();
	}

	template <class K, class V> V & TKeyBuffer <K, V> :: operator [] (const K &inKey)
	{
		size_t i;
		if (Keys.Find (inKey, 0, &i))
		{
			return *Values[i];
		}
		else
		{
			Keys += inKey;
			Values.InsertEmpty (Values.GetLength());
			return *Values[Values.GetLength()-1];
		}
	}

	template <class K, class V> TKeyBuffer <K, V> & TKeyBuffer <K, V> :: operator -= (const K &inKey)
	{
		size_t i;
		if (Keys.Find (inKey, 0, &i))
		{
			Keys.  Delete (i, 1);
			Values.Delete (i, 1);
		}
	}

	template <class K, class V> TBuffer <K> TKeyBuffer <K, V> :: GetKeys () const
	{
		return Keys;
	}

	template <class K, class V> TBuffer <V> TKeyBuffer <K, V> :: GetValues () const
	{
		return Values;
	}

	template <class K, class V> bool TKeyBuffer <K, V> :: Contains (const K &inKey) const
	{
		size_t i;
		return Keys.Find (inKey, 0, &i);
	}

	template <class T> TNode <T> :: TNode 	   (TNode *inParent, T *inValue, const bool inAutoDelete)
				      :	Parent 	   (inParent),
					Value	   (inValue),
					Children   (),
					AutoDelete (inAutoDelete)
	{
	       if (Parent != NULL) Parent -> Children += this;	
	}

	template <class T> TNode <T> :: TNode 	   (const TNode &inNode)
				      : Parent	   (NULL),
					Value	   (new T (*inNode.Value)),
					Children   (NULL),
					AutoDelete (true),
					Destructor (false)
	{
		for (size_t i=inNode.Children.GetLength(), j=0; i>0; i--, j++)
		{
			TNode <T> *newNode = new TNode <T> (**inNode.Children[j]);
			newNode -> Parent = this;
			Children += newNode;
		}
	}

	template <class T> TNode <T> :: ~TNode ()
	{
		Destructor = true;
		Delete ();
	}
	
	template <class T> TNode <T> * TNode <T> :: GetParent () const
	{
		return Parent;
	}	

	template <class T> bool TNode <T> :: SetParent (TNode *inNode)
	{
		if (Parent == inNode || inNode == this) return false;
		if (Parent != NULL) Parent -> Children -= this;
		Parent = inNode;
		if (Parent != NULL) Parent -> Children += this;
		return true;
	}

	template <class T> TNode <T> * TNode <T> :: RemoveParent ()
	{
		TNode <T> *outParent = Parent;
		if (Parent != NULL) Parent -> Children -= this;
		Parent = NULL;
		return outParent;
	}

	template <class T> TBuffer <TNode <T> *> TNode <T> :: GetChildren () const
	{
		return Children;
	}

	template <class T> bool TNode <T> :: InsertChild (T *inValue, const size_t inIndex)
	{
		TNode <T> *newNode = new TNode <T> (NULL, inValue, AutoDelete);
		if (!Children.Insert (newNode, inIndex))
		{
			delete newNode;
			return false;
		}
		newNode -> Parent = this;
		return true;
	}

	template <class T> bool TNode <T> :: InsertChild (TNode *inNode, const size_t inIndex)
	{
		if (inNode -> Parent == this) return false;
		if (!Children.Insert (inNode, inIndex)) return false;
		if (inNode -> Parent != NULL) inNode -> Parent -> Children -= inNode; 
		inNode -> Parent = this;
		return true;
	}

	template <class T> TNode <T> * TNode <T> :: Find (const T *inValue) const
	{
		if (inValue == Value) return const_cast <TNode <T> *> (this);
		for (size_t i=Children.GetLength(); i>0; i--)
		{
			TNode <T> *inChild = (*Children[i-1]) -> Find (inValue);
			if (inChild != NULL) return inChild;
		}
		return NULL;
	}

	template <class T> T * TNode <T> :: GetValue () const
	{
		return Value;
	}

	template <class T> void TNode <T> :: SetValue (T *inValue)
	{
		if (inValue == Value) return;
		if (Value != NULL && AutoDelete) delete Value;
		Value = inValue;
	}

	template <class T> T * TNode <T> :: RemoveValue ()
	{
		T *outValue = Value;
		Value = NULL;
		return outValue;
	}

	template <class T> TBuffer <TNode <T> *> TNode <T> :: GetTree () const
	{
		TBuffer <TNode <T> *> outBuffer;
		outBuffer += const_cast <TNode <T> *> (this);
		for (size_t i=Children.GetLength(), j=0; i>0; i--, j++) outBuffer += (*Children[j]) -> GetTree ();
		return outBuffer;
	}

	template <class T> void TNode <T> :: Delete ()
	{
		while (Children.GetLength() > 0) delete *Children[0];
		if (AutoDelete && Value != NULL) delete Value;
		if (Parent != NULL) Parent -> Children -= this;
		Parent = NULL;
	}
};

#endif
