/*
 * Copyright (c) 2001, 2002 The XDoclet team
 * All rights reserved.
 */
package xdoclet.modules.web;

import xdoclet.XDocletException;
import xdoclet.XDocletMessages;

import xdoclet.XmlSubTask;
import xdoclet.util.Translator;

/**
 * Generates taglib.tld deployment descriptor for JSP taglibs.
 *
 * @author               Ara Abrahamian (ara_e@email.com)
 * @created              July 12, 2001
 * @ant.element          display-name="JSP Taglib" name="jsptaglib" parent="xdoclet.modules.web.WebDocletTask"
 * @version              $Revision: 1.15 $
 * @xdoclet.merge-file   file="taglib-settings.xml" relates-to="taglib.tld" description="Unparsed XML entity containing
 *      (tlib-version, jsp-version, short-name, uri?, display-name?, small-icon?, large-icon?, description?) elements."
 * @xdoclet.merge-file   file="taglib-validator.xml" relates-to="taglib.tld" description="XML document containing the
 *      validator element, used to include a validator which isn't processed by XDoclet."
 * @xdoclet.merge-file   file="taglib-listener.xml" relates-to="taglib.tld" description="Unparsed XML entity containing
 *      additional listener elements, for including listeners not processed by XDoclet."
 * @xdoclet.merge-file   file="taglib-tag.xml" relates-to="taglib.tld" description="Unparsed XML entity containing
 *      additional tag elements, for including tags not processed by XDoclet."
 */
public class JspTaglibSubTask extends XmlSubTask
{
    private static String DEFAULT_TEMPLATE_FILE = "resources/taglib_tld.xdt";

    private static String TLD_PUBLICID_2_0 = "http://java.sun.com/xml/ns/j2ee";
    // xmlns attribute

    private static String TLD_PUBLICID_1_2 = "-//Sun Microsystems, Inc.//DTD JSP Tag Library 1.2//EN";

    private static String TLD_PUBLICID_1_1 = "-//Sun Microsystems, Inc.//DTD JSP Tag Library 1.1//EN";

    private static String TLD_SYSTEMID_2_0 = "http://java.sun.com/xml/ns/j2ee web-jsptaglibrary_2_0.xsd";

    private static String TLD_SYSTEMID_1_2 = "http://java.sun.com/dtd/web-jsptaglibrary_1_2.dtd";

    private static String TLD_SYSTEMID_1_1 = "http://java.sun.com/j2ee/dtds/web-jsptaglibrary_1_1.dtd";

    private static String TLD_XSD_FILE_NAME_2_0 = "resources/web-jsptaglibrary_2_0.xsd";

    private static String TLD_DTD_FILE_NAME_1_2 = "resources/web-jsptaglibrary_1_2.dtd";

    private static String TLD_DTD_FILE_NAME_1_1 = "resources/web-jsptaglibrary_1_1.dtd";

    protected String taglibversion = "1.0";

    protected String jspversion = JspVersionTypes.VERSION_1_2;

    protected String shortname = "";

    protected String uri = "";

    protected String displayname = "";

    protected String smallicon = "";

    protected String largeicon = "";

    protected String description = "";

    protected String filename = "taglib.tld";

    /**
     * Describe what the JspTaglibSubTask constructor does
     */
    public JspTaglibSubTask()
    {
        setTemplateURL(getClass().getResource(DEFAULT_TEMPLATE_FILE));
        setDestinationFile(filename);
    }

    /**
     * Gets the Jspversion attribute of the JspTaglibSubTask object
     *
     * @return   The Jspversion value
     */
    public String getJspversion()
    {
        return jspversion;
    }

    /**
     * Gets the Taglibversion attribute of the JspTaglibSubTask object
     *
     * @return   The Taglibversion value
     */
    public String getTaglibversion()
    {
        return taglibversion;
    }

    /**
     * Gets the Shortname attribute of the JspTaglibSubTask object
     *
     * @return   The Shortname value
     */
    public String getShortname()
    {
        return shortname;
    }

    /**
     * Gets the Uri attribute of the JspTaglibSubTask object
     *
     * @return   The Uri value
     */
    public String getUri()
    {
        return uri;
    }

    /**
     * Gets the Displayname attribute of the JspTaglibSubTask object
     *
     * @return   The Displayname value
     */
    public String getDisplayname()
    {
        return displayname;
    }

    /**
     * Gets the Smallicon attribute of the JspTaglibSubTask object
     *
     * @return   The Smallicon value
     */
    public String getSmallicon()
    {
        return smallicon;
    }

    /**
     * Gets the Largeicon attribute of the JspTaglibSubTask object
     *
     * @return   The Largeicon value
     */
    public String getLargeicon()
    {
        return largeicon;
    }

    /**
     * Gets the Description attribute of the JspTaglibSubTask object
     *
     * @return   The Description value
     */
    public String getDescription()
    {
        return description;
    }

    /**
     * Gets the Filename attribute of the JspTaglibSubTask object
     *
     * @return   The Filename value
     */
    public String getFilename()
    {
        return filename;
    }

    /**
     * The JSP spec version to which the tag library conforms.
     *
     * @param jspversion  The new Jspversion value
     */
    public void setJspversion(JspVersionTypes jspversion)
    {
        this.jspversion = jspversion.getValue();
    }

    /**
     * The tag library implementation version.
     *
     * @param taglibversion  The new Taglibversion value
     */
    public void setTaglibversion(String taglibversion)
    {
        this.taglibversion = taglibversion;
    }

    /**
     * A simple default short name that could be used by a JSP authoring tool to create names with a mnemonic value. For
     * example, it may be used as the prefered prefix value in taglib directives.
     *
     * @param shortname  The new Shortname value
     */
    public void setShortname(String shortname)
    {
        this.shortname = shortname;
    }

    /**
     * A URI uniquely identifying this tag library.
     *
     * @param uri  The new Uri value
     */
    public void setUri(String uri)
    {
        this.uri = uri;
    }

    /**
     * A short name that is intended to be displayed by tools.
     *
     * @param new_display_name  The new Displayname value
     */
    public void setDisplayname(String new_display_name)
    {
        displayname = new_display_name;
    }

    /**
     * Optional small-icon that can be used by tools.
     *
     * @param new_icon  The new Smallicon value
     */
    public void setSmallicon(String new_icon)
    {
        smallicon = new_icon;
    }

    /**
     * Optional large-icon that can be used by tools.
     *
     * @param new_icon  The new Largeicon value
     */
    public void setLargeicon(String new_icon)
    {
        largeicon = new_icon;
    }

    /**
     * A simple string describing the use of this tag library, Should be user discernable.
     *
     * @param new_description  The new Description value
     */
    public void setDescription(String new_description)
    {
        description = new_description;
    }

    /**
     * Specifies the filename of the generated tag library descriptor.
     *
     * @param new_filename  The new Filename value
     */
    public void setFilename(String new_filename)
    {
        filename = new_filename;
        setDestinationFile(filename);
    }

    /**
     * Called to validate configuration parameters.
     *
     * @exception XDocletException
     */
    public void validateOptions() throws XDocletException
    {
        super.validateOptions();

        if (getShortname() == null || getShortname().trim().equals("")) {
            throw new XDocletException(Translator.getString(XDocletMessages.class, XDocletMessages.PARAMETER_MISSING_OR_EMPTY, new String[]{"shortName"}));
        }
    }

    /**
     * Describe what the method does
     *
     * @exception XDocletException
     */
    public void execute() throws XDocletException
    {
        if (getJspversion().equals(JspVersionTypes.VERSION_2_0)) {
            setPublicId(TLD_PUBLICID_2_0);
            // xmlns attribute
            setSystemId(TLD_SYSTEMID_2_0);
            // xsi:schemaLocation
            setSchema(TLD_XSD_FILE_NAME_2_0);
        }
        else if (getJspversion().equals(JspVersionTypes.VERSION_1_1)) {
            setPublicId(TLD_PUBLICID_1_1);
            setSystemId(TLD_SYSTEMID_1_1);
            setDtdURL(getClass().getResource(TLD_DTD_FILE_NAME_1_1));
        }
        else {
            setPublicId(TLD_PUBLICID_1_2);
            setSystemId(TLD_SYSTEMID_1_2);
            setDtdURL(getClass().getResource(TLD_DTD_FILE_NAME_1_2));
        }

        startProcess();
    }

    /**
     * Describe what the method does
     *
     * @exception XDocletException
     */
    protected void engineStarted() throws XDocletException
    {
        System.out.println(Translator.getString(XDocletMessages.class, XDocletMessages.GENERATING_SOMETHING, new String[]{getDestinationFile().toString()}));
    }

    /**
     * Supported JSP spec versions.
     *
     * @author    Ara Abrahamian (ara_e@email.com)
     * @created   July 19, 2001
     */
    public static class JspVersionTypes extends org.apache.tools.ant.types.EnumeratedAttribute
    {
        public final static String VERSION_1_1 = "1.1";
        public final static String VERSION_1_2 = "1.2";
        public final static String VERSION_2_0 = "2.0";

        // We're getting there!

        /**
         * Gets the Values attribute of the JspVersionTypes object
         *
         * @return   The Values value
         */
        public String[] getValues()
        {
            return (new String[]{VERSION_1_1, VERSION_1_2, VERSION_2_0});
        }
    }

}
