# -------------------------------------------------------------------------
#     Copyright (C) 2005-2011 Martin Strohalm <www.mmass.org>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 3 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE.TXT in the
#     main directory of the program
# -------------------------------------------------------------------------

# load libs
import copy
import numpy
from numpy.linalg import solve as solveLinEq

# load stopper
from stopper import CHECK_FORCE_QUIT

# register essential objects and modules
import objects
import blocks


# BASIC CONSTANTS
# ---------------

ELECTRON_MASS = 0.00054857990924
ISOTOPE_DISTANCE = 1.00287
AVERAGE_AMINO = {'C':4.9384, 'H':7.7583, 'N':1.3577, 'O':1.4773, 'S':0.0417}
AVERAGE_BASE = {'C':9.75, 'H':12.25, 'N':3.75, 'O':6, 'P':1}


# BASIC FUNCTIONS
# ---------------

def delta(measuredMass, countedMass, units='ppm'):
    """Calculate error between measuredMass and countedMass in specified units.
        measuredMass: (float)
        countedMass: (float)
        units: ('Da' or 'ppm')
    """
    
    if units == 'ppm':
        return (measuredMass - countedMass)/countedMass*1000000
    elif units == 'Da':
        return (measuredMass - countedMass)
    elif units == '%':
        return (measuredMass - countedMass)/countedMass*100
    else:
        raise ValueError, 'Unknown units for delta mass! -->' + units
# ----


def mz(mass, charge, currentCharge=0, agentFormula='H', agentCharge=1, massType=0):
    """Calculate m/z value for given mass and charge.
        mass: (tuple of (Mo,Av) or float)
        charge: (int) desired charge of ion
        currentCharge: (int) if mass is charged already
        agentFormula: (str or formula) charging agent formula
        agentCharge: (int) charging agent charge
        massType: (0 or 1) used mass type if mass value is float, 0 = monoisotopic, 1 = average
    """
    
    # check agent formula
    if agentFormula != 'e' and not isinstance(agentFormula, objects.compound):
        agentFormula = objects.compound(agentFormula)
    
    # get agent mass
    if agentFormula == 'e':
        agentMass = [ELECTRON_MASS, ELECTRON_MASS]
    else:
        agentMass = agentFormula.mass()
        agentMass = (agentMass[0]-agentCharge*ELECTRON_MASS, agentMass[1]-agentCharge*ELECTRON_MASS)
    
    # recalculate zero charge
    agentCount = currentCharge/agentCharge
    if currentCharge != 0:
        if type(mass) in (tuple, list):
            massMo = mass[0]*abs(currentCharge) - agentMass[0]*agentCount
            massAv = mass[1]*abs(currentCharge) - agentMass[1]*agentCount
            mass = (massMo, massAv)
        else:
            mass = mass*abs(currentCharge) - agentMass[massType]*agentCount
    if charge == 0:
        return mass
    
    # calculate final charge
    agentCount = charge/agentCharge
    if type(mass) in (tuple, list):
        massMo = (mass[0] + agentMass[0]*agentCount)/abs(charge)
        massAv = (mass[1] + agentMass[1]*agentCount)/abs(charge)
        return (massMo, massAv)
    else:
        return (mass + agentMass[massType]*agentCount)/abs(charge)
# ----



# ISOTOPIC PATTERN
# ----------------

def pattern(compound, fwhm=0.1, threshold=0.01, charge=0, agentFormula='H', agentCharge=1):
    """Calculate isotopic pattern for given compound.
        fwhm: (float) gaussian peak width
        threshold: (float) relative intensity threshold in %/100
        charge: (int) charge to be calculated
        agentFormula: (str or formula) charging agent formula
        agentCharge: (int) charging agent charge
    """
    
    # check compound
    if not isinstance(compound, objects.compound):
        compound = objects.compound(compound)
    
    # check agent formula
    if agentFormula != 'e' and not isinstance(agentFormula, objects.compound):
        agentFormula = objects.compound(agentFormula)
    
    # add charging agent to compound
    if charge and agentFormula != 'e':
        formula = compound.formula()
        for atom, count in agentFormula.composition().items():
            formula += '%s%d' % (atom, count*(charge/agentCharge))
        compound = objects.compound(formula)
    
    # get composition
    composition = compound.composition()
    for atom in composition:
        if composition[atom] < 0:
            raise ValueError, 'Pattern cannot be calculated for this formula! --> ' + compound.formula()
    
    # set internal thresholds
    internalThreshold = threshold/100.
    groupingWindow = fwhm/4.
    
    # calculate pattern
    finalPattern = []
    for atom in composition:
        
        # get isotopic profile for current atom or specified isotope only
        atomCount = composition[atom]
        atomPattern = []
        match = objects.elementPattern.match(atom)
        symbol, massNumber, tmp = match.groups()
        if massNumber:
            isotope = blocks.elements[symbol].isotopes[int(massNumber)]
            atomPattern.append([isotope[0], 1.]) # [mass, abundance]
        else:
            for massNumber, isotope in blocks.elements[atom].isotopes.items():
                if isotope[1] > 0.:
                    atomPattern.append(list(isotope)) # [mass, abundance]
        
        # add atoms
        for i in range(atomCount):
            
            CHECK_FORCE_QUIT()
            
            # if pattern is empty (first atom) add current atom pattern
            if not finalPattern:
                finalPattern = normalize(atomPattern)
                continue
            
            # add atom to each peak of final pattern
            currentPattern = []
            for patternIsotope in finalPattern:
                
                # skip peak under relevant abundance threshold
                if patternIsotope[1] < internalThreshold:
                    continue
                
                # add each isotope of current atom to peak
                for atomIsotope in atomPattern:
                    mass = patternIsotope[0] + atomIsotope[0]
                    abundance = patternIsotope[1] * atomIsotope[1]
                    currentPattern.append([mass, abundance])
            
            # group isotopes and normalize pattern
            finalPattern = _groupIsotopes(currentPattern, groupingWindow)
            finalPattern = normalize(finalPattern)
    
    # correct charge
    if charge:
        for i in range(len(finalPattern)):
            finalPattern[i][0] = (finalPattern[i][0]-ELECTRON_MASS*charge)/abs(charge)
    
    # group isotopes and normalize pattern
    finalPattern = _groupIsotopes(finalPattern, groupingWindow)
    finalPattern = normalize(finalPattern)
    
    # discard peaks below threshold
    filteredPeaks = []
    for peak in finalPattern:
        if peak[1] >= threshold:
            filteredPeaks.append(peak)
    finalPattern = filteredPeaks
    
    return finalPattern
# ----


def profile(peaklist, fwhm=0.1, points=10, noise=None, raster=None, forceFwhm=False):
    """Make profile spectrum for given peaklist.
        peaklist: (list of (mz, intensity) or mspy.peaklist) peaklist
        fwhm: (float) default peak fwhm
        points: (int) default number of points per peak width (not used if mzRaster is specified)
        noise: (float) random noise width
        raster: (1D array) m/z values raster
        forceFwhm: (bool) use default fwhm for all peaks
    """
    
    # check peaklist
    if not isinstance(peaklist, objects.peaklist):
        peaklist = objects.peaklist(peaklist)
    
    # get fwhm range
    minFwhm = None
    maxFwhm = None
    if not forceFwhm:
        for peak in peaklist:
            if not peak.fwhm:
                continue
            if not minFwhm or peak.fwhm < minFwhm:
                minFwhm = peak.fwhm
            if not maxFwhm or peak.fwhm > maxFwhm:
                maxFwhm = peak.fwhm
    
    # use default fwhm range if not set
    if not minFwhm or not maxFwhm:
        minFwhm = fwhm
        maxFwhm = fwhm
    
    # get m/z raster
    if raster == None:
        mzRange = (peaklist[0].mz - 5*maxFwhm , peaklist[-1].mz + 5*maxFwhm)
        rasterRange = (minFwhm/points, maxFwhm/points)
        raster = _makeRaster(mzRange, rasterRange)
    
    # get intensity raster
    intensities = numpy.zeros(raster.size, float)
    
    # calulate gaussian peak for each isotope
    for peak in peaklist:
        
        CHECK_FORCE_QUIT()
        
        # get peak fwhm
        if peak.fwhm and not forceFwhm:
            peakFwhm = peak.fwhm
            peakWidth = peak.fwhm/1.66
        else:
            peakFwhm = fwhm
            peakWidth = fwhm/1.66
        
        # calulate peak
        i1 = findIndex(raster, (peak.mz-5*peakFwhm), dim=1)
        i2 = findIndex(raster, (peak.mz+5*peakFwhm), dim=1)
        for i in xrange(i1, i2):
            intensities[i] += peak.intensity*numpy.exp(-1*(pow(raster[i]-peak.mz,2))/pow(peakWidth,2))
    
    # add random noise
    if noise:
        intensities += numpy.random.uniform(-noise/2, noise/2, raster.size)
    
    # make final profile
    raster.shape = (-1,1)
    intensities.shape = (-1,1)
    data = numpy.concatenate((raster, intensities), axis=1)
    data = data.copy()
    
    # add baseline
    baselineData = [[peaklist[0].mz, -peaklist[0].base]]
    for peak in peaklist[1:]:
        if baselineData[-1][0] == peak.mz:
            baselineData[-1][0] -= peak.base
        else:
            baselineData.append([peak.mz, -peak.base])
    data = correctBaseline(points=data, baselineData=baselineData)
    
    return data
# ----


def gaussian(mz, ai, base=0.0, fwhm=0.1, points=500):
    """Make Gaussian peak.
        mz: (float) peak m/z value
        ai: (float) peak ai value
        base: (float) peak baseline value
        fwhm: (float) peak fwhm value
        points: (int) number of points
    """
    
    data = []
    
    minX = mz - (5*fwhm)
    maxX = mz + (5*fwhm)
    step = (maxX - minX) / points
    width = fwhm/1.66
    x = minX
    intensity = ai - base
    
    for i in range(points):
        y = intensity*numpy.exp(-1*(pow(x-mz,2))/pow(width,2)) + base
        data.append([x,y])
        x += step
    
    return numpy.array(data)
# ----


def averagine(mass, charge=0, composition=AVERAGE_AMINO):
    """Calculate average formula for given mass and building block composition.
        mass: (float) neutral mass to be modeled
        charge: (int) charge to be calculated
        composition: (dict) building block composition
    """
    
    # get average mass of block
    blockMass = 0
    for element in composition:
        blockMass += blocks.elements[element].mass[1] * composition[element]
    
    # get block count
    neutralMass = mz(mass, charge=0, currentCharge=charge, massType=1)
    count = max(1, neutralMass/blockMass)
    
    # make formula
    formula = ''
    for element in composition:
        formula += '%s%d' % (element, int(composition[element]*count))
    formula = objects.compound(formula)
    
    # add some hydrogens
    hydrogens = int(round((neutralMass - formula.mass(1)) / blocks.elements['H'].mass[1]))
    hydrogens = max(hydrogens, -1*formula.count('H'))
    formula += 'H%d' % hydrogens
    
    return formula
# ----


def _groupIsotopes(isotopes, window):
    """Group peaks within specified window.
        isotopes: (list of [mass, abundance]) isotopes list
        window: (float) grouping window
    """
    
    isotopes.sort()
    
    buff = []
    buff.append(isotopes[0])
    
    for current in isotopes[1:]:
        previous = buff[-1]
        if (previous[0] + window) > current[0]:
            abundance = previous[1] + current[1]
            mass = (previous[0]*previous[1] + current[0]*current[1]) / abundance
            buff[-1] = [mass, abundance]
        else:
            buff.append(current)
    
    return buff
# ----


def _makeRaster(mzRange, rasterRange):
    """Make m/z raster as linear gradient for given m/z range and edge points differences."""
    
    m = (rasterRange[1] - rasterRange[0]) / (mzRange[1] - mzRange[0])
    b = rasterRange[0] - m * mzRange[0]
    
    size = ((mzRange[1] - mzRange[0]) / rasterRange[0]) + 2
    raster = numpy.zeros(int(size), float)
    
    i = 0
    x = mzRange[0]
    while x <= mzRange[1]:
        raster[i] = x
        x += m*x + b
        i += 1
    
    return raster[:i].copy()
# ----



# SPECTRUM PROCESSING
# -------------------

def noise(points, minX=None, maxX=None, mz=None, window=0.1):
    """Return noise level and width for given data points.
        points: (numpy.array) spectrum points
        minX, maxX: (float or None) points selection
        mz: (float or None) m/z value for which to calculate the noise +- window
        window: (float) points range for noise calculation in %/100, relative to given m/z
    """
    
    # use relevant-portion of the data
    if window == None:
        pass
    elif mz != None:
        window = mz*window
        i1 = findIndex(points, mz-window, dim=2)
        i2 = findIndex(points, mz+window, dim=2)
        points = points[i1:i2]
    elif minX != None and maxX != None:
        i1 = findIndex(points, minX, dim=2)
        i2 = findIndex(points, maxX, dim=2)
    
    # check indexes
    if i1 == i2:
        return 0.0, 1.0
    else:
        points = points[i1:i2]
    
    # unpack data
    x,y = numpy.hsplit(points,2)
    y = y.flatten()
    
    # get noise offset
    noiseLevel = numpy.median(y)
    
    # get noise width
    noiseWidth = numpy.median(numpy.absolute(y - noiseLevel))
    noiseWidth = float(noiseWidth)*2
    
    return noiseLevel, noiseWidth
# ----


def baseline(points, window=0.1, smooth=True, offset=0.):
    """Return baseline data.
        points: (numpy.array)
        window: (float or None) noise calculation window in %/100
        smooth: (bool) smooth final baseline
        offset: (float) global intensity offset in %/100
    """
    
    # single segment baseline
    if window == None:
        noiseLevel, noiseWidth = noise(points)
        noiseLevel -= noiseWidth*offset
        return numpy.array([[points[0][0], noiseLevel, noiseWidth],[points[-1][0], noiseLevel, noiseWidth]])
    
    # make raster
    raster = []
    minimum = max(0, points[0][0])
    x = points[-1][0]
    while x > minimum:
        raster.append(x)
        x -= max(50, x*window)
    raster.append(minimum)
    raster.sort()
    
    # get intensities
    mzArr, intArr = numpy.hsplit(points, 2)
    intArr.flatten()
    
    # make baseline
    levels = []
    widths = []
    for x in raster:
        i1 = findIndex(points, x-x*window, dim=2)
        i2 = findIndex(points, x+x*window, dim=2)
        if i1 == i2:
            noiseLevel = 0.0
            noiseWidth = 1.0
        else:
            noiseLevel = numpy.median(intArr[i1:i2])
            noiseWidth = abs(float(numpy.median(numpy.absolute(intArr[i1:i2] - noiseLevel)))*2)
        levels.append([x, noiseLevel])
        widths.append([x, noiseWidth])
    
    # smooth baseline
    if smooth:
        window = 5 * window * (points[-1][0] - points[0][0])
        levels = smoothSG(numpy.array(levels), window, 2)
        widths = smoothGA(numpy.array(widths), window, 2)
    
    # ensure there are positive values in widths
    for i in range(len(widths)):
        widths[i][1] = abs(widths[i][1])
    
    # apply relative offset and add widths to baseline
    buff = []
    for i, x in enumerate(raster):
        buff.append([x, max(0, levels[i][1]-widths[i][1]*offset), widths[i][1]])
    
    return numpy.array(buff)
# ----


def smoothMA(points, window, cycles=1, style='flat'):
    """Return data points smoothed by moving average.
        points: (numpy.array) points to be smoothed
        window: (float) m/z window size for smoothing
        cycles: (int) number of repeating cycles
    """
    
    # approximate number of points within window
    window = int(window*len(points)/(points[-1][0]-points[0][0]))
    window = min(window, len(points))
    if window < 3:
        return points.copy()
    if not window % 2:
        window -= 1
    
    # unpack mz and intensity
    xAxis, yAxis = numpy.hsplit(points,2)
    xAxis = xAxis.flatten()
    yAxis = yAxis.flatten()
    
    # smooth the points
    while cycles:
        
        CHECK_FORCE_QUIT()
        
        if style == 'flat':
            w = numpy.ones(window,'f')
        elif style == 'gaussian':
            r = numpy.array([(i-(window-1)/2.) for i in range(window)])
            w = numpy.exp(-(r**2/(window/4.)**2))
        else:
            w = eval('numpy.'+style+'(window)')
        
        s = numpy.r_[yAxis[window-1:0:-1], yAxis, yAxis[-2:-window-1:-1]]
        y = numpy.convolve(w/w.sum(), s, mode='same')
        yAxis = y[window-1:-window+1]
        cycles -=1
    
    # return smoothed scan
    xAxis.shape = (-1,1)
    yAxis.shape = (-1,1)
    data = numpy.concatenate((xAxis,yAxis), axis=1)
    
    return data.copy()
# ----


def smoothGA(points, window, cycles=1):
    """Return data points smoothed by Gaussian filter.
        points: (numpy.array) points to be smoothed
        window: (float) m/z window size for smoothing
        cycles: (int) number of repeating cycles
    """
    
    return smoothMA(
        points = points,
        window = window,
        cycles = cycles,
        style = 'gaussian'
    )
# ----


def smoothSG(points, window, cycles=1, order=3):
    """Return data points smoothed by Savitzky-Golay filter.
        points: (numpy.array) points to be smoothed
        window: (float) m/z window size for smoothing
        cycles: (int) number of repeating cycles
        order: (int) order of polynom used
    """
    
    # approximate number of points within window
    window = int(window*len(points)/(points[-1][0]-points[0][0]))
    if window <= order:
        return points.copy()
    
    # unpack axes
    xAxis, yAxis = numpy.hsplit(points,2)
    yAxis = yAxis.flatten()
    
    # coeficients
    orderRange = range(order+1)
    halfWindow = (window-1) // 2
    b = numpy.mat([[k**i for i in orderRange] for k in range(-halfWindow, halfWindow+1)])
    m = numpy.linalg.pinv(b).A[0]
    window = len(m)
    halfWindow = (window-1) // 2
    
    # precompute the offset values for better performance
    offsets = range(-halfWindow, halfWindow+1)
    offsetData = zip(offsets, m)
    
    # smooth the data
    while cycles:
        smoothData = list()
        
        yAxis = numpy.concatenate((numpy.zeros(halfWindow)+yAxis[0], yAxis, numpy.zeros(halfWindow)+yAxis[-1]))
        for i in range(halfWindow, len(yAxis) - halfWindow):
            
            CHECK_FORCE_QUIT()
            
            value = 0.0
            for offset, weight in offsetData:
                value += weight * yAxis[i + offset]
            smoothData.append(value)
        
        yAxis = smoothData
        cycles -=1
    
    # return smoothed data
    yAxis = numpy.array(yAxis)
    yAxis.shape = (-1,1)
    data = numpy.concatenate((xAxis,yAxis), axis=1)
    
    return data.copy()
# ----


def peakIntensity(points, mz):
    """Return interpolated intensity for given m/z.
        points: (numpy array) spectrum data points
        mz: (float) m/z value
    """
    
    # check points
    if len(points) == 0:
        return False
    
    # get mz index
    index = findIndex(points, mz, dim=2)
    if not index or index == len(points):
        return False
    
    # get intensity
    intens = interpolateLine(points[index-1], points[index], x=mz)
    
    return intens
# ----


def peakWidth(points, mz, intensity):
    """Return peak width for given m/z and height.
        points: (numpy array) spectrum data points
        mz: (float) peak m/z value
        intensity: (float) intensity of width measurement
    """
    
    # check points
    if len(points) == 0:
        return None
    
    # get indexes
    index = findIndex(points, mz, dim=2)
    if index < 1 or index == len(points):
        return None
    
    leftIndx = index - 1
    while leftIndx >= 0:
        if points[leftIndx][1] <= intensity:
            break
        leftIndx -= 1
    
    rightIndx = index
    rightMax = len(points) - 1
    while rightIndx < rightMax:
        if points[rightIndx][1] <= intensity:
            break
        rightIndx += 1
    
    # get mz
    leftMZ = interpolateLine(points[leftIndx], points[leftIndx+1], y=intensity)
    rightMZ = interpolateLine(points[rightIndx-1], points[rightIndx], y=intensity)
    
    return abs(rightMZ - leftMZ)
# ----


def labelPoint(points, mz, baselineWindow=0.1, baselineSmooth=True, baselineOffset=0., baselineData=None):
    """Return labeled peak at given m/z value.
        points: (numpy array) spectrum data points
        mz: (float) m/z value to label
        baselineWindow: (float) noise calculation window in %/100
        baselineSmooth: (bool) smooth baseline
        baselineOffset: (float) baseline intensity offset in %/100
        baselineData: (list of [x, noiseLevel, noiseWidth]) precalculated baseline
    """
    
    # check points
    if len(points) == 0:
        return False
    
    # check mz
    if mz <= 0:
        return False
    
    # get peak intensity
    intens = peakIntensity(points, mz)
    if not intens:
        return False
    
    # get baseline data
    if baselineData == None:
        baselineData = baseline(points, window=baselineWindow, smooth=baselineSmooth, offset=baselineOffset)
    
    # get peak baseline and s/n
    i = 0
    while i < len(baselineData)-1 and baselineData[i][0] < mz:
        i += 1
    base = interpolateLine((baselineData[i-1][0], baselineData[i-1][1]), (baselineData[i][0], baselineData[i][1]), x=mz)
    noiseWidth = interpolateLine((baselineData[i-1][0], baselineData[i-1][2]), (baselineData[i][0], baselineData[i][2]), x=mz)
    sn = None
    if noiseWidth:
        sn = (intens - base) / interpolateLine((baselineData[i-1][0], baselineData[i-1][2]), (baselineData[i][0], baselineData[i][2]), x=mz)
    
    # check peak intensity
    if intens <= base:
        return False
    
    # get peak fwhm
    height = base + (intens - base) * 0.5
    fwhm = peakWidth(points, mz, height)
    
    # make peak object
    peak = objects.peak(mz=mz, ai=intens, base=base, sn=sn, fwhm=fwhm)
    
    return peak
# ----


def labelPeak(points, mz=None, minX=None, maxX=None, pickingHeight=0.75, baselineWindow=0.1, baselineSmooth=True, baselineOffset=0., baselineData=None):
    """Return labeled peak in given m/z range.
        mz: (float) single m/z value
        minX: (float) starting m/z value
        maxX: (float) ending m/z value
        pickingHeight: (float) peak picking height for centroiding
        baselineWindow: (float) noise calculation window in %/100
        baselineSmooth: (bool) smooth baseline
        baselineOffset: (float) baseline intensity offset in %/100
        baselineData: (list of [x, noiseLevel, noiseWidth]) precalculated baseline
    """
    
    # check points
    if len(points) == 0:
        return False
    
    # check mz or range
    if mz != None:
        minX = mz
    elif minX == None and maxX == None:
        return False
    
    # check minX
    if minX <= 0:
        return False
    
    # get left index
    i1 = findIndex(points, minX, dim=2)
    indexMax = i1
    if not (0 < i1 < len(points)):
        return False
    
    # get right index
    if maxX != None:
        i2 = findIndex(points, maxX, dim=2)
        if not (0 < i2 < len(points)):
            return False
        
        # find maximum in m/z range
        for x in range(i1,i2+1):
            if points[x][1] > points[indexMax][1]:
                indexMax = x
    
    # get baseline data
    if baselineData == None:
        baselineData = baseline(points, window=baselineWindow, smooth=baselineSmooth, offset=baselineOffset)
    
    # get picking height
    i = 0
    mzMax = points[indexMax][0]
    while i < len(baselineData)-1 and baselineData[i][0] < mzMax:
        i += 1
    noiseLevel = interpolateLine((baselineData[i-1][0], baselineData[i-1][1]), (baselineData[i][0], baselineData[i][1]), x=mzMax)
    height = ((points[indexMax][1] - noiseLevel) * pickingHeight) + noiseLevel
    
    # get centroided m/z
    leftIndx = indexMax-1
    while leftIndx > 0:
        if points[leftIndx][1] <= height:
            break
        leftIndx -= 1
    leftMZ = interpolateLine(points[leftIndx], points[leftIndx+1], y=height)
    
    rightIndx = indexMax
    while rightIndx < (len(points)-1):
        if points[rightIndx][1] <= height:
            break
        rightIndx += 1
    rightMZ = interpolateLine(points[rightIndx-1], points[rightIndx], y=height)
    
    # check range
    if mz == None and (leftMZ < minX or rightMZ > maxX) and (leftMZ != rightMZ):
        return False
    
    # label peak in the newly found selection
    if mz != None and leftMZ != rightMZ:
        peak = labelPeak(
            points = points,
            minX = leftMZ,
            maxX = rightMZ,
            pickingHeight = pickingHeight,
            baselineData = baselineData
        )
        return peak
    
    # label current point
    else:
        peak = labelPoint(
            points = points,
            mz = ((leftMZ + rightMZ)/2),
            baselineData = baselineData
        )
        return peak
# ----


def labelScan(points, minX=None, maxX=None, pickingHeight=0.75, absThreshold=0., relThreshold=0., snThreshold=0., baselineWindow=0.1, baselineSmooth=True, baselineOffset=0., baselineData=None, smoothMethod=None, smoothWindow=0.2, smoothCycles=1):
    """Return centroided peaklist for given data points.
        points: (numpy array) spectrum data points
        minX: (float) starting m/z value
        maxX: (float) ending m/z value
        pickingHeight: (float) peak picking height for centroiding
        absThreshold: (float) absolute intensity threshold
        relThreshold: (float) relative intensity threshold
        snThreshold: (float) signal to noise threshold
        baselineWindow: (float) noise calculation window in %/100
        baselineSmooth: (bool) smooth baseline
        baselineOffset: (float) baseline intensity offset in %/100
        baselineData: (list of [x, noiseLevel, noiseWidth]) precalculated baseline
        smoothMethod: (None, MA GA or SG) smoothing method, MA - moving average, GA - Gaussian, SG - Savitzky-Golay
        smoothWindows: (float) m/z window size for smoothing
        smoothCycles: (int) number of repeating cycles
    """
    
    # check data points
    if len(points) == 0:
        return objects.peaklist()
    
    # get baseline data
    if baselineData == None:
        baselineData = baseline(points, window=baselineWindow, smooth=baselineSmooth, offset=baselineOffset)
    
    CHECK_FORCE_QUIT()
    
    # crop data
    if minX != None and maxX != None:
        i1 = findIndex(points, minX, dim=2)
        i2 = findIndex(points, maxX, dim=2)
        points = points[i1:i2]
    
    # apply smoothing
    if smoothMethod and smoothMethod.upper() == 'MA':
        points = smoothMA(points, smoothWindow, smoothCycles)
    elif smoothMethod and smoothMethod.upper() == 'GA':
        points = smoothGA(points, smoothWindow, smoothCycles)
    elif smoothMethod and smoothMethod.upper() == 'SG':
        points = smoothSG(points, smoothWindow, smoothCycles)
    
    CHECK_FORCE_QUIT()
    
    # get local maxima
    buff = []
    localMaximum = points[0]
    growing = True
    for point in points[1:]:
        if point[1] >= localMaximum[1]:
            localMaximum = point
            growing = True
        elif growing and point[1] < localMaximum[1]:
            peak = [localMaximum[0], localMaximum[1], 0., None, None] # mz, ai, base, sn, fwhm
            buff.append(peak)
            localMax = point
            growing = False
        else:
            localMaximum = point
    
    CHECK_FORCE_QUIT()
    
    # set baseline and sn
    i = 0
    basepeak = 0
    for peak in buff:
        while i < len(baselineData)-1 and baselineData[i][0] < peak[0]:
            i += 1
        b1 = baselineData[i-1]
        b2 = baselineData[i]
        peak[2] = interpolateLine((b1[0], b1[1]), (b2[0], b2[1]), x=peak[0])
        intens = peak[1] - peak[2]
        noiseWidth = interpolateLine((b1[0], b1[2]), (b2[0], b2[2]), x=peak[0])
        if noiseWidth:
            peak[3] = intens/noiseWidth
        if intens > basepeak:
            basepeak = intens
    
    CHECK_FORCE_QUIT()
    
    # remove peaks bellow threshold
    threshold = max(basepeak * relThreshold, absThreshold)
    localMaxima = []
    for peak in buff:
        if peak[0] > 0 and (peak[1] - peak[2]) >= threshold and (not peak[3] or peak[3] >= snThreshold):
            localMaxima.append(peak)
    
    # make centroides
    if pickingHeight == 1.:
        buff = localMaxima
    else:
        buff = []
        previous = None
        for peak in localMaxima:
            
            CHECK_FORCE_QUIT()
            
            # calc peak height
            height = ((peak[1]-peak[2]) * pickingHeight) + peak[2]
            
            # get peak indexes
            index = findIndex(points, peak[0], dim=2)
            if not (0 < index < len(points)):
                continue
            
            leftIndx = index-1
            while leftIndx > 0:
                if points[leftIndx][1] <= height:
                    break
                leftIndx -= 1
            
            rightIndx = index
            rightMax = len(points) - 1
            while rightIndx < rightMax:
                if points[rightIndx][1] <= height:
                    break
                rightIndx += 1
            
            # get mz
            leftMZ = interpolateLine(points[leftIndx], points[leftIndx+1], y=height)
            rightMZ = interpolateLine(points[rightIndx-1], points[rightIndx], y=height)
            peak[0] = (leftMZ + rightMZ)/2
            
            # get intensity
            intensity = peakIntensity(points, peak[0])
            if intensity and intensity <= peak[1]:
                peak[1] = intensity
            else:
                continue
            
            # try to group with previous peak
            if previous != None and leftMZ < previous:
                if peak[1] > buff[-1][1]:
                    buff[-1] = peak
                    previous = rightMZ
            else:
                buff.append(peak)
                previous = rightMZ
    
    CHECK_FORCE_QUIT()
    
    # set baseline, sn and fwhm
    i = 0
    basepeak = 0
    for peak in buff:
        while i < len(baselineData)-1 and baselineData[i][0] < peak[0]:
            i += 1
        b1 = baselineData[i-1]
        b2 = baselineData[i]
        peak[2] = interpolateLine((b1[0], b1[1]), (b2[0], b2[1]), x=peak[0])
        intens = peak[1] - peak[2]
        noiseWidth = interpolateLine((b1[0], b1[2]), (b2[0], b2[2]), x=peak[0])
        if noiseWidth:
            peak[3] = intens/noiseWidth
        peak[4] = peakWidth(points, peak[0], (peak[2] + intens * 0.5))
        if intens > basepeak:
            basepeak = intens
    
    CHECK_FORCE_QUIT()
    
    # remove peaks bellow threshold
    threshold = max(basepeak * relThreshold, absThreshold)
    centroides = []
    for peak in buff:
        if peak[0] > 0 and (peak[1] - peak[2]) >= threshold and (not peak[3] or peak[3] >= snThreshold):
            centroides.append(objects.peak(mz=peak[0], ai=peak[1], base=peak[2], sn=peak[3], fwhm=peak[4]))
    
    # return peaklist object
    return objects.peaklist(centroides)
# ----


def envelopeCentroid(isotopes, height=0.5, intensity='maximum'):
    """Calculate envelope centroid for given isotopes.
        isotopes: (list of peaks)
        height: (float) intensity to calculate centroid width
        intensity: (maximum | sum | average) haw to calculate envelope label intensity
    """
    
    # check isotopes
    if len(isotopes) == 0:
        return False
    elif len(isotopes) == 1:
        return isotopes[0]
    
    # check peaklist object
    if not isinstance(isotopes, objects.peaklist):
        isotopes = objects.peaklist(isotopes)
    
    # get sums
    sumMZ = 0
    sumIntensity = 0
    for isotope in isotopes:
        sumMZ += isotope.mz * isotope.intensity
        sumIntensity += isotope.intensity
    
    # get average m/z
    mz = sumMZ / sumIntensity
    
    # get ai, base and sn
    base = isotopes.basepeak.base
    sn = isotopes.basepeak.sn
    fwhm = isotopes.basepeak.fwhm
    if intensity == 'sum':
        ai = base + sumIntensity
    elif intensity == 'average':
        ai = base + sumIntensity / len(isotopes)
    else:
        ai = isotopes.basepeak.ai
    if isotopes.basepeak.sn:
        sn = (ai - base) * isotopes.basepeak.sn / (isotopes.basepeak.ai - base)
    
    # get envelope width
    minInt = isotopes.basepeak.intensity * height
    i1 = None
    i2 = None
    for x, isotope in enumerate(isotopes):
        if isotope.intensity >= minInt:
            i2 = x
            if i1 == None:
                i1 = x
    
    mz1 = isotopes[i1].mz
    mz2 = isotopes[i2].mz
    if i1 != 0:
        mz1 = interpolateLine((isotopes[i1-1].mz, isotopes[i1-1].ai), (isotopes[i1].mz, isotopes[i1].ai), y=minInt)
    if i2 < len(isotopes)-1:
        mz2 = interpolateLine((isotopes[i2].mz, isotopes[i2].ai), (isotopes[i2+1].mz, isotopes[i2+1].ai), y=minInt)
    if mz1 != mz2:
        fwhm = abs(mz2 - mz1)
    
    # make peak
    peak = objects.peak(mz=mz, ai=ai, base=base, sn=sn, fwhm=fwhm)
    
    return peak
# ----


def envelopeMonoisotope(isotopes, charge, intensity='maximum'):
    """Calculate envelope centroid for given isotopes.
        isotopes: (mspy.peaklist or list of peaks)
        charge: (int) peak charge
        intensity: (maximum | sum | average) haw to calculate envelope label intensity
    """
    
    # check isotopes
    if len(isotopes) == 0:
        return False
    
    # check peaklist object
    if not isinstance(isotopes, objects.peaklist):
        isotopes = objects.peaklist(isotopes)
    
    # calc averagine
    avFormula = averagine(isotopes.basepeak.mz, charge=charge, composition=AVERAGE_AMINO)
    avPattern = avFormula.pattern(fwhm=0.1, threshold=0.001, charge=charge)
    avPattern = objects.peaklist(avPattern)
    
    # get envelope centroid
    points = numpy.array([(p.mz, p.intensity) for p in isotopes])
    baseline = numpy.array([(points[0][0], 0., 0.), (points[-1][0], 0., 0.)])
    centroid = labelPeak(points, mz=isotopes.basepeak.mz, pickingHeight=0.8, baselineData=baseline)
    if not centroid:
        centroid = isotopes.basepeak
    
    # get averagine centroid
    points = numpy.array([(p.mz, p.intensity) for p in avPattern])
    baseline = numpy.array([(points[0][0], 0., 0.), (points[-1][0], 0., 0.)])
    avCentroid = labelPeak(points, mz=avPattern.basepeak.mz, pickingHeight=0.8, baselineData=baseline)
    if not avCentroid:
        avCentroid = avPattern.basepeak
    
    # align profiles and get monoisotopic mass
    shift = centroid.mz - avCentroid.mz
    errors = [(abs(p.mz - avPattern.basepeak.mz - shift), p.mz) for p in isotopes]
    mz = min(errors)[1] - (avPattern.basepeak.mz - avFormula.mz(charge)[0])
    
    # sum intensities
    sumIntensity = 0
    for isotope in isotopes:
        sumIntensity += isotope.intensity
    
    # get ai, base and sn
    base = isotopes.basepeak.base
    sn = isotopes.basepeak.sn
    fwhm = isotopes.basepeak.fwhm
    if intensity == 'sum':
        ai = base + sumIntensity
    elif intensity == 'average':
        ai = base + sumIntensity / len(isotopes)
    else:
        ai = isotopes.basepeak.ai
    if isotopes.basepeak.sn:
        sn = (ai - base) * isotopes.basepeak.sn / (isotopes.basepeak.ai - base)
    
    # make peak
    peak = objects.peak(mz=mz, ai=ai, base=base, sn=sn, fwhm=fwhm, isotope=0)
    
    return peak
# ----


def findIsotopes(peaklist, maxCharge=1, mzTolerance=0.15, intTolerance=0.5, isotopeShift=0.0):
    """In-place calculation of peak charges and isotopes.
        peaklist: (mspy.peaklist) peaklist to process
        maxCharge: (float) max charge to be searched
        mzTolerance: (float) absolute mass tolerance for isotopes
        intTolerance: (float) relative intensity tolerance for isotopes in %/100
        isotopeShift: (float) isotope distance correction (neutral mass)
    """
    
    # check peaklist
    if not isinstance(peaklist, objects.peaklist):
        raise TypeError, "Peak list must be mspy.peaklist object!"
    
    # clear previous results
    for peak in peaklist:
        peak.setCharge(None)
        peak.setIsotope(None)
    
    # get charges
    if maxCharge < 0:
        charges = [-x for x in range(1, abs(maxCharge)+1)]
    else:
        charges = [x for x in range(1, maxCharge+1)]
    charges.reverse()
    
    # walk in a peaklist
    maxIndex = len(peaklist)
    for x, parent in enumerate(peaklist):
        
        CHECK_FORCE_QUIT()
        
        # skip assigned peaks
        if parent.isotope != None:
            continue
        
        # try all charge states
        for z in charges:
            cluster = [parent]
            
            # search for next isotope within m/z tolerance
            difference = (ISOTOPE_DISTANCE + isotopeShift)/abs(z)
            y = 1
            while x+y < maxIndex:
                mzError = (peaklist[x+y].mz - cluster[-1].mz - difference)
                if abs(mzError) <= mzTolerance:
                    cluster.append(peaklist[x+y])
                elif mzError > mzTolerance:
                    break
                y += 1
            
            # no isotope found
            if len(cluster) == 1:
                continue
            
            # get theoretical isotopic pattern
            mass = min(15000, int( mz( parent.mz, 0, z))) / 200
            pattern = patternLookupTable[mass]
            
            # check minimal number of isotopes in the cluster
            limit = 0
            for p in pattern:
                if p >= 0.33:
                    limit += 1
            if len(cluster) < limit and abs(z) > 1:
                continue
            
            # check peak intensities in cluster
            valid = True
            isotope = 1
            limit = min(len(pattern), len(cluster))
            while (isotope < limit):
                
                # calc theoretical intensity from previous peak and current error
                intTheoretical = (cluster[isotope-1].intensity / pattern[isotope-1]) * pattern[isotope]
                intError = cluster[isotope].intensity - intTheoretical
                
                # intensity in tolerance
                if abs(intError) <= (intTheoretical * intTolerance):
                    cluster[isotope].setIsotope(isotope)
                    cluster[isotope].setCharge(z)
                
                # intensity is higher (overlap)
                elif intError > 0:
                    pass
                
                # intensity is lower and first isotope is checked (nonsense)
                elif (intError < 0 and isotope == 1):
                    valid = False
                    break
                
                # try next peak
                isotope += 1
            
            # cluster is OK, set parent peak and skip other charges
            if valid:
                parent.setIsotope(0)
                parent.setCharge(z)
                break
# ----


def deconvolute(peaklist, massType=0):
    """Recalculate peaklist to singly charged.
        peaklist: (mspy.peaklist) peak list to deconvolute
        massType: (0 or 1) mass type used for m/z re-calculation, 0 = monoisotopic, 1 = average
    """
    
    # recalculate peaks
    buff = []
    for peak in copy.deepcopy(peaklist):
        
        CHECK_FORCE_QUIT()
        
        # uncharged peak
        if not peak.charge:
            continue
        
        # correct charge
        elif abs(peak.charge) == 1:
            buff.append(peak)
        
        # recalculate peak
        else:
            
            # set fwhm
            if peak.fwhm:
                newFwhm = abs(peak.fwhm*peak.charge)
                peak.setFwhm(newFwhm)
            
            # set m/z and charge
            if peak.charge < 0:
                newMz = mz(mass=peak.mz, charge=-1, currentCharge=peak.charge, massType=massType)
                peak.setMz(newMz)
                peak.setCharge(-1)
            else:
                newMz = mz(mass=peak.mz, charge=1, currentCharge=peak.charge, massType=massType)
                peak.setMz(newMz)
                peak.setCharge(1)
            
            # store peak
            buff.append(peak)
    
    # finalize peaks
    if buff:
        base = min([p.base for p in buff])
        for peak in buff:
            peak.setSN(None)
            peak.setAi(peak.ai - peak.base + base)
            peak.setBase(base)
    
    # update peaklist
    peaklist = objects.peaklist(buff)
    
    return peaklist
# ----


def correctBaseline(points, window=0.1, smooth=True, offset=0., baselineData=None):
    """Return data points with baseline correction.
        points: (numpy array) spectrum data points
        window: (float) noise calculation window in %/100
        smooth: (bool) smooth final baseline
        offset: (float) global intensity offset in %/100
        baselineData: (list of [x, noiseLevel]) precalculated baseline
    """
    
    # check points
    if len(points) == 0:
        return points
    
    # get baseline
    if baselineData == None:
        baselineData = baseline(points, window=window, smooth=smooth, offset=offset)
    
    CHECK_FORCE_QUIT()
    
    # calculate offsets for points
    offsets = []
    if len(baselineData) == 1:
        offsets.append((0., baselineData[0][1]))
    else:
        i = 1
        m = (baselineData[i][1] - baselineData[i-1][1])/(baselineData[i][0] - baselineData[i-1][0])
        b = baselineData[i-1][1] - m * baselineData[i-1][0]
        
        limit = len(baselineData)-1
        for x in xrange(len(points)):
            
            CHECK_FORCE_QUIT()
            
            while i < limit and baselineData[i][0] < points[x][0]:
                i += 1
                m = (baselineData[i][1] - baselineData[i-1][1])/(baselineData[i][0] - baselineData[i-1][0])
                b = baselineData[i-1][1] - m * baselineData[i-1][0]
            
            offsets.append((0., m * points[x][0] + b))
    
    # shift points to zero level
    shifted = points - numpy.array(offsets)
    
    # remove negative intensities
    minXY = numpy.minimum.reduce(shifted)
    maxXY = numpy.maximum.reduce(shifted)
    shifted = shifted.clip([minXY[0],0.], maxXY)
    
    return shifted
# ----


def unifyRaster(pointsA, pointsB):
    """Return data points with unified x-raster.
        pointsA: (numpy array) spectrum data points
        pointsB: (numpy array) spectrum data points
    """
    
    # convert arrays
    pointsA = list(pointsA)
    pointsB = list(pointsB)
    
    # count arrays
    countA = len(pointsA)
    countB = len(pointsB)
    
    # merge left
    i = 0
    while i<countA and i<countB and pointsA[i][0] < pointsB[i][0]:
        pointsB.insert(i, [pointsA[i][0], 0.0])
        countB += 1
        i += 1
    
    i = 0
    while i<countA and i<countB and pointsA[i][0] > pointsB[i][0]:
        pointsA.insert(i, [pointsB[i][0], 0.0])
        countA += 1
        i += 1
    
    # merge middle
    for i, x in enumerate(pointsA):
        
        CHECK_FORCE_QUIT()
        
        if i == countB:
            break
        
        if pointsA[i][0] < pointsB[i][0]:
            intens = interpolateLine(pointsB[i-1], pointsB[i], x=pointsA[i][0])
            pointsB.insert(i, [pointsA[i][0], intens])
            countB += 1
            
        elif pointsA[i][0] > pointsB[i][0]:
            intens = interpolateLine(pointsA[i-1], pointsA[i], x=pointsB[i][0])
            pointsA.insert(i, [pointsB[i][0], intens])
            countA += 1
    
    # merge right
    if countA < countB:
        for x in pointsB[countA:]:
            pointsA.append([x[0],0.0])
            countA += 1
    
    elif countA > countB:
        for x in pointsA[countB:]:
            pointsB.append([x[0],0.0])
            countB += 1
    
    return pointsA, pointsB
# ----


def normalize(data):
    """Normalize data."""
    
    # get maximum Y
    maximum = data[0][1]
    for item in data:
        if item[1] > maximum:
            maximum = item[1]
    
    # normalize data data
    for x in range(len(data)):
        data[x][1] /= maximum
    
    return data
# ----



# DATA RE-CALIBRATION
# -------------------

def calibration(data, model='linear'):
    """Calculate calibration constants for given references.
        data: (list) pairs of (measured mass, reference mass)
        model: ('linear' or 'quadratic')
        This function uses least square fitting written by Konrad Hinsen.
    """
    
    # single point calibration
    if model == 'linear' and len(data) == 1:
        shift = data[0][1] - data[0][0]
        return _linearModel, (1., shift), 1.0
    
    # set fitting model and initial values
    if model=='linear':
        model = _linearModel
        initials = (0.5, 0)
    elif model=='quadratic':
        model = _quadraticModel
        initials = (1., 0, 0)
    
    # calculate calibration constants
    params = _leastSquaresFit(model, initials, data)
    
    # fce, parameters, chi-square
    return model, params[0], params[1]
# ----


def _linearModel(params, x):
    """Function for linear model."""
    
    a, b = params
    return a*x + b
# ----


def _quadraticModel(params, x):
    """Function for quadratic model."""
    
    a, b, c = params
    return a*x*x + b*x + c
# ----


def _leastSquaresFit(model, parameters, data, maxIterations=None, limit=1e-7):
    """General non-linear least-squares fit using the
    Levenberg-Marquardt algorithm and automatic derivatives."""
    
    n_param = len(parameters)
    p = ()
    i = 0
    for param in parameters:
        p = p + (_DerivVar(param, i),)
        i = i + 1
    id = numpy.identity(n_param)
    l = 0.001
    chi_sq, alpha = _chiSquare(model, p, data)
    niter = 0
    
    while True:
        niter += 1
        
        delta = solveLinEq(alpha+l*numpy.diagonal(alpha)*id,-0.5*numpy.array(chi_sq[1]))
        next_p = map(lambda a,b: a+b, p, delta)
        
        next_chi_sq, next_alpha = _chiSquare(model, next_p, data)
        if next_chi_sq > chi_sq:
            l = 10.*l
        elif chi_sq[0] - next_chi_sq[0] < limit:
            break
        else:
            l = 0.1*l
            p = next_p
            chi_sq = next_chi_sq
            alpha = next_alpha
        
        if maxIterations and niter == maxIterations:
            break
    
    return map(lambda p: p[0], next_p), next_chi_sq[0]
# ----


def _chiSquare(model, parameters, data):
    """Count chi-square."""
    
    n_param = len(parameters)
    alpha = numpy.zeros((n_param, n_param))
    
    chi_sq = _DerivVar(0., [])
    for point in data:
        f = model(parameters, point[0])
        chi_sq += (f-point[1])**2
        d = numpy.array(f[1])
        alpha = alpha + d[:,numpy.newaxis]*d
    
    return chi_sq, alpha
# ----


class _DerivVar:
    """This module provides automatic differentiation for functions with any number of variables."""
    
    def __init__(self, value, index=0):
        self.value = value
        if type(index) == type([]):
            self.deriv = index
        else:
            self.deriv = index*[0] + [1]
    
    def _mapderiv(self, func, a, b):
        nvars = max(len(a), len(b))
        a = a + (nvars-len(a))*[0]
        b = b + (nvars-len(b))*[0]
        return map(func, a, b)
    
    def __getitem__(self, item):
        if item == 0:
            return self.value
        elif item == 1:
            return self.deriv
        else:
            raise IndexError
    
    def __cmp__(self, other):
        if isinstance(other, _DerivVar):
            return cmp(self.value, other.value)
        else:
            return cmp(self.value, other)
    
    def __add__(self, other):
        if isinstance(other, _DerivVar):
            return _DerivVar(self.value + other.value, self._mapderiv(lambda a,b: a+b, self.deriv, other.deriv))
        else:
            return _DerivVar(self.value + other, self.deriv)
    
    def __radd__(self, other):
        if isinstance(other, _DerivVar):
            self.value += other.value
            self.deriv = self._mapderiv(lambda a,b: a+b, self.deriv, other.deriv)
            return self
        else:
            self.value += other
            return self
    
    def __sub__(self, other):
        if isinstance(other, _DerivVar):
            return _DerivVar(self.value - other.value, self._mapderiv(lambda a,b: a-b, self.deriv, other.deriv))
        else:
            return _DerivVar(self.value - other, self.deriv)
    
    def __rsub__(self, other):
        if isinstance(other, _DerivVar):
            self.value -= other.value
            self.deriv = self._mapderiv(lambda a,b: a-b, self.deriv, other.deriv)
            return self
        else:
            self.value -= other
            return self
    
    def __mul__(self, other):
        if isinstance(other, _DerivVar):
            return _DerivVar(self.value * other.value, self._mapderiv(lambda a,b: a+b, map(lambda x,f=self.value:f*x, other.deriv), map(lambda x,f=other.value:f*x, self.deriv)))
        else:
            return _DerivVar(self.value * other, map(lambda x,f=other:f*x, self.deriv))
    
    def __div__(self, other):
        if isinstance(other, _DerivVar):
            inv = 1./other.value
            return _DerivVar(self.value * inv, self._mapderiv(lambda a,b: a-b, map(lambda x,f=inv: f*x, self.deriv), map(lambda x,f=self.value*inv*inv: f*x, other.deriv)))
        else:
            inv = 1./value
            return _DerivVar(self.value * inv, map(lambda x,f=inv:f*x, self.deriv))
    
    def __pow__(self, other):
        val1 = pow(self.value, other-1)
        deriv1 = map(lambda x,f=val1*other: f*x, self.deriv)
        return _DerivVar(val1*self.value, deriv1)
        
    def __abs__(self):
        absvalue = abs(self.value)
        return _DerivVar(absvalue, map(lambda a, d=self.value/absvalue: d*a, self.deriv))
    
# ----



# HELPERS
# -------

def findIndex(points, x, dim=1):
    """Get nearest lower index for selected point."""
    
    lo = 0
    hi = len(points)
    
    # 1D array
    if dim == 1:
        while lo < hi:
            mid = (lo + hi) / 2
            if x < points[mid]:
                hi = mid
            else:
                lo = mid + 1
    # 2D array
    if dim == 2:
        while lo < hi:
            mid = (lo + hi) / 2
            if x < points[mid][0]:
                hi = mid
            else:
                lo = mid + 1
    
    return lo
# ----


def interpolateLine(p1, p2, x=None, y=None):
    """Get line interpolated X or Y value."""
    
    # check points
    if p1[0] == p2[0] and x!=None:
        return max(p1[1], p2[1])
    elif p1[0] == p2[0] and y!=None:
        return p1[0]
    
    # get equation
    m = (p2[1] - p1[1])/(p2[0] - p1[0])
    b = p1[1] - m * p1[0]
    
    # get point
    if x != None:
        return m * x + b
    elif y != None:
        return (y - b) / m
# ----



# PATTERN LOOKUP TABLE
# --------------------

def generatePatternLookupTable(highmass, step=200, composition=AVERAGE_AMINO, table='tuple'):
    """Print pattern lookup table."""
    
    for mass in range(0, highmass, step):
        formula = averagine(mass, composition=composition)
        
        pattern = ''
        for mz, abundance in formula.pattern(fwhm=0.1, threshold=0.001):
            pattern += '%.3f, ' % abundance
        
        if table == 'tuple':
            print '(%s), #%d' % (pattern[:-2], mass)
        elif table == 'dict':
            print '%d: (%s),' % (mass, pattern[:-2])
# ----


# pattern lookup table for amino building block
patternLookupTable = (
    (1.000, 0.059, 0.003), #0
    (1.000, 0.122, 0.013), #200
    (1.000, 0.241, 0.040, 0.005), #400
    (1.000, 0.303, 0.059, 0.008), #600
    (1.000, 0.426, 0.109, 0.020, 0.003), #800
    (1.000, 0.533, 0.166, 0.038, 0.006), #1000
    (1.000, 0.655, 0.244, 0.066, 0.014, 0.002), #1200
    (1.000, 0.786, 0.388, 0.143, 0.042, 0.009, 0.001), #1400
    (1.000, 0.845, 0.441, 0.171, 0.053, 0.013, 0.002), #1600
    (1.000, 0.967, 0.557, 0.236, 0.080, 0.021, 0.005), #1800
    (0.921, 1.000, 0.630, 0.291, 0.107, 0.032, 0.007, 0.001), #2000
    (0.828, 1.000, 0.687, 0.343, 0.136, 0.044, 0.011, 0.002), #2200
    (0.752, 1.000, 0.744, 0.400, 0.171, 0.060, 0.017, 0.004), #2400
    (0.720, 1.000, 0.772, 0.428, 0.188, 0.068, 0.020, 0.005), #2600
    (0.667, 1.000, 0.825, 0.487, 0.228, 0.088, 0.028, 0.007), #2800
    (0.616, 1.000, 0.884, 0.556, 0.276, 0.113, 0.039, 0.010, 0.002), #3000
    (0.574, 1.000, 0.941, 0.628, 0.330, 0.143, 0.052, 0.015, 0.003), #3200
    (0.536, 0.999, 1.000, 0.706, 0.392, 0.179, 0.069, 0.022, 0.005), #3400
    (0.506, 0.972, 1.000, 0.725, 0.412, 0.193, 0.077, 0.025, 0.006), #3600
    (0.449, 0.919, 1.000, 0.764, 0.457, 0.226, 0.094, 0.033, 0.009, 0.001), #3800
    (0.392, 0.853, 1.000, 0.831, 0.543, 0.295, 0.136, 0.053, 0.017, 0.004), #4000
    (0.353, 0.812, 1.000, 0.869, 0.593, 0.336, 0.162, 0.067, 0.023, 0.006), #4200
    (0.321, 0.776, 1.000, 0.907, 0.644, 0.379, 0.190, 0.082, 0.030, 0.009), #4400
    (0.308, 0.760, 1.000, 0.924, 0.669, 0.401, 0.205, 0.090, 0.033, 0.011, 0.001), #4600
    (0.282, 0.729, 1.000, 0.962, 0.723, 0.451, 0.239, 0.110, 0.042, 0.014, 0.003), #4800
    (0.258, 0.699, 1.000, 1.000, 0.780, 0.504, 0.277, 0.132, 0.053, 0.018, 0.004), #5000
    (0.228, 0.645, 0.962, 1.000, 0.809, 0.542, 0.308, 0.153, 0.065, 0.023, 0.007), #5200
    (0.203, 0.598, 0.927, 1.000, 0.839, 0.581, 0.343, 0.176, 0.078, 0.029, 0.010), #5400
    (0.192, 0.577, 0.911, 1.000, 0.854, 0.602, 0.361, 0.189, 0.086, 0.033, 0.011), #5600
    (0.171, 0.536, 0.880, 1.000, 0.884, 0.644, 0.399, 0.216, 0.102, 0.040, 0.014, 0.003), #5800
    (0.154, 0.501, 0.851, 1.000, 0.912, 0.686, 0.439, 0.244, 0.120, 0.050, 0.018, 0.004), #6000
    (0.139, 0.468, 0.823, 1.000, 0.942, 0.730, 0.482, 0.278, 0.141, 0.062, 0.023, 0.007), #6200
    (0.126, 0.441, 0.799, 1.000, 0.969, 0.772, 0.524, 0.310, 0.162, 0.073, 0.028, 0.009), #6400
    (0.121, 0.427, 0.787, 1.000, 0.983, 0.794, 0.547, 0.328, 0.174, 0.080, 0.031, 0.011), #6600
    (0.104, 0.381, 0.732, 0.971, 1.000, 0.848, 0.614, 0.390, 0.219, 0.109, 0.045, 0.016, 0.004), #6800
    (0.092, 0.349, 0.691, 0.944, 1.000, 0.872, 0.648, 0.422, 0.244, 0.125, 0.054, 0.020, 0.006), #7000
    (0.082, 0.321, 0.654, 0.919, 1.000, 0.894, 0.682, 0.456, 0.270, 0.143, 0.063, 0.024, 0.008), #7200
    (0.073, 0.296, 0.620, 0.895, 1.000, 0.917, 0.718, 0.492, 0.299, 0.162, 0.077, 0.030, 0.011), #7400
    (0.069, 0.284, 0.604, 0.884, 1.000, 0.929, 0.735, 0.509, 0.313, 0.172, 0.084, 0.033, 0.012), #7600
    (0.062, 0.262, 0.573, 0.861, 1.000, 0.952, 0.772, 0.548, 0.345, 0.195, 0.098, 0.040, 0.015, 0.003), #7800
    (0.056, 0.243, 0.544, 0.839, 1.000, 0.976, 0.811, 0.589, 0.380, 0.220, 0.114, 0.049, 0.019, 0.005), #8000
    (0.051, 0.227, 0.521, 0.821, 1.000, 0.997, 0.846, 0.628, 0.413, 0.244, 0.130, 0.058, 0.022, 0.007), #8200
    (0.045, 0.206, 0.486, 0.786, 0.980, 1.000, 0.869, 0.660, 0.444, 0.268, 0.147, 0.070, 0.027, 0.010), #8400
    (0.042, 0.196, 0.468, 0.767, 0.968, 1.000, 0.879, 0.676, 0.460, 0.281, 0.156, 0.075, 0.030, 0.011), #8600
    (0.038, 0.179, 0.437, 0.733, 0.947, 1.000, 0.899, 0.705, 0.491, 0.307, 0.173, 0.086, 0.036, 0.013, 0.002), #8800
    (0.033, 0.163, 0.408, 0.701, 0.926, 1.000, 0.919, 0.736, 0.524, 0.335, 0.193, 0.099, 0.043, 0.016, 0.004), #9000
    (0.030, 0.149, 0.382, 0.670, 0.906, 1.000, 0.938, 0.768, 0.558, 0.364, 0.215, 0.113, 0.051, 0.020, 0.006), #9200
    (0.026, 0.132, 0.348, 0.629, 0.877, 1.000, 0.971, 0.823, 0.620, 0.420, 0.258, 0.143, 0.069, 0.028, 0.010), #9400
    (0.024, 0.126, 0.337, 0.616, 0.868, 1.000, 0.981, 0.839, 0.638, 0.437, 0.271, 0.153, 0.074, 0.031, 0.011), #9600
    (0.022, 0.116, 0.317, 0.592, 0.851, 1.000, 1.000, 0.872, 0.676, 0.472, 0.298, 0.172, 0.087, 0.037, 0.014, 0.002), #9800
    (0.020, 0.106, 0.294, 0.561, 0.822, 0.983, 1.000, 0.888, 0.700, 0.498, 0.320, 0.188, 0.099, 0.043, 0.017, 0.004), #10000
    (0.017, 0.096, 0.272, 0.529, 0.790, 0.965, 1.000, 0.905, 0.727, 0.526, 0.346, 0.207, 0.113, 0.050, 0.020, 0.006), #10200
    (0.015, 0.087, 0.251, 0.499, 0.761, 0.946, 1.000, 0.922, 0.755, 0.556, 0.373, 0.227, 0.126, 0.061, 0.024, 0.008), #10400
    (0.014, 0.083, 0.242, 0.486, 0.747, 0.937, 1.000, 0.930, 0.768, 0.570, 0.385, 0.237, 0.134, 0.065, 0.026, 0.009), #10600
    (0.013, 0.075, 0.225, 0.459, 0.720, 0.920, 1.000, 0.947, 0.796, 0.602, 0.415, 0.260, 0.149, 0.075, 0.032, 0.012, 0.001), #10800
    (0.012, 0.069, 0.208, 0.435, 0.695, 0.904, 1.000, 0.963, 0.824, 0.633, 0.443, 0.284, 0.165, 0.085, 0.037, 0.015, 0.002), #11000
    (0.010, 0.063, 0.194, 0.412, 0.669, 0.888, 1.000, 0.980, 0.852, 0.667, 0.475, 0.309, 0.184, 0.098, 0.044, 0.018, 0.005), #11200
    (0.009, 0.057, 0.180, 0.391, 0.646, 0.872, 1.000, 0.997, 0.882, 0.702, 0.509, 0.336, 0.204, 0.113, 0.052, 0.021, 0.006), #11400
    (0.009, 0.054, 0.173, 0.379, 0.631, 0.861, 0.995, 1.000, 0.892, 0.717, 0.523, 0.350, 0.214, 0.119, 0.057, 0.023, 0.008), #11600
    (0.008, 0.049, 0.160, 0.355, 0.602, 0.834, 0.980, 1.000, 0.906, 0.739, 0.548, 0.373, 0.231, 0.132, 0.066, 0.026, 0.010), #11800
    (0.007, 0.042, 0.141, 0.321, 0.557, 0.791, 0.953, 1.000, 0.931, 0.781, 0.596, 0.417, 0.268, 0.158, 0.082, 0.037, 0.014, 0.002), #12000
    (0.006, 0.038, 0.130, 0.301, 0.531, 0.767, 0.939, 1.000, 0.945, 0.805, 0.624, 0.443, 0.289, 0.174, 0.093, 0.043, 0.017, 0.004), #12200
    (0.005, 0.035, 0.120, 0.283, 0.507, 0.744, 0.925, 1.000, 0.960, 0.830, 0.653, 0.470, 0.312, 0.191, 0.106, 0.051, 0.020, 0.006), #12400
    (0.005, 0.033, 0.115, 0.274, 0.495, 0.732, 0.918, 1.000, 0.967, 0.842, 0.668, 0.485, 0.324, 0.200, 0.112, 0.054, 0.023, 0.007), #12600
    (0.004, 0.030, 0.107, 0.257, 0.472, 0.710, 0.904, 1.000, 0.982, 0.868, 0.699, 0.515, 0.351, 0.219, 0.126, 0.063, 0.027, 0.010), #12800
    (0.004, 0.027, 0.098, 0.242, 0.450, 0.689, 0.890, 1.000, 0.997, 0.894, 0.731, 0.547, 0.378, 0.241, 0.141, 0.072, 0.032, 0.012, 0.002), #13000
    (0.003, 0.025, 0.090, 0.224, 0.426, 0.661, 0.867, 0.989, 1.000, 0.911, 0.756, 0.574, 0.402, 0.260, 0.155, 0.082, 0.037, 0.014, 0.003), #13200
    (0.003, 0.022, 0.082, 0.208, 0.402, 0.633, 0.843, 0.975, 1.000, 0.925, 0.777, 0.598, 0.425, 0.279, 0.169, 0.092, 0.043, 0.017, 0.005), #13400
    (0.003, 0.021, 0.079, 0.202, 0.392, 0.621, 0.833, 0.969, 1.000, 0.930, 0.786, 0.609, 0.435, 0.288, 0.176, 0.097, 0.046, 0.018, 0.006), #13600
    (0.003, 0.019, 0.073, 0.188, 0.370, 0.595, 0.810, 0.955, 1.000, 0.943, 0.808, 0.634, 0.460, 0.309, 0.191, 0.108, 0.053, 0.022, 0.007), #13800
    (0.002, 0.017, 0.067, 0.175, 0.350, 0.570, 0.787, 0.942, 1.000, 0.956, 0.831, 0.662, 0.487, 0.331, 0.209, 0.121, 0.062, 0.026, 0.010), #14000
    (0.002, 0.016, 0.061, 0.163, 0.330, 0.547, 0.765, 0.929, 1.000, 0.968, 0.855, 0.690, 0.515, 0.356, 0.227, 0.135, 0.070, 0.031, 0.012, 0.002), #14200
    (0.002, 0.014, 0.056, 0.151, 0.312, 0.524, 0.743, 0.916, 1.000, 0.982, 0.878, 0.718, 0.544, 0.382, 0.247, 0.149, 0.079, 0.037, 0.014, 0.003), #14400
    (0.002, 0.013, 0.054, 0.146, 0.304, 0.514, 0.733, 0.909, 1.000, 0.989, 0.890, 0.733, 0.559, 0.395, 0.257, 0.156, 0.084, 0.039, 0.016, 0.004), #14600
    (0.001, 0.012, 0.047, 0.131, 0.276, 0.478, 0.697, 0.881, 0.989, 1.000, 0.920, 0.777, 0.605, 0.437, 0.292, 0.182, 0.102, 0.051, 0.022, 0.007), #14800
    (0.001, 0.010, 0.043, 0.121, 0.259, 0.454, 0.671, 0.859, 0.977, 1.000, 0.932, 0.797, 0.629, 0.460, 0.312, 0.197, 0.114, 0.058, 0.025, 0.008, 0.001), #15000
)

