
#include <wx/filename.h>
#include <wx/stdpaths.h>

//#include <boost/lexical_cast.hpp>

#include <algorithm>
#include <set>
#include <math.h>

#include "Dictionary.h"
#include "DicomFile.h"
#include "StringConvert.h"
#include "Volume.h"
#include "SeriesHandler.h"

using namespace jcs;
using namespace std;




SeriesHandler::SeriesHandler(const std::string& seriesUid)
: mSeriesUid(seriesUid)
{ 
  seriesInfo.subject_name = "";
  seriesInfo.subject_id = "";
  seriesInfo.study_number = "";
  seriesInfo.series_number = "";
  seriesInfo.StudyDate = "";
  seriesInfo.SeriesDate = "";
  seriesInfo.study_description = "";
  seriesInfo.series_description = "";

}

SeriesHandler::~SeriesHandler()
{
}



/*
SeriesHandler&
SeriesHandler::operator= (const SeriesHandler& rhs)
{
  m_series = rhs.m_series;
  return *this;
}
*/

std::vector<double> 
SeriesHandler::GetIppForFirstSlice(const VolId& id)
{ 
  return position[id]; 
}


vector<double>
SeriesHandler::GetVoxelSize()
{
  vector<double> voxel_size(3, 0);
  string s;
  if(Find("PixelSpacing", s)) {
    voxel_size[0] = stof(s, 0);
    voxel_size[1] = stof(s, 1);
  }
  Find("SliceSpacing", voxel_size[2]);
  if (voxel_size[2] == 0)
    Find("SliceThickness", voxel_size[2]);
  return voxel_size;
}

int
SeriesHandler::GetNumberOfSlices() const
{
  int nVolumes = GetNumberOfVolumes();
  int nFiles = GetNumberOfFiles();
  if (nVolumes != 0)
    return nFiles/nVolumes;
  else
    return nFiles;

}


vector<string> 
SeriesHandler::GetFilenames() const
{
  vector<string> names;
  for(FileMapType::const_iterator it = mFileMap.begin(); it != mFileMap.end(); ++it) 
    names.push_back(it->first);
  return names;

}

bool
SeriesHandler::IsBigEndian() const
{
  if (mFileMap.empty()) return 0;
  DicomFile dicomFile((*mFileMap.begin()).first.c_str());
  return dicomFile.IsBigEndian();
}

int
SeriesHandler::GetNumberOfVolumes() const
{
  std::set<VolId> volSet = GetVolIds();
  /*
  for(FileMapType::const_iterator it = mFileMap.begin(); it != mFileMap.end(); ++it) {
    for (VolListType::const_iterator vit = it->second.begin(); 
      vit != it->second.end(); ++vit) { 
      volSet.insert(*vit);
    }
  }
  */
  return volSet.size();
}

string
SeriesHandler::GetImagePositionPatient(DicomFile& dfile, int frame)
{
  string ipp;

  if (!dfile.Find("ImagePositionPatient", ipp))
    ipp = "0\\0\\" + itos(frame);
  return ipp;

}

int
SeriesHandler::GetRescaleSlopeAndIntercept(const VolId& id, double& slope, double& intercept) const
{
  std::string filename;
  if (!GetFirstFilename(id, filename)) return 0;
  DicomFile dicomFile(filename.c_str());
  return GetRescaleSlopeAndIntercept(dicomFile, slope, intercept);
}

//****
// Seeks RescaleSlope and RescaleIntercept attributes. If
// not found, will set slope to 1 and intercept to 0.
//****
int
SeriesHandler::GetRescaleSlopeAndIntercept(DicomFile& dfile, double& slope, double& intercept, int frame) const
{
  if (!dfile.Find("RescaleSlope", slope)) slope = 1;
  if (!dfile.Find("RescaleIntercept", intercept))  intercept = 0;
  return 1;
}

//****
//****
double
SeriesHandler::GetVolumeInterval() const
{
  map<VolId, vector<double> > vtmap;
  FileMapType::const_iterator it = mFileMap.begin();
  while (it != mFileMap.end()) { 
    DicomFile file(it->first.c_str());
    double time;
    file.Find("AcquisitionTime", time);
    vtmap[it->second.front()].push_back(time);
    ++it;
  }

  if (vtmap.size() <= 1) return 0;

  vector<double> vol_times;
  map<VolId, vector<double> >::iterator vit = vtmap.begin();
  while (vit != vtmap.end()) { 
    vol_times.push_back(*min_element(vit->second.begin(), vit->second.end()));
    ++vit;
  }

  sort(vol_times.begin(), vol_times.end());

  return (vol_times[1] - vol_times[0]);

}

//****
//****
// todo
double
SeriesHandler::GetSliceDuration() const
{
  return 0;
}

int
SeriesHandler::GetColumns()
{
  int columns;
  Find("ImageColumns", columns);
  return columns;
}

int
SeriesHandler::GetRows()
{
  int rows;
  Find("ImageRows", rows);
  return rows;
}

int
SeriesHandler::GetFrames() const
{
  int frames;
  Find("NumberOfFrames", frames);
  return frames;
}

int
SeriesHandler::AddFile(const char* filename)
{
  if (mFileMap.count(filename)) return 0;

  DicomFile dicom_file(filename);

  string series_uid;
  dicom_file.Find("SeriesInstanceUid", series_uid);

  // Fix problems with termination
  string temp(series_uid.c_str());
  series_uid.clear();
  series_uid = temp;

  assert(mSeriesUid == series_uid);

  std::string uid;
  dicom_file.Find("SopInstance", uid);
  if (mInstanceUids.count(uid) == 0 ){
    mInstanceUids.insert(uid);
    VolListType info = ReadVolIds(dicom_file);
    mFileMap[filename] = info;
  }
  
  if (seriesInfo.SeriesDate == "") {
    Find("PatientId", seriesInfo.subject_id);
    Find("PatientName", seriesInfo.subject_name);
    Find("StudyId", seriesInfo.study_number);
    Find("StudyDescription", seriesInfo.study_description);
    int series_number;
    Find("SeriesNumber", series_number);
    seriesInfo.series_number = itos(series_number, 3);
    Find("StudyDate", seriesInfo.StudyDate);
    Find("SeriesDate", seriesInfo.SeriesDate);
    Find("SeriesDescription", seriesInfo.series_description);

  }

  return 1;
}


vector<string>
SeriesHandler::GetStringInfo()
{
  vector<string> v;
  string s;

//  Find("SeriesInstanceUid", s);
  v.push_back(string("Series UID: ") + GetSeriesUid());

  if (Find("StudyDate", s))
    v.push_back(string("Study date: ") + s);

  if (Find("StudyTime", s))
    v.push_back(string("Study time: ") + s);

  if (Find("SeriesDate", s))
    v.push_back(string("Series date: ") + s);

  if (Find("SeriesTime", s))
    v.push_back(string("Series time: ") + s);

  if (Find("PatientName", s))
    v.push_back(string("Subject: ") + s);

  if (Find("SeriesDescription", s))
    v.push_back(string("Series description: ") + s);

  if (Find("Manufacturer", s))
    v.push_back(string("Manufacturer: ") + s);

  if (Find("ModelName", s))
    v.push_back(string("Model name: ") + s);

  if (Find("StudyId", s))
    v.push_back(string("Study id: ") + s);

  if (Find("SeriesNumber", s))
    v.push_back(string("Series number: ") + s);

  if (Find("RepetitionTime", s))
    v.push_back(string("Repetition time (ms): ") + s);

  if (Find("EchoTime", s))
    v.push_back(string("Echo time (ms): ") + s);

  if (Find("InversionTime", s))
    v.push_back(string("Inversion time (ms): ") + s);

  if (Find("FlipAngle", s))
    v.push_back(string("Flip angle: ") + s);

  if (Find("NumberOfAverages", s))
    v.push_back(string("Number of averages: ") + s);

  if(Find("SliceThickness", s))
    v.push_back(string("Slice thickness (mm): ") + s);

  if(Find("SliceSpacing", s)) 
    v.push_back(string("Slice spacing (mm): ") + s);

  if(Find("ImageColumns", s))
    v.push_back(string("Image columns: ") + s);

  if (Find("ImageRows", s))
    v.push_back(string("Image rows: ") + s);

  if (Find("NumberOfFrames", s))
    v.push_back(string("Number of frames: ") + s);

  if (Find("PhaseEncodingDirection", s))
    v.push_back(string("Phase encoding direction: ") + s);


  vector<double> voxels = GetVoxelSize();
  v.push_back(string("Voxel size x (mm): ") + ftos(voxels[0]));
  v.push_back(string("Voxel size y (mm): ") + ftos(voxels[1]));

  int nVolumes = GetNumberOfVolumes();
  v.push_back(string("Number of volumes: ") + itos(nVolumes));
  v.push_back(string("Number of slices: ") + itos(GetNumberOfSlices()));
  v.push_back(string("Number of files: ") + itos(GetNumberOfFiles()));
  v.push_back(string("Number of frames: ") + itos(GetFrames()));


  v.push_back(string("Slice duration (ms) : ") + ftos(GetSliceDuration()));
  if (nVolumes > 1)
    v.push_back(string("Volume interval (s): ") + ftos(GetVolumeInterval()));

  if (!orientations.empty()) {
    if (orientations.front().at(1) > orientations.front().at(0)) // sagital slices
      v.push_back(string("Orientation: sag"));

    else if (orientations.front().at(4) 
        > fabs(orientations.front().at(5))) // transverse slices
      v.push_back(string("Orientation: tra"));
  
    else // coronal slices
      v.push_back(string("Orientation: cor"));
  }

  return v;
}

// DicomFile will set str to empty string if value not found, so we don't have to check.
SeriesHandler::VolListType
SeriesHandler::ReadVolIds(DicomFile& file)
{
  // should use FrameIncrementPointer, but doing it the easy way for now
  int nframes;
  if (!file.Find("NumberOfFrames", nframes)) nframes = 1;

  VolListType v;

  for (int frameno = 0; frameno < nframes; ++frameno) {

    VolId info;

    info.ids.push_back(GetSeriesUid());

    string str;
    file.Find("NumberOfTemporalPositions", str);
    int ndigits = str.size();
  
    str.clear();
    int tpid = 0;
    file.Find("TemporalPositionId", tpid);
    str = itos(tpid, ndigits);
    info.ids.push_back(str);

    str.clear();
    int echonumber = 0;
    file.Find("EchoNumber", echonumber);
    str = itos(echonumber, 2);
    info.ids.push_back(str);
  /*
    str.clear();
    file.Find("TriggerTime", str);
    info.ids.push_back(str);
  */  
    std::string ImageType;
    file.Find("ImageType", ImageType);

    // cute but could cause problems
  //  vector<string> it_parsed = ParseDicomString(ImageType);
  //  info.ids.insert(info.ids.end(), it_parsed.begin(), it_parsed.end());

    if (ImageType.find("DERIVED") != std::string::npos) info.ids.push_back("D");
    else info.ids.push_back("");
    if (ImageType.find("T2 MAP") != std::string::npos) info.ids.push_back("T2MAP");
    else info.ids.push_back("");
    if (ImageType.find("\\M\\") != std::string::npos) info.ids.push_back("M");
    else if (ImageType.find("\\P\\") != std::string::npos) info.ids.push_back("P");
    else info.ids.push_back("");

    vector<double> rotation;
    rotation.resize(9, 0);

    string iop;
    if (!file.Find("ImageOrientationPatient", iop)) {
      iop = "1\\0\\0\\0\\1\\0";
    }
  
    for (int i = 0; i < 6; ++i) {
      double f = stof(iop, i);
      // round off
      rotation[i] = static_cast<double>(static_cast<int>(f * 10000 + 0.5)) / 10000;
    }

    rotation[6] = rotation[1]*rotation[5] - rotation[2]*rotation[4];
    rotation[7] = rotation[2]*rotation[3] - rotation[0]*rotation[5];
    rotation[8] = rotation[0]*rotation[4] - rotation[1]*rotation[3];
  
    vector<vector <double> >::iterator pos;
    pos = find(orientations.begin(), orientations.end(), rotation);
    if (pos != orientations.end()) {
      info.orientNumber = distance(orientations.begin(), pos);
    }
    else {
      info.orientNumber = orientations.size();
      orientations.push_back(rotation);
    }

    info.ids.push_back(itos(info.orientNumber + 1));


    v.push_back(info);
  }
  return v;
}

set<VolId>
SeriesHandler::GetVolIds() const
{
  std::set<VolId> volSet;
  for(FileMapType::const_iterator it = mFileMap.begin(); it != mFileMap.end(); ++it) {
    for (VolListType::const_iterator vit = it->second.begin(); 
      vit != it->second.end(); ++vit) { 
      volSet.insert(*vit);
    }
  }
  return volSet;
}

// returns volid + last file with that id
map<VolId, string>
SeriesHandler::GetVolIdsAndFiles() const
{
  map<VolId, string> volMap;

  FileMapType::const_iterator it = mFileMap.begin();
  while (it != mFileMap.end()) {
    VolListType::const_iterator vit = it->second.begin();
    while (vit != it->second.end()) {
      volMap[*vit] = it->first;
      ++vit;
    }
    ++it;
  }
  return volMap;
}

bool
SeriesHandler::GetFirstFilename(const VolId& id, string& str) const
{
  FileMapType::const_iterator it = mFileMap.begin();
  while (it != mFileMap.end()) {
    VolListType::const_iterator vit = it->second.begin();
    while (vit != it->second.end()) {
      if(*vit == id) {
        str = it->first;
        return true;  //1;
      }
      ++vit;
    }
    ++it;
  }
  return false;  //0;
}


vector<double>
SeriesHandler::GetRotationMatrix(const VolId& id)
{
  vector<double> r(9,0);
  try {
    r = orientations.at(id.orientNumber);
  }
  catch (const std::exception& error) {
    wxLogError(_T("orientNumber: %d; number of orientations: %d"), id.orientNumber, orientations.size());
    wxLogError(wxString(error.what(), wxConvLocal));
  }
  return r;
}


SyngoHandler::SyngoHandler(const string& seriesUid):SeriesHandler(seriesUid)
{
}

int
SyngoHandler::ReadCSAImageHeader(const std::string& tag, std::string& value) const
{
  if (mFileMap.empty()) return 0;
  DicomFile dicomFile((*mFileMap.begin()).first.c_str());
  int err = dicomFile.ReadCSAImageHeader(tag, value);
  return err;
}

int
SyngoHandler::ReadCSASeriesHeader(const std::string& tag, std::string& value) const
{
  if (mFileMap.empty()) return 0;
  DicomFile dicomFile((*mFileMap.begin()).first.c_str());
  return dicomFile.ReadCSASeriesHeader(tag, value);
}

double
SyngoHandler::GetSliceDuration() const
{
  double duration = 0;
  string str;
  int err = ReadCSAImageHeader("SliceMeasurementDuration", str);
  if (err > 0) 
//    duration = boost::lexical_cast<double> (str.c_str());
    duration = stof(str);
  
  return duration;

}

AoCode
SyngoHandler::GetAcquisitionOrder() const
{
  string protocol;
  int err = ReadCSASeriesHeader("MrProtocol", protocol);
  
  if (err == -1) err = ReadCSASeriesHeader("MrPhoenixProtocol", protocol);

  if (err > 0) {
    string::size_type pos = protocol.find("sSliceArray.ucMode");
    pos = protocol.find('=', pos);
    pos = protocol.find_first_not_of(' ', pos+1);
    if (pos != string::npos) {
      string acquisition = protocol.substr(pos, 3);
      if (acquisition == "0x1") return ASCENDING;
      if (acquisition == "0x2") return DESCENDING;
      if (acquisition == "0x4") {
        if (GetNumberOfSlices() % 2 == 0) {
          return INTERLEAVED_EVEN;
        }
        else {
          return INTERLEAVED_ODD;
        }
      }
      else
        return OTHER;
    }
  }
  return UNKNOWN;
}

bool
SyngoHandler::IsMoCo() const
{
  if (mFileMap.empty()) return 0;
  string imageType;
  DicomFile dicomFile((*mFileMap.begin()).first.c_str());
  dicomFile.Find("ImageType", imageType);
  if (imageType.find("MOCO") != string::npos) 
    return true;
  else return false; 

}

void 
SeriesHandler::GetImageComments(std::vector<string>& comments) const 
{
  map<VolId, string> volFileMap = GetVolIdsAndFiles();
  map<VolId, string>::iterator it = volFileMap.begin();
  map<VolId, string>::iterator it_end = volFileMap.end();

  while (it != it_end) {
    string comment;
    DicomFile file(it->second.c_str());
    file.Find("ImageComments", comment);
    comments.push_back(comment);
    ++it;
  }

}


SeriesHandler::VolListType
SyngoHandler::ReadVolIds(DicomFile& file)
{
  VolListType v = SeriesHandler::ReadVolIds(file);

  // needed for DTI
  string str;
  file.Find("SequenceName", str);
  string::size_type pos = str.find('#');
  if (pos != string::npos) {
    int zeros_to_add = 4 + pos - str.length();
    str.insert(pos+1, zeros_to_add, '0');

    pos = str.find('b') + 1;
    if (pos != string::npos) {
      string::size_type pos2 = str.find_first_not_of("0123456789", pos);
      zeros_to_add = 4 + pos - pos2;
      str.insert(pos, zeros_to_add, '0');
    }


  }
  str = RemoveInvalidChars(str);
  v.front().ids.push_back(str);

  int an;
  file.Find("AcquisitionNumber", an);
  
  int width = 4;
  str = itos(an, width);
  v.front().ids.push_back(str);

  str.clear();  
  file.ReadCSAImageHeader("UsedChannelMask", str);
  v.front().ids.push_back(RemoveInvalidChars(str));

  return v;
}

bool
SyngoHandler::IsDti() const
{
  string bValue;
  int err = ReadCSAImageHeader("B_value", bValue);
  return (err > 0);

}

GradientInfo
SyngoHandler::GetGradientInfo()
{

  GradientInfo info;

  map<VolId, string> volFileMap = GetVolIdsAndFiles();
  map<VolId, string>::iterator it = volFileMap.begin();
  map<VolId, string>::iterator it_end = volFileMap.end();

  DicomFile first_file(it->second.c_str());
//  bool needsRotation = !first_file.HasSiemensMrHeader();
  // Rotation always needed!

  string version;
  first_file.Find("SoftwareVersion", version);
  if (version.find("B15") != string::npos) 
    wxLogWarning(_T("Warning: Polarity of the bvecs may be wrong for VB15 data."));
  if (version.find("B13") != string::npos)
    wxLogWarning(_T("Warning: bvecs are sometimes wrong for VB13 data, due to a bug in the algorithm by which Siemen's converted the B_matrix to a principle eigendirection.  Use with caution."));

  while (it != it_end) {
    string bValue;
    string directionality;
    vector<string> gradDir;

    DicomFile file(it->second.c_str());

    int err = file.ReadCSAImageHeader("B_value", bValue);

    if (err > 0) info.values.push_back(stof(bValue));

    err = file.ReadCSAImageHeader("DiffusionGradientDirection", gradDir);
    if ((err < 0) || (gradDir.size() != 3)) {
      info.xGrads.push_back(0);
      info.yGrads.push_back(0);
      info.zGrads.push_back(0);
    }

    else {
      info.xGrads.push_back(stof(gradDir.at(0)));
      info.yGrads.push_back(stof(gradDir.at(1)));
      info.zGrads.push_back(stof(gradDir.at(2)));
    }

    ++it;
  }

  vector<double> r = GetRotationMatrix(volFileMap.begin()->first);
  RotateGradInfo(info, r);

  return info;
}


vector<string>
NumarisMosaicHandler::GetStringInfo()
{
  vector<string> info = SeriesHandler::GetStringInfo();
  info.push_back(string("Mosaic rows: ") + itos(GetRows()));
  info.push_back(string("Mosaic columns: ") + itos(GetColumns()));
  
  string a_order;
  switch (GetAcquisitionOrder()) {
    case ASCENDING :
      a_order = "Ascending";
      break;

    case DESCENDING :
      a_order = "Descending";
      break;

    case INTERLEAVED_ODD :
      a_order = "Interleaved";
      break;

    case OTHER :
      a_order = "Other";
      break;

    case UNKNOWN :
    default:
      a_order = "Unknown";
      break;
  }
  info.push_back(string("Acquisition order: ") + a_order);

  return info;
}



NumarisMosaicHandler::NumarisMosaicHandler(const std::string& seriesUid)
: SeriesHandler(seriesUid)
{
}

SeriesHandler::VolListType
NumarisMosaicHandler::ReadVolIds(DicomFile& file) 
{
  VolListType v = SeriesHandler::ReadVolIds(file);
  string str;
  file.Find("InstanceNumber", str);
  v.front().ids.push_back(str);
  return v;
}

AoCode
NumarisMosaicHandler::GetAcquisitionOrder() const
{
  Dictionary* Numaris = Numaris_Dictionary::Instance();
  string acquisition_order;
  Find(Numaris->Lookup("ORDER_OF_SLICES"), acquisition_order);
  if (acquisition_order == "ASCENDING") return ASCENDING;
  if (acquisition_order == "DECREASING") return DESCENDING;
  if (acquisition_order == "INTERLEAVED") return INTERLEAVED_ODD;
  return OTHER;
}

double
NumarisMosaicHandler::GetVolumeInterval() const
{
  double tr;
  Find("RepetitionTime", tr);
  return tr/1000.0;
}

vector<double>
NumarisMosaicHandler::GetSliceOrder()
{
  Dictionary* Numaris = Numaris_Dictionary::Instance();
  int n_slices = GetNumberOfSlices();
  vector<double> slice_order;
  slice_order.reserve(n_slices);

  double delta_z = GetVoxelSize().back();

  AoCode acquisition_order = GetAcquisitionOrder();

  switch (acquisition_order) {

    case INTERLEAVED_ODD: 
      for (int i = n_slices; i > 0; i-=2)
        slice_order.push_back(delta_z * i);
      for (int i = (n_slices - 1); i > 0; i-=2)
        slice_order.push_back(delta_z * i);
        break;
  
    case DESCENDING: 
      for (int i = 1; i <= n_slices; ++i) 
        slice_order.push_back(i * delta_z);

        break;

    default: 
      for (int i = n_slices; i > 0; --i)
        slice_order.push_back(i * delta_z);
  }

  return slice_order;
}

int
NumarisMosaicHandler::GetRows()
{
  Dictionary* Numaris = Numaris_Dictionary::Instance();
  int mosaic_size = 0;
  Find(Numaris->Lookup("BASE_RAW_MATRIX_SIZE"), mosaic_size);
  return mosaic_size;
}

int NumarisMosaicHandler::GetColumns()
{
  return GetRows();
}

vector<double>
NumarisMosaicHandler::GetVoxelSize()
{
  vector<double> voxel_size(3, 0);
  
  string s;
  if(Find("PixelSpacing", s)) {
    voxel_size[0] = stof(s, 0);
    voxel_size[1] = stof(s, 1);
  }
  
  // Pixel spacing is reported incorrectly
  // for mosaic files -- must correct
  int mosaic_size = GetRows();
  double image_size;
  Find("ImageColumns", image_size);
  voxel_size[0] *= image_size/mosaic_size;
  Find("ImageRows", image_size);
  voxel_size[1] *= image_size/mosaic_size;
  
  Find("SliceThickness", voxel_size[2]);
  double slice_spacing = 0;
  Find("SliceSpacing", slice_spacing);
  voxel_size[2] += slice_spacing;

  return voxel_size;
}

int
NumarisMosaicHandler::GetNumberOfSlices() const
{
  Dictionary* Numaris = Numaris_Dictionary::Instance();
  int slices = 0;
  Find(Numaris->Lookup("NUMBER_OF_SLICES_CURRENT"), slices);
  return slices;
}

SyngoMosaicHandler::SyngoMosaicHandler(const std::string& seriesUid)
: SyngoHandler(seriesUid)
{
}


SeriesHandler::VolListType
SyngoMosaicHandler::ReadVolIds(DicomFile& file) 
{
  VolListType v = SyngoHandler::ReadVolIds(file);

  string description;
  file.Find("SeriesDescription", description);
  if ((description.find("EvaSeries") != string::npos) ||
    (description.find("Superimposed Data") != string::npos)) {

    string str;
    int in;
    file.Find("InstanceNumber", in);
  
    int width = 4;
    str = itos(in, width);
    v.front().ids.push_back(str);
  }

  return v;
}


/*
vector<double>
SyngoMosaicHandler::GetSliceOrder()
{
  int n_slices = GetNumberOfSlices();
  vector<double> slice_order;
  slice_order.reserve(n_slices);

  double delta_z = GetVoxelSize().back();

  bool foot_to_head = true;
  string protocol;
  int err = ReadCSASeriesHeader("MrProtocol", protocol);
  if (err == -1) err = ReadCSASeriesHeader("MrPhoenixProtocol", protocol);

  if (err > 0) {
    string::size_type pos = protocol.find("sSliceArray.ucImageNumbTra");
    foot_to_head = (pos == string::npos);
  }

  if (foot_to_head) {
    for (int i = 1; i <= n_slices; ++i) 
      slice_order.push_back(i * delta_z);
  }
  else {
    for (int i = n_slices; i > 0; --i)
      slice_order.push_back(i * delta_z);
  }

  return slice_order;
}
*/
int
SyngoMosaicHandler::GetRows()
{
  string pe_direction;
  Find("PhaseEncodingDirection", pe_direction);
  double fov_rows;
  if (pe_direction == "ROW")
    fov_rows = GetRoFov();
  else
    fov_rows = GetPhaseFov();

  double pixel_size_y = GetVoxelSize()[1];
  return static_cast<int>(floor(fov_rows/pixel_size_y + 0.5));

}

int
SyngoMosaicHandler::GetColumns()
{
  string pe_direction;
  Find("PhaseEncodingDirection", pe_direction);
  double fov_cols;
  if (pe_direction == "COL")
    fov_cols = GetRoFov();
  else
    fov_cols = GetPhaseFov();

  double pixel_size_x = GetVoxelSize()[0];
  return static_cast<int>(floor(fov_cols/pixel_size_x + 0.5));

}

double
SyngoMosaicHandler::GetPhaseFov() const
{
  double phase_fov = 256;
  string protocol;
  int err = ReadCSASeriesHeader("MrProtocol", protocol);
  if (err == -1) err = ReadCSASeriesHeader("MrPhoenixProtocol", protocol);

  if (err > 0) {
    string::size_type pos = protocol.find("sSliceArray.asSlice[0].dPhaseFOV");
    std::stringstream ss(protocol.substr(pos, 256));
    ss.ignore(256, '=');
    ss >> phase_fov;
  }
  return phase_fov;
}

double
SyngoMosaicHandler::GetRoFov() const
{
  double ro_fov = 256;
  string protocol;
  int err = ReadCSASeriesHeader("MrProtocol", protocol);
  if (err == -1) err = ReadCSASeriesHeader("MrPhoenixProtocol", protocol);

  if (err > 0) {
    string::size_type pos = protocol.find("sSliceArray.asSlice[0].dReadoutFOV");
    std::stringstream ss(protocol.substr(pos, 256));
    ss.ignore(256, '=');
    ss >> ro_fov;
  }
  return ro_fov;
}

int
SyngoMosaicHandler::GetNumberOfSlices() const
{
  string slices;
  int n_slices = 1;

  int err = ReadCSAImageHeader("NumberOfImagesInMosaic", slices);
  if (err == 0) wxLogError(_T("Unable to find number of slices"));
  else if (err == -1) wxLogError(_T("Number of slices not provided"));
  else n_slices = stoi(slices);
  
//boost::lexical_cast<int> (slices.c_str());
  
  return n_slices;

}

/*
vector<double>
SyngoMosaicHandler::GetIppForFirstSlice(const VolId& id)
{
  if (position.find(id) != position.end()) return position[id]; 
  
  vector<double> ipp1;
  ipp1.resize(3,0);

  vector<double> center(3,0);
  string protocol;
  int err = ReadCSASeriesHeader("MrProtocol", protocol);
  if (err == -1) err = ReadCSASeriesHeader("MrPhoenixProtocol", protocol);
  
  if (err <= 0) return ipp1; 
    
  double value;
  string::size_type pos = protocol.find("sSliceArray.asSlice[0].sPosition.dSag");
  if (pos != string::npos) {
    stringstream ss(protocol.substr(pos, 256));
    ss.ignore(256, '=');
    ss >> value;
    center[0] = value;
  }
  pos = protocol.find("sSliceArray.asSlice[0].sPosition.dCor");
  if (pos != string::npos) {
    stringstream ss(protocol.substr(pos, 256));
    ss.ignore(256, '=');
    ss >> value;
    center[1] = value;
  }
  if (pos != string::npos) {
    pos = protocol.find("sSliceArray.asSlice[0].sPosition.dTra");
    stringstream ss(protocol.substr(pos, 256));
    ss.ignore(256, '=');
    ss >> value;
    center[2] = value;
  }

// nb - if we ever had a mosaic for a 3D sequence, we'd have to use the full 3x3 matrix
  vector<double> rotation = orientations.at(0);
  vector<double> voxel_size = GetVoxelSize();
  vector<double> offset(2,0);
  offset[0] = -0.5*GetColumns()*voxel_size[0];
  offset[1] = -0.5*GetRows()*voxel_size[1];
  // offset[slice] = 0;

  ipp1[0] = center[0] + rotation[0]*offset[0] + rotation[3]*offset[1];
  ipp1[1] = center[1] + rotation[1]*offset[0] + rotation[4]*offset[1]; 
  ipp1[2] = center[2] + rotation[2]*offset[0] + rotation[5]*offset[1];

  position[id] = ipp1;
  return ipp1;

}
*/
 
vector<string>
SyngoMosaicHandler::GetStringInfo()
{
  vector<string> info = SeriesHandler::GetStringInfo();
  info.push_back(string("Mosaic rows: ") + itos(GetRows()));
  info.push_back(string("Mosaic columns: ") + itos(GetColumns()));
  
  string a_order;
  switch (GetAcquisitionOrder()) {
    case ASCENDING :
      a_order = "Ascending";
      break;

    case DESCENDING :
      a_order = "Descending";
      break;

    case INTERLEAVED_ODD :
      a_order = "Interleaved Odd";
      break;

    case INTERLEAVED_EVEN :
      a_order = "Interleaved Even";
      break;

    case OTHER :
      a_order = "Other";
      break;

    case UNKNOWN :
    default:
      a_order = "Unknown";
      break;
  }
  info.push_back(string("Acquisition order: ") + a_order);


  DicomFile dicomFile((*mFileMap.begin()).first.c_str());
  vector<string> temp;
  dicomFile.ReadCSAImageHeader("MosaicRefAcqTimes", temp);
  info.insert(info.end(), temp.begin(), temp.end());


  return info;
}

GeDtiRbHandler::GeDtiRbHandler(const std::string& seriesUid)
: SeriesHandler(seriesUid)
{
}

SeriesHandler::VolListType
GeDtiRbHandler::ReadVolIds(DicomFile& file)
{ 
  VolListType v = SeriesHandler::ReadVolIds(file);
  // Can't call default ReadVolId because of problem with trigger time.

  string acquisition_number;
  if (!file.Find("AcquisitionNumber", acquisition_number))
    wxLogError(_T("Acquisition number not found"));
  v.front().ids.push_back(acquisition_number);

  int image_number, number_of_slices;
  if (!file.Find("InstanceNumber", image_number))
    wxLogError(_T("Instance number not found"));

  Dictionary* Excite = Excite_Dictionary::Instance();
  if (!file.Find(Excite->Lookup("Locations_in_acquisition"), number_of_slices))
    wxLogError(_T("Locations_in_acquisition not found"));

  int vol_no = static_cast<int>(floor(static_cast<double>((image_number - 1))/number_of_slices) + 1);

  v.front().ids.push_back(itos(vol_no, 3));

  return v;

}

GradientInfo
GeDtiRbHandler::GetGradientInfo()
{

  GradientInfo info;

  int gradfileno;
  Dictionary* Excite = Excite_Dictionary::Instance();

  double bvalue;
  Find(Excite->Lookup("User_data_9"), bvalue);

  if (Find(Excite->Lookup("User_data_11"), gradfileno)) {
    wxString gradfilename = wxString::Format(_T("dwepi.%d.grads"), gradfileno);
    
    wxFileName gradfile(wxStandardPaths::Get().GetDataDir());
    gradfile.SetFullName(gradfilename);

    std::ifstream input(gradfile.GetFullPath().mb_str(wxConvLocal));

    double value;
    input >> value;
    while (input.good()) {
      info.xGrads.push_back(value);
      input >> value;
      info.yGrads.push_back(value);
      input >> value;
      info.zGrads.push_back(value);
      info.values.push_back(bvalue);
      input >> value;
    }

    // GE -- do not correct
    // assuming one orientation in series
    //vector<double> r = orientations.at(0);
    //RotateGradInfo(info, r);

  }

  return info;
}

GeEpiHandler::GeEpiHandler(const std::string& seriesUid)
: SeriesHandler(seriesUid)
{
}

SeriesHandler::VolListType
GeEpiHandler::ReadVolIds(DicomFile& file)
{ 
  VolListType v = SeriesHandler::ReadVolIds(file);

  int image_number, number_of_slices;

  if (!file.Find("InstanceNumber", image_number))
    wxLogError(_T("Instance number not found"));

  Dictionary* Excite = Excite_Dictionary::Instance();
  if (!file.Find(Excite->Lookup("Locations_in_acquisition"), number_of_slices))
    wxLogError(_T("Locations_in_acquisition not found"));

  int vol_no = static_cast<int>(floor(static_cast<double>((image_number - 1))
    /number_of_slices) + 1);

  v.front().ids.push_back(itos(vol_no, 3));

  return v;

}

GeDti2Handler::GeDti2Handler(const std::string& seriesUid)
: GeEpiHandler(seriesUid)
{
}

SeriesHandler::VolListType
GeDti2Handler::ReadVolIds(DicomFile& file)
{ 
  VolListType v = SeriesHandler::ReadVolIds(file);
  Dictionary* Excite = Excite_Dictionary::Instance();

  int image_number, number_of_slices;

  if (!file.Find("InstanceNumber", image_number))
    wxLogError(_T("Instance number not found"));

  if (!file.Find(Excite->Lookup("Locations_in_acquisition"), number_of_slices))
    wxLogError(_T("Locations_in_acquisition not found"));

  int vol_no = static_cast<int>(floor(static_cast<double>((image_number - 1))
    /number_of_slices) + 1);

  v.front().ids.push_back(itos(vol_no, 3));

  vector<double> g_value;
  string b_direction;
  double value;
  file.Find(Excite->Lookup("User_data_20"), value);
  g_value.push_back(value);

  file.Find(Excite->Lookup("User_data_21"), value);
  g_value.push_back(value);

  file.Find(Excite->Lookup("User_data_22"), value);
  g_value.push_back(value);

  int g_number;
  vector<vector <double> >::iterator pos;
  pos = find(gradients.begin(), gradients.end(), g_value);
  if (pos != gradients.end()) {
    g_number = distance(gradients.begin(), pos);
  }
  else {
    g_number = gradients.size();
    gradients.push_back(g_value);
  }

  v.front().ids.push_back(itos(g_number, 3));

  return v;

}

GradientInfo
GeDti2Handler::GetGradientInfo()
{

  GradientInfo info;

  Dictionary* Excite = Excite_Dictionary::Instance();

  map<VolId, string> volFileMap = GetVolIdsAndFiles();
  map<VolId, string>::iterator it = volFileMap.begin();
  map<VolId, string>::iterator it_end = volFileMap.end();

  while (it != it_end) {

    //vector<string> test = it->first.ids;
    //for (int t = 0; t < test.size(); ++t)
    //  wxLogMessage(test[t].c_str());
    
    info.values.push_back(0); // bvalue not found in these files.

    DicomFile file(it->second.c_str());

    double value;
    int err = file.Find(Excite->Lookup("User_data_20"), value);
    if (err > 0) info.xGrads.push_back(value);

    err = file.Find(Excite->Lookup("User_data_21"), value);
    if (err > 0) info.yGrads.push_back(value);

    err = file.Find(Excite->Lookup("User_data_22"), value);
    if (err > 0) info.zGrads.push_back(value);

    ++it;
  }

  // GE -- do not correct!
//  vector<double> r = GetRotationMatrix(volFileMap.begin()->first);
//  RotateGradInfo(info, r);

  return info;


}

AchievaDtiHandler::AchievaDtiHandler(const std::string& seriesUid)
: SeriesHandler(seriesUid)
{
}

SeriesHandler::VolListType
AchievaDtiHandler::ReadVolIds(DicomFile& file)
{ 
  VolListType v = SeriesHandler::ReadVolIds(file);
  Dictionary* Achieva = PhilipsMr_Dictionary::Instance();

  vector<double> g_value;
  string b_direction;

  double value;
  file.Find(Achieva->Lookup("Diffusion_B-Factor"), value);
  g_value.push_back(value);

  file.Find(Achieva->Lookup("Diffusion_Direction_RL"), value);
  g_value.push_back(value);

  file.Find(Achieva->Lookup("Diffusion_Direction_AP"), value);
  g_value.push_back(value);

  file.Find(Achieva->Lookup("Diffusion_Direction_FH"), value);
  g_value.push_back(value);

  int g_number;
  vector<vector <double> >::iterator pos;
  pos = find(gradients.begin(), gradients.end(), g_value);
  if (pos != gradients.end()) {
    g_number = distance(gradients.begin(), pos);
  }
  else {
    g_number = gradients.size();
    gradients.push_back(g_value);
  }

  v.front().ids.push_back(itos(g_number, 3));

  return v;

}

GradientInfo
AchievaDtiHandler::GetGradientInfo()
{

  GradientInfo info;

  Dictionary* Achieva = PhilipsMr_Dictionary::Instance();

  map<VolId, string> volFileMap = GetVolIdsAndFiles();
  map<VolId, string>::iterator it = volFileMap.begin();
  map<VolId, string>::iterator it_end = volFileMap.end();

  while (it != it_end) {

    //vector<string> test = it->first.ids;
    //for (int t = 0; t < test.size(); ++t)
    //  wxLogMessage(test[t].c_str());

    DicomFile file(it->second.c_str());

    double value;
    file.Find(Achieva->Lookup("Diffusion_B-Factor"), value);
    info.values.push_back(value);

    file.Find(Achieva->Lookup("Diffusion_Direction_RL"), value);
    info.xGrads.push_back(value);

    file.Find(Achieva->Lookup("Diffusion_Direction_AP"), value);
    info.yGrads.push_back(value);

    file.Find(Achieva->Lookup("Diffusion_Direction_FH"), value);
    info.zGrads.push_back(value);

    ++it;
  }

  vector<double> r = GetRotationMatrix(volFileMap.begin()->first);

  RotateGradInfo(info, r);
  return info;


}

void 
jcs::Normalize(GradientInfo& info)
{
  vector<double> norms;
  for (unsigned int i = 0; i < info.xGrads.size(); ++i) {
    double norm = info.xGrads.at(i) * info.xGrads.at(i)
          + info.yGrads.at(i) * info.yGrads.at(i)
          + info.zGrads.at(i) * info.zGrads.at(i);
    norm = sqrt(norm);

    norms.push_back(norm);
  }
  for (unsigned int i = 0; i < info.xGrads.size(); ++i) {
    if ((norms.at(i) != 1.0) && (norms.at(i) != 0)) {
      info.xGrads.at(i) /= norms.at(i);
      info.yGrads.at(i) /= norms.at(i);
      info.zGrads.at(i) /= norms.at(i);
  
    }
  }
}

// nb - inverse(r)*info.xGrads
void
jcs::RotateGradInfo(GradientInfo& info, vector<double>& r)
{
  vector<double> temp (3,0);
  for (unsigned int i = 0; i < info.xGrads.size(); ++i) {
    temp[0] = r[0]*info.xGrads[i] + r[1]*info.yGrads[i] + r[2]*info.zGrads[i];
    temp[1] = r[3]*info.xGrads[i] + r[4]*info.yGrads[i] + r[5]*info.zGrads[i];
    temp[2] = r[6]*info.xGrads[i] + r[7]*info.yGrads[i] + r[8]*info.zGrads[i];

    info.xGrads[i] = temp[0];
    info.yGrads[i] = temp[1];
    info.zGrads[i] = temp[2];
  }

  return;
}


EnhancedMrHandler::EnhancedMrHandler(const std::string& seriesUid)
: SeriesHandler(seriesUid)
{
}

string
EnhancedMrHandler::GetImagePositionPatient(DicomFile& dfile, int frame)
{
  return ipps.at(frame);
}

int
EnhancedMrHandler::GetRescaleSlopeAndIntercept(DicomFile& dfile, double& slope, double& intercept, int frame) const
{
  // use a private vector
  // if it's empty, fill it on the first call
  // or fill it in getvolids

  slope = slopes.at(frame);
  intercept = intercepts.at(frame);

  return 1;
}


vector<string>
EnhancedMrHandler::GetStringInfo()
{
  vector<string> info = SeriesHandler::GetStringInfo();
  return info;
}

SeriesHandler::VolListType 
EnhancedMrHandler::ReadVolIds(DicomFile& file)
{
  string imageType;
  file.Find("ImageType", imageType);
  bool isdti = (imageType.find("DIFFUSION") != string::npos);

  VolListType v;

  string sequence;
  vector<string> dips;
  file.Find("DimensionIndexSequence", sequence);
  file.FindInSequence(sequence, "DimensionIndexPointer", dips);

  unsigned int n_indices = dips.size();

  sequence.clear();
  vector<string> fc_sequence;
  vector<string> po_sequence;
  vector<string> pp_sequence;
  vector<string> diff_sequence;
  vector<string> pm_sequence;
  vector<string> pvt_sequence;

  // This is a type 2 element. Could probably comment this out?
  if (file.Find("SharedFunctionalGroupsSequence", sequence)) {
    file.FindInSequence(sequence, "PlaneOrientationSequence", po_sequence);
    file.FindInSequence(sequence, "PlanePositionSequence", pp_sequence);
  }

  sequence.clear();

  if (!file.Find("PerFrameFunctionalGroupsSequence", sequence)) {
    wxLogError(_T("Unable to find PerFrameFunctionalGroupsSequence"));
    return v;
  }

  
  if (!file.FindInSequence(sequence, "FrameContentSequence", fc_sequence)) {
    wxLogError(_T("Unable to find FrameContentSequence"));
    return v;
  }

  if (po_sequence.size() == 0) {
    if (!file.FindInSequence(sequence, "PlaneOrientationSequence", po_sequence)) {
      wxLogError(_T("Unable to find PlaneOrientationSequence"));
      return v;
    }
  }

  if (pp_sequence.size() == 0) {
    if (!file.FindInSequence(sequence, "PlanePositionSequence", pp_sequence)) {
      wxLogError(_T("Unable to find PlanePositionSequence"));
      return v;
    }
  }
  file.FindInSequence(sequence, "PixelMeasuresSequence", pm_sequence);

  if (isdti) 
    file.FindInSequence(sequence, "MrDiffusionSequence", diff_sequence);

  int pvt_found = file.FindInSequence(sequence, "PixelValueTransformationSequence", pvt_sequence);

  vector<string>::iterator fc_it = fc_sequence.begin();
  vector<string>::iterator po_it = po_sequence.begin();
  vector<string>::iterator pp_it = pp_sequence.begin();
  vector<string>::iterator diff_it = diff_sequence.begin();
  vector<string>::iterator pvt_it = pvt_sequence.begin();

  vector<vector <double> > rotations;
  vector<vector <int> > divs(n_indices);
  vector<int> inStackPositions;
  vector<string> bvecs;
  vector<string> bvals;

  // Assumes pixel spacing same in every frame -- not, strictly speaking,
  // a safe assumption
  vector<string> vals;
  if (!pm_sequence.empty()) {
    file.FindInSequence(pm_sequence.front(), "PixelSpacing", vals);
    if (!vals.empty()) pixelSpacing = vals.front();
  }

  while (fc_it != fc_sequence.end()) {

    vals.clear();
    file.FindInSequence(*fc_it, "DimensionIndexValues", vals);
    for (unsigned int i = 0; i < n_indices; ++i) 
      divs.at(i).push_back(stoi(vals.front(), i));
    vals.clear();

    if (file.FindInSequence(*fc_it, "InStackPositionNumber", vals)) {
      inStackPositions.push_back(stoi(vals.front()));
      vals.clear();
    }
    else inStackPositions.push_back(0);

    double rescale_val;
    if (pvt_found && file.FindInSequence(*pvt_it, "RescaleIntercept", vals)) {
      intercepts.push_back(stof(vals.front()));
      vals.clear();
    }
    else if (file.Find("RescaleIntercept", rescale_val))
      intercepts.push_back(rescale_val);
    else intercepts.push_back(0);

    if (pvt_found && file.FindInSequence(*pvt_it, "RescaleSlope", vals)) {
      slopes.push_back(stof(vals.front()));
      vals.clear();
    }
    else if (file.Find("RescaleSlope", rescale_val))
      intercepts.push_back(rescale_val);
    else slopes.push_back(1);

    if (file.FindInSequence(*pp_it, "ImagePositionPatient", vals))
      ipps.push_back(vals.front());
    else ipps.push_back("0/0/0");
    vals.clear();

    if (!file.FindInSequence(*po_it, "ImageOrientationPatient", vals))
      vals.push_back("1/0/0/0/1/0");
    vector<double> rotation;
    rotation.resize(9, 0);

    for (int i = 0; i < 6; ++i) {
      double f = stof(vals.front(), i);
      // round off
      rotation[i] = static_cast<double>(static_cast<int>(f * 100000 + 0.5)) / 100000;
    }

    rotation[6] = rotation[1]*rotation[5] - rotation[2]*rotation[4];
    rotation[7] = rotation[2]*rotation[3] - rotation[0]*rotation[5];
    rotation[8] = rotation[0]*rotation[4] - rotation[1]*rotation[3];

    rotations.push_back(rotation);

    if (isdti && diff_sequence.size() != 0) {
      vals.clear();
      file.FindInSequence(*diff_it, "DiffusionBValue", vals);
      if (vals.size() != 0) bvals.push_back(vals.front());
      else bvals.push_back("-1");
      vals.clear();
      file.FindInSequence(*diff_it, "DiffusionDirectionality", vals);
      if (vals.front() == "DIRECTIONAL") {
        vals.clear();
        file.FindInSequence(*diff_it, "DiffusionGradientDirectionSequence", vals);
        vector<string> dgo;
        file.FindInSequence(vals.front(), "DiffusionGradientOrientation", dgo);
        bvecs.push_back(dgo.front());
      }
      else bvecs.push_back("0/0/0");
      ++diff_it;
    };

    ++fc_it;
    if (po_sequence.size() == fc_sequence.size()) ++po_it;
    if (pp_sequence.size() == fc_sequence.size()) ++pp_it;
    wxTheApp->Yield();

  }


  vector<bool> use_index;
  for (unsigned int i = 0; i < divs.size(); ++i) {
    // Don't use if indices simply reflect inStackPositions
    bool use = divs.at(i) != inStackPositions;
    // or if they just count up from 1 to nslices
    unsigned int j = 0;
    bool match = use;
    while (match && j < divs.at(i).size()) {
      match = (divs.at(i).at(j) == j+1);
      ++j;
    }
    use &= !match;
    use_index.push_back(use);

  }

  for (unsigned int i = 0; i < rotations.size(); ++i) {
    VolId info;
    info.ids.push_back(GetSeriesUid());

    for (unsigned int j = 0; j < divs.size(); ++j) 
      if (use_index.at(j)) info.ids.push_back(itos(divs.at(j).at(i), 3));

    vector<vector <double> >::iterator pos;
    pos = find(orientations.begin(), orientations.end(), rotations.at(i));

    if (pos != orientations.end()) {
      info.orientNumber = distance(orientations.begin(), pos);
    }
    else {
      info.orientNumber = orientations.size();
      orientations.push_back(rotations.at(i));
    }


    v.push_back(info);
    if (isdti && diff_sequence.size() != 0) {
      bvalMap[info] = bvals.at(i);
      bvecMap[info] = bvecs.at(i);
    }

    wxTheApp->Yield();
  }
  return v;
}

GradientInfo 
EnhancedMrHandler::GetGradientInfo()
{
  GradientInfo info;

  set<VolId> vols = GetVolIds();
  vector<double> r = GetRotationMatrix(*vols.begin());

  for (set<VolId>::iterator it = vols.begin(); it != vols.end(); ++it) {
    info.values.push_back(stof(bvalMap[*it]));
    info.xGrads.push_back(stof(bvecMap[*it], 0));
    info.yGrads.push_back(stof(bvecMap[*it], 1));
    info.zGrads.push_back(stof(bvecMap[*it], 2));
  };

  wxLogWarning(_T("Warning: bvecs have NOT been verified as correct for enhanced MR DICOM files. If you would like to help with this, please email jolinda@uoregon.edu."));

  RotateGradInfo(info, r);

  return info;
}

bool 
EnhancedMrHandler::IsDti() const 
{
  string imageType;
  Find("ImageType", imageType);
  if (imageType.find("DIFFUSION") != string::npos)
    return true;
  else return false;
}
  
vector<double> 
EnhancedMrHandler::GetVoxelSize() 
{
  vector<double> voxel_size(3, 0);

  if (pixelSpacing.empty()) Find("PixelSpacing", pixelSpacing);
  if (!pixelSpacing.empty()) {
    voxel_size[0] = stof(pixelSpacing, 0);
    voxel_size[1] = stof(pixelSpacing, 1);
  }
  Find("SliceSpacing", voxel_size[2]);
  if (voxel_size[2] == 0)
    Find("SliceThickness", voxel_size[2]);
    
  return voxel_size;
}

double
SyngoHandler::GetVolumeInterval() const
{
  map<VolId, vector<double> > vtmap;
  FileMapType::const_iterator it = mFileMap.begin();
  while (it != mFileMap.end()) { 
    DicomFile file(it->first.c_str());
    string str;
    int err = file.ReadCSAImageHeader("TimeAfterStart", str);
    if (err <= 0) return 0;
    double time = stof(str);
    vtmap[it->second.front()].push_back(time);
    ++it;
  }

  if (vtmap.size() <= 1) return 0;

  vector<double> vol_times;
  map<VolId, vector<double> >::iterator vit = vtmap.begin();
  while (vit != vtmap.end()) { 
    vol_times.push_back(*min_element(vit->second.begin(), vit->second.end()));
    ++vit;
  }

  sort(vol_times.begin(), vol_times.end());

  return (vol_times[1] - vol_times[0]);

}
