// file kernel/n/alpha/mul_n2.S: O(n^2) multiplication of natural integers
/*-----------------------------------------------------------------------+
 |  Copyright 2005-2006, Michel Quercia (michel.quercia@prepas.org)      |
 |                                                                       |
 |  This file is part of Numerix. Numerix is free software; you can      |
 |  redistribute it and/or modify it under the terms of the GNU Lesser   |
 |  General Public License as published by the Free Software Foundation; |
 |  either version 2.1 of the License, or (at your option) any later     |
 |  version.                                                             |
 |                                                                       |
 |  The Numerix Library is distributed in the hope that it will be       |
 |  useful, but WITHOUT ANY WARRANTY; without even the implied warranty  |
 |  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU  |
 |  Lesser General Public License for more details.                      |
 |                                                                       |
 |  You should have received a copy of the GNU Lesser General Public     |
 |  License along with the GNU MP Library; see the file COPYING. If not, |
 |  write to the Free Software Foundation, Inc., 59 Temple Place -       |
 |  Suite 330, Boston, MA 02111-1307, USA.                               |
 +-----------------------------------------------------------------------+
 |                                                                       |
 |                        Multiplication quadratique                     |
 |                                                                       |
 +-----------------------------------------------------------------------*/

                        # +---------------------------+
                        # |  Multiplication droule  |
                        # +---------------------------+

   # entre en milieu de boucle :
   #   r1:r0 = retenue entrante
   #   r2  = -(longueur du multiplicande)
   #   r8  = multiplicateur
   #   r16 = adresse multiplicande cadre sur un multiple de 32
   #   r20 = adresse rsultat cadre sur un multiple de 32
   #   r27 = adresse de retour
   #
   # sortie:
   #   (r20) <- r3 * (r16)
   #   r0 <- retenue sortante
   #   r1 <- dernier chiffre du rsultat
   #
   # registres modifis:
   #   r2 <- r2 + 32*ceil(r2/32) 
   #   r3 <- ind
   #   r16 <- adresse suivante pour le multiplicande
   #   r20 <- adresse suivante pour le rsultat

        .align 5
        .globl sn_mulloop
        .ent   sn_mulloop
        .frame $30,0,$27,0
        .prologue 0

        # corps de boucle  drouler (7 instructions, entrer  la 7me)
#define BODY(x) \
        ldq    $3,   x($16)     ;\
        mulq   $3,   $8,   $1   ;\
        umulh  $3,   $8,   $3   ;\
        addq   $0,   $1,   $1   ;\
        cmpult $1,   $0,   $0   ;\
        addq   $3,   $0,   $0   ;\
        stq    $1,   x($20)

	# boucle droule pour 32 chiffres
sn_mulloop:
        BODY(0);   BODY(8);   BODY(16);  BODY(24)
        BODY(32);  BODY(40);  BODY(48);  BODY(56)
        BODY(64);  BODY(72);  BODY(80);  BODY(88)
        BODY(96);  BODY(104); BODY(112); BODY(120)
        BODY(128); BODY(136); BODY(144); BODY(152)
        BODY(160); BODY(168); BODY(176); BODY(184)
        BODY(192); BODY(200); BODY(208); BODY(216)
        BODY(224); BODY(232); BODY(240); BODY(248)
#undef BODY

	lda    $2,   32($2)
	lda    $16,  256($16)
	lda    $20,  256($20)
	blt    $2,   sn_mulloop
	ret    $31,  ($27),1

	.end   sn_mulloop


              # +----------------------------------------------+
              # |  Multiplication droule pour deux chiffres  |
              # +----------------------------------------------+

   # entre en milieu de boucle :
   #   r0:r1 = retenue entrante
   #   r7:r8 = multiplicateur
   #   r2    = -(longueur du multiplicande)
   #   r16   = adresse multiplicande cadre sur un multiple de 32
   #   r20   = adresse rsultat cadre sur un multiple de 32
   #   r27   = adresse de retour
   #
   # sortie:
   #   (r20) <- r7:r8 * (r16)
   #   r0:r1 <- retenue sortante
   #
   # registres modifis:
   #   r2 <- r2 + 32*ceil(r2/32) 
   #   r3,r4,r5,r6 <- ind
   #   r16 <- adresse suivante pour le multiplicande
   #   r20 <- adresse suivante pour le rsultat

        .align 5
        .globl sn_mulloop2
        .ent   sn_mulloop2
        .frame $30,0,$27,0
        .prologue 0

        # corps de boucle  drouler (15 instructions, entrer  la 1re)
#define BODY(x) \
        ldq    $6,   x($16)     ;\
        mulq   $6,   $7,   $3   ;\
        umulh  $6,   $7,   $4   ;\
        mulq   $6,   $8,   $5   ;\
        umulh  $6,   $8,   $6   ;\
        addq   $0,   $3,   $3   ;\
        cmpult $3,   $0,   $0   ;\
        addq   $0,   $4,   $4   ;\
        stq    $3,   x($20)     ;\
        addq   $1,   $5,   $0   ;\
        cmpult $0,   $5,   $5   ;\
        addq   $5,   $6,   $6   ;\
	addq   $0,   $4,   $0   ;\
        cmpult $0,   $4,   $4   ;\
	addq   $4,   $6,   $1

	# boucle droule pour 32 chiffres
sn_mulloop2:
        BODY(0);   BODY(8);   BODY(16);  BODY(24)
        BODY(32);  BODY(40);  BODY(48);  BODY(56)
        BODY(64);  BODY(72);  BODY(80);  BODY(88)
        BODY(96);  BODY(104); BODY(112); BODY(120)
        BODY(128); BODY(136); BODY(144); BODY(152)
        BODY(160); BODY(168); BODY(176); BODY(184)
        BODY(192); BODY(200); BODY(208); BODY(216)
        BODY(224); BODY(232); BODY(240); BODY(248)
#undef BODY

	addq   $2,   32,   $2
	lda    $16,  256($16)
	lda    $20,  256($20)
	blt    $2,   sn_mulloop2
	ret    $31,  ($27),1

	.end   sn_mulloop2

                  # +---------------------------------------+
                  # |  Multiplication et addition droule  |
                  # +---------------------------------------+

   # entre en milieu de boucle :
   #   r1:r0 = retenue entrante
   #   r8  = multiplicateur
   #   r2  = -(longueur du multiplicande)
   #   r16 = adresse multiplicande cadre sur un multiple de 32
   #   r20 = adresse rsultat cadre sur un multiple de 32
   #   r27 = adresse de retour
   #
   # sortie :
   #   (r20) <- (r20) + r8 * (r16)
   #   r0 <- retenue sortante
   #   r1 <- dernier chiffre du rsultat
   #
   # registres modifis :
   #   r2 <- r2 + 32*ceil(r2/32) 
   #   r3 <- ind
   #   r16 <- adresse suivante pour le multiplicande
   #   r20 <- adresse suivante pour le rsultat

        .align 5
        .globl sn_muladdloop
        .ent   sn_muladdloop
        .frame $30,0,$27,0
        .prologue 0

        # corps de boucle  drouler (11 instructions, entrer  la 7me)
#define BODY(x) \
        ldq    $3,   x($16)     ;\
        mulq   $3,   $8,   $1   ;\
        umulh  $3,   $8,   $3   ;\
        addq   $0,   $1,   $1   ;\
        cmpult $1,   $0,   $0   ;\
        addq   $0,   $3,   $0   ;\
        ldq    $3,   x($20)     ;\
        addq   $1,   $3,   $1   ;\
        cmpult $1,   $3,   $3   ;\
        stq    $1,   x($20)     ;\
        addq   $0,   $3,   $0

	# boucle droule pour 32 chiffres
sn_muladdloop:
        BODY(0);   BODY(8);   BODY(16);  BODY(24)
        BODY(32);  BODY(40);  BODY(48);  BODY(56)
        BODY(64);  BODY(72);  BODY(80);  BODY(88)
        BODY(96);  BODY(104); BODY(112); BODY(120)
        BODY(128); BODY(136); BODY(144); BODY(152)
        BODY(160); BODY(168); BODY(176); BODY(184)
        BODY(192); BODY(200); BODY(208); BODY(216)
        BODY(224); BODY(232); BODY(240); BODY(248)
#undef BODY

	addq   $2,   32,   $2
	lda    $16,  256($16)
	lda    $20,  256($20)
	blt    $2,   sn_muladdloop
	ret    $31,  ($27),1

	.end   sn_muladdloop

                # +-------------------------------------------+
                # |  Multiplication et soustraction droule  |
                # +-------------------------------------------+

   # entre en milieu de boucle :
   #   r1:r0 = retenue entrante
   #   r8  = multiplicateur
   #   r2  = -(longueur du multiplicande)
   #   r16 = adresse multiplicande cadre sur un multiple de 32
   #   r20 = adresse rsultat cadre sur un multiple de 32
   #   r27 = adresse de retour
   #
   # sortie:
   #   (r20) <- (r20) - r8 * (r16)
   #   r0 <- retenue sortante
   #   r1 <- dernier chiffre du rsultat
   #
   # registres modifis:
   #   r2 <- r2 + 32*ceil(r2/32) 
   #   r3 <- ind
   #   r16 <- adresse suivante pour le multiplicande
   #   r20 <- adresse suivante pour le rsultat

        .align 5
        .globl sn_mulsubloop
        .ent   sn_mulsubloop
        .frame $30,0,$27,0
        .prologue 0

        # corps de boucle  drouler (11 instructions, entrer  la 7me)
#define BODY(x) \
        ldq    $3,   x($16)     ;\
        mulq   $3,   $8,   $1   ;\
        umulh  $3,   $8,   $3   ;\
        addq   $0,   $1,   $1   ;\
        cmpult $1,   $0,   $0   ;\
        addq   $0,   $3,   $0   ;\
        ldq    $3,   x($20)     ;\
        subq   $3,   $1,   $1   ;\
        cmpult $3,   $1,   $3   ;\
        stq    $1,   x($20)     ;\
        addq   $0,   $3,   $0

	# boucle droule pour 32 chiffres
sn_mulsubloop:
        BODY(0);   BODY(8);   BODY(16);  BODY(24)
        BODY(32);  BODY(40);  BODY(48);  BODY(56)
        BODY(64);  BODY(72);  BODY(80);  BODY(88)
        BODY(96);  BODY(104); BODY(112); BODY(120)
        BODY(128); BODY(136); BODY(144); BODY(152)
        BODY(160); BODY(168); BODY(176); BODY(184)
        BODY(192); BODY(200); BODY(208); BODY(216)
        BODY(224); BODY(232); BODY(240); BODY(248)
#undef BODY

	addq   $2,   32,   $2
	lda    $16,  256($16)
	lda    $20,  256($20)
	blt    $2,   sn_mulsubloop
	ret    $31,  ($27),1

	.end   sn_mulsubloop

        # +----------------------------------------------------------+
        # |  Multiplication et addition droule pour deux chiffres  |
        # +----------------------------------------------------------+

   # entre en milieu de boucle :
   #   r0:r1 = retenue entrante
   #   r7:r8 = multiplicateur
   #   r2    = -(longueur du multiplicande)
   #   r16   = adresse multiplicande cadre sur un multiple de 32
   #   r20   = adresse rsultat cadre sur un multiple de 32
   #   r27   = adresse de retour
   #
   # sortie:
   #   (r20) <- (r20) + r7:r8 * (r16)
   #   r0:r1 <- retenue sortante
   #
   # registres modifis:
   #   r2 <- r2 + 32*ceil(r2/32) 
   #   r3,r4,r5,r6 <- ind
   #   r16 <- adresse suivante pour le multiplicande
   #   r20 <- adresse suivante pour le rsultat

        .align 5
        .globl sn_muladdloop2
        .ent   sn_muladdloop2
        .frame $30,0,$27,0
        .prologue 0

        # corps de boucle  drouler (19 instructions, entrer  la 1re
	# ou  la 12me avec la retenue dans r3:(r1+r4):r6)
#define BODY(x) \
        ldq    $6,   x($16)     ;\
        mulq   $6,   $7,   $3   ;\
        umulh  $6,   $7,   $4   ;\
        mulq   $6,   $8,   $5   ;\
        umulh  $6,   $8,   $6   ;\
        addq   $0,   $3,   $3   ;\
        cmpult $3,   $0,   $0   ;\
        addq   $0,   $4,   $4   ;\
        addq   $1,   $5,   $1   ;\
        cmpult $1,   $5,   $5   ;\
        addq   $5,   $6,   $6   ;\
        ldq    $0,   x($20)     ;\
        addq   $0,   $3,   $3   ;\
        cmpult $3,   $0,   $0   ;\
        addq   $0,   $4,   $4   ;\
        stq    $3,   x($20)     ;\
	addq   $1,   $4,   $0   ;\
        cmpult $0,   $4,   $4   ;\
	addq   $4,   $6,   $1
	
	# boucle droule pour 32 chiffres
sn_muladdloop2:
        BODY(0);   BODY(8);   BODY(16);  BODY(24)
        BODY(32);  BODY(40);  BODY(48);  BODY(56)
        BODY(64);  BODY(72);  BODY(80);  BODY(88)
        BODY(96);  BODY(104); BODY(112); BODY(120)
        BODY(128); BODY(136); BODY(144); BODY(152)
        BODY(160); BODY(168); BODY(176); BODY(184)
        BODY(192); BODY(200); BODY(208); BODY(216)
        BODY(224); BODY(232); BODY(240); BODY(248)
#undef BODY

	addq   $2,   32,   $2
	lda    $16,  256($16)
	lda    $20,  256($20)
	blt    $2,   sn_muladdloop2
	ret    $31,  ($27),1

	.end   sn_muladdloop2

      # +--------------------------------------------------------------+
      # |  Multiplication et soustraction droule pour deux chiffres  |
      # +--------------------------------------------------------------+

   # entre en milieu de boucle :
   #   r0:r1 = retenue entrante
   #   r7:r8 = multiplicateur
   #   r2    = -(longueur du multiplicande)
   #   r16   = adresse multiplicande cadre sur un multiple de 32
   #   r20   = adresse rsultat cadre sur un multiple de 32
   #   r27   = adresse de retour
   #
   # sortie:
   #   (r20) <- (r20) - r7:r8 * (r16)
   #   r0:r1 <- retenue sortante
   #
   # registres modifis:
   #   r2 <- r2 + 32*ceil(r2/32) 
   #   r3,r4,r5,r6 <- ind
   #   r16 <- adresse suivante pour le multiplicande
   #   r20 <- adresse suivante pour le rsultat

        .align 5
        .globl sn_mulsubloop2
        .ent   sn_mulsubloop2
        .frame $30,0,$27,0
        .prologue 0

        # corps de boucle  drouler (19 instructions, entrer  la 1re
	# ou  la 12me avec la retenue dans r3:(r1+r4):r6)
#define BODY(x) \
        ldq    $6,   x($16)     ;\
        mulq   $6,   $7,   $3   ;\
        umulh  $6,   $7,   $4   ;\
        mulq   $6,   $8,   $5   ;\
        umulh  $6,   $8,   $6   ;\
        addq   $0,   $3,   $3   ;\
        cmpult $3,   $0,   $0   ;\
        addq   $0,   $4,   $4   ;\
        addq   $1,   $5,   $1   ;\
        cmpult $1,   $5,   $5   ;\
        addq   $5,   $6,   $6   ;\
        ldq    $0,   x($20)     ;\
        subq   $0,   $3,   $3   ;\
        cmpult $0,   $3,   $0   ;\
        addq   $0,   $4,   $4   ;\
        stq    $3,   x($20)     ;\
	addq   $1,   $4,   $0   ;\
        cmpult $0,   $4,   $4   ;\
	addq   $4,   $6,   $1
	
	# boucle droule pour 32 chiffres
sn_mulsubloop2:
        BODY(0);   BODY(8);   BODY(16);  BODY(24)
        BODY(32);  BODY(40);  BODY(48);  BODY(56)
        BODY(64);  BODY(72);  BODY(80);  BODY(88)
        BODY(96);  BODY(104); BODY(112); BODY(120)
        BODY(128); BODY(136); BODY(144); BODY(152)
        BODY(160); BODY(168); BODY(176); BODY(184)
        BODY(192); BODY(200); BODY(208); BODY(216)
        BODY(224); BODY(232); BODY(240); BODY(248)
#undef BODY

	addq   $2,   32,   $2
	lda    $16,  256($16)
	lda    $20,  256($20)
	blt    $2,   sn_mulsubloop2
	ret    $31,  ($27),1

	.end   sn_mulsubloop2

                      # +------------------------------+
                      # |  Multiplication par un long  |
                      # +------------------------------+

   # unsigned long xn(mul_1)(chiffre *a, long la, unsigned long b, chiffre *c)
   #
   # entre :
   # a = naturel de longueur la
   # b = long >= 0   # c = naturel de longueur la, peut tre confondu avec a
   #
   # sortie :
   # c <- a*b
   # retourne la retenue

#ifdef assembly_sn_mul_1
#define L(x) .Lsn_mul_1_##x
#define _a_  $16
#define _b_  $18
#define _c_  $19
#define _la_ $17

        .align 5
        .globl sn_mul_1
        .ent   sn_mul_1
sn_mul_1:
        .frame $30,0,$26,0
        .prologue 1
	ldgp   $gp,  0($27)

        bne    _la_, 1f
        bis    $31,  $31,  $0	# cas la = 0: retourne 0
        ret    $31,  ($26),1
1:
	subq   $31,  _la_, $2   # r2 <- -la
	and    $2,   31,   $3	# r3 <- (-la) % 32
	bic    $2,   31,   $2   # r2 <- -32*ceil(la/32)
	sll    $3,   3,    $4   # r4 <- 8*((-la) % 32)
	lda    $5,   sn_mulloop
	subq   $4,   $3,   $3   # r3 <- nb d instructions a sauter
	addq   $3,   6,    $3
	s4addq $3,   $5,   $5   # r5 <- adresse d entre dans la boucle
	bis    _b_,  _b_,  $8   # r8 <- multiplicateur
        ldq    $0,  0(_a_)	# r1:r0 <- a[0]*b
	mulq   $0,   _b_,  $1
	umulh  $0,   _b_,  $0
	subq   _a_,  $4,   _a_  # cadre a,c sur le multiple de 32 prcdent
	subq   _c_,  $4,   $20
	bis    $26,  $26,  $27  # retour terminal
	jmp    $31,  ($5)       # effectue les multiplications suivantes

	.end sn_mul_1

#undef L
#undef _a_
#undef _b_
#undef _c_
#undef _la_
#endif /* assembly_sn_mul_1 */

                      # +------------------------------+
                      # |  Multiplication quadratique  |
                      # +------------------------------+

   #  void xn(mul_n2)(chiffre *a, long la, chiffre *b, long lb, chiffre *c)
   #
   #  entre :
   #  a = naturel de longueur la
   #  b = naturel de longueur lb
   #  c = naturel de longueur la+lb, non confondu avec a ou b
   #  contraintes : 0 < lb <= la
   #
   #  sortie :
   #  c <- a*b

#ifdef assembly_sn_mul_n2
#define L(x) .Lsn_mul_n2_##x
#define _a_  $16
#define _b_  $18
#define _c_  $20
#define _la_ $17
#define _lb_ $19

        .align 5
#ifdef debug_mul_n2
        .globl sn_mul_n2_buggy
        .ent   sn_mul_n2_buggy
sn_mul_n2_buggy:
        .frame $30,0,$26,0
        .prologue 1
	ldgp   $gp,  0($27)
#else
        .globl sn_mul_n2
        .ent   sn_mul_n2
sn_mul_n2:
        .frame $30,0,$26,0
        .prologue 1
	ldgp   $gp,  0($27)
L(nogp):
#endif

	# prpare le droulement de la boucle interne
	subq   $31,  _la_, _la_ # la <- -la
	and    _la_,  31,  $3	# r3 <- (-la) % 32
	bic    _la_,  31,  _la_ # la <- -32*ceil(la/32)
	sll    $3,   3,    $4   # r4 <- 8*((-la) % 32)
	subq   _a_,  $4,   _a_  # cadre a,c sur le multiple de 32 prcdent
	subq   _c_,  $4,   _c_
	lda    $21,  sn_muladdloop2
	addq   $3,   $4,   $2   # r2 <- 19*((-la) % 32)
	addq   $2,   $2,   $2
	addq   $2,   $3,   $2
	s4addq $2,   $21,  $21  # r21 <- adresse d entre dans la boucle interne
	blbc   _lb_, L(even)

	# cas lb impair: c <- b[0]*a
	subq   $4,   $3,   $2   # r2 <- 7*((-la) % 32)
	lda    $27,  sn_mulloop
	s4addq $2,   $27,  $27  # r27 <- adresse entre 1re boucle
	lda    _lb_, -1(_lb_)   # lb --
	ldq    $8,   0(_b_)     # r8 <- multiplicateur
        bis    _la_, _la_, $2   # r2 <- compteur
	bis    $31,  $31,  $0   # r0 <- 0 (retenue)
	jsr    $27,  ($27)      # c[0..la-1] += b[0]*a
	stq    $0,   0(_c_)     # c[la] <- retenue
        lda    _b_,  8(_b_)     # b ++
        lda    _c_,  8(_c_)     # c ++
	bne    _lb_, L(next)
        ret    $31,  ($26),1

	# cas lb pair: c <- b[0]:b[1]*a
	.align 5
L(even):
	subq   $4,   $3,   $2   # r2 <- 15*((-la) % 32)
	addq   $4,   $2,   $2
	lda    $27,  sn_mulloop2
	s4addq $2,   $27,  $27  # r27 <- adresse entre 1re boucle
	lda    _lb_, -2(_lb_)   # lb -= 2
	ldq    $7,   0(_b_)     # r7:r8 <- multiplicateur
	ldq    $8,   8(_b_)
        bis    _la_, _la_, $2   # r2 <- compteur
	bis    $31,  $31,  $0   # r0:r1 <- 0 (retenue)
	bis    $31,  $31,  $1
	jsr    $27,  ($27)      # c[0..la-1] += b[0]:b[1]*a
        stq    $0,   0(_c_)     # c[la]:c[la+1] <- retenue
        stq    $1,   8(_c_)
	bne    _lb_, L(loop)
        ret    $31,  ($26),1

	# boucle sur les chiffres suivants de b
	.align 5
L(loop):
        lda    _b_,  16(_b_)    # b += 2
        lda    _c_,  16(_c_)    # c += 2
L(next):
        s8addq _la_, _a_,  _a_  # restaure &a
        s8addq _la_, _c_,  _c_  # restaure &c
	lda    _lb_, -2(_lb_)   # lb -= 2
        ldq    $7,   0(_b_)     # r7:r8 <- multiplicateur
	ldq    $8,   8(_b_)
        bis    _la_, _la_, $2   # r2 <- compteur
	bis    $31,  $31,  $0   # r0:r1 <- 0
	bis    $31,  $31,  $1
	jsr    $27,  ($21)      # c[i..i+la-1] += b[i]:b[i+1]*a
        stq    $0,   0(_c_)     # c[i+la]:c[i+la+1] <- retenue
        stq    $1,   8(_c_)
	bne    _lb_, L(loop)
        ret    $31,  ($26),1
	        
#ifdef debug_mul_n2
	.end sn_mul_n2_buggy
#else
	.end sn_mul_n2
#endif

#undef L
#undef _a_
#undef _b_
#undef _c_
#undef _la_
#undef _lb_
#endif /* assembly_sn_mul_n2 */
#if !defined(assembly_sn_mul_n2) || defined(debug_mul_n2)
	REPLACE(sn_mul_n2)
#endif


                                 # +---------+
                                 # |  Carr  |
                                 # +---------+

   #  void xn(sqr_n2)(chiffre *a, long la, chiffre *b)
   #
   #  entre :
   #  a = naturel de longueur la
   #  b = naturel de longueur 2*la, non confondu avec a
   #  contraintes : 0 < la
   #
   #  sortie :
   #  b <- a^2

#ifdef assembly_sn_sqr_n2
#define L(x) .Lsn_sqr_n2_##x
#define _a_  $16
#define _b_  $18
#define _c_  $20
#define _la_ $17
#define _lb_ $19

        .align 5
#ifdef debug_mul_n2
        .globl sn_sqr_n2_buggy
        .ent   sn_sqr_n2_buggy
sn_sqr_n2_buggy:
        .frame $30,0,$26,0
        .prologue 1
	ldgp   $gp,  0($27)
#else
        .globl sn_sqr_n2
        .ent   sn_sqr_n2
sn_sqr_n2:
        .frame $30,0,$26,0
        .prologue 1
	ldgp   $gp,  0($27)
L(nogp):
#endif
	        
#ifdef debug_mul_n2
	.end sn_sqr_buggy
#else
	.end sn_sqr_n2
#endif
	
	# initialise b avec les carrs des chiffres de a
	lda   _c_,  8(_b_)      # c <- &b[1]
	bis   _a_,  _a_,  $4    # r4 <- &a
	bis   _la_, _la_, _lb_	# lb <- la
	bis   $31,  $31,  $0    # r0 <- 0 (= a[-1])
	.align 5
L(squares):
	lda    _lb_, -1(_lb_)   # la--
	cmplt  $0,   $31,  $1   # r1 <- msb(a[i-1])
	ldq    $0,   0($4)      # r0 <- a[i]
	addq   $0,   $1,   $1   # r1 <- a[i] + msb(a[i-1])
	mulq   $0,   $1,   $2   # r2:r3 <- a[i]*(a[i] + msb(a[i-1])
	umulh  $0,   $1,   $3
	cmoveq $1,   $0,   $3
	stq    $2,   0(_b_)     # range le produit
	stq    $3,   8(_b_)
	lda    $4,   8($4)      # a++
	lda    _b_, 16(_b_)     # b += 2
	bne    _lb_, L(squares)
	lda    _la_, -1(_la_)   # la--
	beq    _la_, L(done)

	# ajoute les doubles produits, deux ranges  la fois
	bis    _a_,  _a_,  _b_  # b <- &a[0]
	lda    _a_,  8(_a_)     # a++
	subq   $31,  _la_, _la_ # la <- 1-la
	and    _la_, 31,   $0   # r0 <- (1-la) % 32
	bic    _la_, 31,   _lb_ # lb <- -32*ceil((la-1)/32)
	sll    $0,   3,    $1   # r1 <- 8*((1-la) % 32)
	subq   _a_,  $1,   _a_  # cadre a,c sur le multiple de 32 prcdent
	subq   _c_,  $1,   _c_
	lda    $21,  sn_muladdloop2
	addq   $1,   $0,   $1   # r1 <- 19*((1-la) % 32)
	addq   $1,   $1,   $1
	addq   $1,   $0,   $1
	s4addq $1,   $21,  $21  # r21 <- adresse d entre dans la boucle interne
	lda    $21,  44($21)    # il faut sauter les 11 premires instructions
	bis    $31,  $31,  $22  # r22 <- 0 (= a[-1])
	bis    $31,  $31,  $23  # r23 <- 0 (retenue en fin de boucle)

	.align 5
L(loop):
	lda    _la_, 2(_la_)    # la -= 2
	cmplt  $22,  $31,  $0   # r0 <- msb(a[i-1])
	ldq    $7,   0(_b_)     # r7:r8 <- 2*a[i]:a[i+1] + msb(a[i-1])
	ldq    $22,  8(_b_)
	cmplt  $7,   $31,  $8
	addq   $7,   $0,   $0
	addq   $7,   $0,   $7
	addq   $22,  $8,   $8
	addq   $22,  $8,   $8
	mulq   $7,   $22,  $3   # r3:r4 <- (2*a[i] + msb(a[i-1]))*a[i+1]
	umulh  $7,   $22,  $4
	bis    $31,  $31,  $1   # r1,r6 <- 0 (retenues)
	bis    $31,  $31,  $6
	bis    _lb_, _lb_, $2  # r2 <- compteur
	jsr    $27,  ($21)     # c[2i+1..i+la-1] += 2*a[i]:a[i+1]*a[i+2..la-1]
	ldq    $2,   0(_c_)    # ajoute la retenue  c[i+la]:c[i+la+1]
	ldq    $3,   8(_c_)
	addq   $0,   $23,  $0
	cmpult $0,   $23,  $23
	addq   $0,   $2,   $0
	cmpult $0,   $2,   $2
	addq   $2,   $23,  $23
	addq   $3,   $23,  $3
	cmpult $3,   $23,  $23
	addq   $1,   $3,   $1
	cmpult $1,   $3,   $3
	addq   $3,   $23,  $23 # r23 <- nouvelle retenue
	stq    $0,   0(_c_)    # range le rsultat
	stq    $1,   8(_c_)
	lda    $21, 152($21)   # avance de 38 instructions
	and    _la_, 30,   $0  # si le compte est multiple de 32 ...
	bne    $0,   1f
	addq   _lb_, 32,  _lb_ # ... dcompte un tour
	lda    $21,  -2432($21)# et repart en dbut de boucle
1:	
	s8addq _lb_, _a_,  _a_ # restaure a
	s8addq _lb_, _c_,  _c_ # restaure c
	lda    _b_,  16(_b_)   # b += 2
	lda    _c_,  16(_c_)   # c += 2
	blt    _la_,  L(loop)

	ldq    $0,   0(_c_)    # ajoute la dernire retenue
	addq   $0,   $23,  $0
	stq    $0,   0(_c_)
L(done):
        ret    $31,  ($26),1

#undef L
#undef  _a_
#undef  _b_
#undef  _c_
#undef  _la_
#undef  _lb_
#endif /* assembly_sn_sqr_n2 */
#if !defined(assembly_sn_sqr_n2) || defined(debug_mul_n2)
	REPLACE(sn_sqr_n2)
#endif
