/*
-----------------------------------------------------------------------------
This source file is part of OGRE
(Object-oriented Graphics Rendering Engine)
For the latest info, see http://www.ogre3d.org

Copyright (c) 2000-2009 Torus Knot Software Ltd
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
-----------------------------------------------------------------------------
*/
#include "OgreShaderMaterialSerializerListener.h"
#include "OgreShaderGenerator.h"
#include "OgreMaterialManager.h"


namespace Ogre {
namespace RTShader {

//-----------------------------------------------------------------------------
SGMaterialSerializerListener::SGMaterialSerializerListener()
{
	mSourceMaterial = NULL;
}

//-----------------------------------------------------------------------------
SGMaterialSerializerListener::~SGMaterialSerializerListener()
{


}

//-----------------------------------------------------------------------------
void SGMaterialSerializerListener::materialEventRaised(MaterialSerializer* ser, 
														MaterialSerializer::SerializeEvent event, 
														bool& skip, const Material* mat)
{
	if (event == MaterialSerializer::MSE_PRE_WRITE)
	{
		MaterialPtr matPtr = MaterialManager::getSingleton().getByName(mat->getName());
		mSourceMaterial = matPtr.get();
		createSGPassList(mSourceMaterial, mSGPassList);
	}

	if (event == MaterialSerializer::MSE_POST_WRITE)
	{
		mSourceMaterial = NULL;
		mSGPassList.clear();
	}
}

//-----------------------------------------------------------------------------
void SGMaterialSerializerListener::techniqueEventRaised(MaterialSerializer* ser, 
										  MaterialSerializer::SerializeEvent event, 
										  bool& skip, const Technique* tech)
{
	// Pre technique write event.
	if (event == MaterialSerializer::MSE_PRE_WRITE)
	{		
		const Any& techUserData = tech->getUserObjectBindings().getUserAny(ShaderGenerator::SGTechnique::UserKey);

		// Skip writing this technique since it was generated by the Shader Generator.
		if (techUserData.isEmpty() == false)
		{
			skip = true;
			return;
		}		
	}
}


//-----------------------------------------------------------------------------
void SGMaterialSerializerListener::passEventRaised(MaterialSerializer* ser, 
												   MaterialSerializer::SerializeEvent event, 
												   bool& skip, const Pass* pass)
{
	// End of pass writing event.
	if (event == MaterialSerializer::MSE_WRITE_END)
	{		
		// Grab the shader generator pass instance.
		ShaderGenerator::SGPass* passEntry = getShaderGeneratedPass(pass);
		
		// Case this pass use as source pass for shader generated pass.
		if (passEntry != NULL)							
			ShaderGenerator::getSingleton().serializePassAttributes(ser, passEntry);
	}	
}

//-----------------------------------------------------------------------------
void SGMaterialSerializerListener::createSGPassList(Material* mat, SGPassList& passList)
{
	for (unsigned short techniqueIndex = 0; techniqueIndex < mat->getNumTechniques(); ++techniqueIndex)
	{
		Technique* curTechnique = mat->getTechnique(techniqueIndex);

		for (unsigned short passIndex = 0; passIndex < curTechnique->getNumPasses(); ++passIndex)
		{
			Pass* curPass = curTechnique->getPass(passIndex);		
			const Any& passUserData = curPass->getUserObjectBindings().getUserAny(ShaderGenerator::SGPass::UserKey);

			// Case this pass created by the shader generator.
			if (passUserData.isEmpty() == false)	
			{
				ShaderGenerator::SGPass* passEntry = any_cast<ShaderGenerator::SGPass*>(passUserData);

				passList.push_back(passEntry);
			}								
		}
	}
}
//-----------------------------------------------------------------------------
ShaderGenerator::SGPass* SGMaterialSerializerListener::getShaderGeneratedPass(const Pass* srcPass)
{
	SGPassListIterator it	 = mSGPassList.begin();
	SGPassListIterator itEnd = mSGPassList.end();

	for (; it != itEnd; ++it)
	{
		ShaderGenerator::SGPass* currPassEntry = *it;

		if (currPassEntry->getSrcPass() == srcPass)
		{
			return currPassEntry;
		}
	}

	return NULL;
}


}
}
