//                                               -*- C++ -*-
/**
 *  @file  Library.cxx
 *  @brief A Library object points to an actual shared library
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: Library.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include "OTconfig.hxx"
#ifdef HAVE_LIBDL
extern "C" {
#include <dlfcn.h>               // for dlopen, dlsym, dlclose, dlerror
}
#endif
#include "Library.hxx"
#include "Log.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Func
    {

      typedef Common::Log Log;
      typedef Common::NotYetImplementedException NotYetImplementedException;

      CLASSNAMEINIT(Library);

      /* Default constructor */
      Library::Library(LibraryHandle handle,
		       const String & path)
	: PersistentObject(),
	  handle_(handle),
	  path_(path)
      {
        // Nothing to do
      }

      /* Virtual constructor */
      Library * Library::clone() const
      {
	return new Library(*this);
      }

      /* Comparison operator */
      Bool Library::operator ==(const Library & other) const
      {
	return true;
      }

      /* String converter */
      String Library::__repr__() const
      {
	return OSS() << "class=" << getClassName()
		     << " name=" << getName()
		     << " path=" << path_
		     << " handle=" << handle_;
      }


      /* Library symbol accessor */
      LibrarySymbol Library::getSymbol(String symbolName) const
	/* throw(DynamicLibraryException) */
      {
#ifdef HAVE_LIBDL
	// We first clear any previous error
	dlerror();

	char * error;

	// Now we look for the symbol in the library
	LibrarySymbol symbol = dlsym(handle_, symbolName.c_str());
	if ((error = dlerror()) != NULL)
	  throw DynamicLibraryException(HERE) << "Cannot dlsym symbol '"
					      << symbolName << "' in library located at '"
					      << path_ << "'. Reason : "
					      << error;
	Log::Info(OSS() << "Found symbol '" << symbolName << "' (" << symbol << ") in library located at '" << path_ << "'");

	return symbol;
#else /* HAVE_LIBDL */
	throw NotYetImplementedException(HERE) << "Library::getSymbol() need libdl";
#endif /* HAVE_LIBDL */
      }





    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
