//                                               -*- C++ -*-
/**
 *  @file  RayleighFactory.cxx
 *  @brief Factory for Rayleigh distribution
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2008-09-24 09:07:16 +0200 (mer. 24 sept. 2008) $
 *  Id:      $Id: RayleighFactory.cxx 942 2008-09-24 07:07:16Z lebrun $
 */
#include <cmath>
#include "RayleighFactory.hxx"
#include "Rayleigh.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Model::DistributionImplementationFactory::NumericalPoint NumericalPoint;

      /* Default constructor */
      RayleighFactory::RayleighFactory():
	DistributionImplementationFactory()
      {
	// Nothing to do
      }

      /* Virtual constructor */
      RayleighFactory * RayleighFactory::clone() const
      {
	return new RayleighFactory(*this);
      }

      /* Here is the interface that all derived class must implement */

      RayleighFactory::DistributionImplementation * RayleighFactory::buildImplementation(const NumericalSample & sample) const /* throw(InvalidArgumentException, InternalException) */
      {
	if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a Rayleigh distribution from an empty sample";
	if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build a Rayleigh distribution only from a sample of dimension 1, here dimension=" << sample.getDimension();
	NumericalScalar gamma(sample.getMin()[0]);
	// Correction for bias control
	const NumericalScalar size(sample.getSize());
	if (gamma > 0.0)
	  {
	    gamma *= 1.0 - 1.0 / (2.0 + size);
	  }
	else
	  {
	    gamma *= 1.0 + 1.0 / (2.0 + size);
	  }
	NumericalScalar sumSquares(0.0);
	for (UnsignedLong i = 0; i < size; ++i)
	  {
	    const NumericalScalar xI(sample[i][0] - gamma);
	    sumSquares += xI * xI;
	  }
	try {
	  return Rayleigh(sqrt(0.5 * sumSquares / size), gamma).clone();
	}
	catch (InvalidArgumentException & ex)
	  {
	    throw InternalException(HERE) << "Error: cannot estimate parameters of a Rayleigh distribution from the given sample";
	  }
      }

      RayleighFactory::DistributionImplementation * RayleighFactory::buildImplementation(const NumericalPointCollection & parameters) const /* throw(InvalidArgumentException, InternalException) */
      {
	try {
	  Rayleigh distribution;
	  distribution.setParametersCollection(parameters);
	  return distribution.clone();
	}
	catch (InvalidArgumentException & ex)
	  {
	    throw InternalException(HERE) << "Error: cannot build a Rayleigh distribution from the given parameters";
	  }
      }

      RayleighFactory::DistributionImplementation * RayleighFactory::buildImplementation() const /* throw(InvalidArgumentException, InternalException) */
      {
	return Rayleigh().clone();
      }

    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
