//                                               -*- C++ -*-
/**
 *  @file  CompositeRandomVector.cxx
 *  @brief An implementation class for Composite random vectors
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: CompositeRandomVector.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include "PersistentObjectFactory.hxx"
#include "CompositeRandomVector.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Model {

      typedef Base::Common::NotYetImplementedException NotYetImplementedException;

      CLASSNAMEINIT(CompositeRandomVector);

      static Base::Common::Factory<CompositeRandomVector> RegisteredFactory("CompositeRandomVector");

      /* Standard constructor */
      CompositeRandomVector::CompositeRandomVector(const NumericalMathFunction & function,
						   const Antecedent & p_antecedent,
						   const String & name)
	/* throw (InvalidArgumentException) */
	: RandomVectorImplementation(name),
	  function_(function),
	  p_antecedent_(p_antecedent)
      {
	if (function.getInputDimension() != p_antecedent->getDimension()) throw InvalidArgumentException(HERE) << "Error: trying to build a CompositeRandomVector from a RandomVector and a NumericalMathFunction with incompatible dimensions, here RandomVector dimension=" << p_antecedent->getDimension() << " and NumericalMathFunction input dimension=" << function.getInputDimension();
	setDescription(function.getOutputDescription());
      }

      /* Standard constructor */
      CompositeRandomVector::CompositeRandomVector(const NumericalMathFunction & function,
						   const RandomVector & antecedent,
						   const String & name)
	/* throw (InvalidArgumentException) */
	: RandomVectorImplementation(name),
	  function_(function),
	  p_antecedent_(antecedent.getImplementation())
      {
	if (function.getInputDimension() != p_antecedent_->getDimension()) throw InvalidArgumentException(HERE) << "Error: trying to build a CompositeRandomVector from a RandomVector and a NumericalMathFunction with incompatible dimensions, here RandomVector dimension=" << p_antecedent_->getDimension() << " and NumericalMathFunction input dimension=" << function.getInputDimension();
	setDescription(function.getOutputDescription());
      }

      /* Virtual constructor */
      CompositeRandomVector * CompositeRandomVector::clone() const
      {
	return new CompositeRandomVector(*this);
      }

      /* String converter */
      String CompositeRandomVector::__repr__() const {
	OSS oss;
	oss << "class=" << CompositeRandomVector::GetClassName()
	    << " function=" << function_
	    << " antecedent=" << p_antecedent_->__repr__();
	return oss;
      }
  


      /* Is the underlying random vector composite ? */
      Bool CompositeRandomVector::isComposite() const
      {
	return true;
      }



      /* Here is the interface that all derived class must implement */


      /* Dimension accessor */
      UnsignedLong CompositeRandomVector::getDimension() const
      {
	return function_.getOutputDimension();
      }

      /* Realization accessor */
      CompositeRandomVector::NumericalPoint CompositeRandomVector::getRealization() const
      {
	return function_(p_antecedent_->getRealization());
      }

      /* Numerical sample accessor */
      CompositeRandomVector::NumericalSample CompositeRandomVector::getNumericalSample(const UnsignedLong size) const
      {
	return function_(p_antecedent_->getNumericalSample(size));
      }


      /* Mean accessor */
      CompositeRandomVector::NumericalPoint CompositeRandomVector::getMean() const
      {
	/* To be implemented by a suitable algorithm, given by the user.
	   It could be the e.g. quadratic cumul algorithm */
	throw NotYetImplementedException(HERE);
      }

      /* Covariance accessor */
      CompositeRandomVector::CovarianceMatrix CompositeRandomVector::getCovariance() const
      {
	/* To be implemented by a suitable algorithm, given by the user.
	   It could be the e.g. quadratic cumul algorithm */
	throw NotYetImplementedException(HERE);
      }

      /* Get the random vector corresponding to the i-th marginal component */
      RandomVectorImplementation::Implementation CompositeRandomVector::getMarginal(const UnsignedLong i) const /* throw(InvalidArgumentException) */
      {
	if (i >= getDimension()) throw InvalidArgumentException(HERE) << "The index of a marginal random vector must be in the range [0, dim-1]";
	return new CompositeRandomVector(function_.getMarginal(i), p_antecedent_);
      }

      /* Get the marginal random vector corresponding to indices components */
      RandomVectorImplementation::Implementation CompositeRandomVector::getMarginal(const Indices & indices) const /* throw(InvalidArgumentException) */
      {
	if (!indices.check(getDimension() - 1)) throw InvalidArgumentException(HERE) << "The indices of a marginal random vector must be in the range [0, dim-1] and  must be different";
	return new CompositeRandomVector(function_.getMarginal(indices), p_antecedent_);
      }

      /* Antecedent accessor */
      CompositeRandomVector::Antecedent CompositeRandomVector::getAntecedent() const
      {
	return p_antecedent_;
      }

      /* Function accessor */
      CompositeRandomVector::NumericalMathFunction CompositeRandomVector::getFunction() const
      {
	return function_;
      }

      /* Distribution accessor */
      Distribution CompositeRandomVector::getDistribution() const
      {
	throw NotYetImplementedException(HERE) << "CompositeRandomVector::Distribution CompositeRandomVector::getDistribution() const";
      }

      /* Method save() stores the object through the StorageManager */
      void CompositeRandomVector::save(StorageManager::Advocate & adv) const
      {
	RandomVectorImplementation::save(adv);
	adv.saveAttribute( "function_", function_ );
	adv.saveAttribute( "antecedent_", *p_antecedent_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void CompositeRandomVector::load(StorageManager::Advocate & adv)
      {
	RandomVectorImplementation::load(adv);
	adv.loadAttribute( "function_", function_ );
	Base::Common::TypedInterfaceObject<RandomVectorImplementation> antecedent;
	adv.loadAttribute( "antecedent_", antecedent );
	p_antecedent_ = antecedent.getImplementation();
      }

    } /* namespace Model */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
