//                                               -*- C++ -*-
/**
 *  @file  Distribution.hxx
 *  @brief Abstract top-level class for all distributions
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: Distribution.hxx 1473 2010-02-04 15:44:49Z dutka $
 */
#ifndef OPENTURNS_DISTRIBUTION_HXX
#define OPENTURNS_DISTRIBUTION_HXX

#include "TypedInterfaceObject.hxx"
#include "DistributionImplementation.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Model {


      /**
       * @class Distribution
       *
       * The class describes the probabilistic concept of distribution.
       * Instances of this class can produce samples following the
       * distribution, can compute PDF or CDF, etc.
       * They are the actual key component of RandomVectors.
       */
      class Distribution 
	: public Base::Common::TypedInterfaceObject<DistributionImplementation>
      {
	CLASSNAME;
      public:

	typedef DistributionImplementation::Implementation            Implementation;
	typedef DistributionImplementation::InvalidArgumentException  InvalidArgumentException;
	typedef DistributionImplementation::InvalidDimensionException InvalidDimensionException;
	typedef DistributionImplementation::NotDefinedException       NotDefinedException;
	typedef DistributionImplementation::NumericalPoint            NumericalPoint;
	typedef DistributionImplementation::NumericalPointWithDescription  NumericalPointWithDescription;
	typedef DistributionImplementation::NumericalPointCollection  NumericalPointCollection;
	typedef DistributionImplementation::NumericalPointWithDescriptionCollection  NumericalPointWithDescriptionCollection;
	typedef DistributionImplementation::Description               Description;
	typedef DistributionImplementation::NumericalSample           NumericalSample;
	typedef DistributionImplementation::CovarianceMatrix          CovarianceMatrix;
	typedef DistributionImplementation::CorrelationMatrix         CorrelationMatrix;
	typedef DistributionImplementation::SquareMatrix              SquareMatrix;
	typedef DistributionImplementation::Indices                   Indices;
	typedef DistributionImplementation::Interval                  Interval;
	typedef DistributionImplementation::IsoProbabilisticTransformation        IsoProbabilisticTransformation;
	typedef DistributionImplementation::InverseIsoProbabilisticTransformation InverseIsoProbabilisticTransformation;
	typedef DistributionImplementation::Graph                     Graph;

	/** Default constructor */
	Distribution();

	/** Default constructor */
	Distribution(const DistributionImplementation & implementation,
		     const String & name = OT::DefaultName);


	/** Constructor from implementation */
	Distribution(const Implementation & p_implementation,
		     const String & name = OT::DefaultName);

#ifndef SWIG
	/** Constructor from implementation pointer */
	Distribution(DistributionImplementation * p_implementation,
		     const String & name = OT::DefaultName);
#endif

	/** Comparison operator */
	Bool operator ==(const Distribution & other) const;

	/** String converter */
	String __repr__() const;

	/** Weight accessor */
	void setWeight(const NumericalScalar w) /* throw(InvalidArgumentException) */;
	NumericalScalar getWeight() const;

	/** Dimension accessor */
	UnsignedLong getDimension() const;

	/* Here is the interface that all derived class must implement */

	/** Get one realization of the distribution */
	NumericalPoint getRealization() const;

	/** Get a numerical sample whose elements follow the distribution */
	NumericalSample getNumericalSample(const UnsignedLong size) const;

	/** Get the DDF of the distribution */
	NumericalScalar computeDDF(const NumericalScalar scalar) const;
	NumericalPoint  computeDDF(const NumericalPoint & point) const;
	NumericalSample computeDDF(const NumericalSample & sample) const;

	/** Get the PDF of the distribution */
	NumericalScalar computePDF(const NumericalScalar scalar) const;
	NumericalScalar computePDF(const NumericalPoint & point) const;
	NumericalSample computePDF(const NumericalSample & sample) const;
	/** Compute the PDF of 1D distributions over a regular grid */
	NumericalSample computePDF(const NumericalScalar xMin,
				   const NumericalScalar xMax,
				   const UnsignedLong pointNumber,
				   const NumericalScalar precision = DistributionImplementation::DefaultPDFEpsilon) const;

	/** Get the CDF of the distribution */
	NumericalScalar computeCDF(const NumericalScalar scalar,
				   const Bool tail = false) const;
	NumericalScalar computeCDF(const NumericalPoint & point,
				   const Bool tail = false) const;
	NumericalSample computeCDF(const NumericalSample & sample,
				   const Bool tail = false) const;
	/** Compute the CDF of 1D distributions over a regular grid */
	NumericalSample computeCDF(const NumericalScalar xMin,
				   const NumericalScalar xMax,
				   const UnsignedLong pointNumber,
				   const NumericalScalar precision = DistributionImplementation::DefaultPDFEpsilon,
				   const Bool tail = false) const;

	/** Get the probability content of an interval */
	NumericalScalar computeProbability(const Interval & interval) const;

	/** Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
	NumericalComplex computeCharacteristicFunction(const NumericalScalar x,
						       const Bool logScale = false) const;

	/** Get the PDF gradient of the distribution */
	NumericalPoint computePDFGradient(const NumericalPoint & point) const;

	/** Get the CDF gradient of the distribution */
	NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

	/** Get the quantile of the distribution */
	NumericalPoint computeQuantile(const NumericalScalar prob,
				       const Bool tail = false) const;
	/** Compute the quantile over a regular grid */
	NumericalSample computeQuantile(const NumericalScalar qMin,
					const NumericalScalar qMax,
					const UnsignedLong pointNumber,
					const NumericalScalar precision = DistributionImplementation::DefaultQuantileEpsilon,
					const Bool tail = false) const;

	/** Get the mathematical and numerical range of the distribution.
	    Its mathematical range is the smallest closed interval outside
	    of which the PDF is zero, and the numerical range is the interval
	    outside of which the PDF is rounded to zero in double precision */
	Interval getRange() const;

	/** Get the roughness, i.e. the L2-norm of the PDF */
	NumericalScalar getRoughness() const;

	/** Get the mean of the distribution */
	NumericalPoint getMean() const /* throw(NotDefinedException) */;

	/** Get the covariance of the distribution */
	CovarianceMatrix getCovariance() const /* throw(NotDefinedException) */;

	/** Cholesky factor of the covariance matrix accessor */
	SquareMatrix getCholesky() const;

	/** Get the raw moments of the standardized distribution */
	NumericalPoint getStandardMoment(const UnsignedLong n) const;

	/** Get the raw moments of the distribution */
	NumericalPoint getMoment(const UnsignedLong n) const;

	/** Get the centered moments about the mean of the distribution */
	NumericalPoint getCenteredMoment(const UnsignedLong n) const;

	/** Inverse of the Cholesky factor of the covariance matrix accessor */
	SquareMatrix getInverseCholesky() const;

	/** Correlation matrix accessor */
	CorrelationMatrix getCorrelation() const;

	/** Get the standard deviation of the distribution */
	NumericalPoint getStandardDeviation() const /* throw(NotDefinedException) */;

	/** Get the skewness of the distribution */
	NumericalPoint getSkewness() const /* throw(NotDefinedException) */;

	/** Get the kurtosis of the distribution */
	NumericalPoint getKurtosis() const /* throw(NotDefinedException) */;

	/** Get the implementation of the distribution */
	Implementation getImplementation() const;

	/** Check if the distribution is elliptical */
        Bool isElliptical() const;

	/** Check if the distribution is constinuous */
        Bool isContinuous() const;

	/** Tell if the distribution has elliptical copula */
	Bool hasEllipticalCopula() const;

	/** Tell if the distribution has independent copula */
	Bool hasIndependentCopula() const;

	/** Get the i-th marginal distribution */
	Distribution getMarginal(const UnsignedLong i) const /* throw(InvalidArgumentException) */;

	/** Get the distribution of the marginal distribution corresponding to indices dimensions */
	Distribution getMarginal(const Indices & indices) const /* throw(InvalidArgumentException) */;

	/** Get the copula of a distribution */
	Distribution getCopula() const;

	/** Compute the DDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
	virtual NumericalScalar computeConditionalDDF(const NumericalScalar x, const NumericalPoint & y) const;

	/** Compute the PDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
	virtual NumericalScalar computeConditionalPDF(const NumericalScalar x, const NumericalPoint & y) const;

	/** Compute the CDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
	virtual NumericalScalar computeConditionalCDF(const NumericalScalar x, const NumericalPoint & y) const;

	/** Compute the quantile of Xi | X1, ..., Xi-1, i.e. x such that CDF(x|y) = q with x = Xi, y = (X1,...,Xi-1) */
	virtual NumericalScalar computeConditionalQuantile(const NumericalScalar q, const NumericalPoint & y) const;

	/** Get the isoprobabilist transformation */
	IsoProbabilisticTransformation getIsoProbabilisticTransformation() const;

	/** Get the inverse isoprobabilist transformation */
	InverseIsoProbabilisticTransformation getInverseIsoProbabilisticTransformation() const;

	/** Get the standard distribution */
	Distribution getStandardDistribution() const;

	/** Draw the PDF of the distribution when its dimension is 1 or 2 */
	virtual Graph drawPDF() const
	  /* throw(InvalidDimensionException, InvalidArgumentException) */;

	/** Draw the PDF of the distribution when its dimension is 1 */
	virtual Graph drawPDF(const NumericalScalar xMin, const NumericalScalar xMax, const UnsignedLong pointNumber = DistributionImplementation::DefaultPointNumber) const
	  /* throw(InvalidDimensionException, InvalidArgumentException) */;
	virtual Graph drawPDF(const UnsignedLong pointNumber) const
	  /* throw(InvalidDimensionException, InvalidArgumentException) */;

	/** Draw the PDF of a 1D marginal */
	virtual Graph drawMarginal1DPDF(const UnsignedLong marginalIndex, const NumericalScalar xMin, const NumericalScalar xMax, const UnsignedLong pointNumber) const
	  /* throw(InvalidArgumentException) */;

	/** Draw the PDF of the distribution when its dimension is 2 */
	virtual Graph drawPDF(const NumericalPoint & xMin, const NumericalPoint & xMax, const NumericalPoint & pointNumber) const
	  /* throw(InvalidDimensionException, InvalidArgumentException) */;
	virtual Graph drawPDF(const NumericalPoint & xMin, const NumericalPoint & xMax) const
	  /* throw(InvalidDimensionException, InvalidArgumentException) */;
	virtual Graph drawPDF(const NumericalPoint & pointNumber) const
	  /* throw(InvalidDimensionException, InvalidArgumentException) */;

	/** Draw the PDF of a 2D marginal */
	virtual Graph drawMarginal2DPDF(const UnsignedLong firstMarginal, const UnsignedLong secondMarginal, const NumericalPoint & xMin, const NumericalPoint & xMax, const NumericalPoint & pointNumber) const
	  /* throw(InvalidArgumentException) */;

	/** Draw the CDF of the distribution when its dimension is 1 or 2 */
	virtual Graph drawCDF() const
	  /* throw(InvalidDimensionException, InvalidArgumentException) */;

	/** Draw the CDF of the distribution when its dimension is 1 */
	virtual Graph drawCDF(const NumericalScalar xMin, const NumericalScalar xMax, const UnsignedLong pointNumber = DistributionImplementation::DefaultPointNumber) const
	  /* throw(InvalidDimensionException, InvalidArgumentException) */;
	virtual Graph drawCDF(const UnsignedLong pointNumber) const
	  /* throw(InvalidDimensionException, InvalidArgumentException) */;

	/** Draw the CDF of the distribution when its dimension is 2 */
	virtual Graph drawCDF(const NumericalPoint & xMin, const NumericalPoint & xMax, const NumericalPoint & pointNumber) const
	  /* throw(InvalidDimensionException, InvalidArgumentException) */;
	virtual Graph drawCDF(const NumericalPoint & xMin, const NumericalPoint & xMax) const
	  /* throw(InvalidDimensionException, InvalidArgumentException) */;
	virtual Graph drawCDF(const NumericalPoint & pointNumber) const
	  /* throw(InvalidDimensionException, InvalidArgumentException) */;

	/** Draw the CDF of a 1D marginal */
	virtual Graph drawMarginal1DCDF(const UnsignedLong marginalIndex, const NumericalScalar xMin, const NumericalScalar xMax, const UnsignedLong pointNumber) const
	  /* throw(InvalidArgumentException) */;

	/** Draw the CDF of a 2D marginal */
	virtual Graph drawMarginal2DCDF(const UnsignedLong firstMarginal, const UnsignedLong secondMarginal, const NumericalPoint & xMin, const NumericalPoint & xMax, const NumericalPoint & pointNumber) const
	  /* throw(InvalidArgumentException) */;

	/** Parameters value and description accessor */
	NumericalPointWithDescriptionCollection getParametersCollection() const;
	void setParametersCollection(const NumericalPointWithDescriptionCollection & parametersCollection);
	void setParametersCollection(const NumericalPointCollection & parametersCollection);
	void setParametersCollection(const NumericalPoint & parameters);

	/** Parameters number */
	UnsignedLong getParametersNumber() const;

	/** Description accessor */
	void setDescription(const Description & description);
	Description getDescription() const;

	/** Accessor to PDF computation precision */
	NumericalScalar getPDFEpsilon() const;

	/** Accessor to CDF computation precision */
	NumericalScalar getCDFEpsilon() const;

	/** Get a positon indicator for a 1D distribution */
	NumericalScalar getPositionIndicator() const;

	/** Get a dispersion indicator for a 1D distribution */
	NumericalScalar getDispersionIndicator() const;

      }; /* class Distribution */


    } /* namespace Model */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_DISTRIBUTION_HXX */
