/*
 Copyright (C) 2011 Christian Dywan <christian@twotoasts.de>

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 See the file COPYING for the full license text.
*/

public class Postler.Messages : Gtk.TreeView {
    Accounts accounts;
    Gtk.TreeStore store;
    Gtk.TreeModelSort sort;
    Gtk.TreePath defer_select;
    Dexter.Dexter dexter = new Dexter.Dexter ();

    public Postler.Content content { get; set; }
    public bool hide_read { get; set; }
    public bool rich_rows { get; set; default = true; }
    public bool show_attachments { get; set; default = false; }

    public string? location { get; private set; }
    public AccountInfo account_info { get; private set; }
    public string? selected_location { get; private set; }

    string to_or_from;
    string[] headers = {};
    string[] filters = {};
    string[] folders = {};
    FileMonitor[] folder_monitors = {};

    enum Columns {
        FLAGGED,
        STATUS,
        ATTACHMENT,
        SUBJECT,
        WEIGHT,
        FROM,
        SIZE,
        LOCATION,
        TIMESTAMP
    }

    bool search_inline (Gtk.TreeModel model, int column, string key,
        Gtk.TreeIter iter) {
        string subject;
        model.get (iter, Columns.SUBJECT, out subject);
        return !(key in subject.down ());
    }

    void selection_changed () {
        Gtk.TreeIter sort_iter;
        if (get_selected_iter (out sort_iter)) {
            string location;
            sort.get (sort_iter, Columns.LOCATION, out location);
            selected_location = location;
        } else
            selected_location = null;
    }

    static string escape_text (string text) {
        /* Like Markup.escape_text, but including UTF-8 validation */
        var escaped = new StringBuilder.sized (text.size () * 2);
        for (unowned string s = text; s.get_char () != 0 ; s = s.next_char ()) {
            unichar character = s.get_char ();
            switch (character) {
            case '&':
                escaped.append ("&amp;");
                break;
            case '<':
                escaped.append ("&lt;");
                break;
            case '>':
                escaped.append ("&gt;");
                break;
            case '\'':
                escaped.append ("&apos;");
                break;
            case '"':
                escaped.append ("&quot;");
                break;
            default:
                if (character.iscntrl ())
                    escaped.append_c (' ');
                else if (!character.validate ())
                    escaped.append_unichar ('�');
                else
                    escaped.append_unichar (character);
                break;
            }
        }
        return escaped.str;
    }

    void render_flag (Gtk.CellLayout layout, Gtk.CellRenderer cell,
        Gtk.TreeModel model, Gtk.TreeIter iter) {
        bool flagged;
        var screen = get_screen ();
        unowned string stock_id = null;
        unowned string prelight_stock_id = null;

        model.get (iter, Columns.FLAGGED, out flagged);
        if (flagged) {
            if (Gtk.IconTheme.get_for_screen (screen).has_icon (STOCK_STARRED))
                stock_id = STOCK_STARRED;
            else
                stock_id = STOCK_EMBLEM_IMPORTANT;
        } else {
            if (Gtk.IconTheme.get_for_screen (screen).has_icon (STOCK_NOT_STARRED))
                prelight_stock_id = STOCK_NOT_STARRED;
        }
        cell.set ("stock-id", stock_id, "prelight-stock-id", prelight_stock_id);
    }

    void render_subject (Gtk.CellLayout layout, Gtk.CellRenderer cell,
        Gtk.TreeModel model, Gtk.TreeIter iter) {
        string charset, subject;
        int weight = Pango.Weight.NORMAL;
        int64 timestamp;
        var renderer = cell as Gtk.CellRendererText;

        model.get (iter, Columns.SUBJECT, out subject,
                         Columns.WEIGHT, out weight,
                         Columns.TIMESTAMP, out timestamp);

        if (timestamp == 0) {
            model.get (iter, Columns.SUBJECT, out subject);
            renderer.markup = subject;
            renderer.xpad = renderer.ypad = 8;
            renderer.xalign = renderer.yalign = 0.5f;
            return;
        }

        renderer.xpad = renderer.ypad = 0;
        renderer.xalign = renderer.yalign = 0.0f;

        if (!rich_rows) {
            model.get (iter, Columns.WEIGHT, out weight);
            renderer.text = parse_encoded (subject, out charset);
            renderer.weight = weight;
            return;
        }

        string from;
        model.get (iter, Columns.FROM, out from);

        subject = escape_text (parse_encoded (subject, out charset));
        string date = Postler.Content.format_timestamp (timestamp);
        long size = date.length;
        for (int j = 0; j < 20 - size; j++)
            date = date + " ";

        from = escape_text (parse_address (parse_encoded (from, out charset))[0]);
        from = dexter.get_name (from) ?? from;

        renderer.ellipsize = Pango.EllipsizeMode.MIDDLE;
        renderer.markup = ("<span weight=\"%d\">%s</span>\n" +
                           "<small><tt>%s      </tt></small> %s").printf (
                           weight, from, date, subject);
    }

    void render_from (Gtk.TreeViewColumn column, Gtk.CellRenderer cell,
        Gtk.TreeModel model, Gtk.TreeIter iter) {
        var renderer = cell as Gtk.CellRendererText;

        if (rich_rows)
            renderer.text = "";
        else {
            string? from;
            model.get (iter, Columns.FROM, out from);
            if (from != null) {
                string charset;
                from = parse_address (parse_encoded (from, out charset))[0];
            }
            renderer.text = from;
        }
    }

    void render_date (Gtk.TreeViewColumn column, Gtk.CellRenderer cell,
        Gtk.TreeModel model, Gtk.TreeIter iter) {
        var renderer = cell as Gtk.CellRendererText;

        if (rich_rows)
            renderer.text = "";
        else {
            int64 timestamp;
            model.get (iter, Columns.TIMESTAMP, out timestamp);
            renderer.text = Postler.Content.format_timestamp (timestamp);
        }
    }

    void render_size (Gtk.TreeViewColumn column, Gtk.CellRenderer cell,
        Gtk.TreeModel model, Gtk.TreeIter iter) {
        var renderer = cell as Gtk.CellRendererText;

        if (rich_rows)
            renderer.text = "";
        else {
            int64 size;
            model.get (iter, Columns.SIZE, out size);
            renderer.text = format_size_for_display (size);
        }
    }

    void renderer_flag_toggled (Gtk.CellRendererToggle renderer,
                                string                 path) {
        Gtk.TreeIter sort_iter = Gtk.TreeIter ();
        if (!sort.get_iter_from_string (out sort_iter, path))
            return;

        string location;
        sort.get (sort_iter, Columns.LOCATION, out location);
        toggle_message_flag (sort_iter, ref location, 'F');
    }

    void renderer_status_toggled (Gtk.CellRendererToggle renderer,
                                  string                 path) {
        Gtk.TreeIter sort_iter = Gtk.TreeIter ();
        if (!sort.get_iter_from_string (out sort_iter, path))
            return;

        string location;
        sort.get (sort_iter, Columns.LOCATION, out location);
        toggle_message_flag (sort_iter, ref location, 'S');
    }

    void on_drag_data_get (Gdk.DragContext context,
        Gtk.SelectionData selection_data, uint info, uint time_) {

        Gtk.TreeIter iter;
        Gtk.TreeModel model;
        var rows = get_selection ().get_selected_rows (out model);
        string[] uris = null;
        foreach (Gtk.TreePath path in rows) {
            model.get_iter_from_string (out iter, path.to_string ());
            string? location;
            model.get (iter, Columns.LOCATION, out location);
            uris += ("file://" + location);
        }
        if (uris != null)
            selection_data.set_uris (uris);
    }

    public Messages (Accounts accounts) {
        /* Drag and Drop */
        Gtk.drag_source_set (this, Gdk.ModifierType.BUTTON1_MASK, {}, Gdk.DragAction.MOVE);
        Gtk.drag_source_add_uri_targets (this);
        drag_data_get.connect (on_drag_data_get);

        this.accounts = accounts;
        store = new Gtk.TreeStore (9, typeof (bool), typeof (string),
            typeof (string), typeof (string), typeof (int), typeof (string),
            typeof (int64), typeof (string), typeof (int64));
        sort = new Gtk.TreeModelSort.with_model (store);
        set_model (sort);
        set_search_equal_func (search_inline);
        get_selection ().set_mode (Gtk.SelectionMode.MULTIPLE);
        get_selection ().changed.connect (selection_changed);
        var column = new Gtk.TreeViewColumn ();
        column.set_title (_("Flagged"));
        var renderer_flag = new Postler.CellRendererToggle ();
        column.pack_start (renderer_flag, true);
        column.set_cell_data_func (renderer_flag, render_flag);
        column.add_attribute (renderer_flag, "active", Columns.FLAGGED);
        renderer_flag.toggled.connect (renderer_flag_toggled);
        insert_column (column, -1);
        var renderer_status = new Postler.CellRendererToggle ();
        insert_column_with_attributes (-1, _("Status"),
            renderer_status, "stock-id", Columns.STATUS);
        renderer_status.toggled.connect (renderer_status_toggled);
        column = new Gtk.TreeViewColumn ();
        column.set_title (_("Subject"));
        var renderer_text = new Gtk.CellRendererText ();
        column.pack_start (renderer_text, true);
        column.set_cell_data_func (renderer_text, render_subject);
        var renderer_icon = new Gtk.CellRendererPixbuf ();
        column.pack_start (renderer_icon, false);
        column.add_attribute (renderer_icon, "stock-id", Columns.ATTACHMENT);
        insert_column (column, -1);
        insert_column_with_data_func (-1, _("From"),
            new Gtk.CellRendererText (), render_from);
        insert_column_with_data_func (-1, _("Date"),
            new Gtk.CellRendererText (), render_date);
        insert_column_with_data_func (-1, _("Size"),
            new Gtk.CellRendererText (), render_size);

        if (rich_rows) {
            get_column (3).visible = false;
            get_column (4).visible = false;
            get_column (5).visible = false;
        }

        unowned Gtk.BindingSet binding_set = Gtk.BindingSet.by_class (get_class ());
        Gtk.BindingEntry.add_signal (binding_set,
            Gdk.keyval_from_name ("Return"), 0,
                "content-display", 0);
        Gtk.BindingEntry.add_signal (binding_set,
            Gdk.keyval_from_name ("ISO_Enter"), 0,
                "content-display", 0);
        Gtk.BindingEntry.add_signal (binding_set,
            Gdk.keyval_from_name ("KP_Enter"), 0,
                "content-display", 0);
        Gtk.BindingEntry.add_signal (binding_set,
            Gdk.keyval_from_name ("Return"), Gdk.ModifierType.CONTROL_MASK,
                "content-display-window", 0);
        Gtk.BindingEntry.add_signal (binding_set,
            Gdk.keyval_from_name ("ISO_Enter"), Gdk.ModifierType.CONTROL_MASK,
                "content-display-window", 0);
        Gtk.BindingEntry.add_signal (binding_set,
            Gdk.keyval_from_name ("KP_Enter"), Gdk.ModifierType.CONTROL_MASK,
                "content-display-window", 0);
        Gtk.BindingEntry.add_signal (binding_set,
            Gdk.keyval_from_name ("space"), 0,
                "content-page-down", 0);
        Gtk.BindingEntry.add_signal (binding_set,
            Gdk.keyval_from_name ("space"), Gdk.ModifierType.SHIFT_MASK,
                "content-page-up", 0);
        Gtk.BindingEntry.add_signal (binding_set,
            Gdk.keyval_from_name ("Delete"), 0, "delete", 0);
        content_display_window.connect (content_new_window);

        notify["hide-read"].connect (() => {
            search (filters[0] ?? "", headers[0] ?? "");
        });
    }

    [Signal (action=true)]
    public virtual signal void content_display () {
        select_cursor_row (false);
    }
    [Signal (action=true)]
    public signal void content_display_window ();
    /* Vala fails to parse out parameters in virtual signals */
    void content_new_window () {
        Gtk.TreeIter sort_iter;
        if (get_selected_iter (out sort_iter)) {
            string location;
            sort.get (sort_iter, Columns.LOCATION, out location);
            mark_message_read (sort_iter, ref location);
            Postler.App.spawn_module ("content", location);
        }
    }
    [Signal (action=true)]
    public virtual signal void content_page_down () {
        content.move_cursor (Gtk.MovementStep.PAGES, 1);
    }
    [Signal (action=true)]
    public virtual signal void content_page_up () {
        content.move_cursor (Gtk.MovementStep.PAGES, -1);
    }
    [Signal (action=true)]
    public virtual signal void delete () {
        delete_selected ();
    }

    static string decode_piece (string encoded, out string charset) {
        if (!encoded.contains ("=?"))
            return encoded;
        int token1 = 0;
        while (!(encoded[token1] == '=' && encoded[token1 + 1] == '?'))
            token1++;
        token1++;
        int token = token1 + 1;
        while (encoded[token] != '?')
            token++;
        charset = encoded[token1 + 1:token].up ();
        /* Encoding aliases */
        if (charset == "KS_C_5601-1987")
            charset = "CP949";

        token++;
        unichar encoding = encoded[token].toupper ();
        if (encoding != 'Q' && encoding != 'B')
            return encoded;
        token++;
        if (encoded[token] != '?')
            return encoded;
        token++;
        string[] pieces = encoded.slice (token, encoded.length).split ("?=");
        if (pieces == null || pieces[0] == null)
            return encoded;
        string unquoted;
        if (encoding == 'Q') {
            unquoted = pieces[0].replace (" =", " ").replace ("_", " ");
            unquoted = Postler.Content.quoted_printable_decode (unquoted);
        }
        else if (encoding == 'B') 
            unquoted = (string)GLib.Base64.decode (pieces[0]);
        else
            unquoted = pieces[0];
        try {
            return encoded.substring (0, token1 - 1)
                + GLib.convert (unquoted, -1, "UTF-8", charset, null) +
                (pieces[1] != null ? pieces[1] : "");
        }
        catch (GLib.ConvertError error) {
            GLib.warning (_("Failed to convert \"%s\": %s"), encoded, error.message);
            return encoded.substring (0, token1 - 1)
                + pieces[0] + (pieces[1] != null ? pieces[1] : "");
        }
    }

    internal static string parse_encoded (string? encoded, out string charset) {
        charset = null;
        return_val_if_fail (encoded != null, null);

        /* format "=?charset?encoding?encoded?=",
           if in doubt, bail out and take the raw data */
        /* We mask "?q?=" as "\nq\n=" here because ?= is our delimiter */
        string eencoded = encoded.replace ("?q?=", "\nq\n=").replace ("?Q?=", "\nQ\n=");
        string[] pieces = eencoded.strip ().split ("?=");
        if (pieces == null || pieces[0] == null)
            return encoded;

        var decoded = new GLib.StringBuilder ();
        foreach (string piece in pieces) {
            piece = piece.replace ("\nq\n=", "?q?=").replace ("\nQ\n=", "?Q?=");
            decoded.append (decode_piece (piece, out charset));
        }
        return decoded.str;
    }

    internal static string[] parse_address (string address)
        ensures (result[0] != null && result[1] != null) {
        if (address.length < 1) {
            GLib.critical ("parse_address: assertion '!address.length < 1' failed");
            return { address, address };
        }

        if (!(">" in address && "<" in address))
            return { address, address };

        long greater = address.length - 1;
        while (address[greater] != '>')
            greater--;
        long lower = greater;
        while (address[lower] != '<')
            lower--;

        string recipient = address.slice (lower + 1, greater);
        if (recipient.has_prefix ("mailto:"))
            recipient = recipient.substring (7, -1);
        if (lower == 0)
            return { recipient, recipient };
        /* TODO: Parse multiple addresses */
        if (">" in recipient)
            return { recipient, recipient };

        /* Remove double or single quotes around the name */
        long first = address.has_prefix ("'") || address.has_prefix ("\"") ? 1 : 0;
        return { address.substring (first, lower - 1)
            .replace ("\\\"", "`")
            .replace ("' ", "").replace ("\"", "").chomp (), recipient };
    }

    public void clear () {
        folder_monitors = {};
        store.clear ();
    }

    static string flags_from_filename (string filename, out string bare_filename) {
        string[]? parts = filename.split (":");
        if (parts == null || parts[0] == null || parts[1] == null) {
            bare_filename = filename;
            return "";
        }
        bare_filename = parts[0];
        return parts[1];
    }

    static string parse_flags (string name, out string flagged, out int weight) {
        /* format "unique:2,DFPRST", ordered alphabetically */
        unowned string status = STOCK_MAIL_UNREAD;
        string bare_filename;
        string flags = flags_from_filename (name, out bare_filename);
        if (flags == "")
            return status;

        foreach (var character in flags.to_utf8 ()) {
            switch (character) {
            case 'D':
                status = STOCK_EMBLEM_DRAFT;
                break;
            case 'F':
                flagged = STOCK_EMBLEM_IMPORTANT;
                break;
            case 'P':
                status = STOCK_MAIL_FORWARDED;
                break;
            case 'R':
                status = STOCK_MAIL_REPLIED;
                break;
            case 'S':
                weight = Pango.Weight.NORMAL;
                if (status == STOCK_MAIL_UNREAD)
                    status = null;
                break;
            case 'T':
                flagged = "T";
                break;
            }
        }
        return status;
    }

    public void search (string filter, string header="subject") {
        headers = {};
        filters = {};
        populate (location, account_info, filter.down (), header);
    }

    private struct Message {
        string location;
        string subject;
        public string get_subject () {
            return subject != null ? subject : _("No subject");
        }
        string status;
        int font_weight;
        bool flagged;
        string from;
        public string get_from () {
            return from != null ? from : _("Unknown");
        }
        int64 timestamp;
        int64 size;
        string attachment;
    }

    private Message? read_message (File contents, bool folder_new) {

        string filename = contents.get_basename ();
        if (filename[0] == '.')
            return null;

        string status = STOCK_MAIL_UNREAD;
        int font_weight = Pango.Weight.BOLD;
        string flags = null;
        if (!folder_new) {
            status = parse_flags (filename, out flags, out font_weight);
            if (hide_read && font_weight != Pango.Weight.BOLD && flags == null)
                return null;
            if (flags != null && flags[0] == 'T')
                return null;
        }

        var message = new Message ();
        message.status = status;
        message.font_weight = font_weight;
        message.flagged = flags != null;
        message.location = contents.get_path ();
        message.subject = null;
        message.from = null;
        message.timestamp = 0;

        string content_type = show_attachments ? null : "";
        try {
            var stream = new DataInputStream (contents.read (null));
            string line;
            string previous_line = "";
            while ((line = stream.read_line (null, null)) != null) {
                if (line == "")
                    break;
                if (line[0] == '\t' || line[0] == ' ')
                    line = previous_line + " " + line.chug ();
                previous_line = line;

                string[] parts = line.split (":", 2);
                if (parts == null || parts[0] == null)
                    continue;
                string field = ascii_strdown (parts[0]);
                if (filters[0] != null && parts[1] != null) {
                    string lowercased = ascii_strdown (parts[1]);
                    if (headers[0] == field
                     && !(filters[0] in lowercased))
                        return null;
                    else if (filters[1] != null && headers[1] == field
                          && !(filters[1] in lowercased))
                        return null;
                }
                if (field == "subject") {
                    message.subject = parts[1].strip ();
                    if (message.from != null && message.timestamp != 0
                     && content_type != null && filters[0] == null)
                        break;
                }
                else if (field == to_or_from) {
                    message.from = parts[1];
                    if (message.subject != null && message.timestamp != 0
                     && content_type != null && filters[0] == null)
                        break;
                    }
                else if (field == "date") {
                    var the_time = Postler.Content.date_from_string (parts[1]);
                    message.timestamp = the_time.to_local ().to_unix ();
                    if (message.subject != null && message.from != null
                     && content_type != null && filters[0] == null)
                        break;
                }
                else if (field == "content-type") {
                    content_type = parts[1].strip ();
                    if (message.subject != null && message.from != null
                     && message.timestamp != 0 && filters[0] == null)
                        break;
                }
            }

            unowned string? fulltext = null;
            if (headers[0] == "fulltext")
                fulltext = filters[0];
            else if (headers[1] == "fulltext")
                fulltext = filters[1];
            if (fulltext != null) {
                bool skip = true;
                while ((line = stream.read_line (null, null)) != null) {
                    if (line.down ().contains (fulltext)) {
                        skip = false;
                        break;
                    }
                }
                if (skip)
                    return null;
            }
        } catch (GLib.Error contents_error) {
            GLib.critical (_("Failed to read message \"%s\": %s"),
                contents.get_path (), contents_error.message);
        }

        message.attachment = null;
        if (show_attachments
         && content_type != null
         && content_type.has_prefix ("multipart/mixed"))
            message.attachment = STOCK_MAIL_ATTACHMENT;
        return message;
    }

    public bool populate (string? location, AccountInfo account_info,
        string? filter=null, string? header=null) {
        clear ();
        if (location == null)
            return true;
        if (Path.is_absolute (location)
         && !FileUtils.test (location + "/cur", FileTest.EXISTS))
            return true;

        model = sort = null;

        var now = GLib.Date ();
        now.set_time_val (GLib.TimeVal ());

        string basename = Path.get_basename (location);
        if (basename == account_info.get_folder (FolderType.SENT)
         || basename == account_info.get_folder (FolderType.QUEUE)
         || basename == account_info.get_folder (FolderType.DRAFTS))
            to_or_from = "to";
        else
            to_or_from = "from";

        try {
            if (location != this.location) {
                folders = {};
                headers = {};
                filters = {};
                if (location.has_prefix ("search:")) {
                    foreach (var folder in accounts.get_folders ()) {
                        folders += folder + "/INBOX/new";
                        folders += folder + "/INBOX/cur";
                    }
                }
                else {
                    folders += location + "/cur";
                    folders += location + "/new";
                }
            }

            if (filters[0] == null) {
                headers = { filter != null && filter != "" ? header : null };
                filters = { filter != "" ? filter : null };

                if (location.has_prefix ("search:")) {
                    string query = location.substring (7);
                    string[] parts = query.split ("/", 2);
                    return_val_if_fail (parts[0] != null && parts[1] != null, false);
                    headers += parts[0];
                    if (filters[0] == null)
                        filters = { parts[1].down () };
                    filters += parts[1].down ();
                }
            }

            foreach (var folder in folders) {
                var folder_dir = File.new_for_path (folder);
                FileEnumerator folder_enumerator;
                try {
                    folder_enumerator = folder_dir.enumerate_children (
                        FILE_ATTRIBUTE_STANDARD_NAME + "," +
                        FILE_ATTRIBUTE_STANDARD_SIZE, 0, null);
                    } catch (GLib.Error enumerator_error) {
                        /* Empty accounts can't be enumerated. */
                        continue;
                    }
                GLib.FileInfo info;

                try {
                    var folder_monitor = folder_dir.monitor_directory (0, null);
                    folder_monitor.changed.connect (folder_monitor_changed);
                    folder_monitors += folder_monitor;
                } catch (GLib.Error monitor_error) {
                    GLib.critical (_("Failed to monitor folder \"%s\": %s"),
                        folder, monitor_error.message);
                }

                folder = folder_dir.get_basename ();
                bool folder_new = folder_dir.get_path ().has_suffix ("new");
                while ((info = folder_enumerator.next_file (null)) != null) {
                    unowned string name = info.get_name ();

                    var message_path = folder_dir.resolve_relative_path (name);
                    var message = read_message (message_path, folder_new);
                    if (message == null)
                        continue;
                    message.size = info.get_size ();

                    store.insert_with_values (null, null, 0,
                        Columns.FLAGGED, message.flagged,
                        Columns.STATUS, message.status,
                        Columns.ATTACHMENT, message.attachment,
                        Columns.SUBJECT, message.get_subject (),
                        Columns.WEIGHT, message.font_weight,
                        Columns.FROM, message.get_from (),
                        Columns.TIMESTAMP, message.timestamp,
                        Columns.SIZE, message.size,
                        Columns.LOCATION, message.location);
                }
            }

            /* Show error for failed search, ie. no results */
            int n_messages = store.iter_n_children (null);
            if (filters[0] != null && n_messages == 0) {
                display_error (_("No messages found"),
                    _("Check the spelling or try a different filter."));
                get_selection().set_select_function( () => { return false; });
                return false;
            } else {
                get_selection().set_select_function( () => { return true; });
            }

            sort = new Gtk.TreeModelSort.with_model (store);
            sort.set_sort_column_id (Columns.TIMESTAMP, Gtk.SortType.DESCENDING);
            model = sort;

            hadjustment.value = vadjustment.value = 0;

        } catch (GLib.Error error) {
            display_error (_("Failed to read folder \"%s\".").printf (location),
                error.message);
        }

        /* Avoid changing 'location' to not cause property notification */
        if (this.location != location)
            this.location = location;
        this.account_info = account_info;
        return false;
    }

    public void display_error (string title, string message) {
        clear ();
        string markup = "<big><b>%s</b></big>\n\n%s".printf (title, message);
        store.insert_with_values (null, null, 0,
                                  Columns.SUBJECT, markup,
                                  Columns.FROM, null);
        sort = new Gtk.TreeModelSort.with_model (store);
        model = sort;
    }

    internal static string toggle_flag (string location, char flag) {
        string folder = Path.get_dirname (location);
        string bare_filename;
        if (folder.has_suffix ("/new"))
            folder = folder.slice (0, -4) + "/cur";
        string flags = flags_from_filename (location, out bare_filename);
        var new_flags = new StringBuilder ();
        bool did_include_flag = false;
        foreach (var character in flags.to_utf8 ()) {
            if (character < flag)
                new_flags.append_c (character);
            else if (character == flag)
                did_include_flag = true;
            else if (character > flag) {
                if (!did_include_flag) {
                    did_include_flag = true;
                    new_flags.append_c (flag);
                }
                new_flags.append_c (character);
            }
        }
        if (!did_include_flag)
            new_flags.append_c (flag);
        return folder + "/" + Path.get_basename (bare_filename) + ":" + new_flags.str;
    }

    void toggle_message_flag (Gtk.TreeIter sort_iter, ref string location, char flag) {
        return_if_fail (location != null);
        string new_location = toggle_flag (location, flag);
        if (FileUtils.rename (location, new_location) == 0) {
            location = new_location;
            int font_weight = Pango.Weight.BOLD;
            string? flagged = null;
            string status = parse_flags (location, out flagged, out font_weight);

            Gtk.TreeIter child_iter;
            sort.convert_iter_to_child_iter (out child_iter, sort_iter);
            store.set (child_iter,
                       Columns.FLAGGED, flagged != null ? true : false,
                       Columns.STATUS, status,
                       Columns.LOCATION, new_location,
                       Columns.WEIGHT, font_weight);
            if (location == selected_location)
                selected_location = new_location;
        }
    }

    bool get_selected_iter (out Gtk.TreeIter sort_iter) {
        GLib.List<Gtk.TreePath> paths = get_selection ().get_selected_rows (null);
        sort_iter = Gtk.TreeIter ();
        var path = paths.nth_data (0);
        if (path != null && sort.get_iter (out sort_iter, path))
            return true;
        return false;
    }

    bool get_message_iter (string location, out Gtk.TreeIter iter) {
        Gtk.TreeIter message_iter;
        if (!store.iter_children (out message_iter, null))
            return false;
        do {
            string existing_location;
            store.get (message_iter, Columns.LOCATION, out existing_location);
            if (existing_location == location) {
                if (&iter != null)
                    iter = message_iter;
                return true;
            }
        } while (store.iter_next (ref message_iter));
        return false;
    }

    bool iter_previous (ref Gtk.TreeIter iter) {
        Gtk.TreePath path = model.get_path (iter);
        return path.prev () && model.get_iter (out iter, path);
    }

    public void select_next_unread (bool forward) {
        Gtk.TreeIter iter;
        var path = get_selection ().get_selected_rows (null).nth_data (0);
        if (path != null)
            model.get_iter (out iter, path);
        else if (forward)
            model.iter_children (out iter, null);
        else
            model.iter_nth_child (out iter, null, model.iter_n_children (null) - 1);
        while ((forward && model.iter_next (ref iter))
           || (!forward && iter_previous (ref iter))) {
            int font_weight;
            model.get (iter, Columns.WEIGHT, out font_weight);
            if (font_weight == Pango.Weight.BOLD) {
                set_cursor (model.get_path (iter), null, false);
                break;
            }
        }
    }

    void folder_monitor_changed (File file, File? other, FileMonitorEvent event) {
        switch (event) {
        case FileMonitorEvent.CREATED:
            if (get_message_iter (file.get_path (), null))
                break;
            bool folder_new = file.get_path ().has_suffix ("/new");
            var message = read_message (file, folder_new);
            if (message == null)
                break;
            message.size = 0; /* FIXME: get file size */
            bool scroll = vadjustment.value == 0;
            store.insert_with_values (null, null, 0,
                Columns.FLAGGED, message.flagged,
                Columns.STATUS, message.status,
                Columns.ATTACHMENT, message.attachment,
                Columns.SUBJECT, message.get_subject (),
                Columns.WEIGHT, message.font_weight,
                Columns.FROM, message.get_from (),
                Columns.TIMESTAMP, message.timestamp,
                Columns.SIZE, message.size,
                Columns.LOCATION, message.location);
            if (scroll)
                vadjustment.value = 0;
            break;
        case FileMonitorEvent.DELETED:
            Gtk.TreeIter iter;
            if (get_message_iter (file.get_path (), out iter))
                store.remove (iter);
            break;
        case FileMonitorEvent.CHANGED:
        case FileMonitorEvent.CHANGES_DONE_HINT:
        case FileMonitorEvent.ATTRIBUTE_CHANGED:
            break;
        default:
            GLib.warning ("Unhandled folder change: " + event.to_string ());
            break;
        }
    }

    void mark_message_read (Gtk.TreeIter sort_iter, ref string location) {
        return_if_fail (location != null);
        string bare_filename;
        if (!flags_from_filename (location, out bare_filename).contains ("S"))
            toggle_message_flag (sort_iter, ref location, 'S');
    }

    public void toggle_selected_flag (char flag) {
        GLib.List<Gtk.TreePath> paths;
        var references = new GLib.List<Gtk.TreeRowReference> ();

        paths = get_selection ().get_selected_rows (null);
        foreach (var path in paths)
            references.prepend (new Gtk.TreeRowReference (sort, path));

        foreach (var reference in references) {
            var path = reference.get_path ();
            Gtk.TreeIter sort_iter;
            if (sort.get_iter (out sort_iter, path)) {
                string location;
                sort.get (sort_iter, Columns.LOCATION, out location);
                toggle_message_flag (sort_iter, ref location, flag);
            }
        }
    }

    void display_message (Gtk.TreeIter sort_iter) {
        string location;
        sort.get (sort_iter, Columns.LOCATION, out location);
        mark_message_read (sort_iter, ref location);
        content.display (location, account_info);
    }

    public static bool ensure_folder (string folder) {
        if (DirUtils.create_with_parents (folder + "/new", 0700) != 0
         || DirUtils.create_with_parents (folder + "/cur", 0700) != 0)
            return false;
        return true;
    }

    public void delete_selected () {
        if (location.has_suffix ("/" + account_info.get_folder (FolderType.TRASH)))
            move_selected (null);
        else
            move_selected (FolderType.TRASH);
    }

    public static uint32 obtain_uid_sequence (string folder, uint32 fallback_sequence) {
        /* Update .uidvalidity number, also read and updated by mbsync */
        uint32 uid_sequence = fallback_sequence;
        string uid_file = folder.slice (0, -4) + ".uidvalidity";
        try {
            string contents;
            FileUtils.get_contents (uid_file, out contents, null);
            string[] parts = contents.split ("\n");
            if (parts[0] != null && parts[1] != null) {
                uint32 old_uid_sequence = (uint32)(parts[1].to_uint64 ());
                FileUtils.set_contents (uid_file, "%s\n%u\n".printf (
                    parts[0], old_uid_sequence + 1), -1);
                uid_sequence = old_uid_sequence + 1;
            }
        }
        catch (Error error) {
            GLib.critical (_("Failed to read folder validity: %s"), error.message);
        }
        return uid_sequence;
    }

    static uint32 maildir_sequence_number = 0;
    public static string generate_filename (string folder, string flags="")
        requires (folder.has_suffix ("/cur/")) {
        /* format "time.pid_sequence.host,U=uid:2,FLAGS"
           This is the format used by mbsync. */
        var now = GLib.TimeVal ();
        now.get_current_time ();
        if (maildir_sequence_number == uint32.MAX)
            maildir_sequence_number = 0;
        else
            maildir_sequence_number++;
        uint32 uid_sequence = obtain_uid_sequence (folder, maildir_sequence_number);
        return folder + "%lu.%d_%u.%s,U=%u:2,%s".printf (
            now.tv_sec,
            Posix.getpid (),
            maildir_sequence_number,
            Environment.get_host_name ().replace ("/", "_").replace (":", "."),
            uid_sequence,
            flags);
    }

    public static string update_filename (string location, string new_folder)
        requires (new_folder.has_suffix ("/cur/")) {

        string bare_filename;
        string flags = flags_from_filename (location, out bare_filename);
        string filename;
        if ("S" in flags)
            filename = Path.get_basename (bare_filename) + ":" + flags;
        else
            filename = Path.get_basename (toggle_flag (location, 'S'));
        string[] parts = filename.split (",");
        return_val_if_fail (parts[0] != null && parts[1] != null
            && parts[2] != null && parts[3] == null, "");
        uint32 uid_sequence = obtain_uid_sequence (new_folder, maildir_sequence_number);
        return new_folder + parts[0] + ",U=" + uid_sequence.to_string () + ":2," + parts[2];
    }

    public void move_selected (FolderType? type) {
        string? destination_path;
        if (type == null)
            destination_path = null;
        else if (account_info.type == AccountType.SEARCH)
            destination_path = "/"; /* Nonsense path, in case of later errors */
        else {
            return_if_fail (account_info.get_folder (type) != null);
            destination_path = account_info.path + "/" + account_info.get_folder (type);
            if (!ensure_folder (destination_path)) {
                GLib.critical (_("Folders for moving messages couldn't be created."));
                return;
            }
        }

        GLib.List<Gtk.TreePath> paths;
        var references = new GLib.List<Gtk.TreeRowReference> ();
        paths = get_selection ().get_selected_rows (null);
        foreach (var path in paths)
            references.prepend (new Gtk.TreeRowReference (sort, path));

        foreach (var reference in references) {
            var path = reference.get_path ();
            Gtk.TreeIter sort_iter;
            if (sort.get_iter (out sort_iter, path)) {
                string location;
                sort.get (sort_iter, Columns.LOCATION, out location);
                var file = File.new_for_path (location);
                try {
                    if (destination_path == null)
                        file.delete (null);
                    else {
                        if (account_info.type == AccountType.SEARCH) {
                            /* Look for the account belonging to the message */
                            string inbox_folder = location.substring (0,
                                location.pointer_to_offset (location.rstr ("/")));
                            return_if_fail (inbox_folder.has_suffix ("/INBOX/new")
                                         || inbox_folder.has_suffix ("/INBOX/cur"));
                            string account_folder = inbox_folder.slice (0, -10);
                            string? folder_name = null;
                            foreach (var info in accounts.get_infos ()) {
                                if (info.path == account_folder) {
                                    folder_name = info.get_folder (type);
                                    break;
                                }
                            }
                            return_if_fail (folder_name != null);
                            destination_path = account_folder + "/" + folder_name;
                            return_if_fail (destination_path != null);
                        }

                        string new_location = update_filename (location,
                            destination_path + "/cur/");
                        var destination_file = File.new_for_path (new_location);
                        file.move (destination_file, GLib.FileCopyFlags.NONE);
                    }

                    /* Move to next, more recent message row */
                    Gtk.TreeIter child_iter;
                    sort.convert_iter_to_child_iter (out child_iter, sort_iter);
                    Gtk.TreePath next_path = sort.get_path (sort_iter);
                    content.clear ();
                    if (store.remove (child_iter)) {
                        next_path.prev ();
                        if (sort.get_iter (out sort_iter, next_path)) {
                            set_cursor (sort.get_path (sort_iter), null, false);
                            display_message (sort_iter);
                        }
                    }
                } catch (GLib.Error error) {
                    unowned string message;
                    if (destination_path == null)
                        message = _("Failed to delete message \"%s\": %s");
                    else
                        message = _("Failed to move message \"%s\": %s");
                    GLib.critical (message, location, error.message);
                }
            }
        }
    }

    public override bool button_press_event (Gdk.EventButton event) {
        Gtk.TreePath path;
        Gtk.TreeViewDropPosition pos;
        get_dest_row_at_pos ((int)event.x, (int)event.y, out path, out pos);

        if (event.type == Gdk.EventType.2BUTTON_PRESS) {
            Gtk.TreeIter sort_iter;
            if (get_selected_iter (out sort_iter)) {
                string location;
                sort.get (sort_iter, Columns.LOCATION, out location);
                mark_message_read (sort_iter, ref location);
                Postler.App.spawn_module ("content", location);
            }
        } else if (event.type == Gdk.EventType.BUTTON_PRESS
                && get_selection ().count_selected_rows () > 1
                && ((event.state & (Gdk.ModifierType.CONTROL_MASK
                                  | Gdk.ModifierType.SHIFT_MASK)) == 0)
                && get_selection ().path_is_selected (path)) {
                    get_selection().set_select_function( () => { return false; });
                    defer_select = path;
        } else {
            /* Re-enable selection */
            get_selection().set_select_function(() => { return true; });
            if (defer_select != null
             && get_dest_row_at_pos ((int)event.x, (int)event.y, out path, out pos)
             && defer_select == path
             && !(event.x == 0 && event.y == 0))
                set_cursor (path, null, false);

            defer_select = null;
        }
        return base.button_press_event (event);
    }

    public override bool button_release_event (Gdk.EventButton event) {
        if ((event.state & Gdk.ModifierType.BUTTON1_MASK) == 0) {
            /* Re-enable selection */
            get_selection().set_select_function(() => { return true; });
            Gtk.TreePath? path = null;
            Gtk.TreeViewDropPosition pos;
            if (defer_select != null
             && get_dest_row_at_pos ((int)event.x, (int)event.y, out path, out pos)
             && defer_select == path
             && !(event.x == 0 && event.y == 0))
                set_cursor (path, null, false);

            defer_select = null;
            return base.button_release_event (event);
        } else {
            get_selection ().set_select_function(() => { return true; });
            Gtk.TreePath? path = null;
            Gtk.TreeViewDropPosition pos;
            get_dest_row_at_pos ((int)event.x, (int)event.y, out path, out pos);
            if (defer_select != null
             && get_dest_row_at_pos ((int)event.x, (int)event.y, out path, out pos)
             && !(event.x == 0 && event.y == 0))
                set_cursor (path, null, false);

            defer_select = null;
        }

        GLib.List<Gtk.TreePath> paths = get_selection ().get_selected_rows (null);
        Gtk.TreeIter sort_iter = Gtk.TreeIter ();
        var path = paths.nth_data (0);
        if (path != null && sort.get_iter (out sort_iter, path)) {
            Gtk.TreeViewColumn column;
            get_path_at_pos ((int)event.x, (int)event.y, null,
                             out column, null, null);
            unowned string title = column.get_title ();
            /* Clickable icons should not display/ mark as read */
            if (title != _("Status") && title != _("Flagged"))
                display_message (sort_iter);
        }
        return base.button_release_event (event);
    }

    public override void row_activated (Gtk.TreePath path, Gtk.TreeViewColumn column) {
        Gtk.TreeIter sort_iter;
        if (sort.get_iter (out sort_iter, path))
            display_message (sort_iter);
    }
}

