import Plugin1DBase
import numpy
from numpy import cos, sin
import sys
import os
try:
    from PyMca import PyMcaQt as qt
    from PyMca import PyMcaDirs
    PYMCADIR = True
except ImportError:
    print "BM5TwoThetaPlugin Using huge PyQt4 import"
    import PyQt4.Qt as qt
    PYMCADIR = False

try:
    from PyMca import QPyMcaMatplotlibSave
    import matplotlib
    MATPLOTLIB = True
except ImportError:
    MATPLOTLIB = False

DEBUG = 0

class ConfigurationWidget(qt.QDialog):
    def __init__(self, parent=None):
        qt.QDialog.__init__(self, parent)
        self.setWindowTitle("BM05 Theta - Theta Analyzer Mesh Plugin")
        self.mainLayout = qt.QGridLayout(self)
        self.mainLayout.setMargin(6)
        self.mainLayout.setSpacing(2)
        labelList = ['Beam Energy (keV):',
                     'd-spacing (Angstrom):',
                     'Angle between diffraction and sample surface plane (deg):']
        row = 0
        col = 0
        self._lineEdit = []
        for text in labelList:
            label = qt.QLabel(self)
            label.setText(text)
            w = qt.QLineEdit(self)
            w._v = qt.QDoubleValidator(w)
            w.setValidator(w._v)
            self.mainLayout.addWidget(label, row, col)
            self.mainLayout.addWidget(w, row, col+1)
            row += 1
            self._lineEdit.append(w)

        hbox = qt.QWidget(self)
        hboxLayout = qt.QHBoxLayout(hbox)
        self.okButton = qt.QPushButton(hbox)
        self.okButton.setAutoDefault(False)
        self.okButton.setText("Accept")

        self.dismissButton = qt.QPushButton(hbox)
        self.dismissButton.setAutoDefault(False)
        self.dismissButton.setText("Dismiss")

        hboxLayout.addWidget(self.okButton)
        hboxLayout.addWidget(self.dismissButton)

        self.mainLayout.addWidget(hbox, row, 0, 1, 2)
        self.connect(self.okButton, qt.SIGNAL('clicked()'), self.accept)
        self.connect(self.dismissButton, qt.SIGNAL('clicked()'), self.reject)

    def setParameters(self, ddict):
        keys = ['energy', 'dspacing', 'phi']
        for i in range(3):
            key = keys[i]
            if ddict.has_key(key):
                self._lineEdit[i].setText("%.7g" % ddict[key])

    def getParameters(self):
        ddict={}
        keys = ['energy', 'dspacing', 'phi']
        for i in range(3):
            key = keys[i]
            ddict[key] = float(self._lineEdit[i].text())
        return ddict

class BM05TwoThetaPlugins(Plugin1DBase.Plugin1DBase):
    def __init__(self, plotWindow, **kw):
        Plugin1DBase.Plugin1DBase.__init__(self, plotWindow, **kw)
        self.methodDict = {'Configure':[self._configure,
                                        "Setup Energy, dspacing and offset",
                                        None],
                           'Convert':[self._convert,
                                      "Convert to Q space",
                                      None],
                           'Last result':[self._showLast,
                                         "Show image of last result",
                                          None],
                           'Save':[self._save,
                                   'Save last result',
                                   None]}
        #scan 281
        self._configuration = {'energy':    15.0,   #in keV
                'dspacing':  1.1084186,              #in A
                #'tthoffset': 43.78,                 # in deg
                'phi':  (180/numpy.pi)*numpy.arccos(8./numpy.sqrt(3.*24.))}
        """
        #scan 195 and 57
        self._configuration = {'energy':    24.0,   #in keV
                'dspacing':  3.1350812,              #in A
                #'tthoffset': 43.78,                 # in deg
                'phi':  0}
        """
        self.configurationWidget = None
        self.matplotlibWidget = None
        self.__lastScan = None
        
    #Methods to be implemented by the plugin
    def getMethods(self, plottype=None):
        """
        A list with the NAMES  associated to the callable methods
        that are applicable to the specified plot.

        Plot type can be "SCAN", "MCA", None, ...        
        """
        names = self.methodDict.keys()
        names.sort()
        return names

    def getMethodToolTip(self, name):
        """
        Returns the help associated to the particular method name or None.
        """
        return self.methodDict[name][1]

    def getMethodPixmap(self, name):
        """
        Returns the pixmap associated to the particular method name or None.
        """
        return self.methodDict[name][2]

    def applyMethod(self, name):
        """
        The plugin is asked to apply the method associated to name.
        """
        if DEBUG:
                apply(self.methodDict[name][0])
        else:
            try:
                apply(self.methodDict[name][0])
            except:
                import sys
                print sys.exc_info()
        return

    def _configure(self):
        if self.configurationWidget is None:
            self.configurationWidget = ConfigurationWidget()
        self.configurationWidget.setParameters(self._configuration)
        ret = self.configurationWidget.exec_()
        if ret == qt.QDialog.Accepted:
            self._configuration.update(self.configurationWidget.getParameters())
            self.configurationWidget.setParameters(self._configuration)

    def _getFilteredActiveCurve(self):
        activeCurve = self.getActiveCurve()
        if activeCurve is None:
            return None
        x, y, legend, info = activeCurve [0:4]
        xmin, xmax =self.getGraphXLimits()
        i1 = numpy.nonzero((x >= xmin) & (x <= xmax))
        x = numpy.take(x, i1)
        y = numpy.take(y, i1)

        #sort
        i1 = x.argsort()
        x = numpy.take(x, i1)
        y = numpy.take(y, i1)

        #remove duplicates
        x=x.ravel()
        i1 = numpy.nonzero((x[1:] > x[:-1]))
        x = numpy.take(x, i1)
        y = numpy.take(y, i1)
        
        x.shape = -1
        y.shape = -1

        return [x, y, legend, info]

    def _convert(self):
        if 0:
            x, y, legend, info = self._getFilteredActiveCurve()
        else:
            x, y, legend, info = self.getActiveCurve()
        #print "INFO = ", info
        header = info['Header'][0]
        #print header
        item = header.split()
        if (item[2] != 'mesh') or\
           (item[3] != 'th') or\
           (item[7] not in ['ta', 'tth']):
            raise ValueError, "This does not seem to be a Theta - TwoTheta scan"

        lambdaA = 12.39842/self._configuration['energy']
        thetaBragg = numpy.arcsin(0.5*lambdaA/self._configuration['dspacing'])
        thetaBragg *= 180./numpy.pi

        #print "Lambda      = ", lambdaA
        #print "Theta Bragg = ", thetaBragg
        
        qModule = 2*numpy.pi/lambdaA
        qModuleArray = numpy.ones(x.shape, numpy.float) * qModule

        #offset = self._configuration['tthoffset']
        #theta = 0.5*(x-offset)*(numpy.pi/180.)
        #qy = qModuleArray * numpy.sin(theta)
        #qz = qModuleArray * numpy.cos(theta)

        #I have qy and qz, I need q parallel and q perpendicular
        #to the sample surface
        #Assume an EXACTLY regular mesh for both motors
        th  = float(item[4]) + ((float(item[5]) - float(item[4]))/int(item[6]))*\
                             numpy.arange(int(item[6])+1)
        tth = float(item[8]) + ((float(item[9]) - float(item[8]))/int(item[10]))*\
                             numpy.arange(int(item[10])+1)

        #in the file, first variate th, then tth/analyzer
        th, tth = numpy.meshgrid(th, tth)
        y.shape = th.shape

        """
        th  =  (th - self._configuration['thoffset']) * (numpy.pi/180.)
        if item[7] == 'ta':
            print "BM05 CASE"
            tth = (self._configuration['tthoffset']-tth) * (numpy.pi/180.)
        else:
            tth = (tth - self._configuration['tthoffset']) * (numpy.pi/180.)
        print "ANGLES = ", th.min(), th.max()
        print "ANGLES = ", tth.min(), tth.max()

        #in the file, first variate th, then tth
        qparallel = numpy.zeros(len(tth)*len(th), numpy.float)
        qnormal   = numpy.zeros(len(tth)*len(th), numpy.float)
        rows      = numpy.zeros(len(tth)*len(th), numpy.float)
        columns   = numpy.zeros(len(tth)*len(th), numpy.float)
        """


        #find the maximum and minimum of the intensity
        iMax = y.max()
        iMin = y.min()

        #find the center of mass of all the points above a certain percentage
        #of the difference

        threshold = iMin + 0.01 * (iMax-iMin)                   
        idx = (y >= threshold)
        
        ysum   =   y[idx].sum()
        thMax  =  (y[idx]*th[idx]).sum()/ysum
        tthMax = (y[idx]*tth[idx]).sum()/ysum 

        #print "thMax = ", thMax
        #print "tthMax = ", tthMax

        #FABIO
        if 0:
            thMax  = th.min() + 0.5 * (th.max() - th.min())
            tthMax = tth.min() + 0.5 * (tth.max() - tth.min())



        #at tthMax we should find twice the Bragg angle
        tth = tth - tthMax  #now it is at zero

        #chi -> Value of analiser at Imax should be 2Theta bragg, this gives
        #       me the offset on the analiser angle. From the BM05 geometry,
        #       a positive change in analyser angle corresponds to a negative
        #       change on 2Theta of the same amount.
        chi = 2*thetaBragg - tth
        deltachi = -tth
        
        #psi -> Value of theta at I max should be Bragg angle + phi
        deltapsi = (th-thMax) 
        psi = thetaBragg+deltapsi-self._configuration['phi'] #The minus sign reproduces Tamzin's values

        #make sure we work in radians
        chi *= numpy.pi/180.
        deltachi *= numpy.pi/180.
        psi *= numpy.pi/180.
        deltapsi *= numpy.pi/180.

        qparallel  = cos(chi-psi)*cos(deltachi-deltapsi)-sin(chi-psi)*sin(deltachi-deltapsi)-\
                    (cos(psi)*cos(deltapsi)-sin(psi)*sin(deltapsi)+cos(chi-psi)-cos(psi))
        qparallel /= lambdaA
        

        qnormal  = sin(psi)*cos(deltapsi)+cos(psi)*sin(deltapsi)-sin(psi)-sin(chi-psi)+\
                   sin(chi-psi)*cos(deltachi-deltapsi)+cos(chi-psi)*sin(deltachi-deltapsi)
        qnormal /= lambdaA

        qparallel *= 1.0e4
        qnormal   *= 1.0e4

        self.__lastScan        = "%s" % info['Header'][0]
        self.__lastConfiguration = {}
        self.__lastConfiguration.update(self._configuration)
        self.__lastQParallel = qparallel
        self.__lastQNormal   = qnormal
        self.__lastIntensity = y

        #matplotlib
        #npoints = int(numpy.sqrt(y.shape[0]*y.shape[0]+y.shape[1]*y.shape[1]))
        npoints=128
        qpmin = qparallel.min()
        qpmax = qparallel.max()

        qnmin = qnormal.min()
        qnmax = qnormal.max()

        if 0:
            qmin = min(qpmin, qnmin)
            qmax = max(qpmax, qnmax)
            xi = numpy.linspace(qmin, qmax, npoints)
            yi = numpy.linspace(qmin, qmax, npoints)
        else:
            xi = numpy.linspace(qpmin, qpmax, npoints)
            yi = numpy.linspace(qnmin, qnmax, npoints)
        zi = matplotlib.mlab.griddata(qparallel.ravel(), qnormal.ravel(), y.ravel(), xi, yi)

        if self.matplotlibWidget is None:
            self.matplotlibWidget=QPyMcaMatplotlibSave.SaveImageSetup(None, zi)
            parameters = self.matplotlibWidget.getParameters()
            parameters['interpolation'] = 'Bilinear'
            parameters['linlogcolormap'] = 'Logarithmic'
            parameters['colormap'] = 'Jet'
            parameters['colorbar'] = 'Horizontal'
            parameters['xaxis'] = 'On'
            parameters['yaxis'] = 'On'
        else:
            self.matplotlibWidget.setImageData(zi)
            parameters = self.matplotlibWidget.getParameters()
        parameters['title'] = "Scan %s" % (info['Header'][0].split()[1])
        parameters['xlabel'] = 'qparallel(micron-1)'
        parameters['ylabel'] = 'qnormal(micron-1)'
        parameters['xorigin'] = xi[0]
        parameters['xpixelsize'] = xi[1]-xi[0]
        parameters['yorigin'] = yi[0]
        parameters['ypixelsize'] = yi[1]-yi[0]
        #parameters['zoomxmin'] = qpmin
        #parameters['zoomxmax'] = qpmax
        #parameters['zoomymin'] = qnmin
        #parameters['zoomymax'] = qnmax
        #print parameters
        self.matplotlibWidget.setParameters(parameters)
        self.matplotlibWidget.updateClicked()
        self.matplotlibWidget.show()
        self.matplotlibWidget.raise_()
        

    def _showLast(self):
        if self.matplotlibWidget is None:
            return
        self.matplotlibWidget.show()
        self.matplotlibWidget.raise_()

    def _save(self):
        if self.__lastScan is None:
            return
        #self.__lastConfiguration
        qparallel = self.__lastQParallel 
        qnormal = self.__lastQNormal
        y = self.__lastIntensity

        #get the output filename
        fileTypes = "CSV Files (*.csv)\n)"
        message   = "Enter output filename"
        if PYMCADIR:           
            wdir = PyMcaDirs.outputDir
        else:
            wdir = None
        filename = qt.QFileDialog.getSaveFileName(None,
                                                  message,
                                                  wdir,
                                                  fileTypes)
        filename = str(filename)
        if not len(filename):
            return
        if not filename.lower().endswith(".csv"):
            filename += ".csv"

        #write the file
        arrayLabels = ['qparallel(micron-1)', 'qnormal(micron-1)', 'I']
        arrayList = [qparallel, qnormal, y]
        csvseparator = ";"
        header = '"qparallel(micron-1)"'
        for label in arrayLabels[1:]:
            header +='%s"%s"' % (csvseparator,label)
        filehandle=open(filename,'w+')
        filehandle.write('%s\n' % header)
        fileline=""
        for i in range(y.shape[0]):
            for j in range(y.shape[1]):
                fileline += "%.7g" % (arrayList[0][i,j])
                for data in arrayList[1:]:
                    fileline += "%s%.7g" % (csvseparator, data[i,j])
                fileline += "\n"
                filehandle.write("%s" % fileline)
                fileline =""
        filehandle.write("\n") 
        filehandle.close()

MENU_TEXT = "BM05TwoThetaPlugin"
def getPlugin1DInstance(plotWindow, **kw):
    ob = BM05TwoThetaPlugins(plotWindow)
    return ob

if __name__ == "__main__":
    from PyMca import Plot1D
    app = qt.QApplication([])
    #w = ConfigurationWidget()
    #w.exec_()
    #sys.exit(0)
    
    DEBUG = 1
    x = numpy.arange(100.)
    y = x * x
    plot = Plot1D.Plot1D()
    plot.addCurve(x, y, "dummy")
    plot.addCurve(x+100, -x*x)
    plugin = getPlugin1DInstance(plot)
    for method in plugin.getMethods():
        print method, ":", plugin.getMethodToolTip(method)
    plugin.applyMethod(plugin.getMethods()[1])
    curves = plugin.getAllCurves()
    for curve in curves:
        print curve[2]
    print "LIMITS = ", plugin.getGraphYLimits()
