///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include"rheolef/geo_element.h"
#include"rheolef/tiny_element.h"
using namespace rheolef;
using namespace std;

namespace rheolef {
static 
char
skip_blancs_and_tabs (istream& is)
{
    if (!is.good()) return 0;
    char c = is.peek();
    while (is.good() && (c == ' ' || c == '\t')) {
	is >> c;
    } while (is.good() && (c == ' ' || c == '\t'));
    return c;
}
istream& 
operator >> (istream& is, geo_element& K)
{
    typedef geo_element::size_type size_type;
    char c;
    if (!is.good()) return is;
    is >> ws >> c;
    if (!isdigit(c)) {
	// has a element specifier, as:
	//  	q  23 17 25 2
        K.set_name(c);
        for (size_type i = 0; i < K.size() && is.good(); i++) {
            is >> K[i];
        }
        return is;
    }
    // has a element specifier, as:
    //  	23 17 25
    size_type tmp [geo_element::max_subgeo_vertex];
    size_type nvertex = 0;
    while (is.good() && isdigit(c)) {
        is.unget(); // or is.putback(c); ?
	is >> tmp [nvertex];
	nvertex++;
        c = skip_blancs_and_tabs (is);
    }
    // default: search the first type available for nvertex
    size_type i;
    for (i = 0; i < geo_element::max_size && geo_element::_n_subgeo [i][0] != nvertex; i++)
   	;
    if (i == geo_element::max_size) {
        error_macro ("undefined element with n_vertex = " << nvertex);
    }
    // initialize K
    geo_element::enum_type typ = geo_element::enum_type(i);
    K.set_type(typ);
    for (size_type i = 0; i < K.size(); i++) {
        K[i] = tmp [i];
    }
    return is;
}
ostream& 
operator << (ostream& os, const geo_element& K)
{
    if (K.type() == geo_element::max_size) {
        error_macro ("unexpected type");
        return os;
    }
    if (K.dimension() >= 2) {
	os << K.name() << "\t";
    }
    for (geo_element::size_type i = 0; i < K.size(); i++) {
        os << K[i];
        if (i != K.size()-1) {
	    os << " ";
	}
    }
    return os;
}
ostream& 
geo_element::dump(ostream& os) const 
{
    os << "geo_element: type = " << type() << endl;
    if (type() == max_size) {
        warning_macro ("unexpected type");
        return os;
    }
    //
    // heap
    //
    os << "  heap_size = " << heap_size() << endl;
    os << "  heap = ";
    for (size_type i = 0; i < heap_size(); i++) {
        os << _heap [i] << " ";
    }
    os << endl;
    //
    // print by subgeo
    //
    for (size_type d = 0; d <= dimension(); d++) {
	size_type nsubgeo = n_subgeo(d);
	os << "  d=" << d << " n_subgeo=" << nsubgeo << endl;
	for (size_type i = 0; i < nsubgeo; i++) {
	    os << "    " << i << ": index=" << subgeo(d, i) << endl;
	}
    }
    return os;
}
void 
geo_element::_heap_close()
{
    if (_heap) { 
	delete_tab_macro(_heap);
	_heap = 0;
    }
}
void 
geo_element::_heap_init()
{
    _heap_close();
    if (type() != max_size) {
	_heap = new_tab_macro(size_type, heap_size());
        fill(_heap, _heap + heap_size(), not_set);
    }
}
geo_element::geo_element (const tiny_element& x)
 : reference_element(x), _heap(0)
{
    set_type(x.type());
    if (x.type() != max_size) {
        for (size_type i = 0; i < x.size(); i++) {
	    operator[](i) = x[i];
        }
    } else {
	fatal_macro ("copy  from uninitialized tiny_element");
    }
}
void
geo_element::copy (const tiny_element& x)
{
    set_type(x.type());
    if (x.type() != max_size) {
        for (size_type i = 0; i < x.size(); i++) {
	    operator[](i) = x[i];
        }
    } else {
	fatal_macro ("copy  from uninitialized tiny_element");
    }
}
geo_element::geo_element(const geo_element& x)
 : reference_element(x), _heap(0)
{
    if (type() != max_size) {
        _heap = new_tab_macro(size_type, heap_size());
        std::copy(x._heap, x._heap + heap_size(), _heap);
    }
}
void
geo_element::copy (const geo_element& x)
{
    set_type(x.type());
    if (type() != max_size) {
        std::copy(x._heap, x._heap + heap_size(), _heap);
    }
}
geo_element& 
geo_element::operator = (const geo_element& x)
{
    if (type() != max_size) {
        set_type(x.type());
        std::copy(x._heap, x._heap + heap_size(), _heap);
        warning_macro("physical copy of geo_element '" << name() << "'");
    }
    return *this;
}

void
geo_element::build_subgeo(size_type subgeo_dim, size_type i_subgeo, geo_element& S) const
{
    size_type subgeo_sz = subgeo_size(subgeo_dim, i_subgeo);
    S.set_type(subgeo_sz, subgeo_dim);
    for (size_type i = 0; i < subgeo_sz; i++) {
        S[i] = subgeo_vertex(subgeo_dim, i_subgeo, i);
    }
}
geo_element::size_type
geo_element::subgeo_local_index(const geo_element& S) const
{
    // TODO: furnish also orientation and amount of shift

    size_type subgeo_dim = S.dimension();
    size_type nsubgeo = n_subgeo(subgeo_dim);
    for (size_type i = 0; i < nsubgeo; i++) {

        size_type subgeo_sz = subgeo_size(subgeo_dim, i);
        if (subgeo_sz != S.size()) {
	    continue;
	}
	size_type j0 = not_set;
	for (size_type j = 0; j < subgeo_sz; j++) {
	   if (S[0] == subgeo_vertex(subgeo_dim, i, j)) {
		j0 = j;	
		break;
           }
        }
	if (j0 == not_set) {
	   continue;
	}
	// now, have a common vertex: check other vertices
	bool found = true;
	for (size_type j = 1; j < subgeo_sz; j++) {
	    size_type j1 = (j0 + j) % subgeo_sz;
	    if (S[j] != subgeo_vertex(subgeo_dim, i, j1)) { 
		found = false;
		break;
	    }
	}
	if (found) {
	    return i;
	}
	// try with opposite orientation
	found = true;
	for (size_type j = 1; j < subgeo_sz; j++) {
	    size_type j1 = (j0 + subgeo_sz - j) % subgeo_sz;
	    if (S[j] != subgeo_vertex(subgeo_dim, i, j1)) { 
		found = false;
		break;
	    }
	}
	if (found) {
	    return i;
	}
    }
    // not found
    return not_set;
}
geo_element::size_type
geo_element::subgeo(const geo_element& S) const
{
    size_type i = subgeo_local_index(S);
    if (i == not_set) {
	return i;
    } else {
        return subgeo(S.dimension(), i);
    }
}
void
geo_element::set_subgeo(const geo_element& S, size_type idx)
{
    size_type i = subgeo_local_index(S);
    if (i == not_set) {
        error_macro ("set_subgeo: element does not contains such subgeo");
    } else {
	set_subgeo (S.dimension(), i, idx);
    }
}
void
geo_element::check() const
{
    for (size_type d = 0; d <= dimension(); d++) {
	size_type nsubgeo = n_subgeo(d);
	for (size_type i = 0; i < nsubgeo; i++) {
	    if (subgeo(d, i) == not_set)  {
		dump();
		error_macro ("index not set on subgeo: d=" << d << " i=" << i);
	    }
	}
    }
}
} // namespace rheolef
