///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/config.h"
#ifdef _RHEOLEF_HAVE_MPI
#include "rheolef/geo.h"
#include "rheolef/geo_domain.h"
#include "rheolef/dis_macros.h"
#include "rheolef/rheostream.h"
#include "rheolef/index_set.h"

namespace rheolef {

extern
array<polymorphic_array<geo_element>::size_type>
geo_partition (const polymorphic_array<geo_element>& elts, size_t dis_nv, size_t map_dim);

// --------------------------------------------------------------------------
// edges & faces renumbering subroutine 
// --------------------------------------------------------------------------
template <class PolymorphicArray>
void
geo_element_renumbering (
    const std::vector<geo_element::size_type>& new_global_vertex_owner,
    const std::vector<geo_element::size_type>& global_new_num_vert,
    const PolymorphicArray&                    ios_ge,
	  size_t			       dis_nv,
          PolymorphicArray&                    ge,
          array<geo_element::size_type>&       ios_ige2dis_ige
    )
{
    using namespace std;
    typedef typename geo_element::size_type size_type;

    communicator comm = ios_ge.ownership().comm();
    size_type dis_nge = ios_ge.dis_size();
    vector<size_type> new_local_ge_owner (dis_nge, 0);

    //
    // 1) global all_reduce
    // TODO: not balanced: ge nums
    // TODO: also, not optimal: O(N) in communication, memory & CPU, instead of O(N/nproc)
    //
    for (size_type ige = 0, nge = ios_ge.size(); ige < nge; ige++) {
      const geo_element& S = ios_ge [ige];
      size_type ios_dis_ige = S.ios_dis_ie();
      size_type owner = 0;
      for (size_type iloc = 0; iloc < S.size(); iloc++) {
        size_type ios_dis_iv = S [iloc];
        assert_macro (ios_dis_iv < dis_nv, "vertex index "<<ios_dis_iv<<" is out of range [0:"<<dis_nv<<"[");
        size_type iproc = new_global_vertex_owner [ios_dis_iv];
        owner = std::max (owner,iproc);
      }
      new_local_ge_owner [ios_dis_ige] = owner;
    }
    vector<size_type> new_global_ge_owner (dis_nge, 0);
    mpi::all_reduce (
  	comm,
  	new_local_ge_owner.begin().operator->(),
  	dis_nge,
  	new_global_ge_owner.begin().operator->(),
  	mpi::maximum<size_type>());
  
    distributor ios_ge_ownership = ios_ge.ownership();
    //
    // 2) redistribute the vertex partition
    //
    array<size_type> ge_partition (ios_ge_ownership);
    for (size_type ios_dis_ige = ge_partition.ownership().first_index(); ios_dis_ige < ge_partition.ownership().last_index(); ios_dis_ige++) {
      ge_partition.dis_entry (ios_dis_ige) = new_global_ge_owner[ios_dis_ige];
    }
    ge_partition.dis_entry_assembly();

    array<size_type>  ige2ios_dis_ige;
    ios_ge.repartition (
	ge_partition,
        ge,
  	ige2ios_dis_ige,
  	ios_ige2dis_ige);
  
    //
    // 3) vertices S[iloc] of new numbered element table still have ios numbering: fix it
    //
    size_type first_dis_ige = ge.ownership().first_index();
    for (size_type ige = 0, nge = ge.size(); ige < nge; ige++) {
      geo_element& S = ge [ige];
      S.set_dis_ie (first_dis_ige + ige);
      for (size_type iloc = 0, nloc = S.size(); iloc < nloc; iloc++) {
        assert_macro (S[iloc] < dis_nv, "vertex index "<<S[iloc]<<" out of range [0:"<<dis_nv<<"[");
        assert_macro (global_new_num_vert[S[iloc]] < dis_nv, "new vertex index "<<global_new_num_vert[S[iloc]] <<" out of range [0:"<<dis_nv<<"[");
        S[iloc] = global_new_num_vert [S[iloc]];
      }
    }
}
// --------------------------------------------------------------------------
// io for geo
// --------------------------------------------------------------------------
template <class T>
idiststream&
geo_rep<T,distributed>::get (idiststream& ips)
{
  using namespace std;
  check_macro (ips.good(), "bad input stream for geo.");
  communicator comm = base::_geo_element[0].ownership().comm();

  size_type io_proc = idiststream::io_proc();
  size_type my_proc = comm.rank();
  size_type nproc   = comm.size();
  if (my_proc == io_proc && !scatch(ips.is(),"\nmesh"))
    error_macro("input stream does not contains a geo.");
  // ------------------------------------------------------------------------
  // 1) read file
  // ------------------------------------------------------------------------
  //
  // 1.1) get header
  //
  size_type dis_nv;
  size_type dis_ne;
  size_type dis_nedg = 0;
  size_type dis_n_fac = 0;

  base::_name = "unnamed";

  ips >> base::_version
      >> base::_dimension
      >> dis_nv
      >> dis_ne;

  if (base::_version < 2) {
    warning_macro ("mesh version < 2 no more supported in distributed version");
  } else {
    if (base::_dimension >= 3) {
      ips >> dis_n_fac;
    }
    if (base::_dimension >= 2) {
      ips >> dis_nedg;
    }
  }
  std::fill (_ios_size_by_variant, _ios_size_by_variant+reference_element::max_size, 0);
  //
  // 1.2) get coordinates
  //
  array<vertex_type> ios_vertex (dis_nv);
  ios_vertex.get_values (ips, _point_get<T>(base::_dimension));
  check_macro (ips.good(), "bad input stream for geo.");
  // set ios_dis_iv index as fisrt field of the idx_vertex pair:
  distributor ios_vertex_ownership = ios_vertex.ownership();
  polymorphic_array<geo_element> ios_geo_element_p (ios_vertex_ownership);
  size_type first_ios_dis_iv = ios_vertex.ownership().first_index();
  for (size_type ios_iv = 0, ios_nv = ios_vertex.size(); ios_iv < ios_nv; ios_iv++) {
    size_type ios_dis_iv = first_ios_dis_iv + ios_iv;
    ios_geo_element_p[ios_iv] = geo_element_p(ios_dis_iv);
    geo_element& P = ios_geo_element_p[ios_iv];
    P.set_ios_dis_ie (ios_dis_iv);
    _ios_size_by_variant [P.variant()]++;
  }
  //
  // 1.3) get elements
  //
  polymorphic_array<geo_element> ios_elts (dis_ne);
  ios_elts.get_values (ips);
  size_type ios_dis_ie_start = ios_elts.ownership().first_index();
  base::_map_dimension = 0;
  for (size_type ie = 0, ne = ios_elts.size(); ie < ne; ie++) {
    geo_element& K = ios_elts [ie];
    base::_map_dimension = std::max (K.dimension(), base::_map_dimension);
    K.set_ios_dis_ie (ios_dis_ie_start + ie);
    if (K.dimension() > 0) { // point already counted; used when dim>0 and map_dim=0
      _ios_size_by_variant [K.variant()]++;
    }
  }
  // merge:  when np > n_element, map_dim=0 on some procs without any elts...
  base::_map_dimension = mpi::all_reduce (comm, base::_map_dimension, mpi::maximum<size_type>());
  //
  // 1.4) get faces & edges
  //
  polymorphic_array<geo_element,distributed> ios_faces;
  if   (base::_version  >= 2 && base::_dimension >= 3) {
      ios_faces.resize (dis_n_fac);
      ios_faces.get_values (ips);
      size_type ios_dis_ifac_start = ios_faces.ownership().first_index();
      for (size_type ifac = 0, nfac = ios_faces.size(); ifac < nfac; ifac++) {
        geo_element& F = ios_faces [ifac];
        F.set_ios_dis_ie (ios_dis_ifac_start + ifac);
        _ios_size_by_variant [F.variant()]++;
      }
  }
  polymorphic_array<geo_element,distributed> ios_edges;
  if   (base::_version  >= 2 && base::_dimension >= 2) {
      ios_edges.resize (dis_nedg);
      ios_edges.get_values (ips);
      size_type ios_dis_iedg_start = ios_edges.ownership().first_index();
      for (size_type iedg = 0, nedg = ios_edges.size(); iedg < nedg; iedg++) {
        geo_element& E = ios_edges [iedg];
        E.set_ios_dis_ie (ios_dis_iedg_start + iedg);
        _ios_size_by_variant [E.variant()]++;
      }
  }
  // ------------------------------------------------------------------------
  // 2) mesh partition & element renumbering
  // ------------------------------------------------------------------------
  array<size_type> partition = geo_partition (ios_elts, dis_nv, map_dimension());
  //
  // 2.1) elements renumbering
  //
  array<size_type> ie2ios_dis_ie; // no more used
  ios_elts.repartition (
	partition,
	base::_geo_element[base::_map_dimension],
  	ie2ios_dis_ie,
  	_ios_ige2dis_ige[base::_map_dimension]);

  size_type first_dis_ie = base::_geo_element[base::_map_dimension].ownership().first_index();
  for (size_type ie = 0, ne = size(); ie < ne; ie++) {
    geo_element& K = operator[] (ie);
    K.set_dis_ie (first_dis_ie + ie);
  }
  // ------------------------------------------------------------------------
  // 3) vertices renumbering 
  // ------------------------------------------------------------------------
  // 3.1) global all_reduce
  // TODO: not balanced: vertex nums
  // TODO: also, not optimal: O(N) in communication, memory & CPU, instead of O(N/nproc)
  //
  vector<size_type> new_local_vertex_owner (dis_nv, 0);
  for (size_type ie = 0, ne = size(); ie < ne; ie++) {
    const geo_element& K = operator[] (ie);
    for (size_type iloc = 0; iloc < K.size(); iloc++) {
      new_local_vertex_owner [K[iloc]] = my_proc;
    }
  }
  vector<size_type> new_global_vertex_owner (dis_nv, 0);
  mpi::all_reduce (
	comm,
	new_local_vertex_owner.begin().operator->(),
	dis_nv,
	new_global_vertex_owner.begin().operator->(),
	mpi::maximum<size_type>());

  // 3.2) redistribute the vertex partition
  array<size_type> vertex_partition (ios_vertex.ownership());
  for (size_type ios_dis_iv = vertex_partition.ownership().first_index(); ios_dis_iv < vertex_partition.ownership().last_index(); ios_dis_iv++) {
      vertex_partition.dis_entry (ios_dis_iv) = new_global_vertex_owner[ios_dis_iv];
  }
  vertex_partition.dis_entry_assembly();

  array<size_type>       iv2ios_dis_iv;
  ios_geo_element_p.repartition (
	vertex_partition,
        base::_geo_element[0],
  	iv2ios_dis_iv,
  	_ios_ige2dis_ige[0]);

  distributor vertex_ownership = base::_geo_element[0].ownership();
  base::_vertex.resize (vertex_ownership);
  ios_vertex.permutation_apply (
  	_ios_ige2dis_ige[0],
        base::_vertex);

  // 3.3) set the element[0] array 
  base::_geo_element[0].resize (vertex_ownership);
  size_type first_dis_iv = vertex_ownership.first_index();
  size_type  last_dis_iv = vertex_ownership.last_index();
  for (size_type iv = 0, nv = base::_geo_element[0].size(); iv < nv; iv++) {
    geo_element& P = base::_geo_element [0][iv];
    P[0] = first_dis_iv + iv;
    P.set_dis_ie (first_dis_iv + iv);
  }
  //
  // 3.3) vertices K[iloc] of new numbered element table K still have ios numbering: fix it
  //
  vector<size_type> local_new_num_vert (dis_nv, 0);
  for (size_type dis_iv = _ios_ige2dis_ige[0].ownership().first_index(),
                 dis_nv = _ios_ige2dis_ige[0].ownership().last_index(); 
                 dis_iv < dis_nv; dis_iv++) {
     size_type iv = dis_iv - _ios_ige2dis_ige[0].ownership().first_index();
     local_new_num_vert [dis_iv] = _ios_ige2dis_ige[0][iv];
  }
  vector<size_type> global_new_num_vert (dis_nv, 0);
  mpi::all_reduce (
	comm,
	local_new_num_vert.begin().operator->(),
	dis_nv,
	global_new_num_vert.begin().operator->(),
	mpi::maximum<size_type>());

  for (size_type ie = 0; ie < size(); ie++) {
    geo_element& K = operator[] (ie);
    for (size_type iloc = 0, nloc = K.size(); iloc < nloc; iloc++) {
      assert_macro (K[iloc] < dis_nv, "vertex index "<<K[iloc]<<" out of range [0:"<<dis_nv<<"[");
      assert_macro (global_new_num_vert[K[iloc]] < dis_nv, "new vertex index "<<global_new_num_vert[K[iloc]] <<" out of range [0:"<<dis_nv<<"[");
      K[iloc] = global_new_num_vert [K[iloc]];
    }
  }
  // ------------------------------------------------------------------------
  // 4) edge & face renumbering 
  // ------------------------------------------------------------------------
  if (base::_version >= 2 && base::_map_dimension >= 2) {
    geo_element_renumbering (
        new_global_vertex_owner,
        global_new_num_vert,
        ios_edges,
	dis_nv,
        base::_geo_element[1],
        _ios_ige2dis_ige[1]);
  }
  if (base::_version >= 2 && base::_map_dimension >= 3) {
    geo_element_renumbering (
        new_global_vertex_owner,
        global_new_num_vert,
        ios_faces,
	dis_nv,
        base::_geo_element[2],
        _ios_ige2dis_ige[2]);
  }
  // ------------------------------------------------------------------------
  // 6) get domain, until end-of-file (requires ios_ige2dis_ige renumbering)
  // ------------------------------------------------------------------------
  do {
    domain_indirect_basic<distributed> dom;
    bool status = dom.get (ips, *this);
    if (!status) break;
    base::_domains.push_back (dom);
  } while (true);
  // ------------------------------------------------------------------------
  // 7) external entities
  // ------------------------------------------------------------------------
  reset_size_by ();
  build_external_entities ();
  set_element_edge_index();
  set_element_face_index();

  /** TODO: autres champs a initialiser :

   	 _geo_element [dim]   : pour dim=1,..,map_dim-1

   et distributed :
    	_ios_ige2dis_ige[dim] : pour dim=1,..,map_dim
   */

  return ips;
}
template <class T>
void
geo_rep<T,distributed>::reset_size_by ()
{
  // _ios_size_by_variant[] may also be set !
  geo_base_rep<T,distributed>::reset_size_by();
  std::fill (_ios_size_by_dimension,   _ios_size_by_dimension+4, 0);
  std::fill (_dis_size_by_dimension,   _dis_size_by_dimension+4, 0);
  std::fill (_dis_size_by_variant,     _dis_size_by_variant+reference_element::max_size, 0);
  for (size_type dim = 0; dim <= base::_map_dimension; dim++) {
    _ios_size_by_dimension [dim] = geo_element_ios_ownership(dim).size();
    _dis_size_by_dimension [dim] = base::_geo_element [dim].ownership().dis_size();
  }
  mpi::all_reduce (base::comm(), base::_size_by_variant, reference_element::max_size, _dis_size_by_variant, std::plus<size_type>());
}
/** ------------------------------------------------------------------------
 * loop on geo_element (edges, faces, etc):
 *       identify some vertices, that are referenced
 *       by locally-managed geo_elements, but these vertices are managed
 *       by another processor: e.g. vertices at a partition boundary.
 * ------------------------------------------------------------------------
 */
template <class T>
void
geo_rep<T,distributed>::build_external_entities ()
{
  distributor vertex_ownership = base::_geo_element[0].ownership();
  size_type first_dis_iv = vertex_ownership.first_index();
  size_type  last_dis_iv = vertex_ownership.last_index();
  size_type       dis_nv = vertex_ownership.dis_size();

  // 1) list external vertex indexes:
  std::set<size_type> ext_vertex_set;
  for (size_type dim = 1; dim <= base::_map_dimension; dim++) {
    for (size_type ige = 0, nge = base::_geo_element[dim].size(); ige < nge; ige++) {
      const geo_element& K = base::_geo_element[dim][ige];
      for (size_type iloc = 0, nloc = K.size(); iloc < nloc; iloc++) {
        assert_macro (K[iloc] < dis_nv, "vertex index "<<K[iloc]<<" out of range [0:"<<dis_nv<<"[");
        size_type dis_iv = K [iloc];
        if (dis_iv >= first_dis_iv && dis_iv < last_dis_iv) continue;
        ext_vertex_set.insert (dis_iv);
      }
    }
  }
  warning_macro ("external: ext_iv_set.size="<<ext_vertex_set.size());
  // 2) get external vertices:
  base::_vertex.get_dis_entry (
	ext_vertex_set,
	_ext_vertex);

  base::_geo_element[0].get_dis_entry (
	ext_vertex_set,
	_ext_geo_element_0);
}
#ifdef TO_CLEAN
// ----------------------------------------------------------
// this paradigm appears very often: x[dis_i]
// TODO: embed it directly in array<T,distributed>
// ----------------------------------------------------------
template<class T, class M>
class ext_array {};

template<class T>
class ext_array<T,distributed> : public array<T,distributed> {
public:
// typedef:
  typedef array<T,distributed>     base;
  typedef typename base::size_type size_type;
  typedef std::map<size_type,T>    map_type;
// allocator:
  ext_array(const distributor& ownership = distributor(), const T init_val = T())
   : base(ownership,init_val), _ext_x() {}
  void resize (const distributor& ownership = distributor(), const T init_val = T()) {
     base::resize (ownership,init_val);
     _ext_x.clear();
  }
  void set_external_indexes (const std::set<size_type>& ext_ix_set) {
    _ext_x.clear();
    base::get_dis_entry (ext_ix_set, _ext_x);
  }
// accessor:
  const T& dis_at (const size_type dis_i) const {
    if (dis_i >= base::ownership().first_index() && dis_i < base::ownership().last_index()) {
      size_type i = dis_i - base::ownership().first_index();
      return base::operator[](i);
    }
    typename map_type::const_iterator iter = _ext_x.find (dis_i);
    check_macro (iter != _ext_x.end(), "unexpected external index="<<dis_i);
    return (*iter).second;
  }
// data:
protected:
  map_type               _ext_x;
};
#endif // TO_CLEAN
/** ------------------------------------------------------------------------
 * on any 2d or 3d geo_element K, set K.dis_iedge(iloc) number
 * ------------------------------------------------------------------------
 */
template <class T>
void
geo_rep<T,distributed>::set_element_edge_index()
{
  if (map_dimension() < 2) return;
  warning_macro ("set edge...");
  size_type           nv   = geo_element_ownership(0).size();
  size_type first_dis_iv   = geo_element_ownership(0).first_index();
  size_type  last_dis_iv   = geo_element_ownership(0). last_index();
  size_type first_dis_iedg = geo_element_ownership(1).first_index();

  // ------------------------------------------------------------------------
  // 1) ball(X) := { E; X is a vertex of E }
  // ------------------------------------------------------------------------
  warning_macro ("set edge [1]");
  index_set empty_set; // TODO: add a global allocator to empty_set
  array<index_set,distributed> ball (geo_element_ownership(0), empty_set);
  std::set<size_t> ext_iv_set; // size=O((N/nproc)^((d-1)/d))
  for (size_type iedg = 0, nedg = geo_element_ownership(1).size(); iedg < nedg; iedg++) {
    const geo_element& E = get_geo_element (1, iedg);
    size_type dis_iedg = first_dis_iedg + iedg;
    for (size_type iloc = 0; iloc < 2; iloc++) {
      size_type dis_iv = E[iloc];
      if (dis_iv >= first_dis_iv && dis_iv < last_dis_iv) {
	size_type iv = dis_iv - first_dis_iv;
	ball [iv] += dis_iedg;
      } else {
	index_set dis_iedg_set;
	dis_iedg_set += dis_iedg;
	ball.dis_entry (dis_iv) += dis_iedg_set; // not so efficient
        ext_iv_set.insert (dis_iv);
      }
    }
  }
  ball.dis_entry_assembly (index_set_add_op<index_set,index_set>());
  // ------------------------------------------------------------------------
  // 2) for all the dis_iv that are not handled by the current process
  //    but are referenced by at least an edge, get ext_ball[dis_iv]
  // ------------------------------------------------------------------------
  // Question: faudra-t'il inclure d'autres sommets dans ext_iv_set ?
  // -> ceux issus des triangles, tetra, ect, et externes a la partition ?
  // Question : est-ce que ca en ajouterait ? reponse : OUI (teste')
  // mais c'est pas utile pour l'instant : ball sert juste aux aretes
  warning_macro ("set edge [2] ext_iv_set.size="<<ext_iv_set.size());
#ifdef TODO
  ball.set_dis_indexes (ext_iv_set);
#endif // TODO

  warning_macro ("set edge done");
}
/** ------------------------------------------------------------------------
 * on any 3d geo_element K, set K.dis_iface(iloc) number
 * ------------------------------------------------------------------------
 */
template <class T>
void
geo_rep<T,distributed>::set_element_face_index()
{
  if (map_dimension() < 3) return;
}
// ----------------------------------------------------------------------------
// read from file
// ----------------------------------------------------------------------------
template <class T>
void
geo_rep<T,distributed>::load (
  std::string filename, 
  const communicator& comm)
{
  idiststream ips;
  ips.open (filename, "geo", comm);
  check_macro(ips.good(), "\"" << filename << "[.geo[.gz]]\" not found.");
  get (ips);
  std::string root_name = delete_suffix (delete_suffix(filename, "gz"), "geo");
  std::string name = get_basename (root_name);
  base::_name = name;
}
// ----------------------------------------------------------------------------
// output
// ----------------------------------------------------------------------------
/// @brief helper permutation class for geo i/o
template <class V = typename polymorphic_traits<geo_element>::derived_type>
struct geo_element_perm {
  typedef geo_element::size_type size_type;
  geo_element_perm (const polymorphic_array<geo_element,distributed,V>& elts) 
    : _elts(elts) {}
  size_type operator[] (size_type ie) const {
    const geo_element& K = _elts [ie];
    return K.ios_dis_ie();
  }
  const polymorphic_array<geo_element,distributed,V>& _elts;
};
template <class T>
odiststream&
geo_rep<T,distributed>::put (odiststream& ops)  const
{
  using namespace std;
  communicator comm = base::_geo_element[0].ownership().comm();
  size_type io_proc = odiststream::io_proc();
  size_type my_proc = comm.rank();
  size_type nproc   = comm.size();
  size_type dis_nv = base::_vertex.dis_size ();
  size_type dis_ne = base::_geo_element[base::_map_dimension].dis_size ();
  ops << "#!geo" << endl
      << endl
      << "mesh" << endl
      << base::_version
      << " " << base::_dimension
      << " " << dis_nv
      << " " << dis_ne;
  if   (base::_version >= 2) {
    if (base::_dimension >= 3) {
      ops << " " << base::_geo_element[2].dis_size();
    }
    if (base::_dimension >= 2) {
      ops << " " << base::_geo_element[1].dis_size();
    }
  }
  ops << endl << endl;

  // build a permutationh array (could be avoided, but requires a template Permutation arg in array::permuted_put
  array<size_type> iv2ios_dis_iv (base::_vertex.ownership());
  for (size_type iv = 0, niv = iv2ios_dis_iv.size(); iv < niv; iv++) {
    const geo_element& P = base::_geo_element[0][iv];
    iv2ios_dis_iv [iv] = P.ios_dis_ie();
  }
  base::_vertex.permuted_put_values (ops, iv2ios_dis_iv, _point_put<T>(base::_dimension));
  ops << endl;

  // elements are permuted back to original order and may
  // refer to original vertices numbering
  std::vector<size_type> vertex_perm ((my_proc == io_proc) ? dis_nv : 0);
  size_type tag_gather = distributor::get_new_tag();
  if (my_proc == io_proc) {
    size_type i_start = iv2ios_dis_iv.ownership().first_index(my_proc);
    size_type i_size  = iv2ios_dis_iv.ownership().size       (my_proc);
    for (size_type i = 0; i < i_size; i++) {
      vertex_perm [i_start + i] = iv2ios_dis_iv [i];
    }
    for (size_type iproc = 0; iproc < nproc; iproc++) {
      if (iproc == my_proc) continue;
      size_type i_start = iv2ios_dis_iv.ownership().first_index(iproc);
      size_type i_size  = iv2ios_dis_iv.ownership().size       (iproc);
      comm.recv (iproc, tag_gather, vertex_perm.begin().operator->() + i_start, i_size);
    }
  } else {
    comm.send (0, tag_gather, iv2ios_dis_iv.begin().operator->(), iv2ios_dis_iv.size());
  }
  geo_element_permuted_put put_element (vertex_perm);

  base::_geo_element[base::_map_dimension].permuted_put_values (
	ops, 
	geo_element_perm<> (base::_geo_element[base::_map_dimension]),
	put_element);
  ops << endl;
  //
  // put faces & edges
  //
  if   (base::_version >= 2 && base::_dimension >= 3) {
      base::_geo_element[2].permuted_put_values (
	ops, 
	geo_element_perm<> (base::_geo_element[2]),
	put_element);
  }
  if   (base::_version >= 2 && base::_dimension >= 2) {
      base::_geo_element[1].permuted_put_values (
	ops, 
	geo_element_perm<> (base::_geo_element[1]),
	put_element);
  }
  //
  // put domains
  //
  for (typename std::vector<domain_indirect_basic<distributed> >::const_iterator
        iter = base::_domains.begin(),
        last = base::_domains.end();
	iter != last; ++iter) {
    ops << endl;
    (*iter).put (ops, *this);
  }
  return ops;
}
template <class T>
void
geo_rep<T,distributed>::dump (std::string name)  const {
  base::_vertex.dump        (name + "-vert");
  base::_geo_element[base::_map_dimension].dump(name + "-elem");
}
// --------------------------------------------------------------------------
// accessors to distributed data
// --------------------------------------------------------------------------
template <class T>
typename geo_rep<T,distributed>::size_type
geo_rep<T,distributed>::dis_ige2ios_dis_ige (size_type dim, size_type dis_ige) const
{
    check_macro (dim == 0, "dimension="<<dim<<": not yet !");
    if (dis_ige >= vertex_ownership().first_index()
     && dis_ige <  vertex_ownership().last_index()) {
        size_type ige = dis_ige - vertex_ownership().first_index();
        const geo_element& P = base::_geo_element[0][ige];
        return P.ios_dis_ie();
    }
    typename polymorphic_map<geo_element>::const_iterator iter = _ext_geo_element_0.find (dis_ige);
    check_macro (iter != _ext_geo_element_0.end(), "unexpected vertex index:"<<dis_ige);
    const geo_element& P = *iter;
    return P.ios_dis_ie();
}
template <class T>
const typename geo_rep<T,distributed>::vertex_type&
geo_rep<T,distributed>::dis_vertex (size_type dis_iv) const
{
    if (dis_iv >= vertex_ownership().first_index()
     && dis_iv <  vertex_ownership().last_index()) {
        size_type iv = dis_iv - vertex_ownership().first_index();
        return base::_vertex [iv];
    }
    // here, dis_iv is not managed by current proc
    // try on external associative table
    typename vertex_map_type::const_iterator iter = _ext_vertex.find (dis_iv);
    check_macro (iter != _ext_vertex.end(), "unexpected vertex index:"<<dis_iv);
    return (*iter).second;
}
// --------------------------------------------------------------------------
// access by geo_element(dim,idx)
// --------------------------------------------------------------------------
template <class T>
distributor
geo_rep<T,distributed>::geo_element_ios_ownership (size_type dim) const
{
  return _ios_ige2dis_ige[dim].ownership();
}
template <class T>
typename geo_rep<T,distributed>::size_type
geo_rep<T,distributed>::ios_ige2dis_ige (size_type dim, size_type ios_ige) const
{
  return _ios_ige2dis_ige[dim][ios_ige];
}
template <class T>
typename geo_rep<T,distributed>::size_type
geo_rep<T,distributed>::ige2ios_dis_ige (size_type dim, size_type ige) const
{
    const geo_element& K = base::_geo_element[dim][ige];
    return K.ios_dis_ie();
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class geo_rep<Float,distributed>;
template geo_basic<Float,distributed> compact (const geo_basic<Float,distributed>&);

} // namespace rheolef
#endif // _RHEOLEF_HAVE_MPI
