///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/config.h"

#ifdef _RHEOLEF_HAVE_MPI

#include "rheolef/space.h"

namespace rheolef {

template <class T>
space_rep<T,distributed>::space_rep (const geo_basic<T,distributed>& omega_in, std::string approx)
  : space_base_rep<T,distributed>::space_base_rep (omega_in, approx),
    _idof2ios_dis_idof(),
    _ios_idof2dis_idof(),
    _ext_dis_idof2blk_dis_iub()
{
    warning_macro ("space cstor...");

    if (base::get_element().name() == "") return; // empty space cstor

    // omega_in is compressed by space_constitution() allocator
    // when it is a domain: then it becomes a geo_domain, with compressed numbering
    // so, omega_in can be different from omega:
    const geo_basic<T,distributed>& omega = base::get_geo();
    warning_macro ("space cstor: name="<<base::get_geo().name());

    // TODO: extend to P1d mixed(t,q), P1d mixed(T,P,H); to P2; to P2d...
    /* can be done by inquiring:
	switch (map_dim) {
         case 2: {
	  size_type t_ndof_per_subgeo[4];
	  size_type q_ndof_per_subgeo[4];
	  basis.get_ndof_per_subgeo (hat_t, t_ndof_per_subgeo);
	  basis.get_ndof_per_subgeo (hat_q, q_ndof_per_subgeo);
	   // P0 -> (0,0,1,0)	when map_dim=2
	   // P1 -> (1,0,0,0)	for all map_dim
	   // P2 -> (1,1,0,0)	when map_dim=2 and hat_t
	   // P2 -> (1,1,1,0)	when map_dim=2 and hat_q
	   // P3 -> (1,2,1,0)	when map_dim=2 and hat_t
	   // P3 -> (1,2,4,0)	when map_dim=2 and hat_q
          if (element.numb.is_continuous) {
	    // sum globaly vert, edges
          } else {
	    // sum locally vert, edges
          }
     */
    distributor ownership;
    distributor ios_ownership;
    std::string numb = base::get_element().get_numbering().name();
    warning_macro ("space cstor: numb="<< numb);
    if (numb == "P1" || numb == "P0") {
      size_type idof_dim;
      if (numb == "P0") {
        idof_dim = base::get_geo().map_dimension();
      } else { // P1
        idof_dim = 0; 
      }
      ownership = omega.geo_element_ownership(idof_dim);
      ios_ownership = omega.geo_element_ios_ownership(idof_dim);
      _idof2ios_dis_idof.resize  (ownership);
      _ios_idof2dis_idof.resize  (ios_ownership);
      for (size_type idof = 0, ndof = ownership.size(); idof < ndof; idof++) {
	_idof2ios_dis_idof [idof] = omega.ige2ios_dis_ige (idof_dim,idof);
      }
      for (size_type ios_idof = 0, ios_ndof = ios_ownership.size(); ios_idof < ios_ndof; ios_idof++) {
        _ios_idof2dis_idof [ios_idof] = omega.ios_ige2dis_ige (idof_dim,ios_idof);
      }
    } else if (numb == "P1d") {
      size_type map_dim = base::get_geo().map_dimension();
      size_type ndof, ios_ndof, dis_ndof, nvert_per_elt;
      switch (map_dim) {
	case 0: {
	  ndof     =     omega.    size_by_variant()[reference_element::p];
	  ios_ndof =     omega.ios_size_by_variant()[reference_element::p];
	  dis_ndof =     omega.dis_size_by_variant()[reference_element::p]; 
          nvert_per_elt = 1;
          warning_macro ("P1d,d=0:     size_by_variant[p] = " << omega.    size_by_variant()[reference_element::p]);
          warning_macro ("P1d,d=0: ios_size_by_variant[p] = " << omega.ios_size_by_variant()[reference_element::p]);
          warning_macro ("P1d,d=0: dis_size_by_variant[p] = " << omega.dis_size_by_variant()[reference_element::p]);
	  break;
        }
	case 1: {
	      ndof =   2*omega.    size_by_variant()[reference_element::e];
	  ios_ndof =   2*omega.ios_size_by_variant()[reference_element::e];
	  dis_ndof =   2*omega.dis_size_by_variant()[reference_element::e]; 
          nvert_per_elt = 2;
	  break;
        }
	case 2: {
	      ndof =   3*omega.    size_by_variant()[reference_element::t] 
	             + 4*omega.    size_by_variant()[reference_element::q]; 
	  ios_ndof =   3*omega.ios_size_by_variant()[reference_element::t] 
	             + 4*omega.ios_size_by_variant()[reference_element::q]; 
	  dis_ndof =   3*omega.dis_size_by_variant()[reference_element::t] 
	             + 4*omega.dis_size_by_variant()[reference_element::q]; 

	  bool mixed   = omega.dis_size_by_variant()[reference_element::t] != 0 &&
	                 omega.dis_size_by_variant()[reference_element::q] != 0;
	  check_macro (!mixed, "unsupported 2d mixed triangle/quadrangle mesh");
	  if (omega.dis_size_by_variant()[reference_element::t] != 0) {
            nvert_per_elt = 3;
	  } else {
            nvert_per_elt = 4;
          }
	  break;
        }
	case 3:
	default: {
	      ndof =   4*omega.    size_by_variant()[reference_element::T]
	             + 6*omega.    size_by_variant()[reference_element::P]
	             + 8*omega.    size_by_variant()[reference_element::H];
	  ios_ndof =   4*omega.ios_size_by_variant()[reference_element::T]
	             + 6*omega.ios_size_by_variant()[reference_element::P]
	             + 8*omega.ios_size_by_variant()[reference_element::H];
	  dis_ndof =   4*omega.dis_size_by_variant()[reference_element::T]
	             + 6*omega.dis_size_by_variant()[reference_element::P]
	             + 8*omega.dis_size_by_variant()[reference_element::H];
	  bool mixed = ( omega.dis_size_by_variant()[reference_element::T] != 0 &&
	                 omega.dis_size_by_variant()[reference_element::P] != 0) ||
	               ( omega.dis_size_by_variant()[reference_element::P] != 0 &&
	                 omega.dis_size_by_variant()[reference_element::H] != 0) ||
	               ( omega.dis_size_by_variant()[reference_element::H] != 0 &&
	                 omega.dis_size_by_variant()[reference_element::T] != 0);
	  check_macro (!mixed, "unsupported 3d mixed tetra/penta or hexa mesh");
	  if (omega.dis_size_by_variant()[reference_element::T] != 0) {
            nvert_per_elt = 4;
	  } else if (omega.dis_size_by_variant()[reference_element::P] != 0) {
            nvert_per_elt = 6;
	  } else {
            nvert_per_elt = 8;
	  }
	  break;
        }
      }
          ownership = distributor (dis_ndof, comm(),     ndof);
      ios_ownership = distributor (dis_ndof, comm(), ios_ndof);
      warning_macro ("P1d:     ndof = " <<     ndof << ", dis_ndof      = " << dis_ndof);
      warning_macro ("P1d: ios_ndof = " << ios_ndof << ", dis_ndof      = " << dis_ndof);
      warning_macro ("P1d: nvert_per_elt = " << nvert_per_elt);
      _idof2ios_dis_idof.resize  (ownership);
      _ios_idof2dis_idof.resize  (ios_ownership);
      warning_macro ("P1d: nge = " << omega.geo_element_ownership(map_dim).size());
      check_macro (ndof == nvert_per_elt*omega.geo_element_ownership(map_dim).size(),
	"incompatible ndof="<<ndof<<" and ne="<<omega.geo_element_ownership(map_dim).size());
      for (size_type ige = 0, nge = omega.geo_element_ownership(map_dim).size(); ige < nge; ige++) {
	size_type ios_dis_ige = omega.ige2ios_dis_ige (map_dim, ige);
	for (size_type iloc = 0; iloc < nvert_per_elt; iloc++) {
	  size_type idof         = nvert_per_elt*ige         + iloc;
	  size_type ios_dis_idof = nvert_per_elt*ios_dis_ige + iloc;
	  _idof2ios_dis_idof [idof] = ios_dis_idof;
        }
      }
      warning_macro ("P1d: map_dim  = " << map_dim);
      warning_macro ("P1d: ios_nge = " << omega.geo_element_ios_ownership(map_dim).size());
      dis_warning_macro ("P1d: barrier");
      check_macro (ios_ndof == nvert_per_elt*omega.geo_element_ios_ownership(map_dim).size(),
	"incompatible ios_ndof="<<ios_ndof<<" and ios_ne="<<omega.geo_element_ios_ownership(map_dim).size());
      for (size_type ios_ige = 0, ios_nge = omega.geo_element_ios_ownership(map_dim).size(); ios_ige < ios_nge; ios_ige++) {
        size_type dis_ige = omega.ios_ige2dis_ige (map_dim, ios_ige);
	for (size_type iloc = 0; iloc < nvert_per_elt; iloc++) {
	  size_type ios_idof = nvert_per_elt*ios_ige + iloc;
	  size_type dis_idof = nvert_per_elt*dis_ige + iloc;
          _ios_idof2dis_idof [ios_idof] = dis_idof;
        }
      }
    } else {
      error_macro ("unexpected approximation numbering: \""<< numb << "\"");
    }
    warning_macro ("space cstor done");
}
/// @brief subroutine used by freeze_body
template <class T>
void
space_rep<T,distributed>::append_external_dof (
    const geo_basic<T,distributed>& dom,
    std::set<size_type>&            ext_dof_set) const
{
    warning_macro ("space external...");
    size_type first_dis_idof = base::ownership().first_index();
    size_type  last_dis_idof = base::ownership().last_index();
    for (size_type ie = 0, ne = dom.size(); ie < ne; ie++) {
        const geo_element& K = dom [ie];
        std::vector<size_type> dis_idof;
        get_dis_idof (K, dis_idof);
#ifdef TO_CLEAN
	warning_macro ("K"<<K.dis_ie()<<": idof.size="<<dis_idof.size());
        for (size_type loc_idof = 0, loc_ndof = dis_idof.size(); loc_idof < loc_ndof; loc_idof++) {
	    warning_macro ("idof["<<loc_idof<<"]="<<dis_idof[loc_idof]);
        }
#endif // TO_CLEAN
        for (size_type loc_idof = 0, loc_ndof = dis_idof.size(); loc_idof < loc_ndof; loc_idof++) {
	    if (dis_idof [loc_idof] >= first_dis_idof && dis_idof [loc_idof] < last_dis_idof) continue;
            ext_dof_set.insert (dis_idof [loc_idof]);
        }
    }
    warning_macro ("space external done");
}
template <class T>
void
space_rep<T,distributed>::freeze_body () const
{
    warning_macro ("space freeze...");
    base::base_freeze_body();
    // -----------------------------------------------------------------------
    // 1) symbolic assembly: 
    // loop on elements & identify some dofs, that are referenced
    // by locally-managed geo_elements, but these dofs are managed
    // by another processor: e.g. dofs associated to vertices on 
    // a partition boundary.
    // -----------------------------------------------------------------------
    // set local numbering with global indexes
    size_type first_dis_iu = base::_iu_ownership.first_index();
    size_type first_dis_ib = base::_ib_ownership.first_index();
    for (size_type idof = 0, ndof = base::_idof2blk_iub.size(); idof < ndof; idof++) {
        size_type first_dis_iub = base::_idof2blk_iub [idof].is_blocked() ? first_dis_ib : first_dis_iu;
        base::_idof2blk_iub [idof].set_iub (base::_idof2blk_iub[idof].iub() + first_dis_iub);
    }
    std::set<size_type> ext_dof_set;

    // 1.1 loop on the main mesh
    warning_macro ("space freeze: external");
    append_external_dof (base::get_geo(), ext_dof_set);

    // 1.2 loop on all domains and insert also external dofs
    if (base::get_element().get_numbering().is_continuous()) {
      for (size_type idom = 0, ndom = base::get_geo().n_domain(); idom < ndom; idom++) {
        const geo_basic<T,distributed>& dom = base::get_geo().get_domain (idom);
        append_external_dof (dom, ext_dof_set);
      }
    }
    warning_macro ("space freeze: scatter...");
    // get external dof numbering:
    base::_idof2blk_iub.get_dis_entry (ext_dof_set, _ext_dis_idof2blk_dis_iub);

    // shift localy-managed dofs to local indexes:
    warning_macro ("space freeze: shift...");
    for (size_type idof = 0, ndof = base::_idof2blk_iub.size(); idof < ndof; idof++) {
        size_type first_dis_iub = base::_idof2blk_iub [idof].is_blocked() ? first_dis_ib : first_dis_iu;
        base::_idof2blk_iub [idof].set_iub (base::_idof2blk_iub[idof].iub() - first_dis_iub);
    }
    warning_macro ("space freeze done");
}
// ----------------------------------------------------------------------------
// accessors for external dofs
// ----------------------------------------------------------------------------
template <class T>
typename space_rep<T,distributed>::size_type
space_rep<T,distributed>::dis_iub (size_type dis_idof) const 
{
    base::freeze_guard();
    if (dis_idof >= base::ownership().first_index() && dis_idof < base::ownership().last_index()) {
        size_type idof = dis_idof - base::ownership().first_index();
        size_type idx = base::iub (idof);
        bool      blk = base::is_blocked (idof);
        size_type idx_start = blk ? base::ib_ownership().first_index() : base::iu_ownership().first_index();
        return idx_start + idx;
    }
    // here, dis_idof is not managed by current proc: try on external associative table
    typename map_pair_type::const_iterator iter = _ext_dis_idof2blk_dis_iub.find (dis_idof);
    check_macro (iter != _ext_dis_idof2blk_dis_iub.end(), "unexpected dis_idof="<<dis_idof);
    return (*iter).second.iub();
}
template <class T>
bool
space_rep<T,distributed>::dis_is_blocked (size_type dis_idof) const
{
    base::freeze_guard();
    if (dis_idof >= base::ownership().first_index() && dis_idof < base::ownership().last_index()) {
        size_type idof = dis_idof - base::ownership().first_index();
        return base::is_blocked (idof);
    }
    // here, dis_idof is not managed by current proc: try on external associative table
    typename map_pair_type::const_iterator iter = _ext_dis_idof2blk_dis_iub.find (dis_idof);
    check_macro (iter != _ext_dis_idof2blk_dis_iub.end(), "unexpected dis_idof="<<dis_idof);
    return (*iter).second.is_blocked();
}
// ----------------------------------------------------------------------------
// space ios setting
// ----------------------------------------------------------------------------
#ifdef TO_CLEAN
template <class T>
void
space_rep<T,distributed>::get_ios_dis_idof (const geo_element& K, std::vector<size_type>& ios_dis_idof) const
{
    base::freeze_guard();
    size_type idof_dim;
    // TODO: depend upon loc_idof, for P2 and such; here only for P0 & P1 => basis::dof_dimension(loc_idof)
    if (base::get_element().name() == "P0") {
      idof_dim = K.dimension();
    } else { // P1
      idof_dim = 0; 
    }
    base::get_dis_idof (K, ios_dis_idof);
    for (size_type loc_idof = 0, loc_ndof = ios_dis_idof.size(); loc_idof < loc_ndof; loc_idof++) {
      ios_dis_idof [loc_idof] = base::get_geo().dis_ige2ios_dis_ige (idof_dim, ios_dis_idof[loc_idof]);
    }
}
#endif // TO_CLEAN
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class space_rep<Float,distributed>;

} // namespace rheolef
#endif // _RHEOLEF_HAVE_MPI
