///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
# include "rheolef/config.h"

# ifdef _RHEOLEF_HAVE_MPI
# include "rheolef/asr.h"
# include "rheolef/csr.h"

# include "rheolef/asr_store.h"
# include "rheolef/mpi_assembly_begin.h"
# include "rheolef/mpi_assembly_end.h"
#include <functional>
#include <rheolef/functional_sgi_ext.h> // compose1

// using namespace std;

namespace rheolef {
// ----------------------------------------------------------------------------
// class member functions
// ----------------------------------------------------------------------------

template<class T>
asr_mpi_rep<T>::asr_mpi_rep(
    const distributor& row_ownership,
    const distributor& col_ownership)
  : asr_seq_rep<T> (row_ownership, col_ownership),
    _dis_nnz(0),
    _stash(),
    _send(),
    _receive(),
    _receive_max_size(0)
{
}
template <class T>
void
asr_mpi_rep<T>::resize (
    const distributor& row_ownership,
    const distributor& col_ownership)
{
  asr_seq_rep<T>::resize(row_ownership, col_ownership);
}
template<class T>
asr_mpi_rep<T>::asr_mpi_rep(
    size_type dis_nrow1,
    size_type dis_ncol1)
  : asr_seq_rep<T>(
    	distributor (dis_nrow1, communicator(), distributor::decide),
        distributor (dis_ncol1, communicator(), distributor::decide)),
    _dis_nnz(0),
    _stash(),
    _send(),
    _receive(),
    _receive_max_size(0)
{
}
template <class T>
void
asr_mpi_rep<T>::resize (
    size_type dis_nrow1,
    size_type dis_ncol1)
{
    if (dis_nrow() != 0 || dis_ncol() != 0) {
	fatal_macro("resize non-empty distributed not yet supported");
    }
    distributor row_ownership (dis_nrow1, communicator(), distributor::decide);
    distributor col_ownership (dis_ncol1, communicator(), distributor::decide);
    asr_seq_rep<T>::resize (row_ownership, col_ownership);
}
template<class T>
asr_mpi_rep<T>::asr_mpi_rep(const asr_mpi_rep& a)
  : asr_seq_rep<T>(a),
    _dis_nnz(a._dis_nnz),
    _stash(),
    _send(),
    _receive(),
    _receive_max_size(0)
{
    assert_macro((a._stash.size() != 0 ||
                  a._send.data.size() != 0 || 
		  a._receive.data.size() != 0),
		 "copy during assembly phase");
    fatal_macro ("physical copy of asr");
}
template<class T>
asr_mpi_rep<T>::asr_mpi_rep(const csr_mpi_rep<T>& a)
  : asr_seq_rep<T>(a.row_ownership(), a.col_ownership()),
    _dis_nnz(a.dis_nnz()),
    _stash(),
    _send(),
    _receive(),
    _receive_max_size(0)
{
    a.to_asr(*this);
}
template <class T>
T&
asr_mpi_rep<T>::dis_entry (size_type dis_i, size_type dis_j)
{
    assert_macro (dis_i < dis_nrow() && dis_j < dis_ncol(), 
		  "indexes ("<<dis_i<<" "<<dis_j<<") out of range [0:"
		  << dis_nrow() << "[x[0:"
		  << dis_ncol() << "[");

    size_type istart = row_ownership().first_index();
    size_type ilast  = row_ownership().last_index();

    if (dis_i >= istart && dis_i <  ilast) {

        // local entry
	row_type& row_i = operator[](dis_i-istart);
	std::pair<typename row_type::iterator,bool> status
	 = row_i.insert (std::pair<const size_type,T>(dis_j,T(0)));
	if (status.second) {
	    // new entry
            asr_seq_rep<T>::_nnz++;
	}
	return (*(status.first)).second;
    }
    // entry into the stash:
    typedef std::pair<size_type,size_type>  ij_t;
    typedef std::pair<const ij_t,T>        aij_t;
    std::pair<typename map_ijv_type::iterator,bool> status 
	 = _stash.insert (aij_t (ij_t (dis_i,dis_j), T(0)));
    return (*(status.first)).second;
}
template <class T>
void
asr_mpi_rep<T>::dis_entry_assembly_begin()
{
    first_op<std::pair<const std::pair<size_type,size_type>, T> > op2;
    first_op<const std::pair<size_type,size_type> >          op1;
	 
    _receive_max_size = mpi_assembly_begin (
	_stash,
	make_apply_iterator(_stash.begin(), compose1(op1,op2)),
	make_apply_iterator(_stash.end(),   compose1(op1,op2)),
        row_ownership(),
        _receive,
        _send);

    _stash.clear();
}
template <class T>
void
asr_mpi_rep<T>::dis_entry_assembly_end()
{
    size_type n_new_entry = mpi_assembly_end (
	_receive,
	_send,
	_receive_max_size,
        asr_make_store(
	    asr_seq_rep<T>::begin() - row_ownership().first_index(),
	    set_add_op<T,T>() ));

    asr_seq_rep<T>::_nnz += n_new_entry;
    _dis_nnz = mpi::all_reduce (row_ownership().comm(), asr_seq_rep<T>::_nnz, std::plus<size_type>());

    _send.waits.clear();
    _send.data.clear();
    _receive.waits.clear();
    _receive.data.clear();
    _receive_max_size = 0;
}
template <class T>
odiststream&
asr_mpi_rep<T>::put (odiststream& ops) const
{
    // send all in a pseudo-sequential matrix on process 0
    size_type io_proc = odiststream::io_proc();
    size_type my_proc = comm().rank();
    distributor io_row_ownership (dis_nrow(), comm(), (my_proc == io_proc ? dis_nrow() : 0));
    distributor io_col_ownership (dis_ncol(), comm(), (my_proc == io_proc ? dis_ncol() : 0));
    asr<T> a (io_row_ownership, io_col_ownership);
    size_type istart = row_first_index();
    for (size_type i = 0; i < asr_seq_rep<T>::nrow(); i++) {
        typename row_type::const_iterator iter = operator[](i).begin();
        typename row_type::const_iterator last = operator[](i).end();
	while (iter != last) {
  	    a.dis_entry (i+istart, (*iter).first) += (*iter).second;
	    iter++;
	}
    }
    a.dis_entry_assembly_begin();
    a.dis_entry_assembly_end();

    if (my_proc == io_proc) {
      // print sequential matrix on io_proc
      a.data().asr_seq_rep<T>::put (ops);
    }
    return ops;
}
template <class T>
idiststream&
asr_mpi_rep<T>::get (idiststream& ips) 
{
    extern void read_mm_header(idiststream&);
    read_mm_header(ips);
    size_type n_row, n_col;
    ips >> n_row >> n_col;
    resize (distributor(n_row), distributor(n_col));
    if (row_ownership().process() == 0) {

        std::istream& is = ips.is();
        size_type gol_nnz;
        size_type n_nz;
        is >> n_nz;
        size_type i, j;
        T aij;
        for (size_type p = 0; p < n_nz; p++) {
            is >> i >> j >> aij;
	    // 1..N convention:
	    i--; j--;
	    dis_entry (i,j) += aij;
        }
    }
    dis_entry_assembly_begin();
    dis_entry_assembly_end();
    return ips;
}
template <class T>
void
asr_mpi_rep<T>::dump(const std::string& name) const
{ 
    asr_seq_rep<T>::dump(name, row_first_index());
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class asr_mpi_rep<Float>;
} // namespace rheolef
# endif // _RHEOLEF_HAVE_MPI
