#ifndef BOOST_PP_IS_ITERATING
#ifndef _RHEOLEF_POLYMORPHIC_MAP_H
#define _RHEOLEF_POLYMORPHIC_MAP_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================

#include "rheolef/polymorphic_traits.h"
#include "rheolef/polymorphic_data_map.h"

namespace rheolef {
namespace mpl = boost::mpl;

// forward declaration:
template <class T, class V, int NV>
struct polymorphic_array_seq_rep;

template <class T, class V, int NV>
struct polymorphic_pair_type {};

template <class T, class V, int NV>
struct polymorphic_map_rep {};

// ----------------------------------------------------------
// run here the recursive inclusion of this file:
// ----------------------------------------------------------
// _RHEOLEF_POLYMORPHIC_MAX_SIZE was defined by "polymorphic_data_vector.h"
#define BOOST_PP_ITERATION_LIMITS (0, _RHEOLEF_POLYMORPHIC_MAX_SIZE)
#define BOOST_PP_FILENAME_1    "rheolef/polymorphic_map.h" // this file
#include BOOST_PP_ITERATE()

} // namespace rheolef
#endif // _RHEOLEF_POLYMORPHIC_MAP_H
#else  // BOOST_PP_IS_ITERATING
// ----------------------------------------------------------
// here, this file is recursively included with growing "N" :
// ----------------------------------------------------------
#define N    BOOST_PP_ITERATION()

/*Class:polymorphic_map
NAME:  polymorphic_map - a map<size_t,T> like polymorphic container (@PACKAGE@-@VERSION@)
SYNOPSYS:
  @example
  template <class T> class polymorphic_map;
  @end example
AUTHORS:
   LJK-IMAG, 38041 Grenoble cedex 9, France
   Pierre.Saramito@imag.fr
DATE:   4 february 2010
End:
*/

// -------------------------------------------------------------
// pair(size_t k; void* ptr): pointer to a hierarchy of classes
//   and k = variant of the derived class
// -------------------------------------------------------------
template <class T, class V>
struct polymorphic_pair_type<T,V,N> : std::pair<std::size_t,void*> {
  typedef std::size_t                                   size_type;
  typedef T                                             element_type;
  typedef std::pair<size_type,void*>                    pair_type;
  typedef polymorphic_data_map<T,V,N>          	        data_type;   // extensible area

#define _RHEOLEF_typedef(z,k,unused)                                    \
    typedef typename mpl::at_c<V,k>::type                 T##k;		\
    typedef typename data_type::map##k##_type             map##k##_type;
    BOOST_PP_REPEAT(N, _RHEOLEF_typedef, ~)
#undef _RHEOLEF_typedef

  polymorphic_pair_type() : pair_type(-1,0) {}
  polymorphic_pair_type(size_type k, void* p) : pair_type(k,p) {}
  void set (size_type k, void* p) { pair_type::first = k; pair_type::second = p; }
  size_type variant()    const { return pair_type::first; }
  void* raw_pointer()    const { return pair_type::second; }

  const T& get_reference() const {
    switch (variant()) {
#define _RHEOLEF_case(z,k,unused)                    	\
      case k : {					\
        typename map##k##_type::iterator iter 		\
          = to_iterator<map##k##_type>(raw_pointer());	\
        return (*iter).second;				\
      }
      BOOST_PP_REPEAT(N, _RHEOLEF_case, ~)
#undef _RHEOLEF_case
      default :{
        error_macro ("unexpected type #"<<variant()<<" variant of "<<typename_macro(T));
	T* p = 0; return *p; // may return something... this statement is never reached
      }
    }
  }
  T& get_reference() {
    switch (variant()) {
#define _RHEOLEF_case(z,k,unused)                    	\
      case k : {					\
        typename map##k##_type::iterator iter 		\
          = to_iterator<map##k##_type>(raw_pointer());	\
        return (*iter).second;				\
      }
      BOOST_PP_REPEAT(N, _RHEOLEF_case, ~)
#undef _RHEOLEF_case
      default :{
        error_macro ("unexpected type #"<<variant()<<" variant of "<<typename_macro(T));
	T* p = 0; return *p; // may return something... this statement is never reached
      }
    }
  }
};
// -------------------------------------------------------------
// representation classes
// -------------------------------------------------------------
template <class T, class V>
class polymorphic_map_rep<T,V,N> {
public:

// typedefs:

    static const size_t                                   _n_variant  = mpl::size<V>::value; // should be = N
    typedef typename std::vector<int>::size_type          size_type;
    typedef T                                             element_type;
    typedef polymorphic_data_map<T,V,N>                   data_type;   // extensible area
    typedef std::map<size_type,
		   polymorphic_pair_type<T,V,N>,
	           std::less<size_type>,
		   heap_allocator<std::pair<size_type, polymorphic_pair_type<T,V,N> > > > 
	  map_type;

#define _RHEOLEF_typedef(z,k,unused)                                    \
    typedef typename mpl::at_c<V,k>::type                 T##k;		\
    typedef typename data_type::map##k##_type             map##k##_type;
    BOOST_PP_REPEAT(N, _RHEOLEF_typedef, ~)
#undef _RHEOLEF_typedef

// allocators:

    polymorphic_map_rep ()
      : _ptr (),
        _data()
      {}

    void clear();

// accessors & modifiers:

    size_type     size() const { return _ptr.size(); }

    template <class U>
    void insert (const std::pair<size_type, U>& dis_idx_value);

    struct const_reference {
        const_reference (const polymorphic_pair_type<T,V,N>& x) : _x(x) {}
        operator const T& () const { return _x.get_reference(); }
      // data:
      protected:
        const polymorphic_pair_type<T,V,N>& _x;
    };
    struct const_iterator {
      typedef typename map_type::const_iterator      raw_iterator;
      typedef std::bidirectional_iterator_tag        iterator_category;
      typedef typename raw_iterator::value_type      value_type;
      typedef typename raw_iterator::difference_type difference_type;
      typedef typename raw_iterator::pointer         pointer;
      typedef typename polymorphic_map_rep<T,V,N>::const_reference reference;

      const_iterator (raw_iterator ri) : _ri(ri) {}
      const_reference operator* () const { return const_reference((*_ri).second); }
      const_iterator& operator++ () { ++_ri; return *this; }
      const_iterator& operator-- () { --_ri; return *this; }
      const_iterator  operator++ (int) {
		const_iterator tmp = *this;
		++_ri;
		return tmp;
      }
      const_iterator  operator-- (int) {
		const_iterator tmp = *this;
		--_ri;
		return tmp;
      }
      bool operator== (const_iterator j) const { return _ri == j._ri; }
      bool operator!= (const_iterator j) const { return _ri != j._ri; }
    protected:
      raw_iterator  _ri;
    };

    const_iterator begin () const;
    const_iterator end () const;
    const_iterator find (size_type dis_i) const;

// i/o

  odiststream& put (odiststream& ops) const;

// data:
protected:
  map_type   _ptr;
  data_type  _data;
};
#ifndef _RHEOLEF_POLYMORPHIC_MAP
#define _RHEOLEF_POLYMORPHIC_MAP
// -------------------------------------------------------------
// wrapper class: polymorphic_map
// -------------------------------------------------------------
//<polymorphic_map:
template <class T, class V = typename polymorphic_traits<T>::derived_type>
class polymorphic_map : public smart_pointer<polymorphic_map_rep<T,V,mpl::size<V>::value> > {
public:

// typedefs:

    static const size_t _n_variant  = mpl::size<V>::value;

    typedef sequential                                 memory_type;
    typedef polymorphic_map_rep<T,V,_n_variant>        rep;
    typedef smart_pointer<rep>                         base;
    typedef typename rep::size_type                    size_type;
    typedef typename rep::const_reference              const_reference;
    typedef typename rep::const_iterator               const_iterator;

// allocators:

    polymorphic_map ();

// accessors & modifiers:

    size_type size () const;
    const_iterator begin () const;
    const_iterator end () const;
    const_iterator find (size_type dis_i) const;

// i/o:

    odiststream& put (odiststream& ops) const;
};
//>pol_ymorphic_map:

template <class T, class V>
inline
polymorphic_map<T,V>::polymorphic_map ()
 : base (new_macro(rep))
{
}
template <class T, class V>
inline
typename polymorphic_map<T,V>::size_type
polymorphic_map<T,V>::size () const
{
    return base::data().size();
}
template <class T, class V>
inline
odiststream&
polymorphic_map<T,V>::put(odiststream& ops) const
{
    return base::data().put(ops);
}
template <class T, class V>
inline
typename polymorphic_map<T,V>::const_iterator
polymorphic_map<T,V>::begin () const
{
    return base::data().begin ();
}
template <class T, class V>
inline
typename polymorphic_map<T,V>::const_iterator
polymorphic_map<T,V>::end () const
{
    return base::data().end ();
}
template <class T, class V>
inline
typename polymorphic_map<T,V>::const_iterator
polymorphic_map<T,V>::find (size_type dis_i) const
{
    return base::data().find (dis_i);
}
#endif // _RHEOLEF_POLYMORPHIC_MAP
// -------------------------------------------------------------
// not inlined : longer code
// -------------------------------------------------------------
#include "rheolef/polymorphic_map.icc"

#endif // BOOST_PP_IS_ITERATING
