///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "P2_numbering.h"
namespace rheolef {
using namespace std;

/* For mopre info: Dhatt et Touzot 2eme ed p110 
 * WARNING:
 * The node numbering correspond to the basis numbering.
 * The node numbering convention is different from Dhatt et Touzot:
 * we use P. L. Georges et H. Borouchaki
 *        "Triangulation de Delaunay et maillages",
 *        Hermes, 1997
 *   - first the vertices
 *   - then the middle of the edges
 *
 * WARNING2: the transformation is still linear
 */

/*
 * global numbering statement: uses symbolic values (do not uses as variable !):
 *
 *    K_idx                 : the current element index in the mesh
 *    K.face(i)             : the i-th element face index in the mesh
 *    K.edge(i)             : the i-th element edge index in the mesh
 *    K[i]                  : the i-th element vertice index in the mesh
 *    mesh_n_geo (dim)      : total number of vertices(dim=0), edges(1), faces(2), volume(3)
 *                            in the mesh
 *    mesh_n_element (type) : total number of element of type 'p', 't', 'q', 'T', 'P', 'H'
 *                            in the mesh
 */
std::string
numbering_P2::name() const
{
  return "P2";
}
numbering_P2::size_type
numbering_P2::idof (
	const size_type*      mesh_n_geo,
	const size_type*      mesh_n_element,
	const geo_element&    K,
	size_type             i_dof_local) const
{
        // all vertices-based dof are numbered first
        // then all edge-based dof
        // and then all dof based on interior nodes (e.g. quadrangles)

	// for mixed triangle-quadrangle mesh : assume triangle are numbered first
        switch (K.type()) {
	  case reference_element::p: 
            return K.index();
	  case reference_element::e: 
            if (i_dof_local < 2) {
              return K [i_dof_local];
            } else {
              return mesh_n_geo[0] + K.index();
            }
	  case reference_element::t: 
            if (i_dof_local < 3) {
              return K [i_dof_local];
            } else {
              return mesh_n_geo[0] + K.edge(i_dof_local - 3);
            }
	  case reference_element::q: 
            if (i_dof_local < 4) {
              return K [i_dof_local];
            } else if (i_dof_local < 8) {
              return mesh_n_geo[0] + K.edge(i_dof_local - 4);
            } else {
              return mesh_n_geo[0] + mesh_n_geo[1]
		  + K.index() - mesh_n_element[reference_element::t];
            }
	  case reference_element::T: 
            if (i_dof_local < 4) {
              return K [i_dof_local];
            } else {
              return mesh_n_geo[0] + K.edge(i_dof_local - 4);
            }
	  case reference_element::P: 
	  case reference_element::H:
	  default:
	    error_macro ("unsupported P2 element on `" << K.name() << "'");
	    return 0;
    	}
}
void
numbering_P2::idof (
	const size_type*      mesh_n_geo,
	const size_type*      mesh_n_element,
	const geo_element&    K, 
	vector<size_type>&    i_dof) const
{
  for (size_type i_dof_local = 0; i_dof_local < K.size(); i_dof_local++)
    i_dof[i_dof_local] 
     = numbering_P2::idof (mesh_n_geo, mesh_n_element, K, i_dof_local);
}
numbering_P2::size_type
numbering_P2::ndof (
              size_type  mesh_map_dimension,
        const size_type* mesh_n_geo,
        const size_type* mesh_n_element) const
{
    switch (mesh_map_dimension) {
      case 0:
	return mesh_n_geo[0];
      case 1:
	return mesh_n_geo[0] + mesh_n_geo[1];
      case 2:
        // number of vertices + nb of edges in the mesh
	// + number of quadrangle (extra interior node)
	return mesh_n_geo[0]
	  + mesh_n_geo[1]
	  + mesh_n_element[reference_element::q];
      case 3:
	return mesh_n_geo[0]
	  + mesh_n_geo[1]
	  + mesh_n_element[reference_element::P]
	  + mesh_n_element[reference_element::H];
      default:
	fatal_macro ("unsupported P2 approximation in `" << mesh_map_dimension << "'");
	return 0;
    }
}
bool
numbering_P2::is_continuous () const
{
  	return true;
}

} // namespace rheolef
