///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is sequential in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/domain_indirect.h"
#include "rheolef/geo.h"
#include "rheolef/geo_connectivity.h"
#include "rheolef/rheostream.h"

namespace rheolef {

// ----------------------------------------------------------------------------
/// side and orientation
// ----------------------------------------------------------------------------
std::istream&
operator>> (std::istream& is, domain_pair_type& x)
{
    char c;
    is >> std::ws >> c;
    if (isdigit(c)) { 
      // positive orientation:
      x._pos = true;
      is.unget(); // or is.putback(c); ?
      return is >> x._ige;
    } else {
      // negative orientation:
      x._pos = false;
      return is >> x._ige;
    }
}
std::ostream&
operator<< (std::ostream& os, const domain_pair_type& x)
{
    if (! x._pos) os << '-';
    return os << x._ige;
}
// ----------------------------------------------------------------------------
/// @brief generic (seq,mpi) i/o in format version 2
// ----------------------------------------------------------------------------
odiststream&
domain_indirect_rep<sequential>::put (odiststream& ops)  const {
  using namespace std;
  ops << "domain" << endl
      << base::_name << endl
      << "2 " << base::_map_dim << " " << size() << endl;
  array<domain_pair_type,sequential>::put_values (ops);
  return ops;
}
// ----------------------------------------------------------------------------
/// @brief sequential i/o in format version 1 or 2 (backward compatibility)
// ----------------------------------------------------------------------------
template<class U>
idiststream&
domain_indirect_rep<sequential>::get (
    idiststream& ips,
    const geo_rep<U,sequential>& omega,
    std::vector<std::set<size_type> > ball[4])
{ 
warning_macro ("get dom...");
  std::istream& is = ips.is();
  if (!scatch(is,"\ndomain")) {
    is.setstate (std::ios::badbit);
    return ips;
  }
warning_macro ("get dom[0]");
  size_type version, n_side;
  is >> base::_name >> version >> base::_map_dim >> n_side;
  array<domain_pair_type,sequential>::resize (n_side);
  check_macro (version <= 2, "unexpected domain format version="<<version);
  if (version == 2 || base::_map_dim == 0) {
    array<domain_pair_type,sequential>::get_values (ips);
    return ips;
  }
  check_macro (version == 1, "unexpected domain format version="<<version);
  check_macro (base::_map_dim <= omega.dimension(), "unexpected domain dimension="
	<<base::_map_dim<<" > geometry dimension = " << omega.dimension());
  //
  // old version=1 format: searh for index of sides
  // => this computation is slow, so format 2 is prefered 
  //    when using large 3d meshes with large boundary domains
  //    or with 3d regions defined as domains
  //
  if (ball[base::_map_dim].size() == 0) {
warning_macro ("get dom: ball...");
    ball[base::_map_dim].resize (omega.n_vertex());
    build_point_to_element_sets (omega.begin(base::_map_dim), omega.end(base::_map_dim), ball[base::_map_dim].begin());
warning_macro ("get dom: ball done");
  }
warning_macro ("get dom: load...");
  polymorphic_array<geo_element,sequential> d_tmp (n_side);
  d_tmp.get_values (ips);
warning_macro ("get dom: solve...");
  polymorphic_array<geo_element,sequential>::const_iterator q = d_tmp.begin();
  for (iterator_ioige p = ioige_begin(), last = ioige_end(); p != last; ++p, ++q) {
    const geo_element& S = *q;
    std::set<size_type> contains_S;
    build_set_that_contains_S (S, ball[base::_map_dim], contains_S);
    check_macro (contains_S.size() >= 1, "domain element not in mesh: " << S);
    check_macro (contains_S.size() <= 1, "problem with domain element: " << S);
    // now, choose the only one mesh (sub-)element matching domain element S
    size_type i_side = *(contains_S.begin());
    const geo_element& S_orig = omega.get_geo_element (base::_map_dim, i_side);
    geo_element::orientation_type orient;
    size_type shift;
    check_macro (S_orig.get_orientation_and_shift (S, orient, shift),
    			"problem with domain element: " << S);
    (*p).set (orient, i_side);
  }
warning_macro ("get dom: solve done");
  return ips;
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template
idiststream&
domain_indirect_rep<sequential>::get (
    idiststream& ips,
    const geo_rep<Float,sequential>&  omega,
    std::vector<std::set<size_type> > ball[4]);

} // namespace rheolef
