# ifndef _SKIT_CSRREP_H
# define _SKIT_CSRREP_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
//
// CSR: Compressed Sparse Row format
// fortran-compatible implementation
// for SPARSKIT
//
// author: Pierre.Saramito@imag.fr
//
// date: 21 january 1997
//

	/*@! \begin{quote}
		The Compressed Sparse Row format is the basic
		format used in SPARSKIT. Its  data structure consists of three arrays.
		
		\begin{itemize}
		
		\item A float array $A$ containing the float values $a_{ij}$ stored row by row,
		from row 0 to $NROW-1$. The length of $A$ is NNZ.
		
		\item An integer array $JA$ containing the column indices 
		of the elements $a_{ij}$ as stored in the array $A$.  The length of 
		$JA$ is NNZ. 
		
		\item An integer array $IA$ containing the pointers to the
		beginning of each row in the arrays $A$ and $JA$. Thus the content of
		$IA(i)$ is the position in arrays $A$ and $JA$ where the $i$-th row
		starts.  The length of $IA$ is $N+1$ with $IA(N)$ containing the
		number $IA(0)+NNZ$, i.e., the address in $A$ and $JA$ of the beginning
		of a fictitious row $N+1$.
		
		\end{itemize}
		The order of the nonzero elements within the same row are {\em a priori} 
		not important.
		
		A variation to this scheme is to {\em sort} the elements in each row
		in such a way that their column positions are in increasing order.
		When this sorting in enforced, it is often possible to
		make substantial savings in the number of operations of
		some well-known algorithms as addition of two sparse matrix.
		
		This variant is preciselly implemented here.
	\end{quote} */

# include "rheolef/array.h"
namespace rheolef { 

template<class T>
class csrrep : public occurence {
    public:
	typedef typename std::vector<int>::size_type       size_type;
	typedef typename Array<size_type>::element_type    element_type;
	typedef typename Array<size_type>::iterator        iterator_index;
	typedef typename Array<T>::iterator                iterator_value;
	typedef typename Array<size_type>::const_iterator  const_iterator_index;
	typedef typename Array<T>::const_iterator          const_iterator_value;
   // private:
	
	//      for I = 0..NROW-1, the index K=IA(I)..IA(I+1)-1 scan row I
	//      in arrays A, JA as A(K), JA(K)
	//!	where 0 $\leq$ JA(K) $\leq$ NCOL-1 are in increasing order,
	//	and IA(NROW) = NNZ = number of non-zero elements.

       	mutable Array<size_type>    IA;   	// pointer array (1:NROW+1)
       	Array<size_type>            JA;   	// column array (1:NNZ)
	Array<T>       	             A;    	// values array (1:NNZ)
       	size_type	            NCOL;       // column dimension
    public:
    // constructors
	csrrep (size_type nrow1=0, size_type ncol1=0, size_type nnz1=0)
	: occurence(), IA(nrow1+1), JA(nnz1), A(nnz1), NCOL(ncol1) 
	{ 
	  fill(IA.begin(), IA.end(), (size_type)(0));
	  IA (nrow1) = nnz1;
	}
	// copy: logical copy for A, JA, IA
	csrrep (const csrrep<T>& b) 
	  : occurence(), IA(b.IA), JA(b.JA), A(b.A), NCOL(b.NCOL)
	    {}
	csrrep (const ssk<T>& b);
	~csrrep(void) {}
    
    //  Access and info functions
        void resize (size_type nr, size_type nc, size_type nz)
	{
	  IA.resize(nr+1);
	  JA.resize(nz); 
	  A.resize(nz); 
	  NCOL = nc;
	  IA (0)  = 0;
	  IA (nr) = nz;
	}
	// read direct access
	const Array<T>&      a() const { return A; }
	const Array<size_type>& ja() const { return JA; }
	const Array<size_type>& ia() const { return IA; }

	// write direct access
	Array<T>&      a() { return A; }
	Array<size_type>& ja() { return JA; }
	Array<size_type>& ia() { return IA; }
	
	size_type nrow () const { return IA.size()-1; }
        size_type ncol () const { return NCOL; }
        size_type nnz () const  { return A.size(); }
};
}// namespace rheolef
# endif // _SKIT_CSRREP_H
